/* Copyright (C) 2002 MySQL AB & Jorge del Conde

  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU Library General Public
  License as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Library General Public License for more details.
    
  You should have received a copy of the GNU Library General Public
  License along with this library; if not, write to the Free
  Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,
  MA 02111-1307, USA 
*/

#ifndef CMYSQLCONNECTION_H
#define CMYSQLCONNECTION_H

#include <qvariant.h>
#include <qobject.h>

#ifdef WIN32
#include <winsock.h>
#endif

#include <mysql.h>
#include <mysqld_error.h>

#include "CConfig.h"
#include "Config.h"
#include "CTextDialog.h"
#include "CMessageWindow.h"

struct ServerVersion {
  uint major;
  uint minor;
  QString rel;
};

class CMySQLConnection : public QObject
{
  Q_OBJECT

public:
  CMySQLConnection(const QString & connectionName, CMessagePanel * msgpanel=0, const QString &path=g_CONNECTION_PATH);
  ~CMySQLConnection();

  void signalConnect(const char *sig, QObject *receiver, const char *member);

  QString getServerVersion();
  QString getHostInfo();
  QString getProtoInfo();
  QString getUptime();
  QString getStatus();

  bool killProcess(unsigned long pid);
  bool flush(const QString & f);
  bool ping();
  bool shutdown();

  bool connect(const QString & dbname = QString::null, bool p = true);
  void disconnect();
  
  ServerVersion getMySQLServerVersion() { return m_ServerVersion; }
  QString Quote(const QString &f);

  bool hasCompletion() { return m_Completion; }

  bool setConnected(bool c, const QString & dbname = QString::null, bool del=false);  
  bool isConnected() { return m_isConnected; }

  bool isSocket() { return m_isSocket; }

  bool setDatabaseName(const QString & dbname);
  QString getDatabaseName() {return m_DatabaseName; }

  QString getFieldSeparator(bool escape = false);
  QString getFieldEncloser(bool escape = false);
  QString getLineTerminator(bool escape = false);
  QString getReplaceEmpty(bool escape = false);

  QString getConnectionName() {return m_ConnectionName; }
  void setConnectionName(const QString & connectionName);

  QString getHostName() {return m_HostName; }  

  QString getPassword() {return m_Password; }
  void setPassword(QString p) { m_Password = p;}

  QString getUserName() {return m_UserName; }
  void setUserName(QString u) { m_UserName = u; }

  QString getSocketName() {return m_SocketFile; }  
  
  bool isBlockingQueries() {return m_BlockingQueries; }
  void setBlockingQueries(bool b) { m_BlockingQueries = b; }

  bool isOneConnection() {return m_OneConnection; }
  void setOneConnection(bool c) { m_OneConnection = c; }

  bool isCompressed() {return m_Compress; }
  void setCompressed(bool c) { m_Compress = c; }

  bool isPromptPassword() {return m_PromptPassword; }
  void setPromptPassword(bool p) { m_PromptPassword = p; }

  bool isRetrieveShowTableStatus() { return m_retrieveShowTableStatus; }

  QString getCreated() { return m_Created; }
  QString getModified() { return m_Modified; }

  int getPort() { return m_Port; }  
  int getTimeout() { return m_Timeout; }  
  
  void update();

  QString getError();
  unsigned int getErrno() { return ((m_isConnected) ? mysql_errno(mysql) : 10061); }

  void PrintError(int type, const QString & msg=QString::null, bool p = true);

  CMessagePanel * messagePanel;

  friend class CMySQLQuery;  

signals:
  void disconnected();  
  void connected();
  void refreshed();
  void statuschanged();

protected:
  void refresh(const QString &path=g_CONNECTION_PATH);
  void setHostName(const QString & h) { m_HostName = h;}  
  void setSocketName(const QString & s) {m_SocketFile = s; }
  void setport(int p) { m_Port = p; }
  MYSQL *mysql;
  
private:
  QString doEscape(const QString &str);
  void initServerVersion();
  ServerVersion m_ServerVersion;
  QString m_Modified;
  QString m_Created;
  QString m_HostName;
  QString m_UserName;
  QString m_Password;
  QString m_ConnectionName;
  QString m_DatabaseName;
  QString m_SocketFile;
  QString m_fieldSeparated;
  QString m_lineTerminated;
  QString m_fieldEnclosed;
  QString m_replaceEmpty;
  bool m_retrieveShowTableStatus;
  bool m_Completion;
  bool m_isSocket;
  bool m_PromptPassword;
  bool m_Compress;
  bool m_isConnected;
  bool m_BlockingQueries;
  bool m_OneConnection;
  int m_Port;
  int m_Timeout;
};

#endif