/*+*****************************************************************************
*                                                                              *
* File: xu.c                                                                   *
*                                                                              *
* Description: X/Xt/Motif utilities                                            *
*                                                                              *
**-****************************************************************************/

#ifndef __lint
char xu_vers[] = "@(#)xu.c	1.17	02/18/99	Written by Lionel Cons";
#endif /* __lint */

/******* Include Files ********************************************************/

#include "xu.h"
#include "util.h"
#include "xn.h"
#include "xmd.h"

#include <X11/X.h>
#include <X11/Xlib.h>
#include <X11/Xatom.h>
#include <X11/IntrinsicP.h>
#include <Xm/DialogS.h>
#include <Xm/MessageB.h>
#include <Xm/MwmUtil.h>

/******* Constants ************************************************************/

#define MAX_WINDOWS	16	/* maximum windows returned by xu_scan */
#define MAX_WIDGETS	16	/* maximum widgets remembered for stay_on_top */

/* types for xu_prompt */
#define XU_QUESTION	0
#define XU_INFORMATION	1
#define XU_ERROR	2

/******* Macros ***************************************************************/

/******* External Stuff *******************************************************/

WININFO xu_window_info[MAX_WINDOWS];
Window  xu_root_window, xu_motif_window;

/******* Local Stuff **********************************************************/

static Display *display;
static Widget toplevel, top_widget = NULL;
static int dpy_width, dpy_height, top_conflicts;
static int window_count = 0, widget_count = 0;
static Widget widget_used[MAX_WIDGETS];
static char *xu_last_atom_name = NULL;

/*
 * return the name of the given atom
 */
char *xu_atom_name(Atom atom)
{
  /* free the previous atom, if any */
  if (xu_last_atom_name) XFree(xu_last_atom_name);
  /* get the atom name */
  xu_last_atom_name = XGetAtomName(display, atom);
  /* return a string anyway */
  return(xu_last_atom_name ? xu_last_atom_name : "???");
}

/*
 * scan all the windows under w to find ones belonging to the X client
 */
static void scan_under(Window w, U32BITS base, U32BITS mask)
{
  unsigned int i, nchildren;
  Status status;
  Window dummy, *children;
  char *name, *cp;
  XWindowAttributes wa;
  int rx, ry;

  /* check this one */
  if (window_count >= MAX_WINDOWS) return;
  if (((U32BITS)w & ~mask) == base) {
    /* this window belongs to the X client */
    if (XFetchName(display, w, &name)) {
      xu_window_info[window_count].id = w;
      xu_window_info[window_count].name = name;
      cp = xu_window_info[window_count].geom;
      if (XGetWindowAttributes(display, w, &wa)) {
	/* same algorithm as xwininfo */
	XTranslateCoordinates (display, w, wa.root, 
			       -wa.border_width, -wa.border_width,
			       &rx, &ry, &dummy);
	sprintf(cp, "%dx%d", wa.width, wa.height);
	cp += strlen(cp);
	if (rx < 0) sprintf(cp, "%d", rx);
	else sprintf(cp, "+%d", rx);
	cp += strlen(cp);
	if (ry < 0) sprintf(cp, "%d", ry);
	else sprintf(cp, "+%d", ry);
      } else {
	strcpy(cp, "?x?+?+?");
      }
      window_count++;
      /* skip subwindows */
      return;
    }
  }
  /* scan the children */
  status = XQueryTree(display, w, &dummy, &dummy, &children, &nchildren);
  if (!status) return;
  for (i=0; i<nchildren; i++) scan_under(children[i], base, mask);
  if (children) XFree(children);
}

/*
 * find all the windows belonging to the X client and fill xu_window_info
 */
int xu_scan(U32BITS base, U32BITS mask)
{
  int i;

  /* free memory used previously */
  for (i=0; i<window_count; i++)
    if (xu_window_info[i].name) XFree(xu_window_info[i].name);
  /* do the job */
  window_count = 0;
  scan_under(xu_root_window, base, mask);
  return(window_count);
}

/*
 * function called when the visibility of a dialog widget changes
 */
static void stay_on_top(Widget w, caddr_t data, XVisibilityEvent *event)
{
  data = data; /* unused parameters */

  if (event->state != VisibilityUnobscured) {
    /* at least partially hidden, should I raise? */
    if (top_widget == NULL) {
      /* nobody is on top yet, I go! */
      top_widget = w;
#ifdef DEBUG
      if (DEBUGGING(DBG_UI))
	dprintf("ui  ", PRIx32BITS " is now on top", top_widget);
#endif
      top_conflicts = 0;
    }
    if (top_widget == w) {
      /* I should be on top! */
#ifdef DEBUG
      if (DEBUGGING(DBG_UI))
	dprintf("ui  ", PRIx32BITS " obscured %d", w, top_conflicts);
#endif
      if (top_conflicts < 9)
	xu_raise(w); /* try to raise */
      else
	top_widget = NULL; /* give up :-( */
      top_conflicts++;
    } else {
      /* hidden by one of my peers, I acknowledge */
      if (top_conflicts > 0) top_conflicts--;
    }
  }
}

/*
 * get rid of a dialog widget
 */
void xu_dismiss(Widget w, XtPointer data)
{
  data = data; /* unused parameters */

  XtDestroyWidget(XtParent(w));
}

/*
 * function called when a dialog widget gets destroyed
 */
static void destroy_cb(Widget w, XtPointer data)
{
  int i;
  data = data; /* unused parameters */

  /* remove me from the widget list */
  for (i=0; i<widget_count; i++)
    if (widget_used[i] == w) widget_used[i] = NULL;
  /* if I was on top... */
  if (top_widget == w) {
    top_widget = NULL;
    /* promote another widget */
    for (i=0; i<widget_count; i++)
      if (widget_used[i] != NULL) {
	top_widget = widget_used[i];
#ifdef DEBUG
	if (DEBUGGING(DBG_UI))
	  dprintf("ui  ", PRIx32BITS " is now on top", top_widget);
#endif
	xu_raise(top_widget);
	break;
      }
  }
}

/*
 * create a dialog widget to show something
 */
static Widget xu_prompt(int type, char *text,
			ANSWER *ok, ANSWER *cancel, ANSWER *help)
{
  Widget shell, dialog;
  Arg args[8];
  int n;
  XmString text_str, ok_str, cancel_str, help_str;
  int x, y;
  unsigned int w, h, bw, d;
  Window dummy;

  /* init */
#ifdef DEBUG
  if (DEBUGGING(DBG_UI))
    switch (type) {
    case XU_QUESTION:
      dprintf("ui/q", "%s", text);
      break;
    case XU_INFORMATION:
      dprintf("ui/i", "%s", text);
      break;
    case XU_ERROR:
      dprintf("ui/e", "%s", text);
      break;
    default:
      die("bad type in xu_prompt: %d", type);
    }
#endif
  /* overkill but to please gcc -Wall */
  text_str = ok_str = cancel_str = help_str = NULL;

  /* dialog shell */
  n = 0;
  XtSetArg(args[n], XmNmwmFunctions, MWM_FUNC_RESIZE | MWM_FUNC_MOVE); n++;
  XtSetArg(args[n], XmNdeleteResponse, XmDO_NOTHING); n++; /* to ignore FUNC_CLOSE */
  shell = NULL; /* to please gcc -Wall */
  switch (type) {
  case XU_QUESTION:
    shell = XmCreateDialogShell(toplevel, "Question", args, n);
    break;
  case XU_INFORMATION:
    shell = XmCreateDialogShell(toplevel, "Information", args, n);
    break;
  case XU_ERROR:
    shell = XmCreateDialogShell(toplevel, "Error", args, n);
    break;
  default:
    die("bad type in xu_prompt: %d", type);
  }
  if (type == XU_QUESTION) {
    /* add handler and callback to stay on top */
    XtAddEventHandler(shell, VisibilityChangeMask, False,
		      (XtEventHandler)stay_on_top, (XtPointer)NULL);
    XtAddCallback(shell, XmNdestroyCallback,
		  (XtCallbackProc)destroy_cb, (XtPointer)NULL);
    /* add the shell to the list of used widgets */
    for (n=0; n<widget_count; n++) if (widget_used[n] == NULL) break;
    if (n < widget_count) {
      /* free slot */
      widget_used[n] = shell;
    } else if (widget_count < MAX_WIDGETS) {
      /* we can store another one */
      widget_used[widget_count++] = shell;
    }
  }

  /* message box */
  n = 0;
  switch (type) {
  case XU_QUESTION:
    XtSetArg(args[n], XmNdialogType, XmDIALOG_QUESTION); n++;
    break;
  case XU_INFORMATION:
    XtSetArg(args[n], XmNdialogType, XmDIALOG_INFORMATION); n++;
    break;
  case XU_ERROR:
    XtSetArg(args[n], XmNdialogType, XmDIALOG_ERROR); n++;
    break;
  }
  XtSetArg(args[n], XmNdefaultButtonType, XmDIALOG_OK_BUTTON); n++;
  text_str = XmStringCreateLtoR(text, XmSTRING_DEFAULT_CHARSET);
  XtSetArg(args[n], XmNmessageString, text_str); n++;
  XtSetArg(args[n], XmNautoUnmanage, False); n++;
  dialog = XmCreateMessageBox(shell, "Message Box", args, n);
  XmStringFree(text_str);
  n = 0;
  if (ok) {
    ok_str = XmStringCreateSimple(ok->text);
    XtSetArg(args[n], XmNokLabelString, ok_str); n++;
  }
  if (cancel) {
    cancel_str = XmStringCreateSimple(cancel->text);
    XtSetArg(args[n], XmNcancelLabelString, cancel_str); n++;
  }
  if (help) {
    help_str = XmStringCreateSimple(help->text);
    XtSetArg(args[n], XmNhelpLabelString, help_str); n++;
  }
  XtSetValues(dialog, args, n);
  if (ok) XmStringFree(ok_str);
  else XtUnmanageChild(XmMessageBoxGetChild(dialog, XmDIALOG_OK_BUTTON));
  if (cancel) XmStringFree(cancel_str);
  else XtUnmanageChild(XmMessageBoxGetChild(dialog, XmDIALOG_CANCEL_BUTTON));
  if (help) XmStringFree(help_str);
  else XtUnmanageChild(XmMessageBoxGetChild(dialog, XmDIALOG_HELP_BUTTON));

  /* callbacks */
  if (ok && ok->callback)
    XtAddCallback(dialog, XmNokCallback, ok->callback, ok->data);
  if (cancel && cancel->callback)
    XtAddCallback(dialog, XmNcancelCallback, cancel->callback, cancel->data);
  if (help && help->callback)
    XtAddCallback(dialog, XmNhelpCallback, help->callback, help->data);

  /* show it */
  XtManageChild(dialog);

  /* center it */
  if (XGetGeometry(display, XtWindow(shell), &dummy, &x, &y, &w, &h, &bw, &d)) {
    x = (dpy_width - w) / 2;
    y = (dpy_height - h) / 2;
    XtMoveWidget(shell, x, y);
  }

  return(shell);
}

/*
 * ask something to the user
 */
Widget xu_ask(char *question, ANSWER *ok, ANSWER *cancel, ANSWER *help)
{
  return(xu_prompt(XU_QUESTION, question, ok, cancel, help));
}

/*
 * show something to the user
 */
Widget xu_info(char *info, ANSWER *ok)
{
  return(xu_prompt(XU_INFORMATION, info, ok, (ANSWER *)NULL, (ANSWER *)NULL));
}

/*
 * show an error to the user
 */
Widget xu_error(char *error)
{
  ANSWER ack;

  ack.text = "Acknowledged";
  ack.callback = (XtCallbackProc)xu_dismiss;
  ack.data = (XtPointer)0;
  return(xu_prompt(XU_ERROR, error, &ack, (ANSWER *)NULL, (ANSWER *)NULL));
}

/*
 * change the message string of a dialog
 */
void xu_text(Widget dialog, char *text)
{
  WidgetList child;
  Cardinal nc;
  XmString str;

  XtVaGetValues(dialog, XmNchildren, &child, XmNnumChildren, &nc, NULL);
  if (nc == 1) {
    /* the only child should be the message box */
    str = XmStringCreateLtoR(text, XmSTRING_DEFAULT_CHARSET);
    XtVaSetValues(child[0], XmNmessageString, str);
    XmStringFree(str);
  }
}

/*
 * raise a dialog
 */
void xu_raise(Widget dialog)
{
  XRaiseWindow(display, XtWindow(dialog));
}

/*
 * query extension information for xn
 */
static void xu_query_extensions(void)
{
  char **names;
  int n, i, opc, evt, err;

  names = XListExtensions(display, &n);
  for (i=0; i<n; i++) {
    if (!XQueryExtension(display, names[i], &opc, &evt, &err)) continue;
#ifdef DEBUG
    if (DEBUGGING(DBG_X) && DEBUGGING(DBG_MISC))
      dprintf("misc", "X extension %s: opcode = %d, event = %d, error = %d",
	      names[i], opc, evt, err);
#endif
    xn_record_extension(opc, names[i]);
  }
  if (names) XFreeExtensionList(names);
}

/*
 * setup the X utilities
 */
void xu_setup(Widget w)
{
  Atom mw, type;
  int format;
  unsigned long length, bytesafter;
  Window *property = NULL;
  Screen *screen;

  toplevel = w;
  display = XtDisplay(toplevel);
  xu_root_window = DefaultRootWindow(display);
  mw = XInternAtom(display, "_MOTIF_DRAG_WINDOW", False);
  xu_motif_window = (Window)NULL;
  if ((XGetWindowProperty(display, xu_root_window, mw,
			  0L, 100000L, False, AnyPropertyType,
			  &type, &format, &length, &bytesafter, 
			  (unsigned char **)&property) == Success) &&
      (type == XA_WINDOW) &&
      (format == 32) &&
      (length == 1)) {
    xu_motif_window = *property;
  }
  screen  = XDefaultScreenOfDisplay(display);
  dpy_width  = XWidthOfScreen(screen);
  dpy_height = XHeightOfScreen(screen);
  xu_query_extensions();
#ifdef DEBUG
  if (DEBUGGING(DBG_X)) {
    dprintf("misc", "screen = %dx%d, root window = "
	    PRIx32BITS ", Motif window = " PRIx32BITS,
	    dpy_width, dpy_height, xu_root_window, xu_motif_window);
    /* well, we are debugging X aren't we? */
    XSynchronize(display, True);
  }
#endif
}
