/*+*****************************************************************************
*                                                                              *
* File: xs.c                                                                   *
*                                                                              *
* Description: X sockets handling                                              *
*                                                                              *
**-****************************************************************************/

#ifndef __lint
char xs_vers[] = "@(#)xs.c	1.12	07/22/99	Written by Lionel Cons";
#endif /* __lint */

/******* Include Files ********************************************************/

#include "xs.h"
#include "util.h"
#include "host.h"
#include "auth.h"
#include "err.h"

#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <sys/un.h>
#include <sys/stat.h>

/******* Constants ************************************************************/

/*
 * from Xproto.h
 */
#define X_TCP_PORT 6000		/* add display number to it */

/*
 * length of the queue for pending connections
 */
#define LISTEN_BACKLOG	10

/*
 * shortcuts because I'm lazy
 */
#define XS_TCP	(XS_WANT_TCP|XS_NEED_TCP)
#define XS_UNIX	(XS_WANT_UNIX|XS_NEED_UNIX)

/******* Macros ***************************************************************/

/******* External Stuff *******************************************************/

int xs_flags = 0;		/* X socket flags */
int xs_fd = -1;			/* fd of the TCP incomming socket */
int xs_fdl = -1;		/* fd of the UNIX incomming socket */

/******* Local Stuff **********************************************************/

static HOST *xs_host;		/* host of the X server */
static int   xs_dispno;		/* display number of the X server */

static char *xs_xsock;		/* path to use for the X sockets */
static char *xs_xdir;		/* directory containing the X sockets */

static char *xs_un_path;	/* path of the UNIX incomming socket */

static char  xs_nbuf[8];	/* buffer to sprintf("%d") to */

/*
 * make a TCP connection to the X server
 * from Xtranssock.c/TRANS(SocketINETConnect)
 */
int xs_connect(void)
{
  struct sockaddr_in s_in;
  int fd;

  memset((char *)&s_in, '\0', sizeof(s_in));
  s_in.sin_family = AF_INET;
  s_in.sin_addr = xs_host->addr;
  s_in.sin_port = htons(X_TCP_PORT + xs_dispno);
  fd = socket(AF_INET, SOCK_STREAM, 0);
  if (fd < 0) {
    sprintf(xs_nbuf, "%d", X_TCP_PORT + xs_dispno);
    err_report(NULL, "can't create TCP socket ", xs_host->ip_name,
	       ":", xs_nbuf, ": ", ERROR, 0);
    return(-1);
  }
#ifdef TCP_NODELAY
  {
    int one = 1;
    setsockopt(fd, IPPROTO_TCP, TCP_NODELAY, &one, sizeof(one));
  }
#endif
  if (connect(fd, (struct sockaddr *)&s_in, sizeof(s_in)) < 0) {
    sprintf(xs_nbuf, "%d", X_TCP_PORT + xs_dispno);
    err_report(NULL, "can't connect TCP socket ", xs_host->ip_name,
	       ":", xs_nbuf, ": ", ERROR, 0);
    return(-1);
  }
  return(fd);
}

/*
 * make a UNIX socket connection to the X server
 * from Xtranssock.c/TRANS(SocketUNIXConnect)
 */
int xs_connect_local(void)
{
  struct sockaddr_un s_un;
  int fd;

  memset((char *)&s_un, '\0', sizeof(s_un));
  s_un.sun_family = AF_UNIX;
  sprintf(s_un.sun_path, xs_xsock, xs_dispno);
  fd = socket(AF_UNIX, SOCK_STREAM, 0);
  if (fd < 0) {
#ifdef DEBUG
    if (DEBUGGING(DBG_WARNING))
      dprintf("warn", "can't create UNIX socket %s: %s", s_un.sun_path, ERROR);
#endif
    return(-1);
  }
  if (connect(fd, (struct sockaddr *)&s_un, sizeof(s_un)) < 0) {
#ifdef DEBUG
    if (DEBUGGING(DBG_WARNING))
      dprintf("warn", "can't connect UNIX socket %s: %s", s_un.sun_path, ERROR);
#endif
    return(-1);
  }
  return(fd);
}

/*
 * accept a TCP connection
 */
int xs_accept(HOST **host)
{
  int newfd;
  struct sockaddr_in s_in;
  int sinlen;
  HOST *newhost;

  /* accept */
  sinlen = sizeof(s_in);
  newfd = accept(xs_fd, (void *)&s_in, &sinlen);
  if (newfd < 0) {
    err_report(NULL, "accept failed: ", ERROR, 0);
    return(-1);
  }

  /* check host */
  newhost = host_create_by_addr(&s_in.sin_addr);
  if (!newhost) {
    err_report(NULL, "accept rejected for ", inet_ntoa(s_in.sin_addr),
	       ": bad or unknown host", 0);
    return(-1);
  }

  /* return */
#ifdef DEBUG
  if (DEBUGGING(DBG_CONN))
    dprintf("conn", "detected connection from %s [%s], port %d",
	      newhost->name, newhost->ip_name, ntohs(s_in.sin_port));
#endif
  *host = newhost;
  return(newfd);
}

/*
 * accept a UNIX socket connection
 */
int xs_accept_local(void)
{
  int newfd;
  struct sockaddr_un s_un;
  int sunlen;

  /* accept */
  sunlen = sizeof(s_un);
  newfd = accept(xs_fdl, (void *)&s_un, &sunlen);
  if (newfd < 0) {
    err_report(NULL, "local accept failed: ", ERROR, 0);
    return(-1);
  }

  /* return */
#ifdef DEBUG
  if (DEBUGGING(DBG_CONN))
    dprintf("conn", "detected local connection");
#endif
  return(newfd);
}

/*
 * create the sockets to act as a fake X server
 * see connection.c and Xtranssock.c
 */
static int xs_listen(int proposed_dispno)
{
  int dispno = proposed_dispno;
  struct sockaddr_in s_in;
  struct sockaddr_un s_un;
  int res, oldumask;

  oldumask = umask(0);
  goto just_do_it;
try_again_sam:
  if (xs_fd  >= 0) close(xs_fd);
  if (xs_fdl >= 0) close(xs_fdl);
  dispno++;
just_do_it:
  /*
   * try to create TCP socket
   */
  if (BITTST(xs_flags, XS_TCP)) {
    memset((char *)&s_in, '\0', sizeof(s_in));
    s_in.sin_family = AF_INET;
    s_in.sin_addr.s_addr = htonl(INADDR_ANY);
    s_in.sin_port = htons(X_TCP_PORT + dispno);
    xs_fd = socket(AF_INET, SOCK_STREAM, 0);
    if (xs_fd < 0) die("can't create TCP socket: %s", ERROR);
#ifdef SO_REUSEADDR
    {
      int one = 1;
      res = setsockopt(xs_fd, SOL_SOCKET, SO_REUSEADDR,
		       (void *)&one, sizeof(one));
      if (res < 0)
	err_report(NULL, "can't setsockopt SO_REUSEADDR: ", ERROR, 0);
    }
#endif /* SO_REUSEADDR */
  }
  /*
   * try to create UNIX socket
   */
  if (BITTST(xs_flags, XS_UNIX)) {
    if (mkdir(xs_xdir, 0777) == 0) chmod(xs_xdir, 0777);
    memset((char *)&s_un, '\0', sizeof(s_un));
    s_un.sun_family = AF_UNIX;
    sprintf(s_un.sun_path, xs_xsock, dispno);
    xs_un_path = str_copy(s_un.sun_path);
    xs_fdl = socket(AF_UNIX, SOCK_STREAM, 0);
    if (xs_fdl < 0) die("can't create UNIX socket: %s", ERROR);
  }
  /*
   * try to bind TCP socket
   */
  if (BITTST(xs_flags, XS_TCP)) {
    res = bind(xs_fd, (void *)&s_in, sizeof(s_in));
    if (res < 0) {
      if ((errno == EADDRINUSE) && BITTST(xs_flags, XS_HUNT))
	goto try_again_sam;
      if (BITTST(xs_flags, XS_NEED_TCP)) {
	die("can't bind TCP socket: %s", ERROR);
      } else {
	err_report(NULL, "can't bind TCP socket: ", ERROR, 0);
	close(xs_fd);
	xs_fd = -1;
      }
    }
  }
  /*
   * try to bind UNIX socket
   */
  if (BITTST(xs_flags, XS_UNIX)) {
    unlink(xs_un_path);
    res = bind(xs_fdl, (void *)&s_un, sizeof(s_un));
    if (res < 0) {
      if ((errno == EADDRINUSE) && BITTST(xs_flags, XS_HUNT))
	goto try_again_sam;
      if (BITTST(xs_flags, XS_NEED_UNIX)) {
	die("can't bind UNIX socket: %s", ERROR);
      } else {
	err_report(NULL, "can't bind UNIX socket: ", ERROR, 0);
	close(xs_fdl);
	xs_fdl = -1;
      }
    }
  }
  /*
   * do we have what we need now?
   */
  if ((xs_fd < 0) && (xs_fdl < 0)) die("can't bind any socket!");
  /*
   * try to set additional TCP socket options
   */
  if (xs_fd >= 0) {
#ifdef SO_LINGER
    {
      static int linger[2] = { 0, 0 };
      res = setsockopt(xs_fd, SOL_SOCKET, SO_LINGER,
		       (void *)linger, sizeof(linger));
      if (res < 0)
	err_report(NULL, "can't setsockopt SO_LINGER: ", ERROR, 0);
    }
#else /* SO_LINGER */
#ifdef SO_DONTLINGER
    {
      res = setsockopt(xs_fd, SOL_SOCKET, SO_DONTLINGER, (void *)NULL, 0);
      if (res < 0)
	err_report(NULL, "can't setsockopt SO_DONTLINGER: ", ERROR, 0);
    }
#endif /* SO_DONTLINGER */
#endif /* SO_LINGER */
  }
  /*
   * try to listen now
   */
  if (xs_fd >= 0) {
    res = listen(xs_fd,  LISTEN_BACKLOG);
    if (res < 0) die("can't listen to TCP socket: %s", ERROR);
  }
  if (xs_fdl >= 0) {
    res = listen(xs_fdl, LISTEN_BACKLOG);
    if (res < 0) die("can't listen to UNIX socket: %s", ERROR);
  }
  /*
   * finally return the new display number
   */
  umask(oldumask);
  return(dispno);
}

/*
 * parse the display to setup xs_dispno and xs_host
 */
static void parse_display(char *dpy)
{
  char *dv, *cp, *dn;

  /* analyze the display, algorithm from XConnDis.c/_XConnectDisplay */
  dv = str_copy(dpy);
  cp = strchr(dv, ':');
  if (cp == NULL) die("bad display (missing colon): %s", dpy);
  if (cp[1] == ':') die("bad display (DECnet is not supported): %s", dpy);
  *cp++ = '\0'; /* end of host part */
  dn = cp;
  cp = strchr(dn, '.');
  if (cp) *cp = '\0'; /* end of number part */
  xs_dispno = atoi(dn);
  if (*dv == '\0' || !strcmp(dv, "unix") || !strcmp(dv, "localhost")) {
    /* local DISPLAY */
    xs_host = localhost;
  } else {
    /* remote DISPLAY */
    xs_host = host_create_by_name(dv);
    if (xs_host == NULL) die("bad display (unknown host %s): %s", dv, dpy);
  }
  str_free(dv);
#ifdef DEBUG
  if (DEBUGGING(DBG_MISC))
    dprintf("misc", "display is %s, using %s X server %d on host %s", dpy,
	    (xs_host == localhost ? "local" : "remote"),
	    xs_dispno, xs_host->name);
#endif
}

/*
 * create a fake X server and initialise some data in order
 * to be able to connect to the real X server when needed
 */
void xs_setup(char *dpy, int *dispno_ret, char *xsock)
{
  char *cp;

  /*
   * check xs_flags
   */
  if (!BITTST(xs_flags, XS_TCP|XS_UNIX))
    die("bad X socket flags: no type defined");

  /*
   * check xsock, see <sys/un.h> for sun_path hard-coded size...
   */
  if (strlen(xsock) > 90) die("bad X socket path (too long): %s", xsock);
  if (!strstr(xsock, "%d")) die("bad X socket path (no %%d): %s", xsock);
  xs_xsock = xsock;
  xs_xdir = str_copy(xsock);
  cp = strrchr(xs_xdir, '/');
  if (cp == NULL) die("bad X socket path (no /): %s", xsock);
  *cp = '\0';
#ifdef DEBUG
  if (DEBUGGING(DBG_MISC))
    dprintf("misc", "X sockets in %s using path %s", xs_xdir, xs_xsock);
#endif

  /*
   * do the remaining setup
   */
  parse_display(dpy);
  auth_setup(xs_host, xs_dispno);
  *dispno_ret = xs_listen(*dispno_ret);
#ifdef DEBUG
  if (DEBUGGING(DBG_MISC))
    dprintf("misc", "created sockets for dispno %d: TCP=%d UNIX=%d",
	    *dispno_ret, xs_fd, xs_fdl);
#endif
}

/*
 * close the fake X server
 */
void xs_close(void)
{
  if (xs_fd  >= 0) close(xs_fd);
  if (xs_fdl >= 0) close(xs_fdl);
  unlink(xs_un_path); /* so that it can be reused immediately... */
  str_free(xs_un_path);
}
