/*+*****************************************************************************
*                                                                              *
* File: util.c                                                                 *
*                                                                              *
* Description: miscellaneous useful things                                     *
*                                                                              *
**-****************************************************************************/

#ifndef __lint
char util_vers[] = "@(#)util.c	1.7	02/19/99	Written by Lionel Cons";
#endif /* __lint */

/******* Include Files ********************************************************/

#include "util.h"

#include <time.h>
#include <stdarg.h>
#include <ctype.h>

/******* Constants ************************************************************/

/******* Macros ***************************************************************/

/******* External Stuff *******************************************************/

#ifdef DEBUG
int debug = 0;			/* debugging flags */
char *timetag = "timetag";	/* special tag to print a time stamp */
#endif

/******* Local Stuff **********************************************************/

static char *argv0 = NULL;	/* saved argv[0] for die() */
#ifdef DEBUG
static char *hex_digit = "0123456789ABCDEF"; /* for hexdump() */

/*
 * dump some data on stderr, nicely formated
 */
void hexdump(char *cp, int len)
{
  int done, n;
  char c, hex_data[50], ascii_data[25];
  
  done = 0;
  while (done < len) {
    /* format one line of 24 bytes */
    for (n=0; n<24 && (done+n)<len; n++) {
      c = cp[done+n];
      hex_data[2*n]   = hex_digit[(c & 0xF0) >> 4];
      hex_data[2*n+1] = hex_digit[ c & 0x0F];
      ascii_data[n] = (isprint((int)c) ? c : '.');
    }
    hex_data[2*n] = ascii_data[n] = '\0';
    /* print the line */
    fprintf(stderr, "%4d: %-48s %s\n", done, hex_data, ascii_data);
    done += n;
  }
}

/*
 * print a message on stderr with an optional tag
 */
void dprintf(char *tag, char *fmt, ...)
{
 va_list ap;

 if (tag) {
   if (tag == timetag) {
     time_t now;
     struct tm *tms;
     
     now = time((time_t *)NULL);
     tms = localtime(&now);
     fprintf(stderr, "%04d/%02d/%02d-%02d:%02d:%02d ",
	     tms->tm_year+1900, tms->tm_mon+1, tms->tm_mday,
	     tms->tm_hour, tms->tm_min, tms->tm_sec);
   } else {
     fprintf(stderr, "%s: ", tag);
   }
 }
 va_start(ap, fmt);
 vfprintf(stderr, fmt, ap);
 va_end(ap);
 fputs("\n", stderr);
}

#endif /* DEBUG */

/*
 * print an error message on stderr and exit
 */
void die(char *fmt, ...)
{
  va_list ap;

  if (argv0) fprintf(stderr, "%s: ", argv0);
  va_start(ap, fmt);
  vfprintf(stderr, fmt, ap);
  va_end(ap);
  fputs("\n", stderr);
  exit(-1);
}

/*
 * malloc with result checking
 */
void *safe_malloc(int size)
{
  void *ptr;

  ptr = malloc(size);
  if (!ptr) die("can't allocate %d bytes: %s", size, ERROR);
  return(ptr);
}

/*
 * returns a copy of the first n bytes of the given string
 */
char *str_ncopy(char *s, int len)
{
  char *new;

  new = (char *)safe_malloc(len + 1);
  strncpy(new, s, len);
  new[len] = '\0';
  return(new);
}

/*
 * returns a copy of the given string
 */
char *str_copy(char *s)
{
  char *new;

  new = (char *)safe_malloc(strlen(s) + 1);
  strcpy(new, s);
  return(new);
}

/*
 * setup this module
 */
void util_setup(char *progname)
{
  argv0 = str_copy(progname);
}
