/*+*****************************************************************************
*                                                                              *
* File: main.c                                                                 *
*                                                                              *
* Description: a Motif X11 connection monitor                                  *
*                                                                              *
**-****************************************************************************/

#ifndef __lint
char main_vers[] = "@(#)main.c	1.25	07/22/99	Written by Lionel Cons";
#endif /* __lint */

/******* Include Files ********************************************************/

#include <stdio.h>
#include <signal.h>
#include <sys/time.h>
#include <sys/resource.h>

#include <X11/Xlib.h>
#include <Xm/Protocols.h>

#include "util.h"
#include "host.h"
#include "xs.h"
#include "iconn.h"
#include "conn.h"
#include "xu.h"
#include "list.h"
#include "opt.h"
#include "cfg.h"
#include "input.h"
#include "popup.h"
#include "sb.h"

/******* Constants ************************************************************/

#define MIN_WIDTH	180	   /* minimum width of the list in pixels */

/******* Macros ***************************************************************/

/******* Local Stuff **********************************************************/

/*
 * destroy a connection
 */
static void connection_destroy(CONN *c)
{
#ifdef DEBUG
  if (DEBUGGING(DBG_CONN))
    dprintf(c->tag, "destroying connection");
  if (DEBUGGING(DBG_LOG))
    dprintf(timetag, "end of connection %s from %s", c->id, c->name);
#endif
  list_remove(c);
  conn_close(c);
  OLD(c);
}

/*
 * callback to accept an incoming connection
 */
static void connection_accept_cb(Widget w, ICONN *ic)
{
  CONN *c;
  int fd;

  /* dismiss the dialog */
  if (w != NULL) xu_dismiss(w, NULL);

  /* create a new connection, first with UNIX socket */
  fd = xs_connect_local();
  /* or, if it fails, with TCP socket */
  if (fd < 0) fd = xs_connect();
  if (fd < 0) {
    iconn_refuse(ic, "Can't connect to X server");
    return;
  }
  c = NEW(CONN);
  conn_init(c, ic, fd);
  list_add(c);
  /* update the input */
  input_update_connection(c);
#ifdef DEBUG
  if (DEBUGGING(DBG_CONN))
    dprintf(c->tag, "accepting%s connection from %s",
	    (ic->local ? " local" : ""), c->host->fq_name);
  if (DEBUGGING(DBG_LOG))
    dprintf(timetag, "connection from %s accepted as %s", c->name, c->id);
#endif

  /* clean up */
  iconn_destroy(ic);
}

/*
 * callback to refuse an incoming connection
 */
static void connection_deny_cb(Widget w, ICONN *ic)
{
#ifdef DEBUG
  if (DEBUGGING(DBG_LOG)) {
    if (ic->name)
      dprintf(timetag, "connection from %s@%s refused", ic->name, ic->host->name);
    else
      dprintf(timetag, "connection from %s refused", ic->host->name);
  }
#endif
  /* dismiss the dialog and refuse the connection */
  if (w != NULL) xu_dismiss(w, NULL);
  iconn_refuse(ic, "Client is not authorized to connect to Server");
}

/*
 * callback to confirm an existing connection
 */
static void connection_confirmed_cb(Widget w, CONN *c)
{
  /* dismiss the dialog and update alert */
  xu_dismiss(w, NULL);
  c->ui.alert = (Widget)NULL;
  /* update the input */
  input_update_connection(c);
}

/*
 * callback to destroy an existing connection
 */
static void connection_destroy_cb(Widget w, CONN *c)
{
  /* dismiss the dialog and update alert */
  xu_dismiss(w, NULL);
  c->ui.alert = (Widget)NULL;
  /* and destroy the connection! */
  connection_destroy(c);
}

/*
 * ask confirmation about a connection
 */
static Widget connection_ask(CONN *c, ICONN *ic)
{
  ANSWER a_accept, a_deny;

  sb_discard();
  a_accept.text = "Accept";
  a_deny.text   = "Deny";
  if (c) {
    sb_add_strings("Accept existing connection ", c->id, " from ", c->name, 0);
    a_accept.callback = (XtCallbackProc)connection_confirmed_cb;
    a_accept.data = (XtPointer)c;
    a_deny.callback = (XtCallbackProc)connection_destroy_cb;
    a_deny.data = (XtPointer)c;
  } else {
    sb_add_string("Accept incoming connection from ");
    if (ic->name) sb_add_strings(ic->name, "@", 0);
    sb_add_string(ic->host->name);
    a_accept.callback = (XtCallbackProc)connection_accept_cb;
    a_accept.data = (XtPointer)ic;
    a_deny.callback = (XtCallbackProc)connection_deny_cb;
    a_deny.data = (XtPointer)ic;
  }
  return(xu_ask(sb_buffer, &a_accept, &a_deny, (ANSWER *)NULL));
}

/*
 * confirm a connection
 */
static void connection_confirm(CONN *c)
{
  /* already displayed? */
  if (c->ui.alert) {
    xu_raise(c->ui.alert);
    return;
  }
  c->ui.alert = connection_ask(c, NULL);
  /* update the input */
  input_update_connection(c);
}

/*
 * callback triggered when a new connection is detected
 */
static void connection_detected_cb(int local, int *source, XtInputId *id)
{
  ICONN *ic;
  int action;
  source = source; id = id; /* unused parameters */

  ic = iconn_create(local, opt.ident);
  if (!ic) return;
  /* default */
  ic->icf = opt.icf;
  action = CFG_ACTION_ASK;
  /* use config file is any */
  if (opt.config[0] != '\0')
    action = cfg_action(opt.config, ic->name, ic->host, &ic->icf);
  switch (action) {
  case CFG_ACTION_ALLOW:
    connection_accept_cb(NULL, ic);
    break;
  case CFG_ACTION_DENY:
    connection_deny_cb(NULL, ic);
    break;
  default:
    connection_ask(NULL, ic);
  }
}

/*
 * miscellaneous things to setup
 */
static void misc_setup(Display *display)
{
  /*
   * setup the sockets
   */
  xs_flags = opt.xflags;
  xs_setup(DisplayString(display), &opt.dispno, opt.xsock);
  if (opt.verbose) {
    fprintf(stdout, "%s:%d\n", localhost->fq_name, opt.dispno);
    fflush(stdout);
  }
  if (xs_fd > 0)
    XtAppAddInput(XtDisplayToApplicationContext(display), xs_fd,
		  (XtPointer)XtInputReadMask,
		  (XtInputCallbackProc)connection_detected_cb, (XtPointer)0);
  if (xs_fdl > 0)
    XtAppAddInput(XtDisplayToApplicationContext(display), xs_fdl,
		  (XtPointer)XtInputReadMask,
		  (XtInputCallbackProc)connection_detected_cb, (XtPointer)1);

  /*
   * increase the maximum number of open files
   */
#ifdef RLIMIT_NOFILE
  {
    struct rlimit rl;

    getrlimit(RLIMIT_NOFILE, &rl);
    rl.rlim_cur = rl.rlim_max;
    setrlimit(RLIMIT_NOFILE, &rl);
  }
#endif /* RLIMIT_NOFILE */

  /*
   * go in the background?
   */
  if (opt.fork) {
    int pid;

    pid = fork();
    if (pid < 0) die("can't fork: %s", ERROR);
    if (pid != 0) exit(0);
  }

  /*
   * close stdin and stdout anyway, see XDISPLAY=`mxconns -verbose &`
   */
  (void) freopen("/dev/null", "r", stdin);
  (void) freopen("/dev/null", "w", stdout);
  /* stderr is kept for debugging and error messages */
}

static void suicide(int sig) {
  sig = sig; /* unused parameters */

#ifdef DEBUG
  if (DEBUGGING(DBG_MISC)) dprintf("misc", "shutting down");
#endif
  xs_close(); /* to remove the UNIX socket */
  exit(0); /* the other connections will be closed here! */
}

#ifdef DEBUG
/*
 * signal handler to toggle debugging
 */
static void toggle_debug(int sig)
{
  sig = sig; /* unused parameters */

  debug = (debug ? 0 : DBG_ALL);
}
#endif /* DEBUG */

/*
 * signal handler to kill the connection with the oldest alert
 */
static void kill_alert(int sig)
{
  CONN *c;
  sig = sig; /* unused parameters */

  c = conn_oldest_alert();
  if (c) connection_destroy(c);
}

/*
 * main
 */
int main(int argc, char *argv[])
{
  Widget top;
  Display *display;
  Dimension width;
  Arg args[5];
  int n;
  struct sigaction new, old;

  /*
   * save program name now to be able to use die()
   */
  util_setup(argv[0]);

  /*
   * X init and argument parsing
   */
  top = opt_init(argc, argv);
  display = XtDisplay(top);
#ifdef DEBUG
  debug = opt.debug;
#endif

  /*
   * main setup
   */
  /* get host name and domain */
  host_setup();
  /* build scrolled list */
  list_setup(top, connection_confirm, popup_handler);
  /* initialise X utilities */
  xu_setup(top);
  /* setup input dispatching */
  input_setup(XtDisplayToApplicationContext(display), connection_destroy);
  /* handle popup menus */
  popup_setup(connection_confirm);
  /* the rest, including sockets */
  misc_setup(display);

  /*
   * set window & icon titles, width
   */
  XtRealizeWidget(top);
  XtVaGetValues(top, XmNwidth, &width, NULL);
  if (width < MIN_WIDTH) width = MIN_WIDTH;
  sprintf(sb_line, "%s:%d", localhost->name, opt.dispno);
  n = 0;
  XtSetArg(args[n], XmNtitle, sb_line); n++;
  XtSetArg(args[n], XmNminWidth, MIN_WIDTH); n++;
  XtSetArg(args[n], XmNwidth, width); n++;
  if (opt.iname[0] != '\0') {
    XtSetArg(args[n], XmNiconName, opt.iname); n++;
  }
  XtSetValues(top, args, n);

  /* 
   * catch the destroy signal from WM & similar 
   */
  XmAddWMProtocolCallback(top, XInternAtom(display, "WM_DELETE_WINDOW", False),
			  (XtCallbackProc)suicide, NULL);
  XtAddCallback(top, XtNdestroyCallback, (XtCallbackProc)suicide, NULL);

  /* 
   * set up signal handlers
   */
  memset((char *)&new, '\0', sizeof(new));
  new.sa_handler = SIG_IGN;
  sigaction(SIGPIPE, &new, &old); /* rather get EPIPE instead */
  new.sa_handler = suicide;
  sigaction(SIGINT,  &new, &old);
  sigaction(SIGHUP,  &new, &old);
  sigaction(SIGTERM, &new, &old);
#ifdef DEBUG
  new.sa_handler = toggle_debug;
  sigaction(SIGUSR1, &new, &old);
#endif
  new.sa_handler = kill_alert;
  sigaction(SIGUSR2, &new, &old);

  /* 
   * just do it (TM)
   */
#ifdef DEBUG
  if (DEBUGGING(DBG_MISC)) dprintf("misc", "in main loop now...");
#endif
  XtAppMainLoop(XtDisplayToApplicationContext(display));
  suicide(0);
  /* to please lint and gcc -Wall but suicide() is fatal... */
  return(0);
}
