/*+*****************************************************************************
*                                                                              *
* File: input.c                                                                *
*                                                                              *
* Description: input handling, i.e. where to listen to with XtAddInput         *
*                                                                              *
**-****************************************************************************/

#ifndef __lint
char input_vers[] = "@(#)input.c	1.7	03/16/99	Written by Lionel Cons";
#endif /* __lint */

/******* Include Files ********************************************************/

#include <X11/Xlib.h>

#include "input.h"
#include "conn.h"
#include "xu.h"
#include "sb.h"

/******* Constants ************************************************************/

#define NO_INTERVAL_ID   (XtIntervalId)0

/******* Macros ***************************************************************/

/******* External Stuff *******************************************************/

/******* Local Stuff **********************************************************/

static XtAppContext input_app_context;
static void (*input_destroy_connection)(CONN *);

/*
 * check and update the input ids
 */
static void input_check(XtInputId *id, int fd, XtInputMask cond,
			XtInputCallbackProc proc, XtPointer data)
{
  switch (cond) {
  case XtInputReadMask:
  case XtInputWriteMask:
    if (*id != NO_INPUT_ID) return;
    *id = XtAppAddInput(input_app_context, fd, (XtPointer)cond, proc, data);
#ifdef DEBUG
    if (DEBUGGING(DBG_DISPATCH)) {
      CONN *c = (CONN *)data;
      dprintf(c->tag, "new %s input id",
	      ((id == &(c->ui.cid.in)) ? "client read " :
	       ((id == &(c->ui.cid.out)) ? "client write" :
		((id == &(c->ui.sid.in)) ? "server read " :
		 ((id == &(c->ui.sid.out)) ? "server write" : "???")))));
    }
#endif
    break;
  case XtInputNoneMask:
    if (*id == NO_INPUT_ID) return;
    XtRemoveInput(*id);
    *id = NO_INPUT_ID;
#ifdef DEBUG
    if (DEBUGGING(DBG_DISPATCH)) {
      CONN *c = (CONN *)data;
      dprintf(c->tag, "no  %s input id",
	      ((id == &(c->ui.cid.in)) ? "client read " :
	       ((id == &(c->ui.cid.out)) ? "client write" :
		((id == &(c->ui.sid.in)) ? "server read " :
		 ((id == &(c->ui.sid.out)) ? "server write" : "???")))));
    }
#endif
    break;
  default:
    die("bad condition in input_check: %d", cond);
  }
}

/*
 * freeze a connection
 */
static void input_freeze(CONN *c)
{
  XtPointer data = (XtPointer)c;
  IO *iop;
  ID *idp;
  
  /* client side */
  iop = &c->client;
  idp = &c->ui.cid;
  input_check(&idp->in,  iop->fd_in,  XtInputNoneMask, NULL, data);
  input_check(&idp->out, iop->fd_out, XtInputNoneMask, NULL, data);
  /* server side */
  iop = &c->server;
  idp = &c->ui.sid;
  input_check(&idp->in,  iop->fd_in,  XtInputNoneMask, NULL, data);
  input_check(&idp->out, iop->fd_out, XtInputNoneMask, NULL, data);
}

/*
 * update the input ids and other things
 */
static void input_update(CONN *c, IO *iop, ID *idp, XtInputCallbackProc proc)
{
  XtPointer data = (XtPointer)c;

  /* frozen connection or dialog shown: don't do I/O */
  if (BITTST(c->flags, CF_FROZEN) || c->ui.alert) {
    input_freeze(c);
    return;
  }

  /* an alert has just been cleared, we have to trigger proc */
  if (c->xinfo.alert) {
    (*proc)(c, NULL, NULL);
    return;
  }

  /* error handling */
  if (BITTST(iop->flags, IO_ERROR)) {
    input_destroy_connection(c);
    return;
  }

  /* normal case */
  if (BITTST(iop->flags, IO_READ))
    input_check(&idp->in,  iop->fd_in,  XtInputReadMask,  proc, data);
  else
    input_check(&idp->in,  iop->fd_in,  XtInputNoneMask,  proc, data);
  if (BITTST(iop->flags, IO_WRITE))
    input_check(&idp->out, iop->fd_out, XtInputWriteMask, proc, data);
  else
    input_check(&idp->out, iop->fd_out, XtInputNoneMask,  proc, data);
}

/*
 * callback to dismiss an alert
 */
static void input_alert_dismiss_cb(Widget w, CONN *c)
{
  xu_dismiss(w, NULL);
  c->ui.alert = (Widget)NULL;
  /* update the input */
  input_update_connection(c);
}

/*
 * callback to replace next packet by a NoOp
 */
static void input_alert_noop_cb(Widget w, CONN *c)
{
  BITSET(c->flags, CF_NOOP);
  input_alert_dismiss_cb(w, c);
}

/*
 * callback to kill a connection
 */
static void input_alert_kill_cb(Widget w, CONN *c)
{
  xu_dismiss(w, NULL);
  c->ui.alert = (Widget)NULL;
  input_destroy_connection(c);
}

/*
 * popup an alert
 */
static void input_alert(CONN *c)
{
  ANSWER allow, killit, noop;

  BITCLR(c->flags, CF_NOOP);
  allow.text = "Allow";
  allow.callback = (XtCallbackProc)input_alert_dismiss_cb;
  allow.data = (XtPointer)c;
  killit.text = "Kill";
  killit.callback = (XtCallbackProc)input_alert_kill_cb;
  killit.data = (XtPointer)c;
  noop.text = "NoOp";
  noop.callback = (XtCallbackProc)input_alert_noop_cb;
  noop.data = (XtPointer)c;
  sprintf(sb_line, "A %s has been detected on connection %s,\ncoming from ",
	  (c->xinfo.alert_level == AL_DUBIOUS ?
	   "dubious X request" : "security violation"), c->id);
  sb_discard();
  sb_add_strings(sb_line, c->name, ":\n     ", c->xinfo.alert,
		 "\nWhat do you want to do?", 0);
  c->ui.alert = xu_ask(sb_buffer, &allow, &killit, &noop);
}

/*
 * what to do when the client side wants to do I/O
 */
static void input_server(CONN *c, int *source, XtInputId *id)
{
  source = source; id = id; /* unused parameters */

  /* frozen connection or dialog shown: don't do I/O! */
  if (BITTST(c->flags, CF_FROZEN) || c->ui.alert) return;

  /* do some work */
  conn_server_read(c);
  conn_server_write(c);

  /* update input */
  input_update(c, &c->server, &c->ui.sid, (XtInputCallbackProc)input_server);
}

/*
 * what to do when the server side wants to do I/O
 */
static void input_client(CONN *c, int *source, XtInputId *id)
{
  source = source; id = id; /* unused parameters */

  /* frozen connection or dialog shown: don't do I/O! */
  if (BITTST(c->flags, CF_FROZEN) || c->ui.alert) return;

  /* do some work */
  conn_client_read(c);
  conn_client_write(c);

  /* alert detected? */
  if (c->xinfo.alert) input_alert(c);

  /* update input */
  input_update(c, &c->client, &c->ui.cid, (XtInputCallbackProc)input_client);
}

/*
 * update all the input callbacks for a connection
 */
void input_update_connection(CONN *c)
{
  input_update(c, &c->client, &c->ui.cid, (XtInputCallbackProc)input_client);
  input_update(c, &c->server, &c->ui.sid, (XtInputCallbackProc)input_server);
}

/*
 * setup
 */
void input_setup(XtAppContext ac, void (*dc)(CONN *))
{
  input_app_context = ac;
  input_destroy_connection = dc;
}
