/***************************************************************************
 Mutella - A commandline/HTTP client for the Gnutella filesharing network.

 This program is free software; you can redistribute it and/or
 modify it under the terms of the GNU General Public License
 as published by the Free Software Foundation; either version 2
 of the License, or (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 uiterminal.cpp  -  User Interface code for the terminal/commandline.

    begin                : Thu Jan 31 2002
    copyright            : (C) 2002 by Max Zaitsev
    email                : maksik@gmx.co.uk
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

//#include <iostream.h>
#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <ctype.h>
#include <time.h>
#include <sys/types.h>	/* waitpid () */
#include <sys/wait.h>	/* waitpid () */
#include <unistd.h>	/* fnctl () */
#include <fcntl.h>	/* fnctl () */

#include "mutella.h"
#include "structures.h"

#include "controller.h"
#include "mui.h"
#include "uiterminal.h"
#include "property.h"
#include "preferences.h"
#include "mprintf.h"
#include "ansicolor.h"
#include "event.h"
#include "messages.h"
#include "lineinput.h"
#include "conversions.h"
#include "common.h"
#include "term_help.h"

#define _isblank isspace

class MUITermPriv;
typedef bool (MUITermPriv::*tComHandler)(char * arg);
typedef set<int> intSet;
typedef set<DWORD> dwordSet;
typedef set<void*> ptrSet;
typedef vector<int> intVec;
typedef vector<DWORD> dwordVec;

struct SUIColors {
	// list/results
	char header[32];
	char id[32];
	char size[32];
	char speed[32];
	char percent[32];
	char time[32];
	char name[32];
	char extra[32];
	char sha1[32];
	char flag1[32];
	char flag2[32];
	char num[32];
	char ip[32];
	char status[32];
	// general purpose
	char deflt[32];
	char message[32];
	char error[32];
	// set
	char variable[32];
	char value[32];
};

/////////////////////////////////////////////////////////////////////////////
// Paged output with color support

class MPrintfColor : public MPrintfMore
{
public:
	MPrintfColor(int nCPL = -1, int nLPP = -1) : MPrintfMore(nCPL, nLPP), m_bEnableColor(true) {}
protected:
	virtual void PrePrint(CString& s){
		// search for '??' combinations which are suposed to define 'style'
		// listed in m_propClr;
		int nStart = 0;
		int nPos;
		int nPos1;
		int n,i;
		bool bStyled = false;
		LPCSTR szPropName;
		int nNameLen;
		while ( -1 != (nPos = s.find("??",nStart)) )
		{
			CString sStyle = s.substr(nPos+2,16); // style names are normally short
			bool bFound = false;
			nPos1 = -1;
			// unfortunately sStyle.find('>'); wont work
			n = sStyle.length();
			for(i = 0; i<n; ++i)
				if (sStyle[i] == '>')
				{
					nPos1 = i;
					break;
				}
				else if (sStyle[i]<'0' || sStyle[i]>'z' || (sStyle[i]>'9' && sStyle[i]<'A'))
				{
					// non-alphanum
					break;
				}
			if (nPos1>=0)
			{
				//found '>' -- we have to cut out &&...> and replace it with the ANSI combination
				sStyle = sStyle.substr(0,nPos1);
				CString sTmp = s.substr(nPos+2+nPos1+1); // '+1' is for spacer after
				s.cut(nPos); // cut without memory realloc
				s += ANSI_NORMAL;
				bStyled = false;
				//
				if (m_bEnableColor && sStyle.size())
				{
					MProperty* pP = m_propClr.FindProperty(sStyle.c_str());
					if (pP)
					{
						if (strlen(pP->GetStringValue()))
						{
							s += ESC_CHAR;
							s += pP->GetStringValue();
							bStyled = true;
						}
					}
				}
				nStart = s.length();
				s += sTmp;
			}
			else
			{
				// '??' case -- just replace it with 'ANSI_NORMAL'
				CString sTmp = s.substr(nPos+2);
				s.cut(nPos); // cut without memory realloc
				s += ANSI_NORMAL;
				nStart = s.length();
				s += sTmp;
				bStyled = false;
			}
		}
		if (bStyled)
			s+= ANSI_NORMAL;
	}
public:
	MPropertyContainer m_propClr;
	bool m_bEnableColor;
};

//////////////////////////////////////////////////////////////////////
// Re-define line input to enable completion

class MyLineInput : public MLineInput
{
public:
	MyLineInput(MUITermPriv* pMaster, MController* pController);
	virtual bool CompleteWord(const CString& part, list<CString>& listCompletions);
protected:
	MController* m_pController;
	MUITermPriv* m_pMaster;
};

////////////////////////////////////////////////////////////////////////
// event display

class MEventPrinter : public MSyncEventReceiver {
public:
protected:
	virtual bool IsOfInterest(MEvent* p){
		ASSERT(p);
		return p->GetSeverity()>=ES_IMPORTANT;
	}
	virtual void OnEvent(MEvent* p, bool){
			ASSERT(p);
			printf("IMPORTANT EVENT:\n"
#ifdef _DEBUG
			             "  type: %d\n  severity: %d\n  id: %d\n"
			             ,
			             p->GetType(), p->GetSeverity(), p->GetID()
#endif
			             );
			if (p->GetID())
				printf("  %s\n"
				       "  %s\n",
				       GetMessageString(p->GetID()).c_str(),
				       p->Format().c_str());
			else
				printf("  %s\n", p->Format().c_str());
	}
};

class MEventCache : public MAsyncEventReceiver {
public:
	MEventCache() : MAsyncEventReceiver(INT_MAX, 0x4000 /*16k*/) {}
protected:
	virtual bool IsOfInterest(MEvent* p){
		ASSERT(p);
		return p->GetSeverity()>=ES_JUSTINCASE;
	}
};

////////////////////////////////////////////////////////////////////////////
// UI implementation

class MUITermPriv {
public:
	MUITermPriv(MController*);
	~MUITermPriv();
	// basic functionality
	bool attach();
	bool init();
	void detach();
	void ui_loop();
	void stop(){m_bContinue = 0;} // !! this will not work for the external "Stop" request
	//
	enum SetMode {
		SET_SET,
		SET_ADD,
		SET_REMOVE
	};
	bool set_helper(MPropertyContainer* pPC, SetMode mode, char * arg);
	bool stop_help(LPCSTR szCom,bool bDelPart, char * arg);
	//
	int m_nCounter;
	dwordVec m_vecConnIDs;
	dwordVec m_vecTransIDs;
	dwordVec m_vecSearchIDs;
	dwordVec m_vecResIDs;
	MController* m_pController;
	int  m_nDummy1;
	bool m_bContinue;
	int  m_nDummy2;
	std::queue<CString> m_command_queue;
	// paged terminal output
	MPrintfColor m_more;
	// terminal input
	MyLineInput m_input;
	// couple of properties
	int m_nMaxResultsDispl;
	bool m_bShowGroups;
	char m_szStartUpScript[1024];
	bool m_bUseColor;
	char m_szColorScheme[1024];
	SUIColors m_colors;
	MEventPrinter m_evPrinter;
	MEventCache m_evCache;

	// command support
	typedef struct {
		char*			name;     // User printable name of the function.
		tComHandler 	handler;  // Function to call to do the job.
		char*			doc_short; // Documentation for this function.
		char*			doc_long;
	} ComEntry;
	typedef struct {
		char*			name;     // User printable name of the function.
		char*			doc_short; // Documentation for this function.
		char*			doc_long;
	} VarEntry;
	static ComEntry commands[];
	static VarEntry variables[];
	ComEntry* find_command(char*);
	bool execute_line (const CString& sLine);
	// commands
	bool com_help(char * arg);
	bool com_exit(char * arg);
	// search
	bool com_find(char * arg);
	bool com_list(char * arg);
	bool com_ls(char * arg);
	bool com_edit(char * arg);
	bool com_delete(char * arg);
	bool com_erase(char * arg);
	bool com_clear(char * arg);
	bool com_results(char * arg);
	// retrive
	bool com_get(char * arg);
	// rename file
	bool com_move(char * arg);
	// stop transfers
	bool com_stop(char * arg);
	bool com_kill(char * arg);
	// info
	bool com_info(char * arg);
	bool com_hosts(char * arg);
	// connections
	bool com_open(char * arg);
	bool com_close(char * arg);
	// misc
	bool com_load(char * arg);   // script
	//bool com_system(char * arg); // execute a shell command
	bool com_set(char * arg);    // envinroment management
	bool com_set_add(char * arg);
	bool com_set_remove(char * arg);
	bool com_color(char * arg); // termUI colors
	// shared files
	bool com_scan(char * arg);
	bool com_library(char * arg);
	//
	bool com_system(char * arg);
	bool com_version(char * arg);
};

MUITermPriv::MUITermPriv(MController* pC) : m_more(80,24), m_input(this, pC)
{
	m_pController = pC;
	m_bContinue = true;
	
}

MUITermPriv::~MUITermPriv()
{
}

bool MUITermPriv::attach()
{
	//
	ED().AddReceiver(&m_evPrinter);
	ED().AddReceiver(&m_evCache);
	//
	MProperty* pP;
	MPropertyContainer* pPC = m_pController->GetPropertyContainer();
	pPC->AddSection("TerminalUI");
	pPC->AddProperty("MaxResultsDisplayed", &m_nMaxResultsDispl,    1000);
	pPC->AddProperty("ShowGroups",          &m_bShowGroups,         true);
	pPC->AddProperty("StartupScript",       m_szStartUpScript,      1024, "~/.mutella/termrc");
	pPC->AddProperty("UseANSIColor",        &m_more.m_bEnableColor, true);
	pPC->AddProperty("ColorScheme",         m_szColorScheme,        1024, "~/.mutella/termclr");
	pP = pPC->AddProperty("TerminalCols",   &m_more.m_nCPL,         MPrintfMore::GuessTermCols());
	if (pP) pP->SetPersistance(false);
	pP = pPC->AddProperty("TerminalLines",  &m_more.m_nLPP,         MPrintfMore::GuessTermLines());
	if (pP) pP->SetPersistance(false);
	pPC->AddProperty("Paginate",            &m_more.m_bBreak,       true);
	pPC->SetCurrentSection(NULL);
	return true;
}

bool MUITermPriv::init()
{
	//
	if (!m_input.Init(stdin, stdout))
	{
		// TODO: post event
		TRACE("Failed to init line-input object");
		return false;
	}
	// prepare property container for the color scheme
	m_more.m_propClr.AddProperty("header",   m_colors.header,  32, "[1m"); //bold
	m_more.m_propClr.AddProperty("id",       m_colors.id,      32, "[1m"); //bold
	m_more.m_propClr.AddProperty("size",     m_colors.size,    32, "[34m"); //blue
	m_more.m_propClr.AddProperty("speed",    m_colors.speed,   32, "[32m"); //green
	m_more.m_propClr.AddProperty("percent",  m_colors.percent, 32, "[33m"); //yellow
	m_more.m_propClr.AddProperty("time",     m_colors.time,    32, "[35m"); //magenta
	m_more.m_propClr.AddProperty("name",     m_colors.name,    32, "[31m"); //red
	m_more.m_propClr.AddProperty("extra",    m_colors.extra,   32, "[32m");
	m_more.m_propClr.AddProperty("sha1",     m_colors.sha1,    32, "[33m");
	m_more.m_propClr.AddProperty("flag1",    m_colors.flag1,   32, "[34m"); //bold
	m_more.m_propClr.AddProperty("flag2",    m_colors.flag2,   32, "[33m"); //bold
	m_more.m_propClr.AddProperty("num",      m_colors.num,     32, "[36m");
	m_more.m_propClr.AddProperty("ip",       m_colors.ip,      32, "[35m");
	m_more.m_propClr.AddProperty("status",   m_colors.status,  32, "[34m");
	// general purpose
	//m_more.m_propClr.AddProperty("default",  m_colors.deflt,   32, "");
	m_more.m_propClr.AddProperty("message",  m_colors.message, 32, "[34m");//blue
	m_more.m_propClr.AddProperty("error",    m_colors.error,   32, "[31m");//red
	// set
	m_more.m_propClr.AddProperty("variable", m_colors.variable,32, "[35m");
	m_more.m_propClr.AddProperty("value",    m_colors.value,   32, "[34m");
	// load the color scheme
	if (strlen(m_szColorScheme))
		m_more.m_propClr.Read(m_szColorScheme);
}

void MUITermPriv::detach()
{
	m_input.Close();
	if (strlen(m_szColorScheme) && m_more.m_bEnableColor)
	{
		TRACE("Saving color scheme...");
		m_more.m_propClr.Write(m_szColorScheme);
	}
	// remove properties from m_pController
	// TODO that
	//
	ED().RemoveReceiver(&m_evPrinter);
	ED().RemoveReceiver(&m_evCache);
	// store the color scheme

}

MUITermPriv::ComEntry MUITermPriv::commands[] = {
  { "help",   &MUITermPriv::com_help,   "Displays a help message. Type `help <command>' for more specific info", NULL},
  { "?",      &MUITermPriv::com_help,   "Synonym for `help'", NULL},
  { "exit",   &MUITermPriv::com_exit,   "Exits Mutella",
  									"    Initiates exit sequence. Cannot be abbreviated to one letter"},
  { "info"   ,&MUITermPriv::com_info,   "Displays various information regarding current network activities",
  									"info [network] [connections] [transfers] [uploads] [downloads] [loop]\n"
  									"    Displays different bits of information regarding current status of the\n"
  									"    client and its activities. Use parameters to select specific types of\n"
  									"    info, e.g. 'info downloads' will only display status of current downloads.\n"
  									"    Parameter 'loop' repeats the info command each 2 seconds until key is\n"
  									"    pressed. 'info loop' can be combined with any other 'info' parameter.\n"
  									"    Parameters can be abbreviated and multiple parameters can be given. Without\n"
  									"    parameters displays all status information.\n"
  									"    NOTE: rates displayed for individual Gnutella-net connecvtions do not\n"
  									"    reflect the bandwidth consumed and are given only for the reference.\n"
  									"    Total connection speed however is quite reliable parameter."},
  { "hosts"  ,&MUITermPriv::com_hosts,  "Displays current content of the hosts cache", NULL },
  { "find",   &MUITermPriv::com_find,   "Adds a query to the search list",
  									"find <list of keywords> [options]\n"
  									"    Adds new query to the current list of searches. All the keywords are\n"
  									"    matched as a boolean AND ignoring upper/lower case.\n"
  									"    NEW: exclusive search is supported: `find hook .avi -hookers' will only\n"
  									"    look for a movie, filtering out porno-crap.\n"
  									"    NEW: Sha1 searches can be created by replacing the search string with\n"
  									"    the string starting from 'sha1:', e.g. 'sha1:XX..XX', where 'XX..XX' is\n"
  									"    base32-encoded sha1 hash. Sha1 searches can be combined with other\n"
  									"    options and '/autoget' in particular."
  									"    Additional search options allow one to specify exact file size with\n"
  									"    `size:XXXX' or minimum file size with `min:XXXX' or approximate size\n"
  									"    within 10% tolerance with `around:XXXX'. Auto-get searches can be created\n"
  									"    by means of adding `/auto' or `/autoget' flag (to be used with care)."
  									"    With no parameters equivalent to `list'" },
  { "list",   &MUITermPriv::com_list,   "Lists current searches",
  									"list [pattern] [options]\n"
  									"    Displays current list of searches. Optionally takes pattern as a\n"
  									"    parameter. When pattern is given only searches that match it are\n"
  									"    listed. This is useful when search list grows above 5-10 entries.\n"
  									"    Additionally it is possible to filter out searches with no hits by\n"
  									"    '-empty' option and automatically generated searches with '-auto'" },
  { "ls",     &MUITermPriv::com_ls,     "Synonym for 'list -empty'", NULL},
  { "edit"   ,&MUITermPriv::com_edit,   "Modifies keyword string for existing search",
  									"edit <search_ID> <new_search_string>\n"
  									"    Enables user to modify existing searches. Useful when automatic keyword\n"
  									"    extraction fails to generate proper search string when searching for\n"
  									"    alternative locations. NOTE that this does not change the file name of\n"
  									"    the file being downloaded"},
  { "delete", &MUITermPriv::com_delete, "Deletes a query or queries from the list of searches",
  									"delete <search_ID(s)>\n"
  									"    Deletes a query or queries from the current list of searches. Numeric\n"
  									"    ID(s) should be taken from last output of the `list' command. IDs can\n"
  									"    be given in fairly relaxed way. For example `delete 2,5,8-10,1' will\n"
  									"    delete searches listed under numbers 1,2,5,8,9,10; `delete 1-' deletes\n"
  									"    all the searches listed by the last `list' command\n" },
  { "erase", &MUITermPriv::com_erase,   "Deletes query(ies) from the search list and erases the partial file(s)",
  									"erase <search_ID(s)>\n"
  									"    Deletes a query or queries from the current list of searches and deletes\n"
  									"    the partial file(s) for the case of automatically generated auto-get\n"
  									"    searches. The partial files for the active downloads will not be deleted.\n"
  									"    You should use 'kill' command instead. Similarly to 'delete', numeric\n"
  									"    ID(s) should be taken from last output of the `list' command. IDs can\n"
  									"    be given in fairly relaxed way. For example `delete 2,5,8-10,1' will\n"
  									"    delete searches listed under numbers 1,2,5,8,9,10; `delete 1-' deletes\n"
  									"    all the searches listed by the last `list' command\n" },

  { "clear",  &MUITermPriv::com_clear,  "Clears results list for the query or query list",
  									"clear <search_ID(s)>\n"
  									"    Clears results of each the queries referenced by numeric IDs. Numeric\n"
  									"    ID(s) are to be taken from last output of the `list' command. IDs can\n"
  									"    be given in fairly relaxed way. For example `clear 2,5,8-10,1' will\n"
  									"    clear up searches listed under numbers 1,2,5,8,9,10; `clear 1-' clears\n"
  									"    all the searches listed by the last `list' command\n"},
  { "results",&MUITermPriv::com_results,"Displays search results",
  									"results [pattern | search_ID(s)]\n"
  									"    Displays results of the search. Searches can be selected by either\n"
  									"    pattern in the same way as for `list' command, or numerical search ID(s)" },
  { "get"    ,&MUITermPriv::com_get,    "Get files from the net",
  									"get <result_ID(s)>\n"
  									"    Initiates download of the file(s) for given result ID(s). Result ID(s)\n"
  									"    should reference search result list as it was produced by the last\n"
  									"    `results' command. IDs can be given in fairly relaxed way. For example\n"
  									"    `get 10, 15, 22-25' will start download of the search results listed\n"
  									"    under numbers 10, 15, 21, 22, 23, 24, 25.\n"
  									"    The progress of download procedure can be examined with `info' command.\n"
  									"    Once download is started Mutella tries to get requested file forever,\n"
  									"    until the transfer is successful or stopped with the `stop' command.\n"
  									"    When download fails to start immediately Mutella initiates the search for\n"
  									"    alternative locations for the file. If host is not responding for last\n"
  									"    25 trials 'auto-get' search is added and download failure is reported.\n"
  									"    When the file appears on the Gnutella horizon Mutella automatically\n"
  									"    initiates the transfer. Auto-get searches are also created for all\n"
  									"    partial files found in the download directory on the client start" },
  { "stop"   ,&MUITermPriv::com_stop,   "Stops the transfer",
  									"stop <transfer_ID(s)>\n"
  									"    Stops transfers corresponding to the given numeric ID(s). Transfer IDs\n"
  									"    are to be taken from the last `info' command output\n"
  									"    NOTE that 'stop' doesn't remove partial file from the download directory\n"
  									"    use `kill' instead" },
  { "kill"   ,&MUITermPriv::com_kill,   "Same as stop, but erases partial file in case of download",
  									"kill <transfer_ID(s)>\n"
  									"    Stops transfers corresponding to the given numeric ID(s) and deletes\n"
  									"    appropriate patrial files in case of download. Transfer IDs are to be taken\n"
  									"    from the last `info' command output"},
  { "move"   ,&MUITermPriv::com_move,   "Modifies filename of the file being downloaded",
  									"move <download_ID> <new_file_name>\n"
  									"    Useful to remove 'SHARE-THIS' rubbish frim the file name so that auto-search\n"
  									"    feature produces more results and finished file look nicer. If the automatic\n"
  									"    search for alternative locations has already been created it will be modified\n"
  									"    SEE ALSO: 'edit'"},
  { "open"   ,&MUITermPriv::com_open,   "Opens new Gnutella-net connection",
  									"open <host> [port]\n"
  									"    Opens Gnutella-net connection to the specified host. When 'port'\n"
  									"    parameter is omitted 6346 is assumed" },
  { "close"  ,&MUITermPriv::com_close,  "Closes specified Gnutella connection(s)",
  									"close <connection_ID(s)>\n"
  									"    Closes connection(s) which correspond to the given numeric ID(s).\n"
  									"    Connection IDs are to be taken from the last `info' command output" },
  { "scan"   ,&MUITermPriv::com_scan,   "(Re)Scans shared directory", NULL },
  { "library",&MUITermPriv::com_library,"Lists currently shared files", NULL },
  { "load"   ,&MUITermPriv::com_load,   "Loads and executes Mutella terminal-mode script", NULL },
  { "set"    ,&MUITermPriv::com_set,    "Accesses Mutella options",
  									"set [Variable [new_value]]\n"
  									"    Used to access Mutella options. Without parameters displays the entire\n"
  									"    list of options. When called up with one parameter displays the value\n"
  									"    of the appropriate variable. When called with two parameters treats\n"
  									"    the first parameter as variable name and the second one as a new value\n"
  									"    To empty the string variable type `set <variable> \"\"'\n"
  									"    All mutella options are stored in `~/.mutella/mutellarc' file" },
  { "set+"   ,&MUITermPriv::com_set_add,"Adds a value to the list-type option",
  									"set+ [Variable [new_value]]\n"
  									"    Enables to add a value to the list. Without parameters displays all the\n"
  									"    list-type options. When called with one parameter displays the value\n"
  									"    of the appropriate variable. When called with two parameters treats\n"
  									"    the first parameter as variable name and the second as a value to be\n"
  									"    added to the list" },
  { "set-"   ,&MUITermPriv::com_set_remove,"Removes a value from the list-type option",
  									"set- [Variable [value]]\n"
  									"    Enables to remove a given value from the list. Without parameters displays\n"
  									"    all the list-type options. When called with one parameter displays the value\n"
  									"    of the appropriate variable. When called with two parameters treats\n"
  									"    the first parameter as variable name and the second as a value to be\n"
  									"    removed from the list" },
  { "color"  ,&MUITermPriv::com_color,  "Sets terminal colors",
  									"color [field [new_ASCI_code]]\n"
  									"    Used to set colors for terminal UI. Syntax is similar to `set'. The\n"
  									"    color scheme is stored in the location ponted by ColorScheme variable\n"
  									"    (default is `~/.mutella/termclr')" },
  { "system" ,&MUITermPriv::com_system, "Executes a shell command given by argument", NULL },
  { "!"      ,&MUITermPriv::com_system, "synonym for 'system'", NULL },
  { "version",&MUITermPriv::com_version,"Displays program version", NULL },

  { NULL, NULL, NULL, NULL }
};

MUITermPriv::VarEntry MUITermPriv::variables[] = {
  { "GWebCache1",			"GWebCache server #1 (host cache) URL",
  							"  GWebCacheX is used to initiate the connection to the Gnutella-net\n"
  							"  Alternatively it will be used when Mutella looses the connection to the\n"
  							"  net for whatever reason and host cache is empty. Accepts standard\n"
  							"  HTTP URL format\n"},
  { "GWebCache2",			"GWebCache server #2. See help on `GWebCache1' for detail", NULL},
  { "GWebCache3",			"GWebCache server #3. See help on `GWebCache1' for detail", NULL},
  { "BandwidthConnects",	"Bandwidth allocated for Guntella-net connections",
  							"  Amount of network bandwidth (in Kbytes per second) allowed to spend for\n"
  							"  Gnutella-net related traffic. Because of the nature of this traffic Mutella\n"
  							"  is unable to follow this limitation exactly. The value given here will only\n"
  							"  approximately limit the connection bandwidth"},
  { "BandwidthTotal",		"Bandwidth allocated for all non-ours traffic",
  							"  Limits amount of the bandwidth (in Kbytes per second) given for both uploads\n"
  							"  and Gnutella-net. Works well in combination with BandwidthConnects, when the\n"
  							"  later is less or equal to 50% of the BandwidthTotal. Actual upload traffic is\n"
  							"  limited by the (BandwidthTotal - actual_connection_traffic)"},
  { "BandwidthTransfer",	"Bandwidth allocated for uploads",
  							"  Limits amount of the bandwidth (in Kbytes per second) allowed for uploads.\n"
  							"  When BandwidthTotal is set actual upload traffic is limited by the minimum\n"
  							"  between BandwidthTransfer and (BandwidthTotal - actual_connection_traffic)"},
  { "ConnectTimeout",		"Connection Timeout", NULL},
  { "DownloadPath",			"Path to the download directory", NULL},
  { "Firewall",				"Set to true if it is impossible to connect to your host", NULL },
  { "ForceIP",				"Your IP address reported to the Gnutella-net",
  							"  Set to your firewall IP if there is a static mapping for your Mutella"},
  { "ForcePort",			"Listening port reported to the Gnutella-net",
  							"  Normally combines with ForceIP. Should be set to the listening port on the\n"
  							"  firewall if there is static mapping for your Mutella"},
  { "DynDnsFirewall",		"Alternative to ForceIP when IP of the firewall is not static",
  							"  Set to the symbolic name of your firewall. The 'ForceIP' variable will be\n"
  							"  automatically updated each 10 minutes"},
  { "AcceptRemoteIpHeader",	"Whether the 'LocalIP' should be updated from RemoteIP replies",
  							"  Must work correctly in most cases. Only effective when ForceIP is set to\n"
  							"  0.0.0.0 and DynDnsFirewall is empty. Setting this option to 'true' enables\n"
  							"  to detect firewall IP automatically in a most robust way" },
  { "GroupFuzzy",			"Group search results by size, but allow some deviations in names",
  							"  Setting this to 'true' allows names to differ insignificantly, like to switch\n"
  							"  word possitions. When set to 'false' grouping require exact(no case) name match"},
  { "LocalIP",				"IP address of the local host",
  							"  The value is automatically defined on the client start and modified later\n"
  							"  once a connection to the Gnutella network is established (this is required\n"
  							"  for determination of the correct IP for box with multiple interfaces"},
  { "LocalPort",			"Listening port",
  							"  Modifying the value during client operation have no effect. Edit\n"
  							"  ~/.mutella/mutellarc file when client is stopped, or set LocalPort in Mutella\n"
  							"  and exit it. Upon the next client start Mutella will listen on the new port"},
  { "MaxConnPerSubnetA",	"Maximum number of the simultaneous connections per A-class subnet", NULL},
  { "MaxConnPerSubnetB",	"Maximum number of the simultaneous connections per B-class subnet", NULL},
  { "MaxConnPerSubnetC",	"Maximum number of the simultaneous connections per C-class subnet", NULL},
  { "MaxConnections",		"Maximum number of the Gnutella-net connections, excluding leaves", NULL},
  { "LeafModeConnects",		"Number of the Gnutella-net connections when in LEAF mode", NULL},
  { "MaxLeaves",			"Maximum mumber of leaves allowed to connect to our ULTRAPEER", NULL},
  { "MaxDownloads",			"Maximum number of the simulteneous downloads", NULL},
  { "MaxFileSize",			"Maximum file size (in K) of the search result",
  							"  Required for spam-protection. Default is very reasonable"},
  { "MaxPerHostDownloads",	"Maximum number of the simultaneous downloads from the same host", NULL},
  { "MaxPerHostUploads",	"Maximum number of the simultaneous uploads to the same host", NULL},
  { "MaxReplies",			"Maximum number of the replies for the incoming search request", NULL},
  { "MaxResults",			"Maximum number of results per search", NULL},
  { "MaxResultsDisplayed",	"Maximum number of results displayed by `results'", NULL},
  { "MaxSearches",			"Maximum number of the simultaneous searches", NULL},
  { "MaxUploads",			"Maximum number of the simultaneous uploads", NULL},
  { "MinConnections",		"Minimum number of connections (not effective in LEAF mode)",
  							"  When number of connections drops below this value Mutella starts to make\n"
  							"  outgoing connections. This is not true when QuietMode is set"},
  { "EnableUltrapeerMode",	"Enable participation in Ultrapeer election",
  							"  When set to 'true' enables mutella to become ulrapeer. In order"
  							"  to do so the client uptime should be substantial and the connection"
  							"  bandwidth should be hight enough (>32K/s)"},
  { "ForceUltrapeerMode",	"Enforce Ultrapeer mode",
  							"  When set to 'true' forces mutella to become ulrapeer, regardles of"
  							"  the election process or local bandwidth limits and so on. Only use"
  							"  this option if you are certain you know what you are doing"},
  { "MinDownloadSpeed",		"Minimum download speed in bytes per second", NULL},
  { "MinFileSize",			"Minimum file size (in K) of the search result",
  							"  Required for protection against millions of partial files people share"},
  { "MinFriends",			"Minimum number of the hosts known to the Gnutella-node", NULL},
  { "MinUploadSpeed",		"Minimum upload speed in bytes per second", NULL},
  { "Paginate",				"Stop between pages",
  							"  When set to `true' terminal output will stop each `TerminalLines' lines"},
  { "PushTimeout",			"Set the timeout for push requests.",
  							"  The timeout for push requests; after this period of time has passed,\n"
							"  the push request attempt is assumed to have failed."},
  { "QuietMode",			"Do not allow Mutella making outgoing connections",
  							"  This mode is useful when it is advisable to reduce amount of the outgoing\n"
  							"  connections, for example when running Mutella from the office computer. Has\n"
  							"  effect on connections and push-uploads"},
  { "ReachableForPush",			"Specify whether your host is reachable for push"
							"  If this is set to false, results from private subnets will be filtered out.\n"
							"  If set to true, your host will attempt a push request to hosts which are on\n"
							"  private subnets." },
  { "ReplyFilePath",		"Add file path to the search reply",
  							"  Defines whether to add a file path to the search replies. Added path is\n"
  							"  relative to the shared directory"},
  { "ReplyIfAvail",			"Send search-replies only if upload slots available", NULL},
  { "ResubmitTime",			"Interval of re-submission for local searches", NULL},
  { "RetryDelay",			"Download retry delay", NULL},
  { "ScreenBusy",			"not functional", NULL},
  { "SearchScreenNodes",	"not functional", NULL},
  { "SharePath",			"Path to the shared directory",
  							"  Path to the shared directory. Changing it during Mutella session has no\n"
  							"  immediate effect. To actualise new shared path you need to issue `scan'\n"
  							"  command. To share files located on different file systems use symbolic links"},
  { "ShareFilter",			"File name filter used when scanning the shared directory",
  							"  Space separated list of file patterns like *.avi *.mp? *.mpeg and so on.\n"
  							"  Empty string means no filterding is done. Changing it during Mutella\n"
  							"  session has no immediate effect in the same way as for 'SharePath'. To\n"
  							"  actualise new filter in is necessary to issue `scan' command."},
  { "ShareDotFiles",		"Whether to share unix hidden files and/or folders", NULL},
  //{ "SpeedDynamic",			"",
  //							""},
  { "SpeedStatic",			"The speed of your connection in Kbps", NULL},
  //{ "SpeedTimeout",			"",
  //							""},
  { "StartupScript",		"The path to the start-up script", NULL},
  //{ "StrictSearch",			"",
  //							""},
  { "TerminalCols",			"Number of columns in the terminal",
  							"  Defined automatically at the client start. Non-persistent. If automatic\n"
  							"  detection fails you can set typical number of columns in the start-up\n"
  							"  script (~/.mutella/termrc)"},
  { "TerminalLines",		"Number of lines in the terminal",
  							"  Defined automatically at the client start. Non-persistent. If automatic\n"
  							"  detection fails you can set typical number of lines in the start-up\n"
  							"  script (~/.mutella/termrc)"},
  //{ "TransferTimeout",		"",
  //							""},
  //{ "",	"",
  //							""},
  { "SaveSearches",			"Defines whether to save searches between sessions", NULL },

  { NULL, NULL, NULL }
};

/* Look up NAME as the name of a command, and return a pointer to that
   command.  Return a NULL pointer if NAME isn't a command name. */
MUITermPriv::ComEntry * MUITermPriv::find_command ( char *name )
{
	register int i;
	for (i = 0; commands[i].name; i++)
		if (strncmp (name, commands[i].name, strlen(name)) == 0)
		{
			if (strcmp (name, "e") == 0)
				return NULL;
			return (&commands[i]);
		}
	return NULL;
}

bool MUITermPriv::execute_line (const CString& sLine)
{
	register int i;
	ComEntry *command;
	char *word;
	int 	pipes [2],
		old_stdout,
		fpid = 0,
		flags;
	bool bOldPageBreak;
	char *args [4], *ptr;
	//
	char *line = (char*) alloca(sLine.size()+1);
	ASSERT(line);
	strncpy(line, sLine.c_str(), sLine.size());
	line[sLine.size()] = '\0';
	//
	/* Isolate the command word. */
	i = 0;
	while (line[i] && _isblank(line[i]))
		i++;
	word = line + i;

	while (line[i] && !_isblank(line[i]))
		i++;

	if (line[i])
		line[i++] = '\0';

	command = find_command (word);

	if (!command)
	{
		printf ("%s: No such command.\n", word);
		return false;
	}

	/* Get argument to command, if any. */
	while (_isblank(line[i]))
		i++;

	word = line + i;

	/* Pipe output to a system command */
	if ((ptr = strchr (word, '|'))) {
		*ptr = 0;
		if (pipe (pipes) == -1) {
			printf ("Failed to create a pipe\n");
			return false;
		}

		if ((fpid = fork ()) == -1) {
			printf ("Failed to create child process\n");
			return false;
		}	

		if (! fpid) {
			/* child */
			close (pipes [1]);

			/* redirect stdin from the pipe */
			if (dup2 (pipes [0], STDIN_FILENO) == -1) {
				printf ("Failed to redirect stdin\n");
				exit (1);
				return false;
			}

			args [0] = "sh";
			args [1] = "-c";
			args [2] = ++ptr;
			args [3] = 0;

			/* Set close_on_exec to 0 */
			flags = fcntl (STDIN_FILENO, F_GETFD, 0);
			flags &= ~FD_CLOEXEC;
			fcntl (STDIN_FILENO, F_SETFD, flags);

			if (execve ("/bin/sh", args, NULL) == -1) {
				printf ("execve() failed\n");
				exit (1);
			}
		} else {
			/* parent */
			close (pipes [0]);
			
			/* Backup stdout */
			if ((old_stdout = dup (STDOUT_FILENO)) == -1) {
				printf ("Failed to backup stdout\n");
				return false;
			}	

			/* Redirect stdout to the pipe */
			if (dup2 (pipes [1], STDOUT_FILENO) == -1) {
				printf ("Failed to redirect stdout\n");
				return false;
		 	}
		 	
			/* set max lines to avoid paging */
			bOldPageBreak = m_more.m_bBreak;
			m_more.m_bBreak = false;

			close (pipes [1]);
		}
	}
	
	/* Call the function. */
	bool bRes = ((*this).*(command->handler))(word);//(this->(*(command->handler)) (word));

	m_more.Flush();

	/* Wait for piped process to exit and restore stdout / term lines */
	if (fpid>0) {
		close (STDOUT_FILENO);
		waitpid (fpid, NULL, 0);
		if (dup2(old_stdout, STDOUT_FILENO)==-1)
			printf("Failed to set stdout back\n");
		close (old_stdout);
		m_more.m_bBreak = bOldPageBreak;
	}

	return bRes;
}


void MUITermPriv::ui_loop()
{
	char *line, *s;
	char tmp[1024];
	CString sLastCommand;
	// startup script
	if (strlen(m_szStartUpScript))
	{
		if (!FileExists(m_szStartUpScript))
		{
			printf("\nCreating default '%s' file\n", m_szStartUpScript);
			printf("Edit it to modify Mutella startup behaviour\n\n");
			FILE* f;
			if (f=fopen(ExpandPath(m_szStartUpScript).c_str(),"w"))
			{
				fprintf(f,"# mutella terminal-mode initialisation script\n");
				fprintf(f,"# edit it to make mutella smarter during startup\n");
				//fprintf(f,"\n# initiate a gnutella-net connection\n");
				//fprintf(f,"open public.bearshare.net\n");
				//fprintf(f,"open gnotella.fileflash.com\n");
				//fprintf(f,"open router.limewire.com\n");
				fprintf(f,"\n# for example set terminal size\n");
				fprintf(f,"# (if automatic detection fails), or just do nothing\n");
				fprintf(f,"\nversion\n\n");

				fclose(f);
			}
		}
		if (FileExists(m_szStartUpScript))
		{
			strncpy(tmp,m_szStartUpScript,1024);
			tmp[1023]='\0';
			com_load(tmp);
		}
	}
	/* Loop reading and executing lines until the user quits. */
	while ( m_bContinue )
	{
		if (m_command_queue.size())
		{
			strncpy(tmp,m_command_queue.front().c_str(),1024);
			tmp[1023]='\0';
			m_command_queue.pop();
			//
			execute_line(tmp);
		}
		else
		{
			if (m_evCache.HaveEvents())
			{
				if (m_evCache.MissedEvents())
				{
					m_more.print("??error>You have missed %d events\n", m_evCache.MissedEvents());
					m_evCache.ResetMissedEvents();
				}
				m_evCache.LockFront();
				while (m_evCache.HaveEvents())
				{
					MEvent* p = m_evCache.UnsafeFront();
					ASSERT(p);
					m_more.print("??header>Event %x:??\n"
					             "  received at: ??time>%s??\n"
#ifdef _DEBUG
					             "  type: %d\n  severity: %d\n  id: %d\n"
#endif
					             ,
					             p,
					             FormatAbsTime(p->GetTime()).c_str()
#ifdef _DEBUG
					             ,
					             p->GetType(), p->GetSeverity(), p->GetID()
#endif
					             );
					if (p->GetID())
						m_more.print("  ??message>%s??\n"
						             "  ??message>%s??\n",
						             GetMessageString(p->GetID()).c_str(),
						             p->Format().c_str());
					else
						m_more.print("  ??message>%s??\n", p->Format().c_str());
					//
					m_evCache.Pop();
				}
				m_evCache.UnlockFront();
				m_more.Flush();
			}
			// Read line of input,
			// remove leading and trailing whitespace from the line.
			CString sLine = StripWhite( m_input.InputLine("> ") );
			if (sLine.empty())
				continue;
			// Then, if there is anything left, execute it.
			execute_line(sLine);
		}
	}
}

MUITerminal::MUITerminal()
{
	m_pPriv = NULL;
}

MUITerminal::~MUITerminal()
{
	delete m_pPriv;
}

bool MUITerminal::Attach(MController* pC)
{
	
	m_pPriv = new MUITermPriv(pC);
	ASSERT(m_pPriv);
	return m_pPriv->attach();
}

bool MUITerminal::Init()
{
	if (m_pPriv)
		return m_pPriv->init();
	return false;
}

void MUITerminal::Detach()
{
	if (m_pPriv)
		m_pPriv->detach();
}

void MUITerminal::Do()
{
	if (m_pPriv)
		m_pPriv->ui_loop();
}

void MUITerminal::Stop()
{
	if (m_pPriv)
		m_pPriv->stop();
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////
// couple of useful parsers
//

bool ParseNumRanges(intSet& seq, char* str, int nMax /*for n- combination*/)
{
	bool bAfterDash = false;
	int n = -1;
	int m = -1;
	int len = strlen(str);
	char* p = NULL;
	for( int i = 0; i<=len; ++i )
	{
		if (str[i]>='0' && str[i]<='9')
		{
			// it is a digit -- set the pointer to it if it is the first one
			if (!p)
				p=str+i;
		}
		else if ( str[i]==',' || str[i]==';' || str[i]=='-' || str[i]==' ' || str[i]=='\t' || str[i]=='\0' )
		{
			// achieved space or separator or range symbol (dash);
			if (p)
				// if we have the pointer to the begining of the number
				if (bAfterDash)
				{
					// there was a dash before -- treat as a range
					if ( str[i]=='-' )
						// nn-nn-
						return false;
					str[i] = '\0';
					m = atoi(p);
					p = NULL;
					if (n<0 && m<n)
						return false;
					if (m>nMax)
						m = nMax;
					for(n++;n<=m;++n)
						seq.insert(n);
					bAfterDash = false;
				}
			 	else
				{
					// just a number -- count it, and if current char is not a separator --
					// leave n untouched for the case of a dash
					bool bComa = ( str[i]==',' || str[i]==';' );
					bAfterDash = ( str[i]=='-' );
					str[i] = '\0';
					n = atoi(p);
					p = NULL;
					seq.insert(n);
					if (bComa)
						n = -1;
				}
			else
			{
				if ( str[i]=='-' )
				{
					if ( n<0 || bAfterDash )
						//   -n or -- situation
						return false;
					bAfterDash = true;
				}
				else if ( str[i]==',' || str[i]==';' )
				{
					if (bAfterDash)
					// n-, situation
					{
						if (n<0)
							return false;
						for (n++;n<=nMax;++n)
							seq.insert(n);
						bAfterDash = false;
					}
					n = -1;
				}
			}
		}
		else
			// unexpected symbol
			return false;
	}
	if ( bAfterDash )
	{
		// n-  situation
		if (n<0)
			return false;
		for (n++;n<=nMax;++n)
			seq.insert(n);
	}
	// sort and compress would be needed by the list, set is already sorted and
	//seq.sort();
	//seq.unique();
	return true;		
}

// not quite parser, but still... useful
static CString FormatSearch(SGnuSearch* pSearch)
{
	CString res = pSearch->m_Search;
	for (set<CString>::iterator it = pSearch->m_setSha1.begin(); it != pSearch->m_setSha1.end(); ++it)
	{
		if (!res.empty())
			res += ' ';
		res += "sha1:";
		res += *it;
	}
	return res;
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////
// command implementation
//

bool MUITermPriv::com_help ( char *arg )
{
	register int i;
	int printed = 0;
	if ( !arg || 0==strlen(arg) )
	{
		m_more.print("??header>Available comands:\n");
		m_more.print("??header>==================\n");
		for (i = 0; commands[i].name; i++)
		{
			m_more.print ("??name>%s\t??%s\n", commands[i].name, commands[i].doc_short);
		}
		m_more.print("??message>\nAll commands accept abbrevations, e.g. `results' can be replaced\n"
		             "with `res' or even `r'\n");
		m_more.print("??flag1>NEW: ??message>All commands support Unix pipes, that is output of the mutella\n"
		             "command can be redirected to the system command using standard Unix\n"
		             "pipleline syntax, e.g. `res 1-3 | grep something'\n");
		m_more.print("??message>Type `help <command>' to get more detailed help on specific command\n");
		m_more.print("??flag1>NEW: ??message>type `help <variable_name>' to get description of the specific\n"
		             "variable. List of mutella variables (options) can be acquired with\n"
		             "`set' command with no parameters\n");
		return true;
	}
	if (m_pController->GetProperty(arg))
	{
		for (i = 0; variables[i].name; i++)
		{
			if ( strcmp(arg, variables[i].name) == 0 )
			{
				if (variables[i].doc_long)
					m_more.print ("\n%s:	%s\n\n%s\n", arg, variables[i].doc_short, variables[i].doc_long);
				else
					m_more.print ("\n%s:	%s\n", arg, variables[i].doc_short);
				printed++;
			}
		}
		if (!printed)
		{
			m_more.print ("\n??message>Sorry, variable `%s' is not documented yet\n", arg);
			printed++;
		}
	}
	else
	{
		for (i = 0; commands[i].name; i++)
		{
			if ( strncmp (arg, commands[i].name, strlen(commands[i].name) ) == 0 )
			{
				if (commands[i].doc_long)
					m_more.print ("\n%s\n", commands[i].doc_long);
				else
					m_more.print ("\n%s\n    %s.\n", commands[i].name, commands[i].doc_short);
				printed++;
			}
		}
	}
	if (!printed)
	{
		if ( arg && strlen(arg) )
			m_more.print("??message>No commands match `%s'.  Possibilties are:\n", arg);
		for (i = 0; commands[i].name; i++)
		{
			m_more.print ("%s\t", commands[i].name);
			// Print in six columns.
			if ( i%6 == 5)
				m_more.print ("\n");
		}
		if ( i%6 != 0)
			m_more.print("\n");
		m_more.print("\n??message>List of mutella variables can be acquired with `set' command\n");
	}
	return true;
}

bool MUITermPriv::com_exit ( char *arg )
{
	// TODO: ask for safe exit
	m_more.print("??message>bye\n");
	m_bContinue = false;
	m_pController->ForceStop();
	return true;
}

bool MUITermPriv::com_system(char * arg)
{
	return -1 != system(arg);
}

bool MUITermPriv::com_load(char * arg)
{
	CString s = ExpandPath(arg);
	FILE* f = fopen(s.c_str(), "r");
	if (f == NULL)
	{
		m_more.print("??message>failed to open `??name>%s??message>'\n", s.c_str());
		return false;
	}
	char tmp[1024];
	char * t;
	while (!feof(f) && !ferror(f))
	{
		if (NULL!=fgets(tmp,1024,f))
		{
			tmp[1023] = '\0';
			t = StripWhite(tmp);
			if (strlen(t) && *t != '#')
				m_command_queue.push(t);
		}
	}
	fclose(f);
	return true;
}

bool parse_num_param(CString& s, int& num, int nStatrAt = 0)
{
	// find the firs 'token' -- something isolated by the spaces
	int nBegin = nStatrAt;
	while (_isblank(s[nBegin]) && nBegin<s.length())
		++nBegin;
	if (nBegin == s.length())
		return false;
	int nEnd = nBegin+1;
	while (!_isblank(s[nEnd]) && nEnd<s.length())
		nEnd++;
	char* tmp = (char*) alloca(nEnd-nBegin+2);
	ASSERT(tmp);
	strncpy(tmp, s.substr(nBegin,nEnd-nBegin).c_str(), nEnd-nBegin+1);
	if (asc2num( tmp, &num))
	{
		s = s.substr(0, nStatrAt) + s.substr(nEnd);
		return true;
	}
	return false;	
}

bool MUITermPriv::com_find(char * arg)
{
	if ( arg==NULL || strlen(arg)==0 )
		return com_list("");
	CString search = StripWhite(arg);
	// check for options
	int nSizeMode = LIMIT_NONE;
	int nSize = 0;
	int nPos = 0;
	// size filters
	if ( (nPos=search.find("min:")) >= 0 )
	{
		if (!parse_num_param(search, nSize, nPos+4))
		{
			m_more.print("??error>find: syntax error after 'min:'\n");
			return false;
		}
		nSizeMode = LIMIT_MORE;
		search = search.substr(0,nPos)+search.substr(nPos+4);
	}
	else if ( (nPos=search.find("size:")) >= 0 )
	{
		if (!parse_num_param(search, nSize, nPos+5))
		{
			m_more.print("??error>find: syntax error after 'size:'\n");
			return false;
		}
		nSizeMode = LIMIT_EXACTLY;
		search = search.substr(0,nPos)+search.substr(nPos+5);
	}
	else if ( (nPos=search.find("around:")) >= 0 )
	{
		if (!parse_num_param(search, nSize, nPos+7))
		{
			m_more.print("??error>find: syntax error after 'around:'\n");
			return false;
		}
		nSizeMode = LIMIT_APPROX;
		search = search.substr(0,nPos)+search.substr(nPos+7);
	}
	// autoget flag "/autoget"
	bool bAutoGet = false;
	if ( (nPos=search.find("/autoget")) >= 0 )
	{
		bAutoGet = true;
		search = search.substr(0,nPos)+search.substr(nPos+8);
	}
	else if ( (nPos=search.find("/auto")) >= 0 )
	{
		bAutoGet = true;
		search = search.substr(0,nPos)+search.substr(nPos+5);
	}
	// add a query to the search list
	if (m_pController->AddSearchUsr(search.c_str(), nSize, nSizeMode, bAutoGet))
		return true;
	m_more.print("??error>adding search `??name>%s??error>' failed.\nprobably there are similar searches already \nor maximum number of searches has been reached.\n", search.c_str());
	return false;
}

bool MUITermPriv::com_edit(char * arg)
{
	CString sArg = StripWhite(arg);
	// check if we have 2 parameters and extract the first one
	int nPos = sArg.find(' ');
	if (nPos<0)
	{
		m_more.print("??error>'edit' requires 2 parameters\n");
		return false;
	}
	
	DWORD dwID = 0;
	int nSNum = atol(sArg.substr(0, nPos).c_str());
	if (nSNum>0 && nSNum<=m_vecSearchIDs.size())
		dwID = m_vecSearchIDs[nSNum-1];
	if (dwID==0)
	{
		m_more.print("??error>'edit' requires valid search ID\n");
		return false;
	}
	CString sSearch = StripWhite(sArg.substr(nPos+1));
	if (!sSearch.length() || !m_pController->ModifySearch(dwID, sSearch.c_str()))
	{
		m_more.print("??error>Editing search failed, may be the ID is invalid or the search string is too short?\n");
		return false;
	}
	//m_more.print("??message>Ok\n");
	return true;
}

struct resParams{
	MUITermPriv* pThis;
	CString  sFilter;
	int      nMaxResultsDisplayed;
	dwordSet seq;
	dwordSet retSet;
};

int compRes(const void* p1, const void* p2)
{
	Result* pr1 = *(Result**)p1;
	Result* pr2 = *(Result**)p2;
	// sord decending by size, than desending by speed, than accending by name
	if (pr1->Size>pr2->Size)
		return -1;
	if (pr1->Size<pr2->Size)
		return 1;
	if (pr1->Speed>pr2->Speed)
		return -1;
	if (pr1->Speed<pr2->Speed)
		return 1;
	if (pr1->Name>pr2->Name)
		return 1;
	if (pr1->Name<pr2->Name)
		return -1;
	return 0;
}

bool resCallback(void* pData, SGnuSearch* pSearch)
{
	resParams* pRP = (resParams*) pData;
	if ( (pSearch->m_nHits != 0 && pRP->sFilter.length()!=0 && QueryMatch(pSearch->m_Search, pRP->sFilter)) ||
		 (pSearch->m_nHits != 0 && pRP->sFilter.length()==0 && pRP->seq.size()==0) ||
		 (pRP->seq.end() != pRP->seq.find(pSearch->m_dwID)) )
	{
		pRP->retSet.insert(pSearch->m_dwID);
	}
	return true;
}

int compGroups(const void* p1, const void* p2)
{
	ResultGroup* pg1 = *(ResultGroup**)p1;
	ResultGroup* pg2 = *(ResultGroup**)p2;
	// sort decending by size, than desending by speed, than acending by name
	if (pg1->Size>pg2->Size)
		return -1;
	if (pg1->Size<pg2->Size)
		return 1;
	if (pg1->AvgSpeed>pg2->AvgSpeed)
		return -1;
	if (pg1->AvgSpeed<pg2->AvgSpeed)
		return 1;
	if (pg1->Name>pg2->Name)
		return 1;
	if (pg1->Name<pg2->Name)
		return -1;
	return 0;
}

bool MUITermPriv::com_results(char * arg)
{
	resParams rp;
	rp.pThis = this;
	if (strlen(arg)==0 ||
	    strpbrk(arg, "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvqxyz") )
	{
		// use string filter approach
		rp.sFilter = arg;
	}
	else
	{
		// try numbersint
		intSet is;
		if (!ParseNumRanges(is, arg, m_vecSearchIDs.size()))
		{
			m_more.print("??error>results: failed to parse parameters\n");
			return false;
		}
		// transform numbers into IDs
		for (intSet::iterator it=is.begin();it!=is.end();++it)
		{
			if (*it<=m_vecSearchIDs.size())
				rp.seq.insert(m_vecSearchIDs[(*it)-1]);
			else
			{
				m_more.print("??error>results: no searches with ID = %d\n", *it);
			}
		}
	}
	rp.nMaxResultsDisplayed = 1000;
	MProperty* pMaxRes = m_pController->GetProperty("MaxResultsDisplayed");
	if (pMaxRes)
		rp.nMaxResultsDisplayed = pMaxRes->GetIntValue();
	m_pController->ForEachSearch((void*)&rp, resCallback);
	//
	m_nCounter = 1;
	m_vecResIDs.clear();
	m_more.print("Current search results, matching your criteria\n");
	for (dwordSet::iterator it = rp.retSet.begin(); it!=rp.retSet.end(); ++it)
	{
		if (rp.nMaxResultsDisplayed < m_nCounter)
			break;
		//
		SGnuSearch gs;
		vector<Result> rv;
		vector<ResultGroup> gv;
		if (!m_pController->GetSearchByID(*it,gs,rv,gv))
			continue;
		//
		m_more.print("  ??header>QUERY:??\"??name>%s??header>\" \t HITS:??num>%d\n", FormatSearch(&gs).c_str(), gs.m_nHits);
		if (m_bShowGroups)
		{
			// now copy result groups to our array so that we could sort it
			int nCount = gv.size();
			ResultGroup** arrpRGps = new ResultGroup*[nCount];
			ASSERT(arrpRGps);
			for (int i = 0; i<nCount; ++i)
			{
				arrpRGps[i]= &gv[i];
			}
			// we dont need a quick sort for a list of 10-20 results
			// but we have it so we use it
			qsort(arrpRGps, nCount, sizeof(Result*),compGroups); // TODO: different ways of sorting
			for (int i=0; i<nCount; i++)
			{
				ResultGroup* pgrp=arrpRGps[i];
				m_vecResIDs.push_back(pgrp->dwID);
				if ( pgrp->ResultSet.size() > 1 )
				{
					m_more.print("??id>%3d)?? `??name>%s??' ??size>%s??\n    LOCATIONS:??num>%-3d??  avg.speed:??speed>%s\n",
						m_vecResIDs.size(), pgrp->Name.c_str(), FormatSize(pgrp->Size).c_str(),
						pgrp->ResultSet.size(), GetSpeedString(pgrp->AvgSpeed).c_str());
				}
				else if (pgrp->ResultSet.size() == 1)
				{
					Result* pres= &rv[*pgrp->ResultSet.begin()];
					m_more.print("??id>%3d)?? `??name>%s??' ??size>%s?? REF:??num>%d\n",
						m_vecResIDs.size(), pgrp->Name.c_str(), FormatSize(pres->Size).c_str(), pres->FileIndex );
				}
				if (pgrp->ResultSet.size()<4)
				{
					int j = 0;
					for ( set<DWORD>::iterator iti = pgrp->ResultSet.begin(); iti != pgrp->ResultSet.end(); ++iti)
					{
						Result* pres= &rv[*iti];
						if (pres->Sha1.size())
							m_more.print("     sha1: ??sha1>%s\n", pres->Sha1.c_str());
						if (pres->Extra.size())
							m_more.print("    extra: ??extra>%s\n", pres->Extra.c_str());
						m_more.print("    ??ip>%16s:%-4d?? speed:??speed>%-12s ??flag1>%-12s?? time:??time>%s\n",
							Ip2Str(pres->Host).c_str(), pres->Port, GetSpeedString(pres->Speed).c_str(), pres->Vendor.c_str(),
							FormatTime(xtime() - pres->ChangeTime).c_str() );
						if (++j>2) break;
					}
				}
			}
			delete [] arrpRGps;
		}
		else
		{
			// now copy references to our array so that we could sort it
			int nCount = rv.size();
			Result** arrpRes = new Result*[nCount];
			ASSERT(arrpRes);
			for (int i=0; i<nCount; i++)
			{
				arrpRes[i]=&rv[i];
			}
			// we dont need a quick sort for a list of 10-20 results
			// but we have it so we use it
			qsort(arrpRes, nCount, sizeof(Result*),compRes); // TODO: different ways of sorting
			for (int i=0; i<nCount; i++)
			{
				Result* pres=arrpRes[i];
				m_vecResIDs.push_back(pres->dwID);
				m_more.print(" ??id>%d)?? `??name>%s??' ??size>%s?? REF:??num>%d??\n    IP:??ip>%s:%d?? \tSPEED:??speed>%s \t??flag1>%s\n",
					m_vecResIDs.size(), pres->Name.c_str(), FormatSize(pres->Size).c_str(), pres->FileIndex,
					Ip2Str(pres->Host).c_str(), pres->Port, GetSpeedString(pres->Speed).c_str(), pres->Vendor.c_str());
			}
			delete [] arrpRes;
		}
	}
	m_more.print("count: ??num>%d\n", m_nCounter-1);
	return true;
}

struct clrParams{
	MUITermPriv* pThis;
	CString  sFilter;
	int      nMaxResultsDisplayed;
	dwordSet seq;
	dwordSet retSet;
};


bool clearCallback(void* pData, SGnuSearch* pSearch)
{
	clrParams* pRP = (clrParams*) pData;
	if (pRP->nMaxResultsDisplayed < pRP->pThis->m_nCounter)
		return false;
	if ( (pRP->sFilter.length()!=0 && QueryMatch(pSearch->m_Search, pRP->sFilter)) ||
		 (pRP->sFilter.length()==0 && pRP->seq.size()==0) ||
		 (pRP->seq.end() != pRP->seq.find(pSearch->m_dwID)) )
	{
		pRP->pThis->m_more.print("\"??name>%s??\"\n", pSearch->m_Search.c_str());
		pRP->retSet.insert(pSearch->m_dwID);
	}
	return true;
}

bool MUITermPriv::com_clear(char * arg)
{
	if (strlen(arg)==0)
	{
		m_more.print("??error>clear: argument is required\n");
		return false;
	}
	clrParams cp;
	cp.pThis = this;
	if (strpbrk(arg, "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvqxyz"))
	{
		// use string filter approach
		cp.sFilter = arg;
	}
	else
	{
		intSet seq;
		// try numbersint
		if (!ParseNumRanges(seq, arg, m_vecSearchIDs.size()))
		{
			m_more.print("??error>clear: failed to parse parameters\n");
			return false;
		}
		// convert to IDs
		for(intSet::iterator it = seq.begin(); it != seq.end(); it++)
		{
			if (*it <= m_vecSearchIDs.size())
				cp.seq.insert(m_vecSearchIDs[(*it)-1]);
		}
	}
	cp.nMaxResultsDisplayed = 1000;
	MProperty* pMaxRes = m_pController->GetProperty("MaxResultsDisplayed");
	if (pMaxRes)
		cp.nMaxResultsDisplayed = pMaxRes->GetIntValue();
	cp.retSet.clear();
	m_nCounter = 1;
	m_more.print("??message>Clearing search results of following searches\n");
	m_pController->ForEachSearch((void*)&cp, clearCallback);
	if (!cp.retSet.empty())
	{
		//pSearch->Clear();
		for (dwordSet::iterator it = cp.retSet.begin();it != cp.retSet.end(); ++it)
			m_pController->ClearSearchByID(*it);
	}
	return true;
}

struct listParams{
	MUITermPriv* pThis;
	CString sFilter;
	bool bNoEmpty;
	bool bNoAutoSearches;
};

bool listCallback(void* pData, SGnuSearch* pSearch)
{
	listParams* pLP = (listParams*) pData;
	if ( (pLP->sFilter.length()==0 || QueryMatch(pSearch->m_Search, pLP->sFilter)) &&
		 (!pLP->bNoEmpty || pSearch->m_nHits) &&
		 (!pLP->bNoAutoSearches || !pSearch->IsAutomatic()) )
	{
		pLP->pThis->m_vecSearchIDs.push_back(pSearch->m_dwID);
		ASSERT(pLP->pThis->m_vecSearchIDs.size()==pLP->pThis->m_nCounter);
		//
		pLP->pThis->m_more.print("??id>%2d) ??name>`%s'\n    ", pLP->pThis->m_nCounter, FormatSearch(pSearch).c_str());
		if (pSearch->m_SizeFilterMode == LIMIT_NONE)
			pLP->pThis->m_more.print("??flag1>no size filter ");
		else if (pSearch->m_SizeFilterMode == LIMIT_MORE)
			pLP->pThis->m_more.print("MIN:??size>%-11s",FormatSize(pSearch->m_SizeFilterValue).c_str());
		else if (pSearch->m_SizeFilterMode == LIMIT_EXACTLY)
			pLP->pThis->m_more.print("SIZE:??size>%-10d",pSearch->m_SizeFilterValue);
		else if (pSearch->m_SizeFilterMode == LIMIT_LESS)
			pLP->pThis->m_more.print("MAX:??size>%-11s",FormatSize(pSearch->m_SizeFilterValue).c_str());
		else if (pSearch->m_SizeFilterMode == LIMIT_APPROX)
			pLP->pThis->m_more.print("AROUND:??size>%-11s",FormatSize(pSearch->m_SizeFilterValue).c_str());
		if (pSearch->m_bAutoget)
			pLP->pThis->m_more.print("??flag2>  AUTOGET");
		else
			pLP->pThis->m_more.print("         ");
		if (pSearch->m_nHits)
			pLP->pThis->m_more.print("  HITS:??num>%d\n", pSearch->m_nHits);
		else
			pLP->pThis->m_more.print("??flag1>  NO HITS\n");
		pLP->pThis->m_nCounter++;
	}
	return true;
}

bool MUITermPriv::com_list(char * arg)
{
	m_nCounter = 1;
	listParams lp;
	lp.pThis = this;
	lp.sFilter = arg;
	lp.bNoEmpty = false;
	lp.bNoAutoSearches = false;
	// switches
	MakeLower(lp.sFilter);
	lp.sFilter = " " + lp.sFilter + " ";
	int n;
	n = lp.sFilter.find(" -empty ");
	if (0<=n)
	{
		lp.sFilter = lp.sFilter.substr(0, n)+ lp.sFilter.substr(n+7);
		lp.bNoEmpty = true;
	}
	n = lp.sFilter.find(" -auto ");
	if (0<=n)
	{
		lp.sFilter = lp.sFilter.substr(0, n)+ lp.sFilter.substr(n+6);
		lp.bNoAutoSearches = true;
	}
	lp.sFilter = StripWhite(lp.sFilter);
	//
	m_vecSearchIDs.clear();
	//
	m_more.print("??message>Searches in progress, matching your criteria\n");
	m_pController->ForEachSearch((void*)&lp, listCallback);
	m_more.print("??message>count: ??num>%d\n", m_nCounter-1);
	return true;
}

bool MUITermPriv::com_ls(char * arg)
{
	if (NULL!=strstr(arg, "-empty"))
		return com_list(arg);
	char* tmp = (char*) alloca(strlen(arg)+16); // to lazy to count letters
	ASSERT(tmp);
	sprintf(tmp, "%s -empty", arg);
	return com_list(tmp);
}

bool MUITermPriv::com_delete(char * arg)
{
	intSet seq;
	if (!ParseNumRanges(seq, arg, m_vecSearchIDs.size()))
	{
		m_more.print("??error>delete: failed to parse parameters\n");
		return false;
	}
	for(intSet::iterator it = seq.begin(); it != seq.end(); it++)
	{
		if (*it<=m_vecSearchIDs.size())
			m_pController->RemoveSearchByID(m_vecSearchIDs[(*it)-1], false);
	}
	return true;
}

bool MUITermPriv::com_erase(char * arg)
{
	intSet seq;
	if (!ParseNumRanges(seq, arg, m_vecSearchIDs.size()))
	{
		m_more.print("??error>erase: failed to parse parameters\n");
		return false;
	}
	for(intSet::iterator it = seq.begin(); it != seq.end(); it++)
	{
		if (*it<=m_vecSearchIDs.size())
			m_pController->RemoveSearchByID(m_vecSearchIDs[(*it)-1], true);
	}
	return true;
}

static const char* FormatPeerMode(SGnuNode* pNode)
{
	if (pNode->m_nStatus != SOCK_CONNECTED)
		return "-";
	switch (pNode->m_nPeerMode) {
		case CM_NORMAL:    return "N";
		case CM_LEAF:      return "L";
		case CM_ULTRAPEER: return "U";
	}
	return "?";
}

bool infoConnCallback(void* pData, SGnuNode* pNode)
{
	MUITermPriv* pThis = (MUITermPriv*) pData;
	pThis->m_vecConnIDs.push_back(pNode->m_dwID);
	ASSERT(pThis->m_vecConnIDs.size()==pThis->m_nCounter);
	pThis->m_more.print("??id>%2d)?? ??ip>%18s:%-5d??  ??status>%s??  ??speed>%5s??:??speed>%-5s?? ??num>%5s??/??num>%-6s?? ??percent>%3d%%??   ??time>%6s??   ??time>%6s\n",
			pThis->m_nCounter, pNode->m_sHost.c_str(), pNode->m_nPort,
			FormatPeerMode(pNode),
			FormatSize((int)pNode->m_dStatisticsRate[NR_BYTE_IN]).c_str(),FormatSize((int)pNode->m_dStatisticsRate[NR_BYTE_OUT]).c_str(),
			FormatNumber(pNode->m_dwFriendsTotal).c_str(),
			FormatKSizeLL(pNode->m_llLibraryTotal).c_str(),
			pNode->m_nEfficiency,
			FormatTime(xtime()-pNode->m_nPeerUptime).c_str(),
			FormatTime(xtime()-pNode->m_nUptime).c_str());
	//m_more.print("      version 0.%d\n",pNode->m_dwVersion);
	pThis->m_nCounter++;
	return true;
}

bool infoUploadCallback(void* pData, SGnuUpload* pUpload)
{
	MUITermPriv* pThis = (MUITermPriv*) pData;
	pThis->m_vecTransIDs.push_back(pUpload->m_dwID);
	ASSERT(pThis->m_vecTransIDs.size()==pThis->m_nCounter);
	pThis->m_more.print("??id>%2d)??name> %s\n    ??ip>%16s:%-4d??  ??percent>%5s  ??speed>%5s/s  ",
		pThis->m_nCounter,
		InsertElypsis(pUpload->m_sFileName, 75).c_str(),
		Ip2Str(pUpload->m_ipHost).c_str(), pUpload->m_nPort,
		FormatPercent(pUpload->m_nBytesCompleted,pUpload->m_nFileLength).c_str(),
		FormatSize((int)pUpload->m_dRate).c_str());
	if (pUpload->m_dRate > 0)
		pThis->m_more.print("ETA:??time>%s  ",
			FormatTime((int) ((pUpload->m_nFileLength-pUpload->m_nBytesCompleted)/pUpload->m_dRate)).c_str());
	pThis->m_more.print("??status>%s\n",
		SGnuUpload::GetErrorString(pUpload->m_nError));
	pThis->m_nCounter++;
	return true;
}

/*struct SGnuDownloadItem
{
	// Node info
	IP		  ipHost;
	WORD      nPort;
	CString	  sVendor;
	// Download info
	int       nStatus;
	int       nLastStatusChange;
	int       nLastActive;
	int       nDisconnectReason;
	DWORD     dwRate;
	//
	bool      bServerIsUp;   // true if we recieved anything from it ever
	int       nConErrors;    // counter of consequitive connection errors
	int       nPushTimeouts; // counter of consequitive push timeouts
};

struct SGnuDownload
{
	// Download Properties
	CString m_sName;
	DWORD   m_dwSearchID;
	// File info
	DWORD   m_dwFileLength;
	DWORD   m_dwBytesCompleted;
	// source addresses, etc
	double m_dRate;          // real download rate in bytes per second
	//
	int    m_nActiveConnections;
	int    m_nCreationTime;
	// status
	int    m_nLastDisconnectReason;
	int    m_nStatus;
	// ID for UI
	DWORD  m_dwID;
	// Download items -- individual hosts
	DownloadItemVec m_vecItems;
	// few static methods
	static LPCSTR GetErrorString(int nCode);
	static LPCSTR GetStatusString(int nStatus);
};*/


bool infoDownloadCallback(void* pData, SGnuDownload* pDownload)
{
	MUITermPriv* pThis = (MUITermPriv*) pData;
	pThis->m_vecTransIDs.push_back(pDownload->m_dwID);
	ASSERT(pThis->m_vecTransIDs.size()==pThis->m_nCounter);
	pThis->m_more.print("??id>%2d)??name> %s\n", pThis->m_nCounter, InsertElypsis(pDownload->m_sName,75).c_str());
	if (pDownload->m_nStatus != TRANSFER_CLOSED && pDownload->m_nStatus != TRANSFER_CLOSING)
	{
		pThis->m_more.print("    ??status>%-7s  A:??num>%2d/%-2d ",
				SGnuDownload::GetStatusString(pDownload->m_nStatus),
				pDownload->m_vecItems.size(),
				pDownload->m_nBadResults);
		pThis->m_more.print("??size>%5s/%-5s  ??percent>%5s  ",
				FormatSize(pDownload->m_dwBytesCompleted).c_str(),
				FormatSize(pDownload->m_dwFileLength).c_str(),
				FormatPercent(pDownload->m_dwBytesCompleted,pDownload->m_dwFileLength).c_str());
		if (pDownload->m_dRate > 0)
			pThis->m_more.print("??speed>%5s/s?? ETA:??time>%s\n",
					FormatSize((int)pDownload->m_dRate).c_str(),
					FormatTime((int) ((pDownload->m_dwFileLength-pDownload->m_dwBytesCompleted)/pDownload->m_dRate)).c_str());
		else
			pThis->m_more.print("\n");
			/*pThis->m_more.print("??status>%s\n",
					SGnuDownload::GetErrorString(pDownload->m_nLastDisconnectReason));*/
		//
		if (pDownload->m_vecItems.size())
		{
		  for (int i=0; i<min((int)pDownload->m_vecItems.size(),3); ++i)
		  {
			pThis->m_more.print("    ??status>%-7s ", SGnuDownload::GetStatusString(pDownload->m_vecItems[i].nStatus));
			pThis->m_more.print("??ip>%16s:%-5d??  %-10s  ??speed>%5s/s??  ConErr:%-2d  PushTO:%-2d\n",
					Ip2Str(pDownload->m_vecItems[i].ipHost).c_str(),
					pDownload->m_vecItems[i].nPort,
					pDownload->m_vecItems[i].bServerIsUp ? "ServerIsUp" : "NoReplyYet",
					FormatSize(pDownload->m_vecItems[i].dwRate).c_str(),
					pDownload->m_vecItems[i].nConErrors,
					pDownload->m_vecItems[i].nPushTimeouts
					);
		  }
		}
	}
	else
		pThis->m_more.print("    INACTIVE , reason: ??status>%s\n", SGnuDownload::GetErrorString(pDownload->m_nLastDisconnectReason));
	
	pThis->m_nCounter++;
	return true;
}

bool MUITermPriv::com_info(char * arg)
{
	bool bNet=true, bConn=true, bUpl=true, bDownl=true, bRealtime = false;
	// parse arg here
	if (arg && strlen(arg))
	{
		bNet=bConn=bUpl=bDownl=false; // start from reseting everything
		CString sArg = StripWhite(arg);
		CString sPar;
		int nPos;
		sArg += " ";// to make parsing simplier
		while (sArg.length()>1) /* this is because of the space at the tail */
		{
			nPos = sArg.find(" ");
			sPar = sArg.substr(0,nPos);
			sArg = StripWhite(sArg.substr(nPos+1));
			sArg += " ";// ah, yeah im to lazy
			//
			if (0==strncmp("network", sPar.c_str(), sPar.length()))
				bNet = true;
			if (0==strncmp("connections", sPar.c_str(), sPar.length()))
				bConn = true;
			if (0==strncmp("uploads", sPar.c_str(), sPar.length()))
				bUpl = true;
			if (0==strncmp("downloads", sPar.c_str(), sPar.length()))
				bDownl = true;
			if (0==strncmp("transfers", sPar.c_str(), sPar.length()))
			{
				bUpl = true;
				bDownl = true;
			}
			if (0==strncmp("loop", sPar.c_str(), sPar.length()))
				bRealtime = true;
		}
		// here we have a little problem: if 'i r' was specified this will
		// reset all the flags -- just fix it manually now
		if (bRealtime && !bNet && !bConn && !bUpl &&!bDownl)
			bNet=bConn=bUpl=bDownl=true;
	}
	//
	do {
		if (bNet)
		{
			m_more.print("??header>host mode: ??name>%s??  ??header>uptime: ??time>%s\n\n",
					m_pController->IsUltrapeer() ? "ULTRAPEER" : "LEAF",
					FormatTime(xtime() - m_pController->GetClientStartTime()).c_str() );
			m_more.print("??header>network horizon:\n");
			m_more.print("??header>----------------\n");
			int nHosts, nSharingHosts, nFiles, nSize;
			m_pController->GetNetStats(nHosts, nSharingHosts, nFiles, nSize);
			//m_more.print("%d %d %d\n",nHosts, nFiles, nSize);
			m_more.print("ReachebleHosts: ??num>%s??  SharingHosts: ??num>%s??  Files: ??num>%s??  Capacity:??num> %s\n\n",
					FormatSize(nHosts).c_str(),
					FormatSize(nSharingHosts).c_str(),
					FormatSize(nFiles).c_str(),
					FormatMSize(nSize).c_str());

			m_more.print("??header>caches:\n");
			m_more.print("??header>-------\n");
			int nCacher, nUltraCatcher, nStore, nGWeb;
			m_pController->GetCacheStats(nCacher, nUltraCatcher, nStore, nGWeb);
			m_more.print("CatchedHosts: ??num>%d??  UltraPeers: ??num>%d??  PersistentHosts: ??num>%d??  WebCacheURLs: ??num>%d\n\n",nCacher, nUltraCatcher, nStore, nGWeb);

			m_more.print("??header>sockets:\n");
			m_more.print("??header>--------\n");
			int nConn, nUpl, nDownlA, nDownlT;
			m_pController->GetConnStats(nConn, nUpl, nDownlA, nDownlT);
			m_more.print("Network: ??num>%d??  Uploads: ??num>%d??  Downloads: ??num>%d?? / ??num>%d\n\n",nConn, nUpl, nDownlA, nDownlT);
			m_more.print("??header>bandwidth:\n");
			m_more.print("??header>----------\n");
			m_pController->GetBandwidthStats(nConn, nUpl, nDownlA);	
			m_more.print("Network:??speed>%s/s??  Uploads:??speed>%s/s??  Downloads:??speed>%s/s??  Total:??speed>%s/s??\n\n",
					FormatSize(nConn).c_str(),
					FormatSize(nUpl).c_str(),
					FormatSize(nDownlA).c_str(),
					FormatSize(nConn+nUpl+nDownlA).c_str());
		}
		//
		if (bConn)
		{
			m_more.print("??header>gnutella network connections:\n");
			m_more.print("??header>-----------------------------\n");
			m_more.print("??header>no.??|           ??header>address??:??header>port?? | ??header>T?? |  ??header>rate i:o??  | ??header>horizon??  | ??header>eff.?? | ??header>uptime?? | ??header>c.time??\n");
			m_nCounter = 1;
			m_vecConnIDs.clear();
			m_pController->ForEachConnection((void*)this, infoConnCallback);
			m_more.print("total connections: ??num>%d??  rate: [??speed>%5s??|??speed>%-5s??]/sec\n\n",
				   m_nCounter-1,
				   FormatSize(m_pController->GetConnRateRecv()).c_str(),
				   FormatSize(m_pController->GetConnRateSend()).c_str());
		}
		//
		m_nCounter = 1;
		if (bUpl)
		{
			m_more.print("??header>uploads:\n");
			m_more.print("??header>--------\n");
			m_vecTransIDs.clear();
			m_pController->ForEachUpload((void*)this, infoUploadCallback);
		}
		//
		if (bDownl)
		{
			m_more.print("??header>downloads:\n");
			m_more.print("??header>----------\n");
			if (!bUpl)
				m_vecTransIDs.clear();
			m_pController->ForEachDownload((void*)this, infoDownloadCallback);
		}
		if (bUpl && bDownl)
			m_more.print("total transfers: ??num>%d\n", m_nCounter-1);
		else if (bUpl)
			m_more.print("total uploads: ??num>%d\n", m_nCounter-1);
		else if (bDownl)
			m_more.print("total downloads: ??num>%d\n", m_nCounter-1);
		if (bRealtime)
		{
			bool bOldPageBreak = m_more.m_bBreak;
			m_more.m_bBreak = false;
			m_more.print("--hit any key to stop looping--");
			m_more.Flush();
			fflush(stdout);
			m_more.m_bBreak = bOldPageBreak;
			if (read_key(2))
				bRealtime = false;
			printf("\r                               \r");
		}
	} while (bRealtime);
	return true;
}

bool MUITermPriv::com_hosts(char * arg)
{
	list<Node> hosts;
	m_pController->GetNodeCache(hosts);
	if (hosts.size())
	{
		m_more.print("??header>discovered hosts\n");
		m_more.print("??header>----------------\n");
		for (list<Node>::iterator it = hosts.begin(); it != hosts.end(); ++it)
		{
			m_more.print("??ip>%16s:%-4d ??num>%6d  ??size>%5s  ??speed>%s\n",
				Ip2Str(it->Host).c_str(), it->Port,
				it->ShareCount,
				FormatKSize(it->ShareSize).c_str(),
				GetSpeedString(it->Speed).c_str());
		}
		m_more.print("total: ??num>%d\n", hosts.size());
	}
	else
	{
		m_more.print("??message>hosts cache is empty\n");
	}
}

bool MUITermPriv::com_close(char * arg)
{
	intSet seq;
	int nConn, nUpl, nDownlA, nDownlT;
	m_pController->GetConnStats(nConn, nUpl, nDownlA, nDownlT);
	if (!ParseNumRanges(seq, arg, nConn))
	{
		m_more.print("??error>close: failed to parse parameters\n");
		return false;
	}
	for(intSet::iterator it = seq.begin(); it != seq.end(); it++)
	{
		if (m_vecConnIDs.size()<*it)
			break;
		m_pController->CloseConnectionByID(m_vecConnIDs[(*it)-1]);
	}
	return true;
}

bool MUITermPriv::stop_help(LPCSTR szCom,bool bDelPart, char * arg)
{
	intSet seq;
	int nConn, nUpl, nDownlA, nDownlT;
	m_pController->GetConnStats(nConn, nUpl, nDownlA, nDownlT);
	if (!ParseNumRanges(seq, arg, nUpl+nDownlT))
	{
		m_more.print("??error>%s: failed to parse parameters\n", szCom);
		return false;
	}
	for(intSet::iterator it = seq.begin(); it != seq.end(); it++)
	{
		if (m_vecTransIDs.size()<*it)
			break;
		if ((*it)>0)
			m_pController->RemoveTransferByID(m_vecTransIDs[(*it)-1], bDelPart);
	}
	return true;
}

bool MUITermPriv::com_stop(char * arg)
{
	return stop_help("stop", false, arg);
}

bool MUITermPriv::com_kill(char * arg)
{
	return stop_help("kill", true, arg);
}

bool MUITermPriv::com_open(char * arg)
{
	int nPort = 6346;
	arg = StripWhite ( arg );
	char* pPort = strchr(arg,':');
	if (pPort == NULL)
		pPort = strchr(arg,' ');
	if ( pPort )
	{
		*pPort='\0';
		nPort = atoi(pPort+1);
		if (nPort<1024)
			nPort = 6346;
	}
	if (strlen(arg))
	{
		m_more.print("??message>opening connection to '??ip>%s??message>' port ??num>%d\n", arg, nPort);
		m_pController->OpenConnection(arg, nPort);
		return true;
	}
	return false;
}

bool MUITermPriv::com_set(char * arg)
{
	return set_helper(m_pController->GetPropertyContainer(), SET_SET, arg);
}

bool MUITermPriv::com_set_add(char * arg)
{
	return set_helper(m_pController->GetPropertyContainer(), SET_ADD, arg);
}

bool MUITermPriv::com_set_remove(char * arg)
{
	return set_helper(m_pController->GetPropertyContainer(), SET_REMOVE, arg);
}

bool MUITermPriv::com_color(char * arg)
{
	return set_helper(&m_more.m_propClr, SET_SET, arg);
}

bool MUITermPriv::set_helper(MPropertyContainer* pPC, SetMode mode, char * arg)
{
	//
	char tmp[1024];
	CString sArg = StripWhite(arg);
	sArg += ' ';
	int nSpacePos = sArg.find(" ");
	CString sPropName = sArg.substr(0,nSpacePos);
	CString sPropVal = StripWhite(sArg.substr(nSpacePos+1));
	if (sPropName.length())
	{
		MProperty* pP = pPC->FindProperty(sPropName.c_str());
		if (pP && (mode == SET_SET || pP->IsSet()))
		{
			if (sPropVal.length())
			{
				strncpy(tmp,sPropVal.c_str(),1024);
				switch (mode)
				{
					case SET_SET:    if (pP->SetFromStr(tmp)) return true;
						break;
					case SET_ADD:    if (pP->GetSet()->InsertStr(tmp)) return true;
						break;
					case SET_REMOVE: if (pP->GetSet()->RemoveStr(tmp)) return true;
						break;
				}
				m_more.print("??error>failed to parse `%s'\n", sPropVal.c_str());
				return false;
			}
			else
			{
		        m_more.print("??variable>%s?? = ??value>%s\n", pP->GetPropertyName(), pP->Format().c_str());
			}
			return true;
		}
		if (mode == SET_SET)
			m_more.print("??error>there is no `%s' variable\n", sPropName.c_str());
		else
			m_more.print("??error>there is no `%s' variable of type 'list'\n", sPropName.c_str());
		return false;
	}
	//
	set<CString> propset;
	MPropertyContainer::iterator it;
	for (it = pPC->begin(); it!= pPC->end(); it++)
	{
		if (mode == SET_SET || (pPC->GetProperty(it)->IsSet()))
			propset.insert(pPC->GetPropertyName(it));;
	}
	int s = propset.size();
	MPropertyContainer::sec_iterator its;
	for (its=pPC->sec_begin(); its!= pPC->sec_end(); its++)
	{
		MPropertySection* pS = pPC->GetSection(its);
		bool bSectionPrinted = false;
		for (it = pS->begin();it!=pS->end();it++)
		{
			MProperty* pP = pPC->GetProperty(it);
			if (propset.find(pPC->GetPropertyName(it))!=propset.end())
			{
				if (!bSectionPrinted)
				{
					m_more.print("[ %s ]\n",pPC->GetSectionName(its));
					bSectionPrinted = true;
				}
				m_more.print("??variable>%32s?? = ??value>%s\n", pPC->GetPropertyName(it), pP->Format().c_str());
				propset.erase(pPC->GetPropertyName(it));
				//cout << propset.size() << ") " << pPC->GetPropertyName(it) << endl;
			}
		}
	}
	if (propset.size())
	{
		if (pPC->HasSections())
			m_more.print("*** no section assigned ***\n");
		for (set<CString>::iterator itn = propset.begin(); itn!=propset.end();itn++)
		{
			MProperty* pP = pPC->FindProperty(itn->c_str());
			ASSERT(pP);
			if (pP)
			{
				m_more.print("??variable>%32s?? = ??value>%s\n", itn->c_str(), pP->Format().c_str());
			}
		}
	}
	return true;
}


struct getParams{
	MUITermPriv* pThis;
	intSet   seq;
};


bool MUITermPriv::com_get(char * arg)
{
	getParams gp;
	gp.pThis = this;
	
	if (!ParseNumRanges(gp.seq, arg, m_vecResIDs.size()))
	{
		m_more.print("??error>get: failed to parse parameters\n");
		return false;
	}
	if (gp.seq.empty())
	{
		m_more.print("??error>get requires numeric parameters\n");
		return false;
	}
	//
	ResultVec resVec;
	std::queue<int> toErase;
	for (intSet::iterator it=gp.seq.begin();it!=gp.seq.end(); ++it)
	{
		if (*it <= m_vecResIDs.size() && m_pController->GetResultsByID(m_vecResIDs[(*it)-1],resVec))
		{
			ASSERT(resVec.size());
			if (m_pController->AddDownload(resVec))
				m_more.print("??message>starting download of ??name>%s\n", resVec[0].Name.c_str());
			else
				m_more.print("??error>failed to start download of ??name>%s\n", resVec[0].Name.c_str());
			//
			resVec.clear();
			toErase.push(*it);
		}
	}
	while (toErase.size())
	{
		gp.seq.erase(toErase.front());
		toErase.pop();
	}	
	if (gp.seq.empty())
		return true;
	m_more.print("??error>no results with number(s)");
	for (intSet::iterator it=gp.seq.begin(); it!=gp.seq.end(); it++)
	{
		m_more.print(" ??error>%d", (*it));
	}
	m_more.print("\n");
	return false;
}

bool MUITermPriv::com_move(char * arg)
{
	CString sArg = StripWhite(arg);
	// check if we have 2 parameters and extract the first one
	int nPos = sArg.find(' ');
	if (nPos<0)
	{
		m_more.print("??error>'move' requires 2 parameters\n");
		return false;
	}
	
	DWORD dwID = 0;
	int nSNum = atol(sArg.substr(0, nPos).c_str());
	if (nSNum>0 && nSNum<=m_vecTransIDs.size())
		dwID = m_vecTransIDs[nSNum-1];
	if (dwID==0)
	{
		m_more.print("??error>'move' requires valid download ID\n");
		return false;
	}
	CString sFileName = StripWhite(sArg.substr(nPos+1));
	if (!sFileName.length() || !m_pController->RenameDownload(dwID, sFileName.c_str()))
	{
		m_more.print("??error>Renaming file failed, may be ID or file name is invalid?\n");
		return false;
	}
	//m_more.print("??message>Ok\n");
	return true;
}

bool MUITermPriv::com_scan(char * arg)
{
	m_pController->Rescan();
	return true;
}

bool MUITermPriv::com_library(char * arg)
{
	vector<SharedFile> SharedFiles;
	m_pController->GetSharedFiles(SharedFiles);
	if (SharedFiles.size()==0)
	{
		m_more.print("??message>no files shared\n");
		return true;
	}
	m_more.print("??header>shared files\n");
	m_more.print("??header>============\n");
	int nCounter = 1;
	for (vector<SharedFile>::iterator itFile = SharedFiles.begin(); itFile != SharedFiles.end(); itFile++, nCounter++)
	{
		if (itFile->Hidden)
			continue;
		m_more.print("??id>%4d)?? ??name>%-42s ??size>%5s??  hits:??num>%-5d?? uploads:??num>%-4d\n",nCounter,
			   InsertElypsis(itFile->Name, 42).c_str(),
			   FormatSize(itFile->Size).c_str(),
			   itFile->Matches,itFile->Uploads);
		if (!itFile->Sha1Hash.empty())
			m_more.print("      sha1: ??sha1>%s\n", itFile->Sha1Hash.c_str());
	}
	return true;
}

bool MUITermPriv::com_version(char * arg)
{
	m_more.print("??message>You are currently using Mutella v%s\n  ??flag1>Enjoy!\n", VERSION);
}

// command completion
MyLineInput::MyLineInput(MUITermPriv* pMaster, MController* pController) : m_pMaster(pMaster), m_pController(pController)
{
}

bool MyLineInput::CompleteWord(const CString& part, list<CString>& listCompletions)
{
	ASSERT(m_pMaster);
	ASSERT(m_pController);
	// parse the string to isolate the last word
	if (part[part.size()-1]==' ')
		return false; // sort of quick workaround for situation like command<SPACE>
	CString s = StripWhite(part);
	int nPos = s.rfind(" ");
	// look up the commands if its the first word
	if (nPos<0) // no spaces
	{
		for (int i = 0; MUITermPriv::commands[i].name; i++)
			if (strncmp (s.c_str(), MUITermPriv::commands[i].name, s.length()) == 0)
			{
				listCompletions.push_front(MUITermPriv::commands[i].name);
			}
		return listCompletions.size();
	}
	// otherwise try variables or colors if the first word is set or color command
	CString sCom = StripWhite(s.substr(0, nPos));
	if (sCom=="set" || sCom=="set+" || sCom=="set-" ||sCom=="help")
	{
		CString sPart = s.substr(nPos+1);
		// first check against commands if we have "help" command
		if (sCom=="help")
		{
			for (int i = 0; MUITermPriv::commands[i].name; i++)
				if (strncmp (sPart.c_str(), MUITermPriv::commands[i].name, sPart.length()) == 0)
				{
					listCompletions.push_front(MUITermPriv::commands[i].name);
				}
		}
		// now load properties
		MPropertyContainer* pPC = m_pController->GetPropertyContainer();
		MPropertyContainer::iterator it;
		for (it = pPC->begin(); it!= pPC->end(); ++it)
			if (strncmp (sPart.c_str(), pPC->GetPropertyName(it), sPart.length()) == 0)
			{
				if ( (sCom!="set+" && sCom!="set-") || pPC->GetProperty(it)->IsSet() )
					listCompletions.push_front(pPC->GetPropertyName(it));
			}
		return listCompletions.size();
	} // otherwise try variables or colors if the first word is set or color command
	else if (sCom=="color")
	{
		CString sVar = s.substr(nPos+1);
		//
		MPropertyContainer* pPC = &m_pMaster->m_more.m_propClr;
		MPropertyContainer::iterator it;
		for (it = pPC->begin(); it!= pPC->end(); ++it)
			if (strncmp (sVar.c_str(), pPC->GetPropertyName(it), sVar.length()) == 0)
			{
				listCompletions.push_front(pPC->GetPropertyName(it));
			}
		return listCompletions.size();
	}
	return false;
}

