/***************************************************************************
 Mutella - A commandline/HTTP client for the Gnutella filesharing network.

 This program is free software; you can redistribute it and/or
 modify it under the terms of the GNU General Public License
 as published by the Free Software Foundation; either version 2
 of the License, or (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 uiremote.cpp  -  User interface code for the HTTP UI.

    begin                : Thu Jan 31 2002
    copyright            : (C) 2002 by Max Zaitsev
    email                : maksik@gmx.co.uk
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "mutella.h"
#include "structures.h"

#include "mui.h"
#include "uiremote.h"
#include "controller.h"
#include "property.h"
#include "event.h"
#include "messages.h"

#include "asyncsocket.h"

#include "conversions.h"
#include "common.h"

#include <unistd.h>
#include <sys/mman.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

class MUIRTimer;
class MUIREventLog;
class MUIRSocket;
struct SLoopContext;
struct SListOptions;
struct SSearchRes;

// A date format for HTTP.
#define RFC1123FMT "%a, %d %b %Y %H:%M:%S GMT"

const char * HTTP404 =  "<HTML>\r\n"
						"<HEAD><TITLE>404 Not Found</TITLE></HEAD>\r\n"
						"<BODY>\r\n"
						"<H1>Not Found</H1>\r\n"
						"The requested file was not found on this server.\r\n"
						"Check if <b>RemoteTemplatePath</b> variable is set correctly.\r\n"
						"<HR NOSHADE SIZE=1>\r\n"
						"Powered by <I><B><A HREF=\"http://mutella.sourceforge.net\">Mutella</A></B></I>.\r\n"
						"</BODY>\r\n"
						"</HTML>\r\n\r\n";
						
const char * HTTP401 =  "<HTML>\r\n"
						"<HEAD><TITLE>401 Authorization Required</TITLE></HEAD>\r\n"
						"<BODY>\r\n"
						"<H1>Authorization Required</H1>\r\n"
						"The server requires athorizetion for the requested resource.\r\n"
						"<HR NOSHADE SIZE=1>\r\n"
						"Powered by <I><B><A HREF=\"http://mutella.sourceforge.net\">Mutella</A></B></I>.\r\n"
						"</BODY>\r\n"
						"</HTML>\r\n\r\n";

//////////////////////////////////////////////////////////
// define the class which will really do all the hard work
// we dont put it into the header to remove dependences

class MUIRemotePriv
{
public:
	MUIRemotePriv(MController* pC, MUIRemote* pOwner);
	~MUIRemotePriv();
	// required functionality -- copy of the interface functions
	bool attach();
	bool init();
	void detach();
	void ui_loop();
	void stop();
	bool on_http_request(const IP& ipRemoteHost, LPCSTR szPath, LPCSTR szHandshake, const char* pBuffRest, int nFD);
	//
	void on_timer(int nTimeStamp);
	// internal use members
	bool process_buff(const char * pBuf, int nBufLen, CString & sPocessedBuf, bool bProcessed, bool bOnlyTags, SLoopContext* pContext);
protected:
	// other methods
	bool on_command(const CString& sCommand, const map<CString, CString>& sArgMap, LPCSTR szHandshake, SLoopContext& Context);
	void send_file(const CString& sRelPath, LPCSTR szHandshake, int nFD, SLoopContext* pContext = NULL);
	bool process_html_buff(const char * pFileBuf, int nBufLen, CString & sPocessedBuf, SLoopContext* pContext = NULL);
	void process_loop(const char * pLoopBuf, int nLoopBufLen, const char * pLoopNoneBuf, int nLoopNoneBufLen, const char* szLoopName, CString & sPocessedBuf, bool bListedLoop, SLoopContext* pContext);
	bool parse_trail(const CString& trail, map<CString, CString>& values);
	void loopResults(const char * pLoopBuf, int nBufLen, CString & sPocessedBuf, bool bListedLoop, SLoopContext* pContext);
	void loopGroups(const char * pLoopBuf, int nBufLen, CString & sPocessedBuf, bool bListedLoop, SLoopContext* pContext);
	void loopItems(const char * pLoopBuf, int nBufLen, CString & sPocessedBuf, bool bListedLoop, SLoopContext* pContext);
	void loopEvents(const char * pLoopBuf, int nBufLen, CString & sPocessedBuf, bool bListedLoop, SLoopContext* pContext);
	//
	bool replaceFormParams(CString& sLine, bool bLineModified, SLoopContext* pContext);
	bool replaceContextTags(CString& sLine, bool bLineModified, int nTags, map<const char *, CString>& mapTagValues, SLoopContext* pContext);
	// log access
	void logAccess(int nTime, IP ipRemote, LPCSTR szMessage, LPCSTR szPath);
	// data fields
	MController*   m_pController;
	MUIRemote*     m_pOwner;
	MWaitCondition m_waitStop;
	MUIRTimer*     m_pTimer;
	MUIREventLog*  m_pEventLog;
	list<MUIRSocket*> m_listSockets;
	// properties
	bool  m_bEnableRemote;
	bool  m_bRemoteLogAccess;
	IPNET m_netAllowed;
	char  m_szRemoteDir[128];
	char  m_szRemoteTemplPath[1024];
	char  m_szRemoteUser[128];
	char  m_szRemotePass[128];
	char  m_szRemoteDefSort[128];
	int   m_nMaxResInGroup;
};

////////////////////////////////////
// MUIRemote only forwards the calls

MUIRemote::MUIRemote()
{
	m_pPriv = NULL;
}

MUIRemote::~MUIRemote()
{
	delete m_pPriv;
}

bool MUIRemote::Attach(MController* pC)
{
	m_pPriv = new MUIRemotePriv(pC, this);
	ASSERT(m_pPriv);
	return m_pPriv->attach();
}

bool MUIRemote::Init()
{
	if (m_pPriv)
		return m_pPriv->init();
	return false;
}

void MUIRemote::Detach()
{
	if (m_pPriv)
		m_pPriv->detach();
}

void MUIRemote::Do()
{
	if (m_pPriv)
		m_pPriv->ui_loop();
}

void MUIRemote::Stop()
{
	if (m_pPriv)
		m_pPriv->stop();
}

bool MUIRemote::OnHttpRequest(const IP& ipRemoteHost, LPCSTR szPath, LPCSTR szHandshake, const char* pBuffRest, int nFD)
{
	if (m_pPriv)
		return m_pPriv->on_http_request(ipRemoteHost, szPath, szHandshake, pBuffRest, nFD);
	return false;
}

////////////////////////////////////////////////////////////////////////
// MUIRemotePriv service classes declaration/implementation

class MUIRTimer : public MSyncEventReceiver {
public:
	MUIRTimer(MUIRemotePriv* pMaster) : m_pMaster(pMaster){}
protected:
	virtual bool IsOfInterest(MEvent* p){
		ASSERT(p);
		return p->GetID()==MSG_TIMER;
	}
	virtual void OnEvent(MEvent* p, bool){
		ASSERT(p);
		ASSERT(m_pMaster);
		m_pMaster->on_timer(((MIntEvent*)p)->GetValue());
	}
	//
	MUIRemotePriv* m_pMaster;
};

/////////////////////////////////////////////////////////
// service class to store events
// this class has nothing to do with HTTP access log
class MUIREventLog : public MAsyncEventReceiver {
public:
	MUIREventLog() : MAsyncEventReceiver(INT_MAX, 0x4000 /*16k*/) {}
protected:
	virtual bool IsOfInterest(MEvent* p){
		ASSERT(p);
		return p->GetSeverity()>=ES_JUSTINCASE;
	}
};

class MUIRSocket : public MAsyncSocket
{
public:
	MUIRSocket(int nCode, const char* pBuf, int nBufLen) :
			m_nBufLen(nBufLen),
			m_nBytesSent(0),
			m_bError(false)
	{
		m_nLastActive = xtime();
		m_pBuf = new char[nBufLen];
		ASSERT(m_pBuf);
		memcpy(m_pBuf, pBuf, nBufLen);
		// create a header
		m_sHeader = "HTTP/1.0 " + DWrdtoStr(nCode) + "\r\nServer: Mutella\r\n";
		if (nBufLen>6 && 0==strncasecmp(pBuf,"<HTML>",6))
			m_sHeader += "Content-type:text/html\r\n";
		else if (nBufLen>6 && 0==strncasecmp(pBuf,"<style",6))
			m_sHeader += "Content-type:text/css\r\n";
		else if (nBufLen>5 && 0==strncmp(pBuf,"GIF87",5))
			m_sHeader += "Content-type:image/gif\r\n";
		else if (nBufLen>5 && 0==strncmp(pBuf,"GIF89",5))
			m_sHeader += "Content-type:image/gif\r\n";
		else if (nBufLen>10 && 0==strncmp(pBuf+6,"JFIF",4))
			m_sHeader += "Content-type:image/jpeg\r\n";
		else if (nBufLen>4 && 0==strncmp(pBuf+1,"PNG",3))
			m_sHeader += "Content-type:image/png\r\n";
		else
			m_sHeader += "Content-type:binary\r\n";

		// datestamp
		char beforetimebuf[100];
		char nowtimebuf[100];
	 	char aftertimebuf[100];	
		time_t before = 0;
		time_t now = xtime(); 
		time_t after = now + (60 * 60 * 24 * 365);

		(void) strftime( beforetimebuf, sizeof(beforetimebuf),
			RFC1123FMT, gmtime( &before ) );
		(void) strftime( nowtimebuf, sizeof(nowtimebuf),
			RFC1123FMT, gmtime( &now ) );
		(void) strftime( aftertimebuf, sizeof(aftertimebuf),
			RFC1123FMT, gmtime( &after ) ); 

		m_sHeader += "Date: " + CString(nowtimebuf) + "\r\n";

		// Add dated headers.
		if (nBufLen>6 && 0==strncasecmp(pBuf,"<HTML>",6))
		{
			m_sHeader += CString("Last-Modified: ") + nowtimebuf;
			m_sHeader += "\r\n";
			m_sHeader += CString("Expires: ") + nowtimebuf;
			m_sHeader += "\r\n";
		}
		else
		{
			m_sHeader += CString("Last-Modified: ") + beforetimebuf;
			m_sHeader += "\r\n";
			m_sHeader += CString("Expires: ") + aftertimebuf;
			m_sHeader += "\r\n";
		}

		// authorisation
		if (nCode == 401)
			m_sHeader += "WWW-Authenticate: Basic realm=\"MutellaRemote\"\r\n";
		m_sHeader += "\r\n";
	}
	~MUIRSocket()
	{
		delete [] m_pBuf;
	}
	inline bool Closed(){return m_hSocket == INVALID_SOCKET;}
	inline bool Completed(){return m_nBytesSent>=m_sHeader.length()+m_nBufLen;}
	inline bool Error(){return m_bError;}
	inline int  LastActiveTime(){return m_nLastActive;}
	//
	virtual void OnSend(int nErrorCode){
		// if we have completed sending and reseived next "OnSend" it would be valid to assume
		// that the send buffer has been transmitted
		if (Completed())
		{
			//TRACE("Transmission via socket seems to be completed");
			//ModifySelectFlags(0, FD_WRITE); // to remove busy-looping
			Close();
		}
		m_nLastActive = xtime();
		int nSentNow = 0;
		if (m_sHeader.length()>m_nBytesSent)
			nSentNow = Send(m_sHeader.c_str()+m_nBytesSent, m_sHeader.length()-m_nBytesSent);
		else
			nSentNow = Send(m_pBuf+m_nBytesSent-m_sHeader.length(),m_nBufLen-m_nBytesSent+m_sHeader.length());
		if (nSentNow<0){
			m_bError = true;
			Close();
			return;
		}
		m_nBytesSent+=nSentNow;
	}
	//OnReceive{}
protected:
	int     m_nLastActive;
	int     m_nBytesSent;
	bool    m_bError;
	//
	char*   m_pBuf;
	int     m_nBufLen;
	//
	CString m_sHeader;
};

////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////
// MUIRemotePriv implementation -- all the functionality is defined here

MUIRemotePriv::MUIRemotePriv(MController* pC, MUIRemote* pOwner) : m_pController(pC), m_pOwner(pOwner)
{
	m_pTimer = new MUIRTimer(this);
	ASSERT(m_pTimer);
	m_pEventLog =  new MUIREventLog();
}

MUIRemotePriv::~MUIRemotePriv()
{
	delete m_pTimer;
	delete m_pEventLog;
}

bool MUIRemotePriv::attach()
{
	MProperty* pP;
	MPropertyContainer* pPC = m_pController->GetPropertyContainer();
	pPC->AddSection("RemoteUI");
	pPC->AddProperty("RemoteEnable",        &m_bEnableRemote,       false); // default is OFF for security reasons
	pPC->AddProperty("RemoteLogAccess",     &m_bRemoteLogAccess,     true);
	pPC->AddProperty("AllowedRemoteNet",    &m_netAllowed,          MakeIPNET(127,0,0,1,32)); // default is localhost 127.0.0.1/32
	pPC->AddProperty("RemoteDir",           m_szRemoteDir,          128,  "/remote"); // requres restart to have an effect
	pPC->AddProperty("RemoteTemplatePath",  m_szRemoteTemplPath,    1024, INSTALL_DATA_DIR "/" PACKAGE "/remote/template");
	pPC->AddProperty("RemoteUser",          m_szRemoteUser,         128,  "");
	pPC->AddProperty("RemotePass",          m_szRemotePass,         128,  "");
	pPC->AddProperty("RemoteDefSortOrder",  m_szRemoteDefSort,      128,  "size");
	pPC->AddProperty("RemoteMaxResInGrp",   &m_nMaxResInGroup,      4); // sets maximum number of the results listed for each group
	pPC->SetCurrentSection(NULL);
	return true;
}

bool MUIRemotePriv::init()
{
	VERIFY(m_pController->RegisterHttpFolder(m_pOwner, m_szRemoteDir));
	ED().AddReceiver(m_pTimer);
	ED().AddReceiver(m_pEventLog);
	return true;
}

void MUIRemotePriv::detach()
{
	ED().RemoveReceiver(m_pTimer);
	ED().RemoveReceiver(m_pEventLog);
	VERIFY(m_pController->UnregisterHttpFolder(m_pOwner, m_szRemoteDir));
}

void MUIRemotePriv::ui_loop()
{
	// more than simplistic implementation
	// may be later I will find some usefull
	// job for this thread
	m_waitStop.wait();
}

void MUIRemotePriv::stop()
{
	TRACE("MUIRemotePriv::ui_loop: STOP");
	m_waitStop.wakeAll();
}

// problem: we want OnTimer to come to us from the network thread
// broadcast event from the MGnuDirector::OnTimer
void MUIRemotePriv::on_timer(int nTimeStamp)
{
	// do updates, check for timeouts, etc
	// TODO: lock mutex here
	for (list<MUIRSocket*>::iterator it = m_listSockets.begin(); it != m_listSockets.end(); ++it)
	{
		//if ((*it)->Closed())
		//	cout << "on_timer: socket is closed\n";
		//if ((*it)->Error())
		//	cout << "on_timer: socket error\n";
		//if ((*it)->LastActiveTime()+60 < nTimeStamp)
		//	cout << "on_timer: socket is dead, last active time = " << (*it)->LastActiveTime() << " timestamp = " << nTimeStamp << endl;
		if ((*it)->Closed() ||
		    (*it)->Error()     ||
		    (*it)->LastActiveTime()+60 < nTimeStamp )
		{
			// close and remove this socket
			(*it)->Close();
			delete (*it);
			//TODO: lock the mutex
			m_listSockets.erase(it);
			// idonno safer way to restore the iterator
			it = m_listSockets.begin();
		}
	}
}

struct SLoopContext{
	MController*  pController;
	SGnuNode*     pNode;
	SGnuUpload*   pUpload;
	SGnuDownload* pDownload;
	SGnuDownloadItem*
	              pDownloadItem;
	SGnuSearch*   pSearch;
	DWORD         dwLoopCounter;
	SListOptions* pListOptions;
	CString*      psRoot;
	CString*      psUrl;
	CString*      psForm;
	CString*      psMessage;
	CString*      psStatus;
	SSearchRes*   pSearchRes;
	Result*       pResult;
	ResultGroup*  pResGroup;
	int           nResultsSortOrder;
	set<DWORD>*   pSetIDs;
	map<CString, CString>*
	              pMapFormParams;
	MEvent*       pEvent;
	// constructor
	SLoopContext(){
		// init with zeroes
		memset(this, 0, sizeof(SLoopContext));
	}
};

struct SListOptions{
	bool bNoEmpty;
	bool bNoAuto;
	CString sFilter;
};

enum ResultsSortOrder{
	RSO_NoSort = 0,
	RSO_Size   = 1,
	RSO_Name   = 2,
	RSO_Speed  = 3,
	RSO_IP     = 4
};

bool MUIRemotePriv::on_http_request(const IP& ipRemoteHost, LPCSTR szPath, LPCSTR szHandshake, const char* pBuffRest, int nFD)
{
	// check if we at al enabled
	if (!m_bEnableRemote)
		return false;
	// just a sanity check
	if (strstr(szPath, m_szRemoteDir)!=szPath)
		return false;
	// protect us from the crackers... somehow (./../.. will bring us to the home directory)
	// get rid of paths with spaces as well
	// WHAT THE HELL I DID TRY TO DO HERE?
	if (strstr(szPath,"..") ||  strchr(szPath,' '))
	if ( 
		szPath[0]=='/' ||   //starts with a backslash - http://server//
		strcmp(szPath,"..") == 0 ||   // contains ..
		strncmp(szPath,"../",3) == 0 || // reverse check.
		strstr(szPath, "/../" ) != (char*) 0 ||  // contains the string
		strcmp( &(szPath[strlen(szPath)-3]), "/.." ) == 0 // ends with
		) return false;
	// check for source IP
	SOCKADDR_IN sockAddr;
	memset(&sockAddr, 0, sizeof(sockAddr));
	socklen_t nAddLen = sizeof(sockAddr);
	if (0!=getpeername(nFD, (SOCKADDR*)&sockAddr, &nAddLen))
		return false;
	IP ipRemote;
	ipRemote.S_addr = sockAddr.sin_addr.s_addr;
	//TRACE2("request from ", Ip2Str(ipRemote));
	if (!IsIpInSubnet(ipRemote, m_netAllowed))
	{
		logAccess(xtime(), ipRemote, "IpIsNotPermitted", szPath);
		return false;
	}
	if (strcmp(m_szRemotePass, "no_password") != 0)
	{
		// login -- HTTP-Basic so far
		bool bLoginOK = false;
		char * pszAuth = NULL;
		if ( (pszAuth = strstr(szHandshake, "Authorization:")) )
		{
			// extract authorization line
			char * pRN = strstr(pszAuth, "\r\n");
			CString sAuthLine(pszAuth + strlen("Authorization:"), pRN-pszAuth-strlen("Authorization:"));
			int nPos = sAuthLine.find("Basic");
			if (nPos>=0)
			{
				CString sCred = Base64::decode(StripWhite(sAuthLine.substr(nPos + strlen("Basic")+1)));
				CString sShouldBe = CString(m_szRemoteUser) + ":" + m_szRemotePass;
				if (sShouldBe == sCred)
					bLoginOK = true;
			}
			if (m_bRemoteLogAccess && !bLoginOK)
			{
				// authorisation failed -- log this
				logAccess(xtime(), ipRemote, "AuthorisationFailed", szPath);
			}
		}
		if (!bLoginOK)
		{
			MUIRSocket* pSock = new MUIRSocket(401, HTTP401, strlen(HTTP401));
			ASSERT(pSock);
			// TODO: lock mutex
			m_listSockets.push_front(pSock);
			pSock->Attach(nFD, FD_READ | FD_WRITE | FD_CLOSE);
			return true;
		}
	}
	if (m_bRemoteLogAccess)
	{
		// authorisation succeeded -- log this
		logAccess(xtime(), ipRemote, "AccessGranted", szPath);
	}

	// Cookies
	map<CString, CString> mapFormValues;
	const char* pszCookie;
	if( (pszCookie = strstr(szHandshake, "Cookie:")) )
	{
		char * pRN = strstr(pszCookie, "\r\n");
		CString sCookieLine(pszCookie + strlen("Cookie:"), pRN-pszCookie-strlen("Cookie:"));
		// we will spil the line by "; " combination
		list<CString> Cookies;
		int nPos;
		if (0<split_str(sCookieLine,"; ",Cookies))
		{
			for (list<CString>::iterator itC = Cookies.begin(); itC != Cookies.end(); ++itC)
			{
				nPos = itC->find('=');
				if (nPos>0)
					mapFormValues[StripWhite(itC->substr(0,nPos))] = StripWhite(itC->substr(nPos+1));
			}
		}
	}
	//cout << szHandshake << endl;
	//cout << pBuffRest << endl;
	// isolate relative path and remove the trailing slashes, etc.
	CString sPathCmd = StripAnyOf(CString(szPath, strlen(m_szRemoteDir), INT_MAX),"/ \t");
	// now isolate "form-params"
	int nPos;
	CString sPath(sPathCmd);
	CString sForm;
	if (0<=(nPos=sPath.find('?')))
	{
		sForm=StripAnyOf(sPath.substr(nPos),"/ \t");
		sPath.cut(nPos);
	}
	if (sPath.empty())
		sPath = "index.html";
	//TRACE4("HTTP-request: path = '", sPath,"', form param = ", sForm);
	// first parse the form's parameters
	// if there is a "CMD" parameter -- perform the command
	// than set the "Context" object based on the parameters
	// and send the file
	CString sRoot = m_szRemoteDir;
	CString sUrl = sRoot + "/" + sPathCmd;
	SLoopContext Context;
	Context.psRoot = &sRoot;
	Context.psUrl = &sUrl;
	CString sMessage;
	CString sStatus;
	set<DWORD> setIDs;
	Context.pSetIDs = &setIDs;
	SListOptions lo;
	lo.bNoEmpty = false;
	lo.bNoAuto  = false;
	if (parse_trail(sForm, mapFormValues))
	{
		// make form parameters known to the context object
		Context.psForm = &sForm;
		// form set of IDs -- find everything that starts from 'id'
		DWORD dwID;
		map<CString, CString>::const_iterator itid = mapFormValues.lower_bound("id");
		while (itid != mapFormValues.end() && itid->first[0] == 'i' && itid->first[1] == 'd')
		{
			if ((dwID=atol(itid->second.c_str()))>0)
				setIDs.insert(dwID);
			++itid;
		}
		// process the command
		map<CString, CString>::iterator itc = mapFormValues.find("CMD");
		if (itc != mapFormValues.end())
		{
			Context.psMessage = &sMessage;
			Context.psStatus = &sStatus;
			if (!on_command(itc->second, mapFormValues, szHandshake, Context))
			{
				cout << "on command failed\n";
				#warning generate error page here
				return false;
			}
		}
		// recognize various oprions
		map<CString, CString>::iterator itop;
		// "Searches" loop options
		bool bSOpt = false;
		itop = mapFormValues.find("sNoEmpty");
		if (itop != mapFormValues.end())
		{
			lo.bNoEmpty = true;
			bSOpt = true;
		}
		itop = mapFormValues.find("sNoAuto");
		if (itop != mapFormValues.end())
		{
			lo.bNoAuto = true;
			bSOpt = true;
		}
		itop = mapFormValues.find("sFilter");
		if (itop != mapFormValues.end())
		{
			lo.sFilter = itop->second;
			bSOpt = true;
		}
		if (bSOpt)
			Context.pListOptions = &lo;
		// "Results" & "Groups" loops options
		//map<CString, CString> mapArgValues;
		CString sSortBy;
		itop = mapFormValues.find("rSortBy");
		if (itop!=mapFormValues.end())
			sSortBy = itop->second;
		else if (strlen(m_szRemoteDefSort))
		{
			mapFormValues["rSortBy"] = sSortBy = m_szRemoteDefSort;
		}
		if (sSortBy.length())
		{
			if (sSortBy=="size")
				Context.nResultsSortOrder = RSO_Size;
			else if (sSortBy=="name")
				Context.nResultsSortOrder = RSO_Name;
			else if (sSortBy=="speed")
				Context.nResultsSortOrder = RSO_Speed;
			else if (sSortBy=="ip")
				Context.nResultsSortOrder = RSO_IP;
		}
		
		// load the form params into the map of substitutions
		Context.pMapFormParams = &mapFormValues;
	}
	send_file(sPath, szHandshake, nFD, &Context);
	return true;
}

void MUIRemotePriv::send_file(const CString& sRelPath, LPCSTR szHandshake, int nFD, SLoopContext* pContext /*=NULL*/)
{
	// this function will be able to parse the file for <!--foreach--> statemenst, etc, but later
	// for now will just send a context of the file
	CString sDiskPath = ExpandPath(CString(m_szRemoteTemplPath)+"/"+sRelPath);
	if (!FileExists(sDiskPath))
	{
		sDiskPath += ".html";
		if (!FileExists(sDiskPath))
		{
			MUIRSocket* pSock = new MUIRSocket(404, HTTP404, strlen(HTTP404));
			ASSERT(pSock);
			// TODO: lock mutex
			m_listSockets.push_front(pSock);
			pSock->Attach(nFD, FD_READ | FD_WRITE | FD_CLOSE);
			return;
		}
	}
	//
	int hFile = open(sDiskPath.c_str(), O_RDONLY);
	ASSERT(hFile>=0);
	int nFileSize = lseek(hFile,0,SEEK_END);
	lseek(hFile,0,SEEK_SET);
	MUIRSocket* pSock;
	char * pFileBuf = (char *) mmap(NULL, nFileSize, PROT_READ, MAP_SHARED, hFile, 0);
	// here we add check if the file is the HTML one and parse and possibly modify its content
	CString sPocessedBuf;
	if (process_html_buff(pFileBuf, nFileSize, sPocessedBuf, pContext))
	{
		//send modified buffer
		pSock = new MUIRSocket(200, sPocessedBuf.c_str(), sPocessedBuf.length());
		ASSERT(pSock);
	}
	else
	{
		// send unmodified file
		pSock = new MUIRSocket(200, pFileBuf, nFileSize);
		ASSERT(pSock);
	}
	munmap(pFileBuf, nFileSize);
	::close(hFile);
	
	// TODO: lock mutex
	m_listSockets.push_front(pSock);
	pSock->Attach(nFD, FD_READ | FD_WRITE | FD_CLOSE);
}

struct SSearchRes{
	ResultGroupVec  vecGrp;
	ResultVec       vecRes;
};

#define CREATE_STATUS \
if (bOk && !bFail)\
{\
	*Context.psMessage += " succeeded";\
	*Context.psStatus = "Success";\
}\
else if (bOk && bFail)\
{\
	*Context.psMessage += " succeeded partly";\
	*Context.psStatus = "Partial success";\
}\
else if (!bOk && bFail)\
{\
	*Context.psMessage += " failed";\
	*Context.psStatus = "Failure";\
}\
else\
{\
	*Context.psMessage = CString("'") + sCommand + "' command requires ID list";\
	*Context.psStatus = "Error";\
}

bool MUIRemotePriv::on_command(const CString& sCommand, const map<CString, CString>& mapFormArgs, LPCSTR szHandshake, SLoopContext& Context)
{
	ASSERT(Context.psMessage);
	ASSERT(Context.psStatus);
	ASSERT(Context.pSetIDs);
	// clear the status buffers
	*Context.psMessage = "";
	*Context.psStatus = "";
	if (sCommand=="exit")
	{
		// a bit rough
		//::close(nFD);
		//
		stop();
		return false; // this is a bit rough
	}
	else if (sCommand=="close")
	{
		bool bOk = false;
		bool bFail = false;
		for (set<DWORD>::iterator it = Context.pSetIDs->begin(); it != Context.pSetIDs->end(); ++it)
		{
			if (m_pController->CloseConnectionByID(*it))
				bOk = true;
			else
				bFail = true;
		}
		*Context.psMessage = "closing connection(s)";
		CREATE_STATUS
		return true;
	}
	else if (sCommand=="stop")
	{
		bool bOk = false;
		bool bFail = false;
		for (set<DWORD>::iterator it = Context.pSetIDs->begin(); it != Context.pSetIDs->end(); ++it)
		{
			if (m_pController->RemoveTransferByID(*it, false))
				bOk = true;
			else
				bFail = true;
		}
		*Context.psMessage = "stopping transfer(s)";
		CREATE_STATUS
		return true;
	}
	else if (sCommand=="kill")
	{
		bool bOk = false;
		bool bFail = false;
		for (set<DWORD>::iterator it = Context.pSetIDs->begin(); it != Context.pSetIDs->end(); ++it)
		{
			if (m_pController->RemoveTransferByID(*it, true))
				bOk = true;
			else
				bFail = true;
		}
		*Context.psMessage = "killing transfer(s)";
		CREATE_STATUS
		return true;
	}
	else if (sCommand=="get")
	{
		bool bOk = false;
		bool bFail = false;
		for (set<DWORD>::iterator it = Context.pSetIDs->begin(); it != Context.pSetIDs->end(); ++it)
		{
			ResultVec rv;
			if (m_pController->GetResultsByID(*it, rv) && m_pController->AddDownload(rv))
			{
				bOk = true;
			}
			else
			{
				bFail = true;
			}
		}
		*Context.psMessage = "creating download(s)";
		CREATE_STATUS
		return true;
	}
	else if (sCommand=="delete" || sCommand=="erase")
	{
		bool bOk = false;
		bool bFail = false;
		for (set<DWORD>::iterator it = Context.pSetIDs->begin(); it != Context.pSetIDs->end(); ++it)
		{
			if (m_pController->RemoveSearchByID(*it, sCommand=="erase"))
				bOk = true;
			else
				bFail = true;
		}
		*Context.psMessage = (sCommand=="erase") ? "deleting search(es) and partial file(s)" : "deleting search(es)";
		CREATE_STATUS
		return true;
	}
	else if (sCommand=="clear")
	{
		bool bOk = false;
		bool bFail = false;
		for (set<DWORD>::iterator it = Context.pSetIDs->begin(); it != Context.pSetIDs->end(); ++it)
		{
			if (m_pController->ClearSearchByID(*it))
				bOk = true;
			else
				bFail = true;
		}
		*Context.psMessage = "clearing search(es)";
		CREATE_STATUS
		return true;
	}
	else if (sCommand=="find")
	{
		map<CString, CString>::const_iterator itQ = mapFormArgs.find("q");
		if (itQ==mapFormArgs.end() || itQ->second.empty())
		{
			*Context.psMessage = CString("'") + sCommand + "' command requires non-empty quiery text ('q' filed)";
			*Context.psStatus = "Error";
		}
		else
		{
			int nSizeFilter = 0;
			int nSizeFilterMode = 0;
			bool bAutoget = false;
			// size filter mode "sfm"
			map<CString, CString>::const_iterator itSFM = mapFormArgs.find("sfm");
			if (itSFM != mapFormArgs.end())
				nSizeFilterMode = atol(itSFM->second.c_str());
			if (nSizeFilterMode)
			{
				// size filter value "sf"
				map<CString, CString>::const_iterator itSF = mapFormArgs.find("sf");
				if (itSF != mapFormArgs.end())
					if (!asc2num(itSF->second, &nSizeFilter))
					{
						*Context.psMessage = "Cannot parse size filter parameters, adding search failed";
						*Context.psStatus = "Failure";
						return true;
					}
			}
			// autoget flag "ag"
			map<CString, CString>::const_iterator itAG = mapFormArgs.find("ag");
			if (itAG != mapFormArgs.end())
				bAutoget = (itAG->second == "on" || itAG->second == "ON");
			// add search
			if (m_pController->AddSearchUsr(itQ->second.c_str(), nSizeFilter, nSizeFilterMode, bAutoget))
			{
					*Context.psMessage = CString("Adding search '") + itQ->second + "' succeeded";
					*Context.psStatus = "Success";
			}
			else
			{
				*Context.psMessage = CString("Adding search '") + itQ->second + "' failed";
				*Context.psStatus = "Failure";
			}
		}
		return true;
	}
	else if (sCommand=="edit")
	{
		set<DWORD>::iterator it = Context.pSetIDs->begin();
		DWORD dwID = 0;
		if (it != Context.pSetIDs->end())
			dwID = *it;
		CString sSearch;
		map<CString, CString>::const_iterator itQ = mapFormArgs.find("nq");
		if (itQ != mapFormArgs.end())
			sSearch = itQ->second;
		if (dwID && sSearch.length()>3 && m_pController->ModifySearch(dwID, sSearch.c_str()))
		{
			*Context.psMessage = "Search string has been successfully modified";
			*Context.psStatus = "Success";
		}
		else
		{
			*Context.psMessage = "Failed to modify search string: wrong ID or invalid search string";
			*Context.psStatus = "Failure";
		}
		return true;
	}
	else if (sCommand=="move")
	{
		set<DWORD>::iterator it = Context.pSetIDs->begin();
		DWORD dwID = 0;
		if (it != Context.pSetIDs->end())
			dwID = *it;
		CString sFileName;
		map<CString, CString>::const_iterator itQ = mapFormArgs.find("nn");
		if (itQ != mapFormArgs.end())
			sFileName = itQ->second;
		if (dwID && sFileName.length()>3 && m_pController->RenameDownload(dwID, sFileName.c_str()))
		{
			*Context.psMessage = "Download has been successfully renamed";
			*Context.psStatus = "Success";
		}
		else
		{
			*Context.psMessage = "Failed to rename download: wrong ID or invalid new name";
			*Context.psStatus = "Failure";
		}
		return true;
	}
	return false;
}

enum TagResolv{
	TR_Globals,
	TR_Message,
	TR_GetHostMode,
	TR_GetHostUptime,
	TR_GetNetStats,
	TR_GetConnStats,
	TR_GetCacheStats,
	TR_GetBandwidthStats,
	TR_GetConnRate,
	TR_LoopCounter,
	TR_ForEachNode,
	TR_ForEachUpload,
	TR_ForEachDownload,
	TR_ForEachDownloadItem,
	TR_ForEachSearch,
	TR_Result,
	TR_ResultGroup,
	TR_Event
};

/////////////////////////////////////////////////////////////////
// parsing of the buffer with replacement of the tags
// and loop detection

#define TAG_REMOTEROOT  "#REMOTEROOT#"
#define TAG_CURRENTURL  "#CURRENTURL#"
#define TAG_CURRENTFORM "#CURRENTFORM#"
#define TAG_CURRENTTIME "#CURRENTTIME#"
#define TAG_MESSAGE     "#MESSAGE#"
#define TAG_STATUS      "#STATUS#"
#define TAG_NETHOSTS    "#NETHOSTS#"
#define TAG_NETSHOSTS   "#NETSHOSTS#"
#define TAG_NETFILES    "#NETFILES#"
#define TAG_NETSIZE     "#NETSIZE#"
#define TAG_NETSOCKS    "#NETSOCKS#"
#define TAG_HOSTMODE    "#HOSTMODE#"
#define TAG_HOSTUPTIME  "#HOSTUPTIME#"
#define TAG_UPLSOCKS    "#UPLSOCKS#"
#define TAG_DOWNLSOCKS  "#DOWNLSOCKS#"
#define TAG_DOWNLCOUNT  "#DOWNLCOUNT#"
#define TAG_TRANSSOCKS  "#TRANSSOCKS#"
#define TAG_NETBAND     "#NETBAND#"
#define TAG_UPLBAND     "#UPLBAND#"
#define TAG_DOWNLBAND   "#DOWNLBAND#"
#define TAG_TOTALBAND   "#TOTALBAND#"
#define TAG_NETBANDIN   "#NETBANDIN#"
#define TAG_NETBANDOUT  "#NETBANDOUT#"
#define TAG_CATCHERHOSTS "#CATCHERHOSTS#"
#define TAG_ULTRAHOSTS  "#ULTRAHOSTS#"
#define TAG_STOREHOSTS  "#STOREHOSTS#"
#define TAG_WEBCACHEURLS "#WEBCACHEURLS#"
#define TAG_LOOPCOUNTER "#LOOPCOUNTER#"
#define TAG_LOOPMOD2    "#LOOPMOD2#"
#define TAG_LOOPMOD3    "#LOOPMOD3#"
#define TAG_LOOPMOD4    "#LOOPMOD4#"
#define TAG_NODEID      "#NODEID#"
#define TAG_NODEIP      "#NODEIP#"
#define TAG_NODEPORT    "#NODEPORT#"
#define TAG_NODECLI     "#NODECLI#"
#define TAG_NODEBANDIN  "#NODEBANDIN#"
#define TAG_NODEBANDOUT "#NODEBANDOUT#"
#define TAG_NODEPACKIN  "#NODEPACKIN#"
#define TAG_NODEPACKOUT "#NODEPACKOUT#"
#define TAG_NODEREPLYIN "#NODEREPLYIN#"
#define TAG_NODEDROP    "#NODEDROP#"
#define TAG_NODEQUEUE   "#NODEQUEUE#"
#define TAG_NODEHOSTS   "#NODEHOSTS#"
#define TAG_NODESIZE    "#NODESIZE#"
#define TAG_NODETIME    "#NODETIME#"
#define TAG_NODESTIME   "#NODESTIME#"
#define TAG_NODEMODE    "#NODEMODE#"
#define TAG_NODEEFF     "#NODEEFF#"
#define TAG_UPLID       "#UPLID#"
#define TAG_UPLIP       "#UPLIP#"
#define TAG_UPLPORT     "#UPLPORT#"
#define TAG_UPLAGENT    "#UPLAGENT#"
#define TAG_UPLFILE     "#UPLFILE#"
#define TAG_UPLSIZE     "#UPLSIZE#"
#define TAG_UPLCOMPL    "#UPLCOMPL#"
#define TAG_UPLPERCENT  "#UPLPERCENT#"
#define TAG_UPLRATE     "#UPLRATE#"
#define TAG_UPLETA      "#UPLETA#"
#define TAG_UPLSTATUS   "#UPLSTATUS#"
#define TAG_DOWNLID     "#DOWNLID#"
#define TAG_DOWNLIP     "#DOWNLIP#"
#define TAG_DOWNLPORT   "#DOWNLPORT#"
#define TAG_DOWNLFILE   "#DOWNLFILE#"
#define TAG_DOWNLALTRN  "#DOWNLALTRN#"
#define TAG_DOWNLBADALTRN "#DOWNLBADALTRN#"
#define TAG_DOWNLSIZE   "#DOWNLSIZE#"
#define TAG_DOWNLCOMPL  "#DOWNLCOMPL#"
#define TAG_DOWNLPERCENT "#DOWNLPERCENT#"
#define TAG_DOWNLRATE   "#DOWNLRATE#"
#define TAG_DOWNLETA    "#DOWNLETA#"
#define TAG_DOWNLSTATUS "#DOWNLSTATUS#"
#define TAG_DOWNLEXTSTATUS "#DOWNLEXTSTATUS#"
#define TAG_ITEMIP      "#ITEMIP#"
#define TAG_ITEMPORT    "#ITEMPORT#"
#define TAG_ITEMISUP    "#ITEMISUP#"
#define TAG_ITEMRATE    "#ITEMRATE#"
#define TAG_ITEMCONNERR "#ITEMCONNERR#"
#define TAG_ITEMPUSHTO  "#ITEMPUSHTO#"
#define TAG_ITEMDISCREASON "#ITEMDISCREASON#"
#define TAG_ITEMSTATUS  "#ITEMSTATUS#"
#define TAG_SEARID      "#SEARID#"
#define TAG_SEARNAME    "#SEARNAME#"
#define TAG_SEARSTRING  "#SEARSTRING#"
#define TAG_SEARSIZEFLT "#SEARSIZEFLT#"
#define TAG_SEARAUTOGET "#SEARAUTOGET#"
#define TAG_SEARCANERASE "#SEARCANERASE#"
#define TAG_SEARHITS    "#SEARHITS#"
#define TAG_SEARGROUPS  "#SEARGROUPS#"
#define TAG_RESID       "#RESID#"
#define TAG_RESFILENAME "#RESFILENAME#"
#define TAG_RESFILEREF  "#RESFILEREF#"
#define TAG_RESFILESIZE "#RESFILESIZE#"
#define TAG_RESHOSTIP   "#RESHOSTIP#"
#define TAG_RESHOSTPORT "#RESHOSTPORT#"
#define TAG_RESSPEEDNUM "#RESSPEEDNUM#"
#define TAG_RESSPEED    "#RESSPEED#"
#define TAG_RESTIME     "#RESTIME#"
#define TAG_RESVENDOR   "#RESVENDOR#"
#define TAG_RESEXTRA    "#RESEXTRA#"
#define TAG_RESSHA1     "#RESSHA1#"
#define TAG_GRPID       "#GRPID#"
#define TAG_GRPFILENAME "#GRPFILENAME#"
#define TAG_GRPFILESIZE "#GRPFILESIZE#"
#define TAG_GRPLOCATIONS "#GRPLOCATIONS#"
#define TAG_GRPAVGSPEEDNUM "#GRPAVGSPEEDNUM#"
#define TAG_GRPAVGSPEED "#GRPAVGSPEED#"
#define TAG_EVTIME      "#EVTIME#"
#define TAG_EVRELTIME   "#EVRELTIME#"
#define TAG_EVTYPE      "#EVTYPE#"
#define TAG_EVSEVERITY  "#EVSEVERITY#"
#define TAG_EVID        "#EVID#"
#define TAG_EVMESSAGE   "#EVMESSAGE#"

struct STagEntry{
	char*     tag;
	TagResolv res;
};

STagEntry tags[] = {{TAG_REMOTEROOT,	TR_Globals},
					{TAG_CURRENTURL,	TR_Globals},
					{TAG_CURRENTFORM,	TR_Globals},
					{TAG_CURRENTTIME,	TR_Globals},
					{TAG_MESSAGE,		TR_Message},
					{TAG_STATUS,		TR_Message},
					{TAG_NETHOSTS,		TR_GetNetStats},
					{TAG_NETSHOSTS,		TR_GetNetStats},
					{TAG_NETFILES,		TR_GetNetStats},
					{TAG_HOSTMODE,		TR_GetHostMode},
					{TAG_HOSTUPTIME,	TR_GetHostUptime},
					{TAG_NETSIZE,		TR_GetConnStats},
					{TAG_NETSOCKS,		TR_GetConnStats},
					{TAG_UPLSOCKS,		TR_GetConnStats},
					{TAG_DOWNLSOCKS,	TR_GetConnStats},
					{TAG_DOWNLCOUNT,	TR_GetConnStats},
					{TAG_TRANSSOCKS,	TR_GetConnStats},
					{TAG_NETBAND,		TR_GetBandwidthStats},
					{TAG_UPLBAND,		TR_GetBandwidthStats},
					{TAG_DOWNLBAND,		TR_GetBandwidthStats},
					{TAG_TOTALBAND,		TR_GetBandwidthStats},
					{TAG_CATCHERHOSTS,	TR_GetCacheStats},
					{TAG_ULTRAHOSTS,	TR_GetCacheStats},
					{TAG_STOREHOSTS,	TR_GetCacheStats},
					{TAG_WEBCACHEURLS,	TR_GetCacheStats},
					{TAG_NETBANDIN,		TR_GetConnRate},
					{TAG_NETBANDOUT,	TR_GetConnRate},
					{TAG_LOOPCOUNTER,	TR_LoopCounter},
					{TAG_LOOPMOD2,		TR_LoopCounter},
					{TAG_LOOPMOD3,		TR_LoopCounter},
					{TAG_LOOPMOD4,		TR_LoopCounter},
					{TAG_NODEID,		TR_ForEachNode},
					{TAG_NODEIP,		TR_ForEachNode},
					{TAG_NODEPORT,		TR_ForEachNode},
					{TAG_NODECLI,		TR_ForEachNode},
					{TAG_NODEBANDIN,	TR_ForEachNode},
					{TAG_NODEBANDOUT,	TR_ForEachNode},
					{TAG_NODEPACKIN,	TR_ForEachNode},
					{TAG_NODEPACKOUT,	TR_ForEachNode},
					{TAG_NODEREPLYIN,	TR_ForEachNode},
					{TAG_NODEDROP,		TR_ForEachNode},
					{TAG_NODEQUEUE,		TR_ForEachNode},
					{TAG_NODEHOSTS,		TR_ForEachNode},
					{TAG_NODESIZE,		TR_ForEachNode},
					{TAG_NODETIME,		TR_ForEachNode},
					{TAG_NODESTIME,		TR_ForEachNode},
					{TAG_NODEMODE,		TR_ForEachNode},
					{TAG_NODEEFF,		TR_ForEachNode},
					{TAG_UPLID,			TR_ForEachUpload},
					{TAG_UPLIP,			TR_ForEachUpload},
					{TAG_UPLPORT,		TR_ForEachUpload},
					{TAG_UPLAGENT,		TR_ForEachUpload},
					{TAG_UPLFILE,		TR_ForEachUpload},
					{TAG_UPLSIZE,		TR_ForEachUpload},
					{TAG_UPLCOMPL,		TR_ForEachUpload},
					{TAG_UPLPERCENT,	TR_ForEachUpload},
					{TAG_UPLRATE,		TR_ForEachUpload},
					{TAG_UPLETA,		TR_ForEachUpload},
					{TAG_UPLSTATUS,		TR_ForEachUpload},
					{TAG_DOWNLID,		TR_ForEachDownload},
					{TAG_DOWNLIP,		TR_ForEachDownload},
					{TAG_DOWNLPORT,		TR_ForEachDownload},
					{TAG_DOWNLFILE,		TR_ForEachDownload},
					{TAG_DOWNLALTRN,	TR_ForEachDownload},
					{TAG_DOWNLBADALTRN,	TR_ForEachDownload},
					{TAG_DOWNLSIZE,		TR_ForEachDownload},
					{TAG_DOWNLCOMPL,	TR_ForEachDownload},
					{TAG_DOWNLPERCENT,	TR_ForEachDownload},
					{TAG_DOWNLRATE,		TR_ForEachDownload},
					{TAG_DOWNLETA,		TR_ForEachDownload},
					{TAG_DOWNLSTATUS,	TR_ForEachDownload},
					{TAG_DOWNLEXTSTATUS,TR_ForEachDownload},
					{TAG_ITEMIP,		TR_ForEachDownloadItem},
					{TAG_ITEMPORT,		TR_ForEachDownloadItem},
					{TAG_ITEMISUP,		TR_ForEachDownloadItem},
					{TAG_ITEMRATE,		TR_ForEachDownloadItem},
					{TAG_ITEMCONNERR,	TR_ForEachDownloadItem},
					{TAG_ITEMPUSHTO,	TR_ForEachDownloadItem},
					{TAG_ITEMDISCREASON,TR_ForEachDownloadItem},
					{TAG_ITEMSTATUS,	TR_ForEachDownloadItem},
					{TAG_SEARID,		TR_ForEachSearch},
					{TAG_SEARNAME,		TR_ForEachSearch},
					{TAG_SEARSTRING,	TR_ForEachSearch},
					{TAG_SEARSIZEFLT,	TR_ForEachSearch},
					{TAG_SEARAUTOGET,	TR_ForEachSearch},
					{TAG_SEARCANERASE,	TR_ForEachSearch},
					{TAG_SEARHITS,		TR_ForEachSearch},
					{TAG_SEARGROUPS, 	TR_ForEachSearch},
					{TAG_RESID,			TR_Result},
					{TAG_RESFILENAME,	TR_Result},
					{TAG_RESFILEREF,	TR_Result},
					{TAG_RESFILESIZE,	TR_Result},
					{TAG_RESHOSTIP,		TR_Result},
					{TAG_RESHOSTPORT,	TR_Result},
					{TAG_RESSPEEDNUM,	TR_Result},
					{TAG_RESSPEED,		TR_Result},
					{TAG_RESTIME,		TR_Result},
					{TAG_RESVENDOR,		TR_Result},
					{TAG_RESEXTRA,		TR_Result},
					{TAG_RESSHA1,		TR_Result},
					{TAG_GRPID,			TR_ResultGroup},
					{TAG_GRPFILENAME,	TR_ResultGroup},
					{TAG_GRPFILESIZE,	TR_ResultGroup},
					{TAG_GRPLOCATIONS,	TR_ResultGroup},
					{TAG_GRPAVGSPEEDNUM, TR_ResultGroup},
					{TAG_GRPAVGSPEED,	TR_ResultGroup},
					{TAG_EVTIME,		TR_Event},
					{TAG_EVRELTIME,		TR_Event},
					{TAG_EVTYPE,		TR_Event},
					{TAG_EVSEVERITY,	TR_Event},
					{TAG_EVID,			TR_Event},
					{TAG_EVMESSAGE,		TR_Event}
				};

void acquire_tag(const char * szTag, TagResolv resolver, map<const char *, CString>& mapTagValues, SLoopContext* pContext);

const char * loops[] = {"Conn", "Upload", "Download", "Search", "Result", "Group", "Item", "Event"};

bool MUIRemotePriv::process_html_buff(const char * pFileBuf, int nBufLen, CString & sPocessedBuf, SLoopContext* pContext /*=NULL*/)
{
	if (nBufLen<=6 ||
		(0!=strncmp(pFileBuf,"<HTML>",6) && (0!=strncmp(pFileBuf,"<html>",6))  &&
		 0!=strncmp(pFileBuf,"<STYLE",6) && (0!=strncmp(pFileBuf,"<style",6))) )
			return false;
	// we allocate context object just in case there is no external one
	SLoopContext Context;
	if (pContext)
		pContext->pController = m_pController;
	else
		Context.pController = m_pController;
	// we only substitute tags in style files
	bool bOnlyTags = (0==strncmp(pFileBuf,"<STYLE",6) || (0==strncmp(pFileBuf,"<style",6)));
	//
	return process_buff(pFileBuf, nBufLen, sPocessedBuf, false, bOnlyTags, pContext ? pContext : &Context);
}

bool MUIRemotePriv::replaceFormParams(CString& sLine, bool bLineModified, SLoopContext* pContext )
{
	int nPos;
	if ((nPos=sLine.find('['))>=0)
	{
		// extract field name
		int nPos1 = sLine.find(']', nPos+1);
		if (nPos1>nPos)
		{
			CString sInpTag = sLine.substr(nPos, nPos1-nPos+1);
			CString sReplace;
			if (pContext && pContext->pMapFormParams)
			{
				for (map<CString, CString>::const_iterator itfp = pContext->pMapFormParams->begin(); itfp != pContext->pMapFormParams->end(); ++itfp)
				{
					if (sInpTag == CString("[")+itfp->first+"_checked]")
					{
						sReplace = "checked";
						break;
					}
					if (sInpTag == CString("[")+itfp->first+"_"+itfp->second+"_checked]")
					{
						sReplace = "checked";
						break;
					}
					if (sInpTag == CString("[")+itfp->first+"_value]")
					{
						sReplace = itfp->second;
						break;
					}
				}
			}
			bLineModified = true;
			ReplaceSubStr(sLine, sInpTag, sReplace);
		}
	}
	return bLineModified;
}

bool MUIRemotePriv::replaceContextTags(CString& sLine, bool bLineModified, int nTags, map<const char *, CString>& mapTagValues, SLoopContext* pContext)
{
	int nPos;
	if (sLine.find('#')>=0)
	{
		for (int n=0; n<nTags; ++n)
		{
			if ((nPos=sLine.find(tags[n].tag))>=0)
			{
				bLineModified = true;
				map<const char *, CString>::iterator it = mapTagValues.end();
				if ((it=mapTagValues.find(tags[n].tag))==mapTagValues.end())
				{
					// have to acquire value of the tag
					acquire_tag(tags[n].tag, tags[n].res, mapTagValues, pContext);
					// find it again
					it=mapTagValues.find(tags[n].tag);
				}
				//ASSERT(it!=mapTagValues.end());
				if (it!=mapTagValues.end())
					ReplaceSubStr(sLine, it->first, it->second);
			}
		}
	}
	return bLineModified;
}


bool MUIRemotePriv::process_buff(const char * pBuf, int nBufLen, CString & sPocessedBuf, bool bProcessed, bool bOnlyTags, SLoopContext* pContext)
{
	// map to cache resolved tags
	map<const char *, CString> mapTagValues;
	// size of the tags array
	int nTags = sizeof(tags)/sizeof(STagEntry);
	int nPos;
	bool bLineModified;
	for (const char* pB = pBuf; pB && pB-pBuf<nBufLen;)
	{
		const char* pEL = strchr(pB,'\n');// look for the end ol line (\r\n\ still covered)
		if (!pEL)
			pEL = pBuf+nBufLen-1; // will point to the last char in the buffer
		if (pB==pEL)
		{
			++pB;
			continue;
		}
		// get the line for processing
		// it would be better not to create a copy, but... too much hassle
		CString sLine(pB, pEL-pB);// this might leave trailing \r but we dont care
		// process the line
		bLineModified = false;
		if (!bOnlyTags)
		{
			//TODO: add general <!--# ... --> parser
			// first check for <!--#include filename="..." -->
			if ((nPos=sLine.find("<!--#include "))>=0)
			{
				// process the included file and replace <> with the result
				// problems: sLine IS SUPPOSED to be just a line!
				// we can get around that by spliting the line into 3 parts:
				// 1: before <!--
				// 2: inside <!-- ... -->
				// 3: after -->
				// than first process part 1 as a separate line, replace 2 with the
				// included processed file and process 3 as a separate line.
				// Question: how we do that?
				// so why dont we cut the line here and let 1 be processed and wory about 2 and 3 later
				sLine.cut(nPos);
			}
			// now check for <!--#if statement
			if ((nPos=sLine.find("<!--#if"))>=0)
			{
				// same as <!--#include
				sLine.cut(nPos);
			}
			// now check if we have <!-- ForEach in the line and trim it appropriately
			if ((nPos=sLine.find("<!-- ForEach"))>=0)
			{
				// same as <!--#include
				sLine.cut(nPos);
			}
			// now check if we have tag markers in the line
			// form parameters
			bLineModified = replaceFormParams(sLine, bLineModified, pContext);
		}
		// or mutella context-specific tags
		bLineModified = replaceContextTags(sLine, bLineModified, nTags, mapTagValues, pContext);
		// we are done with the line, now lets put it to the buffer if its needed
		if (bLineModified && !bProcessed)
		{
			sPocessedBuf.reserve(nBufLen);      // the output buffer will be at least same long as the input one
			sPocessedBuf.assign(pBuf, pB-pBuf); // this will copy data only once
			bProcessed = true;
		}
		if (bProcessed)
			sPocessedBuf += sLine; // now we are forced to copy all other lines. TODO: optimize

		if (!bOnlyTags)
		{
			// now it's time to check for tags which modify the structure of the document
			// loops and includes
			// we have to restore the original sLine content
			sLine = CString(pB, pEL-pB);
			// check for include statements again
			if ((nPos=sLine.find("<!--#include "))>=0)
			{
				const char * pBegin = pB + nPos;
				// find the end of the tag
				int nPos1;
				if ((nPos1=sLine.find("-->"))<0)
					break; // we did not find the 'end of comment' in this line
				const char* pCont = pB + nPos1+3;
				// 'parse out' the file name
				CString sParam(pBegin+13,nPos1-nPos-13); //13=strlen("<!--#include ")
				// im too lasy to parse things like "filename="
				CString sRelFilePath = StripAnyOf(sParam," \"");
				//TRACE2("including file ", sRelFilePath);
				// now load the file and feed it to 'process_buffer()'
				// than tail it to what we have already and call pCoont a new line
				// load
				CString sIncluded;
				char * pIncludeBuf = NULL;
				bool bIncludeProcessed = false;
				//
				CString sDiskPath = ExpandPath(CString(m_szRemoteTemplPath)+"/"+sRelFilePath);
				int hFile = open(sDiskPath.c_str(), O_RDONLY);
				int nFileSize = 0;
				if (hFile>=0)
				{
					nFileSize = lseek(hFile,0,SEEK_END);
					lseek(hFile,0,SEEK_SET);
					pIncludeBuf = (char *) mmap(NULL, nFileSize, PROT_READ, MAP_SHARED, hFile, 0);
					// here we add check if the file is the HTML one and parse and possibly modify its content
					bIncludeProcessed = process_buff(pIncludeBuf, nFileSize, sIncluded, bIncludeProcessed, false, pContext);
				}
				else
				{
					pIncludeBuf = "<!-- FAILED TO INCLUDE FILE HERE -->";
				}
				// tail
				if (!bProcessed)
				{
					sPocessedBuf.reserve(nBufLen);      // the output buffer will be at least same long as the input one
					sPocessedBuf.assign(pBuf, pBegin-pBuf); // this will copy data only once
					bProcessed = true;
				}
				if (bIncludeProcessed)
					sPocessedBuf += sIncluded;
				else
					sPocessedBuf += pIncludeBuf;
				// close and unmmap
				if (hFile>=0)
				{
					munmap(pIncludeBuf, nFileSize);
					::close(hFile);
				}
				// continue from where we stoped in the original buffer
				pB=pCont;
				continue;
			}
			// now check for <!--#if
			if ((nPos=sLine.find("<!--#if"))>=0)
			{
				// it's going to be a bit similar to 'ForEach' case
				// 1. Get the expression out
				// 2. Find end marker
				// 3. Substitute expression
				// 4. Check what comes out and either leave the stuff in or out

				int nPosExprStart = nPos;

				if ((nPos=sLine.find("-->", nPos))<0)
					break; // we did not find the 'end of comment' in this line
				CString sExpr = StripWhite(sLine.substr(nPosExprStart+7, nPos-nPosExprStart-7));
				//
				const char* pThenStart = pB + nPos+3;
				const char* pThenEnd = strstr(pThenStart,"<!--#endif");
				if (!pThenEnd)
					break;// failed to find the end marker
				// pThenEnd now points to the end of the 'if-endif'
				// now lets find the new "continue" point
				pB = strstr(pThenEnd, "-->");
				if (pB)
					pEL = pB + 2; // the pointer wiil be incremented by 1, see below
				else
					pEL = pBuf+nBufLen-1; // will point to the last char in the buffer
					                      // and thus will cause loop to finish
				// now lets find <!--#else --> between #if and #endif
				const char* pElseEnd = NULL;
				const char* pElseStart = strstr(pThenStart,"<!--#else");
				if (pElseStart && pElseStart < pThenEnd)
				{
					// got it between #if and #endif
					pElseEnd = pThenEnd;
					pThenEnd = pElseStart;
					pElseStart = strstr(pThenEnd, "-->");
					if (pElseStart==NULL || pElseStart >= pElseEnd)
						break;
					pElseStart += 3;
				}
				// apply standard substitutions to sExpr
				replaceFormParams(sExpr, false, pContext);
				replaceContextTags(sExpr, false, nTags, mapTagValues, pContext);
				// ready to rock!
				CString sIf;
				if (sExpr.length() && sExpr != "0" && sExpr != "false" && sExpr != "off")
				{
					// apply 'then' part
					sIf = CString(pThenStart, pThenEnd-pThenStart);
				}
				else
				{
					// apply 'else' part if we have it
					if (pElseStart)
						sIf = CString(pElseStart, pElseEnd-pElseStart);
				}
				if (!sIf.empty())
				{
					// process if's interior
					CString sIfProc;
					process_buff(sIf.c_str(), sIf.length(), sIfProc, false, false, pContext);
					// tail
					if (!bProcessed)
					{
						sPocessedBuf.reserve(nBufLen);      // the output buffer will be at least same long as the input one
						sPocessedBuf.assign(pBuf, pThenStart-pBuf); // this will copy data only once
						bProcessed = true;
					}
					if (sIfProc.empty())
						sPocessedBuf += sIf;
					else
						sPocessedBuf += sIfProc;
				}
			}
			// now check for the loops
			int nLoops = sizeof(loops)/sizeof(const char*);
			// TODO: following assumes that the line was not eddited!
			if ((nPos=sLine.find("<!-- ForEach"))>=0)
			{
				// yep, we have it. now lets see if it's one of the recognized loops
				int nL = strlen("<!-- ForEach");
				for (int n = 0; n<nLoops; ++n)
				{
					if (sLine.substr(nPos+nL,strlen(loops[n]))==loops[n])
					{
						// now we have it!
						CString sLoopName = sLine.substr(nPos,nL+strlen(loops[n]));
						// the only thing left to check is if its' really the begining
						bool bStart = sLine.substr(nPos+nL+strlen(loops[n]), 5)=="Start";
						bool bListedStart = sLine.substr(nPos+nL+strlen(loops[n]), 11)=="ListedStart";
						if (bStart || bListedStart)
						{
							// Ok, here we have the loop marker for the loop start. search for the end of comment and
							// this will be the start of the loop text module
							// than we have to find ForEach...End and begining
							// of the comment will be the end of the loop and
							// the end of comment will be than new position for
							// the normal parsing.
							if ((nPos=sLine.find("-->"))<0)
								break; // we did not find the 'end of comment' in this line
							const char* pLoop = pB + nPos+3;
							// now look up the end marker
							CString sEndMarker = sLoopName + "End";
							const char* pEndMarker = strstr(pLoop,sEndMarker.c_str());
							if (!pEndMarker)
								break;// failed to find the end marker
							// pEndMarker now points to the end of the loop text
							// now lets find the new "continue" point
							pB = strstr(pEndMarker, "-->");
							if (pB)
								pEL = pB + 2; // the pointer wiil be incremented by 1, see below
							else
								pEL = pBuf+nBufLen-1; // will point to the last char in the buffer
								                      // and thus will cause loop to finish
							// now lets find <!-- ForEachXxxxNone --> marker between the
							// pLoop and pEndMarker
							CString sNoneMarker = sLoopName + "None";
							const char* pNoneMarker = strstr(pLoop,sNoneMarker.c_str()); // may be we can save searching here?
							// well, in the current incarnation "None" comment will stay in the file that is sent
							// cause I'm to lasy to look for the end of comment
							if (pNoneMarker == NULL ||  pNoneMarker>pEndMarker)
								pNoneMarker = pEndMarker;
							// and only now we can enter the recursive call to process the content of the loop
							// but before we have to ensure that we are "copying" the buffer
							if (!bProcessed)
							{
								sPocessedBuf.reserve(nBufLen);      // the output buffer will be at least same long as the input one
								sPocessedBuf.assign(pBuf, pLoop-pBuf); // this will copy data only once
								bProcessed = true;
							}
							process_loop(pLoop, pNoneMarker-pLoop, pNoneMarker, pEndMarker-pNoneMarker, loops[n], sPocessedBuf, bListedStart, pContext);
						}
						//
						break;
					}
				}
			}
		}
		////////////////////////
		// increment the pointer
		pB = pEL+1;
	}
	// do actual processing: search for the markers and tags
	return bProcessed;
}

/////////////////////////////////////////////////////////////////////
// resolve the tag value, querying MController if needed and possible

LPCSTR FormatEventSeverity(int n)
{
	switch (n)
	{
		case ES_NONE:        return "undefined";
		case ES_DEBUG:       return "debug";
		case ES_MINIMAL:     return "minimal";
		case ES_UNIMPORTANT: return "low";
		case ES_JUSTINCASE:  return "lower";
		case ES_GOODTOKNOW:  return "medium";
		case ES_IMPORTANT:   return "high";
		case ES_CRITICAL:    return "critical";
	}
	return "INVALID";
}

LPCSTR FormatEventType(int n)
{
	switch (n)
	{
		case ET_NONE:         return "UNDEFINED";
		case ET_ERROR:        return "ERROR";
		case ET_WARNING:      return "WARNING";
		case ET_MESSAGE:      return "MESSAGE";
		case ET_STATUSUPDATE: return "STATUS CHANGE";
	}
	return "INVALID";
}

static const char* FormatPeerMode(SGnuNode* pNode)
{
	if (pNode->m_nStatus != SOCK_CONNECTED)
		return "---";
	switch (pNode->m_nPeerMode) {
		case CM_NORMAL:    return "normal";
		case CM_LEAF:      return "leaf";
		case CM_ULTRAPEER: return "ultrapeer";
	}
	return "invalid";
}

static CString FormatSearch(SGnuSearch* pSearch)
{
	CString res = pSearch->m_Search;
	for (set<CString>::iterator it = pSearch->m_setSha1.begin(); it != pSearch->m_setSha1.end(); ++it)
	{
		if (!res.empty())
			res += ' ';
		res += "sha1:";
		res += *it;
	}
	return res;
}

void acquire_tag(const char * szTag, TagResolv resolver, map<const char *, CString>& mapTagValues, SLoopContext* pCnxt)
{
	switch (resolver)
	{
		case TR_Globals:
			mapTagValues[TAG_REMOTEROOT] = pCnxt->psRoot ? *pCnxt->psRoot : "";;
			mapTagValues[TAG_CURRENTURL] = pCnxt->psUrl  ? *pCnxt->psUrl  : "";
			mapTagValues[TAG_CURRENTFORM] = pCnxt->psForm ? *pCnxt->psForm : "";
			mapTagValues[TAG_CURRENTTIME] = FormatAbsTime(xtime());
			break;
		case TR_Message:
			mapTagValues[TAG_MESSAGE] = pCnxt->psMessage ? *pCnxt->psMessage : "";
			mapTagValues[TAG_STATUS] = pCnxt->psStatus ? *pCnxt->psStatus : "";
			break;
		case TR_GetHostMode:
			if(!pCnxt->pController)
			{
				mapTagValues[szTag]=CString("[") + szTag + "(out of context)]";
				return;
			}
			mapTagValues[TAG_HOSTMODE] = pCnxt->pController->IsUltrapeer() ? "UltraPeer" : "Leaf";
			break;
		case TR_GetHostUptime:
			if(!pCnxt->pController)
			{
				mapTagValues[szTag]=CString("[") + szTag + "(out of context)]";
				return;
			}
			mapTagValues[TAG_HOSTUPTIME] = FormatTime(xtime() - pCnxt->pController->GetClientStartTime());
			break;
		case TR_GetNetStats:
			{
				if(!pCnxt->pController)
				{
					mapTagValues[szTag]=CString("[") + szTag + "(out of context)]";
					return;
				}
				int nHosts, nSharingHosts, nFiles, nSize;
				pCnxt->pController->GetNetStats(nHosts, nSharingHosts, nFiles, nSize);
				mapTagValues[TAG_NETHOSTS]  = FormatSize(nHosts);
				mapTagValues[TAG_NETSHOSTS] = FormatSize(nSharingHosts);
				mapTagValues[TAG_NETFILES]  = FormatSize(nFiles);
				mapTagValues[TAG_NETSIZE]   = FormatMSize(nSize);
			}
			break;
		case TR_GetCacheStats:
			{
				if(!pCnxt->pController)
				{
					mapTagValues[szTag]=CString("[") + szTag + "(out of context)]";
					return;
				}
				int nCacher, nUltra, nStore, nGWeb;
				pCnxt->pController->GetCacheStats(nCacher, nUltra, nStore, nGWeb);
				mapTagValues[TAG_CATCHERHOSTS]  = FormatSize(nCacher);
				mapTagValues[TAG_ULTRAHOSTS]    = FormatSize(nUltra);
				mapTagValues[TAG_STOREHOSTS]    = FormatSize(nStore);
				mapTagValues[TAG_WEBCACHEURLS]  = FormatSize(nGWeb);
			}
			break;
		case TR_GetConnStats:
			{
				if(!pCnxt->pController)
				{
					mapTagValues[szTag]=CString("[") + szTag + "(out of context)]";
					return;
				}
				int nConn, nUpl, nDownlA, nDownlT;
				pCnxt->pController->GetConnStats(nConn, nUpl, nDownlA, nDownlT);
				mapTagValues[TAG_NETSOCKS]   = FormatSize(nConn);
				mapTagValues[TAG_UPLSOCKS]   = FormatSize(nUpl);
				mapTagValues[TAG_DOWNLSOCKS] = FormatSize(nDownlA);
				mapTagValues[TAG_DOWNLCOUNT] = FormatSize(nDownlT);
				mapTagValues[TAG_TRANSSOCKS] = FormatSize(nUpl+nDownlA);
			}
			break;
		case TR_GetBandwidthStats:
			{
				if(!pCnxt->pController)
				{
					mapTagValues[szTag]=CString("[") + szTag + "(out of context)]";
					return;
				}
				int nConn, nUpl, nDownl;
				pCnxt->pController->GetBandwidthStats(nConn, nUpl, nDownl);
				mapTagValues[TAG_NETBAND]   = FormatSize(nConn);
				mapTagValues[TAG_UPLBAND]   = FormatSize(nUpl);
				mapTagValues[TAG_DOWNLBAND] = FormatSize(nDownl);
				mapTagValues[TAG_TOTALBAND] = FormatSize(nConn+nUpl+nDownl);
			}
			break;
		case TR_GetConnRate:
			if(!pCnxt->pController)
			{
				mapTagValues[szTag]=CString("[") + szTag + "(out of context)]";
				return;
			}
			mapTagValues[TAG_NETBANDIN]  = FormatSize(pCnxt->pController->GetConnRateRecv());
			mapTagValues[TAG_NETBANDOUT] = FormatSize(pCnxt->pController->GetConnRateSend());
			break;
		case TR_LoopCounter:
			mapTagValues[TAG_LOOPCOUNTER] = DWrdtoStr(pCnxt->dwLoopCounter);
			mapTagValues[TAG_LOOPMOD2]    = DWrdtoStr(pCnxt->dwLoopCounter % 2);
			mapTagValues[TAG_LOOPMOD3]    = DWrdtoStr(pCnxt->dwLoopCounter % 3);
			mapTagValues[TAG_LOOPMOD4]    = DWrdtoStr(pCnxt->dwLoopCounter % 4);
			break;
		case TR_ForEachNode:
			if(!pCnxt->pNode)
			{
				mapTagValues[szTag]=CString("[") + szTag + "(out of context)]";
				return;
			}
			mapTagValues[TAG_NODEID]      = DWrdtoStr(pCnxt->pNode->m_dwID);
			mapTagValues[TAG_NODEIP]      = pCnxt->pNode->m_sHost;
			mapTagValues[TAG_NODECLI]     = pCnxt->pNode->m_sRemoteClient;
			mapTagValues[TAG_NODEPORT]    = DWrdtoStr(pCnxt->pNode->m_nPort);
			mapTagValues[TAG_NODEBANDIN]  = FormatSize((int)pCnxt->pNode->m_dStatisticsRate[NR_BYTE_IN]);
			mapTagValues[TAG_NODEBANDOUT] = FormatSize((int)pCnxt->pNode->m_dStatisticsRate[NR_BYTE_OUT]);
			mapTagValues[TAG_NODEPACKIN]  = FormatSize((int)(pCnxt->pNode->m_dStatisticsRate[NR_PACKET_IN]  * 60.0 + 0.5));
			mapTagValues[TAG_NODEPACKOUT] = FormatSize((int)(pCnxt->pNode->m_dStatisticsRate[NR_PACKET_OUT] * 60.0 + 0.5));
			mapTagValues[TAG_NODEREPLYIN] = FormatSize((int)(pCnxt->pNode->m_dStatisticsRate[NR_QREPLY_IN]  * 60.0 + 0.5));
			mapTagValues[TAG_NODEDROP]    = DWrdtoStr(pCnxt->pNode->m_nAvgDroppedPackets);
			mapTagValues[TAG_NODEQUEUE]   = DWrdtoStr(pCnxt->pNode->m_nSendQueueSize);
			mapTagValues[TAG_NODEHOSTS]   = FormatNumber(pCnxt->pNode->m_dwFriendsTotal);
			mapTagValues[TAG_NODESIZE]    = FormatKSizeLL(pCnxt->pNode->m_llLibraryTotal);
			mapTagValues[TAG_NODETIME]    = FormatTime(xtime()-pCnxt->pNode->m_nUptime);
			mapTagValues[TAG_NODESTIME]   = FormatTime(xtime()-pCnxt->pNode->m_nPeerUptime);
			mapTagValues[TAG_NODEMODE]    = FormatPeerMode(pCnxt->pNode);
			mapTagValues[TAG_NODEEFF]     = FormatPercent(pCnxt->pNode->m_nEfficiency, 100);
			break;
		case TR_ForEachUpload:
			if(!pCnxt->pUpload)
			{
				mapTagValues[szTag]=CString("[") + szTag + "(out of context)]";
				return;
			}
			mapTagValues[TAG_UPLID]          = DWrdtoStr(pCnxt->pUpload->m_dwID);
			mapTagValues[TAG_UPLIP]          = Ip2Str(pCnxt->pUpload->m_ipHost);
			mapTagValues[TAG_UPLPORT]        = DWrdtoStr(pCnxt->pUpload->m_nPort);
			mapTagValues[TAG_UPLAGENT]       = pCnxt->pUpload->m_sUserAgent;
			mapTagValues[TAG_UPLFILE]        = pCnxt->pUpload->m_sFileName;
			mapTagValues[TAG_UPLSIZE]        = FormatSize(pCnxt->pUpload->m_nFileLength);
			mapTagValues[TAG_UPLCOMPL]       = FormatSize(pCnxt->pUpload->m_nBytesCompleted);
			mapTagValues[TAG_UPLPERCENT]     = FormatPercent(pCnxt->pUpload->m_nBytesCompleted,pCnxt->pUpload->m_nFileLength);
			mapTagValues[TAG_UPLRATE]        = FormatSize((int)pCnxt->pUpload->m_dRate);
			mapTagValues[TAG_UPLETA]         = (pCnxt->pUpload->m_dRate > 0) ? FormatTime((int) ((pCnxt->pUpload->m_nFileLength-pCnxt->pUpload->m_nBytesCompleted)/pCnxt->pUpload->m_dRate)) : "--:--";
			mapTagValues[TAG_UPLSTATUS]      = SGnuUpload::GetErrorString(pCnxt->pUpload->m_nError);
			break;
		case TR_ForEachDownload:
			if(!pCnxt->pDownload)
			{
				mapTagValues[szTag]=CString("[") + szTag + "(out of context)]";
				return;
			}
			mapTagValues[TAG_DOWNLID]           = DWrdtoStr(pCnxt->pDownload->m_dwID);
			mapTagValues[TAG_DOWNLFILE]         = pCnxt->pDownload->m_sName;
			mapTagValues[TAG_DOWNLALTRN]        = DWrdtoStr(pCnxt->pDownload->m_vecItems.size());
			mapTagValues[TAG_DOWNLBADALTRN]     = DWrdtoStr(pCnxt->pDownload->m_nBadResults);
			mapTagValues[TAG_DOWNLSIZE]         = FormatSize(pCnxt->pDownload->m_dwFileLength);
			mapTagValues[TAG_DOWNLCOMPL]        = FormatSize(pCnxt->pDownload->m_dwBytesCompleted);
			mapTagValues[TAG_DOWNLPERCENT]      = FormatPercent(pCnxt->pDownload->m_dwBytesCompleted,pCnxt->pDownload->m_dwFileLength);
			mapTagValues[TAG_DOWNLRATE]         = FormatSize((int)pCnxt->pDownload->m_dRate);
			mapTagValues[TAG_DOWNLETA]          = (pCnxt->pDownload->m_dRate > 0) ? FormatTime((int) ((pCnxt->pDownload->m_dwFileLength-pCnxt->pDownload->m_dwBytesCompleted)/pCnxt->pDownload->m_dRate)) : "--:--";
			mapTagValues[TAG_DOWNLSTATUS]       = SGnuDownload::GetStatusString(pCnxt->pDownload->m_nStatus);
			if (pCnxt->pDownload->m_vecItems.size())
			{
				mapTagValues[TAG_DOWNLIP]       = Ip2Str(pCnxt->pDownload->m_vecItems[0].ipHost);
				mapTagValues[TAG_DOWNLPORT]     = DWrdtoStr(pCnxt->pDownload->m_vecItems[0].nPort);
			}
			else
			{
				mapTagValues[TAG_DOWNLIP]       = "";
				mapTagValues[TAG_DOWNLPORT]     = "";
			}
			if (pCnxt->pDownload->m_nStatus != TRANSFER_CLOSED && pCnxt->pDownload->m_nStatus != TRANSFER_CLOSING)
				mapTagValues[TAG_DOWNLEXTSTATUS]    = "ACTIVE";
			else
				mapTagValues[TAG_DOWNLEXTSTATUS]    = CString("INACTIVE, reason: ") + SGnuDownload::GetErrorString(pCnxt->pDownload->m_nLastDisconnectReason);
			break;
		case TR_ForEachDownloadItem:
			if(!pCnxt->pDownloadItem)
			{
				mapTagValues[szTag]=CString("[") + szTag + "(out of context)]";
				return;
			}
			mapTagValues[TAG_ITEMIP]         = Ip2Str(pCnxt->pDownloadItem->ipHost);
			mapTagValues[TAG_ITEMPORT]       = DWrdtoStr(pCnxt->pDownloadItem->nPort);
			mapTagValues[TAG_ITEMISUP]       = pCnxt->pDownloadItem->bServerIsUp ? "ServerIsUp" : "NoReplyYet";
			mapTagValues[TAG_ITEMRATE]       = FormatSize(pCnxt->pDownloadItem->dwRate);
			mapTagValues[TAG_ITEMCONNERR]    = DWrdtoStr(pCnxt->pDownloadItem->nConErrors);
			mapTagValues[TAG_ITEMPUSHTO]     = DWrdtoStr(pCnxt->pDownloadItem->nPushTimeouts);
			mapTagValues[TAG_ITEMDISCREASON] = (REASON_UNDEFINED != pCnxt->pDownloadItem->nDisconnectReason) ? SGnuDownload::GetErrorString(pCnxt->pDownloadItem->nDisconnectReason) : "";
			mapTagValues[TAG_ITEMSTATUS]     = SGnuDownload::GetStatusString(pCnxt->pDownloadItem->nStatus);
			break;
		case TR_ForEachSearch:
			if(!pCnxt->pSearch)
			{
				mapTagValues[szTag]=CString("[") + szTag + "(out of context)]";
				return;
			}
			mapTagValues[TAG_SEARID]       = DWrdtoStr(pCnxt->pSearch->m_dwID);
			mapTagValues[TAG_SEARNAME]     = FormatSearch(pCnxt->pSearch);
			mapTagValues[TAG_SEARSTRING]   = pCnxt->pSearch->m_Search;
			mapTagValues[TAG_SEARAUTOGET]  = pCnxt->pSearch->m_bAutoget ? "AUTOGET" : "&nbsp;";
			mapTagValues[TAG_SEARCANERASE] = (pCnxt->pSearch->m_bAutoget && pCnxt->pSearch->m_nType != ST_USER) ? "true" : "false";
			mapTagValues[TAG_SEARHITS]     = DWrdtoStr(pCnxt->pSearch->m_nHits);
			mapTagValues[TAG_SEARGROUPS]   = DWrdtoStr(pCnxt->pSearch->m_nGroups);
			if (pCnxt->pSearch->m_SizeFilterMode == LIMIT_NONE)
				mapTagValues[TAG_SEARSIZEFLT] = "no size filter";
			else if (pCnxt->pSearch->m_SizeFilterMode == LIMIT_MORE)
				mapTagValues[TAG_SEARSIZEFLT] = CString("MIN: ") + FormatSize(pCnxt->pSearch->m_SizeFilterValue);
			else if (pCnxt->pSearch->m_SizeFilterMode == LIMIT_EXACTLY)
				mapTagValues[TAG_SEARSIZEFLT] = CString("SIZE: ") + DWrdtoStr(pCnxt->pSearch->m_SizeFilterValue);
			else if (pCnxt->pSearch->m_SizeFilterMode == LIMIT_LESS)
				mapTagValues[TAG_SEARSIZEFLT] = CString("MAX: ") + FormatSize(pCnxt->pSearch->m_SizeFilterValue);
			else if (pCnxt->pSearch->m_SizeFilterMode == LIMIT_APPROX)
				mapTagValues[TAG_SEARSIZEFLT] = CString("AROUND: ") + FormatSize(pCnxt->pSearch->m_SizeFilterValue);
			else
				mapTagValues[TAG_SEARSIZEFLT] = "wrong value";
			break;
		case TR_Result:
			if (!pCnxt->pResult)
			{
				mapTagValues[szTag]=CString("[") + szTag + "(out of context)]";
				return;
			}
			mapTagValues[TAG_RESID]       = DWrdtoStr(pCnxt->pResult->dwID);
			mapTagValues[TAG_RESFILENAME] = pCnxt->pResult->Name;
			mapTagValues[TAG_RESFILEREF]  = DWrdtoStr(pCnxt->pResult->FileIndex);
			mapTagValues[TAG_RESFILESIZE] = FormatSize(pCnxt->pResult->Size);
			mapTagValues[TAG_RESHOSTIP]   = Ip2Str(pCnxt->pResult->Host);
			mapTagValues[TAG_RESHOSTPORT] = DWrdtoStr(pCnxt->pResult->Port);
			mapTagValues[TAG_RESSPEEDNUM] = DWrdtoStr(pCnxt->pResult->Speed);
			mapTagValues[TAG_RESSPEED]    = GetSpeedString(pCnxt->pResult->Speed);
			mapTagValues[TAG_RESTIME]     = FormatTime(xtime() - pCnxt->pResult->ChangeTime);
			mapTagValues[TAG_RESVENDOR]   = pCnxt->pResult->Vendor;
			mapTagValues[TAG_RESSHA1]     = pCnxt->pResult->Sha1;
			if (!pCnxt->pResult->Extra.empty())
			{
				CString sExtra = pCnxt->pResult->Extra;
				ReplaceSubStr(sExtra, "</l>", "</l>\n");
				ReplaceSubStr(sExtra, "<", "&lt;");
				ReplaceSubStr(sExtra, ">", "&gt;");
				mapTagValues[TAG_RESEXTRA] = sExtra;
			}
			else
				mapTagValues[TAG_RESEXTRA] = "";
			break;
		case TR_ResultGroup:
			if(!pCnxt->pResGroup)
			{
				mapTagValues[szTag]=CString("[") + szTag + "(out of context)]";
				return;
			}
			mapTagValues[TAG_GRPID]           = DWrdtoStr(pCnxt->pResGroup->dwID);
			mapTagValues[TAG_GRPFILENAME]     = pCnxt->pResGroup->Name;
			mapTagValues[TAG_GRPFILESIZE]     = FormatSize(pCnxt->pResGroup->Size);
			mapTagValues[TAG_GRPLOCATIONS]    = DWrdtoStr(pCnxt->pResGroup->ResultSet.size());
			mapTagValues[TAG_GRPAVGSPEEDNUM]  = DWrdtoStr(pCnxt->pResGroup->AvgSpeed);
			mapTagValues[TAG_GRPAVGSPEED]     = GetSpeedString(pCnxt->pResGroup->AvgSpeed);
			break;
		case TR_Event:
			if(!pCnxt->pEvent)
			{
				mapTagValues[szTag]=CString("[") + szTag + "(out of context)]";
				return;
			}
			mapTagValues[TAG_EVTIME]     = FormatAbsTime(pCnxt->pEvent->GetTime());
			mapTagValues[TAG_EVRELTIME]  = FormatTime(xtime() - pCnxt->pEvent->GetTime());
			mapTagValues[TAG_EVTYPE]     = FormatEventType(pCnxt->pEvent->GetType());
			mapTagValues[TAG_EVSEVERITY] = FormatEventSeverity(pCnxt->pEvent->GetSeverity());
			mapTagValues[TAG_EVID]       = DWrdtoStr(pCnxt->pEvent->GetID());
			mapTagValues[TAG_EVMESSAGE]  = pCnxt->pEvent->GetID() ?
												(GetMessageString(pCnxt->pEvent->GetID()) + "\n" + pCnxt->pEvent->Format()) :
												(pCnxt->pEvent->Format());
			break;
		default:
			mapTagValues[szTag]=CString("[") + szTag + "]";
	}
}

/////////////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////
// Loops

struct SCallbackParam{
	MUIRemotePriv* pThis;
	const char *   pLoopBuf;
	int            nBufLen;
	CString*       psPocessedBuf;
	SLoopContext*  pContext;
	bool           bListedLoop;
};

bool loopConnCallback(void* pData, SGnuNode* pNode)
{
	SCallbackParam* pCP = (SCallbackParam*)pData;
	if (pCP->bListedLoop &&
		(pCP->pContext->pSetIDs==NULL ||
		 pCP->pContext->pSetIDs->find(pNode->m_dwID)==pCP->pContext->pSetIDs->end()))
	{
		// not in the set -- leave out
		return true;
	}
	pCP->pContext->pNode = pNode;
	pCP->pContext->dwLoopCounter++;
	pCP->pThis->process_buff(pCP->pLoopBuf, pCP->nBufLen, *(pCP->psPocessedBuf), true, false, pCP->pContext);
	return true;
}

bool loopUploadCallback(void* pData, SGnuUpload* pUpload)
{
	SCallbackParam* pCP = (SCallbackParam*)pData;
	if (pCP->bListedLoop &&
		(pCP->pContext->pSetIDs==NULL ||
		 pCP->pContext->pSetIDs->find(pUpload->m_dwID)==pCP->pContext->pSetIDs->end()))
	{
		// not in the set -- leave out
		return true;
	}

	pCP->pContext->pUpload = pUpload;
	pCP->pContext->dwLoopCounter++;
	pCP->pThis->process_buff(pCP->pLoopBuf, pCP->nBufLen, *(pCP->psPocessedBuf), true, false, pCP->pContext);
	return true;
}

bool loopDownloadCallback(void* pData, SGnuDownload* pDownload)
{
	SCallbackParam* pCP = (SCallbackParam*)pData;
	if (pCP->bListedLoop &&
		(pCP->pContext->pSetIDs==NULL ||
		 pCP->pContext->pSetIDs->find(pDownload->m_dwID)==pCP->pContext->pSetIDs->end()))
	{
		// not in the set -- leave out
		return true;
	}
	pCP->pContext->pDownload = pDownload;
	pCP->pContext->dwLoopCounter++;
	pCP->pThis->process_buff(pCP->pLoopBuf, pCP->nBufLen, *(pCP->psPocessedBuf), true, false, pCP->pContext);
	return true;
}

bool loopSearchCallback(void* pData, SGnuSearch* pSearch)
{
	SCallbackParam* pCP = (SCallbackParam*)pData;
	if (pCP->bListedLoop &&
		(pCP->pContext->pSetIDs==NULL ||
		 pCP->pContext->pSetIDs->find(pSearch->m_dwID)==pCP->pContext->pSetIDs->end()))
	{
		// not in the set -- leave out
		return true;
	}
	if ( pCP->pContext->pListOptions == NULL || // no options means ALL
		 ((pCP->pContext->pListOptions->sFilter.length()==0 || QueryMatch(pSearch->m_Search, pCP->pContext->pListOptions->sFilter)) &&
		  (!pCP->pContext->pListOptions->bNoEmpty || pSearch->m_nHits) &&
		  (!pCP->pContext->pListOptions->bNoAuto  || !pSearch->IsAutomatic())) )
	{
		pCP->pContext->pSearch = pSearch;
		pCP->pContext->dwLoopCounter++;
		pCP->pThis->process_buff(pCP->pLoopBuf, pCP->nBufLen, *(pCP->psPocessedBuf), true, false, pCP->pContext);
	}
	return true;
}

void MUIRemotePriv::process_loop(const char * pLoopBuf, int nBufLen, const char * pLoopNoneBuf, int nLoopNoneBufLen, const char* szLoopName, CString & sPocessedBuf, bool bListedLoop, SLoopContext* pContext)
{
	SLoopContext Context(*pContext);
	// loop counter requires some special handling
	Context.dwLoopCounter = 0;
	// prepare params
	SCallbackParam cp;
	cp.pThis = this;
	cp.pLoopBuf = pLoopBuf;
	cp.nBufLen = nBufLen;
	cp.psPocessedBuf = &sPocessedBuf;
	cp.pContext = &Context;
	cp.bListedLoop = bListedLoop;
	// relay problems to the appropriate callback function
	// const char * loops[] = {"Conn", "Upload", "Download", "Search", "Result"};
	if (0==strcmp(szLoopName, "Conn"))
		m_pController->ForEachConnection((void*)&cp, loopConnCallback);
	else if (0==strcmp(szLoopName, "Upload"))
		m_pController->ForEachUpload((void*)&cp, loopUploadCallback);
	else if (0==strcmp(szLoopName, "Download"))
		m_pController->ForEachDownload((void*)&cp, loopDownloadCallback);
	else if (0==strcmp(szLoopName, "Search"))
		m_pController->ForEachSearch((void*)&cp, loopSearchCallback);
	else if (0==strcmp(szLoopName, "Result"))
	{
		loopResults(pLoopBuf, nBufLen, sPocessedBuf, bListedLoop, &Context);
	}
	else if (0==strcmp(szLoopName, "Group"))
	{
		loopGroups(pLoopBuf, nBufLen, sPocessedBuf, bListedLoop, &Context);
	}
	else if (0==strcmp(szLoopName, "Item"))
	{
		loopItems(pLoopBuf, nBufLen, sPocessedBuf, bListedLoop, &Context);
	}
	else if (0==strcmp(szLoopName, "Event"))
	{
		loopEvents(pLoopBuf, nBufLen, sPocessedBuf, bListedLoop, &Context);
	}
	// now check if the loop counter is still zero. if yes and nLoopNoneBufLen>0
	// copy pLoopNoneBuf to the buffer
	if (Context.dwLoopCounter == 0 &&
		nLoopNoneBufLen>0          &&
		pLoopNoneBuf != NULL)
	{
		sPocessedBuf += CString(pLoopNoneBuf, nLoopNoneBufLen); // this will copy buffer twice. TODO: fix
	}
}


void MUIRemotePriv::loopItems(const char * pLoopBuf, int nBufLen, CString & sPocessedBuf, bool bListedLoop, SLoopContext* pContext)
{
	if (pContext->pDownload == NULL || pContext->pDownload->m_vecItems.size()==0 )
		return;
	for (int i=0; i<min((int)pContext->pDownload->m_vecItems.size(),3); ++i)
	{
		pContext->pDownloadItem = &pContext->pDownload->m_vecItems[i];
		pContext->dwLoopCounter++;
		process_buff(pLoopBuf, nBufLen, sPocessedBuf, true, false, pContext);
	}
}

bool sortResBySize(const Result& r1, const Result& r2)
{
	// sort decending by size, than decending by speed, than accending by name
	if (r1.Size>r2.Size)
		return true;
	if (r1.Size<r2.Size)
		return false;
	if (r1.Speed>r2.Speed)
		return true;
	if (r1.Speed<r2.Speed)
		return false;
	if (r1.Name<r2.Name)
		return true;
	// equivalent elements return false
	return false;
}

bool sortResBySpeed(const Result& r1, const Result& r2)
{
	// sort decending by speed, than decending by size, than accending by name
	if (r1.Speed>r2.Speed)
		return true;
	if (r1.Speed<r2.Speed)
		return false;
	if (r1.Size>r2.Size)
		return true;
	if (r1.Size<r2.Size)
		return false;
	if (r1.Name<r2.Name)
		return true;
	// equivalent elements return false
	return false;
}

bool sortResByName(const Result& r1, const Result& r2)
{
	// sort accending by name, than decending by size, than decending by speed
	if (r1.Name<r2.Name)
		return true;
	if (r1.Name>r2.Name)
		return false;
	if (r1.Size>r2.Size)
		return true;
	if (r1.Size<r2.Size)
		return false;
	if (r1.Speed>r2.Speed)
		return true;
	// equivalent elements return false
	return false;
}

bool sortResByIP(const Result& r1, const Result& r2)
{
	// sort accending by IP, than accending by Port, than call sortResByName
	if (r1.Host.S_ip.a<r2.Host.S_ip.a)
		return true;
	if (r1.Host.S_ip.a>r2.Host.S_ip.a)
		return false;
	if (r1.Host.S_ip.b<r2.Host.S_ip.b)
		return true;
	if (r1.Host.S_ip.b>r2.Host.S_ip.b)
		return false;
	if (r1.Host.S_ip.c<r2.Host.S_ip.c)
		return true;
	if (r1.Host.S_ip.c>r2.Host.S_ip.c)
		return false;
	if (r1.Host.S_ip.d<r2.Host.S_ip.d)
		return true;
	if (r1.Host.S_ip.d>r2.Host.S_ip.d)
		return false;
	if (r1.Port<r2.Port)
		return true;
	if (r1.Port>r2.Port)
		return false;
	return sortResByName(r1, r2);
}

void MUIRemotePriv::loopResults(const char * pLoopBuf, int nBufLen, CString & sPocessedBuf, bool bListedLoop, SLoopContext* pContext)
{
	SGnuSearch gs;
	SSearchRes sr;
	if ( pContext->pSearch!=NULL && pContext->pSearchRes==NULL &&
		 m_pController->GetSearchByID(pContext->pSearch->m_dwID, gs, sr.vecRes, sr.vecGrp))
	{
		pContext->pSearchRes = &sr;
		pContext->pSearch = &gs;
	}
	// we have all the vecotrs and we dont have to call "for each"
	// there is no "for-each-result" anyway
	if (pContext->pSearchRes)
	{
		// prepare the vector with results in it
		// the loop might be inclosed into the "Groups" loop, than we have to perform some hacking
		ResultVec vecResults;
		if (pContext->pResGroup)
		{
			vecResults.reserve(pContext->pResGroup->ResultSet.size());
			for (set<DWORD>::iterator itr = pContext->pResGroup->ResultSet.begin(); itr != pContext->pResGroup->ResultSet.end(); ++itr)
				if (*itr < pContext->pSearchRes->vecRes.size())
					vecResults.push_back(pContext->pSearchRes->vecRes[*itr]);
		}
		else
			vecResults = pContext->pSearchRes->vecRes;
		// sort the list first
		switch (pContext->nResultsSortOrder){
			case RSO_Size: stable_sort(vecResults.begin(), vecResults.end(), sortResBySize);
				break;
			case RSO_Name: stable_sort(vecResults.begin(), vecResults.end(), sortResByName);
				break;
			case RSO_Speed: stable_sort(vecResults.begin(), vecResults.end(), sortResBySpeed);
				break;
			case RSO_IP: stable_sort(vecResults.begin(), vecResults.end(), sortResByIP);
				break;
		}
		// reserve buffer space to minimise allocations
		sPocessedBuf.reserve(sPocessedBuf.length()*2+vecResults.size()*nBufLen*2);
		//
		for (ResultVec::iterator it = vecResults.begin(); it!=vecResults.end(); ++it)
		{
			if (bListedLoop &&
			(pContext->pSetIDs==NULL ||
			 pContext->pSetIDs->find(it->dwID)==pContext->pSetIDs->end()))
			{
				// not in the set -- leave out
				continue;
			}
			if (m_nMaxResInGroup>0 &&
			    pContext->dwLoopCounter>=m_nMaxResInGroup &&
			    pContext->pResGroup) // only stop if we are inside Groups loop
			{
				// stop looping
				return;
			}
			pContext->pResult = &(*it); // gcc-3 stl...
			pContext->dwLoopCounter++;
			process_buff(pLoopBuf, nBufLen, sPocessedBuf, true, false, pContext);
		}
	}
	else
	{
		// this will look like "error-message"
		pContext->dwLoopCounter++;
		process_buff(pLoopBuf, nBufLen, sPocessedBuf, true, false, pContext);
	}
}

bool sortGrpBySize(const ResultGroup& g1, const ResultGroup& g2)
{
	// sort decending by size, than decending by speed, than accending by name
	if (g1.Size>g2.Size)
		return true;
	if (g1.Size<g2.Size)
		return false;
	if (g1.AvgSpeed>g2.AvgSpeed)
		return true;
	if (g1.AvgSpeed<g2.AvgSpeed)
		return false;
	if (g1.Name<g2.Name)
		return true;
	// equivalent elements return false
	return false;
}

bool sortGrpBySpeed(const ResultGroup& g1, const ResultGroup& g2)
{
	// sort decending by speed, than decending by size, than accending by name
	if (g1.AvgSpeed>g2.AvgSpeed)
		return true;
	if (g1.AvgSpeed<g2.AvgSpeed)
		return false;
	if (g1.Size>g2.Size)
		return true;
	if (g1.Size<g2.Size)
		return false;
	if (g1.Name<g2.Name)
		return true;
	// equivalent elements return false
	return false;
}

bool sortGrpByName(const ResultGroup& g1, const ResultGroup& g2)
{
	// sort accending by name, than decending by size, than decending by speed
	if (g1.Name<g2.Name)
		return true;
	if (g1.Name>g2.Name)
		return false;
	if (g1.Size>g2.Size)
		return true;
	if (g1.Size<g2.Size)
		return false;
	if (g1.AvgSpeed>g2.AvgSpeed)
		return true;
	// equivalent elements return false
	return false;
}

void MUIRemotePriv::loopGroups(const char * pLoopBuf, int nBufLen, CString & sPocessedBuf, bool bListedLoop, SLoopContext* pContext)
{
	SGnuSearch gs;
	SSearchRes sr;
	if ( pContext->pSearch!=NULL && pContext->pSearchRes==NULL &&
		 m_pController->GetSearchByID(pContext->pSearch->m_dwID, gs, sr.vecRes, sr.vecGrp))
	{
		pContext->pSearchRes = &sr;
		pContext->pSearch = &gs;
	}
	// we have all the vecotrs and we dont have to call "for each"
	// there is no "for-each-result" anyway
	if (pContext->pSearchRes)
	{
		// sort the list first
		switch (pContext->nResultsSortOrder){
			case RSO_Size: stable_sort(pContext->pSearchRes->vecGrp.begin(), pContext->pSearchRes->vecGrp.end(), sortGrpBySize);
				break;
			case RSO_Name: stable_sort(pContext->pSearchRes->vecGrp.begin(), pContext->pSearchRes->vecGrp.end(), sortGrpByName);
				break;
			case RSO_Speed: stable_sort(pContext->pSearchRes->vecGrp.begin(), pContext->pSearchRes->vecGrp.end(), sortGrpBySpeed);
				break;
		}
		// reserve buffer space to minimise allocations
		sPocessedBuf.reserve(sPocessedBuf.length()*2+pContext->pSearchRes->vecGrp.size()*nBufLen*2);
		//
		for (ResultGroupVec::iterator it = pContext->pSearchRes->vecGrp.begin(); it!=pContext->pSearchRes->vecGrp.end(); ++it)
		{
			if (bListedLoop &&
			(pContext->pSetIDs==NULL ||
			 pContext->pSetIDs->find(it->dwID)==pContext->pSetIDs->end()))
			{
				// not in the set -- leave out
				continue;
			}
			pContext->pResGroup = &(*it);
			// enable result tags in the loop to display the first result details
			if ( it->ResultSet.size() &&
				 (*it->ResultSet.begin())<pContext->pSearchRes->vecRes.size()) // just a sanity check
			{
				pContext->pResult = &pContext->pSearchRes->vecRes[*it->ResultSet.begin()];
			}
			pContext->dwLoopCounter++;
			process_buff(pLoopBuf, nBufLen, sPocessedBuf, true, false, pContext);
		}
	}
	else
	{
		// this will look like "error-message"
		pContext->dwLoopCounter++;
		process_buff(pLoopBuf, nBufLen, sPocessedBuf, true, false, pContext);
	}
}

void MUIRemotePriv::loopEvents(const char * pLoopBuf, int nBufLen, CString & sPocessedBuf, bool bListedLoop, SLoopContext* pContext)
{
	if (m_pEventLog->HaveEvents())
	{
		MUIREventLog::queue_type que_copy;
		m_pEventLog->CopyQueue(que_copy);
		for (MUIREventLog::reverse_iterator ite = que_copy.rbegin(); ite != que_copy.rend(); ++ite)
		{
			pContext->pEvent = *ite;
			pContext->dwLoopCounter++;
			process_buff(pLoopBuf, nBufLen, sPocessedBuf, true, false, pContext);
			pContext->pEvent = NULL;
		}
	}
}

///////////////////////////////////////////////////////////////////////////////////////////////////
// few parsers

#define CONV_HEX_DIGIT(_d) \
	if (_d>='0'&&_d<='9')\
		_d -= '0';\
	else if (_d>='a'&&_d<='f')\
		_d -= 'a'-10;\
	else if (_d>='A'&&_d<='F')\
		_d -= 'A'-10;

CString restore_string(const CString& s)
{
	// change + to spaces
	CString o(s);
	int nPos=0;
	while ((nPos=o.find('+',nPos))>=0)
		o[nPos++]=' ';
	// and recover %xx hex values
	nPos = 0;
	while ((nPos=o.find('%',nPos))>=0 && nPos < o.length()-2)
	{
		u_char c1 = o[nPos+1],
			   c2 = o[nPos+2];
		if ( ((c1>='0'&&c1<='9')||(c1>='a'&&c1<='f')||(c1>='A'&&c1<='F')) &&
			 ((c2>='0'&&c2<='9')||(c2>='a'&&c2<='f')||(c2>='A'&&c2<='F')) )
		{
			o = o.substr(0,nPos+1) + o.substr(nPos+3);
			CONV_HEX_DIGIT(c1)
			CONV_HEX_DIGIT(c2)
			o[nPos] = (c1<<4) | c2;
		}
		++nPos;
	}
	return o;
}

bool MUIRemotePriv::parse_trail(const CString& trail, map<CString, CString>& values)
{
	if (0!=trail.find('?') || trail.length()<2)
		return false;
	// return parse_params(trail, "&", "=", values); will not work because of the "url encoding"
	list<CString> pairs;
	if (0==split_str(trail.substr(1,trail.length()),'&',pairs))
		return false;
	for(list<CString>::iterator it=pairs.begin(); it!=pairs.end(); ++it){
		list<CString> pair;
		list<CString>::iterator it1;
		switch (split_str(*it,'=',pair)){
			case 1: values[restore_string(*pair.begin())] = "";
				break;
			case 2:
				it1 = pair.begin();
				++it1;
				values[restore_string(*pair.begin())] = restore_string(*it1);
				break;
			default:
				TRACE("parse_trail: split returned something strange");
		}
	}
	return true;
}

void MUIRemotePriv::logAccess(int nTime, IP ipRemote, LPCSTR szMessage, LPCSTR szPath)
{
	CString s;
	s.format("%s\t %s\t %s\n", Ip2Str(ipRemote).c_str(), szMessage, szPath);
	POST_EVENT( MStringEvent(
				ET_MESSAGE,
				ES_NONE,
				s,
				MSG_REMOTE_ACCESS,
				MSGSRC_REMOTE_UI
			));
}

