#define _LARGEFILE64_SOURCE     /* required for GLIBC to enable stat64 and friends */
#include <sys/types.h>
#include <regex.h>
#include <string.h>
#include <strings.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <pwd.h>
#include <unistd.h>

#include "mt.h"
#include "mem.h"
#include "error.h"
#include "utils.h"
#include "color.h"
#include "term.h"
#include "exec.h"
#include "globals.h"
#include "config.h"

/* "local global" */
int cur_colorscheme_nr = -1;
int cur_filterscheme_nr = -1;
int cur_editscheme_nr = -1;

void do_load_config(char *dummy, char *file);


int config_yes_no(char *what)
{
	if (what[0] == '1' || strcasecmp(what, "yes") == 0 || strcasecmp(what, "on") == 0)
	{
		return 1;
	}

	return 0;
}

long long int kb_str_to_value(char *field, char *str)
{
	char *mult;
	long long int bytes = atoll(str);
	if (bytes < -1)
		error_exit("%s: value cannot be < -1\n", field);

	mult = &str[strlen(str) - 2];
	if (strcasecmp(mult, "kb") == 0)
		bytes *= 1024;
	else if (strcasecmp(mult, "mb") == 0)
		bytes *= 1024 * 1024;
	else if (strcasecmp(mult, "gb") == 0)
		bytes *= 1024 * 1024 * 1024;

	return bytes;
}

void add_cs_re(char *incmd, char *par)
{
	if (use_colors)
	{
		char *re = NULL, *val = NULL;
		char *cmd = &incmd[5];
		char *colon;

		if (strncmp(cmd, "_val", 4) == 0)
		{
			val = find_next_par(par);
			if (!val)
				error_exit("cs_re_val...-entry malformed: value missing\n");

			re = find_next_par(val);
		}
		else
			re = find_next_par(par);

		if (re == NULL)
			error_exit("cs_re-entry malformed: color or regular expression missing (%s:%s)\n", cmd, par);

		if (cur_colorscheme_nr == -1)
			error_exit("for cs_re one needs to define a colorscheme-name first\n");

		/* find colorscheme */
		if (cschemes[cur_colorscheme_nr].color_script.script)
			error_exit("one cannot let a colorscript have the same name has a colorscheme");

		/* grow/create list */
		cschemes[cur_colorscheme_nr].pentries = (color_scheme_entry *)myrealloc(cschemes[cur_colorscheme_nr].pentries, (cschemes[cur_colorscheme_nr].n + 1) * sizeof(color_scheme_entry), "do_load_config: color for regexp");

		/* add to list */
		if (cmd[0] == 0x00)
			cschemes[cur_colorscheme_nr].pentries[cschemes[cur_colorscheme_nr].n].flags = 0;
		else if (strcmp(cmd, "_s") == 0)
			cschemes[cur_colorscheme_nr].pentries[cschemes[cur_colorscheme_nr].n].flags = CSREFLAG_SUB;
		else if (strcmp(cmd, "_val_less") == 0)
			cschemes[cur_colorscheme_nr].pentries[cschemes[cur_colorscheme_nr].n].flags = CSREFLAG_CMP_VAL_LESS;
		else if (strcmp(cmd, "_val_bigger") == 0)
			cschemes[cur_colorscheme_nr].pentries[cschemes[cur_colorscheme_nr].n].flags = CSREFLAG_CMP_VAL_BIGGER;
		else if (strcmp(cmd, "_val_equal") == 0)
			cschemes[cur_colorscheme_nr].pentries[cschemes[cur_colorscheme_nr].n].flags = CSREFLAG_CMP_VAL_EQUAL;

		/* sanity check */
		if (cmd[0] != 0x00 && strchr(re, '(') == NULL)
			error_exit("%s without substring selections! ('(' and ')')\n", cmd);

		if (val) cschemes[cur_colorscheme_nr].pentries[cschemes[cur_colorscheme_nr].n].cmp_value = atof(val);
		cschemes[cur_colorscheme_nr].pentries[cschemes[cur_colorscheme_nr].n].merge_color = incmd[0] == 'm' ? MY_TRUE : MY_FALSE;

		cschemes[cur_colorscheme_nr].pentries[cschemes[cur_colorscheme_nr].n].cdef.ac_index = 0;
		colon = strchr(par, '|');
		if (colon)
		{
			*colon = 0x00;
			cschemes[cur_colorscheme_nr].pentries[cschemes[cur_colorscheme_nr].n].cdef.use_alternating_colors = MY_TRUE;
			cschemes[cur_colorscheme_nr].pentries[cschemes[cur_colorscheme_nr].n].cdef.attrs2 = parse_attributes(colon + 1);
		}
		else
		{
			cschemes[cur_colorscheme_nr].pentries[cschemes[cur_colorscheme_nr].n].cdef.use_alternating_colors = MY_FALSE;
		}
		cschemes[cur_colorscheme_nr].pentries[cschemes[cur_colorscheme_nr].n].cdef.attrs1 = parse_attributes(par);

		/* compile regular expression */
		compile_re(&cschemes[cur_colorscheme_nr].pentries[cschemes[cur_colorscheme_nr].n].regex, re);

		cschemes[cur_colorscheme_nr].n++;
	}
}

void add_colorscheme(char *cmd, char *par)
{
	if (use_colors)
	{
		char *descr = find_next_par(par);

		cur_colorscheme_nr = n_cschemes++;

		cschemes = (color_scheme *)myrealloc(cschemes, n_cschemes * sizeof(color_scheme), "do_load_config: list of colorschemes");
		memset(&cschemes[cur_colorscheme_nr], 0x00, sizeof(color_scheme));

		cschemes[cur_colorscheme_nr].descr = mystrdup(descr?descr:"", "do_load_config: cscheme descr");
		cschemes[cur_colorscheme_nr].name  = mystrdup(par, "do_load_config: do_load_config cs_re");
	}
}

void add_colorscript(char *cmd, char *par)
{
	char *dummy1, *dummy2;

	dummy1 = find_next_par(par); /* find script */
	dummy2 = find_next_par(dummy1); /* find description */

	cur_colorscheme_nr = n_cschemes++;

	cschemes = (color_scheme *)myrealloc(cschemes, n_cschemes * sizeof(color_scheme), "do_load_config: list of colorschemes");
	memset(&cschemes[cur_colorscheme_nr], 0x00, sizeof(color_scheme));

	cschemes[cur_colorscheme_nr].name  = mystrdup(par, "do_load_config: do_load_config cs_re");
	cschemes[cur_colorscheme_nr].descr = mystrdup(dummy2, "do_load_config: cscheme descr");
	cschemes[cur_colorscheme_nr].color_script.script = mystrdup(dummy1, "do_load_config: name of script");

	cur_colorscheme_nr = -1;
}

void add_editscheme(char *cmd, char *par)
{
	char *descr = find_next_par(par);

	pes = (editscheme *)myrealloc(pes, sizeof(editscheme) * (n_es + 1), "do_load_config: list of editschemes");
	memset(&pes[n_es], 0x00, sizeof(editscheme));

	pes[n_es].es_name = mystrdup(par, "es name");
	pes[n_es].es_desc = mystrdup(descr?descr:"", "es descr");
	cur_editscheme_nr = n_es++;
}

void add_editrule(char *cmd, char *par)
{
	char *type_str = par;
	char *par1 = find_next_par(type_str);
	char *par2 = NULL;
	striptype_t type = STRIP_TYPE_REGEXP;
	int rule_index = -1;

	if (!par1)
		error_exit("editrule:%s requires a parameter\n", type_str);

	if (strcmp(type_str, "kr") == 0)
		type = STRIP_TYPE_RANGE;
	else if (strcmp(type_str, "ke") == 0)
		type = STRIP_TYPE_REGEXP;
	else if (strcmp(type_str, "kc") == 0)
		type = STRIP_TYPE_COLUMN;
	else
		error_exit("editrule requirs either ke, kr or kc\n");

	if (type == STRIP_TYPE_RANGE || type == STRIP_TYPE_COLUMN)
	{
		par2 = find_next_par(par1);
		if (!par2)
			error_exit("editrule:%s requires another parameter\n", type_str);
	}

	rule_index = pes[cur_editscheme_nr].n_strips;
	pes[cur_editscheme_nr].strips = (strip_t *)myrealloc(pes[cur_editscheme_nr].strips, (pes[cur_editscheme_nr].n_strips + 1) * sizeof(strip_t), "list of editscheme");
	memset(&pes[cur_editscheme_nr].strips[pes[cur_editscheme_nr].n_strips], 0x00, sizeof(strip_t));
	pes[cur_editscheme_nr].n_strips++;

	pes[cur_editscheme_nr].strips[rule_index].type = type;

	if (type == STRIP_TYPE_RANGE)
	{
		pes[cur_editscheme_nr].strips[rule_index].start = atoi(par1);
		pes[cur_editscheme_nr].strips[rule_index].end   = atoi(par2);
	}
	else if (type == STRIP_TYPE_REGEXP)
	{
		pes[cur_editscheme_nr].strips[rule_index].regex_str = mystrdup(par1, "regexp str");
		compile_re(&pes[cur_editscheme_nr].strips[rule_index].regex, par1);
	}
	else if (type == STRIP_TYPE_COLUMN)
	{
		pes[cur_editscheme_nr].strips[rule_index].del = mystrdup(par1, "delimiter");
		pes[cur_editscheme_nr].strips[rule_index].col_nr = atoi(par2);
	}
	else
		error_exit("internal error\n");
}

void add_filterscheme(char *cmd, char *par)
{
	char *descr = find_next_par(par);

	pfs = (filterscheme *)myrealloc(pfs, sizeof(filterscheme) * (n_fs + 1), "do_load_config: list of filterschemes");
	memset(&pfs[n_fs], 0x00, sizeof(filterscheme));

	pfs[n_fs].fs_name = mystrdup(par, "fs name");
	pfs[n_fs].fs_desc = mystrdup(descr?descr:"", "fs descr");
	cur_filterscheme_nr = n_fs++;
}

void add_filterscheme_rule(char *cmd, char *par)
{
	char *type = par;
	int use_regex = 0x00;
	char *re_cmd = NULL;
	char *re_str = find_next_par(par);
	if (!re_str)
		error_exit("missing regular expression in rule-line for scheme %s\n", pfs[cur_filterscheme_nr].fs_name);

	if (type[0] != 'e')
		error_exit("regular expression type '%s' is not recognized\n", type);

	if (type[1] == 0x00 || type[1] == 'm')
		use_regex = 'm';
	else if (type[1] == 'v')
		use_regex = 'v';
	else if (type[1] == 'c')
		use_regex = 'c';
	else if (type[1] == 'C')
		use_regex = 'C';
	else if (toupper(type[1]) == 'X')
	{
		char *dummy = find_next_par(re_str);
		if (!dummy)
			error_exit("missing command for rule of type 'e%c' for scheme %s\n", type[1], pfs[cur_filterscheme_nr].fs_name);

		re_cmd = mystrdup(dummy, "command");
		use_regex = type[1];

		if (use_regex == 'X' && (strchr(re_str, '(') == NULL || strchr(re_str, ')') == NULL))
			error_exit("filterscheme rule: -eX requires a regular expression which selectes a substring using '(' and ')'\n");
	}

	pfs[cur_filterscheme_nr].pre = (re *)myrealloc(pfs[cur_filterscheme_nr].pre, (pfs[cur_filterscheme_nr].n_re + 1) * sizeof(re), "list of regexps for filterscheme");
	memset(&pfs[cur_filterscheme_nr].pre[pfs[cur_filterscheme_nr].n_re], 0x00, sizeof(re));
	pfs[cur_filterscheme_nr].pre[pfs[cur_filterscheme_nr].n_re].use_regex = use_regex;
	pfs[cur_filterscheme_nr].pre[pfs[cur_filterscheme_nr].n_re].regex_str = mystrdup(re_str, "regexp str");
	compile_re(&pfs[cur_filterscheme_nr].pre[pfs[cur_filterscheme_nr].n_re].regex, re_str);
	pfs[cur_filterscheme_nr].pre[pfs[cur_filterscheme_nr].n_re].cmd = re_cmd;
	pfs[cur_filterscheme_nr].n_re++;
}

void add_convert(char *cmd, char *par)
{
	char *conv_name = par;
	char *conv_type = NULL;
	conversion_t type = 0;
	char *conv_script = NULL;
	char *conv_re = NULL;
	int loop;

	/* parse type */
	conv_type = find_next_par(conv_name);
	if (!conv_type)
		error_exit("'convert'-entry malformed: conversion type missing\n");

	if (strncmp(conv_type, "script:", 7) == 0)
	{
		conv_script = find_next_par(conv_type);
		if (conv_script)
			conv_re = find_next_par(conv_script);
		else
			error_exit("convert: script missing\n");
	}
	else
		conv_re = find_next_par(conv_type);

	if (!conv_re)
		error_exit("'convert'-entry malformed: type or regular expression missing\n");

	/* find this conversion: is it from an already known group? */
	for(loop=0; loop<n_conversions; loop++)
	{
		if (strcmp(conversions[loop].name, conv_name) == 0)
			break;
	}
	/* create new group */
	if (loop == n_conversions)
	{
		n_conversions++;
		conversions = (conversion *)myrealloc(conversions, sizeof(conversion) * n_conversions, "list of conversions-sets");
		memset(&conversions[loop], 0x00, sizeof(conversion));
		conversions[loop].name = mystrdup(conv_name, "conversion name");
	}

	if (strcmp(conv_type, "ip4tohost") == 0)
		type = CONVTYPE_IP4TOHOST;
	else if (strcmp(conv_type, "epochtodate") == 0)
		type = CONVTYPE_EPOCHTODATE;
	else if (strcmp(conv_type, "errnotostr") == 0)
		type = CONVTYPE_ERRNO;
	else if (strcmp(conv_type, "hextodec") == 0)
		type = CONVTYPE_HEXTODEC;
	else if (strcmp(conv_type, "dectohex") == 0)
		type = CONVTYPE_DECTOHEX;
	else if (strcmp(conv_type, "tai64todate") == 0)
		type = CONVTYPE_TAI64NTODATE;
	else if (strcmp(conv_type, "script") == 0)
		type = CONVTYPE_SCRIPT;
	else if (strcmp(conv_type, "abbrtok") == 0)
		type = CONVTYPE_ABBRTOK;
	else
		error_exit("convert %s: '%s' is a not recognized conversion type\n", conv_name, conv_type);

	conversions[loop].pcb = (conversion_bundle_t *)myrealloc(conversions[loop].pcb, sizeof(conversion_bundle_t) * (conversions[loop].n + 1), "list of conversion-types per set");
	conversions[loop].pcs = (script *)myrealloc(conversions[loop].pcs, sizeof(script) * (conversions[loop].n + 1), "list of scripts for conversions");

	conversions[loop].pcb[conversions[loop].n].type = type;


	memset(&conversions[loop].pcs[conversions[loop].n], 0x00, sizeof(script));
	conversions[loop].pcs[conversions[loop].n].script = conv_script?mystrdup(conv_script, "script name for conversion"):NULL;

	compile_re(&conversions[loop].pcb[conversions[loop].n].regex, conv_re);
	conversions[loop].pcb[conversions[loop].n].match_count = 0;
	conversions[loop].n++;
}

void use_filterscheme(char *cmd, char *par)
{
	char *re, *cur_fs = par;
	int fs;

	if ((re = find_next_par(par)) == NULL)		/* format: cs_re:color:regular expression */
		error_exit("scheme-entry malformed: schemename or regular expression missing (%s:%s)\n", cmd, par);

	/* find colorschemes */
	for(;;)
	{
		char *colon = strchr(cur_fs, ',');
		if (colon) *colon = 0x00;

		fs = find_filterscheme(cur_fs);
		if (fs == -1)
			error_exit("filterscheme '%s' is unknown: you should first define a filterscheme before using it\n", par);

		add_pars_per_file(re, NULL, -1, -1, -1, fs, -1, NULL);

		if (!colon)
			break;

		cur_fs = colon + 1;
	}
}

void use_editscheme(char *cmd, char *par)
{
	char *re, *cur_es = par;
	int es;

	if ((re = find_next_par(par)) == NULL)		/* format: cs_re:color:regular expression */
		error_exit("editscheme-entry malformed: editscheme or regular expression missing (%s:%s)\n", cmd, par);

	/* find editscheme */
	for(;;)
	{
		char *colon = strchr(cur_es, ',');
		if (colon) *colon = 0x00;

		es = find_editscheme(cur_es);
		if (es == -1)
			error_exit("editscheme '%s' is unknown: you should first define a editscheme before using it\n", par);

		add_pars_per_file(re, NULL, -1, -1, -1, -1, es, NULL);

		if (!colon)
			break;

		cur_es = colon + 1;
	}
}

void set_default_convert(char *cmd, char *par)
{
	char *re;
	char *cur_cv = par;

	if ((re = find_next_par(par)) == NULL)		/* format: cs_re:color:regular expression */
		error_exit("default_convert-entry malformed: schemename or regular expression missing (%s:%s)\n", cmd, par);
	
	for(;;)
	{
		char *colon = strchr(cur_cv, ',');
		if (colon) *colon = 0x00;

		add_pars_per_file(re, NULL, -1, -1, -1, -1, -1, cur_cv);

		if (!colon)
			break;

		cur_cv = colon + 1;
	}
}

void scheme(char *cmd, char *par)
{
	if (use_colors)
	{
		char *re;
		char *cur_cs = par;

		if ((re = find_next_par(par)) == NULL)		/* format: cs_re:color:regular expression */
			error_exit("scheme-entry malformed: schemename or regular expression missing (%s:%s)\n", cmd, par);

		/* find colorschemes */
		for(;;)
		{
			char *colon = strchr(cur_cs, ',');
			if (colon) *colon = 0x00;

			add_pars_per_file(re, cur_cs, -1, -1, -1, -1, -1, NULL);

			if (!colon)
				break;

			cur_cs = colon + 1;
		}
	}
}

void set_defaultcscheme(char *cmd, char *par)
{
	defaultcscheme = mystrdup(par, "do_load_config defaultcscheme");
}

void bind_char(char *cmd, char *par)
{
	char *proc;

	if ((proc = find_next_par(par)) == NULL)
		error_exit("bind-entry malformed: key or binding missing (%s:%s)\n", cmd, par);

	if (strlen(par) > 2) error_exit("bind-entry malformed: unknown keyselection (%s:%s)\n", cmd, par);

	keybindings = (keybinding *)myrealloc(keybindings, sizeof(keybinding) * (n_keybindings + 1), "keybindings array");

	if (par[0] == '^')
		keybindings[n_keybindings].key = toupper(par[1]) - 'A' + 1;
	else
		keybindings[n_keybindings].key = par[0];

	keybindings[n_keybindings].command = mystrdup(proc, "do_load_config: keybinding command");
	n_keybindings++;
}

void set_suppress_empty_lines(char *cmd, char *par)
{
	suppress_empty_lines = config_yes_no(par);
}

void set_close_closed_windows(char *cmd, char *par)
{
	do_not_close_closed_windows = !config_yes_no(par);
}

void set_follow_filename(char *cmd, char *par)
{
	default_follow_filename = config_yes_no(par);
}

void set_default_linewrap(char *cmd, char *par)
{
	default_linewrap = par[0];

	if (default_linewrap == 'o')
	{
		char *dummy = find_next_par(par);
		if (dummy)
			default_line_wrap_offset = atoi(dummy);
		else
			error_exit("default_linewrap:o needs a wrap offset parameter\n");
	}
}

void set_umask(char *cmd, char *par)
{
	def_umask = strtol(par, NULL, 0);
}

void set_shell(char *cmd, char *par)
{
	shell = mystrdup(par, "do_load_config shell");
}

void set_statusline_above_data(char *cmd, char *par)
{
	statusline_above_data = config_yes_no(par);
}

void set_caret_notation(char *cmd, char *par)
{
	caret_notation = config_yes_no(par);
}

void set_searches_case_insensitive(char *cmd, char *par)
{
	re_case_insensitive = MY_TRUE;
}

void set_beep_method(char *cmd, char *par)
{
	if (strcmp(par, "beep") == 0)
		beep_method = BEEP_BEEP;
	else if (strcmp(par, "flash") == 0)
		beep_method = BEEP_FLASH;
	else if (strcmp(par, "popup") == 0)
		beep_method = BEEP_POPUP;
	else if (strcmp(par, "none") == 0)
		beep_method = BEEP_NONE;
	else
		error_exit("'%s' is an unknown beep method\n");
}

void set_beep_popup_length(char *cmd, char *par)
{
	beep_popup_length = atof(par);
	if (beep_popup_length < 0.0)
		error_exit("beep_popup_length must be >= 0.0\n");
}

void set_allow_8bit(char *cmd, char *par)
{
	allow_8bit = config_yes_no(par);
}

void set_dsblwm(char *cmd, char *par)
{
	no_linewrap = 1 - config_yes_no(par);
}

void set_warn_closed(char *cmd, char *par)
{
	warn_closed = config_yes_no(par);
}

void set_basename(char *cmd, char *par)
{
	filename_only = config_yes_no(par);
}

void set_bright(char *cmd, char *par)
{
	bright_colors = config_yes_no(par);
}

void set_ts_format(char *cmd, char *par)
{
	ts_format = mystrdup(par, "do_load_config: timestamp format");
}

void set_cnv_ts_format(char *cmd, char *par)
{
	cnv_ts_format = mystrdup(par, "do_load_config: conversion timestamp format");
}

void set_statusline_ts_format(char *cmd, char *par)
{
	statusline_ts_format = mystrdup(par, "do_load_config: statusline timestamp format");
}

void set_markerline_attrs(char *cmd, char *par)
{
	markerline_attrs = parse_attributes(par);
}

void set_idleline_color(char *cmd, char *par)
{
	idleline_attrs = parse_attributes(par);
}

void set_msgline_color(char *cmd, char *par)
{
	msgline_attrs = parse_attributes(par);
}

void set_changeline_color(char *cmd, char *par)
{
	changeline_attrs = parse_attributes(par);
}

void set_statusline_attrs(char *cmd, char *par)
{
	statusline_attrs = parse_attributes(par);
}

void set_splitline_attrs(char *cmd, char *par)
{
	splitline_attrs = parse_attributes(par);
}

void set_inverse_attrs(char *cmd, char *par)
{
	inverse_attrs = attrstr_to_nr(par);
}

void set_splitline(char *cmd, char *par)
{
	if (strcmp(par, "none") == 0)
		splitline_mode = SL_NONE;
	else if (strcmp(par, "regular") == 0)
		splitline_mode = SL_REGULAR;
	else if (strcmp(par, "attributes") == 0)
		splitline_mode = SL_ATTR;
	else
		error_exit("parameter '%s' to 'splitline' not recognized\n", par);
}

void set_show_subwindow_id(char *cmd, char *par)
{
	show_subwindow_id = config_yes_no(par);
}

void set_markerline_timestamp(char *cmd, char *par)
{
	timestamp_in_markerline = config_yes_no(par);
}

void set_global_default_nlines(char *cmd, char *par)
{
	default_maxnlines = atoi(par);
}

void set_global_default_nkb(char *cmd, char *par)
{
	default_maxbytes = kb_str_to_value(cmd, par);
}

void set_default_nlines(char *cmd, char *par)
{
	char *re;
	int n_lines;

	if ((re = find_next_par(par)) == NULL)		/* format: cs_re:color:regular expression */
		error_exit("scheme-entry malformed: schemename or regular expression missing (%s:%s)\n", cmd, par);

	/* find colorscheme */
	n_lines = atoi(par);
	if (n_lines < -1)
		error_exit("default_nlines: value cannot be < -1\n");

	add_pars_per_file(re, NULL, n_lines, -1, -1, -1, -1, NULL);
}

void set_default_mark_change(char *cmd, char *par)
{
	char *re;

	if ((re = find_next_par(par)) == NULL)		/* format: cs_re:color:regular expression */
		error_exit("scheme-entry malformed: scheme name or regular expression missing (%s:%s)\n", cmd, par);

	add_pars_per_file(re, NULL, -1, -1, config_yes_no(par), -1, -1, NULL);
}

void set_default_bytes(char *cmd, char *par)
{
	int bytes;
	char *re;

	if ((re = find_next_par(par)) == NULL)		/* format: cs_re:color:regular expression */
		error_exit("scheme-entry malformed: schemename or regular expression missing (%s:%s)\n", cmd, par);

	bytes = kb_str_to_value(cmd, par);

	add_pars_per_file(re, NULL, -1, bytes, -1, -1, -1, NULL);
}

void set_check_mail(char *cmd, char *par)
{
	check_for_mail = get_value_arg("check_mail", par, VAL_ZERO_POSITIVE);
}

void set_tab_stop(char *cmd, char *par)
{
	tab_width = atoi(par);
}

void set_tail(char *cmd, char *par)
{
	tail = mystrdup(par, "do_load_config: where to find tail");
}

void set_titlebar(char *cmd, char *par)
{
	set_title = mystrdup(par, "do_load_config: titlebar");
}

void set_abbreviate_filesize(char *cmd, char *par)
{
	afs = config_yes_no(par);
}

void set_replace_by_markerline(char *cmd, char *par)
{
	replace_by_markerline = mystrdup(par, "do_load_config: replace_by_markerline");
}

void set_popup_refresh_interval(char *cmd, char *par)
{
	popup_refresh_interval = get_value_arg("popup_refresh_interval", par, VAL_POSITIVE);
}

void set_msgline_char(char *cmd, char *par)
{
	msgline_char = par[0];
}

void set_idleline_char(char *cmd, char *par)
{
	idleline_char = par[0];
}

void set_changeline_char(char *cmd, char *par)
{
	changeline_char = par[0];
}

void set_markerline_char(char *cmd, char *par)
{
	markerline_char = par[0];
}

void set_global_mark_change(char *cmd, char *par)
{
	global_marker_of_other_window = config_yes_no(par);
}

void set_default_bufferwhat(char *cmd, char *par)
{
	char what = par[0];

	if (what != 'a' && what != 'f')
		error_exit("default_bufferwhat expects either 'a' or 'f' (got: '%c')\n", what);

	default_bufferwhat = what;
}

void set_abort_key(char *cmd, char *par)
{
	int dummy = atoi(par);

	if (dummy < 0 || dummy > 255)
		error_exit("abort_key expects an ascii value which is >= 0 && <= 255\n");

	abort_key = dummy;
}

void set_exit_key(char *cmd, char *par)
{
	int dummy = atoi(par);

	if (dummy < 0 || dummy > 255)
		error_exit("exit_key expects an ascii value which is >= 0 && <= 255\n");

	exit_key = dummy;
}

void set_line_ts_format(char *cmd, char *par)
{
	line_ts_format = mystrdup(par, "set_line_ts_format");
}

void set_default_min_shrink(char *cmd, char *par)
{
	default_min_shrink = get_value_arg("min_shrink", par, VAL_POSITIVE);
}

void set_scrollback_show_winnrs(char *cmd, char *par)
{
	default_sb_showwinnr = config_yes_no(par);
}

void  set_wordwrapmaxlength(char *cmd, char *par)
{
	wordwrapmaxlength = get_value_arg("wordwrapmaxlength", par, VAL_POSITIVE);
}

void set_searchhistory_file(char *cmd, char *par)
{
	if (par[0] == 0x00)
	{
		search_h.history_file = NULL;
		search_h.history_size = 0;
	}
	else
	{
		search_h.history_file = myrealpath(par);
	}
}

void set_searchhistory_size(char *cmd, char *par)
{
	int hs = atoi(par);

	if (hs <= 0)
	{
		search_h.history_file = NULL;
		search_h.history_size = 0;
	}
	else
	{
		search_h.history_size = hs;
	}
}

void set_cmdfile_history_file(char *cmd, char *par)
{
	if (par[0] == 0x00)
	{
		cmdfile_h.history_file = NULL;
		cmdfile_h.history_size = 0;
	}
	else
	{
		cmdfile_h.history_file = myrealpath(par);
	}
}

void set_cmdfile_history_size(char *cmd, char *par)
{
	int hs = atoi(par);

	if (hs <= 0)
	{
		cmdfile_h.history_file = NULL;
		cmdfile_h.history_size = 0;
	}
	else
	{
		cmdfile_h.history_size = hs;
	}
}

config_file_keyword cf_entries[] = {
	{ "abbreviate_filesize", set_abbreviate_filesize },
	{ "abort_key", set_abort_key },
	{ "allow_8bit", set_allow_8bit },
	{ "basename", set_basename },
	{ "beep_method", set_beep_method },
	{ "beep_popup_length", set_beep_popup_length },
	{ "bind", bind_char },
	{ "bright", set_bright },
	{ "caret_notation", set_caret_notation },
	{ "changeline_char", set_changeline_char },
	{ "changeline_color", set_changeline_color },
	{ "check_mail", set_check_mail },
	{ "close_closed_windows", set_close_closed_windows },
	{ "cmdfile_history_file", set_cmdfile_history_file },
	{ "cmdfile_history_size", set_cmdfile_history_size },
	{ "cnv_ts_format", set_cnv_ts_format },
	{ "colorscheme", add_colorscheme },
	{ "colorscript", add_colorscript },
	{ "convert", add_convert },
	{ "cs_re", add_cs_re },
	{ "cs_re_s", add_cs_re },
	{ "cs_re_val_bigger", add_cs_re },
	{ "cs_re_val_equal", add_cs_re },
	{ "cs_re_val_less", add_cs_re },
	{ "default_bufferwhat", set_default_bufferwhat },
	{ "default_bytes", set_default_bytes },
	{ "default_convert", set_default_convert },
	{ "default_linewrap", set_default_linewrap },
	{ "default_mark_change", set_default_mark_change },
	{ "default_nlines", set_default_nlines },
	{ "defaultcscheme", set_defaultcscheme },
	{ "dsblwm", set_dsblwm },
	{ "editrule", add_editrule },
	{ "editscheme", add_editscheme },
	{ "exit_key", set_exit_key },
	{ "filterscheme", add_filterscheme },
	{ "follow_filename", set_follow_filename },
	{ "global_default_nkb", set_global_default_nkb },
	{ "global_default_nlines", set_global_default_nlines },
	{ "global_mark_change", set_global_mark_change },
	{ "idleline_char", set_idleline_char },
	{ "idleline_color", set_idleline_color },
	{ "include", do_load_config },
	{ "inverse", set_inverse_attrs },
	{ "line_ts_format", set_line_ts_format },
	{ "markerline_char", set_markerline_char },
	{ "markerline_color", set_markerline_attrs },
	{ "markerline_timestamp", set_markerline_timestamp },
	{ "mcsre", add_cs_re },
	{ "mcsre_s", add_cs_re },
	{ "mcsre_val_bigger", add_cs_re },
	{ "mcsre_val_equal", add_cs_re },
	{ "mcsre_val_less", add_cs_re },
	{ "min_shrink", set_default_min_shrink },
	{ "msgline_char", set_msgline_char },
	{ "msgline_color", set_msgline_color },
	{ "popup_refresh_interval", set_popup_refresh_interval },
	{ "replace_by_markerline", set_replace_by_markerline },
	{ "rule", add_filterscheme_rule },
	{ "scheme", scheme },
	{ "scrollback_show_winnrs", set_scrollback_show_winnrs },
	{ "searches_case_insensitive", set_searches_case_insensitive },
	{ "searchhistory_file", set_searchhistory_file },
	{ "searchhistory_size", set_searchhistory_size },
	{ "shell", set_shell },
	{ "show_subwindow_id", set_show_subwindow_id },
	{ "splitline", set_splitline },
	{ "splitline_attrs", set_splitline_attrs },
	{ "statusline_above_data", set_statusline_above_data },
	{ "statusline_attrs", set_statusline_attrs },
	{ "statusline_ts_format", set_statusline_ts_format },
	{ "suppress_empty_lines", set_suppress_empty_lines },
	{ "tab_stop", set_tab_stop },
	{ "tail", set_tail },
	{ "titlebar", set_titlebar },
	{ "ts_format", set_ts_format },
	{ "umask", set_umask },
	{ "useeditscheme", use_editscheme },
	{ "usefilterscheme", use_filterscheme },
	{ "warn_closed", set_warn_closed },
	{ "wordwrapmaxlength", set_wordwrapmaxlength }
};

int find_config_entry_in_dispatch_table(char *cmd_name)
{
	int left = 0;
	int right = (sizeof(cf_entries) / sizeof(config_file_keyword)) - 1;

	while(left <= right)
	{
		int mid = (left + right) / 2;
		int compare = strcmp(cmd_name, cf_entries[mid].config_keyword);

		if (compare > 0)
			left = mid + 1;
		else if (compare < 0)
			right = mid - 1;
		else
			return mid;
	}

	return -1;
}

int config_file_entry(char *cmd)
{
	int function_nr;
	char *par = NULL;

	/* remove spaces at the beginning of the line */
	while (isspace(*cmd)) cmd++;

	/* skip empty lines and comments */
	if (cmd[0] == 0x00 || cmd[0] == '#' || cmd[0] == ';')
		return 0;

	/* lines are in the format of command:parameter */
	if ((par = find_next_par(cmd)) == NULL)
		error_exit("Malformed configline found: %s (command delimiter (:) missing)\n", cmd);

	function_nr = find_config_entry_in_dispatch_table(cmd);
	if (function_nr == -1)
		return -1;

	cf_entries[function_nr].function(cmd, par);

	return 0;
}

int sort_colorschemes_compare(const void *a, const void *b)
{
	color_scheme *pa = (color_scheme *)a;
	color_scheme *pb = (color_scheme *)b;

	return strcmp(pa -> name, pb -> name);
}

/* returns the default color scheme or -1 if none */
void do_load_config(char *dummy, char *file)
{
	static char sorted = 0;
	FILE *fh;
	int linenr = 0;

	/* given file */
	fh = fopen(file, "r");
	if (fh == NULL)
	{
		if (errno == ENOENT)	/* file does not exist, not an error */
			return;

		error_exit("do_load_config: error loading configfile '%s'\n", file);
	}

	for(;;)
	{
		char read_buffer[CONFIG_READ_BUFFER];
		char *dummy;
		char *cmd = fgets(read_buffer, sizeof(read_buffer) - 1, fh);
		if (!cmd)
			break;

		linenr++;

		/* strip LF */
		dummy = strchr(cmd, '\n');
		if (dummy)
			*dummy = 0x00;
		else
			error_exit("line %d of file '%s' is too long!\n", linenr, file);

		/* LOG("%d: %s\n", linenr, cmdin); */
		if (config_file_entry(cmd) == -1)
			error_exit("Configurationparameter %s is unknown (file: %s, line: %d)\n", read_buffer, file, linenr);
	}
	fclose(fh);

	if (!sorted)
	{
		sorted = 1;
		qsort(cschemes, n_cschemes, sizeof(color_scheme), sort_colorschemes_compare);
	}
}

void load_configfile_wrapper(char *config_file)
{
	/* load configurationfile (if any) */
	if (load_global_config)
		do_load_config(NULL, CONFIG_FILE);

	if (config_file)
	{
		do_load_config(NULL, config_file);
	}
	else
	{
		int path_max = find_path_max();
		char *path = mymalloc(path_max + 1, "path");
		char *home = getenv("HOME");
		struct passwd *pp = getuserinfo();

		if (home)
			snprintf(path, path_max, "%s/.multitailrc", home);
		else
			snprintf(path, path_max, "%s/.multitailrc", pp -> pw_dir);

		do_load_config(NULL, path);

		myfree(path);
	}
}

char load_configfile(char *config_file)
{
	static char config_loaded = 0;

	if (config_loaded == 0)
	{
		/* load configurationfile (if any) */
		load_configfile_wrapper(config_file);

		config_loaded = 1;

		return 1;
	}

	return 0;
}
