#define _LARGEFILE64_SOURCE	/* required for GLIBC to enable stat64 and friends */
#include <ctype.h>
#include <sys/types.h>
#include <pwd.h>
#include <dirent.h>
#include <errno.h>
#include <fcntl.h>
#include <limits.h>
#include <regex.h>
#if !defined(__APPLE__) && !defined(__CYGWIN__)
#include <search.h>
#endif
#include <math.h>
#include <signal.h>
#include <stdarg.h>
#include <stdio.h>
#include <limits.h>
#include <stdlib.h>
#include <string.h>
#include <locale.h>
#include <time.h>
#include <glob.h>
#include <sys/ioctl.h>
#include <sys/stat.h>
#ifndef AIX
#include <sys/termios.h> /* needed on Solaris 8 */
#endif
#include <sys/time.h>
#include <sys/wait.h>
#include <unistd.h>
#if defined(__GLIBC__) && ( __GLIBC__ < 2 || (__GLIBC__ == 2 && __GLIBC_MINOR__ < 2))
#include <sys/sysinfo.h>
#endif

#include "mt.h"
#include "globals.h"
#include "error.h"
#include "mem.h"
#include "utils.h"
#include "scrollback.h"
#include "help.h"
#include "term.h"
#include "cv.h"
#include "selbox.h"
#include "stripstring.h"
#include "color.h"
#include "misc.h"
#include "ui.h"
#include "exec.h"
#include "diff.h"
#include "config.h"
#include "cmdline.h"

/* #define KEYB_DEBUG */

void LOG(char *s, ...)
{
#ifdef _DEBUG
	va_list ap;
	FILE *fh = fopen("log.log", "a+");
	if (!fh)
	{
		endwin();
		error_exit("LOG: error logging\n");
	}

	va_start(ap, s);
	vfprintf(fh, s, ap);
	va_end(ap);

	fclose(fh);
#endif
}

void free_subentry(proginfo *entry)
{
	int loop;

	/* free up filters */
	for(loop=0; loop<entry -> n_strip; loop++)
		myfree((entry -> pstrip)[loop].del);
	myfree(entry -> pstrip);

	/* free all those allocated memory blocks */
	myfree(entry -> filename);
	myfree(entry -> field_del);
	if (entry -> status)
		delete_popup(entry -> status);
	if (entry -> data)
		delete_popup(entry -> data);

	/* free buffers for diff (if any) */
	if (entry -> bcur)
		delete_array(entry -> bcur, entry -> ncur);
	if (entry -> bprev)
		delete_array(entry -> bprev, entry -> nprev);

	/* delete regular expressions */
	for(loop=0; loop<entry -> n_re; loop++)
		free_re(&(entry -> pre)[loop]);
	myfree(entry -> pre);

	/* stop process */
	stop_process(entry -> pid);

	/* close pipe to (tail) process */
	myclose(entry -> fd);
	if (entry -> wfd != entry -> fd) myclose(entry -> wfd);
}

void buffer_replace_pi_pointers(int f_index, proginfo *org, proginfo *new)
{
	int loop;

	for(loop=0; loop<lb[f_index].curpos; loop++)
	{
		if ((lb[f_index].pi)[loop] == org)
		{
			if (!new)
			{
				myfree((lb[f_index].Blines)[loop]);
				(lb[f_index].Blines)[loop] = NULL;
				(lb[f_index].pi)[loop] = NULL;
			}
			else
			{
				(lb[f_index].pi)[loop] = new;
			}
		}
	}
}

char delete_entry(int f_index, proginfo *sub)
{
	char delete_all = 0;

	/* no children? then sub must be pointing to current */
	if (pi[f_index].next == NULL)
	{
		sub = NULL;
	}

	/* stop the process(es) we're watching ('tail' most of the time) */
	if (sub == NULL) /* delete all? */
	{
		proginfo *cur = &pi[f_index];

		do
		{
			free_subentry(cur);
			cur = cur -> next;
		}
		while(cur);

		/* free the subwindows (if any) */
		cur = pi[f_index].next;
		while(cur)
		{
			proginfo *dummy = cur -> next;
			myfree(cur);
			cur = dummy;
		}

		delete_all = 1;
	}
	else
	{
		free_subentry(sub);
	}

	/* remove entry from array */
	if (sub == NULL) /* delete entry in the main array */
	{
		int n_to_move = (nfd - f_index) - 1;

		/* free buffers */
		delete_array(lb[f_index].Blines, lb[f_index].curpos);

		if (n_to_move > 0)
		{
			int loop;

			/* update buffer proginfo-pointers */
			for(loop=f_index + 1; loop<nfd; loop++)
			{
				/* replace lb[f_index].pi -> pi[loop] met pi[loop-1] */
				buffer_replace_pi_pointers(loop, &pi[loop], &pi[loop - 1]);
			}

			/* prog info */
			memmove(&pi[f_index], &pi[f_index+1], sizeof(proginfo) * n_to_move);
			/* buffers */
			memmove(&lb[f_index], &lb[f_index+1], sizeof(buffer) * n_to_move);
		}

		nfd--;

		/*  shrink array */
		if (nfd == 0)	/* empty? */
		{
			myfree(pi);
			pi = NULL;
			myfree(lb);
			lb = NULL;
		}
		else		/* not empty, just shrink */
		{
			pi = (proginfo *)myrealloc(pi, nfd * sizeof(proginfo), "delete_entry: proginfo list");
			lb = (buffer *)myrealloc(lb, nfd * sizeof(buffer), "delete_entry: nfd list");
		}
	}
	else		/* delete sub */
	{
		if (sub != &pi[f_index])	/* not in main array? */
		{
			proginfo *parent = &pi[f_index];

			/* find parent of 'sub' */
			while (parent -> next != sub)
				parent = parent -> next;

			parent -> next = sub -> next;

			buffer_replace_pi_pointers(f_index, sub, NULL);

			myfree(sub);
		}
		else				/* main array, find next */
		{
			proginfo *oldnext = pi[f_index].next;
			/* first, delete the entries of that entry (which is in the array) */
			buffer_replace_pi_pointers(f_index, &pi[f_index], NULL);
			/* then, 'rename' the pointers (original address -> address in array)... */
			buffer_replace_pi_pointers(f_index, oldnext, &pi[f_index]);
			/* and move the object to the array */
			memmove(&pi[f_index], oldnext, sizeof(proginfo));
			/* free the obsolete entry */
			myfree(oldnext);
		}
	}

	return delete_all;
}

/** do_exit
 * - in:      int sig
 * - returns: nothing (doesn't return!)
 * this function is called when MultiTail receives the TERM-signal. it is also
 * called by, for example, wait_for_keypress when ^c is pressed. it stops all
 * child-processes, ends the curses-library and exits with SUCCESS status
 */
void do_exit(void)
{
	proginfo *cur;
	int loop;

	/* kill tail processes */
	for(loop=0; loop<nfd; loop++)
	{
		cur = &pi[loop];

		do
		{
			int r_i;

			for(r_i = 0; r_i < cur -> n_redirect; r_i++)
			{
				if ((cur -> predir)[r_i].type != 0)
				{
					close((cur -> predir)[r_i].fd);

					if ((cur -> predir)[r_i].type == REDIRECTTO_PIPE_FILTERED || (cur -> predir)[r_i].type == REDIRECTTO_PIPE)
						stop_process((cur -> predir)[r_i].pid);
				}
			}

			stop_process(cur -> pid);

			cur = cur -> next;
		}
		while(cur);
	}

	if (set_title)
		gui_window_header("");

#ifdef _DEBUG
	/* to be able to check for memory leaks, (try to :-]) free up the whole
	 * array of windows and such. normally, don't do this: libc & kernel will
	 * take care of that
	 */
	dump_mem(SIGHUP);
#endif

	endwin();

	exit(EXIT_SUCCESS);
}

void signal_handler(int sig)
{
	signal(sig, SIG_IGN);

	switch(sig)
	{
		case SIGTERM:
			do_exit();
			break;
	
		case SIGHUP:
			got_sighup = 1;
			break;

		case SIGWINCH:
			terminal_changed = 1;
			break;

		default:
			error_exit("unexpected signal %d received\n", sig);
	}

	signal(sig, signal_handler);
}

void select_display_start_and_end(char *string, char mode, int wrap_offset, int win_width, int *prt_start, int *prt_end)
{
	int nbytes = strlen(string);

	*prt_start = 0;
	*prt_end = nbytes;

	/* figure out what to display (because of linewraps and such) */
	if (mode == 'a')
	{
		/* default is everything */
	}
	else if (mode == 'l')
	{
		*prt_end = min(nbytes, win_width);
	}
	else if (mode == 'r')
	{
		*prt_start = max(0, *prt_end - win_width);
	}
	else if (mode == 'S')
	{
		*prt_start = find_char_offset(string, ':');
		if (*prt_start == -1)
			*prt_start = 0;
	}
	else if (mode == 's')
	{
		if (nbytes > 16)
			*prt_start = find_char_offset(&string[16], ' ');
		else
			*prt_start = -1;

		if (*prt_start == -1)
			*prt_start = 0;
	}
	else if (mode == 'o')
	{
		*prt_start = min(wrap_offset, nbytes);
	}
}

int find_in_regexec_resultset(regmatch_t *matches, int set_size, int offset)
{
	int index = -1;
	int loop;

	for(loop=0; loop<set_size; loop++)
	{
		if (offset >= matches[loop].rm_so && offset < matches[loop].rm_eo)
		{
			index = loop;
			break;
		}
		else if (matches[loop].rm_so == -1)
		{
			break;
		}
	}

	return index;
}

void draw_tab(NEWWIN *win)
{
	if (tab_width)
	{
		int cury, curx, move, loop;

		/* get current coordinate */
		getyx(win -> win, cury, curx);

		/* calculate new coordinate */
		move = (((curx / tab_width) + 1) * tab_width) - curx;

		for(loop=0; loop<move; loop++)
			waddch(win -> win, ' ');
	}
}

void do_color_print(proginfo *cur, char *use_string, int prt_start, int prt_end, color_offset_in_line *cmatches, int n_cmatches, char start_reverse, regmatch_t *matches, int matching_regex, char use_regex, NEWWIN *win)
{
	int offset;
	myattr_t cdev = { -1, -1 };
	char reverse = start_reverse;
	char default_reverse_state = 0;
	int n_chars;

	/* find things to invert */
	if (use_regex == 'c' || use_regex == 'C' || use_regex == 'b')
	{
		if (matches != NULL)
		{
			/* is this offset in the string in the given (-> function-parameter)
			 * resultset? then invert
			 */
			if ((cur -> pre)[matching_regex].invert_regex == 0)
				default_reverse_state = 1;
		}
		else
		{
			if ((cur -> pre)[matching_regex].invert_regex)
				default_reverse_state = 1;
		}
	}

	use_string += prt_start;
	n_chars = prt_end - prt_start;

	/* print text */
	for(offset=0; offset<n_chars; offset++)
	{
		char re_inv = default_reverse_state;
		char is_control_or_extended_ascii = 0;
		unsigned char current_char = (unsigned char)*use_string++;

		/* find things to colorize */
		if (cur -> colorize == 'S' || cur -> colorize == 'T')
		{
			register int cmatches_index = 0;
			myattr_t new_cdev = { -1, -1 };

			/* if there's a list of offsets where colors should be displayed, check if the
			 * current offset in the string (the one we're going to display) is somewhere
			 * in that list off offsets
			 */
			for(cmatches_index=0; cmatches_index<n_cmatches; cmatches_index++)
			{
				if (offset >= cmatches[cmatches_index].start && offset < cmatches[cmatches_index].end)
				{
					new_cdev = cmatches[cmatches_index].attrs;
					break;
				}
			}

			/* new color selected? then switch off previous color */
			if (cdev.colorpair_index != new_cdev.colorpair_index || cdev.attrs != new_cdev.attrs)
			{
				myattr_off(win, cdev);

				if (new_cdev.colorpair_index != -1 || new_cdev.attrs != -1)
				{
					myattr_on(win, new_cdev);

					if (new_cdev.attrs != -1 && new_cdev.attrs & A_REVERSE)
						reverse = 1;
				}
				else
				{
					reverse = 0;
				}

				cdev = new_cdev;
			}
		}

		/* control-characters will be displayed as an inverse '.' */
		if (current_char < 32  || current_char == 127 || (current_char > 127 && !allow_8bit))
		{
			is_control_or_extended_ascii = 1;

			if (current_char != 10 && current_char != 13 && (current_char != 9 && tab_width > 0))
			{
				re_inv = 1;
			}
		}

		if (re_inv)
		{
			if (!reverse)
			{
				inverse_on(win);
				reverse = 1;
			}
		}

		if (!is_control_or_extended_ascii)
		{
			waddch(win -> win, (const chtype)current_char);
		}
		else
		{
			if (current_char == 9)	/* TAB? */
			{
				draw_tab(win);
			}
			else if (current_char == 10)
			{
				waddch(win -> win, '\n');
			}
			else if (current_char == 13)
			{
				/* silently ignore */
			}
			else if (current_char > 126)
			{
				waddch(win -> win, '.');
			}
			else
			{
				if (caret_notation)
					wprintw(win -> win, "^%c", 'a' + current_char - 1);
				else
					waddch(win -> win, '.');
			}
		}

		if (re_inv && reverse)
		{
			inverse_off(win);
			reverse = 0;
		}
	}

	if (cur -> colorize)
		myattr_off(win, cdev);
}

void color_print(NEWWIN *win, proginfo *cur, char *string, regmatch_t *matches, int matching_regex, mybool_t force_to_winwidth)
{
	char reverse = 0;
	myattr_t cdev = { -1, -1 };
	int prt_start = 0, prt_end;
	int mx = -1;
	char use_regex = 0;
	color_offset_in_line *cmatches = NULL;
	int n_cmatches = 0;
	char *use_string = NULL;
	int y, x;

	/* stop if there's no window tou output too */
	if (!win)
		return;

	/* check if the cursor is not at the beginning of the line
	 * if it is not and it is also not at the most right position, move it
	 * to the left (when it is already at the right most position, it'll move
	 * to the left itself when emitting text)
	 */
	mx = getmaxx(win -> win);
	getyx(win -> win, y, x);
	if (x != 0 && x != mx)
		wprintw(win -> win, "\n");

	/* cur == NULL? Markerline! */
	if (IS_MARKERLINE(cur))
	{
		draw_marker_line(win, string, cur);

		return;
	}

	if (matching_regex != -1)
	{
		use_regex = (cur -> pre)[matching_regex].use_regex;
	}

	/* select color or generate list of offset for colors */
	if (cur -> colorize) /* no need to do this for b/w terminals */
	{
		/* choose_color not only generates a list of colors but if there are terminal-
		 * codes in that string which set a color those are stripped as well
		 * the stripping part should be moved to a seperate function
		 */
		cdev = choose_color(string, cur, &cmatches, &n_cmatches, &use_string);

		/* if not using colorschemes (which can have more then one color
		 * per line), set the color
		 */
		if (cur -> colorize != 'S' && cur -> colorize != 'T')
		{
			myattr_on(win, cdev);

			if (cdev.attrs != -1 && cdev.attrs & A_REVERSE)
				reverse = 1;
		}
	}

	if (!use_string)
		use_string = mystrdup(string, "color_print: disp string");

	/* select start and end of string to display */
	select_display_start_and_end(use_string, force_to_winwidth == MY_TRUE?'l':cur -> line_wrap, cur -> line_wrap_offset, mx, &prt_start, &prt_end);

	/* and display on terminal */
	do_color_print(cur, use_string, prt_start, prt_end, cmatches, n_cmatches, reverse, matches, matching_regex, use_regex, win);

	myattr_off(win, cdev);

	myfree(use_string);
	myfree(cmatches);
}

void check_filter_exec(char *cmd, char *matching_string)
{
	int str_index, par_len = strlen(matching_string);
	int cmd_len = strlen(cmd);
	char *command = mymalloc(cmd_len + 1/* cmd + space */
			+ 1 + (par_len * 2) + 1 + 1, "check_filter_exec: command string");     /* "string"\0 */
	int loop;

	memcpy(command, cmd, cmd_len);
	str_index = cmd_len;
	command[str_index++] = ' ';
	command[str_index++] = '\"';
	for(loop=0; loop<par_len; loop++)
	{
		if (matching_string[loop] == '"' || matching_string[loop] == '`')
		{
			command[str_index++] = '\\';
		}

		command[str_index++] = matching_string[loop];
	}
	command[str_index++] = '\"';
	command[str_index] = 0x00;

	if (execute_program(command, 1) == -1)
		error_exit("check_filter_exec: Failed to execute command '%s'!\n", command);

	myfree(command);
}

/* check if the current line matches with a regular expression
   returns 0 if a regexp failed to execute
   */

/* this code can be optimized quit a bit but I wanted to get it to work quickly so I expanded everything (2006/03/28) */
char check_filter(proginfo *cur, char *string, regmatch_t **pmatch, char **error, int *matching_regex, char do_re, char *display)
{
	int loop;
	/* if no regexps, then always matches so it's the default */
	char re_exec_ok = 1;

	*error = NULL;
	*pmatch = NULL;

	*display = 1;

	/* do this regular expression? */
	for(loop=0; loop<cur -> n_re; loop++)
	{
		int rc;
		char cmd = (cur -> pre)[loop].use_regex;
		char invert = (cur -> pre)[loop].invert_regex;

		/* SKIP DISABLED REGEXPS */
		if (!cmd)
			continue;

		/* EXECUTE THE REGEXPS */
		if (pmatch)
		{
			if (*pmatch == NULL)
				*pmatch = (regmatch_t *)mymalloc(sizeof(regmatch_t) * MAX_N_RE_MATCHES, "check_filter: matching regular expressions");

			rc = regexec(&(cur -> pre)[loop].regex, string, MAX_N_RE_MATCHES, *pmatch, 0);
		}
		else
		{
			rc = regexec(&(cur -> pre)[loop].regex, string, 0, NULL, 0);
		}

		/* IF ERROR, ABORT EVERYTHING */
		if (rc != 0 && rc != REG_NOMATCH)
		{
			*error = convert_regexp_error(rc, &(cur -> pre)[loop].regex);
			re_exec_ok = 0;
			break;
		}

LOG("match: %s, cmd: %c, invert: %d, string: '%s'\n", rc==0?"YES":"no ", cmd, invert, string);
		/* MATCHED? */
		if (rc == 0)
		{
			if (!invert)
			{
				if (cmd == 'm')
				{
					*display = 1;
					*matching_regex = loop;
					(cur -> pre)[loop].match_count++;
					return 1;
				}
				else if (cmd == 'v')
				{
					*display = 0;
					*matching_regex = loop;
					(cur -> pre)[loop].match_count++;
					return 1;
				}
				else if (cmd == 'x' && do_re)
				{
					check_filter_exec((cur -> pre)[loop].cmd, string);
					*matching_regex = loop;
					(cur -> pre)[loop].match_count++;
				}
				else if (toupper(cmd) == 'B' && do_re)
				{
					beep();
					*matching_regex = loop;
					(cur -> pre)[loop].match_count++;
				}
			}
			else
			{
				if (cmd == 'm')
				{
					*display = 0;
				}
			}
		}
		else
		{
			myfree(*pmatch);
			*pmatch = NULL;

			if (invert)
			{
				if (cmd == 'm')
				{
					*display = 1;
					*matching_regex = loop;
					(cur -> pre)[loop].match_count++;
					return 1;
				}
				else if (cmd == 'v')
				{
					*display = 0;
					*matching_regex = loop;
					(cur -> pre)[loop].match_count++;
					return 1;
				}
				else if (cmd == 'x' && do_re)
				{
					check_filter_exec((cur -> pre)[loop].cmd, string);
					*matching_regex = loop;
					(cur -> pre)[loop].match_count++;
				}
				else if (toupper(cmd) == 'B' && do_re)
				{
					beep();
					*matching_regex = loop;
					(cur -> pre)[loop].match_count++;
				}
			}
			else
			{
				if (cmd == 'm')
				{
					*display = 0;
				}
			}
		}
	}

	myfree(*pmatch);
	*pmatch = NULL;
	*matching_regex = -1;

	return re_exec_ok;
}

void do_print(int f_index, proginfo *cur, char *string, regmatch_t *matches, int matching_regex)
{
	/* check filter */
	/* if cur == NULL, then marker line: don't check against filter */
	if (IS_MARKERLINE(cur))
	{
		color_print(pi[f_index].data, cur, string, NULL, -1, MY_FALSE);
	}
	else
	{
		color_print(pi[f_index].data, cur, string, matches, matching_regex, MY_FALSE);
	}
}

void do_buffer(int f_index, proginfo *cur, char *string, char filter_match)
{
	/* remember string */
	if (lb[f_index].bufferwhat == 'a' || filter_match == 1)
	{
		int curline;
		int linelen = 0;

		if (string) linelen = strlen(string);

		/* remove enough lines untill there's room */
		while((lb[f_index].curpos   >= lb[f_index].maxnlines && lb[f_index].maxnlines > 0) ||
				((lb[f_index].curbytes + linelen) >= lb[f_index].maxbytes  && lb[f_index].maxbytes  > 0 && linelen < lb[f_index].maxbytes))
		{
			/* delete oldest */
			if ((lb[f_index].Blines)[0])
			{
				lb[f_index].curbytes -= strlen((lb[f_index].Blines)[0]);
				myfree((lb[f_index].Blines)[0]);
			}

			/* move entries over the deleted one */
			memmove(&(lb[f_index].Blines)[0], &(lb[f_index].Blines)[1], (lb[f_index].curpos-1) * sizeof(char *));
			memmove(&(lb[f_index].pi)[0], &(lb[f_index].pi)[1], (lb[f_index].curpos-1) * sizeof(proginfo *));

			lb[f_index].curpos--;

			if (lb[f_index].curpos == 0)
				break;
		}

		/* grow array */
		lb[f_index].Blines = (char **)myrealloc(lb[f_index].Blines, sizeof(char *) *
				(lb[f_index].curpos + 1), "do_buffer: lines buffer");

		lb[f_index].pi    = (proginfo **)myrealloc(lb[f_index].pi, sizeof(proginfo *) *
				(lb[f_index].curpos + 1), "do_buffer: lines buffer progrinfo");

		curline = lb[f_index].curpos++;

		/* add the logline itself */
		if (string)
		{
			(lb[f_index].Blines)[curline] = mystrdup(string, "do_buffer: print_and_buffer");
			lb[f_index].curbytes += linelen;
		}
		else
			(lb[f_index].Blines)[curline] = NULL;

		/* remember pointer to subwindow (required for setting colors etc.) */
		(lb[f_index].pi)[curline] = cur;
	}
}

void do_redirect(redirect_t *predir, char *buffer, int nbytes, mybool_t add_lf)
{
	int failed = 0;

	if (WRITE(predir -> fd, buffer, nbytes) != nbytes)
		failed = 1;

	if (add_lf == MY_TRUE && WRITE(predir -> fd, "\n", 1) != 1)
		failed = 1;

	if (failed)
	{
		error_popup("Error redirecting output", HELP_REDIRECT_FAILED, "Redirecting the output failed: %s", strerror(errno));

		/* stop redirecting */
		close(predir -> fd);
		predir -> fd = -1;
		predir -> type = REDIRECTTO_NONE;
	}
}

void redirect(proginfo *cur, char *data, int n_bytes, mybool_t is_filtered)
{
	int r_i;

	for(r_i=0; r_i < cur -> n_redirect; r_i++)
	{
		if (is_filtered == MY_TRUE &&
				((cur -> predir)[r_i].type == REDIRECTTO_FILE_FILTERED || (cur -> predir)[r_i].type == REDIRECTTO_PIPE_FILTERED))
		{
			do_redirect(&(cur -> predir)[r_i], data, n_bytes, 1);
		}
		else if (is_filtered == MY_FALSE &&
				((cur -> predir)[r_i].type == REDIRECTTO_FILE || (cur -> predir)[r_i].type == REDIRECTTO_PIPE))
		{
			do_redirect(&(cur -> predir)[r_i], data, n_bytes, 0);
		}
	}
}

void add_marker_if_changed(int f_index, proginfo *cur)
{
	if (lb[f_index].last_win != cur && (lb[f_index].marker_of_other_window || global_marker_of_other_window) && lb[f_index].marker_of_other_window != -1)
	{
		add_markerline(f_index, cur, MARKER_CHANGE, NULL);

		lb[f_index].last_win = cur;
	}
}

void emit_to_buffer_and_term(int f_index, proginfo *cur, char *line)
{
	regmatch_t *pmatch = NULL;
	char *error = NULL;
	int matching_regex = -1;
	char display = 1;
	char *dummyline = NULL;
	char timestamp[1024] = { 0 }, subwindowid[16] = { 0 };
	char *new_line;
	char do_print_and_buffer = 0;
	char marker_added = 0;

	if (replace_by_markerline && strcmp(&line[strlen(line) - strlen(replace_by_markerline)], replace_by_markerline) == 0)
	{
		add_markerline(f_index, cur, MARKER_REGULAR, NULL);
		return;
	}

	new_line = convert(cur, line);

	(void)check_filter(cur, new_line, &pmatch, &error, &matching_regex, 1, &display);

	/* error happened while processing regexp? */
	if (error)
	{
		if (!marker_added)
		{
			marker_added = 1;
			add_marker_if_changed(f_index, cur);
		}
		add_markerline(f_index, cur, MARKER_MSG, error);
		do_print_and_buffer = 1;

		myfree(error);
	}

	/* add a subwindow-ID: [xx] in front of each line */
	if (show_subwindow_id && pi[f_index].next != NULL)
	{
		proginfo *ppi = &pi[f_index];
		int subwin_nr = 0;

		while(ppi != cur && ppi -> next != NULL)
		{
			subwin_nr++;
			ppi = ppi -> next;
		}

		snprintf(subwindowid, sizeof(subwindowid), "[%02d]", subwin_nr);
	}

	/* add a timestamp in front of each line */
	if (cur -> add_timestamp)
		get_now_ts(ts_format, timestamp, sizeof(timestamp));

	if (timestamp[0] || subwindowid[0])
	{
		int dl_len = strlen(new_line) + strlen(timestamp)+1 + strlen(subwindowid)+1 + 1;
		dummyline = (char *)mymalloc(dl_len, "emit_to_buffer_and_term: line + timestamp + subwindowid");

		snprintf(dummyline, dl_len, "%s%s%s%s%s", subwindowid, subwindowid[0]?" ":"", timestamp, timestamp[0]?" ":"", new_line);
	}
	else
		dummyline = new_line;

	if (cur -> suppress_repeating_lines)
	{
		if ((!cur -> last_line && !dummyline) ||
				(cur -> last_line != NULL && dummyline != NULL && strcmp(cur -> last_line, dummyline) == 0))
		{
			cur -> n_times_repeated++;
		}
		else
		{
			do_print_and_buffer = 1;
			if (dummyline)
				cur -> last_line = mystrdup(dummyline, "last_line");
			else
				cur -> last_line = NULL;
		}
	}
	else
		do_print_and_buffer = 1;

	if (do_print_and_buffer)
	{
		if ((cur -> n_times_repeated > 0 || display) && !marker_added)
		{
			marker_added = 1;
			add_marker_if_changed(f_index, cur);
		}

		if (cur -> n_times_repeated)
		{
			char message[128];

			snprintf(message, sizeof(message), "Last message repeated %d times", cur -> n_times_repeated);
			do_print(f_index, cur, message, NULL, -1);
			do_buffer(f_index, cur, message, 1);

			cur -> n_times_repeated = 0;
			myfree(cur -> last_line);
			cur -> last_line = NULL;
		}

		if (display)
		{
			char *stripped = do_strip(cur, dummyline);
			/* output new text */
			do_print(f_index, cur, stripped, pmatch, matching_regex);
			myfree(stripped);

			redirect(cur, dummyline, strlen(dummyline), 1);
		}

		do_buffer(f_index, cur, dummyline, display);
	}

	if (pmatch) myfree(pmatch);
	if (dummyline != new_line) myfree(dummyline);
	if (new_line != line) myfree(new_line);
}

void update_statusline(NEWWIN *status, int win_nr, proginfo *cur)
{
	if (mode_statusline > 0 && status != NULL && cur != NULL)
	{
		int dx, dy;
		myattr_t attrs = statusline_attrs;
		int statusline_len;
		off64_t	fsize = (off64_t)-1;
		time_t	ts = time(NULL);
		char show_f1 = 0;
		int help_str_offset = 0;
		int total_info_len = 0;
		int win_width;
		char *fname = cur -> filename;
		char timestamp[1024];

		if (win_nr == terminal_index)
			attrs.colorpair_index = find_colorpair(COLOR_RED, -1, 0);
		else if (mail)
			attrs.colorpair_index = find_colorpair(COLOR_GREEN, -1, 0);

		if ((ts - mt_started) < 5)
			show_f1 = 1;

		myattr_on(status, attrs);

		draw_line(status, LINE_BOTTOM);

		if (filename_only)
		{
			char *dummy = strrchr(cur -> filename, '/');
			if (dummy) fname = dummy + 1;
		}

		win_width = getmaxx(status -> win);
		mvwprintw(status -> win, 0, 0, "%02d] %s", win_nr, shorten_filename(cur -> win_title?cur -> win_title:fname, win_width - 4));

		if (!cur -> is_command && cur -> is_stdin == MY_FALSE)
			(void)file_info(cur -> filename, &fsize, TT_MTIME, &ts, NULL);

		get_now_ts(statusline_ts_format, timestamp, sizeof(timestamp));

		help_str_offset = 4 + strlen(cur -> win_title?cur -> win_title:fname); /* 4: '%02d] '!! (window nr.) */
		statusline_len = help_str_offset + strlen(timestamp) + 1; /* 4: '%02d] '!! (window nr.) */

		if (win_nr == terminal_index)
			wprintw(status -> win, ", press <CTRL>+<a>, <d> to exit");
		else if (mail)
			wprintw(status -> win, " You've got mail!");

		getmaxyx(status -> win, dy, dx);
		if (dx >= (statusline_len + 13))
		{
			if (cur -> paused)
			{
				color_on(status, find_colorpair(COLOR_YELLOW, -1, 0));
				mvwprintw(status -> win, 0, dx - 10, "  Paused  ");
				color_off(status, find_colorpair(COLOR_YELLOW, -1, 0));
			}
			else if (cur -> is_command)
			{
				int vmsize = get_vmsize(cur -> pid);

				total_info_len = statusline_len + 12;

				if (vmsize != -1 && dx >= (statusline_len + 30))
				{
					int str_x = dx - strlen(timestamp) - 30;
					char *vmsize_str = amount_to_str(vmsize);
					mvwprintw(status -> win, 0, str_x, "%6s (VMsize) %5d (PID) - %s", vmsize_str, cur -> pid, timestamp);
					myfree(vmsize_str);

					total_info_len = statusline_len + 30;
				}
				else
				{
					if (cur -> last_exit_rc != 0)
					{
						mvwprintw(status -> win, 0, dx - strlen(timestamp) - 26, "Last rc: %d, %5d (PID) - %s", WEXITSTATUS(cur -> last_exit_rc), cur -> pid, timestamp);
					}
					else
						mvwprintw(status -> win, 0, dx - strlen(timestamp) - 12, "%5d (PID) - %s", cur -> pid, timestamp);
				}
			}
			else if (fsize == -1)
			{
				if (cur -> is_stdin == MY_FALSE)
				{
					mvwprintw(status -> win, 0, dx - strlen(timestamp) - 6, "??? - %s", timestamp);
					total_info_len = statusline_len + 6;
				}
				else
					mvwprintw(status -> win, 0, dx - strlen(timestamp), "%s", timestamp);
			}
			else
			{
				int cur_len = 0;

				if (fsize < cur -> last_size)
					add_markerline(win_nr, cur, MARKER_MSG, " file got truncated");
				cur -> last_size = fsize;

				if (afs)
				{
					char *filesize = amount_to_str(fsize);
					cur_len = 3 + strlen(filesize);
					mvwprintw(status -> win, 0, dx - (strlen(timestamp) + cur_len), "%s - %s", filesize, timestamp);
					free(filesize);
				}
				else
				{
					cur_len = 13;
					/* is this trick still neccessary as I moved from off_t to off64_t? */
#if 0
					/* this trick is because on MacOS X 'off_t' is specified as a 64 bit integer */
#endif
					if (sizeof(off64_t) == 8)
						mvwprintw(status -> win, 0, dx - (strlen(timestamp) + cur_len), "%10lld - %s", fsize, timestamp);
					else
						mvwprintw(status -> win, 0, dx - (strlen(timestamp) + cur_len), "%10ld - %s", fsize, timestamp);
				}

				total_info_len = statusline_len + cur_len;
			}
		}

		if (show_f1)
		{
			if (use_colors) color_on(status, find_colorpair(COLOR_YELLOW, -1, 1));
			if (dx >= (total_info_len + 32))
				mvwprintw(status -> win, 0, help_str_offset, " *Press F1/<CTRL>+<h> for help* ");
			else if (dx >= (total_info_len + 21))
				mvwprintw(status -> win, 0, help_str_offset, " F1/<CTRL>+<h>: help ");
			else if (dx >= (total_info_len + 13))
				mvwprintw(status -> win, 0, help_str_offset, " F1/^h: help ");
			if (use_colors) color_off(status, find_colorpair(COLOR_YELLOW, -1, 1));
		}

		myattr_off(status, attrs);

		if (check_date() && use_colors)
		{
			int str_x = dx - strlen(timestamp);

			color_on(status, find_or_init_colorpair(COLOR_GREEN, COLOR_WHITE, 1));
			wattron(status -> win, A_BLINK);
			wattron(status -> win, A_BOLD);
			mvwprintw(status -> win, 0, str_x, "%s", timestamp);
			wattroff(status -> win, A_BOLD);
			wattroff(status -> win, A_BLINK);
			color_off(status, find_colorpair(COLOR_GREEN, COLOR_WHITE, 0));
		}

		update_panels();
	}
}

void create_window_set(int startx, int width, int nwindows, int indexoffset)
{
	int loop;

	int term_offset = 0;
	int last_not_hidden_window = -1;
	int n_not_hidden = 0;
	int n_height_not_set = 0;
	int n_lines_requested = 0, n_with_nlines_request = 0;
	int n_lines_available = 0;
	int n_lines_available_per_win = 0;
	int reserved_lines = mode_statusline >= 0?1:0;
	int *lines_per_win = (int *)mymalloc(nwindows * sizeof(int), "create_window_set: lines per win");

	if (SIG_ERR == signal(SIGWINCH, SIG_IGN)) error_exit("create_window_set: signal (SIGWINCH/SIG_IGN) failed\n");

	memset(lines_per_win, 0x00, sizeof(int) * nwindows);

	/* count the number of windows that are not hidden */
	for(loop=0; loop<nwindows; loop++)
	{
		int cur_win_index = loop + indexoffset;
		if (cur_win_index >= nfd) break;

		if (pi[cur_win_index].hidden == 0)
		{
			last_not_hidden_window = loop;
			n_not_hidden++;
		}
	}
	if (n_not_hidden == 0)
		return;

	/* count the number of lines needed by the windows for which
	 * the height is explicitly set
	 * also count the number of windows with this request set
	 */
	for(loop=0; loop<nwindows; loop++)
	{
		int cur_win_index = loop + indexoffset;
		if (cur_win_index >= nfd) break;

		if (pi[cur_win_index].hidden == 0 && pi[cur_win_index].win_height != -1)
		{
			n_lines_requested += (pi[cur_win_index].win_height + reserved_lines);
			n_with_nlines_request++;
		}
	}

	/* fill in the array with heights */
	/* not enough lines? simple fill-in */
	n_lines_available = max_y;
	n_height_not_set = n_not_hidden - n_with_nlines_request;
	if (n_height_not_set > 0)
		n_lines_available_per_win = (max_y - n_lines_requested) / n_height_not_set;
	if (n_height_not_set > 0 && n_lines_available_per_win < 3)
	{
		int lost_lines;
		int n_windows_that_will_fit;

		n_lines_available_per_win = max(3, max_y / n_not_hidden);
		n_windows_that_will_fit = max_y / n_lines_available_per_win;

		lost_lines = max_y - (n_lines_available_per_win * n_windows_that_will_fit);

		for(loop=0; loop<nwindows; loop++)
		{
			int cur_win_index = loop + indexoffset;
			if (cur_win_index >= nfd) break;

			if (pi[cur_win_index].hidden == 0)
			{
				lines_per_win[loop] = n_lines_available_per_win - reserved_lines;
				if (lost_lines)
				{
					lines_per_win[loop]++;
					lost_lines--;
				}
				n_lines_available -= n_lines_available_per_win;

				if (n_lines_available < n_lines_available_per_win)
					break;
			}
		}
	}
	else	/* enough lines */
	{
		int lost_lines = (max_y - n_lines_requested) - (n_lines_available_per_win * (n_not_hidden - n_with_nlines_request));

		/* not enough lines available? then ignore all requests */
		for(loop=0; loop<nwindows; loop++)
		{
			/* need to keep space for a statusline? */
			int cur_win_index = loop + indexoffset;
			if (cur_win_index >= nfd) break;

			if (!pi[cur_win_index].hidden)
			{
				LOG("1: lines available: %d\n", n_lines_available);
				if (pi[cur_win_index].win_height != -1)
				{
					lines_per_win[loop] = pi[cur_win_index].win_height;
					n_lines_available -= (pi[cur_win_index].win_height + reserved_lines);
				}
				else
				{
					lines_per_win[loop] = n_lines_available_per_win - reserved_lines;
					if (lost_lines)
					{
						lines_per_win[loop]++;
						lost_lines--;
					}
					n_lines_available -= (n_lines_available_per_win + reserved_lines);
				}

				LOG("2: lpw: %d, nla: %d, ll: %d\n", lines_per_win[loop], n_lines_available, lost_lines);
			}
		}
	}


	/* re-create windows */
	for(loop=0; loop<nwindows; loop++)
	{
		int redraw_loop;

		int cur_index = loop + indexoffset;
		if (cur_index >= nfd) break;

		if (pi[cur_index].hidden)
			continue;

		if (lines_per_win[loop] == 0)
			continue;

		/* create data window, clear en set scroll ok */
		if (statusline_above_data)
			pi[cur_index].data = mynewwin(lines_per_win[loop], width, term_offset + 1, startx);
		else
			pi[cur_index].data = mynewwin(lines_per_win[loop], width, term_offset, startx);
		bottom_panel(pi[cur_index].data -> pwin);
		werase(pi[cur_index].data -> win);
		scrollok(pi[cur_index].data -> win, TRUE);/* supposed to always return OK, according to the manpage */

		/* create status window */
		if (mode_statusline >= 0)
		{
			if (statusline_above_data)
				pi[cur_index].status = mynewwin(1, width, term_offset, startx);
			else
				pi[cur_index].status = mynewwin(1, width, term_offset + lines_per_win[loop], startx);
			bottom_panel(pi[cur_index].status -> pwin);
			werase(pi[cur_index].status -> win);

			/* create status-line */
			update_statusline(pi[cur_index].status, cur_index, &pi[cur_index]);
		}

		term_offset += lines_per_win[loop] + (mode_statusline >= 0?1:0);

		/* display old strings */
		for(redraw_loop=max(0, lb[cur_index].curpos - lines_per_win[loop]); redraw_loop<lb[cur_index].curpos; redraw_loop++)
		{
			if (IS_MARKERLINE(lb[cur_index].pi[redraw_loop]) && lb[cur_index].Blines[redraw_loop] != NULL)
			{
				do_print(cur_index, lb[cur_index].pi[redraw_loop], lb[cur_index].Blines[redraw_loop], NULL, -1);
			}
			else if ((lb[cur_index].Blines)[redraw_loop])
			{
				char display;
				int matching_regex = -1;
				regmatch_t *pmatch = NULL;
				char *error = NULL;

				/* check filter */
				(void)check_filter(lb[cur_index].pi[redraw_loop], lb[cur_index].Blines[redraw_loop], &pmatch, &error, &matching_regex, 0, &display);
				if (error)
				{
					do_print(cur_index, (lb[cur_index].pi)[redraw_loop], error, NULL, -1);
					myfree(error);
				}

				if (display)
				{
					char *stripped = do_strip((lb[cur_index].pi)[redraw_loop], (lb[cur_index].Blines)[redraw_loop]);
					do_print(cur_index, (lb[cur_index].pi)[redraw_loop], stripped, pmatch, matching_regex);
					myfree(stripped);
				}

				myfree(pmatch);
			}
		}

		update_panels();
	}

	myfree(lines_per_win);

	/* set signalhandler for terminal resize */
	if (SIG_ERR == signal(SIGWINCH, signal_handler)) error_exit("create_window_set: signal (SIGWINCH/SIG_IGN) failed\n");
}

void create_windows(void)
{
	int loop;
	int n_not_hidden = 0;

	/* close windows */
	for(loop=0; loop<nfd; loop++)
	{
		if (pi[loop].status)
		{
			delete_popup(pi[loop].status);
			pi[loop].status = NULL;
		}
		if (pi[loop].data)
		{
			delete_popup(pi[loop].data);
			pi[loop].data = NULL;
		}
	}

	if (splitlines)
	{
		for(loop=0; loop<n_splitlines; loop++)
			delete_popup(splitlines[loop]);
		myfree(splitlines);
		splitlines = NULL;
		n_splitlines = 0;
	}

	if (menu_win)
	{
		delete_popup(menu_win);
		menu_win = NULL;
	}

	werase(stdscr);

	for(loop=0; loop<nfd; loop++)
	{
		if (!pi[loop].hidden)
			n_not_hidden++;
	}

	/* no windows? display list of keys */
	if (nfd == 0 || n_not_hidden == 0)
	{
		menu_win = mynewwin(max_y, max_x, 0, 0);
		werase(menu_win -> win);

		wprintw(menu_win -> win, version_str);
		wprintw(menu_win -> win, "\n\n");

		wprintw(menu_win -> win, "%s\n", F1);
	}
	else
	{
		if (split > 0 && nfd > 1 && n_not_hidden >= split)
		{
			int cols_per_col = max_x / split;
			int win_per_col = nfd / split;
			int x = 0, indexoffset = 0;
			int *vs = (int *)mymalloc(sizeof(int) * split, "create_windows: window widths");
			int *vn = (int *)mymalloc(sizeof(int) * split, "create_windows: n windows");

			if (vertical_split)
			{
				int cols_in_use = 0;
				int cols_set = 0;
				int other_cols_size;

				for(loop=0; loop<split; loop++)
				{
					if (vertical_split[loop] >= 4)
					{
						vs[loop] = vertical_split[loop];
						cols_in_use += vertical_split[loop];
						cols_set++;
					}
				}

				/* determine width of other (=not set) columns */
				if (cols_set < split)
				{
					other_cols_size = (max_x - cols_in_use) / (split - cols_set);

					/* doesn't fit? */
					if (other_cols_size < 4)
					{
						/* then use the predetermined size */
						for(loop=0; loop<split; loop++)
							vs[loop] = cols_per_col;
					}
					else /* fill in the not given windows */
					{
						for(loop=0; loop<split; loop++)
						{
							if (vertical_split[loop] < 4)
								vs[loop] = other_cols_size;
						}
					}
				}
			}
			else
			{
				for(loop=0; loop<split; loop++)
					vs[loop] = cols_per_col;
			}

			if (n_win_per_col)
			{
				int win_in_use = 0;
				int win_set = 0;
				int other_win_n;

				for(loop=0; loop<split; loop++)
				{
					if (n_win_per_col[loop] > 0)
					{
						vn[loop] = n_win_per_col[loop];
						win_in_use += n_win_per_col[loop];
						win_set++;
					}
				}

				if (win_set < split)
				{
					other_win_n = (nfd - win_in_use) / (split - win_set);

					if (other_win_n == 0)
					{
						for(loop=0; loop<split; loop++)
							vn[loop] = win_per_col;
					}
					else
					{
						for(loop=0; loop<split; loop++)
						{
							if (n_win_per_col[loop] < 1)
								vn[loop] = other_win_n;
						}
					}
				}
			}
			else
			{
				for(loop=0; loop<split; loop++)
					vn[loop] = win_per_col;
			}

			n_splitlines = split - 1;
			if (n_splitlines > 0)
				splitlines = (NEWWIN **)mymalloc(sizeof(NEWWIN *) * n_splitlines, "create_windows: splitlines");
			else
				splitlines = NULL;

			for(loop=0; loop<split; loop++)
			{
				if (loop == (split - 1))
				{
					int max_n_win_per_col = max_y / 4;

					create_window_set(x, max_x - x, min(max_n_win_per_col, nfd - indexoffset), indexoffset);
				}
				else
				{
					create_window_set(x, vs[loop] - 1, vn[loop], indexoffset);
					splitlines[loop] = mynewwin(max_y, 1, 0, x + vs[loop] - 1);
					bottom_panel(splitlines[loop] -> pwin);
					myattr_on(splitlines[loop], statusline_attrs);
					draw_line(splitlines[loop], LINE_LEFT);
					myattr_off(splitlines[loop], statusline_attrs);

					x += vs[loop];
					indexoffset += vn[loop];
				}
			}

			myfree(vn);
			myfree(vs);
		}
		else if (nfd != 0)
		{
			create_window_set(0, max_x, nfd, 0);
		}
	}

	mydoupdate();
}

void do_set_bufferstart(int f_index, char store_what_lines, int maxnlines)
{
	lb[f_index].bufferwhat = store_what_lines;

	if (maxnlines < lb[f_index].maxnlines)
	{
		delete_array(lb[f_index].Blines, lb[f_index].curpos);
		/* delete array already frees the Blines-array */
		lb[f_index].Blines = NULL;

		myfree(lb[f_index].pi);
		lb[f_index].pi = NULL;

		lb[f_index].curpos = 0;
	}

	lb[f_index].maxnlines = maxnlines;
}

char close_window(int winnr, proginfo *cur)
{
	if (winnr == terminal_index)
		terminal_index = -1;

	/* make sure it is really gone */
	stop_process(cur -> pid);

	/* restart window? */
	if (cur -> restart >= 0)
	{
		int initial_n_lines_tail;

		/* close old fds */
		if (myclose(cur -> fd) == -1) error_exit("close_window: closing read filedescriptor failed\n");
		if (cur -> fd != cur -> wfd)
		{
			if (myclose(cur -> wfd) == -1) error_exit("close_window: closing write filedescriptor failed\n");
		}

		/* do diff */
		if (cur -> do_diff)
			generate_diff(winnr, cur);

		if (cur -> initial_n_lines_tail == -1)
			initial_n_lines_tail = max_y / (nfd + 1);
		else
			initial_n_lines_tail = cur -> initial_n_lines_tail;

		/* restart tail-process */
		if (start_proc(cur, initial_n_lines_tail) != -1)
			return -1;

		/* ...and if that fails, go on with the closing */
	}

	if (warn_closed)
	{
		NEWWIN *mywin;
		int subwinnr = 0;
		proginfo *dummy = &pi[winnr];
		int win_width = 18 + 4;
		int win_name_len = strlen(cur -> filename);

		while(dummy -> next)
		{
			subwinnr++;
			dummy = dummy -> next;
		}

		if (win_name_len > win_width && (win_name_len + 4) < max_x)
			win_width = win_name_len;
		mywin = create_popup(5, win_width);

		color_on(mywin, 1);
		mvwprintw(mywin -> win, 1, 2, "Window %d/%d closed", winnr, subwinnr);
		mvwprintw(mywin -> win, 2, 2, "%s", cur -> filename);
		mvwprintw(mywin -> win, 3, 2, "Exit code: %d", WEXITSTATUS(cur -> last_exit_rc));
		box(mywin -> win, 0, 0);
		color_off(mywin, 1);

		wmove(mywin -> win, 1, 2);
		mydoupdate();

		/* wait_for_keypress(HELP_WINDOW_CLOSED, 0); */
		(void)getch();

		delete_popup(mywin);
	}

	/* file no longer exists (or so) */
	return delete_entry(winnr, cur);
}

char * key_to_keybinding(char what)
{
	static char buffer[3];

	buffer[1] = buffer[2] = 0x00;

	if (what < 32)
	{
		buffer[0] = '^';
		buffer[1] = 'a' + what - 1;
	}
	else
	{
		buffer[0] = what;
	}

	return buffer;
}

void write_escape_str(FILE *fh, char *string)
{
	int loop, len = strlen(string);

	fprintf(fh, "\"");
	for(loop=0; loop<len; loop++)
	{
		if (string[loop] == '\"')
			fprintf(fh, "\\");
		fprintf(fh, "%c", string[loop]);
	}
	fprintf(fh, "\"");
}

void emit_myattr_t(FILE *fh, myattr_t what)
{
	if (what.colorpair_index != -1)
	{
		/* FG */
		if (cp.fg_color[what.colorpair_index] != -1)
			fprintf(fh, "%s", color_to_string(cp.fg_color[what.colorpair_index]));

		/* BG */
		if (cp.bg_color[what.colorpair_index] != -1)
			fprintf(fh, ",%s", color_to_string(cp.bg_color[what.colorpair_index]));
	}

	/* attributes */
	if (what.attrs > 0)
	{
		if (what.colorpair_index == -1)
			fprintf(fh, ",,");
		else
			fprintf(fh, ",");

		fprintf(fh, "%s", attr_to_str(what.attrs));
	}
}

void add_pars_per_file(char *re, int cs, int n_buffer_lines, int buffer_bytes, char change_win_marker)
{
	int loop;

	for(loop=0; loop<n_pars_per_file; loop++)
	{
		if (strcmp(ppf[loop].re_str, re) == 0)
			break;
	}

	if (loop == n_pars_per_file)
	{
		int rc;

		n_pars_per_file++;

		/* add to list */
		ppf = (pars_per_file *)myrealloc(ppf, sizeof(pars_per_file) * n_pars_per_file, "add_pars_per_file: pars_per_file struct");
		ppf[loop].re_str = mystrdup(re, "add_pars_per_file: default scheme: re string");

		/* compile & store regular expression */
		rc = regcomp(&ppf[loop].regex, re, REG_EXTENDED);
		if (rc != 0)
			error_exit("failed to compile regular expression '%s': %s\n", re, convert_regexp_error(rc, &ppf[loop].regex));

		ppf[loop].n_colorschemes = 0;
		ppf[loop].colorschemes = NULL;
		ppf[loop].buffer_maxnlines = -1;
		ppf[loop].buffer_bytes = -1;
		ppf[loop].change_win_marker = 0;
	}

	if (cs != -1)
	{
		/* store internal scheme-nr */
		ppf[loop].n_colorschemes++;
		ppf[loop].colorschemes = (int *)myrealloc(ppf[loop].colorschemes, sizeof(int) * ppf[loop].n_colorschemes, "list of default colorschemes");

		ppf[loop].colorschemes[ppf[loop].n_colorschemes - 1] = cs;
	}

	if (n_buffer_lines != -1)
		ppf[loop].buffer_maxnlines = n_buffer_lines;

	if (buffer_bytes != -1)
		ppf[loop].buffer_bytes = buffer_bytes;

	if (change_win_marker != -1)
		ppf[loop].change_win_marker = change_win_marker;
}

void redraw_statuslines(void)
{
	int loop;

	/* update statuslines */
	for(loop=0; loop<nfd; loop++)
		update_statusline(pi[loop].status, loop, &pi[loop]);
}

int exec_bind(char key)
{
	int executed = 0;
	int loop;

	for(loop=0; loop<n_keybindings; loop++)
	{
		if (keybindings[loop].key == key)
		{
			gui_window_header(keybindings[loop].command);

			if (execute_program(keybindings[loop].command, 0) == 0)
				executed = 1;

			break;
		}
	}

	return executed;
}

void set_default_parameters_if_not_given_do(proginfo *cur, int pi_index)
{
	int maxnlines = default_maxnlines, maxbytes = default_maxbytes;
	int pm = find_path_max();
	char *real_fname = (char *)mymalloc(pm, "real path");

	do
	{
		int ppf_index;

		if (cur -> is_command || cur -> is_stdin)
		{
			strncpy(real_fname, cur -> filename, pm);
			real_fname[pm - 1] = 0x00;
		}
		else
		{
			if (!realpath(cur -> filename, real_fname))
				error_exit("error obtaining complete (real) path of %s\n", cur -> filename);
		}

		/* check if any default parameters are given in the configfile */
		for(ppf_index=0; ppf_index<n_pars_per_file; ppf_index++)
		{
			int rc;

			if ((rc = regexec(&ppf[ppf_index].regex, real_fname, 0, NULL, 0)) == 0)
			{
				/* set default colorscheme if not already given */
				if (ppf[ppf_index].n_colorschemes > 0 && (cur -> n_color_schemes == 0 && cur -> is_command == 0 && (cur -> colorize == 'n' || cur -> colorize == 0)))
				{
					int c_nr;

					for(c_nr=0; c_nr<ppf[ppf_index].n_colorschemes; c_nr++)
					{
						add_color_scheme(&cur -> color_schemes, &cur -> n_color_schemes, ppf[ppf_index].colorschemes[c_nr]);
					}
					cur -> colorize = 'S';
				}

				if (ppf[ppf_index].buffer_maxnlines != -1)
					maxnlines = ppf[ppf_index].buffer_maxnlines;

				if (ppf[ppf_index].buffer_bytes != -1)
				{
					maxbytes = ppf[ppf_index].buffer_bytes;
					LOG("pff: %d\n", maxbytes);
				}

				break;
			}
			/* we should inform the user of any errors while executing
			 * the regexp! */
			else
			{
				char *error = convert_regexp_error(rc, &ppf[ppf_index].regex);

				if (error)
					error_exit("%s\n", error);
			}
		}

		cur = cur -> next;
	} while(cur);

	if (pi_index != -1)
	{
		LOG("set default %d lines %d bytes\n", maxnlines, maxbytes);
		if (lb[pi_index].maxnlines == -1 && lb[pi_index].maxbytes == -1)
		{
			if (maxbytes)
			{
				lb[pi_index].maxbytes  = maxbytes;
				lb[pi_index].maxnlines = 0;
			}
			else
			{
				lb[pi_index].maxnlines = maxnlines;
				lb[pi_index].maxbytes  = 0;
			}
		}
		else if (lb[pi_index].maxnlines == -1)
		{
			lb[pi_index].maxnlines = maxnlines;
		}
		else /* if (lb[pi_index].maxbytes == -1) */
		{
			lb[pi_index].maxbytes = maxbytes;
		}
	}

	myfree(real_fname);
}

void set_default_parameters_if_not_given(void)
{
	int pi_index;

	for(pi_index=0; pi_index<nfd; pi_index++)
		set_default_parameters_if_not_given_do(&pi[pi_index], pi_index);
}

void start_all_processes(char *nsubwindows)
{
	int loop;

	for(loop=0; loop<nfd; loop++)
	{
		proginfo *cur = &pi[loop];
		int cur_win_size = max_y / nsubwindows[loop];

		do
		{
			cur -> start_ts = time(NULL);

			if (cur -> is_stdin == MY_TRUE)
			{
				int old_0 = mydup(0);

				if (old_0 == -1) error_exit("start_all_processes: cannot dup(0)\n");

				if (close(0) == -1) error_exit("start_all_processes: error closing fd 0\n");

				if (open("/dev/tty", O_RDONLY) != 0) error_exit("start_all_processes: new fd != 0\n");

				cur -> fd = old_0;
				cur -> wfd = -1;
				cur -> pid = -1;
			}
			else
			{
				if (cur -> initial_n_lines_tail == -1)
					cur -> initial_n_lines_tail = cur_win_size;

				/* start the tail process for this file/command */
				if (start_proc(cur, cur -> initial_n_lines_tail) == -1)
					error_exit("failed to start process! (%s)\n", cur -> filename);
			}

			cur = cur -> next;
		} while(cur);
	}
}

void version(void)
{
	printf("%s", version_str);
	printf("\n\nThank you for using MultiTail.\n");
	printf("If you have any suggestion on how I can improve this program,\n");
	printf("do not hesitate to contact me at folkert@vanheusden.com\n");
	printf("Website is available at: http://www.vanheusden.com/multitail/\n\n\n");
}

void init_check_for_mail(void)
{
	if (check_for_mail > 0 && mail_spool_file != NULL)
	{
		if (stat64(mail_spool_file, &msf_info) == -1)
		{
			check_for_mail = 0;

			printf("Could not determine size of file '%s' (which is supposed to be ", mail_spool_file);
			printf("your mailfile): mail-check is disabled.\n");
			printf("You can prevent this message by adding the line 'check_mail:0' ");
			printf("in /etc/multitail.conf or in .multitailrc in your home-directory.\n\n");
			printf("Press enter to continue...");
			fflush(NULL);
			getchar();
		}
		else
		{
			msf_prev_size = msf_info.st_size;

			msf_last_check = time(NULL);
		}
	}
}

void main_loop(void)
{
	/* create windows */
	do_refresh = 2;

	for(;;)
	{
		int c = wait_for_keypress(HELP_MAIN_MENU, 0, NULL, 0);

		if (terminal_index == -1)
		{
			int uc = toupper(c);

			do_refresh = 1;

			if (mail)
			{
				mail = 0;
				redraw_statuslines();
			}

			if (uc == 'Q' || uc == 'X')
			{
				break;
			}
			else if (uc == 'A')
			{
				if (add_window())
					do_refresh = 2;
				continue;
			}
			else if (uc == 'H' || uc == '?')
			{
				show_help(HELP_MAIN_MENU);
				continue;
			}
			else if (uc == 'R')
			{
				do_refresh = 2;
				continue;
			}
			else if (uc == 'I')
			{
				info();
				continue;
			}
			else if (c == 't')
			{
				statistics_menu();
				continue;
			}
			else if (c == 'T')
			{
				truncate_file();
				continue;
			}
			else if (uc == 'J')
			{
				if (set_window_sizes())
					do_refresh = 2;
				continue;
			}
			else if (uc == 'L')
			{
				list_keybindings();
				continue;
			}

			if (nfd == 0)
			{
				wrong_key();
				do_refresh = 0;
				continue;
			}

			else if (uc == 'E' || uc == '\\')
			{
				if (edit_regexp())
					do_refresh = 2;
			}
			else if (uc == 'F')
			{
				if (edit_strippers())
					do_refresh = 2;
			}
			else if (uc == 'D')
			{
				if (delete_window())
					do_refresh = 2;
				continue;
			}
			else if (uc == 'V')
			{
				if (toggle_vertical_split())
					do_refresh = 2;
			}
			else if (c == 'c')
			{
				if (toggle_colors())
					do_refresh = 2;
			}
			else if (c == 'C' && use_colors)
			{
				if (can_change_color())
					(void)color_management(NULL, NULL);
				else
					error_popup("Edit colors", HELP_CANNOT_EDIT_COLOR, "Your terminal doesn't support editing of colors.");
			}
			else if (uc == 'S')
			{
				if (swap_window())
					do_refresh = 2;
				continue;
			}
			else if (c == 'z')
			{
				if (hide_window())
					do_refresh = 2;
			}
			else if (c == 'u')
			{
				if (hide_all_but())
					do_refresh = 2;
			}
			else if (c == 'U')
			{
				if (unhide_all_windows())
					do_refresh = 2;
			}
			else if (uc == 'G')
			{
				screendump();
			}
			else if (uc == 'W')
			{
				write_script();
			}
			else if (uc == 'M')
			{
				set_buffering();
			}
			else if (uc == 'B')
			{
				scrollback();
			}
			else if (c == 'p')
			{
				do_pause();
			}
			else if (c == 'P')	/* pause an individual screen */
			{
				selective_pause();
			}
			else if (uc >= '0' && uc <= '9')
			{
				int index = uc - '0';

				if (index < nfd)
					add_markerline(index, NULL, MARKER_REGULAR, NULL);
				else
					wrong_key();
			}
			else if (uc == 'K')
			{
				terminal_mode();
			}
			else if (uc == 26)	/* ^Z */
			{
				/* we're back from suspend */
				redraw_statuslines();
			}
			else if (uc == ' ')	/* space */
			{
				/* mail = 0;
				   redraw_statuslines(); */
			}
			else if (c == 'o')	/* wipe window */
			{
				wipe_window();
			}
			else if (c == 'O')	/* wipe all windows */
			{
				wipe_all_windows();
			}
			else if (c == 'y')	/* set linewrap */
			{
				set_linewrap();
				do_refresh = 2;
			}
			else if (c == 'Y')	/* send a signal to a window */
			{
				send_signal();
			}
			else
			{
				if (exec_bind(c))
				{
					do_refresh = 1;
				}
				else
				{
					wrong_key();
					do_refresh = 0;
				}
			}
		}
		else
		{
			do_terminal((char)c);
		}
	}
}

int main(int argc, char *argv[])
{
	int loop;
	char *dummy = getenv("MAILCHECK");

	cp.n_def = cp.size = 0;
	cp.fg_color = cp.bg_color = NULL;

	/* removed because if gives regexp compilation problems for some
	 * LANG=... environment variables settings
	 * setlocale(LC_ALL, ""); */

	time(&mt_started);

	if (SIG_ERR == signal(SIGTERM, signal_handler)) error_exit("main: setting of signal failed\n");

	mail_spool_file = getenv("MAIL");
	if (dummy) check_for_mail = atoi(dummy);

	/* calc. buffer length (at least a complete terminal screen) */
	initscr();
	min_n_bufferlines = max(MIN_N_BUFFERLINES, LINES);
	if (has_colors())
	{
		start_color();
		use_default_colors();
		use_colors = 1;
	}

	init_colors();

	init_colornames();

	/* verify size of terminal window */
	if (LINES < 24 || COLS < 78)
		error_exit("Your terminal(-window) is %dx%d. That is too small for MultiTail (at least 78x24 is required).\n", COLS, LINES);

	/* default parameters for the statusline */
	statusline_attrs.colorpair_index = find_or_init_colorpair(COLOR_WHITE, COLOR_BLACK, 0);
	statusline_attrs.attrs = A_REVERSE;

	endwin();

	do_commandline(argc, argv);

	load_configfile(config_file);

	if (markerline_attrs.colorpair_index == -1 && markerline_attrs.attrs == -1)
		markerline_attrs = find_attr(COLOR_RED, COLOR_BLACK, A_REVERSE);

	init_children_reaper();

	/* initialise mail-stuff */
	init_check_for_mail();

	/* start curses library */
	init_ncurses();

        /* it seems that on at least FreeBSD ncurses forgets all defined
         * colorpairs after an endwin()
         * so here were explicitly re-setting them
         */
        for(loop=0; loop<cp.n_def; loop++)
                init_pair(loop, cp.fg_color[loop], cp.bg_color[loop]);

	/* get terminal type */
	get_terminal_type();

	set_default_parameters_if_not_given();

	/* start processes */
	start_all_processes(nsubwindows);
	myfree(nsubwindows);

	signal(SIGHUP, signal_handler);

	main_loop();

	/* kill tail processes */
	do_exit();

	return 0;
}

void sighup_restart_tails(void)
{
	int f_index;

	for(f_index=0; f_index<nfd; f_index++)
	{
		proginfo *cur = &pi[f_index];

		do
		{
			if (!cur -> is_command && cur -> is_stdin != MY_TRUE)
			{
				int pipefd[2];

				stop_process(cur -> pid);

				if (myclose(cur -> fd) == -1) error_exit("sighup_restart_tails: closing read filedescriptor failed\n");
				if (cur -> fd != cur -> wfd)
				{
					if (myclose(cur -> wfd) == -1) error_exit("sighup_restart_tails: closing write filedescriptor failed\n");
				}

				/* create a pipe, will be to child-process */
				if (-1 == pipe(pipefd)) error_exit("sighup_restart_tails: error creating pipe\n");

				cur -> pid = start_tail(cur -> filename, cur -> retry_open, cur -> follow_filename, min_n_bufferlines, pipefd);
				cur -> fd = pipefd[0];
				cur -> wfd = pipefd[1];
			}

			cur = cur -> next;
		} while(cur);
	}
}

int check_for_exited_processes(void)
{
	int win_got_closed = 0;
	int f_index;

	/* verify if any of the processes exited */
	for(f_index=0; f_index<nfd; f_index++)
	{
		int deleted_entry_in_array = 0;
		proginfo *cur = &pi[f_index];

		do
		{
			if (cur -> is_stdin == MY_FALSE)
			{
				/* see if the process exited */
				pid_t rc = waitpid(cur -> pid, &cur -> last_exit_rc, WNOHANG | WUNTRACED);

				/* error while waiting? */
				if (rc == -1 && errno != ECHILD)
					error_exit("check_for_exited_processes: waitpid for pid %d failed\n", cur -> pid);

				/* did it exit? */
				if (rc != 0) /* equal to: rc == cur -> pid */
				{
					win_got_closed = 1;
					deleted_entry_in_array = close_window(f_index, cur);
					/* is an entry deleted? (>=0) or restarted? (==-1) */
					if (deleted_entry_in_array >= 0)
					{
						do_refresh = 2;
						break;
					}
					else if (cur -> do_diff && do_refresh == 0)
					{
						if (do_refresh != 2) do_refresh = 1;
					}
				}
			}

			cur = cur -> next;
		}
		while(cur);

		if (deleted_entry_in_array > 0)
			break;
	}

	return win_got_closed;
}

int find_window(char *filename, int *index, proginfo **cur)
{
	int loop;

	for(loop=0; loop<nfd; loop++)
	{
		proginfo *pc = &pi[loop];

		do
		{
			if (strcmp(pc -> filename, filename) == 0)
			{
				if (index) *index = loop;
				if (cur) *cur = pc;

				return 0;
			}

			pc = pc -> next;
		}
		while(pc);
	}

	return -1;
}

void create_new_win(proginfo **cur, int *nr)
{
	int loop;

	proginfo *newpi = (proginfo *)myrealloc(pi, (nfd + 1) * sizeof(proginfo), "add_window: proginfo list");
	lb = (buffer *)myrealloc(lb, (nfd + 1) * sizeof(buffer), "add_window: buffer list");

	/* 'lb' has pointers to 'pi'-structures. now since we've realloced the pi-
	 * array, we need to update to pointers to pi in the lb-array
	 */
	for(loop=0; loop<nfd; loop++)
		buffer_replace_pi_pointers(loop, &pi[loop], &newpi[loop]);

	/* now that everything is updated, we can safely forget the previous pointer */
	pi = newpi;
	memset(&pi[nfd], 0x00, sizeof(proginfo));

	/* initialize the new structure */
	memset(&lb[nfd], 0x00, sizeof(buffer));
	lb[nfd].maxnlines = min_n_bufferlines;
	lb[nfd].bufferwhat = 'f';

	if (cur) *cur = &pi[nfd];
	if (nr) *nr = nfd;

	nfd++;
}

int check_paths(void)
{
	int new_wins = 0;
	int loop;

	for(loop=0; loop<n_cdg; loop++)
	{
		time_t now = time(NULL);

		if ((now - cdg[loop].last_check) >= cdg[loop].check_interval)
		{
			int fi;
			glob_t files;

			/* get list of files that match */
			if (glob(cdg[loop].glob_str, GLOB_ERR | GLOB_NOSORT, NULL, &files) != 0)
				continue;

			/* check each found file */
			for(fi=0; fi<files.gl_pathc; fi++)
			{
				time_t last_mod;
				mode_t ftype;

				if (file_info(files.gl_pathv[fi], NULL, cdg[loop].new_only, &last_mod, &ftype) == -1)
					continue;

				/* file got changed and/or is new and is a regular file?
				 * and is created after multitail started?
				 */
				if (last_mod >= cdg[loop].last_check &&
						find_window(files.gl_pathv[fi], NULL, NULL) == -1 &&
						S_ISREG(ftype) &&
						((cdg[loop].new_only && last_mod >= mt_started) || !cdg[loop].new_only)
				   )
				{
					proginfo *cur = NULL;
					int win_nr = -1;

					/* create new structure containing all info and such */
					if (!cdg[loop].in_one_window)
					{
						create_new_win(&cur, &win_nr);
					}
					else if (cdg[loop].window_nr == -1)
					{
						create_new_win(&cur, &cdg[loop].window_nr);
						win_nr = cdg[loop].window_nr;
					}
					else
					{
						cur = &pi[cdg[loop].window_nr];

						/* skip to end of chain */
						while (cur -> next) cur = cur -> next;

						/* allocate new entry */
						cur -> next = (proginfo *)mymalloc(sizeof(proginfo), "check_paths: proginfo");
						cur = cur -> next;
						memset(cur, 0x00, sizeof(proginfo));
					}

					/* fill in */
					cur -> conversion_index = -1;
					cur -> filename = mystrdup(files.gl_pathv[fi], "check_paths: new filename");
					cur -> line_wrap = 'a';
					cur -> win_height = -1;
					cur -> sccfirst = 1;
					set_default_parameters_if_not_given_do(cur, win_nr);

					/* start the tail process for this file/command */
					if (start_proc(cur, max_y / nfd) == -1)
						error_exit("failed to start process! (%s)\n", cur -> filename);

					new_wins = 1;
				}
			} /* check all matched files */

			cdg[loop].last_check = now;
		} /* time for a check? */
	} /* check all search patterns  */

	return new_wins;
}

void resize_terminal_do(NEWWIN *popup)
{
	determine_terminal_size(&max_y, &max_x);

	if (ERR == resizeterm(max_y, max_x)) error_exit("wait_for_keypress: problem resizing terminal\n");

	endwin();
	refresh(); /* <- as specified by ncurses faq, was: doupdate(); */

	create_windows();

	if (popup)
	{
		if ((popup -> x_off + popup -> width) > max_x ||
				(popup -> y_off + popup -> height) > max_y)
		{
			popup -> x_off = max(max_x - (popup -> width + 1), 0);
			popup -> y_off = max(max_y - (popup -> height + 1), 0);
			move_panel(popup -> pwin, popup -> y_off, popup -> x_off);
			mydoupdate();
		}
	}
}

int wait_for_keypress(int what_help, double max_wait, NEWWIN *popup, char cursor_shift)
{
	int c = -1;
	double max_wait_start = get_ts();

	for(;;)
	{
		int last_fd = 0, rc, loop;
		fd_set rfds;
		struct timeval tv;
		time_t now;
		proginfo *last_changed_window = NULL;
		char prev_mail_status = mail;
		static time_t lastupdate = 0;

		if (heartbeat_interval > 0 && (get_ts() - heartbeat_t) >= heartbeat_interval)
		{
			heartbeat();
			heartbeat_t = get_ts();
		}

		tv.tv_sec = 0;
		tv.tv_usec = 100000;

		FD_ZERO(&rfds);

		/* add stdin to fd-set: needed for monitoring key-presses */
		FD_SET(0, &rfds);

		/* add fd's of pipes to fd-set */
		for(loop=0; loop<nfd; loop++)
		{
			proginfo *cur = &pi[loop];

			if (cur -> paused)
				continue;

			do
			{
				if (cur -> fd != -1)
				{
					FD_SET(cur -> fd, &rfds);
					last_fd = max(last_fd, cur -> fd);
				}

				cur = cur -> next;
			}
			while(cur);
		}

		time(&now);

		/* check for mail */
		do_check_for_mail(now);

		/* need to check any paths? */
		if (check_paths())
			do_refresh = 2;

		/* update screen? */
		if (do_refresh)
		{
			if (do_refresh == 2)
				create_windows();

			if (difftime(now, lastupdate) >= update_interval)
			{
				do_refresh = 0;
				lastupdate = now;
				mydoupdate();
			}
		}

		/* wait for any data or key press */
		if ((rc = select(last_fd + 1, &rfds, NULL, NULL, &tv)) == -1)
		{
			if (errno != EINTR && errno != EAGAIN)
				error_exit("wait_for_keypress: select returned an error! (%d)\n", errno);

			continue;
		}

		if (terminal_changed)
		{
			terminal_changed = 0;
			resize_terminal_do(popup);
		}

		if (max_wait != 0.0 && (get_ts() - max_wait_start) >= max_wait)
			break;

		/* see if any processes have exited (processes started
		 * by matchin regular expressions)
		 */
		check_exited_children();

		if (got_sighup)
		{
			sighup_restart_tails();
		}

		/* any key pressed? */
		if (FD_ISSET(0, &rfds))
		{
			int uc;

			c = getch();
			uc = toupper(c);

			mail = 0;
			redraw_statuslines();

			if (c == KEY_RESIZE || c == KEY_F(5))
			{
				resize_terminal_do(popup);
				do_refresh = 2;
			}
			else if (c == 3)     /* ^C */
			{
				do_exit();
				error_exit("wait_for_keypress: this should not be reached\n");
			}
			else if (c == 26)/* ^Z */
			{
				endwin();

				if (kill(getpid(), SIGTSTP) == -1)
					error_exit("wait_for_keypress: problem suspending\n");

				mydoupdate();
			}
			else if (c == 8 || c == KEY_F(1))       /* HELP (^h / F1) */
			{
				show_help(what_help);
			}
			else if ((cursor_shift == 0 && c == KEY_LEFT) || (cursor_shift == 1 && c == KEY_SLEFT))
			{
				if (popup != NULL && popup -> x_off > 0)
				{
					popup -> x_off--;
					move_panel(popup -> pwin, popup -> y_off, popup -> x_off);
					mydoupdate();
				}
				else
					wrong_key();
			}
			else if (cursor_shift == 0 && c == KEY_UP)
			{
				if (popup != NULL && popup -> y_off > 0)
				{
					popup -> y_off--;
					move_panel(popup -> pwin, popup -> y_off, popup -> x_off);
					mydoupdate();
				}
				else
					wrong_key();
			}
			else if ((cursor_shift == 0 && c == KEY_RIGHT) || (cursor_shift == 1 && c == KEY_SRIGHT))
			{
				if (popup != NULL && (popup -> x_off + popup -> width) < max_x)
				{
					popup -> x_off++;
					move_panel(popup -> pwin, popup -> y_off, popup -> x_off);
					mydoupdate();
				}
				else
					wrong_key();
			}
			else if (cursor_shift == 0 && c == KEY_DOWN)
			{
				if (popup != NULL && (popup -> y_off + popup -> height) < max_y)
				{
					popup -> y_off++;
					move_panel(popup -> pwin, popup -> y_off, popup -> x_off);
					mydoupdate();
				}
				else
					wrong_key();
			}
			else
			{
				break;
			}

			/* if not a valid key, then reset */
			c = -1;
		}

		/* go through all fds */
		for(loop=0; loop<nfd; loop++)
		{
			int deleted_entry_in_array = 0;
			proginfo *cur = &pi[loop];
			NEWWIN *status = pi[loop].status;

			if (cur -> paused)
				continue;

			do
			{
				if (cur -> fd == -1)
					continue;

				if (FD_ISSET(cur -> fd, &rfds))
				{
					char *buffer = NULL;
					int nbytes, len_il = cur -> incomplete_line ? strlen(cur -> incomplete_line) : 0;

					buffer = myrealloc(cur -> incomplete_line, len_il + 65535 + 1, "wait_for_keypress: input-buffer(1)");
					cur -> incomplete_line = NULL;

					nbytes = read(cur -> fd, &buffer[len_il], 65535);

					/* read error or EOF? */
					if ((nbytes == -1 && errno != EINTR && errno != EAGAIN) || (nbytes == 0 && cur -> is_stdin == MY_FALSE))
					{
						if (last_changed_window == cur) last_changed_window = NULL;
						deleted_entry_in_array = close_window(loop, cur);

						if (deleted_entry_in_array >= 0)
						{
							do_refresh = 2;
							goto closed_window;
						}
						else if (cur -> do_diff && do_refresh == 0)
						{
							if (do_refresh != 2) do_refresh = 1;
						}
					}
					else if (nbytes != -1)	/* if nbytes == -1 it must be an interrupt while READ */
					{
						char *pnt = buffer;

						/* statistics */
						cur -> bytes_processed += nbytes;

						/* redirect to some other file/pipe? */
						redirect(cur, &buffer[len_il], nbytes, MY_FALSE);

						buffer[len_il + nbytes] = 0x00;

						if (strchr(&buffer[len_il], '\n'))
						{
							if (cur -> cont) /* reconnect lines with '\' */
							{
								char *contsearch = pnt;

								while((contsearch = strstr(contsearch, "\\\n")))
								{
									memmove(contsearch, contsearch + 2, strlen(contsearch + 2) + 1);
								}
							}

							/* remember this window as it might be displayed in the GUI
							 * terminal window
							 */
							last_changed_window = cur;

							/* display lines */
							for(;*pnt;)
							{
								char *end = strchr(pnt, '\n');
								if (!end)
									break;

								*end = 0x00;

								/* gen stats */
								store_statistics(cur, now);

								/* is this the output of a program which we should diff and such? */
								if (cur -> do_diff)
								{
									store_for_diff(cur, pnt);
								}
								else /* no, just output */
								{
									if (do_refresh == 0)
										do_refresh = 1;	/* after update interval, update screen */

									emit_to_buffer_and_term(loop, cur, pnt);
								}

								pnt = end + 1;
							}
						}

						if (*pnt != 0x00)
						{
							int line_len = strlen(pnt) + 1;
							cur -> incomplete_line = mymalloc(line_len, "wait_for_keypress: incomplete line(2)");
							memcpy(cur -> incomplete_line, pnt, line_len);
						}

						if (do_refresh)
						{
							/* display statusline? */
							update_statusline(status, loop, cur);

							update_panels();
						}
					}

					myfree(buffer);
				}

				/* close idle, if requested */
				if (cur -> close_idle > 0 && (time(NULL) - cur -> lastevent) > cur -> close_idle)
				{
					if (last_changed_window == cur) last_changed_window = NULL;
					deleted_entry_in_array = close_window(loop, cur);
					if (deleted_entry_in_array >= 0)
					{
						do_refresh = 2;
						goto closed_window;
					}
				}

				if (cur -> mark_interval)
				{
					if ((time(NULL) - cur -> lastevent) >= cur -> mark_interval)
					{
						add_markerline(loop, cur, MARKER_IDLE, NULL);

						if (do_refresh == 0) do_refresh = 1;

						cur -> lastevent = time(NULL);
					}
				}

				cur = cur -> next;
			}
			while(cur);

			if (deleted_entry_in_array > 0)
				break;
		}

		if (check_for_exited_processes())
			last_changed_window = NULL;
closed_window:

		/* any window changed? then we may want to update the terminal window header */
		if ((last_changed_window != NULL || mail != prev_mail_status) && set_title != NULL)
		{
			prev_mail_status = mail;

			draw_gui_window_header(last_changed_window);
		}
	}

	return c;
}
