#include "version.h"

#define _LARGEFILE64_SOURCE	/* required for GLIBC to enable stat64 and friends */
#include <ctype.h>
#include <sys/types.h>
#include <pwd.h>
#include <dirent.h>
#include <errno.h>
#include <fcntl.h>
#include <limits.h>
#include <regex.h>
#if !defined(__APPLE__) && !defined(__CYGWIN__)
#include <search.h>
#endif
#include <math.h>
#include <signal.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <glob.h>
#include <sys/ioctl.h>
#include <sys/stat.h>
#ifndef AIX
#include <sys/termios.h> /* needed on Solaris 8 */
#endif
#include <sys/time.h>
#include <sys/wait.h>
#include <unistd.h>
#if defined(__GLIBC__) && ( __GLIBC__ < 2 || (__GLIBC__ == 2 && __GLIBC_MINOR__ < 2))
#include <sys/sysinfo.h>
#endif
#include <sys/utsname.h>

#include "mt.h"
#include "error.h"
#include "my_pty.h"
#include "utils.h"
#include "colors.h"
#include "scrollback.h"
#include "help.h"
#include "term.h"


/* #define KEYB_DEBUG */

proginfo *pi = NULL;
buffer *lb = NULL;
int nfd = 0;
int max_y, max_x;
char terminal_changed = 0;
char split = 0;
char banner = 1;
NEWWIN **splitlines = NULL;
int n_splitlines = 0;
NEWWIN *menu_win = NULL;
int mode_statusline = 1;
char warn_closed = 1;
int n_colors = -1;
char use_colors = 0;
int min_n_bufferlines = 25;
int path_max = 0;
int heartbeat = 0;
color_scheme *cschemes = NULL;
int n_cschemes = 0;
scheme_per_file *spf = NULL;
int n_scheme_per_file = 0;
char do_refresh = 0;
char mail = 0;
char check_for_mail = 5;	/* check for mail every 5 seconds */
char tab_width = 4;		/* some people use 8 */
time_t mt_started;
int *vertical_split = NULL;
int *n_win_per_col = NULL;
int terminal_index = -1;
char prev_term_char = -1;
int n_keybindings = 0;
keybinding *keybindings = NULL;
pid_t children_list[MAX_N_SPAWNED_PROCESSES];
int n_children = 0;
extern int term_type;
char *set_title = NULL;
pid_t tail_proc = 0;	/* process used by checker-proc */
char bright_colors = 0;
char *mail_spool_file = NULL;
struct stat64 msf_info;
off64_t msf_prev_size = 0;
time_t msf_last_check = 0;
time_t hb_time = 0;
int update_interval = 0;
char *tail = "tail";
char afs = 0;	/* abbreviate filesize */

void LOG(char *s, ...)
{
#ifdef _DEBUG
	va_list ap;
	FILE *fh = fopen("log.log", "a+");
	if (!fh)
	{
		endwin();
		printf("error\n");
	}

	va_start(ap, s);
	vfprintf(fh, s, ap);
	va_end(ap);

	fclose(fh);
#endif
}

char *F1 = "For help at any time press F1.";

void init_curses(void)
{
	initscr();
	if (use_colors)
	{
		start_color(); /* don't care if this one failes */
	}
	keypad(stdscr, TRUE);
	cbreak();
	intrflush(stdscr, FALSE);
	leaveok(stdscr, TRUE);
	noecho();
	nonl();
	refresh();
	nodelay(stdscr, FALSE);
	meta(stdscr, TRUE);	/* enable 8-bit input */
	raw();			/* to be able to catch ctrl+c */
	idlok(stdscr, TRUE);	/* may give a little clunky screenredraw */

#ifdef N_CURSES
	if (use_colors)
	{
		init_pair(MY_RED, COLOR_RED, -1);
		init_pair(MY_GREEN, COLOR_GREEN, -1);
		init_pair(MY_YELLOW, COLOR_YELLOW, -1);
		init_pair(MY_BLUE, COLOR_BLUE, -1);
		init_pair(MY_MAGENTA, COLOR_MAGENTA, -1);
		init_pair(MY_CYAN, COLOR_CYAN, -1);
		init_pair(MY_WHITE, COLOR_WHITE, -1);
		init_pair(MY_BLACK, COLOR_BLACK, -1);
	}
	n_colors = 8;
#endif

	max_y = LINES;
	max_x = COLS;
}

void do_resize(int s)
{
	terminal_changed = 1;
}

void setup_for_childproc(int fd, char close_fd_0)
{
	if (close_fd_0) if (-1 == myclose(0)) error_exit("close failed\n");
	if (-1 == myclose(1)) error_exit("close failed\n");
	if (-1 == myclose(2)) error_exit("close failed\n");
	if (close_fd_0) if (-1 == dup(fd)) error_exit("dup failed\n");
	if (-1 == dup(fd)) error_exit("dup failed\n");
	if (-1 == dup(fd)) error_exit("dup failed\n");

	/*
	 * not doing this: it also clears the 'PATH'
	 * I could make first do a getenv for PATH and then put it
	 * back after the clearenv, but what would be the point of
	 * doing the clearenv in the first place?
	 *
	 *	if (clearenv() == -1)
	 *	{
	 *		fprintf(stderr, "WARNING: could not clear environment variables: %s (%d)\n", strerror(errno), errno);
	 *		exit(1);
	 *	}
	 */

	/* set terminal */
	if (putenv("TERM=dumb") == -1)
		fprintf(stderr, "Could not set TERM environment-variable: %s (%d)\n", strerror(errno), errno);

	(void)umask(007);
}

int execute_program(char *execute, char bg)
{
	int status;
	pid_t child;

	if (bg)
	{
		/* to prevent meltdowns, only a limited number of
		 * processes can be executed
		 */
		if (n_children >= MAX_N_SPAWNED_PROCESSES)
			return 0;
	}
	else
		endwin();

	child = fork();
	if (child == 0)
	{
		if (bg)
		{
			int fd = myopen("/dev/null", O_RDWR);
			if (fd == -1)
				error_exit("Failed to open /dev/null\n");

			setup_for_childproc(fd, 1);
		}

		/* start process */
		if (-1 == execlp("/bin/sh", "/bin/sh", "-c", execute, (void *)NULL)) error_exit("execlp of %s failed\n", execute);

		/* if execlp returns, an error occured */
		error_exit("error while starting process!\n");
	}
	else if (child == -1)
	{
		error_exit("failed to fork child process\n");
	}

	if (bg)
	{
		/* remember this childprocess: we'll see if it has
		 * exited in the main-loop
		 */
		children_list[n_children++] = child;
	}
	else
	{
		/* wait for the childprocess to exit */
		if (waitpid(child, &status, 0) == -1)
			error_exit("failt waiting for process to exit!\n");

		/* restore (n)curses */
		mydoupdate();
	}

	return 0;
}

/** create_subwindow_list
 * - in:      int f_index     window number
 *            char ***swlist  pointer to an array of strings
 * - returns: int             number of elements in the array of strings
 * this function creates for a given window (f_index) a list of subwindows:
 * subwindows are created when you're merging the output of several files/
 * commands
 */
int create_subwindow_list(int f_index, char ***swlist)
{
	char **list = NULL;
	int n = 0;
	proginfo *cur = &pi[f_index];

	do
	{
		list = (char **)myrealloc(list, (n + 1) * sizeof(char *), "subwindow list");

		list[n] = mystrdup(cur -> filename, "subwindowlist filename");

		n++;

		cur = cur -> next;
	}
	while(cur);

	*swlist = list;

	return n;
}

void free_re(re *cur_re)
{
	if (cur_re)
	{	
		myfree(cur_re -> regex_str);
		if (cur_re -> use_regex)
			regfree(&cur_re -> regex);

		myfree(cur_re -> cmd);
	}
}

void free_subentry(proginfo *entry)
{
	int loop;

	/* free up filters */
	for(loop=0; loop<entry -> n_strip; loop++)
		myfree((entry -> pstrip)[loop].del);
	myfree(entry -> pstrip);

	/* free all those allocated memory blocks */
	myfree(entry -> filename);
	myfree(entry -> field_del);
	if (entry -> status)
		delete_popup(entry -> status);
	if (entry -> data)
		delete_popup(entry -> data);

	/* free buffers for diff (if any) */
	if (entry -> bcur)
		delete_array(entry -> bcur, entry -> ncur);
	if (entry -> bprev)
		delete_array(entry -> bprev, entry -> nprev);

	/* delete regular expressions */
	for(loop=0; loop<entry -> n_re; loop++)
		free_re(&(entry -> pre)[loop]);
	myfree(entry -> pre);

	/* stop process */
	stop_process(entry -> pid);

	/* close pipe to (tail) process */
	myclose(entry -> fd);
	myclose(entry -> wfd);
}

void store_for_diff(proginfo *cur, char *string)
{
	cur -> bcur = (char **)myrealloc(cur -> bcur, sizeof(char *) * (cur -> ncur + 1), "difference list");

	(cur -> bcur)[cur -> ncur] = mystrdup(string, "store_for_diff");

	cur -> ncur++;
}

void buffer_replace_pi_pointers(int f_index, proginfo *org, proginfo *new)
{
	int loop;

	for(loop=0; loop<lb[f_index].curpos; loop++)
	{
		if ((lb[f_index].pi)[loop] == org)
		{
			if (!new)
			{
				myfree((lb[f_index].Blines)[loop]);
				(lb[f_index].Blines)[loop] = NULL;
				(lb[f_index].pi)[loop] = NULL;
			}
			else
			{
				(lb[f_index].pi)[loop] = new;
			}
		}
	}
}

char delete_entry(int f_index, proginfo *sub)
{
	char delete_all = 0;

	/* no children? then sub must be pointing to current */
	if (pi[f_index].next == NULL)
	{
		sub = NULL;
	}

	/* stop the process(es) we're watching ('tail' most of the time) */
	if (sub == NULL) /* delete all? */
	{
		proginfo *cur = &pi[f_index];

		do
		{
			free_subentry(cur);
			cur = cur -> next;
		}
		while(cur);

		/* free the subwindows (if any) */
		cur = pi[f_index].next;
		while(cur)
		{
			proginfo *dummy = cur -> next;
			myfree(cur);
			cur = dummy;
		}

		delete_all = 1;
	}
	else
	{
		free_subentry(sub);
	}

	/* remove entry from array */
	if (sub == NULL) /* delete entry in the main array */
	{
		int n_to_move = (nfd - f_index) - 1;

		/* free buffers */
		delete_array(lb[f_index].Blines, lb[f_index].curpos);

		if (n_to_move > 0)
		{
			int loop;

			/* update buffer proginfo-pointers */
			for(loop=f_index + 1; loop<nfd; loop++)
			{
				/* replace lb[f_index].pi -> pi[loop] met pi[loop-1] */
				buffer_replace_pi_pointers(loop, &pi[loop], &pi[loop - 1]);
			}

			/* prog info */
			memmove(&pi[f_index], &pi[f_index+1], sizeof(proginfo) * n_to_move);
			/* buffers */
			memmove(&lb[f_index], &lb[f_index+1], sizeof(buffer) * n_to_move);
		}

		nfd--;

		/*  shrink array */
		if (nfd == 0)	/* empty? */
		{
			myfree(pi);
			pi = NULL;
			myfree(lb);
			lb = NULL;
		}
		else		/* not empty, just shrink */
		{
			pi = (proginfo *)myrealloc(pi, nfd * sizeof(proginfo), "proginfo list");
			lb = (buffer *)myrealloc(lb, nfd * sizeof(buffer), "nfd list");
		}
	}
	else		/* delete sub */
	{
		if (sub != &pi[f_index])	/* not in main array? */
		{
			proginfo *parent = &pi[f_index];

			/* find parent of 'sub' */
			while (parent -> next != sub)
				parent = parent -> next;

			parent -> next = sub -> next;

			buffer_replace_pi_pointers(f_index, sub, NULL);

			myfree(sub);
		}
		else				/* main array, find next */
		{
			proginfo *oldnext = pi[f_index].next;
			/* first, delete the entries of that entry (which is in the array) */
			buffer_replace_pi_pointers(f_index, &pi[f_index], NULL);
			/* then, 'rename' the pointers (original address -> address in array)... */
			buffer_replace_pi_pointers(f_index, oldnext, &pi[f_index]);
			/* and move the object to the array */
			memmove(&pi[f_index], oldnext, sizeof(proginfo));
			/* free the obsolete entry */
			myfree(oldnext);
		}
	}

	return delete_all;
}

/** do_exit
 * - in:      int sig
 * - returns: nothing (doesn't return!)
 * this function is called when MultiTail receives the TERM-signal. it is also
 * called by, for example, wait_for_keypress when ^c is pressed. it stops all
 * child-processes, ends the curses-library and exits with SUCCESS status
 */
void do_exit(int sig)
{
	proginfo *cur;
	int loop;

	/* kill tail processes */
	for(loop=0; loop<nfd; loop++)
	{
		cur = &pi[loop];

		do
		{
			stop_process(cur -> pid);
			cur = cur -> next;
		}
		while(cur);
	}

	if (set_title)
		gui_window_header("");

#ifdef _DEBUG
	/* to be able to check for memory leaks, (try to :-]) free up the whole
	 * array of windows and such. normally, don't do this: libc & kernel will
	 * take care of that
	 */
	dump_mem(SIGHUP);
#endif

	endwin();

	exit(EXIT_SUCCESS);
}

char * find_most_recent_file(char *filespec, char *cur_file)
{
	glob_t files;
	char *selected_file = NULL;
	unsigned int loop;
	time_t prev_ts = (time_t)0;
	off64_t dummy;

	LOG("find file for %s\n", filespec);

	/* get timestamp of previous file */
	if (cur_file)
	{
		if (file_info(cur_file, &dummy, &prev_ts) == -1)
		{
			LOG("could not get fileinfo for %s\n", cur_file);
			prev_ts = (time_t)0;
		}
	}

	/* get list of files that match */
#if defined(__APPLE__) || defined(__CYGWIN__)
	if (glob(filespec, GLOB_ERR | GLOB_NOSORT, NULL, &files) != 0)
#else
		if (glob(filespec, GLOB_ERR | GLOB_NOSORT | GLOB_NOESCAPE, NULL, &files) != 0)
#endif
		{
			LOG("glob failed %d\n", errno);
			return NULL;
		}

	LOG("number of files found: %d\n", files.gl_pathc);

	/* see if any of them is more recent than the current one */
	for(loop=0; loop<files.gl_pathc; loop++)
	{
		time_t new_ts;

		/* current file? then skip it */
		if (cur_file != NULL && strcmp(cur_file, files.gl_pathv[loop]) == 0)
			continue;

		/* get the modificationtime of this found file */
		if (file_info(files.gl_pathv[loop], &dummy, &new_ts) == -1)
		{
			LOG("loop: failed fetching info on file %s: %d\n", files.gl_pathv[loop], errno);
			new_ts = (time_t)0;
		}

		/* more recent? */
		if (new_ts > prev_ts)
		{
			selected_file = files.gl_pathv[loop];
			prev_ts = new_ts;
		}
	}

	/* found a file? then remember the filename */
	if (selected_file != NULL)
	{
		selected_file = mystrdup(selected_file, "find_most_recent_file: new file filename");
	}

	globfree(&files);

	return selected_file;
}

int start_tail(char *filename, char retry_open, char follow_filename, int initial_tail, int *pipefd)
{
	pid_t pid;

	/* start child process */
	if ((pid = fork()) == 0)
	{
		char *pars[8], *posix_version = NULL;
		int npars = 0;
		char nlines_buffer[32];

		setup_for_childproc(pipefd[1], 0);

		/* Linux' tail has the --retry option, but not all
		 * other UNIX'es have this, so I implemented it
		 * myself
		 */
		if (retry_open)
		{
			int rc;
			struct stat64 buf;

			for(;;)
			{
				rc = stat64(filename, &buf);
				if (rc == -1)
				{
					if (errno != ENOENT)
					{
						fprintf(stderr, "Error while looking for file %s: %d\n", filename, errno);
						exit(EXIT_FAILURE);
					}
				}
				else if (rc == 0)
					break;

				usleep(WAIT_FOR_FILE_DELAY * 1000);
			}
		}

		/* the command to start */
		pars[npars++] = tail;

		/* create command for take last n lines & follow and start tail */

		/* get the posix compliance level */
		posix_version = getenv("_POSIX2_VERSION");

		/* follow filename is only supported on *BSD and Linux */
#if defined(__FreeBSD__) || defined(__NetBSD__) || defined(__OpenBSD__) || defined(linux) || defined(__CYGWIN__) || defined(__APPLE__) || defined(__GNU__)
		if (follow_filename)
		{
#if defined(linux) || defined(__CYGWIN__) || defined(__GNU__)
			pars[npars++] = "--follow=name";
#elif defined(__OpenBSD__)
			pars[npars++] = "-f";
#else
			pars[npars++] = "-F";
#endif
			/* number of lines to tail initially */
			pars[npars++] = "-n";
			sprintf(nlines_buffer, "%d", initial_tail);
			pars[npars++] = nlines_buffer;
		}
		else
#endif
		{
			/* check the posix compliance level */
			if (posix_version && atoi(posix_version) >= 200112)
			{
				pars[npars++] = "-f";
				pars[npars++] = "-n";
				sprintf(nlines_buffer, "%d", initial_tail);
				pars[npars++] = nlines_buffer;
			}
			else
			{
				/* number of lines to tail initially and 'follow file' ('f') */
				sprintf(nlines_buffer, "-%dlf", initial_tail);
				pars[npars++] = nlines_buffer;
			}
		}

		/* add the filename to monitor */
		pars[npars++] = filename;
		pars[npars] = NULL;

		/* run tail! */
		if (-1 == execvp(pars[0], pars)) error_exit("execvp of tail failed");

		/* if execlp returns, an error occured */
		error_exit("error while starting process!\n");
	}

	return pid;
}

void check_proc_sigh(int sig)
{
	signal(SIGTERM, SIG_IGN);

	stop_process(tail_proc);

	exit(1);
}

void sighup_handler(int sig)
{
	int loop;

	signal(SIGHUP, SIG_IGN);

	for(loop=0; loop<nfd; loop++)
	{
		proginfo *cur = &pi[loop];

		do
		{
			if (!cur -> is_command)
			{
				int pipefd[2];

				stop_process(cur -> pid);

				if (myclose(cur -> fd) == -1) error_exit("closing read filedescriptor failed\n");
				if (cur -> fd != cur -> wfd)
				{
					if (myclose(cur -> wfd) == -1) error_exit("closing write filedescriptor failed\n");
				}

				/* create a pipe, will be to child-process */
				if (-1 == pipe(pipefd)) error_exit("error creating pipe\n");

				cur -> pid = start_tail(cur -> filename, cur -> retry_open, cur -> follow_filename, min_n_bufferlines, pipefd);
				cur -> fd = pipefd[0];
				cur -> wfd = pipefd[1];
			}

			cur = cur -> next;
		} while(cur);
	}

	signal(SIGHUP, sighup_handler);
}

int start_proc(proginfo *cur, int initial_tail)
{
	if (cur -> is_command)
	{
		int fd_master, fd_slave;

		/* allocate pseudo-tty & fork*/
		cur -> pid = get_pty_and_fork(&fd_master, &fd_slave);
		if (-1 == cur -> pid) error_exit("failed to create pseudo-tty & fork\n");

		/* child? */
		if (cur -> pid == 0)
		{
			/* reset signal handler for SIGTERM */
			signal(SIGTERM, SIG_DFL);

			/* sleep if requested and only when 2nd or 3d (etc.) execution time */
			if (cur -> restart && cur -> first == 0)
				sleep(cur -> restart);

			/* connect slave-fd to stdin/out/err */
			setup_for_childproc(fd_slave, 1);

			/* start process */
			if (-1 == execlp("/bin/sh", "/bin/sh", "-c", cur -> filename, (void *)NULL)) error_exit("execlp of %s failed\n", cur -> filename);

			/* if execlp returns, an error occured */
			error_exit("error while starting process!\n");
		}

#if defined(sun) || defined(__sun) || defined(AIX) || defined(hpux) || defined(__hpux) || defined(OSF1)
		/* these platforms only have the slave-fd available in the childprocess
		 * so don't try to close it as the parent process doesn't have it
		 */
#else
		if (myclose(fd_slave) == -1) error_exit("error closing slave-fd (pseudo tty, fd %d)", fd_slave);
#endif

		/* remember master-fd (we'll read from that one) */
		cur -> fd = fd_master;
		cur -> wfd = fd_master;

		/* next time, sleep */
		cur -> first = 0;
	}
	else
	{
		int pipefd[2];

		/* create a pipe, will be to child-process */
		if (-1 == pipe(pipefd)) error_exit("error creating pipe\n");

		if (cur -> check_interval)
		{
			/* start the process that will check every 'interval' seconds
			 * if a matching file with a more recent mod-time is available
			 */
			cur -> pid = fork();
			if (cur -> pid == 0)
			{
				char *cur_file = NULL, *new_file;

				signal(SIGTERM, check_proc_sigh);

				for(;;)
				{
					/* find (new) file */
					new_file = find_most_recent_file(cur -> filename, cur_file);

					/* if find_most_recent_file returned NOT null, a file was found
					 * which is more recent
					 */
					if (new_file != NULL)
					{
						/* if there was a previous file, see if it is different
						 * from the new filename. this should always be the case!
						 */
						if (cur_file && strcmp(new_file, cur_file) != 0)
						{
							LOG("killing process for %s\n", cur_file);
							if (kill(tail_proc, SIGTERM) == -1)
								error_exit("failed to stop pid %d\n", tail_proc);

							usleep(200);

							kill(SIGKILL, tail_proc);
						}

						/* remember new filename */
						myfree(cur_file);
						cur_file = new_file;

						LOG("new_file: %s\n", cur_file);

						/* and start a proc process */
						tail_proc = start_tail(cur_file, cur -> retry_open, cur -> follow_filename, initial_tail, pipefd);
						if (tail_proc == -1)
						{
							LOG("failed starting process for file %s: %d\n", cur_file, errno);
							break;
						}
					}
					else
					{
						LOG("no file found for pattern %s\n", cur -> filename);
					}

					sleep(cur -> check_interval);
				}

				LOG("stopped checking for file pattern %s\n", cur -> filename);

				exit(1);
			}
		}
		else
		{
			cur -> pid = start_tail(cur -> filename, cur -> retry_open, cur -> follow_filename, initial_tail, pipefd);
		}

		cur -> fd = pipefd[0];
		cur -> wfd = pipefd[1];
	}

	if (cur -> pid > -1)
		return 0;

	return -1;
}

int gen_color(char *start, char *end)
{
	char *loop;
	int chk = 0;

	for(loop=start; loop<end; loop++)
	{
		chk ^= *loop;
	}

	return abs(chk) % n_colors;
}

int match_files(char *search_for, char **path, char ***found, char **isdir)
{
	DIR *dir;
	struct dirent *entry;
	char *cur_dir = mymalloc(path_max + 1, "current file");
	char *fname;
	char **list = NULL;
	int nfound = 0;
	size_t fname_size;
	int path_len;
	int s1, s2;
	char *slash = strrchr(search_for, '/');
	if (slash)
	{
		fname = mystrdup(slash + 1, "match_files: filename");
		*(slash + 1) = 0x00;
		*path = mystrdup(search_for, "match_files: current path");
	}
	else
	{
		*path = mystrdup("./", "match_files: current path");
		fname = mystrdup(search_for, "match_files: filename");
	}
	fname_size = strlen(fname);
	path_len = strlen(*path);

	dir = opendir(*path);
	if (!dir)
	{
		free(cur_dir);
		return 0;
	}

	memcpy(cur_dir, *path, path_len + 1);

	while((entry = readdir(dir)) != NULL)
	{
		if ((fname_size == 0 || strncmp(entry -> d_name, fname, fname_size) == 0) && strcmp(entry -> d_name, ".") != 0 &&
				strcmp(entry -> d_name, "..") != 0)
		{
			struct stat finfo;

			/* get filename */
			list = (char **)myrealloc(list, (nfound + 1) * sizeof(char *), "directory list");
			list[nfound] = mystrdup(entry -> d_name, "match_files dir entry");

			/* check if the file is a directory */
			*isdir = (char *)myrealloc(*isdir, (nfound + 1) * sizeof(char), "is-dir list");
			strncpy(&cur_dir[path_len], entry -> d_name, max(0,path_max - path_len));
			if (stat(cur_dir, &finfo) == -1)
			{
				if (errno != ENOENT)	/* file did not disappear? then something is very wrong */
					error_exit("stat error");
			}
			(*isdir)[nfound] = S_ISDIR(finfo.st_mode)?1:0;

			nfound++;
		}
	}

	if (closedir(dir) == -1)
		error_exit("closedir failed");

	/*	qsort( (void *)list, (size_t)nfound, sizeof(char *), compare_filenames); */
	for(s1=0; s1<(nfound - 1); s1++)
	{
		for(s2=s1+1; s2<nfound; s2++)
		{
			if (strcasecmp(list[s2], list[s1]) < 0)
			{
				char *fdummy = list[s1], ddummy = (*isdir)[s1];

				list[s1] = list[s2];
				(*isdir)[s1] = (*isdir)[s2];
				list[s2] = fdummy;
				(*isdir)[s2] = ddummy;
			}
		}
	}

	*found = list;

	myfree(fname);
	myfree(cur_dir);

	return nfound;
}

char generate_string(char *whereto, void **list, char type, int wcols, int index)
{
	char invert = 0;

	if (type == SEL_WIN)
	{
		snprintf(whereto, wcols + 1, "%02d %s", index, ((proginfo *)list)[index].filename);
		invert = ((proginfo *)list)[index].hidden | ((proginfo *)list)[index].paused;
	}
	else if (type == SEL_SUBWIN || type == SEL_FILES)
		strncpy(whereto, ((char **)list)[index], min(strlen(((char **)list)[index]), wcols) + 1);
	else if (type == SEL_CSCHEME)
		strncpy(whereto, ((color_scheme *)list)[index].name, min(strlen(((color_scheme *)list)[index].name), wcols) + 1);
	else if (type == SEL_COLORS)
	{
		LOG("index: %d\n", index + 1);
		LOG("p: %p\n", MY_COLORS[index + 1]);
		LOG("str: %s\n", MY_COLORS[index + 1]);
		strcpy(whereto, MY_COLORS[index + 1]);
	}

	whereto[min(strlen(whereto), wcols)] = 0x00;

	return invert;
}

int find_sb_string(void **list, char type, int nentries, char *compare_string)
{
	int index, len = strlen(compare_string);

	for(index=0; index<nentries; index++)
	{
		char *cur = NULL;

		if (type == SEL_WIN)
			cur = ((proginfo *)pi)[index].filename;
		else if (type == SEL_SUBWIN || type == SEL_FILES)
			cur = ((char **)list)[index];
		else if (type == SEL_CSCHEME)
			cur = ((color_scheme *)list)[index].name;
		else if (type == SEL_COLORS)
			cur = MY_COLORS[index + 1];

		if (strncmp(cur, compare_string, len) == 0)
			return index;
	}

	return -1;
}

int selection_box(void **list, char *needs_mark, int nlines, char type, int what_help, char *heading)
{
	NEWWIN *mywin;
	int wlines = min(nlines, (max_y - 1) - 4);
	int total_win_size = wlines + 4;
	int wcols = (max_x / 4) - 4;
	int pos = 0, ppos = -1, offs = 0, poffs = -1;
	int loop = 0, sel = -1;
	char first = 1;
	char *dummy = (char *)mymalloc(wcols + 1, "windowname string");
	char *selstr = (char *)mymalloc(path_max + 1, "selection_box: enter selection"), selfound = 0;

	selstr[0] = 0x00;

	mywin = create_popup(total_win_size, max_x / 4);

	for(;;)
	{
		int c;

		/* draw list */
		if (pos != ppos)
		{
			int entries_left = (nlines - pos);

			werase(mywin -> win);

			if (heading)
				win_header(mywin, heading);
			else if (type == SEL_WIN)
				win_header(mywin, "Select window");
			else if (type == SEL_SUBWIN)
				win_header(mywin, "Select subwindow");
			else if (type == SEL_FILES)
				win_header(mywin, "Select file");
			else if (type == SEL_CSCHEME)
				win_header(mywin, "Select color scheme");
			else if (type == SEL_COLORS)
				win_header(mywin, "Select color");

			for(loop=0; loop<min(entries_left, wlines); loop++)
			{
				char invert = generate_string(dummy, list, type, wcols, loop + pos);
				if (loop == offs)
					inverse_on(mywin);
				if (invert) color_on(mywin, MY_YELLOW);
				if (needs_mark && needs_mark[loop + pos])
					mvwprintw(mywin -> win, loop + 2, 1, "*");
				mvwprintw(mywin -> win, loop + 2, 2, "%s", dummy);
				if (invert) color_off(mywin, MY_YELLOW);
				if (loop == offs)
					inverse_off(mywin);
			}

			box(mywin -> win, 0, 0);

			ppos = pos;
			poffs = offs;
		}
		else if (poffs != offs)
		{
			char invert = generate_string(dummy, list, type, wcols, poffs + pos);
			if (invert) color_on(mywin, MY_YELLOW);
			mvwprintw(mywin -> win, poffs + 2, 2, "%s", dummy);
			if (invert) color_off(mywin, MY_YELLOW);

			invert = generate_string(dummy, list, type, wcols, offs + pos);

			inverse_on(mywin);
			if (invert) color_on(mywin, MY_YELLOW);
			if (needs_mark && needs_mark[offs + pos])
				mvwprintw(mywin -> win, loop + 2, 1, "*");
			mvwprintw(mywin -> win, offs + 2, 2, "%s", dummy);
			if (invert) color_off(mywin, MY_YELLOW);
			inverse_off(mywin);

			poffs = offs;
		}

		if (first)
		{
			first = 0;
			color_on(mywin, MY_GREEN);
			mvwprintw(mywin -> win, total_win_size - 2, 2, "Press ^G to abort");
			color_off(mywin, MY_GREEN);
		}
		else
		{
			int loop, len = strlen(selstr);

			for(loop=0; loop<wcols; loop++)
				mvwprintw(mywin -> win, total_win_size - 2, 1 + loop, " ");

			if (!selfound) color_on(mywin, MY_RED);
			mvwprintw(mywin -> win, total_win_size - 2, 1, "%s", &selstr[max(0, len - wcols)]);
			if (!selfound) color_off(mywin, MY_RED);
		}

		mydoupdate();

		c = wait_for_keypress(what_help, 0, mywin, 1);

		if (c == KEY_UP)
		{
			if ((offs + pos) > 0)
			{
				if (offs)
					offs--;
				else
					pos--;
			}
			else
			{
				wrong_key();
			}
		}
		else if (c == KEY_DOWN)
		{
			if ((pos + offs) < (nlines-1))
			{
				if (offs < (wlines-1))
					offs++;
				else
					pos++;
			}
			else
			{
				wrong_key();
			}
		}
		else if (c == KEY_NPAGE)
		{
			if ((pos + offs) < (nlines - 1))
			{
				pos += min(wlines, (nlines - 1) - (pos + offs));
			}
			else
			{
				wrong_key();
			}
		}
		else if (c == KEY_PPAGE)
		{
			if ((pos + offs - wlines) >= 0)
			{
				if (pos > wlines)
				{
					pos -= wlines;
				}
				else
				{
					pos -= (wlines - offs);
					offs = 0;
				}
			}
			else if (offs > 0)
			{
				offs = 0;
			}
			else if (pos > 0)
			{
				pos = 0;
			}
			else
			{
				wrong_key();
			}
		}
		else if (c == KEY_ENTER || c == 13 || c == 10)
		{
			sel = pos + offs;
			break;
		}
		else if (c == 7)
		{
			break;
		}
		else if ((c > 31 && c != 127) || (c == KEY_BACKSPACE))
		{
			int index, curlen;

			curlen = strlen(selstr);
			if (c == KEY_BACKSPACE)
			{
				if (curlen > 0)
					selstr[curlen - 1] = 0x00;
				else
					wrong_key();
			}
			else if (curlen < path_max)
			{
				selstr[curlen] = c;
				selstr[curlen + 1] = 0x00;
			}
			else
				wrong_key();


			curlen = strlen(selstr);
			if (curlen > 0)
			{
				index = find_sb_string(list, type, nlines, selstr);
				if (index != -1)
				{
					ppos = -1;
					sel = pos = index;
					selfound = 1;
				}
				else
				{
					selfound = 0;
				}
			}
		}
		else
		{
			wrong_key();
		}
	}

	delete_popup(mywin);

	myfree(dummy);
	myfree(selstr);

	return sel;
}

int select_window(int what_help, char *heading)
{
	return selection_box((void **)pi, NULL, nfd, SEL_WIN, what_help, heading);
}

proginfo * select_subwindow(int f_index, int what_help, char *heading)
{
	proginfo *cur = NULL;
	char **list;
	int list_n, index, loop;

	if (f_index == -1)
		return NULL;

	list_n = create_subwindow_list(f_index, &list);

	index = selection_box((void **)list, NULL, list_n, SEL_SUBWIN, what_help, heading);

	if (index != -1)
	{
		cur = &pi[f_index];
		for(loop=0; loop<index; loop++)
		{
			cur = cur -> next;
		}
	}

	delete_array(list, list_n);

	return cur;
}

char * select_file(char *input, int what_help)
{
	char **list = NULL, *isdir = NULL, *path = NULL;
	char *new_fname = NULL;
	struct stat64 statbuf;
	int list_n, index;

	list_n = match_files(input, &path, &list, &isdir);
	if (list_n == 0)
	{
		myfree(path);
		flash();
		return NULL;
	}

	index = selection_box((void **)list, isdir, list_n, SEL_FILES, what_help, NULL);
	if (index != -1)
	{
		new_fname = (char *)mymalloc(path_max + 1, "new filename");

		sprintf(new_fname, "%s%s", path, list[index]);

		if (stat64(new_fname, &statbuf) == -1)
		{
			myfree(new_fname);
			new_fname = NULL;
			flash();
		}
		else
		{
			if (S_ISDIR(statbuf.st_mode))
			{
				strcat(new_fname, "/");
			}
		}
	}

	delete_array(list, list_n);

	myfree(isdir);
	myfree(path);

	return new_fname;
}

int choose_color(char *string, proginfo *cur, NEWWIN *win, color_offset_in_line *cmatches, int *n_cmatches, char **new_string)
{
	int color = -1, loop;
	char *sp = string, *dummy = NULL;
	regmatch_t colormatches[MAX_N_RE_MATCHES];

	*new_string = NULL;

	LOG("terminal: %d\n", cur -> term_emul);

	if (cur -> term_emul != TERM_IGNORE)
	{
		int len = strlen(string), new_offset = 0;
		LOG("interp terminal\n");

		*new_string = (char *)mymalloc(len + 1, "new string");

		*n_cmatches = 0;

		/* FIXME: this is ANSI-only and should be moved to some other function */
		for(loop=0; loop<len; loop++)
		{
			if (string[loop] == 27)
			{
				int ci = -1;

				if (string[loop + 1] == '[')
				{
					int find_cmd;

					for(find_cmd=loop + 2; find_cmd < len; find_cmd++)
					{
						char cur_char = tolower(string[find_cmd]);
						if (cur_char >= 'a' && cur_char <= 'z')
							break;
					}

					if (string[find_cmd] == 'm')
					{
						int cc;
						char *sc;

						string[find_cmd] = 0x00;

						sc = strchr(&string[loop + 2], ';');
						if (sc)
							cc = atoi(sc + 1);
						else
							cc = atoi(&string[loop + 2]);
						if (cc == 34)
							ci = MY_BLUE;
						else if (cc == 32)
							ci = MY_GREEN;
						else if (cc == 36)
							ci = MY_CYAN;
						else if (cc == 31)
							ci = MY_RED;
						else if (cc == 35)
							ci = MY_RED; /* FIXME: purple */
						else if (cc == 33)
							ci = MY_YELLOW; /* FIXME: brown */
						else if (cc == 37)
							ci = MY_WHITE; /* FIXME: gray */
						else
							ci = MY_BLACK;

						string[find_cmd] = 'm';
					}

					loop = find_cmd;
				}

				if (ci != -1 && *n_cmatches < MAX_N_RE_MATCHES)
				{
					cmatches[*n_cmatches].start = new_offset;
					cmatches[*n_cmatches].end = -1;
					cmatches[*n_cmatches].color_index = ci;
					(*n_cmatches)++;
				}
			}
			else
			{
				(*new_string)[new_offset++] = string[loop];
			}
		}
		for(loop=0; loop<(*n_cmatches - 1); loop++)
			cmatches[loop].end = cmatches[loop + 1].start;
		cmatches[*n_cmatches - 1].end = new_offset;

		(*new_string)[new_offset] = 0x00;

		for(loop=0; loop<(*n_cmatches); loop++)
			LOG("color %d: %d-%d / %d\n", loop, cmatches[loop].start, cmatches[loop].end, cmatches[loop].color_index);

		return MY_WHITE;
	}

	/* find color */
	switch(cur -> colorize)
	{
		case 'i':
			color = cur -> color_index;
			break;

		case 's':
			/* use the program name for the color */
			if (strlen(string) >= 16)
			{
				sp = strchr(&string[16], ' ');
			}

			while(sp && isspace(*sp)) sp++;

			if (sp)
			{
				char *end1 = strchr(sp, '[');
				char *end2 = strchr(sp, ' ');
				char *end3 = strchr(sp, ':');
				char *end = NULL;

				end = end1;
				if ((end2 && end2 < end) || (end == NULL))
					end = end2;
				if ((end3 && end3 < end) || (end == NULL))
					end = end3;

				if (end)
					color = gen_color(sp, end);
				else
					color = gen_color(sp, &sp[strlen(sp)]);
			}
			else
			{
				color = 0;
			}
			break;

		case 'f':
			/* user selected field for coloring */
			{
				int nlen = strlen(cur -> field_del);

				for(loop=0; loop<cur -> field_nr; loop++)
				{
					sp = strstr(sp, cur -> field_del);
					while(sp)
					{
						if (strncmp(sp, cur -> field_del, nlen) == 0)
						{
							sp += nlen;
						}
						else
						{
							break;
						}
					}

					if (!sp)
						break;
				}

				if (sp)
					dummy = strstr(sp, cur -> field_del);

				if (dummy != NULL && sp != NULL)
					color = gen_color(sp, dummy);
				else if (sp)
					color = gen_color(sp, &sp[strlen(sp)]);
				else
					color = 0;
			}
			break;

		case 'm':
			/* complete string */
			color = gen_color(string, &string[strlen(string)]);
			break;

		case 'S':
			/* color scheme */
			for(loop=0; loop<cschemes[cur -> color_scheme].n; loop++)
			{
				int offset = 0;
				char match = 1;

				while(offset < strlen(string) && match)
				{
					int rc;

					match = 0;

					if ((rc = regexec(&cschemes[cur -> color_scheme].regex[loop], &string[offset], MAX_N_RE_MATCHES, colormatches, offset?REG_NOTBOL:0)) == 0)
					{
						int loop2;

						match = 1;

						for(loop2=0; loop2<MAX_N_RE_MATCHES; loop2++)
						{
							if (colormatches[loop2].rm_so == -1)
								break;

							cmatches[*n_cmatches].start = colormatches[loop2].rm_so + offset;
							cmatches[*n_cmatches].end   = colormatches[loop2].rm_eo + offset;
							cmatches[*n_cmatches].color_index = cschemes[cur -> color_scheme].color[loop];

							offset = max(offset, colormatches[loop2].rm_eo + offset);

							(*n_cmatches)++;
							if (*n_cmatches == MAX_COLORS_PER_LINE)
								return -1;
						}
					}
					else	/* an error occured */
					{
						char *error = convert_regexp_error(rc, &cschemes[cur -> color_scheme].regex[loop]);

						if (error)
						{
							wprintw(win -> win, "%s", error);
							myfree(error);
						}
					}
				}
			}
			break;

	}

	return color;
}

void select_display_start_and_end(char *string, char mode, int wrap_offset, int win_width, int *prt_start, int *prt_end)
{
	int nbytes = strlen(string);

	*prt_start = 0;
	*prt_end = nbytes;

	/* figure out what to display (because of linewraps and such) */
	if (mode == 'a')
	{
		/* default is everything */
	}
	else if (mode == 'l')
	{
		*prt_end = min(nbytes, win_width);
	}
	else if (mode == 'r')
	{
		*prt_start = max(0, *prt_end - win_width);
	}
	else if (mode == 'S')
	{
		*prt_start = find_char_offset(string, ':');
		if (*prt_start == -1)
			*prt_start = 0;
	}
	else if (mode == 's')
	{
		if (nbytes > 16)
			*prt_start = find_char_offset(&string[16], ' ');
		else
			*prt_start = -1;

		if (*prt_start == -1)
			*prt_start = 0;
	}
	else if (mode == 'o')
	{
		*prt_start = min(wrap_offset, nbytes);
	}
}

int find_in_regexec_resultset(regmatch_t *matches, int set_size, int offset)
{
	int index = -1;
	int loop;

	for(loop=0; loop<set_size; loop++)
	{
		if (offset >= matches[loop].rm_so && offset < matches[loop].rm_eo)
		{
			index = loop;
			break;
		}
		else if (matches[loop].rm_so == -1)
		{
			break;
		}
	}

	return index;
}

void draw_tab(NEWWIN *win)
{
	if (tab_width)
	{
		int cury, curx, move, loop;

		/* get current coordinate */
		getyx(win -> win, cury, curx);

		/* calculate new coordinate */
		move = (((curx / tab_width) + 1) * tab_width) - curx;

		for(loop=0; loop<move; loop++)
			waddch(win -> win, ' ');
	}
}

void color_print(NEWWIN *win, proginfo *cur, unsigned char *string, regmatch_t *matches, int matching_regex)
{
	char reverse = 0;
	int loop;
	int color = -1;
	int prt_start = 0, prt_end;
	int mx = -1;
	char use_regex = 0;
	color_offset_in_line cmatches[MAX_COLORS_PER_LINE]; /* max. 80 colors per line */
	int n_cmatches = 0;
	char *use_string = NULL;
	int y, x;

	/* already created a window? */
	if (!win)
		return;

	mx = getmaxx(win -> win);
	getyx(win -> win, y, x);
	if (x != 0 && x != mx)
		wprintw(win -> win, "\n");

#if 0
	LOG("matches: %p, index: %d\n", matches, matching_regex);
	if (matching_regex != -1 && matches != NULL)
	{
		printf("use regex: %c, invert: %d\n", use_regex, (cur -> pre)[matching_regex].invert_regex);
	}
#endif

	/* cur == NULL? Markerline! */
	if (!cur)
	{
		draw_marker_line(win);

		return;
	}

	if (matching_regex != -1)
	{
		use_regex = (cur -> pre)[matching_regex].use_regex;
	}

	/* select color or generate list of offset for colors */
	if (use_colors && cur -> colorize) /* no need to do this for b/w terminals */
	{
		color = choose_color(string, cur, win, cmatches, &n_cmatches, &use_string);

		/* if not using colorschemes (which can have more then one color
		 * per line), set the color
		 */
		if (cur -> colorize != 'S' && cur -> colorize != 'T')
		{
			if (color == MY_BOLD)
				wattron(win -> win, A_BOLD);
			else if (color == MY_BLINK)
				wattron(win -> win, A_BLINK);
			else if (color == MY_REVERSE)
			{
				if (!reverse) inverse_on(win);
				reverse = 1;
			}
			else if (color == MY_UNDERLINE)
				wattron(win -> win, A_UNDERLINE);
			else
				color_on(win, color);
		}
	}

	if (!use_string)
		use_string = mystrdup(string, "disp string");

	/* select start and end of string to display */
	select_display_start_and_end(use_string, cur -> line_wrap, cur -> line_wrap_offset, mx, &prt_start, &prt_end);
	/* line has same length as the window size? then suppress the lf for the next line as it would
	 * give empty lines otherwhise
	 */
	LOG("CP: %d %d - %d (%d) {%s}\n", prt_start, prt_end, mx, prt_end - prt_start, use_string);

	/* print text */
	for(loop=prt_start; loop<prt_end; loop++)
	{
		char re_inv = 0;

		/* find things to invert */
		if (use_regex == 'c' || use_regex == 'C' || use_regex == 'b')
		{
			if (matches != NULL)
			{
				/* is this offset in the string in the given (-> function-parameter)
				 * resultset? then invert
				 */
				if ((cur -> pre)[matching_regex].invert_regex == 0)
					re_inv = 1;
			}
			else
			{
				if ((cur -> pre)[matching_regex].invert_regex)
					re_inv = 1;
			}
		}

		/* find things to colorize */
		if (use_colors && (cur -> colorize == 'S' || cur -> colorize == 'T'))
		{
			int loop2, new_color = -1;

			for(loop2=0; loop2<n_cmatches; loop2++)
			{
				if (loop >= cmatches[loop2].start && loop < cmatches[loop2].end)
				{
					new_color = cmatches[loop2].color_index;
					break;
				}
			}

			if (color != new_color)
			{
				if (color == MY_BOLD)
					wattroff(win -> win, A_BOLD);
				else if (color == MY_BLINK)
					wattroff(win -> win, A_BLINK);
				else if (color == MY_REVERSE)
				{
					if (reverse) inverse_off(win);
					reverse = 0;
				}
				else if (color == MY_UNDERLINE)
					wattroff(win -> win, A_UNDERLINE);
				else
					color_off(win, color);
				color = -1;
			}

			if (new_color != -1 && color != new_color)
			{
				color = new_color;
				if (new_color == MY_BOLD)
					wattron(win -> win, A_BOLD);
				else if (new_color == MY_BLINK)
					wattron(win -> win, A_BLINK);
				else if (new_color == MY_REVERSE)
				{
					if (!reverse) inverse_on(win);
					reverse = 1;
				}
				else if (new_color == MY_UNDERLINE)
					wattron(win -> win, A_UNDERLINE);
				else
					color_on(win, color);
			}
		}

		/* control-characters will be displayed as an inverse '.' */
		if (use_string[loop] < 32  || use_string[loop] == 127)
		{
			if (use_string[loop] != 10 && (use_string[loop] != 9 && tab_width > 0))
			{
				re_inv = 1;
			}
		}

		if (re_inv)
		{
			if (!reverse)
			{
				inverse_on(win);
				reverse = 1;
			}
		}

		if (use_string[loop] == 9)	/* TAB? */
		{
			draw_tab(win);
		}
		else if (use_string[loop] >= 32 && use_string[loop] != 127)	/* control-characters? */
		{
			waddch(win -> win, use_string[loop]);
		}
		else if (use_string[loop] != 10)
		{
			if (use_string[loop] != 13)
				waddch(win -> win, '.');
		}
		else
		{
			waddch(win -> win, '\n');
		}

		if (reverse)
		{
			inverse_off(win);
			reverse = 0;
		}
	}

	if (cur -> colorize && color != -1)
	{
		if (color == MY_BOLD)
			wattroff(win -> win, A_BOLD);
		else if (color == MY_BLINK)
			wattroff(win -> win, A_BLINK);
		else if (color == MY_REVERSE)
			inverse_off(win);
		else if (color == MY_UNDERLINE)
			wattroff(win -> win, A_UNDERLINE);
		else
			color_off(win, color);
	}

	free(use_string);
}

/* check if the current line matches with a regular expression
   returns 0 if a regexp failed to execute
   */
char check_filter(proginfo *cur, char *string, regmatch_t **pmatch, char **error, int *matching_regex, char do_re, char *display)
{
	int loop;
	/* if no regexps, then always matches so it's the default */
	char re_exec_ok = 1;

	*error = NULL;
	*pmatch = NULL;

	*display = 1;

	/* do this regular expression? */
	for(loop=0; loop<cur -> n_re; loop++)
	{
		int rc;

		/* re disabled? */
		if (!(cur -> pre)[loop].use_regex)
		{
			LOG("check_filter: %d disabled\n", loop);
			continue;
		}

		/* match */
		if (pmatch)
		{
			if (*pmatch == NULL)
				*pmatch = (regmatch_t *)mymalloc(sizeof(regmatch_t) * MAX_N_RE_MATCHES, "matching regular expressions");

			rc = regexec(&(cur -> pre)[loop].regex, string, MAX_N_RE_MATCHES, *pmatch, 0);
		}
		else
		{
			rc = regexec(&(cur -> pre)[loop].regex, string, 0, NULL, 0);
		}

		/* no error? */
		if (rc == 0 || rc == REG_NOMATCH)
		{
			/* if there was no match, free up allocated structure */
			if (rc == REG_NOMATCH && pmatch != NULL)
			{
				myfree(*pmatch);
				*pmatch = NULL;
			}

			LOG("[%s] %c %d [%s] %d\n", string, (cur -> pre)[loop].use_regex, (cur -> pre)[loop].invert_regex, (cur -> pre)[loop].regex_str, rc);

			/* match? */
			if (rc == 0)
			{
				*matching_regex = loop;

				/* do not print when matches? */
				if ((cur -> pre)[loop].use_regex == 'v')
				{
					LOG("-ev: matching so suppress");
					*display = 0;
					(cur -> pre)[loop].count++;
					return 1;
				}
			}
			else
			{
				/* do not print when NOT matching? */
				if ((cur -> pre)[loop].use_regex == 'm')
				{
					LOG("-e[m]: NOT matching so suppress");
					*display = 0;
					(cur -> pre)[loop].count++;
					return 1;
				}
			}

			if (do_re && ((rc == 0 && (cur -> pre)[loop].invert_regex == 0) ||
						(rc == REG_NOMATCH && (cur -> pre)[loop].invert_regex == 1)
				     )
			   )
			{
				(cur -> pre)[loop].count++;

				/* execute command if regexp matches? */
				if ((cur -> pre)[loop].use_regex == 'x')
				{
					int str_index, par_len = strlen(string);
					int cmd_len = strlen((cur -> pre)[loop].cmd);
					char *command = mymalloc(cmd_len + 1/* cmd + space */
							+ 1 + (par_len * 2) + 1 + 1, "command string");     /* "string"\0 */
					int loop2;

					memcpy(command, (cur -> pre)[loop].cmd, cmd_len);
					str_index = cmd_len;
					command[str_index++] = ' ';
					command[str_index++] = '\"';
					for(loop2=0; loop2<par_len; loop2++)
					{
						if (string[loop2] == '"' ||
								string[loop2] == '`'
						   )
						{
							command[str_index++] = '\\';
						}

						command[str_index++] = string[loop2];
					}
					command[str_index++] = '\"';
					command[str_index] = 0x00;

					if (execute_program(command, 1) == -1)
						error_exit("Failed to execute command '%s'!", command);

					myfree(command);
				}
				/* sound bell when matches */
				else if (toupper((cur -> pre)[loop].use_regex) == 'B')
				{
					beep();
				}
			}

			if ((rc == 0 && (cur -> pre)[loop].invert_regex == 0) || (rc == REG_NOMATCH && (cur -> pre)[loop].invert_regex == 1))
			{
				return 1;
			}
		}
		/* no match: error actually! */
		else
		{
			*error = convert_regexp_error(rc, &(cur -> pre)[loop].regex);
			re_exec_ok = 0;
			break;
		}
	}

	myfree(*pmatch);
	*pmatch = NULL;
	*matching_regex = -1;

	return re_exec_ok;
}

void do_print(int f_index, proginfo *cur, char *string, regmatch_t *matches, int matching_regex)
{
	/* check filter */
	/* if cur == NULL, then marker line: don't check against filter */
	if (cur)
	{
		color_print(pi[f_index].data, cur, (unsigned char *)string, matches, matching_regex);
	}
	else
	{
		color_print(pi[f_index].data, NULL, NULL, NULL, -1);
	}
}

void do_buffer(int f_index, proginfo *cur, char *string, char filter_match)
{
	/* remember string */
	if (lb[f_index].markset == 'a' || filter_match == 1)
	{
		int curline;

		if (lb[f_index].curpos < lb[f_index].maxnlines || lb[f_index].maxnlines == 0)
		{
			lb[f_index].Blines = (char **)myrealloc(lb[f_index].Blines, sizeof(char *) *
					(lb[f_index].curpos + 1), "lines buffer");

			lb[f_index].pi    = (proginfo **)myrealloc(lb[f_index].pi, sizeof(proginfo *) *
					(lb[f_index].curpos + 1), "lines buffer progrinfo");

			curline = lb[f_index].curpos;
			lb[f_index].curpos++;
		}
		else
		{
			/* delete oldest */
			myfree((lb[f_index].Blines)[0]);
			curline = lb[f_index].maxnlines-1;
			memmove(&(lb[f_index].Blines)[0], &(lb[f_index].Blines)[1], (lb[f_index].maxnlines-1) * sizeof(char *));
			memmove(&(lb[f_index].pi)[0], &(lb[f_index].pi)[1], (lb[f_index].maxnlines-1) * sizeof(proginfo *));
		}

		if (string)
			(lb[f_index].Blines)[curline] = mystrdup(string, "print_and_buffer");
		else
			(lb[f_index].Blines)[curline] = NULL;
		(lb[f_index].pi   )[curline] = cur;
	}
}

void do_strip(proginfo *cur, char *in)
{
	int loop, len = strlen(in);
	int offset = 0;
	char *strip_what = (char *)mymalloc(len, "strip_what");

	memset(strip_what, 0x01, len);

	for(loop=0; loop<cur -> n_strip; loop++)
	{
		if ((cur -> pstrip)[loop].type == STRIP_TYPE_RANGE)
		{
			memset(&strip_what[(cur -> pstrip)[loop].start], 0x00, (cur -> pstrip)[loop].end - (cur -> pstrip)[loop].start);
		}
		else if ((cur -> pstrip)[loop].type == STRIP_TYPE_REGEXP)
		{
			int search_offset = 0;
			regmatch_t matches[MAX_N_RE_MATCHES];

			do
			{
				int new_offset = -1;

				if (regexec(&(cur -> pstrip)[loop].regexp, &in[search_offset], MAX_N_RE_MATCHES, matches, 0) != REG_NOMATCH)
				{
					int match_i;

					for(match_i=0; match_i<MAX_N_RE_MATCHES; match_i++)
					{
						if (matches[match_i].rm_so == -1 ||
								matches[match_i].rm_eo == -1)
							break;

						memset(&strip_what[matches[match_i].rm_so + search_offset], 0x00, matches[match_i].rm_eo - matches[match_i].rm_so);

						new_offset = matches[match_i].rm_eo + search_offset;
					}
				}

				search_offset = new_offset;
			}
			while (search_offset != -1);
		}
		else if ((cur -> pstrip)[loop].type == STRIP_TYPE_COLUMN)
		{
			char *p = in;
			int del_len = strlen((cur -> pstrip)[loop].del);
			int col_i;

			while(strncmp(p, (cur -> pstrip)[loop].del, del_len) == 0) p += del_len;
			if (!*p) continue;
			for(col_i=0; col_i<(cur -> pstrip)[loop].col_nr; col_i++)
			{
				p = strstr(p, (cur -> pstrip)[loop].del);
				if (!p) break;
				while(strncmp(p, (cur -> pstrip)[loop].del, del_len) == 0) p += del_len;
			}
			if (!p) continue;

			/* strip the column */
			while(strncmp(p, (cur -> pstrip)[loop].del, del_len) != 0 && *p != 0x00)
			{
				strip_what[(int)(p - in)] = 0;
				p++;
			}
			/* and strip the delimiter(s) behind it */
			while(strncmp(p, (cur -> pstrip)[loop].del, del_len) == 0)
			{
				int loop2;
				for(loop2=0; loop2<del_len; loop2++)
					strip_what[(int)(p - in) + loop2] = 0;
				p += del_len;
			}
		}
		else
			error_exit("do_strip: unknown strip-command detected (%d)", (cur -> pstrip)[loop].type);
	}

	for(loop=0; loop<len; loop++)
	{
		if (strip_what[loop])
			in[offset++] = in[loop];
	}
	in[offset] = 0x00;

	myfree(strip_what);
}

void emit_to_buffer_and_term(int f_index, proginfo *cur, char *line)
{
	regmatch_t *pmatch = NULL;
	char *error = NULL;
	int matching_regex = -1;
	char display = 1;

	if (line && strlen(line) > 0)
		do_strip(cur, line);

	(void)check_filter(cur, line, &pmatch, &error, &matching_regex, 1, &display);

	/* error happened while processing regexp? */
	if (error)
	{
		do_print(f_index, cur, error, NULL, -1);
		do_buffer(f_index, cur, error, 0);

		myfree(error);
	}

	/* output new text */
	if (display)
	{
		do_print(f_index, cur, line, pmatch, matching_regex);
	}
	do_buffer(f_index, cur, line, display);

	if (pmatch) myfree(pmatch);
}

void update_statusline(NEWWIN *status, int win_nr, proginfo *cur)
{
	if (mode_statusline > 0 && status != NULL && cur != NULL)
	{
		char *dummy;
		int dx, dy;
		int color = -1;
		int statusline_len;
		off64_t	fsize = (off64_t)-1;
		time_t	ts = time(NULL);
		char show_f1 = 0;
		int help_str_offset = 0;
		int total_info_len = 0;

		if (win_nr == terminal_index)
			color = MY_RED;
		else if (mail)
			color = MY_GREEN;

		if ((ts - mt_started) < 5)
			show_f1 = 1;

		if (color != -1)
			color_on(status, color);

		draw_line(status, LINE_BOTTOM);

		inverse_on(status);

		mvwprintw(status -> win, 0, 0, "%02d] %s", win_nr, cur -> win_title?cur -> win_title:cur -> filename);

		if (!cur -> is_command)
			(void)file_info(cur -> filename, &fsize, &ts);
		dummy = ctime(&ts);

		help_str_offset = 4 + strlen(cur -> win_title?cur -> win_title:cur -> filename); /* 4: '%02d] '!! (window nr.) */
		statusline_len = help_str_offset + strlen(dummy) + 1; /* 4: '%02d] '!! (window nr.) */

		if (win_nr == terminal_index)
			wprintw(status -> win, ", press <CTRL>+<a>, <d> to exit");
		else if (mail)
			wprintw(status -> win, " You've got mail!");

		getmaxyx(status -> win, dy, dx);
		if (dx >= (statusline_len + 13))
		{
			if (cur -> paused)
			{
				color_on(status, MY_YELLOW);
				mvwprintw(status -> win, 0, dx - 10, "  Paused  ");
				color_off(status, MY_YELLOW);
			}
			else if (cur -> is_command)
			{
				int vmsize = get_vmsize(cur -> pid);

				total_info_len = statusline_len + 12;

				if (vmsize != -1 && dx >= (statusline_len + 30))
				{
					char *vmsize_str = amount_to_str(vmsize);

					mvwprintw(status -> win, 0, dx - strlen(dummy) - 30, "%6s (VMsize) %5d (PID) - %s", vmsize_str, cur -> pid, dummy);

					myfree(vmsize_str);

					total_info_len = statusline_len + 30;
				}
				else
					mvwprintw(status -> win, 0, dx - strlen(dummy) - 12, "%5d (PID) - %s", cur -> pid, dummy);
			}
			else if (fsize == -1)
			{
				mvwprintw(status -> win, 0, dx - strlen(dummy) - 6, "??? - %s", dummy);
				total_info_len = statusline_len + 6;
			}
			else
			{
				int cur_len = 0;
				/* remove the lf at the end as it isn't viewable and should not take up space */
				char *lf = strchr(dummy, '\n');
				if (lf) *lf = 0x00;

				if (afs)
				{
					char *filesize = amount_to_str(fsize);
					cur_len = 3 + strlen(filesize);
					mvwprintw(status -> win, 0, dx - (strlen(dummy) + cur_len), "%s - %s", filesize, dummy);
					free(filesize);
				}
				else
				{
					cur_len = 13;
					/* is this trick still neccessary as I moved from off_t to off64_t? */
#if 0
					/* this trick is because on MacOS X 'off_t' is specified as a 64 bit integer */
#endif
					if (sizeof(off64_t) == 8)
						mvwprintw(status -> win, 0, dx - (strlen(dummy) + cur_len), "%10lld - %s", fsize, dummy);
					else
						mvwprintw(status -> win, 0, dx - (strlen(dummy) + cur_len), "%10ld - %s", fsize, dummy);
				}

				total_info_len = statusline_len + cur_len;
			}
		}

		if (show_f1)
		{
			color_on(status, MY_YELLOW);
			if (dx >= (total_info_len + 32))
				mvwprintw(status -> win, 0, help_str_offset, " *Press F1/<CTRL>+<h> for help* ");
			else if (dx >= (total_info_len + 21))
				mvwprintw(status -> win, 0, help_str_offset, " F1/<CTRL>+<h>: help ");
			else if (dx >= (total_info_len + 13))
				mvwprintw(status -> win, 0, help_str_offset, " F1/^h: help ");
			color_off(status, MY_YELLOW);
		}

		inverse_off(status);

		if (color != -1)
			color_off(status, color);

		update_panels();
	}
}

void create_window_set(int startx, int width, int nwindows, int indexoffset)
{
	int loop;
	int n_window_lines, add_lines;
	int n_not_hidden = 0, win_counter=0;
	int added = 0;
	int term_offset = 0;

	if (SIG_ERR == signal(SIGWINCH, SIG_IGN)) error_exit("signal (SIGWINCH/SIG_IGN) failed");

	for(loop=0; loop<nwindows; loop++)
	{
		if (pi[loop + indexoffset].hidden == 0)
			n_not_hidden++;
	}

	if (n_not_hidden == 0)
		return;

	/* calculate the size of each window */
	n_window_lines = max_y / n_not_hidden;

	/* each window is less then 3 lines? then hide one or more
	 * windows
	 */
	if (n_window_lines < 3)
	{
		n_not_hidden = max_y / 3;
		n_window_lines = 3;
	}
	/* the number of windows multiplied with their sizes may be
	 * less then the height of the terminalwindow, in that case
	 * one or more windows can be 1 line longer
	 */
	add_lines = max_y - (n_window_lines * n_not_hidden);

	/* re-create windows */
	for(loop=0; loop<nwindows; loop++)
	{
		int cur_index = loop + indexoffset;

		if (pi[cur_index].hidden == 0)
		{
			int add = 0;
			int cur_win_length;
			int cur_term_size = max_y - term_offset;
			int loop2;
			regmatch_t *pmatch = NULL;
			char *error = NULL;
			int n_windows_left = n_not_hidden - win_counter;

			if (add_lines > 0)
			{
				add = 1;
				add_lines--;
			}

			cur_win_length = (n_window_lines - (mode_statusline >= 0?1:0)) + add;

			if (pi[cur_index].win_height != -1)
			{
				int dummy = pi[cur_index].win_height - (mode_statusline >= 0?1:0);

				if (dummy < cur_win_length)
				{
					add_lines += (cur_win_length - dummy);
				}
				else if (dummy > cur_win_length)
				{
					add_lines -= (dummy - cur_win_length);
					if (add_lines < 0)
						add_lines = 0;
				}

				cur_win_length = dummy;
			}

			if (cur_term_size < (cur_win_length + 1))
			{
				cur_win_length = (cur_term_size / n_windows_left) - 1;
				add_lines = max_y - ((cur_win_length + 1) * n_windows_left);
			}

			if (cur_win_length < 2)
			{
				if (n_windows_left > 1 && cur_win_length == 1)
					cur_win_length = 2;
				else
					continue;
			}

			if (win_counter == (n_not_hidden - 1))
				cur_win_length = cur_term_size -1;

			/* creat data window, clear en set scroll ok */
			pi[cur_index].data   = mynewwin(cur_win_length, width, term_offset, startx);
			bottom_panel(pi[cur_index].data -> pwin);
			werase(pi[cur_index].data -> win);

			scrollok(pi[cur_index].data -> win, TRUE);/* supposed to always return OK, according to the manpage */

			/* create status window */
			if (mode_statusline >= 0)
			{
				pi[cur_index].status = mynewwin(1, width, term_offset + cur_win_length, startx);
				bottom_panel(pi[cur_index].status -> pwin);
				werase(pi[cur_index].status -> win);

				color_on(pi[cur_index].status, 0);

				/* create status-line */
				update_statusline(pi[cur_index].status, cur_index, &pi[cur_index]);
			}

			/* display old strings */
			for(loop2=max(0, lb[cur_index].curpos - cur_win_length); loop2<lb[cur_index].curpos; loop2++)
			{
				if (!lb[cur_index].pi[loop2] && lb[cur_index].Blines[loop2] != NULL)
				{
					do_print(cur_index, NULL, NULL, NULL, -1);
				}
				else if ((lb[cur_index].Blines)[loop2])
				{
					char display;
					int matching_regex = -1;

					/* check filter */
					(void)check_filter(lb[cur_index].pi[loop2], lb[cur_index].Blines[loop2], &pmatch, &error, &matching_regex, 0, &display);
					if (error)
					{
						do_print(cur_index, (lb[cur_index].pi)[loop2], error, NULL, -1);
						myfree(error);
					}

					if (display)
					{
						do_print(cur_index, (lb[cur_index].pi)[loop2], (lb[cur_index].Blines)[loop2], pmatch, matching_regex);
					}

					myfree(pmatch);
				}
			}
			update_panels();

			if (add)
			{
				added++;
			}

			/* keep track of the number windows displayed on the terminal-
			 * window
			 */
			win_counter++;

			term_offset += cur_win_length + 1;

			/* n_not_hidden not only indicates the number of windows that are
			 * not hidden, it also indicates the number of windows that *can*
			 * be put in the terminal (due to the height of the terminal).
			 * so if the number of displayed windows is equal to the number of
			 * windows that fit in the terminal, break out of the loop
			 */
			if (win_counter == n_not_hidden)
				break;
		}
	}

	/* set signalhandler for terminal resize */
	if (SIG_ERR == signal(SIGWINCH, do_resize)) error_exit("signal (SIGWINCH/SIG_IGN) failed");
}

void create_windows(void)
{
	int loop;

	/* close windows */
	for(loop=0; loop<nfd; loop++)
	{
		if (pi[loop].status)
		{
			delete_popup(pi[loop].status);
			pi[loop].status = NULL;
		}
		if (pi[loop].data)
		{
			delete_popup(pi[loop].data);
			pi[loop].data = NULL;
		}
	}

	if (splitlines)
	{
		for(loop=0; loop<n_splitlines; loop++)
			delete_popup(splitlines[loop]);
		myfree(splitlines);
		splitlines = NULL;
		n_splitlines = 0;
	}

	if (menu_win)
	{
		delete_popup(menu_win);
		menu_win = NULL;
	}

	werase(stdscr);

	/* no windows? display list of keys */
	if (nfd == 0)
	{
		menu_win = mynewwin(max_y, max_x, 0, 0);
		werase(menu_win -> win);

		wprintw(menu_win -> win, version_str, VERSION);
		wprintw(menu_win -> win, "\n\n");

		wprintw(menu_win -> win, "%s\n", F1);
	}
	else
	{
		if (split > 0 && nfd > 1)
		{
			int cols_per_col = max_x / split;
			int win_per_col = nfd / split;
			int x = 0, indexoffset = 0;
			int *vs = (int *)mymalloc(sizeof(int) * split, "window widths");
			int *vn = (int *)mymalloc(sizeof(int) * split, "n windows");

			if (vertical_split)
			{
				int cols_in_use = 0;
				int cols_set = 0;
				int other_cols_size;

				for(loop=0; loop<split; loop++)
				{
					if (vertical_split[loop] >= 4)
					{
						vs[loop] = vertical_split[loop];
						cols_in_use += vertical_split[loop];
						cols_set++;
					}
				}

				/* determine width of other (=not set) columns */
				if (cols_set < split)
				{
					other_cols_size = (max_x - cols_in_use) / (split - cols_set);

					/* doesn't fit? */
					if (other_cols_size < 4)
					{
						/* then use the predetermined size */
						for(loop=0; loop<split; loop++)
							vs[loop] = cols_per_col;
					}
					else /* fill in the not given windows */
					{
						for(loop=0; loop<split; loop++)
						{
							if (vertical_split[loop] < 4)
								vs[loop] = other_cols_size;
						}
					}
				}
			}
			else
			{
				for(loop=0; loop<split; loop++)
					vs[loop] = cols_per_col;
			}

			if (n_win_per_col)
			{
				int win_in_use = 0;
				int win_set = 0;
				int other_win_n;

				for(loop=0; loop<split; loop++)
				{
					if (n_win_per_col[loop] > 0)
					{
						vn[loop] = n_win_per_col[loop];
						win_in_use += n_win_per_col[loop];
						win_set++;
					}
				}

				if (win_set < split)
				{
					other_win_n = (nfd - win_in_use) / (split - win_set);

					if (other_win_n == 0)
					{
						for(loop=0; loop<split; loop++)
							vn[loop] = win_per_col;
					}
					else
					{
						for(loop=0; loop<split; loop++)
						{
							if (n_win_per_col[loop] < 1)
								vn[loop] = other_win_n;
						}
					}
				}
			}
			else
			{
				for(loop=0; loop<split; loop++)
					vn[loop] = win_per_col;
			}

			n_splitlines = split - 1;
			if (n_splitlines > 0)
				splitlines = (NEWWIN **)mymalloc(sizeof(NEWWIN *) * n_splitlines, "splitlines");
			else
				splitlines = NULL;

			for(loop=0; loop<split; loop++)
			{
				if (loop == (split - 1))
				{
					int max_n_win_per_col = max_y / 4;

					create_window_set(x, max_x - x, min(max_n_win_per_col, nfd - indexoffset), indexoffset);
				}
				else
				{
					create_window_set(x, vs[loop] - 1, vn[loop], indexoffset);
					splitlines[loop] = mynewwin(max_y, 1, 0, x + vs[loop] - 1);
					bottom_panel(splitlines[loop] -> pwin);
					draw_line(splitlines[loop], LINE_LEFT);

					x += vs[loop];
					indexoffset += vn[loop];
				}
			}

			myfree(vn);
			myfree(vs);
		}
		else if (nfd != 0)
		{
			create_window_set(0, max_x, nfd, 0);
		}
	}

	mydoupdate();
}

void determine_terminal_size(void)
{
	struct winsize size;

	max_x = max_y = 0;

	/* changed from 'STDIN_FILENO' as that is incorrect: we're
	 * outputting to stdout!
	 */
	if (ioctl(1, TIOCGWINSZ, &size) == 0)
	{
		max_y = size.ws_row;
		max_x = size.ws_col;
	}

	if (!max_x || !max_y)
	{
		char *dummy = getenv("COLUMNS");
		if (dummy)
			max_x = atoi(dummy);
		else
			max_x = 80;

		dummy = getenv("LINES");
		if (dummy)
			max_x = atoi(dummy);
		else
			max_x = 24;
	}
}

int swap_window(void)
{
	int swapped = 0;

	if (nfd > 1)
	{
		int f1_index, f2_index;

		f1_index = select_window(HELP_SWAP_WIN1, "Swap windows, select window 1");
		if (f1_index != -1)
		{
			f2_index = select_window(HELP_SWAP_WIN2, "Swap windows, select window 2");
			if (f2_index != -1)
			{
				proginfo dummy;
				buffer dummy2;

				buffer_replace_pi_pointers(f1_index, &pi[f1_index], &pi[f2_index]);
				buffer_replace_pi_pointers(f2_index, &pi[f2_index], &pi[f1_index]);

				memmove(&dummy, &pi[f1_index], sizeof(proginfo));
				memmove(&pi[f1_index], &pi[f2_index], sizeof(proginfo));
				memmove(&pi[f2_index], &dummy, sizeof(proginfo));

				memmove(&dummy2, &lb[f1_index], sizeof(buffer));
				memmove(&lb[f1_index], &lb[f2_index], sizeof(buffer));
				memmove(&lb[f2_index], &dummy2, sizeof(buffer));

				swapped = 1;
			}
		}
	}
	else
	{
		error_popup("Swap windows", -1, "There's only 1 window!");
	}

	return swapped;
}

int delete_window(void)
{
	int deleted = 0;
	int f_index = 0;

	/* only popup selectionbox when more then one window */
	if (nfd > 1)
		f_index = select_window(HELP_DELETE_SELECT_WINDOW, "Select window to delete");

	/* user did not press q/x? */
	if (f_index != -1)
	{
		char all = 1;

		/* multiple files for this window? then ask if delete all
		 * or just one
		 */
		if (pi[f_index].next)
		{
			NEWWIN *mywin = create_popup(13, 24);

			win_header(mywin, "Delete window");
			escape_print(mywin, 3, 2, "Delete all? (^y^/^n^)");
			mydoupdate();

			all = ask_yes_no(HELP_DELETE_WINDOW_DELETE_ALL_SUBWIN, mywin);

			delete_popup(mywin);
		}

		if (all == 1)	/* pressed 'Y' */
		{
			delete_entry(f_index, NULL);
			deleted = 1;
		}
		else if (all == 0) /* pressed 'N' */
		{
			proginfo *cur = select_subwindow(f_index, HELP_DELETE_SELECT_SUBWINDOW, "Select subwindow");

			if (cur)
			{
				delete_entry(f_index, cur);
				deleted = 1;
			}
		}
		/* otherwhise: pressed 'Q' */
	}

	return deleted;
}

int hide_window(void)
{
	int f_index = -1;

	f_index = select_window(HELP_HIDE_WINDOW, "Select window to hide");
	if (f_index != -1)
	{
		pi[f_index].hidden = 1 - pi[f_index].hidden;
	}

	return f_index != -1 ? 1 : 0;
}

void info(void)
{
	NEWWIN *mywin = create_popup(16, 49);
	int line = 7;
	struct utsname uinfo;

	mvwprintw(mywin -> win, 1, 2, "-=* MultiTail " VERSION " *=-");
	mvwprintw(mywin -> win, 3, 2, "Written by folkert@vanheusden.com");
	mvwprintw(mywin -> win, 4, 2, "Website: http://www.vanheusden.com/multitail/");
	if (!use_colors)
	{
		mvwprintw(mywin -> win, line++, 2, "Your terminal doesn't support colors");
	}

	if (uname(&uinfo) == -1)
		error_exit("uname() failed");

	line++;
	mvwprintw(mywin -> win, line++, 2, "Running on:");
	mvwprintw(mywin -> win, line++, 2, "%s/%s", uinfo.nodename, uinfo.sysname);
	mvwprintw(mywin -> win, line++, 2, "%s %s", uinfo.release, uinfo.version);
#ifdef _GNU_SOURCE
	mvwprintw(mywin -> win, line++, 2, "%s %s", uinfo.machine, uinfo.domainname);
#else
	mvwprintw(mywin -> win, line++, 2, "%s", uinfo.machine);
#endif
	mvwprintw(mywin -> win, line++, 2, "Terminal size: %dx%d", max_x, max_y);

	mvwprintw(mywin -> win, 14, 1, "Press any key to exit this screen");

#if defined(__FreeBSD__) || defined(linux) || defined(__OpenBSD__) || defined(__NetBSD__) || defined(__APPLE__) || defined(sun) || defined(__sun) || defined(__GNU__)
	for(;;)
	{
		char *dummy = get_load();
		mvwprintw(mywin -> win, 6, 2, "Current load of system: %s", dummy);
		myfree(dummy);
		mydoupdate();

		if (wait_for_keypress(-1, 5, mywin, 0) != -1)
			break;
	}
#else
	mydoupdate();
	wait_for_keypress(-1, 0, mywin, 0);
#endif

	delete_popup(mywin);
}

char ask_negate_regexp(NEWWIN *win, int line)
{
	escape_print(win, line, 2, "Negate regular expression? (^y^/^n^)");
	mydoupdate();

	return ask_yes_no(HELP_NEGATE_REGEXP, win);
}

char ask_colors(NEWWIN *win, int line, char cur, char *fieldnr, char **fielddel, int *cscheme_index, int *color_index)
{
	char ok = 0;

	escape_print(win, line, 1, "Colors? (^s^yslog/^m^isc/^f^ield/^n^one/^S^cheme/^i^ndex)");

	mydoupdate();
	for(;;)
	{
		int c = wait_for_keypress(HELP_ASK_COLORS, 0, win, 0);

		switch(c)
		{
			case 7:		/* ctrl+g */
			case 'q':
				return -1;

			case 'S':
				{
					int dummy = selection_box((void *)cschemes, NULL, n_cschemes, SEL_CSCHEME, HELP_ASK_COLORS_SELECT_SCHEME, NULL);
					if (dummy != -1)
					{
						*cscheme_index = dummy;
						return 'S';
					}
				}
				break;

			case 's':
			case 'm':
				return 'm';

			case 'n':
				return 0;

			case 'i':
				{
					int dummy = selection_box((void *)MY_COLORS, NULL, MY_N_COLORS, SEL_COLORS, HELP_ASK_COLORS_SELECT_COLOR, NULL);
					if (dummy != -1)
					{
						*color_index = dummy;
						return 'i';
					}
				}
				break;

			case 'f':
				{
					int loop, fld_dummy;
					char *dummy, nr[5]; /* xxx\0 and a little :-) */

					sprintf(nr, "%d", *fieldnr);

					mvwprintw(win -> win, line + 2, 1, "Enter field number: ");
					dummy = edit_string(win, line + 3, 42, 39, 1, nr, HELP_COLORS_FIELD_NR);
					if (dummy)
					{
						fld_dummy = atoi(dummy);
						myfree(dummy);

						mvwprintw(win -> win, line + 4, 1, "Enter delimiter (eg a space):");
						dummy = edit_string(win, line + 5, 42, 39, 0, *fielddel, HELP_COLORS_FIELD_DELIMITER);

						if (dummy)
						{
							myfree(*fielddel);
							*fielddel = dummy;
							*fieldnr = fld_dummy;
							ok = 1;
						}
					}

					for(loop=0; loop<4; loop++)
						mvwprintw(win -> win, line + 1 + loop, 1, "                              ");
				}

				if (ok)
					return 'f';
				break;

			case 13:
				if (cur != -1)
					return cur;
		}

		wrong_key();
	}
}

char ask_regex_type(NEWWIN *win, int line)
{
	escape_print(win, line+0, 2, "Usage of regexp? (^m^atch, ^v^ do not match");
	escape_print(win, line+1, 2, "^C^olor, ^B^ell, ^b^ell + colorize, e^x^ecute)");
	mydoupdate();

	for(;;)
	{
		int c = wait_for_keypress(HELP_REGEXP_USAGE, 0, win, 0);

		if (c == 'm' || toupper(c) == 'C' || toupper(c) == 'B' || c == 'x' || c == 'v')
			return c;

		if (toupper(c) == 'Q' || c == 7)
			return -1;

		wrong_key();
	}
}

int add_window(void)
{
	int added = 0;
	char *fname;
	NEWWIN *mywin = create_popup(12, 50);
	int col, fc;
	proginfo *cur = NULL;
	char ask_add_to;
	int what_help = -1;

	if (nfd)
		ask_add_to = 1;
	else
		ask_add_to = 0;

	for(;;)
	{
		int cscheme = -1;
		char field_nr = 0;
		char *field_del = NULL;
		char follow_filename = 0;
		char see_difference = 0;
		int color_index = -1;
		char entry_added = 0;

		win_header(mywin, "Add window");

		if (ask_add_to)
		{
			int rc;

			ask_add_to = 0;
			mvwprintw(mywin -> win, 2, 2, "Add (merge) to existing window?");
			mydoupdate();

			rc = ask_yes_no(HELP_ADD_WINDOW_MERGE_SUBWIN, mywin);

			if (rc == 1)	/* pressed 'Y' */
			{
				int index = 0;

				if (nfd > 1)
					index = select_window(HELP_ADD_WINDOW_SELECT_MERGE_WINDOW, NULL);

				if (index == -1)
					break;

				cur = &pi[index];
			}
			else if (rc == -1)	/* pressed 'Q' */
			{
				break;
			}
		}

		/* check if this extra window will still fit */
		if (!cur)
		{
			if ((max_y / (nfd + 1)) < 3)
			{
				error_popup("Add window", -1, "More windows would not fit.");
				break;
			}
		}

		escape_print(mywin, 2, 2, "File or command? (^f^/^c^)         ");
		mydoupdate();
		for(;;)
		{
			fc = toupper(wait_for_keypress(HELP_ADD_FILE_OR_CMD, 0, mywin, 0));

			if (fc == 'F' || fc == 'C' || fc == 'Q' || fc == 7)
				break;

			wrong_key();
		}

		if (fc == 'Q' || fc == 7)
			break;
		else if (fc == 'F')
		{
			mvwprintw(mywin -> win, 2, 2, "Enter filename:       ");
			what_help = HELP_ENTER_FILENAME_TO_MONITOR;
		}
		else
		{
			mvwprintw(mywin -> win, 2, 2, "Enter command:        ");
			what_help = HELP_ENTER_CMD_TO_MONITOR;
		}

		fname = edit_string(mywin, 3, 41, path_max, 0, NULL, what_help);
		if (!fname)
			break;

		if (fc == 'F')
		{
			escape_print(mywin, 4, 2, "Follow filename? (^y^/^n^)");
			mydoupdate();
			follow_filename = ask_yes_no(HELP_ADD_FILE_FOLLOW_FILENAME, mywin);
		}

		col = ask_colors(mywin, 5, -1, &field_nr, &field_del, &cscheme, &color_index);
		if (col == -1)
			break;

		if (cur == NULL)
		{
			int loop;
			proginfo *newpi = (proginfo *)myrealloc(pi, (nfd + 1) * sizeof(proginfo), "proginfo list");
			lb = (buffer *)myrealloc(lb, (nfd + 1) * sizeof(buffer), "buffer list");

			/* 'lb' has pointers to 'pi'-structures. now since we've realloced the pi-
			 * array, we need to update to pointers to pi in the lb-array
			 */
			for(loop=0; loop<nfd; loop++)
			{
				buffer_replace_pi_pointers(loop, &pi[loop], &newpi[loop]);
			}
			/* now that everything is updated, we can safely forget the previous
			 * pointer
			 */
			pi = newpi;

			/* initialize the new structure */
			memset(&lb[nfd], 0x00, sizeof(buffer));
			lb[nfd].maxnlines = min_n_bufferlines;
			lb[nfd].markset = 'm';

			cur = &pi[nfd];
			entry_added = 1;

			added = 1;
		}
		else
		{
			/* skip to end of chain */
			while (cur -> next)
				cur = cur -> next;

			/* allocate new entry */
			cur -> next = (proginfo *)mymalloc(sizeof(proginfo), "proginfo");
			/* and set pointer to the newly allocated entry */
			cur = cur -> next;
		}
		memset(cur, 0x00, sizeof(proginfo));

		cur -> restart = -1;
		if (fc == 'C')
		{
			char *dummy;

			mvwprintw(mywin -> win, 6, 2, "Repeat command interval: (empty for don't)");
			dummy = edit_string(mywin, 7, 10, 10, 1, NULL, HELP_ADD_WINDOW_REPEAT_INTERVAL);
			if (dummy)
			{
				cur -> restart = atoi(dummy);
				myfree(dummy);

				mvwprintw(mywin -> win, 8, 2, "Do you whish to only see the difference?");
				mydoupdate();
				see_difference = ask_yes_no(HELP_ADD_FILE_DISPLAY_DIFF, mywin);
			}
		}
		cur -> filename = fname;
		cur -> status = cur -> data = NULL;
		cur -> is_command = fc == 'F' ? 0 : 1;
		cur -> retry_open = 0;
		cur -> follow_filename = follow_filename;
		cur -> do_diff = see_difference;
		cur -> next = NULL;
		cur -> first = 1;
		cur -> line_wrap = 'a';
		cur -> win_height = -1;

		cur -> colorize = col;
		cur -> field_nr = field_nr;
		cur -> field_del = field_del;
		cur -> color_scheme = cscheme;
		cur -> color_index = color_index;

		/* start tail-process */
		if (start_proc(cur, max_y / (nfd + 1)) == -1)
		{
			mvwprintw(mywin -> win, 8, 2, "error opening file!");
			mydoupdate();
			wait_for_keypress(HELP_FAILED_TO_START_TAIL, 0, mywin, 0);
			nfd--;
			break;
		}

		if (entry_added)
			nfd++;

		mvwprintw(mywin -> win, 10, 2, "Add another to this new window?");
		mydoupdate();
		if (ask_yes_no(HELP_MERGE_ANOTHER_WINDOW, mywin) <= 0)
			break;

		werase(mywin -> win);
		box(mywin -> win, 0, 0);
	}

	delete_popup(mywin);

	return added;
}

void toggle_colors(void)
{
	int f_index = 0;

	if (nfd > 1)
		f_index = select_window(HELP_TOGGLE_COLORS_SELECT_WINDOW, "Toggle colors: select window");

	if (f_index != -1)
	{
		char col;
		proginfo *cur = &pi[f_index];
		NEWWIN *mywin = NULL;
		char *dummy = NULL;

		if (cur -> next)
			cur = select_subwindow(f_index, HELP_TOGGLE_COLORS_SELECT_SUBWINDOW, "Toggle colors: select subwindow");

		if (cur)
		{
			mywin = create_popup(9 + nfd, 42);
			win_header(mywin, "Toggle colors");

			dummy = mystrdup(cur -> filename, "toggle_colors: filename");
			dummy[min(strlen(dummy), 40)] = 0x00;
			mvwprintw(mywin -> win, 3, 1, dummy);

			col = ask_colors(mywin, 4, cur -> colorize, &cur -> field_nr, &cur -> field_del, &cur -> color_scheme, &cur -> color_index);
			if (col != -1)
				cur -> colorize = col;

			delete_popup(mywin);
			myfree(dummy);
		}
	}
}

char zerotomin(char c)
{
	if (c == 0)
		return 'n';
	if (c == 1)
		return 'y';

	return c;
}

int enter_regexp(void)
{
	int changed = 0;
	NEWWIN *mywin;
	proginfo *cur;
	int f_index = 0;
	int cur_re = 0;

	/* select window */
	if (nfd > 1)
		f_index = select_window(HELP_ENTER_REGEXP_SELECT_WINDOW, "Select window (reg.exp. editing)");

	if (f_index == -1)	/* user pressed Q/X */
		return 0;

	/* select subwindow */
	cur = &pi[f_index];
	if (cur -> next)
		cur = select_subwindow(f_index, HELP_ENTER_REGEXP_SELECT_SUBWINDOW, "Select subwindow (reg.exp. editing)");
	if (!cur)
		return 0;

	/* create window */
	mywin = create_popup(24, 70);
	win_header(mywin, "Edit reg.exp.");
	mvwprintw(mywin -> win, 2, 2, "%s", cur -> filename);
	escape_print(mywin, 3, 2, "^a^dd, ^e^dit, ^d^elete, ^q^uit");

	for(;;)
	{
		int c, key;
		int loop;
		char buffer[58 + 1];
		buffer[58] = 0x00;

		/* clear */
		memset(buffer, ' ', 58);
		for(loop=4; loop<23; loop++)
			mvwprintw(mywin -> win, loop, 1, buffer);

		/* display them lines */
		for(loop=0; loop<cur -> n_re; loop++)
		{
			strncpy(buffer, (cur -> pre)[loop].regex_str, 34);
			if (loop == cur_re)
				inverse_on(mywin);
			mvwprintw(mywin -> win, 4 + loop, 1, "%c%c %s", 
					zerotomin((cur -> pre)[loop].invert_regex),
					zerotomin((cur -> pre)[loop].use_regex),
					buffer);
			if ((cur -> pre)[loop].use_regex == 'x')
			{
				char dummy[18];
				strncpy(dummy, (cur -> pre)[loop].cmd, min(17, strlen((cur -> pre)[loop].cmd)));
				dummy[17]=0x00;
				mvwprintw(mywin -> win, 4 + loop, 42, dummy);
				wmove(mywin -> win, 4 + loop, 41);
			}
			mvwprintw(mywin -> win, 4 + loop, 60, "%d", (cur -> pre)[loop].count);
			if (loop == cur_re)
				inverse_off(mywin);
		}
		mydoupdate();

		/* wait for key */
		for(;;)
		{
			key = wait_for_keypress(HELP_REGEXP_MENU, 0, mywin, 1);
			c = toupper(key);

			/* convert return to 'E'dit */
			if (key == 13)
				key = c = 'E';

			/* any valid keys? */
			if (c == 'Q' || c == 7 || c == 'X' || c == 'A' || c == 'E' || c == 'D' || key == KEY_DOWN || key == 13 || key == KEY_UP)
				break;

			wrong_key();
		}
		/* exit this screen? */
		if (c == 'Q' || c == 7 || c == 'X')
			break;

		if (key == KEY_UP)
		{
			if (cur_re > 0)
				cur_re--;
			else
				wrong_key();
		}
		else if (key == KEY_DOWN || key == 13)
		{
			if (cur_re < (cur -> n_re -1))
				cur_re++;
			else
				wrong_key();
		}

		/* add or edit */
		if (c == 'A' || c == 'E')
		{
			int invert_regex = 0, regex_type;
			int rc;
			char *regex_str, *cmd_str = NULL;

			/* max. 10 regular expressions */
			if (c == 'A' && cur -> n_re == 10)
			{
				wrong_key();
				continue;
			}
			if (c == 'E' && cur -> n_re == 0)
			{
				wrong_key();
				continue;
			}

			/* ask new reg exp */
			mvwprintw(mywin -> win, 15, 2, "Edit regular expression:");
			if (c == 'E')
				regex_str = edit_string(mywin, 16, 58, 128, 0, (cur -> pre)[cur_re].regex_str, HELP_ENTER_REGEXP);
			else
				regex_str = edit_string(mywin, 16, 58, 128, 0, NULL, HELP_ENTER_REGEXP);

			/* user did not abort edit? */
			if (regex_str == NULL)
				continue;

			regex_type = ask_regex_type(mywin, 18);
			if (regex_type == -1)
				continue;

			if (regex_type != 'm' && regex_type != 'v')
			{
				invert_regex = ask_negate_regexp(mywin, 17);
				if (invert_regex == -1)	/* pressed 'Q' */
					continue;
			}

			if (regex_type == 'x')
			{
				mvwprintw(mywin -> win, 20, 2, "Edit command:");
				if (c == 'E')
					cmd_str = edit_string(mywin, 21, 58, 128, 0, (cur -> pre)[cur_re].cmd, HELP_ENTER_CMD);
				else
					cmd_str = edit_string(mywin, 21, 58, 128, 0, NULL, HELP_ENTER_CMD);

				/* did the user cancel things? */
				if (!cmd_str)
				{
					/* then free the memory for the new regexp
					 * and continue with the menu
					 */
					myfree(regex_str);
					continue;
				}
			}

			changed = 1;

			/* edit: free previous */
			if (c == 'E')
			{
				regfree(&(cur -> pre)[cur_re].regex);
				myfree((cur -> pre)[cur_re].regex_str);
			}
			/* add: allocate new */
			else
			{
				cur_re = cur -> n_re++;
				cur -> pre = (re *)myrealloc(cur -> pre, cur -> n_re * sizeof(re), "list of regular expressions");
				memset(&(cur -> pre)[cur_re], 0x00, sizeof(re));
			}

			/* wether to negate this expression or not */
			(cur -> pre)[cur_re].invert_regex = invert_regex;

			/* compile */
			if ((rc = regcomp(&(cur -> pre)[cur_re].regex, regex_str, REG_EXTENDED)))
			{
				char *error = convert_regexp_error(rc, &(cur -> pre)[cur_re].regex);

				mvwprintw(mywin -> win, 17, 2, "Failed to compile regular expression!");

				if (error)
				{
					if (strlen(error) > 58)
					{
						mvwprintw(mywin -> win, 18, 2, "%58", error);
						mvwprintw(mywin -> win, 19, 2, "%s", &error[58]);
					}
					else
					{
						mvwprintw(mywin -> win, 18, 2, "%s", error);
					}
				}

				if (c == 'A')
				{
					cur -> n_re--;
					if (cur -> n_re == cur_re)
						cur_re--;
				}

				mvwprintw(mywin -> win, 22, 2, "Press any key...");
				mydoupdate();
				wait_for_keypress(error?HELP_COMPILE_REGEXP_FAILED:-1, 0, mywin, 0);
				myfree(regex_str);

				myfree(error);
			}
			else
			{
				/* compilation went well, remember everything */
				(cur -> pre)[cur_re].regex_str = regex_str;
				(cur -> pre)[cur_re].use_regex = regex_type;
				(cur -> pre)[cur_re].cmd       = cmd_str;
			}
		}

		/* delete entry */
		if (c == 'D')
		{
			changed = 1;

			/* delete entry */
			free_re(&(cur -> pre)[cur_re]);

			/* update administration */
			if (cur -> n_re == 1)
			{
				myfree(cur -> pre);
				cur -> pre = NULL;
			}
			else
			{
				int n_to_move = (cur -> n_re - cur_re) - 1;

				if (n_to_move > 0)
					memmove(&(cur -> pre)[cur_re], &(cur -> pre)[cur_re+1], sizeof(re) * n_to_move);
			}
			cur -> n_re--;

			/* move cursor */
			if (cur_re > 0 && cur_re == cur -> n_re)
			{
				cur_re--;
			}
		}
	}

	delete_popup(mywin);

	return changed;
}

int toggle_vertical_split(void)
{
	int changed = 0;

	if (split)
	{
		if ((max_y / nfd) < 3)
		{
			error_popup("Toggle vertical split", -1, "That is not possible: the new configuration won't fit.");
		}
		else
		{
			split = 0;
			changed = 1;
			myfree(vertical_split);
			vertical_split = NULL;
			myfree(n_win_per_col);
			n_win_per_col = NULL;
		}
	}
	else
	{
		if (nfd == 1)
			error_popup("Toggle vertical split", -1, "There's only 1 window!");
		else
		{
			char *str;
			NEWWIN *mywin = create_popup(7, 35);

			escape_print(mywin, 2, 2, "Enter number of columns");
			str = edit_string(mywin, 4, 20, 3, 1, NULL, HELP_SET_VERTICAL_SPLIT_N_WIN);
			if (str)
			{
				split = atoi(str);
				if (split > 1)
				{
					changed = 1;
				}
				else
				{
					split = 0;
				}

				myfree(str);
			}

			delete_popup(mywin);
		}
	}

	return changed;
}

void do_set_bufferstart(int f_index, char store_what_lines, int maxnlines)
{
	lb[f_index].markset = store_what_lines;

	if (maxnlines < lb[f_index].maxnlines)
	{
		delete_array(lb[f_index].Blines, lb[f_index].curpos);
		/* delete array already frees the Blines-array */
		lb[f_index].Blines = NULL;

		myfree(lb[f_index].pi);
		lb[f_index].pi = NULL;

		lb[f_index].curpos = 0;
	}

	lb[f_index].maxnlines = maxnlines;
}

void do_pause(void)
{
	NEWWIN *mywin = create_popup(3, 8);

	color_on(mywin, 1);
	box(mywin -> win, 0, 0);
	mvwprintw(mywin -> win, 1, 1, "Paused");
	color_off(mywin, 1);

	wmove(mywin -> win, 1, 2);
	mydoupdate();

	(void)getch();

	delete_popup(mywin);
}

void set_mark(void)
{
	char winchoice, whatlines = 'a', maxnlines = 0;
	NEWWIN *mywin = create_popup(13, 40);

	win_header(mywin, "Set buffer size");

	if (nfd > 1)
	{
		escape_print(mywin, 3, 2, "Set on ^a^ll or ^o^ne");
		mvwprintw(mywin -> win, 4, 2, "window(s)?");
		mydoupdate();

		for(;;)
		{
			winchoice = toupper(wait_for_keypress(HELP_SET_MARK, 0, mywin, 0));

			if (winchoice == 'Q' || winchoice == 'X' || winchoice == 7)
			{
				winchoice = 'Q';
				break;
			}

			if (winchoice == 'A' || winchoice == 'O')
				break;

			wrong_key();
		}
	}
	else
	{
		winchoice = 'A';
	}

	/* ask wether to store all lines or just the ones matchine to the
	 * regular expression (if any)
	 */
	if (winchoice != 'Q' && winchoice != 7)
	{
		escape_print(mywin, 5, 2, "Store ^a^ll or ^m^atching regular");
		mvwprintw(mywin -> win, 6, 2, "expressions (if any)?");
		mydoupdate();
		for(;;)
		{
			whatlines = toupper(wait_for_keypress(HELP_SET_MARK_STORE_WHAT, 0, mywin, 0));
			if (whatlines == 'Q' || whatlines == 'X' || whatlines == 7)
			{
				winchoice = 'Q';
				break;
			}

			if (whatlines == 'A' || whatlines == 'M')
				break;

			wrong_key();
		}
	}

	/* get number of lines to store */
	if (winchoice != 'Q' && winchoice != 7)
	{
		char *dummy;

		mvwprintw(mywin -> win, 7, 2, "Store how many lines? (0 for all)");

		dummy = edit_string(mywin, 8, 40, 7, 1, NULL, HELP_ENTER_NUMBER_OF_LINES_TO_STORE);

		if (!dummy)
		{
			winchoice = 'Q';
		}
		else
		{
			maxnlines = atoi(dummy);

			myfree(dummy);
		}
	}

	/* do set mark */
	if (winchoice != 'Q' && winchoice != 7)
	{
		if (winchoice == 'A')
		{
			int loop;

			for(loop=0; loop<nfd; loop++)
			{
				do_set_bufferstart(loop, whatlines == 'A' ? 'a' : 'm', maxnlines);
			}
		}
		else
		{
			int window = select_window(HELP_SET_MARK_SELECT_WINDOW, NULL);

			if (window != -1)
			{
				do_set_bufferstart(window, whatlines == 'A' ? 'a' : 'm', maxnlines);
			}
		}
	}

	delete_popup(mywin);
}

void generate_diff(int winnr, proginfo *cur)
{
	int loop;

	/* calculate & print difference */
	if (cur -> nprev == 0 && cur -> ncur > 0)
	{
		for(loop=0; loop<cur -> ncur; loop++)
		{
			emit_to_buffer_and_term(winnr, cur, cur -> bcur[loop]);
		}
	}
	else if (cur -> nprev > 0 && cur -> ncur > 0)
	{
		int curprev = 0;

		for(loop=0; loop<cur -> ncur; loop++)
		{
			int indexfound = -1, loop2;

			for(loop2=curprev; loop2<cur -> nprev; loop2++)
			{
				if (strcmp((cur -> bprev)[loop2], (cur -> bcur)[loop]) == 0)
				{
					indexfound = loop2;
					break;
				}
			}

			if (indexfound == -1)
			{
				/* output cur[loop] */
				emit_to_buffer_and_term(winnr, cur, cur -> bcur[loop]);
			}
			else
			{
				curprev = indexfound + 1;
			}
		}
	}
	update_panels();

	/* free up previous */
	delete_array(cur -> bprev, cur -> nprev);

	/* remember current */
	cur -> bprev = cur -> bcur;
	cur -> nprev = cur -> ncur;
	cur -> bcur = NULL;
	cur -> ncur = 0;

	/* update statusline */
	update_statusline(pi[winnr].status, winnr, cur);
}

char close_window(int winnr, proginfo *cur)
{
	if (winnr == terminal_index)
		terminal_index = -1;

	/* make sure it is really gone */
	stop_process(cur -> pid);

	/* restart window? */
	if (cur -> restart >= 0)
	{
		int initial_n_lines_tail;

		/* close old fds */
		if (myclose(cur -> fd) == -1) error_exit("closing read filedescriptor failed\n");
		if (cur -> fd != cur -> wfd)
		{
			if (myclose(cur -> wfd) == -1) error_exit("closing write filedescriptor failed\n");
		}

		/* do diff */
		if (cur -> do_diff)
			generate_diff(winnr, cur);

		if (cur -> initial_n_lines_tail == -1)
			initial_n_lines_tail = max_y / (nfd + 1);
		else
			initial_n_lines_tail = cur -> initial_n_lines_tail;

		/* restart tail-process */
		if (start_proc(cur, initial_n_lines_tail) != -1)
			return -1;

		/* ...and if that fails, go on with the closing */
	}

	if (warn_closed)
	{
		NEWWIN *mywin;

		mywin = create_popup(3, 18 + 4);

		color_on(mywin, 1);
		box(mywin -> win, 0, 0);
		mvwprintw(mywin -> win, 1, 3, "Window %d closed", winnr);
		color_off(mywin, 1);

		wmove(mywin -> win, 1, 2);
		mydoupdate();

		/* wait_for_keypress(HELP_WINDOW_CLOSED, 0); */
		(void)getch();

		delete_popup(mywin);
	}

	/* file no longer exists (or so) */
	return delete_entry(winnr, cur);
}

char * key_to_keybinding(char what)
{
	static char buffer[3];

	buffer[1] = buffer[2] = 0x00;

	if (what < 32)
	{
		buffer[0] = '^';
		buffer[1] = 'a' + what - 1;
	}
	else
	{
		buffer[0] = what;
	}

	return buffer;
}

void list_keybindings(void)
{
	NEWWIN *mywin;
	int prevpos = -1, curpos = 0;

	mywin = create_popup(20, 40);

	for(;;)
	{
		int c;

		if (curpos != prevpos)
		{
			int loop;

			werase(mywin -> win);

			win_header(mywin, "Keybindings");
			box(mywin -> win, 0, 0);

			for(loop=0; loop<17; loop++)
			{
				if ((loop + curpos) >= n_keybindings)
					break;

				inverse_on(mywin);
				mvwprintw(mywin -> win, 2 + loop, 2, "%2s", key_to_keybinding(keybindings[loop + curpos].key));
				inverse_off(mywin);
				mvwprintw(mywin -> win, 2 + loop, 5, "%s", keybindings[loop + curpos].command);
			}

			mydoupdate();

			prevpos = curpos;

		}

		c = wait_for_keypress(HELP_LIST_KEYBINDINGS, 0, mywin, 0);

		if (toupper(c) == 'Q' || c == 7)
			break;
		else if (c == KEY_UP && curpos > 0)
			curpos--;
		else if ((c == KEY_DOWN || c == 13) && curpos < (n_keybindings - 1))
			curpos++;
		else
			wrong_key();
	}

	delete_popup(mywin);
}

void usage(void)
{
	printf(version_str, VERSION);
	printf("\n\nmultitail [-cs|-Cs|-c-] [-i] inputfile [-i anotherinputfile] [...]\n");
	printf("-l	parameter is a command to be executed\n");
	printf("-L	see -l but add to the previous window\n");
	printf("-r interval\n");
	printf("	restart the command when it exited after `interval' seconds\n");
	printf("-R interval\n");
	printf("	same as -r, only with this one only the difference is displayed\n");
	printf("-i	the following parameter is a filename (in case it starts with a dash)\n");
	printf("-n	initial number of lines to tail\n");
	printf("-I	see -i but add to the previous window\n");
	printf("-s x	vertical split screen (in 'x' columns)\n");
	printf("-sw x,x,...	at what columns to split the screen, use '0' for automatic size\n");
	printf("-wh x	height of window\n");
	printf("-f	follow the following filename, not the descriptor\n");
	printf("--retry	keep trying to open the following file if it is inaccessible\n");
	printf("-e[m]	print only when matching with this regexp\n");
	printf("-ev	print only when NOT matching with this regexp\n");
	printf("-ec	use regular expression but display the matches inverted on following file\n");
	printf("-eC	use regexp, display everything but matches inverted on following file\n");
	printf("-ex	execute command ('-ex regexp command') when matches\n");
	printf("-E	use regular expression on following files\n");
	printf("-Ec	use regular expression but display the matches inverted on following files\n");
	printf("-EC	use regexp, display everything but matches inverted on following files\n");
	printf("-ke x   strip parts of the input using regular expression 'x'\n");
	printf("-kr x y strip parts of the input starting at offset x and ending (not including!) offset y\n");
	printf("-kc x y strip parts of the input: strip column 'y' with delimiter 'x'\n");
	printf("-v      invert next regular expression (do not use with -ev/em)\n");
	printf("-c	colorize current\n");
	printf("-cS scheme\n");
	printf("	use colorscheme 'scheme' (as defined in multitail.conf)\n");
	printf("-Cs	colorize all following files with syslog-scheme\n");
	printf("-C	colorize all following files\n");
	printf("-Cf/-cf field delimiter\n");
	printf("        colorize next/all file(s) depending on the given field number. fields\n");
	printf("        are delimited with the given field-delimiter\n");
	printf("-ci color  use 'color' (red, green, etc), usefull when merging multiple inputs\n");
	printf("-c-	do NOT colorize the following file\n");
	printf("-C-	do NOT colorize the following files\n");
	printf("-cT term");
	printf("	interpret terminal-codes from file/command (for terminal type 'term')\n");
	printf("-d	do NOT update the status-line\n");
	printf("-D	do not display a status-line at all\n");
	printf("-z	do not \"window closed\" windows\n");
	printf("-w	do not use colors\n");
	printf("-m nlines\n");
	printf("	immediately set buffer size\n");
	printf("-u	set update interval (for slow links)\n");
	printf("-H interval\n");
	printf("	visual heart-beat (usefull when you want to keep your session alive)\n");
	printf("-p x [y]\n");
	printf("	set linewrap (l=left/a=all/r=right/s=syslog,S=syslog w/o procname, o=offset -> 'y')\n");
	printf("-b n	set TAB-width\n");
	printf("-x str	switch on the xtermtitlebar stuff\n");
	printf("-h	this help\n");
	printf("\n");
	printf("You can have multiple regular expressions per file/command. Be warned: if\n");
	printf("you define multiple and one of them is specified with '-E' (=for every\n");
	printf("following file), _all_ of the current regular expressions are for all\n");
	printf("following files!\n");
	printf("\n");

	printf("%s\n", F1);
}

void write_escape_str(FILE *fh, char *string)
{
	int loop, len = strlen(string);

	fprintf(fh, "\"");
	for(loop=0; loop<len; loop++)
	{
		if (string[loop] == '\"')
			fprintf(fh, "\\");
		fprintf(fh, "%c", string[loop]);
	}
	fprintf(fh, "\"");
}

void write_script(void)
{
	char *fname;
	NEWWIN *mywin = create_popup(12, 42);

	win_header(mywin, "Write script");

	mvwprintw(mywin -> win, 2, 2, "Enter filename:");

	fname = edit_string(mywin, 3, 41, path_max, 0, NULL, HELP_WRITE_SCRIPT);
	if (fname)
	{
		FILE *fh = fopen(fname, "w");
		if (fh)
		{
			int loop;

			fprintf(fh, "#!/bin/sh\n\n");

			fprintf(fh, "multitail");

			if (heartbeat)
				fprintf(fh, " -H %d", heartbeat);

			for(loop=0; loop<nfd; loop++)
			{
				proginfo *cur = &pi[loop];
				char first = 1;

				do
				{
					if (cur -> line_wrap != 'a')
					{
						fprintf(fh, " -p %c", cur -> line_wrap);
					}

					if (cur -> win_height != -1)
						fprintf(fh, " -wh %d", cur -> win_height);

#if 0
					if (cur -> regex_str)
					{
						if (cur -> invert_regex)
							fprintf(fh, " -v");
						if (cur -> use_regex == 'm')
							fprintf(fh, " -e ");
						else if (cur -> use_regex)
							fprintf(fh, " -e%c ", cur -> use_regex);
						write_escape_str(fh, cur -> regex_str);
					}
#endif

					if (cur -> colorize)
					{
						if (cur -> colorize == 's')
							fprintf(fh, " -cs");
						else if (cur -> colorize == 'm')
							fprintf(fh, " -c");
						else if (cur -> colorize == 'f')
						{
							fprintf(fh, " -cf %d ", cur -> field_nr);
							write_escape_str(fh, cur -> field_del);
						}
					}

					if (first)
					{
						first = 0;

						if (cur -> is_command)
						{
							if (cur -> restart != -1)
								fprintf(fh, " -%c %d", cur -> do_diff?'R':'r', cur -> restart);

							fprintf(fh, " -l ");
						}
						else if ((cur -> filename)[0] == '-')
							fprintf(fh, " -i ");
						else
							fprintf(fh, " ");
					}
					else
					{
						if (cur -> is_command)
							fprintf(fh, " -L ");
						else
							fprintf(fh, " -I ");
					}

					write_escape_str(fh, cur -> filename);

					cur = cur -> next;
				}
				while(cur);
			}
			if (split)
				fprintf(fh, " -s");

			if (fchmod(fileno(fh), S_IRUSR | S_IWUSR | S_IXUSR | S_IRGRP | S_IXGRP | S_IROTH | S_IXOTH) == -1)
			{
				error_exit("error setting mode-bits on file");
			}

			fprintf(fh, "\n");

			fclose(fh);
		}
		else
		{
			mvwprintw(mywin -> win, 4, 2, "Error creating file!");
			mydoupdate();

			wait_for_keypress(-1, 0, mywin, 0);
		}

		myfree(fname);
	}

	delete_popup(mywin);
}

int find_colorscheme(char *name)
{
	int loop;

	for(loop=0; loop<n_cschemes; loop++)
	{
		if (strcasecmp(cschemes[loop].name, name) == 0)
			return loop;
	}

	return -1;
}

int parse_color(char *str)
{
	if (strcasecmp(str, "red") == 0)
		return MY_RED;
	else if (strcasecmp(str, "green") == 0)
		return MY_GREEN;
	else if (strcasecmp(str, "yellow") == 0)
		return MY_YELLOW;
	else if (strcasecmp(str, "blue") == 0)
		return MY_BLUE;
	else if (strcasecmp(str, "magenta") == 0)
		return MY_MAGENTA;
	else if (strcasecmp(str, "cyan") == 0)
		return MY_CYAN;
	else if (strcasecmp(str, "white") == 0)
		return MY_WHITE;
	else if (strcasecmp(str, "black") == 0)
		return MY_BLACK;
	else if (strcasecmp(str, "bold") == 0)
		return MY_BOLD;
	else if (strcasecmp(str, "blink") == 0)
		return MY_BLINK;
	else if (strcasecmp(str, "reverse") == 0)
		return MY_REVERSE;
	else if (strcasecmp(str, "underline") == 0)
		return MY_UNDERLINE;

	error_exit("%s is not a known color\n", str);

	return -1;
}

/* returns the default color scheme or -1 if none */
int load_config(char *file)
{
	FILE *fh;
	char *cur_scheme = NULL;
	char *defaultcscheme = NULL;
	int defaultcindex = -1;
	int linenr = 0;

	/* given file */
	fh = fopen(file, "r");
	if (fh == NULL)
	{
		if (errno == ENOENT)	/* file does not exist, not an error */
			return -1;

		return -2;
	}

	for(;;)
	{
		char read_buffer[4096];
		char *dummy, *par;
		char *cmd = fgets(read_buffer, sizeof(read_buffer) - 1, fh);
		if (!cmd)
			break;

		linenr++;

		/* strip LF */
		dummy = strchr(cmd, '\n');
		if (dummy)
			*dummy = 0x00;
		else
			error_exit("line %d of file '%s' is too long!", linenr, file);

		/* LOG("%d: %s\n", linenr, cmdin); */

		/* remove spaces at the beginning of the line */
		while (isspace(*cmd)) cmd++;

		/* see if there's anything in this line anway */
		if (strlen(cmd) == 0)
			continue;

		/* skip comments */
		if (cmd[0] == '#' || cmd[0] == ';')
			continue;

		/* lines are in the format of command:parameter */
		dummy = strchr(cmd, ':');
		if (!dummy)
			error_exit("Malformed configline found: %s (command delimiter (:) missing)\n", cmd);
		*dummy = 0x00;
		par = dummy + 1; 

		if (strcasecmp(cmd, "defaultcscheme") == 0)	/* default colorscheme */
		{
			defaultcscheme = mystrdup(par, "load_config defaultcscheme");
		}
		else if (strcasecmp(cmd, "bright") == 0)
		{
			bright_colors = atoi(par);
		}
		else if (strcasecmp(cmd, "colorscheme") == 0)	/* name for colorscheme */
		{
			if (cur_scheme)
				myfree(cur_scheme);

			cur_scheme = mystrdup(par, "load_config colorscheme");
		}
		else if (strcasecmp(cmd, "cs_re") == 0)		/* ColorScheme_RegularExpression */
		{
			char *re;
			int sn, rc;

			dummy = strchr(par, ':');		/* format: cs_re:color:regular expression */
			if (!dummy)
				error_exit("cs_re-entry malformed: color or regular expression missing (%s:%s)\n", cmd, par);
			*dummy = 0x00;
			re = dummy + 1;

			/* find colorscheme */
			sn = find_colorscheme(cur_scheme);
			if (sn == -1)
			{
				sn = n_cschemes++;
				cschemes = (color_scheme *)myrealloc(cschemes, n_cschemes * sizeof(color_scheme), "list of colorschemes");

				cschemes[sn].name = mystrdup(cur_scheme, "load_config cs_re");
				cschemes[sn].n = 0;
				cschemes[sn].color = NULL;
				cschemes[sn].regex = NULL;
			}

			/* add to list */
			cschemes[sn].color = (int *)myrealloc(cschemes[sn].color, (cschemes[sn].n + 1) * sizeof(int), "color for regexp");
			cschemes[sn].regex = (regex_t *)myrealloc(cschemes[sn].regex, (cschemes[sn].n + 1) * sizeof(regex_t), "regexp for colorscheme");
			cschemes[sn].color[cschemes[sn].n] = parse_color(par);

			/* compile regular expression */
			if ((rc = regcomp(&cschemes[sn].regex[cschemes[sn].n], re, REG_EXTENDED)))
				error_exit("While loading configurationfile: failed to compile regular expression '%s': %s\n", re, convert_regexp_error(rc, &cschemes[sn].regex[cschemes[sn].n]));

			cschemes[sn].n++;
		}
		else if (strcasecmp(cmd, "scheme") == 0)	/* default colorscheme for file */
		{
			char *re;
			int cs, rc;

			dummy = strchr(par, ':');		/* format: cs_re:color:regular expression */
			if (!dummy)
				error_exit("scheme-entry malformed: schemename or regular expression missing (%s:%s)\n", cmd, par);
			*dummy = 0x00;
			re = dummy + 1;

			/* find colorscheme */
			cs = find_colorscheme(par);
			if (cs == -1)
				error_exit("colorscheme '%s' is unknown: you should first define a colorscheme before using it\n", par);

			/* add to list */
			spf = (scheme_per_file *)myrealloc(spf, sizeof(scheme_per_file) * (n_scheme_per_file + 1), "scheme_per_file struct");
			/* store internal scheme-nr */
			spf[n_scheme_per_file].scheme_nr = cs;

			/* compile & store regular expression */
			rc = regcomp(&spf[n_scheme_per_file].regex, re, REG_EXTENDED);
			if (rc != 0)
				error_exit("failed to compile regular expression '%s': %s\n", re, convert_regexp_error(rc, &spf[n_scheme_per_file].regex));

			n_scheme_per_file++;
		}
		else if (strcasecmp(cmd, "check_mail") == 0)	/* how often to check for mail */
		{
			check_for_mail = atoi(par);
		}
		else if (strcasecmp(cmd, "tab_stop") == 0)	/* length of a tab */
		{
			tab_width = atoi(par);
		}
		else if (strcasecmp(cmd, "bind") == 0)		/* program to execute when user presses a key */
		{
			char *proc;

			dummy = strchr(par, ':');		/* format: cs_re:color:regular expression */
			if (!dummy) error_exit("bind-entry malformed: key or binding missing (%s:%s)\n", cmd, par);
			*dummy = 0x00;
			proc = dummy + 1;

			if (strlen(par) > 2) error_exit("bind-entry malformed: unknown keyselection (%s:%s)\n", cmd, par);

			keybindings = (keybinding *)myrealloc(keybindings, sizeof(keybinding) * (n_keybindings + 1), "keybindings array");

			if (par[0] == '^')
				keybindings[n_keybindings].key = toupper(par[1]) - 'A' + 1;
			else
				keybindings[n_keybindings].key = par[0];

			keybindings[n_keybindings].command = mystrdup(proc, "keybinding command");
			n_keybindings++;
		}
		else if (strcasecmp(cmd, "titlebar") == 0)	/* do things with xterm title bar */
		{
			set_title = mystrdup(par, "load_config titlebar");
		}
		else if (strcasecmp(cmd, "tail") == 0)	/* where to find tail */
		{
			tail = mystrdup(par, "where to find tail");
		}
		else if (strcasecmp(cmd, "abbreviate_filesize") == 0)
		{
			afs = atoi(par);
		}
		else
			error_exit("Configurationparameter %s:%s is unknown (file: %s, line: %d)\n", cmd, par, file, linenr);
	}
	fclose(fh);

	if (defaultcscheme)
	{
		defaultcindex = find_colorscheme(defaultcscheme);

		if (defaultcindex == -1)
			error_exit("Default colorscheme '%s' is not defined! Check multitail.conf\n", defaultcscheme);
	}

	return defaultcindex;
}

int do_load_configfile(char *config_file)
{
	/* load configurationfile (if any) */
#ifdef CSW
	int default_color_scheme = load_config("/opt/csw/etc/multitail.conf");
#else
	int default_color_scheme = load_config("/etc/multitail.conf");
#endif

	if (config_file)
		default_color_scheme = load_config(config_file);
	else
	{
		char *path = mymalloc(path_max + 1, "path");
		struct passwd *pp = getuserinfo();

		snprintf(path, path_max, "%s/.multitailrc", pp -> pw_dir);

		default_color_scheme = load_config(path);

		myfree(path);
	}

	return default_color_scheme;
}

void get_terminal_type(void)
{
	char *dummy = getenv("TERM");

	if (dummy && strstr(dummy, "xterm") != NULL)
	{
		term_type = TERM_XTERM;
	}
}

void statistics(void)
{
	NEWWIN *mywin = create_popup(nfd + 7, 73);

	for(;;)
	{
		int loop;
		int vmsize = get_vmsize(getpid());
		int c;

		escape_print(mywin, 1, 2, "^Window^");
		escape_print(mywin, 1, 36, "^Average^");
		escape_print(mywin, 1, 44, "^std dev^");
		escape_print(mywin, 1, 54, "^Tendency^");
		escape_print(mywin, 1, 63, "^# lines^");

		for(loop=0; loop<nfd; loop++)
		{
			int y = 2 + loop;
			char *copy = mystrdup(pi[loop].filename, "statistics filename");
			char *dummy = copy;
			double avg, dev;

			if (strlen(copy) > 30)
				dummy += strlen(copy) - 30;

			mvwprintw(mywin -> win, y, 2, "%02d %s", loop, dummy);
			mvwprintw(mywin -> win, y, 63, "%d", pi[loop].n_events);

			if (pi[loop].n_events == 0)
			{
				mvwprintw(mywin -> win, y, 36, "Not yet available");
			}
			else
			{
				avg = pi[loop].med / (double)pi[loop].n_events;
				mvwprintw(mywin -> win, y, 36, "%.2f", avg);

				dev = sqrt((pi[loop].dev / (double)pi[loop].n_events) - pow(avg, 2.0));
				mvwprintw(mywin -> win, y, 44, "%.2f", dev);

				if (pi[loop].n_events > 1)
					mvwprintw(mywin -> win, y, 54, "%.2f", pi[loop].total_deltat / ((double)pi[loop].n_events - 1));
			}

			myfree(copy);
		}

		mvwprintw(mywin -> win, nfd+3, 2, "Run-time: %.2f hours", difftime(time(NULL), mt_started) / 3600.0);
		if (vmsize != -1)
		{
			char *vmsize_str = amount_to_str(vmsize);
			mvwprintw(mywin -> win, nfd+4, 2, "Memory usage: %s", vmsize_str);
			myfree(vmsize_str);
		}
		escape_print(mywin, nfd+5, 2, "Press ^r^ to reset counters, ^q^ to exit");

		mydoupdate();

		c = toupper(wait_for_keypress(HELP_STATISTICS, 5, mywin, 0));

		if (c == 'R')
		{
			for(loop=0; loop<nfd; loop++)
			{
				pi[loop].med = pi[loop].dev = 0;
				pi[loop].n_events = 0;
				pi[loop].prev_deltat = pi[loop].total_deltat = 0;
				pi[loop].lastevent = 0;
			}
		}

		if (c == 'Q' || c == 7)
			break;

		if (c != -1)
			wrong_key();

		werase(mywin -> win);
		box(mywin -> win, 0, 0);
	}

	delete_popup(mywin);
}

void redraw_statuslines(void)
{
	int loop;

	/* update statuslines */
	for(loop=0; loop<nfd; loop++)
		update_statusline(pi[loop].status, loop, &pi[loop]);
}

void store_statistics(int index, time_t now)
{
	if (pi[index].lastevent)
	{
		double cur_deltat = difftime(now, pi[index].lastevent);

		if (pi[index].n_events == 1)
		{
			pi[index].total_deltat += (cur_deltat - pi[index].prev_deltat);
			pi[index].prev_deltat = cur_deltat;
		}

		pi[index].med += cur_deltat;
		pi[index].dev += pow(cur_deltat, 2.0);
		pi[index].n_events++;
	}

	pi[index].lastevent = now;
}

int do_heartbeat(time_t ts)
{
	if (heartbeat)
	{
		if (difftime(ts, hb_time) >= heartbeat)
		{
			static int hb_x = 0, hb_y = 0, hb_dx = 1, hb_dy = 1;
			int delta = myrand(2) + 1;

			move(hb_y, hb_x);

			hb_y += hb_dy;
			hb_x += hb_dx;

			if (hb_y >= max_y - 1)
			{
				hb_y = max_y - 1;
				hb_dy = -delta;
			}
			else if (hb_y < 0)
			{
				hb_y = 0;
				hb_dy = delta;
			}
			if (hb_x >= max_x - 1)
			{
				hb_x = max_x - 1;
				hb_dx = -delta;
			}
			else if (hb_x < 0)
			{
				hb_x = 0;
				hb_dx = delta;
			}

			hb_time = ts;

			return 1;
		}
	}

	return 0;
}

int set_window_widths(void)
{
	int resized = 0;
	NEWWIN *mywin = create_popup(5 + MAX_N_COLUMNS, 35);
	int loop, y = 0;

	for(;;)
	{
		int key;

		werase(mywin -> win);
		escape_print(mywin, 1, 2, "^Manage columns^");
		escape_print(mywin, 2, 2, "^a^dd, ^d^elete, ^e^dit, ^q^uit");
		escape_print(mywin, 3, 2, "width           # windows");

		if (split < 2)
		{
			color_on(mywin, MY_RED);
			escape_print(mywin, 5, 2, "No columns (no vertical split)");
			color_off(mywin, MY_RED);
		}
		else
		{
			for(loop=0; loop<split; loop++)
			{
				int vs = 0;
				int nw = 0;

				if (vertical_split) vs = vertical_split[loop];
				if (n_win_per_col) nw = n_win_per_col[loop];

				if (loop == y) inverse_on(mywin);
				mvwprintw(mywin -> win, 4 + loop, 2, "%2d              %2d", vs, nw);
				if (loop == y) inverse_off(mywin);
			}
		}
		box(mywin -> win, 0, 0);
		mydoupdate();

		key = toupper(wait_for_keypress(HELP_MANAGE_COLS, 0, mywin, 1));
		if (key == 'Q' || key == 7)
			break;

		if (key == KEY_UP)
		{
			if (y)
				y--;
			else
				wrong_key();
		}
		else if (key == KEY_DOWN)
		{
			if (y < (split - 1))
				y++;
			else
				wrong_key();
		}
		else if (key == 'A')
		{
			if (split == MAX_N_COLUMNS)
			{
				error_popup("Add column", -1, "Maximum number of columns (%d) reached", MAX_N_COLUMNS);
			}
			else
			{
				int is_new = 0;

				if (split == 0)
				{
					is_new = 1;
					split = 2;
				}
				else
					split++;

				if (!vertical_split) is_new = 1;

				vertical_split = (int *)myrealloc(vertical_split, sizeof(int) * split, "column widths");
				n_win_per_col = (int *)myrealloc(n_win_per_col, sizeof(int) * split, "# windows per col");

				if (is_new)
				{
					memset(vertical_split, 0x00, sizeof(int) * split);
					memset(n_win_per_col , 0x00, sizeof(int) * split);
				}
				else
				{
					vertical_split[split - 1] = 0;
					n_win_per_col[split - 1] = 0;
				}

				resized = 1;
			}
		}
		else if (key == 'D')
		{
			resized = 1;

			if (split == 2)
			{
				split = 0;
				myfree(vertical_split);
				vertical_split = NULL;
				myfree(n_win_per_col);
				n_win_per_col = NULL;
			}
			else
			{
				int n_to_move = (split - y) - 1;
				if (vertical_split) memmove(&vertical_split[y], &vertical_split[y+1], sizeof(int) * n_to_move);
				if (n_win_per_col ) memmove(&n_win_per_col [y], &n_win_per_col [y+1], sizeof(int) * n_to_move);
				split--;
			}

			if (y >= split && split > 0)
				y--;

			resized = 1;
		}
		else if (key == 'E')
		{
			char oldval[64], *result;
			NEWWIN *editwin = create_popup(6, 33);

			escape_print(editwin, 1, 2, "Column width:");
			escape_print(editwin, 3, 2, "Number of windows per column:");

			if (!vertical_split)
			{
				vertical_split = mymalloc(split * sizeof(int), "vertical split");
				memset(vertical_split, 0x00, split * sizeof(int));
			}
			if (!n_win_per_col)
			{
				n_win_per_col = mymalloc(split * sizeof(int), "n win per col");
				memset(n_win_per_col, 0x00, split * sizeof(int));
			}

			sprintf(oldval, "%d", vertical_split[y]);
			result = edit_string(editwin, 2, 20, 3, 1, oldval, -1 /* FIXME HELP_... */);
			if (result)
			{
				int vs = atoi(result);
				myfree(result);

				sprintf(oldval, "%d", n_win_per_col[y]);
				result = edit_string(editwin, 4, 20, 3, 1, oldval, -1 /* FIXME HELP_... */);
				if (result)
				{
					vertical_split[y] = vs;
					n_win_per_col [y] = atoi(result);

					myfree(result);

					resized = 1;
				}
			}

			delete_popup(editwin);
		}
		else
			wrong_key();
	}

	delete_popup(mywin);

	return resized;
}

int set_window_sizes(void)
{
	int resized = 0;
	NEWWIN *mywin = create_popup(7, 35);

	escape_print(mywin, 1, 2, "^m^ manage columns");
	escape_print(mywin, 2, 2, "^h^ set height of a window");
	escape_print(mywin, 3, 2, "^q^ quit this menu");
	mydoupdate();

	for(;;)
	{
		int key;

		wmove(mywin -> win, 4, 2);

		key = toupper(wait_for_keypress(HELP_SET_WINDOWSIZES, 0, mywin, 0));
		if (key == 'Q' || key == 7)
			break;

		if (key == 'M')
		{
			resized |= set_window_widths();
		}
		else if (key == 'H')
		{
			if (nfd > 1)
			{
				int window = select_window(HELP_SET_WINDOW_HEIGHT_SELECT_WINDOW, NULL);
				if (window != -1)
				{
					char *str;
					char oldval[5];
					char *spaces = "                                 ";

					snprintf(oldval, 5, "%d", pi[window].win_height);

					mvwprintw(mywin -> win, 4, 2, "Enter number of lines:");
					mvwprintw(mywin -> win, 5, 1, spaces);
					mydoupdate();
					str = edit_string(mywin, 5, 20, 3, 1, oldval, HELP_SET_WINDOW_HEIGHT);
					mvwprintw(mywin -> win, 4, 1, spaces);
					mvwprintw(mywin -> win, 5, 1, spaces);
					if (str)
					{
						int dummy = atoi(str);

						if (dummy >= -1 && dummy < (max_y - ((nfd - 1) * 3)) && dummy != 0 && dummy != 1)
						{
							pi[window].win_height = dummy;
							resized = 1;
							break;
						}

						mvwprintw(mywin -> win, 5, 2, "Invalid height!");

						myfree(str);
					}
				}
			}
			else
			{
				mvwprintw(mywin -> win, 5, 2, "You need at least 2 windows.");
			}

			mydoupdate();
		}
		else
		{
			wrong_key();
		}
	}

	delete_popup(mywin);

	return resized;
}

void terminal_mode(void)
{
	int index = 0;

	if (nfd > 1)
	{
		NEWWIN *mywin = create_popup(8 + nfd, 40);
		win_header(mywin, "Enter terminal mode");
		mydoupdate();

		index = select_window(HELP_TERMINAL_MODE_SELECT_WINDOW, NULL);

		delete_popup(mywin);
	}

	if (index != -1)
	{
		terminal_index = index;
		prev_term_char = -1;

		redraw_statuslines();
	}
}

int exec_bind(char key)
{
	int executed = 0;
	int loop;

	for(loop=0; loop<n_keybindings; loop++)
	{
		if (keybindings[loop].key == key)
		{
			gui_window_header(keybindings[loop].command);

			if (execute_program(keybindings[loop].command, 0) == 0)
				executed = 1;

			break;
		}
	}

	return executed;
}

void draw_gui_window_header(proginfo *last_changed_window)
{
	char *in;
	int pos = 0;
	int str_len;
	struct utsname uinfo;

	if (!set_title) return;

	in = mystrdup(set_title, "draw_gui_window set_title");
	str_len = strlen(in);

	for(;pos < str_len;)
	{
		if (in[pos] == '%')
		{
			char *new_str = NULL;
			char *repl_str = NULL;

			switch(in[pos+1])
			{
				case 'f':	/* changed file */
					repl_str = mystrdup(last_changed_window?last_changed_window -> filename:"", "dgwh changed file");
					break;

				case 'h':	/* hostname */
					{
						if (uname(&uinfo) == -1)
							error_exit("uname() failed");

						repl_str = mystrdup(uinfo.nodename, "dgwh hostname");
					}
					break;

				case 'l':	/* current load */
					{
						char *dummy;
						repl_str = get_load();
						dummy = strchr(repl_str, ' ');
						if (dummy) *dummy = 0x00;
					}
					break;

				case 'm':
					repl_str = mystrdup(mail?"New mail!":"", "dwgh new_mail");
					break;

				case 'u':	/* username */
					{
						struct passwd *pp = getuserinfo();
						repl_str = mystrdup(pp -> pw_name, "dwgh username");
					}
					break;

				case 't':	/* atime */
					if (last_changed_window)
					{
						char *dt = ctime(&last_changed_window -> lastevent);
						char *dummy = strchr(dt, '\n');
						if (dummy) *dummy = 0x00;
						repl_str = mystrdup(dt, "dwgh atime");
					}
					else
					{
						repl_str = mystrdup("", "dwgh atime(2)");
					}
					break;

				case '%':	/* % */
					repl_str = mystrdup("%", "dwgh %");
					break;
			}

			if (repl_str)
			{
				new_str = replace_string(in, pos, pos + 1, repl_str);
				myfree(in);
				in = new_str;
				str_len = strlen(in);
				pos += strlen(repl_str);
				myfree(repl_str);
			}
			else
			{
				pos += 2;
			}
		}
		else
		{
			pos++;
		}
	}

	gui_window_header(in);

	myfree(in);
}

void selective_pause()
{
	int selected_window = select_window(HELP_PAUSE_A_WINDOW, "Pause a window");

	if (selected_window != -1)
	{
		pi[selected_window].paused = 1 - pi[selected_window].paused;
		update_statusline(pi[selected_window].status, selected_window, &pi[selected_window]);
	}
}

int get_value_arg(char *par, char *string, int check)
{
	long int result;
	int len = strlen(string), loop;

	for(loop=0; loop<len; loop++)
	{
		if (!isdigit(string[loop]))
			error_exit("%s needs a value as parameter");
	}

	result = strtol(string, NULL, 10);

	if (result > INT_MAX || result == LONG_MIN || result == LONG_MAX)
		error_exit("value %s for parameter %s is too large", string, par);

	if (check == VAL_ZERO_POSITIVE)
	{
		if (result < 0)
			error_exit("value for %s must be >= 0", par);
	}
	else if (check == VAL_POSITIVE_NOT_1)
	{
		if (result < 0 || result == 1)
			error_exit("value for %s must be >= 0 and not 1", par);
	}
	else if (check == VAL_POSITIVE)
	{
		if (result <= 0)
			error_exit("value for %s must be > 0", par);
	}
	else
		error_exit("get_value_arg: internal error (%d)", check);

	return (int)result;
}

void wipe_window(void)
{
	NEWWIN *mywin = create_popup(5, 35);

	escape_print(mywin, 1, 2, "^Clear window^");
	escape_print(mywin, 3, 2, "Press 0...9");
	mydoupdate();

	for(;;)
	{
		int c = wait_for_keypress(HELP_WIPE_WINDOW, 0, mywin, 0);

		if (c == 7 || toupper(c) == 'Q')
			break;

		if (c < '0' || c > '9')
		{
			wrong_key();
			continue;
		}

		c -= '0';

		if (c >= nfd)
		{
			wrong_key();
			continue;
		}

		if (pi[c].hidden)
		{
			wrong_key();
			continue;
		}

		werase(pi[c].data -> win);
		mydoupdate();

		break;
	}

	delete_popup(mywin);
}

void wipe_all_windows(void)
{
	int loop;

	for(loop=0; loop<nfd; loop++)
	{
		if (pi[loop].data)
			werase(pi[loop].data -> win);
	}
}

int find_path_max(void)
{
#ifdef PATH_MAX
	int path_max = PATH_MAX;
#else
	int path_max = pathconf("/", _PC_PATH_MAX);
	if (path_max <= 0)
	{
		if (errno) error_exit("pathconf(_PC_PATH_MAX) failed\n");

		path_max = 4096;
	}
	else
		path_max++; /* since its relative to root */
#endif
	if (path_max > 4096)
		path_max = 4096;

	return path_max;
}

void start_all_processes(char *nsubwindows)
{
	int loop;

	for(loop=0; loop<nfd; loop++)
	{
		proginfo *cur = &pi[loop];
		int cur_win_size = lb[loop].maxnlines / nsubwindows[loop];

		do
		{
			int initial_n_lines_tail;

			/* find colorscheme if not selected */
			if (cur -> color_scheme == -1 && cur -> is_command == 0 && (cur -> colorize == 'n' || cur -> colorize == 0))
			{
				int loop2;

				for(loop2=0; loop2<n_scheme_per_file; loop2++)
				{
					int rc;

					if ((rc = regexec(&spf[loop2].regex, cur -> filename, 0, NULL, 0)) == 0)
					{
						cur -> color_scheme = spf[loop2].scheme_nr;
						cur -> colorize = 'S';
						break;
					}
					/* we should inform the user of any errors while executing
					 * the regexp! */
					else
					{
						char *error = convert_regexp_error(rc, &spf[loop2].regex);

						if (error)
							error_exit("%s", error);
					}
				}
			}

			if (cur -> initial_n_lines_tail == -1)
				initial_n_lines_tail = cur_win_size == 0 ? min_n_bufferlines : cur_win_size;
			else
				initial_n_lines_tail = cur -> initial_n_lines_tail;

			/* start the tail process for this file/command */
			if (start_proc(cur, initial_n_lines_tail) == -1)
				error_exit("failed to start process! (%s)\n", cur -> filename);

			cur = cur -> next;
		} while(cur);
	}
}

void version(void)
{
	printf(version_str, VERSION);
	printf("\n\nThank you for using MultiTail.\n");
	printf("If you have any suggestion on how I can improve this program,\n");
	printf("do not hesitate to contact me at folkert@vanheusden.com\n");
	printf("Website is available at: http://www.vanheusden.com/multitail/\n\n\n");
}

void init_check_for_mail(void)
{
	if (check_for_mail > 0 && mail_spool_file != NULL)
	{
		if (stat64(mail_spool_file, &msf_info) == -1)
		{
			check_for_mail = 0;

			printf("Could not determine size of file '%s' (which is supposed to be ", mail_spool_file);
			printf("your mailfile): mail-check is disabled.\n");
			printf("You can prevent this message by adding the line 'check_mail:0' ");
			printf("in /etc/multitail.conf or in .multitailrc in your home-directory.\n\n");
			printf("Press enter to continue...");
			fflush(NULL);
			getchar();
		}
		else
		{
			msf_prev_size = msf_info.st_size;

			msf_last_check = time(NULL);
		}
	}
}

void add_line(int index)
{
	do_print(index, NULL, "", NULL, -1);
	do_buffer(index, NULL, "", 1);

	update_panels();
}

void do_terminal(char c)
{
	if (prev_term_char == 1)	/* ^A */
	{
		if (c == 'd')
		{
			terminal_index = -1;
			redraw_statuslines();
			mydoupdate();
		}
		else
		{
			WRITE(pi[terminal_index].wfd, &prev_term_char, 1);
			WRITE(pi[terminal_index].wfd, &c, 1);
		}
	}
	else
	{
		if (c != 1)	/* ^A */
			WRITE(pi[terminal_index].wfd, &c, 1);
	}

	prev_term_char = c;
}

int main(int argc, char *argv[])
{
	int loop;
	char curcolor = 'n', allcolor = 'n';
	char regex_mode = 0;
	char fi = 0;
	char *fd = NULL;
	char follow_filename = 0, retry = 0, invert_regex = 0;
	char *expr = NULL, exprall = 0;
	char *nsubwindows = NULL;
	int maxlines = 0;
	char setmark = 0, allmark = 0;
	int restart = -1;
	char do_diff = 0;
	char line_wrap = 'a';
	int line_wrap_offset = 0;
	re *pre = NULL;
	int n_re = 0;
	char config_loaded = 0;
	char *config_file = NULL;
	int cur_color_scheme = -1;
	int color_index = -1;
	int default_color_scheme = -1;
	int window_height = -1;
	int initial_n_lines_tail = -1;
	char *win_title = NULL;
	int cur_term_emul = TERM_IGNORE;
	strip_t *pstrip = NULL;
	int n_strip = 0;

	time(&mt_started);

	if (SIG_ERR == signal(SIGTERM, do_exit)) error_exit("setting of signal failed");

	mail_spool_file = getenv("MAIL");

	/* determine PATH_MAX */
	path_max = find_path_max();

	/* calc. buffer length (at least a complete terminal screen) */
	initscr();
	min_n_bufferlines = max(MIN_N_BUFFERLINES, LINES);
#ifdef N_CURSES
	if (has_colors())
	{
		use_default_colors();
		use_colors = 1;
	}
#endif
	/* verify size of terminal window */
	if (LINES < 24 || COLS < 78)
		error_exit("Your terminal(-window) is %dx%d. That is too small for MultiTail (at least 78x24 is required).\n", COLS, LINES);

	endwin();

	/* init random thing */
	time(&hb_time);
	srand((unsigned int)hb_time);

	/* parse commandline */
	for(loop=1; loop<argc; loop++)
	{
		if (strcmp(argv[loop], "-V") == 0)
		{
			version();

			return 0;
		}
		else if (strcmp(argv[loop], "-t") == 0)
		{
			win_title = mystrdup(argv[++loop], "-t");
		}
		else if (strcmp(argv[loop], "-x") == 0)
		{
			set_title = mystrdup(argv[++loop], "-x");
		}
		else if (strcmp(argv[loop], "-p") == 0)
		{
			line_wrap = argv[++loop][0];
			if (line_wrap == 'o')
				line_wrap_offset = get_value_arg("-p", argv[++loop], VAL_ZERO_POSITIVE);
			else if (line_wrap != 'a' && line_wrap != 'l' && line_wrap != 'r' && toupper(line_wrap) != 'S')
				error_exit("Invalid mode fore -p");
		}
		else if (strcmp(argv[loop], "--retry") == 0)
		{
			retry = 1;
		}
		else if (strcmp(argv[loop], "-n") == 0)
		{
			initial_n_lines_tail = get_value_arg("-n", argv[++loop], VAL_ZERO_POSITIVE);
		}
		else if (strcmp(argv[loop], "-b") == 0)
		{
			tab_width = get_value_arg("-b", argv[++loop], VAL_ZERO_POSITIVE);
		}
		else if (strcmp(argv[loop], "-u") == 0)
		{
			update_interval = get_value_arg("-u", argv[++loop], VAL_ZERO_POSITIVE);
		}
		else if (strcasecmp(argv[loop], "-r") == 0)
		{
			if (argv[loop][1] == 'R')
				do_diff = 1;

			restart = get_value_arg("-r/R", argv[++loop], VAL_ZERO_POSITIVE);
		}
		else if (strcmp(argv[loop], "-s") == 0)
		{
			split = get_value_arg("-s", argv[++loop], VAL_POSITIVE_NOT_1);
		}
		else if (strcmp(argv[loop], "-sw") == 0)
		{
			char *dummy = argv[++loop];

			if (split != 0)
				error_exit("-s and -sw are mutual exclusive");

			split = 0;
			for(;;)
			{
				int cur_width;
				char *pnt;

				pnt = strtok(dummy, ",");
				if (!pnt) break;

				split++;
				vertical_split = (int *)myrealloc(vertical_split, split * sizeof(int), "vertical split widths");

				cur_width = get_value_arg("-sw", pnt, VAL_POSITIVE);
				dummy = NULL;

				if (cur_width < 4)
				{
					if (cur_width == 0)
						cur_width = -1;
					else
						error_exit("width of a column must be 4 or greater (or '0' for automatic size)");
				}

				vertical_split[split - 1] = cur_width;
			}

			if (split == 1)
				error_exit("you have to give the width for each window! or set it to 0 (=auto width)");
		}
		else if (strcmp(argv[loop], "-sn") == 0)
		{
			char *dummy = argv[++loop];
			int index = 0;

			if (split == 0)
				error_exit("first use -s or -sw to define the number of columns");

			for(;;)
			{
				int cur_n;
				char *pnt;

				pnt = strtok(dummy, ",");
				if (!pnt) break;

				index++;
				n_win_per_col = (int *)myrealloc(n_win_per_col, index * sizeof(int), "vertical split n_win_per_col");

				cur_n = get_value_arg("-sn", pnt, VAL_POSITIVE);
				dummy = NULL;

				if (cur_n < 0)
					error_exit("number of windows must be either 0 (=auto) or >= 1");

				n_win_per_col[index - 1] = cur_n;
			}
		}
		else if (strcmp(argv[loop], "-wh") == 0)
		{
			window_height = get_value_arg("-wh", argv[++loop], VAL_POSITIVE);
		}
		else if (argv[loop][0] == '-' && toupper(argv[loop][1]) == 'E')
		{
			int rc;
			char *cmd = NULL;

			/* -e => only for this file, -E => for all following files */
			if (argv[loop][1] == 'E')
				exprall = 1;

			/* c/C/m define colors, x says: execute */
			if (toupper(argv[loop][2]) == 'C')
				regex_mode = argv[loop][2];
			else if (toupper(argv[loop][2]) == 'B')
				regex_mode = argv[loop][2];
			else if (argv[loop][2] == 'm' || argv[loop][2] == 0x00)
				regex_mode = 'm';	/* m = match, only print when matches */
			else if (argv[loop][2] == 'v')
				regex_mode = 'v';	/* v = !match, only print when not matching */
			else if (argv[loop][2] == 'x')
				regex_mode = 'x';	/* x = execute */
			else
				error_exit("%s is an unknown switch", argv[loop]);

			/* get expression */
			expr = argv[++loop];

			/* and if there's anything to execute, get commandline */
			if (regex_mode == 'x')
				cmd = argv[++loop];

			/* compile & set expression */
			/* allocate new structure */
			pre = (re *)myrealloc(pre, sizeof(re) * (n_re + 1), "list of regular expressions");

			/* initialize structure */
			memset(&pre[n_re], 0x00, sizeof(re));

			/* compile */
			if ((rc = regcomp(&pre[n_re].regex, expr, REG_EXTENDED)))
				error_exit("failed to compile regular expression \"%s\": %s\n", expr, convert_regexp_error(rc, &pre[n_re].regex));

			/* remember string for later edit */
			pre[n_re].regex_str = mystrdup(expr, "-e regexp");

			/* set flag on current file */
			pre[n_re].use_regex = regex_mode;
			if (exprall == 0)
				regex_mode = 0;

			/* wether to invert the reg exp or not */
			if ((regex_mode == 'v' || regex_mode == 'm') && invert_regex)
			{
				fprintf(stderr, "-e[m] / -ev cannot be used together with -v\n\n");
				usage();
				return(1);
			}
			pre[n_re].invert_regex = invert_regex;
			if (exprall == 0)
				invert_regex = 0;

			/* what to execute (if...) */
			if (cmd)
				pre[n_re].cmd = mystrdup(cmd, "-e cmd");
			else
				pre[n_re].cmd = NULL;

			n_re++;

		}
		else if (strcmp(argv[loop], "-v") == 0)
		{
			invert_regex = 1;
		}
		else if (argv[loop][0] == '-' && toupper(argv[loop][1]) == 'C')
		{
			char *pdummy = argv[loop];
			char dummy = -1, doall = 0;

			if (argv[loop][1] == 'C')
				doall = 1;

			if (argv[loop][2] == 's')	/* syslog-file coloring? */
				dummy = 's';
			else if (argv[loop][2] == 'i')	/* use one specific color */
			{
				dummy = 'i';
				color_index = parse_color(argv[++loop]);
			}
			else if (argv[loop][2] == 'T')	/* terminal mode */
			{
				++loop;

				if (strcasecmp(argv[loop], "ANSI") == 0)
					cur_term_emul = TERM_ANSI;
				else
					error_exit("Terminal emulation '%s' is not known\n", argv[loop]);

				dummy = 'T';
			}
			else if (argv[loop][2] == 'S')	/* use colorscheme */
			{
				char *cur_cscheme = argv[++loop];

				if (config_loaded == 0)
				{
					/* load configurationfile (if any) */
					default_color_scheme = do_load_configfile(config_file);

					config_loaded = 1;
				}

				cur_color_scheme = find_colorscheme(cur_cscheme);
				if (cur_color_scheme == -1)
					error_exit("Color scheme %s not found! Check your configfile (%s)\n", cur_cscheme, config_file);
				dummy = 'S';
			}
			else if (argv[loop][2] == '-')	/* do not color current */
				dummy = 'n';
			else if (argv[loop][2] == 'f')	/* select field for coloring */
			{
				dummy = 'f';
				fi = get_value_arg(pdummy, argv[++loop], VAL_ZERO_POSITIVE);
				fd = argv[++loop];
			}
			else if (argv[loop][2] == 0x00)	/* use complete line for coloring */
				dummy = 'm';
			else
				error_exit("invalid -c mode: '%c'", argv[loop][2]);

			if (doall)
				allcolor = dummy;
			else
				curcolor = dummy;
		}
		else if (strcmp(argv[loop], "-f") == 0)
		{
			follow_filename = 1;
		}
		else if (strcmp(argv[loop], "-w") == 0)
		{
			use_colors = 0;
		}
		else if (argv[loop][0] == '-' && argv[loop][1] == 'k')
		{
			char *pdummy = argv[loop];

			pstrip = myrealloc(pstrip, (n_strip + 1) * sizeof(strip_t), "strip_t");

			if (argv[loop][2] == 'e')
			{
				int rc;

				pstrip[n_strip].type = STRIP_TYPE_REGEXP;
				if ((rc = regcomp(&pstrip[n_strip].regexp, argv[++loop], REG_EXTENDED)))
					error_exit("failed to compile regular expression \"%s\": %s\n", argv[loop], convert_regexp_error(rc, &pstrip[n_strip].regexp));
			}
			else if (argv[loop][2] == 'r')
			{
				pstrip[n_strip].type = STRIP_TYPE_RANGE;
				pstrip[n_strip].start = get_value_arg(pdummy, argv[++loop], VAL_ZERO_POSITIVE);
				pstrip[n_strip].end = get_value_arg(pdummy, argv[++loop], VAL_ZERO_POSITIVE);
				if (pstrip[n_strip].end <= pstrip[n_strip].start)
					error_exit("'-kr start end': end must be bigger then start");
			}
			else if (argv[loop][2] == 'c')
			{
				pstrip[n_strip].type = STRIP_TYPE_COLUMN;
				pstrip[n_strip].del = mystrdup(argv[++loop], "strip column delimiter");
				pstrip[n_strip].col_nr = get_value_arg(pdummy, argv[++loop], VAL_ZERO_POSITIVE);
			}
			else
				error_exit("'%s' is not recognized", argv[loop]);

			n_strip++;
		}
		else if (strcmp(argv[loop], "-m") == 0 || strcmp(argv[loop], "-M") == 0)
		{
			setmark = 1;
			if (argv[loop][1] == 'M')
				allmark = 1;
			maxlines = get_value_arg("-m/M", argv[++loop], VAL_ZERO_POSITIVE);
		}
		else if (strcasecmp(argv[loop], "-i") == 0 || argv[loop][0] != '-' ||
				strcasecmp(argv[loop], "-l") == 0 ||
				strcasecmp(argv[loop], "-iw") == 0)
		{
			struct stat64 buf;
			char *dummy;
			char is_cmd = 0;
			char is_sub = 0;
			char is_giw = 0;
			int check_interval = 0;
			proginfo *cur;

			if (config_loaded == 0)
			{
				/* load configurationfile (if any) */
				default_color_scheme = do_load_configfile(config_file);
				if (default_color_scheme == -2)
					error_exit("could not load configfile %s", config_file);

				config_loaded = 1;
			}

			if (strcasecmp(argv[loop], "-l") == 0)
			{
				is_cmd = 1;
			}

			if (strcasecmp(argv[loop], "-iw") == 0)
			{
				if ((argc - loop) > 2 && isdigit(argv[loop+2][0]))
				{
					is_giw = 1;
					check_interval = atoi(argv[loop + 2]);
				}
				else
				{
					check_interval = 5;
				}
			}

			if (strcmp(argv[loop], "-L") == 0 || strcmp(argv[loop], "-I") == 0 || strcmp(argv[loop], "-Iw") == 0)
			{
				is_sub = 1;
			}

			if (argv[loop][0] == '-')
			{
				loop++;
			}

			dummy = argv[loop];

			if (is_sub == 1 && nfd > 0)
			{
				cur = &pi[nfd - 1];

				while(cur -> next)
				{
					cur = cur -> next;
				}

				cur -> next = (proginfo *)mymalloc(sizeof(proginfo), "proginfo");

				cur = cur -> next;

				nsubwindows[nfd-1]++;
			}
			else
			{
				pi = (proginfo *)myrealloc(pi, (nfd + 1) * sizeof(proginfo), "proginfo");

				lb = (buffer *)myrealloc(lb, (nfd + 1) * sizeof(buffer), "buffers");

				nsubwindows = (char *)myrealloc(nsubwindows, (nfd + 1) * sizeof(char), "subwindows");
				nsubwindows[nfd] = 1;
				memset(&lb[nfd], 0x00, sizeof(buffer));
				if (setmark)
				{
					lb[nfd].maxnlines = maxlines;

					if (!allmark)
						setmark = 0;
				}
				else
				{
					lb[nfd].maxnlines = min_n_bufferlines;
				}
				lb[nfd].markset = 'm';

				cur = &pi[nfd];
				nfd++;
			}
			memset(cur, 0x00, sizeof(proginfo));

			/* see if file exists */
			if (check_interval == 0 && is_cmd == 0 && retry == 0 && stat64(dummy, &buf) == -1)
			{
				error_exit("error opening %s (%d)\n", dummy, errno);
			}

			/* init. struct. for this file */
			if (!dummy) error_exit("No filename given!\n");
			cur -> filename = mystrdup(dummy, "cur -> filename");
			cur -> is_command = is_cmd;
			cur -> check_interval = check_interval;

			cur -> win_title = win_title;
			win_title = NULL;

			/* initial number of lines to tail */
			cur -> initial_n_lines_tail = initial_n_lines_tail;
			initial_n_lines_tail = -1;

			/* default window height */
			cur -> win_height = window_height;
			window_height = -1;

			/* store regular expression(s) */
			cur -> n_re = n_re;
			cur -> pre = pre;
			if (exprall == 0)
			{
				n_re = 0;
				pre = NULL;
			}

			/* hide this window? */
			cur -> hidden = 0;

			/* strip */
			cur -> pstrip = pstrip;
			cur -> n_strip = n_strip;
			pstrip = NULL;
			n_strip = 0;

			/* line wrap */
			cur -> line_wrap = line_wrap;
			line_wrap = 'a';
			cur -> line_wrap_offset = line_wrap_offset;

			cur -> retry_open = retry;
			cur -> follow_filename = follow_filename;

			/* 'watch' functionality configuration (more or less) */
			cur -> restart = restart;
			restart = -1; /* invalidate for next parameter */
			cur -> first = 1;
			cur -> do_diff = do_diff;
			do_diff = 0;

			/*  colors */
			if (cur_color_scheme == -1)
				cur_color_scheme = default_color_scheme;

			cur -> color_index = color_index;
			cur -> color_scheme = cur_color_scheme;
			if (curcolor != 'n' && curcolor != 0)
			{
				cur -> colorize = curcolor;
			}
			else
			{
				if (allcolor == 'n' && curcolor == 'n' && default_color_scheme != -1)
				{
					cur -> colorize = 'S';
				}
				else if (allcolor == 'n' || curcolor == 'n')
				{
					cur -> colorize = 0;
				}
				else
				{
					cur -> colorize = allcolor;
				}
			}
			cur -> field_nr = fi;
			if (fd)
			{
				cur -> field_del = mystrdup(fd, "cur -> field_del");
			}
			else
			{
				cur -> field_del = NULL;
			}

			curcolor = 'n';
			retry = follow_filename = 0;

			if (is_giw)	/* skip over the checkinterval */
				loop++;

			cur -> term_emul = cur_term_emul;
			cur_term_emul = TERM_IGNORE;
		}
		else if (strcmp(argv[loop], "-d") == 0)
		{
			mode_statusline = 0;
		}
		else if (strcmp(argv[loop], "-D") == 0)
		{
			mode_statusline = -1;
		}
		else if (strcmp(argv[loop], "-z") == 0)
		{
			warn_closed = 0;
		}
		else if (strcmp(argv[loop], "-H") == 0)
		{
			heartbeat = get_value_arg("-H", argv[++loop], VAL_ZERO_POSITIVE);
		}
		else if (strcmp(argv[loop], "--config") == 0)
		{
			config_file = argv[++loop];
			if (file_exist(config_file) == -1)
				error_exit("Config-file %s does not exist", config_file);
		}
		else
		{
			if (strcmp(argv[loop], "-h") != 0)
			{
				fprintf(stderr, "unknown parameter '%s'\n", argv[loop]);
			}

			usage();

			return 1;
		}
	}

	if (config_loaded == 0)
	{
		do_load_configfile(config_file);
		if (default_color_scheme == -2)
			error_exit("could not load configfile %s", config_file);

		config_loaded = 1;
	}

	/* init list of pids to watch for exit */
	memset(children_list, 0x00, sizeof(children_list));

	/* initialise mail-stuff */
	init_check_for_mail();

	/* start curses library */
	init_curses();

	/* get terminal type */
	get_terminal_type();

	/* start processes */
	start_all_processes(nsubwindows);
	myfree(nsubwindows);

	signal(SIGHUP, sighup_handler);

	/* create windows */
	do_refresh = 2;

	for(;;)
	{
		int c = wait_for_keypress(HELP_MAIN_MENU, 0, NULL, 0);

		if (terminal_index == -1)
		{
			int uc = toupper(c);

			do_refresh = 1;

			if (mail)
			{
				mail = 0;
				redraw_statuslines();
			}

			if (uc == 'Q' || uc == 'X')
			{
				break;
			}
			else if (uc == 'A')
			{
				if (add_window())
					do_refresh = 2;
				continue;
			}
			else if (uc == 'H' || uc == '?')
			{
				show_help(HELP_MAIN_MENU);
				continue;
			}
			else if (uc == 'R')
			{
				do_refresh = 2;
				continue;
			}
			else if (uc == 'I')
			{
				info();
				continue;
			}
			else if (uc == 'T')
			{
				statistics();
				continue;
			}
			else if (uc == 'J')
			{
				if (set_window_sizes())
					do_refresh = 2;
				continue;
			}
			else if (uc == 'L')
			{
				list_keybindings();
				continue;
			}

			if (nfd == 0)
			{
				wrong_key();
				do_refresh = 0;
				continue;
			}

			else if (uc == 'E' || uc == '\\')
			{
				if (enter_regexp())
					do_refresh = 2;
			}
			else if (uc == 'D')
			{
				if (delete_window())
					do_refresh = 2;
				continue;
			}
			else if (uc == 'V')
			{
				if (toggle_vertical_split())
					do_refresh = 2;
			}
			else if (uc == 'C' && use_colors)
			{
				toggle_colors();
			}
			else if (uc == 'S')
			{
				if (swap_window())
					do_refresh = 2;
				continue;
			}
			else if (uc == 'Z')
			{
				if (hide_window())
					do_refresh = 2;
			}
			else if (uc == 'W')
			{
				write_script();
			}
			else if (uc == 'M')
			{
				set_mark();
			}
			else if (uc == 'N')
			{
				delete_mark();
			}
			else if (uc == 'B')
			{
				scrollback();
			}
			else if (c == 'p')
			{
				do_pause();
			}
			else if (c == 'P')	/* pause an individual screen */
			{
				selective_pause();
			}
			else if (uc >= '0' && uc <= '9')
			{
				int index = uc - '0';

				if (index < nfd)
					add_line(index);
				else
					wrong_key();
			}
			else if (uc == 'K')
			{
				terminal_mode();
			}
			else if (uc == 26)	/* ^Z */
			{
				/* we're back from suspend */
				redraw_statuslines();
			}
			else if (uc == ' ')	/* space */
			{
				/* mail = 0;
				   redraw_statuslines(); */
			}
			else if (c == 'o')	/* wipe window */
			{
				wipe_window();
			}
			else if (c == 'O')	/* wipe all windows */
			{
				wipe_all_windows();
			}
			else
			{
				if (exec_bind(c))
				{
					do_refresh = 1;
				}
				else
				{
					wrong_key();
					do_refresh = 0;
				}
			}
		}
		else
		{
			do_terminal((char)c);
		}
	}

	/* kill tail processes */
	do_exit(0);

	return 0;
}

void do_check_for_mail(time_t ts)
{
	if (check_for_mail > 0 && mail_spool_file != NULL && difftime(ts, msf_last_check) >= check_for_mail)
	{
		/* get current filesize */
		if (stat64(mail_spool_file, &msf_info) == -1)
		{
			if (errno != ENOENT)
				error_exit("Error doing stat64() on file %s.", mail_spool_file);
		}

		/* filesize changed? */
		if (msf_info.st_size != msf_prev_size)
		{
			/* file became bigger: new mail
			 * if it became less, the file changed because
			 * mail was deleted or so
			 */
			if (msf_info.st_size > msf_prev_size)
			{
				mail = 1;

				redraw_statuslines();

				if (do_refresh != 2) do_refresh = 1;
			}

			msf_prev_size = msf_info.st_size;

			msf_last_check = ts;
		}
	}
}

int wait_for_keypress(int what_help, int max_wait, NEWWIN *popup, char cursor_shift)
{
	int waited = 0;
	int c = -1;

	for(;;)
	{
		int last_fd = 0, rc, loop;
		fd_set rfds;
		struct timeval tv;
		time_t now;
		proginfo *last_changed_window = NULL;
		char prev_mail_status = mail;
		static time_t lastupdate = 0;

		time(&now);

		tv.tv_sec = 0;
		tv.tv_usec = 100000;

		FD_ZERO(&rfds);

		/* add stdin to fd-set: needed for monitoring key-presses */
		FD_SET(0, &rfds);

		/* add fd's of pipes to fd-set */
		for(loop=0; loop<nfd; loop++)
		{
			proginfo *cur = &pi[loop];

			if (cur -> paused)
				continue;

			do
			{
				if (cur -> fd != -1)
				{
					FD_SET(cur -> fd, &rfds);
					last_fd = max(last_fd, cur -> fd);
				}

				cur = cur -> next;
			}
			while(cur);
		}

		/* heartbeat? */
		if (do_heartbeat(now))
		{
			if (do_refresh != 2) do_refresh = 1;
		}

		/* check for mail */
		do_check_for_mail(now);

		/* update screen? */
		if (do_refresh)
		{
			if (do_refresh == 2)
				create_windows();

			if (difftime(now, lastupdate) >= update_interval)
			{
				do_refresh = 0;
				lastupdate = now;
				mydoupdate();
			}
		}

		/* wait for any data or key press */
		if ((rc = select(last_fd + 1, &rfds, NULL, NULL, &tv)) == -1)
		{
			if (errno != EINTR && errno != EAGAIN)
				error_exit("select returned an error! (%d)\n", errno);

			continue;
		}

		if (terminal_changed)
		{
			terminal_changed = 0;

			determine_terminal_size();

#ifdef N_CURSES
			if (ERR == resizeterm(max_y, max_x)) error_exit("problem resizing terminal\n");
#endif

			endwin();
			refresh(); /* <- as specified by ncurses faq, was: doupdate(); */

			create_windows();
		}

		waited++;
		if ((waited / 10) >= max_wait && max_wait != 0)
			break;

		/* see if any processes have exited (processes started
		 * by matchin regular expressions)
		 */
		for(loop=0; loop<n_children; loop++)
		{
			pid_t rc = waitpid(children_list[loop], NULL, WNOHANG | WUNTRACED);

			/* error while waiting? */
			if (rc == -1 && errno != ECHILD)
				error_exit("waitpid for pid %d failed\n", children_list[loop]);

			if (rc != 0)
			{
				int n_pids_to_move = (n_children - loop) - 1;

				if (n_pids_to_move > 0)
					memmove(&children_list[loop], &children_list[loop + 1], sizeof(pid_t) * n_pids_to_move);

				n_children--;
				/* also decrease pointer! as we move the other pids up! */
				loop--;
			}
		}


		/* any fd's set? */
		if (rc == 0)
		{
			/* verify if any of the processes exited */
			for(loop=0; loop<nfd; loop++)
			{
				int deleted_entry_in_array = 0;
				proginfo *cur = &pi[loop];

				do
				{
					/* see if the process exited */
					pid_t rc = waitpid(cur -> pid, NULL, WNOHANG | WUNTRACED);

					/* error while waiting? */
					if (rc == -1 && errno != ECHILD)
						error_exit("waitpid for pid %d failed\n", cur -> pid);

					/* did it exit? */
					if (rc != 0) /* equal to: rc == cur -> pid */
					{
						deleted_entry_in_array = close_window(loop, cur);
						/* is an entry deleted? (>=0) or restarted? (==-1) */
						if (deleted_entry_in_array >= 0)
						{
							do_refresh = 2;
							break;
						}
						else if (cur -> do_diff && do_refresh == 0)
						{
							if (do_refresh != 2) do_refresh = 1;
						}
					}

					cur = cur -> next;
				}
				while(cur);

				if (deleted_entry_in_array > 0)
					break;
			}

			/* and since no fd's were set, redo the select()-call */
			continue;
		}

		/* any key pressed? */
		if (FD_ISSET(0, &rfds))
		{
			int uc;

			c = getch();
			uc = toupper(c);

			mail = 0;
			redraw_statuslines();

#ifdef N_CURSES
			if (c == KEY_RESIZE || c == KEY_F(5))
#else
				if (c == KEY_F(5))
#endif
				{
					werase(stdscr);
					determine_terminal_size();
					if (popup)
					{
						if ((popup -> x_off + popup -> width) > max_x ||
								(popup -> y_off + popup -> height) > max_y)
						{
							popup -> x_off = max(max_x - (popup -> width + 1), 0);
							popup -> y_off = max(max_y - (popup -> height + 1), 0);
							move_panel(popup -> pwin, popup -> y_off, popup -> x_off);
							mydoupdate();
						}
					}
					do_refresh = 2;
				}
				else if (c == 3)     /* ^C */
				{
					do_exit(-1);
					error_exit("this should not be reached\n");
				}
				else if (c == 26)/* ^Z */
				{
					endwin();

					if (kill(getpid(), SIGTSTP) == -1)
						error_exit("problem suspending\n");

					mydoupdate();
				}
				else if (c == 8 || c == KEY_F(1))       /* HELP (^h / F1) */
				{
					show_help(what_help);
				}
				else if ((cursor_shift == 0 && c == KEY_LEFT) || (cursor_shift == 1 && c == KEY_SLEFT))
				{
					if (popup != NULL && popup -> x_off > 0)
					{
						popup -> x_off--;
						move_panel(popup -> pwin, popup -> y_off, popup -> x_off);
						mydoupdate();
					}
					else
						wrong_key();
				}
				else if (cursor_shift == 0 && c == KEY_UP)
				{
					if (popup != NULL && popup -> y_off > 0)
					{
						popup -> y_off--;
						move_panel(popup -> pwin, popup -> y_off, popup -> x_off);
						mydoupdate();
					}
					else
						wrong_key();
				}
				else if ((cursor_shift == 0 && c == KEY_RIGHT) || (cursor_shift == 1 && c == KEY_SRIGHT))
				{
					if (popup != NULL && (popup -> x_off + popup -> width) < max_x)
					{
						popup -> x_off++;
						move_panel(popup -> pwin, popup -> y_off, popup -> x_off);
						mydoupdate();
					}
					else
						wrong_key();
				}
				else if (cursor_shift == 0 && c == KEY_DOWN)
				{
					if (popup != NULL && (popup -> y_off + popup -> height) < max_y)
					{
						popup -> y_off++;
						move_panel(popup -> pwin, popup -> y_off, popup -> x_off);
						mydoupdate();
					}
					else
						wrong_key();
				}
				else
				{
					break;
				}

			/* if not a valid key, then reset */
			c = -1;
		}

		/* go through all fds */
		for(loop=0; loop<nfd; loop++)
		{
			int deleted_entry_in_array = 0;
			proginfo *cur = &pi[loop];
			NEWWIN *status = pi[loop].status;

			if (cur -> paused)
				continue;

			do
			{
				if (cur -> fd == -1)
					continue;

				if (FD_ISSET(cur -> fd, &rfds))
				{
					char *buffer = NULL;
					int nbytes, len_il = cur -> incomplete_line ? strlen(cur -> incomplete_line) : 0;

					buffer = myrealloc(cur -> incomplete_line, len_il + 65535 + 1, "input-buffer(1)");
					cur -> incomplete_line = NULL;

					nbytes = read(cur -> fd, &buffer[len_il], 65535);

					/* read error or EOF? */
					if ((nbytes == -1 && errno != EINTR && errno != EAGAIN) || nbytes == 0)
					{
						deleted_entry_in_array = close_window(loop, cur);

						if (deleted_entry_in_array >= 0)
						{
							do_refresh = 2;
							goto closed_window;
						}
						else if (cur -> do_diff && do_refresh == 0)
						{
							if (do_refresh != 2) do_refresh = 1;
						}
					}
					else if (nbytes != -1)	/* if nbytes == -1 it must be an interrupt while READ */
					{
						char *pnt = buffer;

						buffer[len_il + nbytes] = 0x00;

						if (strchr(&buffer[len_il], '\n'))
						{
							/* remember this window as it might be displayed in the GUI
							 * terminal window
							 */
							last_changed_window = cur;

							/* display lines */
							for(;*pnt;)
							{
								char *end = strchr(pnt, '\n');
								if (!end)
									break;

								*end = 0x00;

								/* gen stats */
								store_statistics(loop, now);

								/* is this the output of a program which we should diff and such? */
								if (cur -> do_diff)
								{
									store_for_diff(cur, pnt);
								}
								else /* no, just output */
								{
									if (do_refresh == 0)
										do_refresh = 1;	/* after update interval, update screen */

									emit_to_buffer_and_term(loop, cur, pnt);
								}

								pnt = end + 1;
							}
						}

						if (*pnt != 0x00)
						{
							int line_len = strlen(pnt) + 1;
							cur -> incomplete_line = mymalloc(line_len, "incomplete line(2)");
							memcpy(cur -> incomplete_line, pnt, line_len);
						}

						if (do_refresh)
						{
							/* display statusline? */
							update_statusline(status, loop, cur);

							update_panels();
						}
					}

					myfree(buffer);
				}

				cur = cur -> next;
			}
			while(cur);

			if (deleted_entry_in_array > 0)
				break;
		}
closed_window:

		/* any window changed? then we may want to update the terminal window header */
		if ((last_changed_window != NULL || mail != prev_mail_status) && set_title != NULL)
		{
			prev_mail_status = mail;

			draw_gui_window_header(last_changed_window);
		}
	}

	return c;
}
