#define _LARGEFILE64_SOURCE	/* required for GLIBC to enable stat64 and friends */
#include <ctype.h>
#include <sys/types.h>
#include <regex.h>
#include <string.h>
#include <stdlib.h>
#include <pwd.h>

#include "mt.h"
#include "error.h"
#include "my_pty.h"
#include "utils.h"
#include "term.h"
#include "colors.h"
#include "help.h"

extern proginfo *pi;
extern buffer *lb;
extern int path_max;
extern int nfd;
extern int max_x, max_y;
extern int min_n_bufferlines;

int find_string(int window, char *find, int offset)
{
	int loop, index = -1;
        regex_t regex;

	/* compile the searchstring (which can be a regular expression) */
	if (regcomp(&regex, find, REG_EXTENDED))
		return -1;	/* failed -> not found */

	for(loop=offset; loop<lb[window].curpos; loop++)
	{
		if (regexec(&regex, lb[window].Blines[loop], 0, NULL, 0) == 0)
		{
			index = loop;
			break;
		}
	}

       	regfree(&regex);

	return index;
}

void scrollback_savefile(int window)
{
	char *file = NULL;
	NEWWIN *mywin = create_popup(8, 40);

	win_header(mywin, "Save buffer to file");

	mvwprintw(mywin -> win, 4, 2, "Select file");
	file = edit_string(mywin, 5, 40, path_max, 0, NULL, HELP_SCROLLBACK_SAVEFILE_ENTER_FILENAME);
	if (file)
	{
		FILE *fh = fopen(file, "w");
		if (fh)
		{
			int loop;

			for(loop=0; loop<lb[window].curpos; loop++)
			{
				if (lb[window].Blines[loop])
				{
					char display;
					char *error;
					int dummy = -1;

					/* check filter */
					(void)check_filter(lb[window].pi[loop], lb[window].Blines[loop], NULL, &error, &dummy, 0, &display);
					if (error)
					{
						fprintf(fh, "---> Error: %s\n", error);
						myfree(error);
					}
					if (display)
					{
						fprintf(fh, "%s", lb[window].Blines[loop]);
					}
				}
			}

			fclose(fh);
		}
		else
		{
			mvwprintw(mywin -> win, 6, 2, "Cannot write to file!");
			mydoupdate();

			wait_for_keypress(-1, 0, mywin, 0);
		}
	}

	delete_popup(mywin);
}

int get_lines_needed(char *string, int terminal_width)
{
	if (string)
		return (strlen(string) + terminal_width - 1) / terminal_width;
	else
		return 1;
}

void scrollback_displayline(NEWWIN *win, int window, int offset, int terminal_offset)
{
	proginfo *cur_line_meta = lb[window].pi[offset];
	char *cur_line = lb[window].Blines[offset];
	regmatch_t *pmatch = NULL;
	int matching_regex = -1;
	char *error = NULL;

	wmove(win -> win, terminal_offset, 0);

	if (cur_line)
	{
		if (!cur_line_meta) /* markerline? */
		{
			color_print(win, NULL, NULL, NULL, -1);
		}
		else /* just a buffered line */
		{
			char display;
			(void)check_filter(cur_line_meta, cur_line, &pmatch, &error, &matching_regex, 0, &display);
			if (error)
			{
				color_print(win, cur_line_meta, error, NULL, -1);
				myfree(error);
			}
			if (display)
			{
				color_print(win, cur_line_meta, cur_line, pmatch, matching_regex);
			}
		}
	}
	else /* an empty line */
	{
		/* do nothing */
	}

	myfree(pmatch);
}

void scrollback(void)
{
	int window = 0;
	NEWWIN *mywin1 = create_popup(7, 22);

	win_header(mywin1, "Scrollback");

	if (nfd > 1)
	{
		window = select_window(HELP_SCROLLBACK_SELECT_WINDOW, NULL);
	}

	if (window != -1)
	{
		if (lb[window].markset == 0)
		{
			mvwprintw(mywin1 -> win, 3, 2, "Cannot scrollback:");
			mvwprintw(mywin1 -> win, 4, 2, "buffering is disabled!");
			mydoupdate();
			wait_for_keypress(HELP_SCROLLBACK_NO_MARK, 0, mywin1, 0);
		}
	}

	delete_popup(mywin1);

	if (window != -1 && lb[window].markset != 0)
	{
		char *find = NULL;
		NEWWIN *mywin2;
		int nlines = max_y - 6, ncols = max_x - 6;
		int offset = max(0, lb[window].curpos - nlines);
		char redraw = 2;
		int *line_lengths = (int *)mymalloc(lb[window].curpos * sizeof(int), "line lengths");
		int loop;

		for(loop=0; loop<lb[window].curpos; loop++)
			line_lengths[loop] = get_lines_needed(lb[window].Blines[loop], ncols);

		mywin1 = create_popup(max_y - 4, max_x - 4);
		inverse_on(mywin1);
		mvwprintw(mywin1 -> win, max_y - 5, 1, "%02d] %s - %d buffered lines", window, pi[window].filename, lb[window].curpos);
		inverse_off(mywin1);
		wnoutrefresh(mywin1 -> win);

		mywin2 = create_popup(nlines, ncols);
		scrollok(mywin2 -> win, FALSE); /* supposed to always return OK, according to the manpage */

		for(;;)
		{
			int c;

			if (redraw == 2)
			{
				int index = 0;
				int lines_used = 0;

				werase(mywin2 -> win);

				for(;(offset + index) < lb[window].curpos;)
				{
					int prev_lines_used = lines_used;

					lines_used += line_lengths[offset + index];
					if (lines_used > nlines)
						break;

					scrollback_displayline(mywin2, window, offset + index, prev_lines_used);
					index++;
				}

				redraw = 1;
			}

			if (redraw == 1)
			{
				mydoupdate();

				redraw = 0;
			}

			c = wait_for_keypress(HELP_SCROLLBACK_HELP, 0, NULL, 0);

			if (toupper(c) == 'Q' || toupper(c) == 'X' || c == 7)
			{
				break;
			}
			else if (c == KEY_UP && offset > 0)
			{
				int n_to_move;

				offset--;

                                wmove(mywin2 -> win, 0, 0);
				n_to_move = line_lengths[offset];
				winsdelln(mywin2 -> win, n_to_move);

				scrollback_displayline(mywin2, window, offset, 0);

				redraw = 1;
			}
			else if ((c == KEY_DOWN || c == KEY_NPAGE) && offset < (lb[window].curpos - 1))
			{
				int dummy = 0;
				while(offset < (lb[window].curpos - 1))
				{
					offset++;
					dummy += line_lengths[offset];
					if ((dummy > nlines && c == KEY_NPAGE) ||
					    (dummy > (nlines/4) && c == KEY_DOWN))
					{
						offset--;
						break;
					}
				}
				redraw = 2;
			}
			else if (c == KEY_PPAGE && offset > 0)
			{
				int dummy = 0;
				while(offset > 0)
				{
					offset--;
					dummy += line_lengths[offset];
					if (dummy > nlines)
					{
						offset++;
						break;
					}
				}
				redraw = 2;
			}
			else if (c == KEY_HOME && offset > 0)
			{
				offset = 0;
			}
#ifdef N_CURSES
			else if (c == KEY_END && offset < (lb[window].curpos - 1))
			{
				offset = lb[window].curpos - 1;
			}
#endif
			else if (toupper(c) == 'F' || c == '/')
			{
				char *dummy;
				int new_f_index;
				NEWWIN *mywin = create_popup(5, 40);

				win_header(mywin, "Find");

				dummy = edit_string(mywin, 3, 40, 80, 0, find, HELP_SCROLLBACK_EDIT_SEARCH_STRING);
				myfree(find);
				find = dummy;

				redraw = 2; /* force redraw */

				new_f_index = find_string(window, find, 0);
				if (new_f_index == -1)
					wrong_key();
				else
					offset = new_f_index;

				delete_popup(mywin);
			}
			else if (toupper(c) == 'N')
			{
				if (find)
				{
					int new_f_index = find_string(window, find, offset + 1);
					if (new_f_index == -1)
						wrong_key();
					else
						offset = new_f_index;
				}
				else
					wrong_key();
			}
			else if (toupper(c) == 'S')
			{
				scrollback_savefile(window);
				redraw = 2;	/* force redraw */
			}
			else if (toupper(c) == 'H')
			{
				show_help(HELP_SCROLLBACK_HELP);
			}
			else if (toupper(c) == 'C')
			{
				toggle_colors();
				redraw = 2;	/* force redraw */
			}
			else
			{
				wrong_key();
			}
		}

		delete_popup(mywin2);
		delete_popup(mywin1);

		myfree(find);
		myfree(line_lengths);
	}
}

void delete_mark(void)
{
	int window = 0;
	NEWWIN *mywin = create_popup(13, 40);

	win_header(mywin, "Delete mark");

	if (nfd > 1)
	{
		window = select_window(HELP_SCROLLBACK_DELETE_MARK_SELECT_WINDOW, NULL);
	}

	if (window != -1)
	{
		if (lb[window].markset == 0)
		{
			mvwprintw(mywin -> win, 3, 2, "No mark was set!");
			mydoupdate();
			wait_for_keypress(HELP_SCROLLBACK_DELETE_MARK_NO_MARK, 0, mywin, 0);
		}
		else
		{
			lb[window].markset = 0;
			lb[window].maxnlines = min_n_bufferlines;

			delete_array(lb[window].Blines, lb[window].curpos);
			lb[window].Blines = NULL;
			myfree(lb[window].pi);

			lb[window].curpos = 0;
		}
	}

	delete_popup(mywin);
}
