/*
 * $Id: rip_proto.c,v 1.15 1999/04/29 19:02:57 masaki Exp $
 * routines shared by RIP-2 and RIPng
 */

#include <stdarg.h>
#include <mrt.h>
#include <api6.h>
#include <array.h>
#include <config_file.h>
#include <rip.h>


#ifdef notdef
static DATA_PTR
LL_Add_uniq (LINKED_LIST *ll, DATA_PTR p)
{
    DATA_PTR q;

    LL_Iterate (ll, q) {
	if (p == q)
	    break;
    }
    if (q) LL_RemoveFn (ll, q, NULL);
    LL_Add (ll, p);
    return (p);
}
#endif


rip_attr_t *
rip_new_attr (rip_t * rip, int metric)
{
    rip_attr_t *attr;

    attr = New (rip_attr_t);
    attr->type = rip->proto;
    attr->metric = metric;
    attr->pref = RIP_PREF;
    attr->ctime = time (NULL);
    attr->utime = 0;
    attr->dtime = 0;

    return (attr);
}


void
rip_del_attr (rip_attr_t * attr)
{
    assert (attr);
    deref_nexthop (attr->nexthop);
    Delete (attr);
}


static int
rip_comp_attr (rip_attr_t *a, rip_attr_t *b)
{
    if (a->pref != b->pref)
	return (a->pref - b->pref);
    if ((a->type == PROTO_RIP || a->type == PROTO_RIPNG) 
	    && a->type == b->type)
        return (a->metric - b->metric);
    return (a->type - b->type);
}


/* 
 * create/allocate new rip_route_t structure and insert
 * it in the prefix route hash table
 */
rip_route_t *
rip_new_route (rip_t * rip, prefix_t * prefix, rip_attr_t * attr)
{
    rip_route_t *route;

    route = New (rip_route_t);
    route->prefix = Ref_Prefix (prefix);
    route->received = LL_Create (LL_CompareFunction, rip_comp_attr,
                                  LL_AutoSort, True,
				  LL_DestroyFunction, rip_del_attr, 0);
    route->imported = LL_Create (LL_CompareFunction, rip_comp_attr,
                                 LL_AutoSort, True,
				 LL_DestroyFunction, rip_del_attr, 0);
    if (attr->type == PROTO_RIP || attr->type == PROTO_RIPNG) {
	route->current = attr;
    }
    else {
        LL_Add (route->imported, attr);
    }
    route->active = attr;
    route->flags |= RT_RIP_CHANGE;
    HASH_Insert (rip->hash, route);
    return (route);
}


/* 
 * free route memory and remove from hash
 */
static void
rip_delete_route (rip_route_t * route)
{
    Deref_Prefix (route->prefix);
    LL_Destroy (route->imported);
    LL_Destroy (route->received);
    if (route->current)
        rip_del_attr (route->current);
    Delete (route);
}


static void
rip_set_flash_update (rip_t * rip)
{
    time_t t;

    if (rip->flash_update_waiting) {
	trace (TR_TRACE, rip->trace, "flash timer already running\n");
	return;
    }
/* check to see the RIP and RIPNG */
    t = RIP_FLASH_DELAY_MIN +
	rand () % (RIP_FLASH_DELAY_MAX - RIP_FLASH_DELAY_MIN + 1);
    Timer_Set_Time (rip->flash, t);
    Timer_Turn_ON (rip->flash);
    rip->flash_update_waiting++;
}


static void
rip_receive_update (rip_t * rip)
{
#ifdef HAVE_IPV6
    if (rip->proto == PROTO_RIPNG) {
	ripng_receive_update (NULL);
    }
    else
#endif /* HAVE_IPV6 */
    if (rip->proto == PROTO_RIP) {
	rip2_receive_update (NULL);
    }
    else {
	assert (0);
    }
}


static u_int
rip_hash_fn (prefix_t * prefix, u_int size)
{
#ifdef HAVE_IPV6
    if (prefix->family == AF_INET6) {
	u_int val, buff[4];
	IN6_ADDR_COPY ((struct in6_addr *) buff, prefix_toaddr6 (prefix));
	netmasking (prefix->family, (char *) buff, prefix->bitlen);
	/* Pedro's suggestion */
	val = buff[0] ^ buff[1] ^ buff[2] ^ buff[3];
	val ^= (val >> 16);
	val = val % size;
	return (val);
    }
    else
#endif /* HAVE_IPV6 */
    if (prefix->family == AF_INET) {
	u_int val;
	u_char dest[4];
	memcpy (dest, prefix_tochar (prefix), 4);
	netmasking (prefix->family, dest, prefix->bitlen);
	val = dest[0] + dest[1] + dest[2] + dest[3];
	val = val % size;
	return (val);
    }
    else {
	assert (0);
    }
    /* NEVER REACHES */
    return (0);
}


static u_int
rip_lookup_fn (prefix_t * a, prefix_t * b)
{
    return (prefix_compare (a, b));
}


static generic_attr_t *
rip2gen (rip_attr_t *attr)
{
    generic_attr_t *gattr;

    if (attr == NULL)
	return (NULL);
    gattr = New_Generic_Attr (attr->type);
    gattr->gateway = attr->gateway;
    gattr->nexthop = ref_nexthop (attr->nexthop);
    return (gattr);
}


static void
rip_update_call_fn (rip_t * rip, prefix_t * prefix, rip_attr_t * new, 
		    rip_attr_t *old)
{
    generic_attr_t *gnew;
    generic_attr_t *gold;
    rib_update_route_t fn;

    assert (prefix);
    if ((fn = MRT->rib_update_route) == NULL)
	return;

    gnew = rip2gen (new);
    gold = rip2gen (old);

    fn (prefix, gnew, gold, (gnew)? new->pref: 0, 0);

    Deref_Generic_Attr (gnew);
    Deref_Generic_Attr (gold);
}


/*
 * 1) timoute routes (change metric to 16 and set change bit)
 * 2) garbage collect (delete route from hash, free memory and notify
 *    routing table 
 * Don't need to explicitly do outbound processing, as this will be picked up
 * by the regular 30 second timer
 */
static void
rip_timeout_routes (rip_t * rip)
{
    rip_route_t *route;
    time_t now, t;
    time_t nexttime = 0;
    int changed = 0;

    trace (TR_TRACE, rip->trace, "timer (age) fired\n");

    time (&now);
    nexttime = now + RIP_TIMEOUT_INTERVAL;

    HASH_Iterate (rip->hash, route) {
	time_t t;

next_route:
	/* garbage collect and delete route (rip & imported) */
	if (BIT_TEST (route->flags, RT_RIP_DELETE)) {
	    assert (route->current != NULL ||
			LL_GetCount (route->imported) == 1);
	    if (now - route->active->dtime >= RIP_GARBAGE_INTERVAL) {
		rip_route_t *next = HASH_GetNext (rip->hash, route);
		trace (TR_TRACE, rip->trace,
		       "deleted %p nh %a proto %r (garbage collection)\n",
		       route->prefix, route->active->nexthop->prefix,
		       route->active->type);
		/* deleteing a hash item while looping */
    		HASH_Remove (rip->hash, route);
		if (next == NULL)
		    break;
		route = next;
		goto next_route;
	    }
	    else {
	        if ((t = route->active->dtime + RIP_GARBAGE_INTERVAL) 
			< nexttime)
		    nexttime = t;
	    }
	}
	/* timeout route -- set metric to 16 and set change flag */
	else if (route->current) { /* rip routes there */
	    rip_attr_t *attr;
	    LL_Iterate (route->received, attr) {
		rip_attr_t *prev;
		if (attr->utime > 0 /* timeout is on */ &&
		        now - attr->utime >= RIP_TIMEOUT_INTERVAL) {
	            trace (TR_TRACE, rip->trace,
		           "timing out %p nh %a\n",
		           route->prefix, attr->nexthop->prefix);
		    prev = LL_GetPrev (route->received, attr);
		    LL_Remove (route->received, attr);
		    attr = prev;
		}
	        else {			/* live routes */
	            /* see the earliest next time to be checked */
	            if ((t = attr->utime + RIP_TIMEOUT_INTERVAL) < nexttime)
		        nexttime = t;
		}
	    }

	    if (route->current->utime > 0 /* timeout is on */ &&
		    now - route->current->utime >= RIP_TIMEOUT_INTERVAL) {

	        trace (TR_TRACE, rip->trace,
		       "timing out %p nh %a (current)\n",
		       route->prefix, route->current->nexthop->prefix);
		if (LL_GetCount (route->received) > 0) {
		    rip_attr_t *imported = LL_GetHead (route->imported);

		    rip_del_attr (route->current);
		    route->current = LL_GetHead (route->received);
		    LL_RemoveFn (route->received, route->current, NULL);
		    rip_update_call_fn (rip, route->prefix, route->current, 
					NULL);
		    if (imported == NULL || 
			    imported->pref >= route->current->pref) {
			route->active = route->current;
		        route->flags |= RT_RIP_CHANGE;
		        changed++;
		    }
		}
		else if (LL_GetCount (route->imported) > 0) {
		    /* no rip routes but imported one */
		    if (route->current == route->active) {
			route->active = LL_GetHead (route->imported);
		        route->flags |= RT_RIP_CHANGE;
		        changed++;
		    }
		    rip_update_call_fn (rip, route->prefix, NULL, 
					route->current);
		    rip_del_attr (route->current);
		    route->current = NULL;
		}
		else { /* no other rip and no imported */
		    /* keep the current to send negative route */
		    assert (route->active == route->current);
		    route->current->metric = RIP_METRIC_INFINITY;
		    route->current->dtime = now;
		    route->flags |= RT_RIP_DELETE;
		    route->flags |= RT_RIP_CHANGE;
		    changed++;
	            if ((t = route->current->dtime + RIP_GARBAGE_INTERVAL)
			    < nexttime) {
		        nexttime = t;
		    }
		}
	    }
	    else {
		/* live imported routes. nothing to do here */
	    }
	}
    }

    if (changed)
	rip_set_flash_update (rip);

#define RIP_MIN_TIMEOUT_INTERVAL 5
    if ((t = nexttime - time (NULL)) <= 0)
	t = RIP_MIN_TIMEOUT_INTERVAL;	/* don't want so strict? */
    Timer_Set_Time (rip->age, t);
    Timer_Turn_ON (rip->age);
}


/* 
 * announce all change routes to our peers
 */
static void
rip_output_processing (rip_t * rip, int change)
{
    rip_interface_t *rip_interface;
    rip_route_t *route;

    /* on flushing changes, this keeps locking the route table for a long time
       do we need to change our code ? XXX */

    /* announce routes */
    LL_Iterate (rip->ll_rip_interfaces, rip_interface) {
	if (!BITX_TEST (&rip->interface_mask, rip_interface->interface->index))
	    continue;
	if (!BIT_TEST (rip_interface->interface->flags, IFF_UP))
	    continue;
#ifdef HAVE_IPV6
	if (rip->proto == PROTO_RIPNG)
	    ripng_send_routes (rip_interface, NULL, change);
	else
#endif /* HAVE_IPV6 */
	    rip2_send_routes (rip_interface, NULL, change);
    }
    /* clearing change flag */
    HASH_Iterate (rip->hash, route) {
	route->flags &= ~(RT_RIP_CHANGE);
    }
}


int
check_networks (prefix_t * prefix, LINKED_LIST * ll_networks)
{
    prefix_t *network;

    assert (ll_networks);
    LL_Iterate (ll_networks, network) {
	if (network->family == prefix->family &&
	    a_include_b (network, prefix))
	    return (1);
    }
    return (0);
}


/* 
 * run policy on a rip route 
 * returns -1 when policy rejects it
 * otherwise returns new metric
 */
int
rip_policy (rip_t * rip, prefix_t * prefix, rip_attr_t * attr,
	    rip_interface_t * out)
{
    int cost, num;

#ifdef HAVE_IPV6
    if (rip->proto == PROTO_RIPNG) {
	struct in6_addr *addr = (struct in6_addr *) prefix_tochar (prefix);

	if (prefix->bitlen > 128) {
	    trace (TR_PACKET, rip->trace,
		   "  x %p metric %d (invalid prefix length)\n",
		   prefix, attr->metric);
	    return (-1);
	}

	/* see if the destination prefix valid */
	if (IN6_IS_ADDR_MULTICAST (addr) ||
	    (IN6_IS_ADDR_V4COMPAT (addr) && prefix->bitlen > 0) ||
	    IN6_IS_ADDR_LINKLOCAL (addr)) {
	    trace (TR_PACKET, RIPNG->trace,
		   "  x %p metric %d (non global)\n", prefix, attr->metric);
	}
    }
    else
#endif /* HAVE_IPV6 */
    if (rip->proto == PROTO_RIP) {
	if (prefix->bitlen > 32) {
	    trace (TR_PACKET, rip->trace,
		   "  x %p metric %d (invalid prefix length)\n",
		   prefix, attr->metric);
	    return (-1);
	}
    }
    else {
	assert (0);
    }


    if (out == NULL) {
	/* input policy processing */

	if (attr->metric < 1 || attr->metric > RIP_METRIC_INFINITY) {
	    trace (TR_PACKET, rip->trace,
		   "  x %p metric %d (invalid metric)\n",
		   prefix, attr->metric);
	    return (-1);
	}

	/* check distribute-list for in */
	if (attr->gateway && attr->gateway->interface) {
	    int index = attr->gateway->interface->index;
	    rip_interface_t *rip_interface = rip->rip_interfaces[index];

	    assert (rip_interface);
	    if ((num = rip_interface->dlist_in) >= 0) {
	        if (apply_access_list (num, prefix) == 0) {
		    trace (TR_PACKET, rip->trace,
		           "  x %p metric %d (a-list %d)\n",
		           prefix, attr->metric, num);
		    return (-1);
	        }
	    }
	    cost = attr->metric + rip_interface->metric_in;
	}
	else {
	    cost = attr->metric + 1;
	}
    }
    else {
#ifdef notdef
	if (attr->type == PROTO_CONNECTED &&
	    check_networks (prefix, rip->ll_networks)) {
	    /* OK */
	}
	else if (attr->type != rip->proto &&
		 !BIT_TEST (MRT->redist[attr->type], (1 << rip->proto))) {
	    trace (TR_PACKET, rip->trace,
		   "  x %p metric %d (proto %r)\n",
		   prefix, attr->metric, attr->type);
	    return (-1);
	}
#endif
	/* split horizon (without poisoned reverse) */
	/* do not send to the same interface on which we got the route */
	if (attr->type == rip->proto &&
	/* multicast capable and p-to-p interfaces */
	/* I'm not sure how NBMA will be supported */
	    ((BIT_TEST (out->interface->flags, IFF_MULTICAST) ||
	      BIT_TEST (out->interface->flags, IFF_POINTOPOINT)) &&
	     attr->gateway && attr->gateway->interface == out->interface)) {
	    /* No split horizon on NBMA which has not yet supported, though */
#ifdef RIP_POISONED_REVERSE
	    trace (TR_PACKET, rip->trace,
		   "  o %p metric %d (poisoned reverse)\n",
		   prefix, RIP_METRIC_INFINITY);
	    return (RIP_METRIC_INFINITY);
#else
	    trace (TR_PACKET, rip->trace,
		   "  x %p metric %d (split horion)\n", prefix, attr->metric);
	    return (-1);
#endif /* RIP_POISONED_REVERSE */
	}
	if (attr->type == PROTO_CONNECTED && attr->gateway &&
		attr->gateway->interface == out->interface) {
	    trace (TR_PACKET, rip->trace,
		   "  x %p metric %d (direct)\n", prefix, attr->metric);
	    return (-1);
	}
	/* check distribute-list for out */
	if ((num = out->dlist_out) >= 0) {
	    if (apply_access_list (num, prefix) == 0) {
		trace (TR_PACKET, rip->trace,
		       "  x %p metric %d (a-list %d)\n",
		       prefix, attr->metric, num);
		return (-1);
	    }
	}
	cost = attr->metric + out->metric_out;
    }

    if (cost > RIP_METRIC_INFINITY)
	cost = RIP_METRIC_INFINITY;

    return (cost);
}


static int
rip_process_route (rip_t *rip, rip_route_t *route, rip_attr_t *rip_attr)
{
    char nexthopstr[MAXLINE], tagstr[MAXLINE];
    time_t now;

    time (&now);

	if (rip_attr->nexthop != rip_attr->gateway) {
	    sprintf (nexthopstr, " -> %s",
		     prefix_toa (rip_attr->nexthop->prefix));
	}
	else {
	    nexthopstr[0] = '\0';
	}
	if (rip_attr->tag) {
	    sprintf (tagstr, " tag 0x%lx", rip_attr->tag);
	}
	else {
	    tagstr[0] = '\0';
	}

	    /* we already have a route for this prefix. Two cases:
	     *  1) from same gateway -- 
	     *     just update time, delete, or implicit withdraw.
	     *     if tag changed it should be updated and propagated -- masaki
	     *  2) from different gateway -- 
	     *     if metric better use this, otherwise
	     *     just ignore it. We'll hear about it again in 30 seconds...
	     */

	if (BIT_TEST (route->flags, RT_RIP_DELETE)) {

	    /* first process in case the route is being deleted */
	    	assert (LL_GetCount (route->received) == 0 &&
			   (route->current != NULL ||
			    LL_GetCount (route->imported) == 1));

	        if (rip_attr->metric >= RIP_METRIC_INFINITY) {
		    trace (TR_PACKET, rip->trace,
		           "  x %p metric %d (infinity)\n",
		           route->prefix, rip_attr->metric);
		    rip_del_attr (rip_attr);
		    return (0);
	        }

		if (route->current) {
		    assert (LL_GetCount (route->imported) == 0);
		    assert (LL_GetCount (route->received) == 0);
		    rip_del_attr (route->current);
		}
		else {
		    assert (LL_GetCount (route->received) == 0);
		    assert (LL_GetCount (route->imported) == 1);
		    assert (route->active == LL_GetHead (route->imported));
		    LL_Remove (route->imported, route->active);
		}
		route->current = rip_attr;
		rip_update_call_fn (rip, route->prefix, route->current, NULL);
		route->active = route->current;
		route->flags &= ~RT_RIP_DELETE;
		route->flags |= RT_RIP_CHANGE;
		trace (TR_PACKET, rip->trace,
		       "  o %p metric %d%s (delete -> active)\n",
		       route->prefix, rip_attr->metric, tagstr);
		return (1);
	}
	else if (route->current == NULL) {
	    /* imported route only */
	    assert (LL_GetCount (route->imported) > 0);
	    assert (route->active == LL_GetHead (route->imported));

	    if (rip_attr->metric >= RIP_METRIC_INFINITY) {
		trace (TR_PACKET, rip->trace,
		       "  x %p metric %d (infinity)\n",
		       route->prefix, rip_attr->metric);
		rip_del_attr (rip_attr);
		return (0);
	    }

	    route->current = rip_attr;
	    rip_update_call_fn (rip, route->prefix, route->current, NULL);
	    if (route->active->pref >= route->current->pref) {
	        route->active = route->current;
	        route->flags |= RT_RIP_CHANGE;
	        return (1);
	    }
	    return (0);
	}
	else if (route->current->gateway == rip_attr->gateway) {
	    /* from the same gateway */
	    rip_attr_t *imported = LL_GetHead (route->imported);

	    /* new is infinity */
	    if (rip_attr->metric >= RIP_METRIC_INFINITY) {

		trace (TR_TRACE, rip->trace,
			"  o %p%s metric %d%s (shift to delete)\n",
			route->prefix,
			nexthopstr, rip_attr->metric, tagstr);

		rip_del_attr (rip_attr);

		if (LL_GetCount (route->received) <= 0 &&
			LL_GetCount (route->imported) <= 0) {
		    assert (route->current == route->active);
		    route->current->metric = RIP_METRIC_INFINITY;
		    route->current->dtime = now;
	    	    rip_update_call_fn (rip, route->prefix, NULL, 
					route->current);
		    route->flags |= RT_RIP_DELETE;
		    route->flags |= RT_RIP_CHANGE;
		    return (1);
		}
		else if (LL_GetCount (route->received) > 0) {

		    route->current = LL_GetHead (route->received);
		    LL_RemoveFn (route->received, route->current, NULL);
	    	    rip_update_call_fn (rip, route->prefix, route->current, 
					NULL);

		    if (route->active == imported) {
		        if (route->active->pref >= route->current->pref) {
			    route->active = route->current;
			    route->flags |= RT_RIP_CHANGE;
			    return (1);
			}
		    }
		    else {
			route->active = route->current; 
		        if (imported && route->active->pref > imported->pref)
			    route->active = imported;
			route->flags |= RT_RIP_CHANGE;
			return (1);
		    }
		}
		else {
		    /* no rip, but imported */

		    assert (LL_GetCount (route->imported) > 0);
		    assert (LL_GetCount (route->received) == 0);

	    	    rip_update_call_fn (rip, route->prefix, NULL,
					route->current);
		    rip_del_attr (route->current);
		    if (route->current == route->active) {
		        route->current = NULL;
			route->active = imported;
			route->flags |= RT_RIP_CHANGE;
			return (1);
		    }
		    route->current = NULL;
		}
	        return (0);
	    }
	    /* new is not infinity */
	    else if (route->current->metric != rip_attr->metric ||
		    /* ID says nothing about tag but I think it should be */
			route->current->tag != rip_attr->tag ||
			route->current->nexthop != rip_attr->nexthop) {

		    trace (TR_TRACE, rip->trace,
			   "  o %s%s metric %d%s (change)\n",
			   prefix_toax (route->prefix),
			   nexthopstr, rip_attr->metric, tagstr);

		    rip_del_attr (route->current);
		    route->current = rip_attr;
	    	    rip_update_call_fn (rip, route->prefix, route->current,
					NULL);
		    if (route->active == imported) {
		        if (route->active->pref >= route->current->pref) {
			    route->active = route->current;
			    route->flags |= RT_RIP_CHANGE;
			    return (1);
			}
		    }
		    else {
			route->active = route->current;
			if (imported && route->active->pref > imported->pref)
			    route->active = imported;
			route->flags |= RT_RIP_CHANGE;
			return (1);
		    }
		return (0);
	    }
	    else {
		/* update the time */
		trace (TR_PACKET, rip->trace,
			   "  x %p metric %d (same)\n",
			   route->prefix, rip_attr->metric);
		route->current->utime = rip_attr->utime;
		rip_del_attr (rip_attr);
		return (0);
	    }
	}
	else {
	    /* from different gateway */
	    rip_attr_t *imported = LL_GetHead (route->imported);
	    rip_attr_t *attr;

	    LL_Iterate (route->received, attr) {
		if (rip_attr->gateway == attr->gateway)
		    break;
	    }

	    if (attr) {
		/* remove the previous one */
		LL_Remove (route->received, attr);
	    }

	    /* new is infinity */
	    if (rip_attr->metric >= RIP_METRIC_INFINITY) {
		rip_del_attr (rip_attr);
		return (0);
	    }

	    if (route->current->metric > rip_attr->metric /* better metric */ ||
		(route->current->metric != RIP_METRIC_INFINITY &&
		/* suggested heuristic in case of the same metric */
		    (route->current->metric == rip_attr->metric &&
		     (now - route->current->utime) >=
			 RIP_TIMEOUT_INTERVAL / 2))) {

		    trace (TR_TRACE, rip->trace,
			   "  o %p%s metric %d%s (change)\n",
			   route->prefix, nexthopstr, rip_attr->metric, tagstr);
		    LL_Add (route->received, route->current);
		    route->current = rip_attr;
		    rip_update_call_fn (rip, route->prefix, route->current, 
					NULL);
		    if (route->active == imported) {
			if (route->active->pref >= route->current->pref) {
			    route->active = route->current;
			    route->flags |= RT_RIP_CHANGE;
			    return (1);
			}
		    }
		    else {
			route->active = route->current;
			if (imported && route->active->pref > imported->pref)
			    route->active = imported;
			route->flags |= RT_RIP_CHANGE;
			return (1);
		    }
		return (0);
	    }
	    else {
		trace (TR_PACKET, rip->trace,
			"  x %p metric %d (>= existing %d)\n", route->prefix,
			rip_attr->metric, route->current->metric);
	    	LL_Add (route->received, rip_attr);
		return (0);
	    }
        }
    /* MUST NOT REACH HERE */
    assert (0);
    return (0);
}


/* 
 * process linked list of routes (prefix & attribute combinations)
 * check if 1) just updating time on already received route
 *          2) or a change (different attributes from same gateway,
 *             or better metric and attributes from new gateway
 */
void
rip_process_update (rip_t * rip, LINKED_LIST * ll_prefixes,
		    LINKED_LIST * ll_attr)
{
    prefix_t *prefix;
    rip_attr_t *rip_attr;
    rip_route_t *route;
    int metric, change = 0;
    char nexthopstr[MAXLINE], tagstr[MAXLINE];

    rip_attr = LL_GetHead (ll_attr);
    LL_Iterate (ll_prefixes, prefix) {

	/* do checks here -- if fail, log and ignore
	 * 1) destination prefix valid (i.e. not multicast)
	 * 2) prefix length valid 
	 * 3) metric valid (between 1 and 16 inclusive)
	 */

	/* see if valid and do policy to add cost to the metric */
	if ((metric = rip_policy (rip, prefix, rip_attr, NULL)) < 0) {
	    goto next;
	}
	rip_attr->metric = metric;

#ifdef HAVE_IPV6
#ifdef notdef
	IN6_ADDR_COPY (&in6, prefix_toaddr6 (prefix));
	netmasking (prefix->family, addr, prefix->bitlen);
	if (IN6_ADDR_COMP (&in6, prefix_toaddr6 (prefix)) != 0) {
	    trace (TR_PACKET, rip->trace,
		   "prefix %s will be truncated by prefixlen %d\n",
		   inet_ntop (AF_INET6, &in6, tmp6, sizeof tmp6),
		   prefix->bitlen);
	}
#endif
/* XXX */
#endif /* HAVE_IPV6 */
	if (rip_attr->nexthop != rip_attr->gateway) {
	    sprintf (nexthopstr, " -> %s",
		     prefix_toa (rip_attr->nexthop->prefix));
	}
	else {
	    nexthopstr[0] = '\0';
	}
	if (rip_attr->tag) {
	    sprintf (tagstr, " tag 0x%lx", rip_attr->tag);
	}
	else {
	    tagstr[0] = '\0';
	}

	/* new route */
	if ((route = HASH_Lookup (rip->hash, prefix)) == NULL) {

	    if (rip_attr->metric >= RIP_METRIC_INFINITY) {
		trace (TR_PACKET, rip->trace,
		       "  x %p metric %d (infinity)\n",
		       prefix, rip_attr->metric);
		rip_del_attr (rip_attr);
		goto next;
	    }

	    trace (TR_TRACE, rip->trace,
		   "  o %p metric %d%s (new)\n",
		   prefix, rip_attr->metric, tagstr);

	    route = rip_new_route (rip, prefix, rip_attr);
	    route->flags |= RT_RIP_CHANGE;
	    change++;

	    assert (route->active == rip_attr);
	    /* export the route */
	    rip_update_call_fn (rip, route->prefix, route->active, NULL);
	}
	/* exiting route */
	else {
	    change += rip_process_route (rip, route, rip_attr);
	}
next:
	rip_attr = LL_GetNext (ll_attr, rip_attr);
    }

    assert (rip_attr == NULL);

    if (change)
	rip_set_flash_update (rip);
}


/*
 * rip_update_route()
 * will be called from rib, that is, they are called from
 * other threads. Even though RIPng runs with a single thread,
 * locking on the route table and flash update flag have to be
 * protected by locking.
 *
 * -> changes to use the scheduler. So, I think locking is no need but... 
 */

static void
rip_import (rip_t * rip, prefix_t * prefix, generic_attr_t * new,
	    generic_attr_t * old, int pref)
{
    rip_attr_t *rip_attr, *attr;
    rip_route_t *route;
    time_t now;

    /* this routine doesn't handle a rip route */
    assert (new == NULL || new->type != rip->proto);
    assert (old == NULL || old->type != rip->proto);

    time (&now);

    if (old) {

	if ((route = HASH_Lookup (rip->hash, prefix)) == NULL) {
	    trace (TR_WARN, rip->trace,
	           "delete %p nh %a proto %r (not found)\n",
	           prefix, old->nexthop->prefix, old->type);
	    goto quit;
	}

	LL_Iterate (route->imported, attr) {
	    if (attr->type == old->type &&
	         (old->gateway == NULL || attr->gateway == old->gateway))
		break;
	}

	if (attr == NULL) {
	    trace (TR_WARN, rip->trace,
	           "delete %p nh %a proto %r (not found)\n",
	           prefix, old->nexthop->prefix, old->type);
	    goto quit;
	}

	trace (TR_TRACE, rip->trace, "delete %p nh %a proto %r\n",
	       prefix, old->nexthop->prefix, old->type);

	assert (LL_GetCount (route->imported) >= 1);

	if (route->active == attr) {

	    if (route->current == NULL) {
		if (LL_GetCount (route->imported) <= 1) {
		    attr->metric = RIP_METRIC_INFINITY;
		    attr->dtime = now;
		    route->flags |= RT_RIP_DELETE;
		}
		else {
		    LL_Remove (route->imported, attr);
		    route->active = LL_GetHead (route->imported);
		}
	    }
	    else {
		if (LL_GetCount (route->imported) <= 1) {
		    route->active = route->current;
		}
		else {
		    route->active = LL_GetHead (route->imported);
		    if (route->active == NULL || 
			    route->active->pref >= route->current->pref)
			route->active = route->current;
		}
		LL_Remove (route->imported, attr);
	    }
	    route->flags |= RT_RIP_CHANGE;
	    rip_set_flash_update (rip);
	}
	else {
	    /* just remove it */
	    LL_Remove (route->imported, attr);
	}
    }

    if (new) {

#define RIP_IMPORTED_METRIC 1
	rip_attr = rip_new_attr (rip, RIP_IMPORTED_METRIC);
	rip_attr->type = new->type;
	rip_attr->nexthop = ref_nexthop (new->nexthop);
	rip_attr->gateway = new->gateway;
	rip_attr->pref = pref;
	rip_attr->tag = new->tag;

	if ((route = HASH_Lookup (rip->hash, prefix)) == NULL) {
	    /* new route */
	    route = rip_new_route (rip, prefix, rip_attr);
    	    trace (TR_TRACE, rip->trace,
	   	    "add %p nh %a proto %r (new prefix)\n",
	   	    prefix, new->nexthop->prefix, new->type);
	}
	else if (route->flags & RT_RIP_DELETE) {
	    assert (route->active->metric >= RIP_METRIC_INFINITY);

	    if (route->active == LL_GetHead (route->imported)) {
		assert (route->current == NULL);
		LL_Remove (route->imported, route->active);
	    }
	    else {
		assert (route->current);
		rip_del_attr (route->current);
		route->current = NULL;
	    }
	    LL_Add (route->imported, rip_attr);
	    route->active = rip_attr;
	    route->flags &= ~RT_RIP_DELETE;
	}
	else {
	    LL_Iterate (route->imported, attr) {
	        if (attr->type == new->type /* &&
	             (new->gateway == NULL || attr->gateway == new->gateway) */)
		    break;
	    }

	    if (attr != NULL) {
	        rip_attr_t *old_best = LL_GetHead (route->imported);
	        rip_attr_t *new_best;

		/* updating the existing one */
		LL_Remove (route->imported, attr);
	        LL_Add (route->imported, rip_attr);
	        trace (TR_TRACE, rip->trace, "update %p nh %a proto %r\n",
		       prefix, new->nexthop->prefix, new->type);
		new_best = LL_GetHead (route->imported);
		if (old_best != new_best) {
		   /* the best changed */
		    if (route->active == route->current) {
		        if (route->active->pref > new_best->pref) {
			    route->active = new_best;
	    		    route->flags |= RT_RIP_CHANGE;
	    		    rip_set_flash_update (rip);
			}
		    }
		    else {
			assert (route->active == old_best);
		        route->active = new_best;
		        if (route->current &&
			        route->active->pref >= route->current->pref) {
			    route->active = route->current;
		        }
	    	        route->flags |= RT_RIP_CHANGE;
	    	        rip_set_flash_update (rip);
		    }
		}
	    }
	    else {
	        LL_Add (route->imported, rip_attr);
	        trace (TR_TRACE, rip->trace, "add %p nh %a proto %r\n",
		       prefix, new->nexthop->prefix, new->type);
	        assert (route->active);
		if (route->active == route->current) {
		    if (route->active->pref >= rip_attr->pref) {
		        assert (rip_attr == LL_GetHead (route->imported));
			route->active = rip_attr;
	    		route->flags |= RT_RIP_CHANGE;
	    		rip_set_flash_update (rip);
		    }
		}
		else {
		    if (rip_attr == LL_GetHead (route->imported)) {
			route->active = rip_attr;
			if (route->current &&
				route->active->pref >= route->current->pref)
			    route->active = route->current;
	    		route->flags |= RT_RIP_CHANGE;
	    		rip_set_flash_update (rip);
		    }
		}
	    }
	}
    }

  quit:
    Deref_Prefix (prefix);
    if (new)
	Deref_Generic_Attr (new);
    if (old)
	Deref_Generic_Attr (old);
}


/* this is called from threads other than rip, so it must be shceduled */
void
rip_update_route (prefix_t * prefix, generic_attr_t * new,
		  generic_attr_t * old, int pref)
{
    rip_t *rip = RIP;

    /* in case the rib wants to delete them */
    Ref_Prefix (prefix);
    if (new)
	Ref_Generic_Attr (new);
    if (old)
	Ref_Generic_Attr (old);

#ifdef HAVE_IPV6
    if (prefix->family == AF_INET6)
	rip = RIPNG;
#endif /* HAVE_IPV6 */

    schedule_event2 ("rip_import",
		 rip->schedule, rip_import, 5, rip, prefix, new, old, pref);
}


/* 
 * change/set rip attributes.
 */
void
rip_set (int proto, int first,...)
{
    va_list ap;
    enum RIP_ATTR attr;
    rip_t *rip = RIP;

#ifdef HAVE_IPV6
    if (proto == PROTO_RIPNG)
	rip = RIPNG;
#endif /* HAVE_IPV6 */

    /* Process the Arguments */
    va_start (ap, first);

    for (attr = (enum RIP_ATTR) first;
	 attr;
	 attr = va_arg (ap, enum RIP_ATTR)) {

	switch (attr) {
/*
	case RIP_RT_UPDATE_FN:
	    rip->update_call_fn = va_arg (ap, int_fn_t);
	    break;
*/
	case RIP_TRACE_STRUCT:
	    rip->trace = va_arg (ap, trace_t *);
	    break;
	default:
	    assert (0);
	    break;
	}
    }
    va_end (ap);
}


/* 
 * dump various rip stats to a socket
 * usually called by UII (user interactive interface 
 */
int
rip_show (int proto, uii_connection_t * uii)
{
    rip_t *rip = RIP;
    rip_interface_t *rip_interface;

#ifdef HAVE_IPV6
    if (proto == PROTO_RIPNG) {
	rip = RIPNG;
    }
#endif /* HAVE_IPV6 */
    uii_add_bulk_output (uii, "Routing Protocol is \"%s\"\n",
		   proto2string (rip->proto));

    if (rip->sockfd < 0)
	uii_add_bulk_output (uii, "Not listening for announcements\n");
    else
	uii_add_bulk_output (uii, "Listening on port %d (socket %d)\n",
		       rip->port, rip->sockfd);

    if (rip->timer->time_next_fire > 0) {
        uii_add_bulk_output (uii,
      		   "Sending updates every %d seconds jitter [%d..%d], "
		   "next due in %d seconds\n", RIP_UPDATE_INTERVAL, 
		    rip->timer->time_jitter_low,
		    rip->timer->time_jitter_high,
		    rip->timer->time_next_fire - time (NULL));
        uii_add_bulk_output (uii, "Triggered update and split horizon "
		       "(no poisoned reverse) implemented\n");
        uii_add_bulk_output (uii,
	     "Invalid after %d, hold down %d, flushed after %d seconds\n",
	      RIP_TIMEOUT_INTERVAL, RIP_GARBAGE_INTERVAL, 
	      RIP_TIMEOUT_INTERVAL+RIP_GARBAGE_INTERVAL);

        uii_add_bulk_output (uii, "Interface enabled:");
        LL_Iterate (rip->ll_rip_interfaces, rip_interface) {
	    uii_add_bulk_output (uii, " %s", 
		rip_interface->interface->name);
	}
        uii_add_bulk_output (uii, "\n");
        uii_add_bulk_output (uii, "Number of routes in routing table: %d\n", 
			     HASH_GetCount (rip->hash));
    }
    else {
	uii_add_bulk_output (uii, "Not sending announcements\n");
    }
    return (1);
}


static int
rip_route_compare (rip_route_t * a, rip_route_t * b)
{
    return (prefix_compare2 (a->prefix, b->prefix));
}


/*
 * dump routing table to socket. Usually called by user interactive interface
 */
int
rip_show_routing_table (int proto, uii_connection_t * uii, char *ifname)
{
    interface_t *interface = NULL;
    rip_t *rip = RIP;
    rip_route_t *route;
    time_t now;
    char stmp[MAXLINE];
    rip_route_t **array;
    int i, c, t;
    u_int nel;

#ifdef HAVE_IPV6
    if (proto == PROTO_RIPNG) {
	rip = RIPNG;
    }
#endif /* HAVE_IPV6 */

    if (ifname) {
	if ((interface = find_interface_byname (ifname)) == NULL) {
	    config_notice (TR_ERROR, uii,
	    		   "no such interface: %s\n", ifname);
	    return (-1);
	}
    }

    if (rip->hash == NULL)
        return (0);

    time (&now);
    array = (rip_route_t **) HASH_ToArray (rip->hash, NULL, &nel);

    ARRAY_Sort (array, nel, (DATA_PTR)rip_route_compare);

    sprintf (stmp, "%-4s %-4s", "Cost", "Time");
    rib_show_route_head (uii, stmp);

    for (i = 0; i < nel; i++) {
	rip_attr_t *attr;
	route = array[i];

	LL_Iterate (route->received, attr) {

	    if (interface && attr->gateway->interface &&
		    interface != attr->gateway->interface)
		continue;

	    sprintf (stmp, "%4d %4ld", attr->metric, now - attr->utime);
	    rib_show_route_line (uii, ' ', ' ', attr->type,
				     attr->pref, now - attr->ctime,
				     route->prefix, attr->nexthop->prefix,
				     attr->gateway, stmp);
	}

	if ((attr = route->current)) {
	    if (interface == NULL || interface == attr->gateway->interface) {

	        c = (route->active == attr)? '>' : '*';
		t = now - attr->utime;
		if (BIT_TEST (route->flags, RT_RIP_DELETE)) {
		    c = 'D';
		    t = now - attr->dtime;
		}
	        sprintf (stmp, "%4d %4d", attr->metric, t);
	        rib_show_route_line (uii, c, ' ', attr->type,
				     attr->pref, now - attr->ctime,
				     route->prefix, attr->nexthop->prefix,
				     attr->gateway, stmp);
	    }
	}

	LL_Iterate (route->imported, attr) {

	    if (interface && attr->gateway->interface &&
		    interface != attr->gateway->interface)
		continue;

	    c = (route->active == attr)? '>' : ' ';
	    sprintf (stmp, "%4d ----", attr->metric);
	    if (BIT_TEST (route->flags, RT_RIP_DELETE)) {
		c = 'D';
	        sprintf (stmp, "%4d %4ld", attr->metric, now - attr->dtime);
	    }

	    rib_show_route_line (uii, c, ' ', attr->type,
				     attr->pref, now - attr->ctime,
				     route->prefix, attr->nexthop->prefix,
				     attr->gateway, stmp);
	}
    }
    Delete (array);
    return (1);
}


static void
rip_timer_update (rip_t * rip)
{
    trace (TR_TRACE, rip->trace, "timer (update) fired\n");
    if (rip->flash_update_waiting)
	rip->flash_update_waiting = 0;	/* clear flash update */
    rip_output_processing (rip, 0);
}


static void
rip_flash_update (rip_t * rip)
{
    trace (TR_TRACE, rip->trace, "timer (flash update) fired\n");
    if (rip->flash_update_waiting) {
	rip->flash_update_waiting = 0;
	rip_output_processing (rip, 1);
    }
}


static int
rip_init_listen (rip_t * rip)
{
#ifdef HAVE_IPV6
    if (rip->proto == PROTO_RIPNG)
	return (ripng_init_listen (NULL));
    else
#endif /* HAVE_IPV6 */
    if (rip->proto == PROTO_RIP)
	return (rip2_init_listen (NULL));
    else {
	assert (0);
    }
    return (-1);
}


#ifdef notdef
static void
rip_send_request (rip_t * rip, LINKED_LIST * ll_prefixes)
{
#ifdef HAVE_IPV6
    if (rip->proto == PROTO_RIPNG)
	ripng_send_request (ll_prefixes);
    else
#endif /* HAVE_IPV6 */
    if (rip->proto == PROTO_RIP)
	rip2_send_request (ll_prefixes);
    else
	assert (0);
}
#endif


/* run under rip thread */
/* run for all the interfaces. it is an easier way
   because MRT allows two ways to specify prefix and interface name */
void
rip_interface_recheck (rip_t * rip)
{
    prefix_t *prefix;
    char *name;
    interface_t *table[MAX_INTERFACES];
    LINKED_LIST *ll;
    int i;
    interface_t *interface;
    rip_interface_t *rip_interface;
    event_fn_t fn;

#ifdef HAVE_IPV6
    if (rip->proto == PROTO_RIPNG)
	fn = ripng_interface;
    else
#endif /* HAVE_IPV6 */
    if (rip->proto == PROTO_RIP)
	fn = rip2_interface;
    else {
	assert (0);
	return;
    }

    memset (table, 0, sizeof (table));
    LL_Iterate (rip->ll_networks, prefix) {
        if ((ll = find_network (prefix)) != NULL) {
	    LL_Iterate (ll, interface) {
		table[interface->index] = interface;
	    }
	    LL_Destroy (ll);
	}
    }
    LL_Iterate (rip->ll_networks2, name) {
	if ((ll = find_interface_byname_all (name)) != NULL) {
	    LL_Iterate (ll, interface) {
		table[interface->index] = interface;
	    }
	    LL_Destroy (ll);
	}
    }

    for (i = 0; i < sizeof (table)/sizeof (table[0]); i++) {
	interface = table[i];
	rip_interface = rip->rip_interfaces[i];
	if (interface == NULL) {
	    if (!BITX_TEST (&rip->interface_mask, i))
		continue;
	    assert (rip_interface);
	    assert (rip_interface->interface->index == i);
	    trace (TR_TRACE, rip->trace, "interface %s (off)\n",
		   interface->name);
            fn (rip_interface, 0);
	    BITX_RESET (&rip->interface_mask, i);
	    LL_Remove (rip->ll_rip_interfaces, rip_interface);
	}
	else {
	    assert (interface->index == i);
	    if (BITX_TEST (&rip->interface_mask, i))
		continue;
#ifdef HAVE_IPV6
	    if (rip->proto == PROTO_RIPNG) {
		if (!BIT_TEST (interface->flags, IFF_MULTICAST) &&
		    BIT_TEST (interface->flags, IFF_POINTOPOINT) &&
			(interface->link_local == NULL ||
			 interface->link_local->broadcast == NULL)) {
                        trace (TR_ERROR, rip->trace,
                                     "RIPNG on interface %s ignored "
                            "due to no multicast or link-local dest addr\n",
                                       interface->name);
			continue;
		    }
	    }
#endif /* HAVE_IPV6 */

	    trace (TR_TRACE, rip->trace, "interface %s (on)\n",
				       interface->name);
	    BITX_SET (&rip->interface_mask, i);
	    LL_Add (rip->ll_rip_interfaces, rip_interface);
            fn (rip_interface, 1);
	}
    }
}


void
rip_distribute_list_recheck (rip_t *rip)
{
    rip_interface_t *rip_interface;
    dlist_t *dlist;

    /* check distribute-list */
    /* reset all first */
    LL_Iterate (rip->ll_rip_interfaces, rip_interface) {
	rip_interface->dlist_out = -1;
	rip_interface->dlist_in = -1;
    }

    /* find out distribute-list without interface */
    /* this is default */
    LL_Iterate (rip->ll_dlists, dlist) {
	if (dlist->interface)
	    continue;
        LL_Iterate (rip->ll_rip_interfaces, rip_interface) {
	    if (dlist->out)
		rip_interface->dlist_out = dlist->num;
	    else
		rip_interface->dlist_in = dlist->num;
	}
    }

    LL_Iterate (rip->ll_dlists, dlist) {
	if (dlist->interface == NULL)
	    continue;
	if (!BITX_TEST (&rip->interface_mask, dlist->interface->index))
	    continue;
	rip_interface = rip->rip_interfaces[dlist->interface->index];
	assert (rip_interface);
	if (dlist->out)
	    rip_interface->dlist_out = dlist->num;
	else
	    rip_interface->dlist_in = dlist->num;
	
    }
}


void
rip_start (int proto, int port)
{
    rip_route_t route;
    interface_t *interface;
    rip_t *rip = RIP;

#ifdef HAVE_IPV6
    if (proto == PROTO_RIPNG) {
	rip = RIPNG;
    }
#endif /* HAVE_IPV6 */
    rip->hash = HASH_Create (RIP_TABLE_HASH_SIZE,
			     HASH_KeyOffset, 
			     HASH_Offset (&route, &route.prefix),
			     HASH_LookupFunction, rip_lookup_fn,
			     HASH_HashFunction, rip_hash_fn,
			     HASH_DestroyFunction, rip_delete_route,
			     NULL);

    rip->ll_networks = LL_Create (LL_DestroyFunction, Deref_Prefix, 0);
    rip->ll_networks2 = LL_Create (LL_DestroyFunction, free, 0);
    rip->ll_dlists = LL_Create (LL_DestroyFunction, free, 0);
    rip->ll_rip_interfaces = LL_Create (0);
    memset (&rip->interface_mask, 0, sizeof (rip->interface_mask));

    /* copy all interfaces at this point, so all new interfaces created later
       by the kernel should be entered individually into ll_rip_interfaces 
       structure */
    LL_Iterate (INTERFACE_MASTER->ll_interfaces, interface) {
	rip_interface_t *rip_interface = New (rip_interface_t);
	rip_interface->interface = interface;
        rip_interface->dlist_in = -1;
        rip_interface->dlist_out = -1;
        rip_interface->metric_in = 1;
        rip_interface->metric_out = 0;
        rip_interface->default_pref = -1;
        rip_interface->sockfd = -1;
	rip->rip_interfaces[interface->index] = rip_interface;
    }
    MRT->proto_update_route[proto] = rip_update_route;

    rip->port = port;
    if ((rip->sockfd = rip_init_listen (rip)) < 0) {
	trace (TR_ERROR, rip->trace, "aborted due to error(s)\n");
	return;
    }
    select_add_fd_event ("rip_receive_update", rip->sockfd, SELECT_READ, 1,
                         rip->schedule, rip_receive_update, 1, rip);

    Timer_Turn_ON (rip->timer);
    Timer_Turn_ON (rip->age);

    /* send initial request to routers */
    /* rip_send_request (rip, NULL); */
}


void
rip_stop (int proto)
{
    rip_t *rip = RIP;
    int i;

#ifdef HAVE_IPV6
    if (proto == PROTO_RIPNG) {
	rip = RIPNG;
    }
#endif /* HAVE_IPV6 */

    if (rip->sockfd >= 0) {
	trace (TR_INFO, rip->trace, "Closing scoket %d\n", rip->sockfd);
	select_delete_fd (rip->sockfd);
	rip->sockfd = -1;
    }

    /* stop all interfaces */
    LL_Clear (rip->ll_networks);
    LL_Clear (rip->ll_networks2);
    rip_interface_recheck (rip);
    LL_Destroy (rip->ll_networks);
    LL_Destroy (rip->ll_networks2);
    LL_Clear (rip->ll_dlists);
    rip_distribute_list_recheck (rip);
    LL_Destroy (rip->ll_dlists);
    LL_Destroy (rip->ll_rip_interfaces);
    memset (&rip->interface_mask, 0, sizeof (rip->interface_mask));

    for (i = 0; i <= PROTO_MAX; i++) {
        if (BGP4_BIT_TEST (rip->redistribute_mask, i)) {
            if (MRT->rib_redistribute_request)
                MRT->rib_redistribute_request (rip->proto, i, 0);
        }   
    }       
    rip->redistribute_mask = 0;

    for (i = 0; i < MAX_INTERFACES; i++) {
        if (rip->rip_interfaces[i]) {
	    Delete (rip->rip_interfaces[i]);
	    rip->rip_interfaces[i] = NULL;
	}
    }
    MRT->proto_update_route[proto] = NULL;

    Timer_Turn_OFF (rip->timer);
    Timer_Turn_OFF (rip->age);
}


/*
 * initialize rip stuff
 */
int
rip_init (int proto, trace_t * tr)
{
    rip_t *rip;
    char *name = "RIP";
    int port = RIP_DEFAULT_PORT;

#ifdef HAVE_IPV6
    if (proto == PROTO_RIPNG) {
	name = "RIPNG";
	port = RIPNG_DEFAULT_PORT;
    }
#endif /* HAVE_IPV6 */
    rip = New (rip_t);
    rip->trace = trace_copy (tr);
    rip->proto = proto;
    rip->sockfd = -1;
    rip->port = port;

    set_trace (rip->trace, TRACE_PREPEND_STRING, name, 0);
    rip->schedule = New_Schedule (name, rip->trace);
    rip->timer = New_Timer2 ("RIP update timer", RIP_UPDATE_INTERVAL, 0,
			     rip->schedule, rip_timer_update, 1, rip);
    timer_set_jitter2 (rip->timer, -50, 50);
    rip->age = New_Timer2 ("RIP aging timer", RIP_TIMEOUT_INTERVAL, 
			   TIMER_ONE_SHOT, rip->schedule,
			   rip_timeout_routes, 1, rip);
    rip->flash = New_Timer2 ("RIP flash timer", 0, TIMER_ONE_SHOT,
			     rip->schedule, rip_flash_update, 1, rip);
#ifdef HAVE_IPV6
    if (proto == PROTO_RIPNG) {
	ripng_init ();
	RIPNG = rip;
    }
    else
#endif /* HAVE_IPV6 */
    if (proto == PROTO_RIP) {
	RIP = rip;
    }
    else {
	assert (0);
    }

    mrt_thread_create2 (name, rip->schedule, NULL, NULL);
    return (1);
}

