/*
 * $Id: rip2.c,v 1.10 1999/04/27 01:14:38 masaki Exp $
 */

#include <sys/fcntl.h>
#include <mrt.h>
#include <api6.h>
#include <rip.h>


static void
rip2_process_packet_response (gateway_t * gateway, u_char * update, int bytes,
			      LINKED_LIST * ll_prefixes, LINKED_LIST * ll_attr,
			      int pref)
{
    u_char *cp;
    prefix_t *prefix;
    rip_attr_t *p_attr;
    time_t now;

#define RIP_RTELEN 20

    if ((bytes % RIP_RTELEN) != 0) {
	trace (TR_WARN, RIP->trace, "invalid RTE size %d\n", bytes);
    }

    time (&now);
    cp = update;

    while (cp < update + bytes) {
	int afi, tag, prefixlen, metric;
	struct in_addr addr, mask, nhop;

	BGP_GET_SHORT (afi, cp);
	BGP_GET_SHORT (tag, cp);
	BGP_GET_ADDR (&addr, cp);
	BGP_GET_ADDR (&mask, cp);
	BGP_GET_ADDR (&nhop, cp);
	BGP_GET_LONG (metric, cp);

	if (afi == 0xffff /* Authentication */ ) {
	    trace (TR_ERROR, RIP->trace,
		   "Authentication not supported (type %d)\n", tag);
	    continue;
	}
	/* IP = 1, IP6 = 2 defined in RFC 1700 p.91 but RIP RFC says 2 for IP */
	if (afi != 2 /* IP AFI */ ) {
	    trace (TR_ERROR, RIP->trace, "unknown afi = %d\n", afi);
	    continue;
	}

	prefixlen = mask2len (&mask, 4);
	prefix = New_Prefix (AF_INET, &addr, prefixlen);

	p_attr = rip_new_attr (RIP, metric);
	p_attr->gateway = gateway;
	if (nhop.s_addr == INADDR_ANY) {
	    p_attr->nexthop = ref_nexthop (gateway);
	}
	else {
	    prefix_t *prefix = New_Prefix (AF_INET, &nhop, 32);
	    p_attr->nexthop = add_nexthop (prefix, gateway->interface);
	    Deref_Prefix (prefix);
	}
	p_attr->metric = metric;
	p_attr->tag = tag;
	p_attr->utime = now;
	p_attr->pref = pref;
	LL_Add (ll_prefixes, prefix);
	LL_Add (ll_attr, p_attr);
    }
}


/* 
 * Given 1) the gateway we received the packet from, 
 *       2) a pointer to the packet, and
 *       3) the length (num bytes) of the packet, 
 * fill up metrics if exists
 *
 * return value >0: ordinal request (number of entries)
 *              -1: whole-table request
 */

static int
rip2_process_packet_request (struct sockaddr_in *from, u_char * update,
			     int bytes, rip_interface_t * rip_interface)
{
    u_char *cp = update;
    prefix_t *prefix;
    rip_route_t *route;
    int n = 0;

    if ((bytes % RIP_RTELEN) != 0) {
	trace (TR_WARN, RIP->trace, "invalid RTE size %d\n", bytes);
    }

    prefix = New_Prefix (AF_INET, &from->sin_addr, 32);

    while (cp < update + bytes) {
	int afi, tag, prefixlen, metric;
	struct in_addr addr, mask, nhop;

	if ((update + bytes) - cp < RIP_RTELEN)
	    break;

	BGP_GET_SHORT (afi, cp);
	BGP_GET_SHORT (tag, cp);
	BGP_GET_ADDR (&addr, cp);
	BGP_GET_ADDR (&mask, cp);
	BGP_GET_ADDR (&nhop, cp);
	BGP_GET_LONG (metric, cp);

	prefixlen = mask2len (&mask, 4);

	/* Address family field in RIP REQUEST must be zero. (RFC1058 3.4.1) 
		"Vladimir V. Ivanov" <vlad@elis.tusur.ru> */
	if (cp == update + bytes && afi == 0 &&
	    metric == RIP_METRIC_INFINITY /* &&
	    prefixlen == 0 && tag == 0 && 
	    addr.s_addr == INADDR_ANY && mask.s_addr == INADDR_ANY &&
	    nhop.s_addr == INADDR_ANY */) {
	    trace (TR_PACKET, RIP->trace,
		   "  whole-table request, responding\n");
	    Deref_Prefix (prefix);
	    return (-1);
	}

	prefix = Change_Prefix (AF_INET, &addr, prefixlen, prefix);

	if (afi != 2 /* IP AFI */) {
	    trace (TR_WARN, RIP->trace, "unknown afi = %d\n", afi);
	    continue;
	}

	if ((route = HASH_Lookup (RIP->hash, prefix))) {
	    rip_attr_t *attr = route->active;

	    tag = attr->tag;

	    if (ntohs (from->sin_port) == RIP->port) {
		/* router's query, apply policy for that interface */
		if ((metric = rip_policy (RIP, prefix, attr,
					  rip_interface)) < 0) {
		    metric = RIP_METRIC_INFINITY;
		}
	    }
	    else {
		trace (TR_PACKET, RIP->trace, "  o %p metric %d\n",
		       prefix, metric);
	    }
	}
	else {
	    metric = RIP_METRIC_INFINITY;
	    tag = 0;
	    trace (TR_PACKET, RIP->trace, "  x %p metric %d (no exist)\n",
		   prefix, metric);
	}
	cp -= 18;
	BGP_PUT_SHORT (tag, cp);
	cp += 4;		/* skip ip address */
	cp += 4;		/* skip subnet mask */
	BGP_PUT_NETLONG (INADDR_ANY, cp);		/* nexthop itself XXX */
	BGP_PUT_LONG (metric, cp);
	n++;
    }

    Deref_Prefix (prefix);
    return (n);
}


/*
 * start listening for broadcast rip updates
 */
int
rip2_init_listen (prefix_t *prefix)
{
    struct sockaddr_in serv_addr;
    const char zero = 0;
    const int one = 1;
    int sockfd;

    if ((sockfd = socket (AF_INET, SOCK_DGRAM, 0)) < 0) {
	trace (TR_ERROR, RIP->trace, "socket: %m\n");
	return (-1);
    }

    setsockopt (sockfd, SOL_SOCKET, SO_REUSEADDR, (char *)&one, sizeof (one));
#ifdef SO_REUSEPORT
    setsockopt (sockfd, SOL_SOCKET, SO_REUSEPORT, (char *)&one, sizeof (one));
#endif /* SO_REUSEPORT */

#ifdef IP_MULTICAST_LOOP
    /* disable loopback */
    if (setsockopt (sockfd, IPPROTO_IP, IP_MULTICAST_LOOP,
		    &zero, sizeof (zero)) < 0) {
	trace (TR_ERROR, RIP->trace,
	       "setsockopt IP_MULTICAST_LOOP: %m\n");
    }
#endif /* IP_MULTICAST_LOOP */

    memset (&serv_addr, 0, sizeof (serv_addr));
    /* check that NOT listening to portmaster! */
    serv_addr.sin_port = htons (RIP->port);
    serv_addr.sin_family = AF_INET;
    if (prefix) {
        serv_addr.sin_addr.s_addr = prefix_tolong (prefix);
    }
    else {
        serv_addr.sin_addr.s_addr = INADDR_ANY;
    }

    if (bind (sockfd,
	      (struct sockaddr *) &serv_addr, sizeof (serv_addr)) < 0) {
	trace (TR_ERROR, RIP->trace, "bind to port %d: %m\n", RIP->port);
	return (-1);
    }
    trace (TR_TRACE, RIP->trace, "listening on port %d at %s\n",
           ntohs (serv_addr.sin_port), (prefix)? prefix_toa (prefix): "*");
    /* RIP->sockfd = sockfd; */
    return (sockfd);
}


/*
 * turn on the interface
 */
void
rip2_interface (rip_interface_t *rip_interface, int on)
{
    struct ip_mreq mreq;
    int cmd = IP_DROP_MEMBERSHIP;
    char *s_cmd = "IP_DROP_MEMBERSHIP";
    char stmp[INET_ADDRSTRLEN];
    interface_t *interface = rip_interface->interface;

    /*
     * Join the specified multicast address
     */

#define RIP2_MULTI_ADDR htonl(0xE0000009)

    memset (&mreq, 0, sizeof (mreq));
    mreq.imr_multiaddr.s_addr = RIP2_MULTI_ADDR;
    mreq.imr_interface.s_addr = prefix_tolong (interface->primary->prefix);

    if (on) {
        cmd = IP_ADD_MEMBERSHIP;
        s_cmd = "IP_ADD_MEMBERSHIP";
    }

    if (setsockopt (RIP->sockfd, IPPROTO_IP, cmd,
			(char *) &mreq, sizeof (mreq)) >= 0) {
	trace (TR_TRACE, RIP->trace, "%s %s on %s\n", s_cmd,
	       inet_ntop (AF_INET, &mreq.imr_multiaddr, stmp, sizeof stmp),
	       interface->name);

    }
    else {
	trace (TR_ERROR, RIP->trace,
	       "setsockopt %s %s for %s: %m\n", s_cmd,
	       inet_ntop (AF_INET, &mreq.imr_multiaddr, stmp, sizeof stmp),
	       interface->name);
	return;
    }

    if (on) {
	BITX_SET (&RIP->interface_mask, interface->index);
	/* send initial request to routers */
	rip2_send_request (rip_interface, NULL);
    }
    else {
	BITX_RESET (&RIP->interface_mask, interface->index);
    }
}


static int
rip2_sendmsgto (rip_interface_t *rip_interface,
		u_char * buffer, int buflen, u_long flag,
		struct sockaddr_in *host)
{
    int n;
    struct sockaddr_in sin;
    char stmp[INET_ADDRSTRLEN];
    interface_t *interface = rip_interface->interface;
    int sockfd = (rip_interface->sockfd >= 0)?
		    rip_interface->sockfd: RIP->sockfd;

    if (host) {
	memcpy (&sin, host, sizeof (sin));
    }
    else {
        if (!BIT_TEST (interface->flags, IFF_UP))
	    return (-1);
        if (interface->primary == NULL)
	    return (-1);

	memset (&sin, 0, sizeof sin);
	sin.sin_port = htons (RIP->port);
	sin.sin_family = AF_INET;

	if (BIT_TEST (interface->flags, IFF_MULTICAST)) {
	    if (sockfd == RIP->sockfd) {
	        if (setsockopt (sockfd, IPPROTO_IP, IP_MULTICAST_IF,
			        prefix_tochar (interface->primary->prefix),
			        sizeof (struct in_addr)) < 0) {
		    trace (TR_ERROR, RIP->trace,
		           "setsockopt IP_MULTICAST_IF for %s: %m\n",
		           interface->name);
		}
	    }
	    sin.sin_addr.s_addr = RIP2_MULTI_ADDR;
	}
	else if (interface->primary->broadcast) {
	    sin.sin_addr.s_addr = prefix_tolong (interface->primary->broadcast);
	}
	else {
	    assert (0);
	}
    }
    if ((n = sendto (sockfd, (char *) buffer, buflen, 0,
		     (struct sockaddr *) &sin, sizeof (sin))) < 0) {
	trace (TR_ERROR, RIP->trace, "sendto %s on %s: %m\n",
	       inet_ntop (AF_INET, &sin.sin_addr, stmp, sizeof stmp),
	       (interface)? interface->name: "?");
    }
    else {
	trace (TR_TRACE, RIP->trace, "send %d bytes to %s on %s\n", buflen,
	       inet_ntop (AF_INET, &sin.sin_addr, stmp, sizeof stmp),
	       (interface)? interface->name: "?");
    }
    return (n);
}


/*
 * broadcast request for list of prefixes
 * if NULL, send a request for complete routing table 
 */
void
rip2_send_request (rip_interface_t *rip_interface, LINKED_LIST * ll_prefixes)
{
    u_char buffer[RIP_MAX_PDU], *cp = buffer;

    assert (ll_prefixes == NULL);	/* XXX not yet implemented */

    BGP_PUT_BYTE (RIP_REQUEST, cp);
    BGP_PUT_BYTE (RIP_VERSION, cp);
    BGP_PUT_SHORT (0, cp);	/* unused */
    BGP_PUT_SHORT (0, cp);     /* AF_UNSPEC */
    BGP_PUT_SHORT (0, cp);	/* tag */
    BGP_PUT_NETLONG (0, cp);	/* ip address */
    BGP_PUT_NETLONG (0, cp);	/* subnet mask */
    BGP_PUT_NETLONG (0, cp);	/* next hop */
    BGP_PUT_LONG (RIP_METRIC_INFINITY, cp);

    if (rip2_sendmsgto (rip_interface, buffer, cp - buffer, 0, NULL) >= 0)
	    trace (TR_TRACE, RIP->trace, "send request on %s\n",
		   rip_interface->interface->name);
}


/*
 * given an interface, broadcast rip routes (according to policy) 
 * we do this every RIP_UPDATE_INTERVAL
 */
void
rip2_send_routes (rip_interface_t * rip_interface, struct sockaddr_in *host, 
		  int change)
{
    u_char buffer[RIP_MAX_PDU], *cp = buffer;
    rip_route_t *route;
    int routes = 0;

#ifdef notdef
    if (host) {
	char stmp[INET_ADDRSTRLEN];
	trace (TR_TRACE, RIP->trace, "preparing for %s\n",
	       inet_ntop (AF_INET, &host->sin_addr, stmp, sizeof stmp));
    }
    else if (rip_interface)
	trace (TR_TRACE, RIP->trace, "preparing for %s\n", 
	      rip_interface->interface->name);
    else
	assert (0);
#endif

    BGP_PUT_BYTE (RIP_RESPONSE, cp);
    BGP_PUT_BYTE (RIP_VERSION, cp);
    BGP_PUT_SHORT (0, cp);	/* unused */

    /* on flashing update, locking should be done outer side
       because of keeping change flags and reset them after sending out */

    HASH_Iterate (RIP->hash, route) {
    
	prefix_t *prefix = route->prefix;
	rip_attr_t *attr = route->active;
	int metric;
	struct in_addr addr;
	char tagstr[MAXLINE];

	/* doing ouput processing and only sending changed routes */
	if (change && (!BIT_TEST (route->flags, RT_RIP_CHANGE))) {
	    continue;
	}

	if ((metric = rip_policy (RIP, prefix, attr, rip_interface)) < 0)
	    continue;

	if (attr->tag) {
	    sprintf (tagstr, " tag 0x%lx", attr->tag);
	}
	else {
	    tagstr[0] = '\0';
	}
	trace (TR_PACKET, RIP->trace, "  o %p metric %d%s\n",
	       prefix, metric, tagstr);

	BGP_PUT_SHORT (2, cp);	/* afi */
	BGP_PUT_SHORT (attr->tag, cp);	/* tag */
	BGP_PUT_ADDR (prefix_tochar (prefix), cp);	/* ip address */
	len2mask (prefix->bitlen, &addr, sizeof addr);
	BGP_PUT_ADDR (&addr, cp);	/* subnet mask */
	BGP_PUT_NETLONG (0, cp);	/* next hop, itself only XXX */
	BGP_PUT_LONG (metric, cp);
	routes++;

	/* see if we have filled up the buffer */
	if (cp - buffer > RIP_MAX_PDU - RIP_RTELEN) {
	    if (rip2_sendmsgto (rip_interface, buffer, cp - buffer, 0,
				host) < 0)
		break;

	    if (host) {
		char stmp[INET_ADDRSTRLEN];
		trace (TR_TRACE, RIP->trace, "send %d routes to %s\n",
		       routes, inet_ntop (AF_INET, &host->sin_addr,
					  stmp, sizeof stmp));
	    }
	    else if (rip_interface)
		trace (TR_TRACE, RIP->trace, "send %d routes on %s\n",
		       routes, rip_interface->interface->name);
	    else
		assert (0);
	    cp = buffer;
	    BGP_PUT_BYTE (RIP_RESPONSE, cp);
	    BGP_PUT_BYTE (RIP_VERSION, cp);
	    BGP_PUT_SHORT (0, cp);	/* unused */
	    routes = 0;
	}

    }

    if (routes > 0) {
	if (rip2_sendmsgto (rip_interface, buffer, cp - buffer, 0,
			    host) >= 0) {
	    if (host) {
		char stmp[INET_ADDRSTRLEN];
		trace (TR_TRACE, RIP->trace, "send %d routes to %s\n",
		       routes, inet_ntop (AF_INET, &host->sin_addr,
					  stmp, sizeof stmp));
	    }
	    else if (rip_interface)
		trace (TR_TRACE, RIP->trace, "send %d routes on %s\n",
		       routes, rip_interface->interface->name);
	    else
		assert (0);
	}
    }
}


/* rip_receive_update
 * read and process and RIP update packet recieved on
 * our interface
 */
void
rip2_receive_update (rip_interface_t *rip_interface)
{
#define RIP_MAX_PDU 512
    u_char buffer[RIP_MAX_PDU], *cp = buffer;
    int n, fromlen;
    int command, version, zero;
    struct sockaddr_in from;
    prefix_t *prefix;
    gateway_t *gateway;
    interface_t *interface;
    int sockfd;

    sockfd = (rip_interface && rip_interface->sockfd)? 
		rip_interface->sockfd: RIP->sockfd;
    fromlen = sizeof (from);
    n = recvfrom (sockfd, (char *)buffer, sizeof buffer, 
		  O_NONBLOCK, (struct sockaddr *) &from, &fromlen);
    select_enable_fd (sockfd);
    if (n <= 0) {
/* zero may be OK? XXX */
	trace (TR_ERROR, RIP->trace, "recvmsg: %m\n");
	return;
    }

    prefix = New_Prefix (AF_INET, &from.sin_addr, 32);
    interface = find_interface (prefix);

    if (interface == NULL) {
	trace (TR_WARN, RIP->trace,
	       "discard packet from %a (interface unknown)\n", prefix);
	goto ignore;
    }
    if (rip_interface == NULL) {
        trace (TR_TRACE, RIP->trace,
               "packet from %a at RIP socket %d\n", prefix, sockfd);
        rip_interface = RIP->rip_interfaces[interface->index];
        assert (rip_interface);
    }
    else if (rip_interface != RIP->rip_interfaces[interface->index]) {
        trace (TR_ERROR, RIP->trace,
               "confusion: from %a on %s but must be on %s\n", prefix,
               interface->name, rip_interface->interface->name);
        interface = rip_interface->interface;
    }

    if (!BITX_TEST (&RIP->interface_mask, interface->index)) {
        trace (TR_WARN, RIP->trace,
               "packet from %a on disabled interface %s\n",                     
               prefix, interface->name);         
        goto ignore;
    }

    assert (rip_interface);

    /*
     * get command first for later check
     */
    BGP_GET_BYTE (command, cp);
    BGP_GET_BYTE (version, cp);
    BGP_GET_SHORT (zero, cp);

    if (version != RIP_VERSION) {
	trace (TR_WARN, RIP->trace,
	       "unsupported version %d from %a on %s\n",
	       version, prefix, interface->name);
	goto ignore;
    }

    if (zero) {
	trace (TR_WARN, RIP->trace,
	       "non-zero pad field (value 0x%x) from %a on %s\n",
	       zero, prefix, interface->name);
/* XXX  goto ignore; */
    }

    if (command != RIP_RESPONSE && command != RIP_REQUEST) {
	trace (TR_WARN, RIP->trace,
	       "unsupported command %d from %a on %s, ignore!\n",
	       command, prefix, interface->name);
	goto ignore;
    }

    if (command == RIP_RESPONSE) {

	/* register the gateway */
	gateway = add_gateway (prefix, 0, interface);

	/* don't listen to things broadcast from our own interface */
	if (find_interface_local (prefix)) {
	    trace (TR_PACKET, RIP->trace,
		   "recv ignore own response from %a on %s, ignore!\n",
		   prefix, interface->name);
	    goto ignore;
	}

	/* check if this interface is configured for RIP */
	if (!BITX_TEST (&RIP->interface_mask, interface->index)) {
	    trace (TR_PACKET, RIP->trace, "received response "
		   "from %a on unconfigured interface %s, ignore!\n",
		   gateway->prefix, interface->name);
	    goto ignore;
	}

	if (ntohs (from.sin_port) != RIP->port) {
	    trace (TR_INFO, RIP->trace,
		   "non-standard source port %d from %a on %s, "
		   "ignored!\n", ntohs (from.sin_port),
		   gateway->prefix, interface->name);
	    goto ignore;
	}

	trace (TR_TRACE, RIP->trace,
	       "recv response %d bytes from %a on %s\n",
	       n, gateway->prefix, interface->name);

	if (n - (cp - buffer)) {
	    LINKED_LIST *ll_prefixes, *ll_attr;

	    ll_prefixes = LL_Create (LL_DestroyFunction, Deref_Prefix, 0);
	    ll_attr = LL_Create (0);

	    /* munge the rip packet
	       and return list of prefixes and attributes */

	    rip2_process_packet_response (gateway, cp, n - (cp - buffer),
					  ll_prefixes, ll_attr,
		(rip_interface->default_pref >= 0)? 
			rip_interface->default_pref: RIP_PREF);

	    /* update our tables */
	    rip_process_update (RIP, ll_prefixes, ll_attr);

	    LL_Destroy (ll_prefixes);
	    LL_Destroy (ll_attr);
	}
    }
    else if (command == RIP_REQUEST) {

	/* register the gateway */
	if (ntohs (from.sin_port) == RIP->port)
	    gateway = add_gateway (prefix, 0, interface);

	/* don't listen to things broadcast from our own interface
	   except for a query */
	if (find_interface_local (prefix) &&
	    ntohs (from.sin_port) == RIP->port) {
	    goto ignore;
	}

	trace (TR_TRACE, RIP->trace,
	       "recv request %d bytes port %d from %a on %s\n",
	       n, ntohs (from.sin_port), prefix, interface->name);

	if (n - (cp - buffer) > 0) {
	    /* more than 1 entry */
	    if (rip2_process_packet_request (&from, cp, n - (cp - buffer),
					     rip_interface) >= 0) {
		cp = buffer;
		BGP_PUT_BYTE (RIP_RESPONSE, cp);
		BGP_PUT_BYTE (RIP_VERSION, cp);
		BGP_PUT_SHORT (0, cp);

		if (rip2_sendmsgto (rip_interface, buffer, n, 0, &from) < 0) {
		    goto ignore;
		}
	    }
	    else {
		rip2_send_routes (rip_interface, &from, 0);
	    }
	    trace (TR_PACKET, RIP->trace,
		   "recv request answered to %a port %d on %s\n",
		   prefix, ntohs (from.sin_port), interface->name);
	}
	else {
	    trace (TR_WARN, RIP->trace,
		   "recv request no entry from %a on %s, discard it!\n",
		   prefix, interface->name);
	}
    }
  ignore:
    if (prefix)
	Deref_Prefix (prefix);

}
