/*
 * $Id: rib.c,v 1.20 1999/04/29 19:02:50 masaki Exp $
 */

#include <mrt.h>
#include <rib.h>

int num_active_generic_attr = 0;
int num_active_route_head = 0;
int num_active_route_node = 0;


static int
generic_attr_equal (generic_attr_t *a, generic_attr_t *b)
{
    return (a->type == b->type &&
	    a->gateway == b->gateway &&
	    a->nexthop == b->nexthop &&
	    a->tag == b->tag);
}


static buffer_t *
route_toa_buffer (prefix_t *prefix, generic_attr_t *attr, int pref, 
		  u_long flags, buffer_t *buffer)
{
    if (buffer == NULL)
	buffer = New_Buffer (0);
    
    if (attr->nexthop !=  NULL) 
      buffer_printf (buffer, "%p nh %a", prefix, attr->nexthop->prefix);
    else
      buffer_printf (buffer, "%p nh unknown", prefix);

    if (attr->gateway) {
	if (!prefix_compare (attr->gateway->prefix,
			     attr->nexthop->prefix)) {
	    buffer_printf (buffer, " gw %a", attr->gateway->prefix);
	}
    }
    buffer_printf (buffer, " proto %s pref %d",
	     	   proto2string (attr->type), pref);
    if (flags != 0)
        buffer_printf (buffer, " flags 0x%x", flags);
    return (buffer);
}


static route_node_t *
New_Route_Node (route_head_t *route_head, generic_attr_t *attr, int pref,
		u_long flags)
{

    route_node_t *route_node = New (route_node_t);
    route_node->route_head = route_head;
    route_node->attr = Ref_Generic_Attr (attr);
    route_node->pref = pref;
    route_node->flags = flags;
    time (&route_node->time);
    num_active_route_node++;
    
    return (route_node);
}


static void 
Delete_Route_Node (route_node_t * rt_node)
{
    Deref_Generic_Attr (rt_node->attr);
    Delete (rt_node);
    num_active_route_node--;
}


static route_head_t *
New_Route_Head (prefix_t * prefix)
{
    route_head_t *route_head = New (route_head_t);

    route_head->prefix = Ref_Prefix (prefix);
    route_head->ll_route_nodes = 
	LL_Create (LL_DestroyFunction, Delete_Route_Node, 0);
    route_head->active = NULL;

    num_active_route_head++;
    return (route_head);
}


static void 
Delete_Route_Head (route_head_t *route_head)
{
    Deref_Prefix (route_head->prefix);
    LL_Destroy (route_head->ll_route_nodes);
    Delete (route_head);
    num_active_route_head--;
}


generic_attr_t *
New_Generic_Attr (int proto)
{
    generic_attr_t *attr;

    attr = New (generic_attr_t);
    attr->type = proto;
    attr->gateway = NULL;
    attr->nexthop = NULL;
    attr->ref_count = 1;
    pthread_mutex_init (&attr->mutex_lock, NULL);

    num_active_generic_attr++;
    return (attr);
}


generic_attr_t *
Ref_Generic_Attr (generic_attr_t * attr)
{
    if (attr == NULL)
	return (attr);
    pthread_mutex_lock (&attr->mutex_lock);
    attr->ref_count++;
    assert (attr->ref_count > 0);
    pthread_mutex_unlock (&attr->mutex_lock);
    return (attr);
}


void 
Deref_Generic_Attr (generic_attr_t * attr)
{
    if (attr == NULL)
	return;
#ifdef notdef
    if (attr->type == PROTO_RIPNG)
        ripng_deref_attr ((ripng_attr_t *)attr);
    else if (attr->type == PROTO_BGP)
	bgp_deref_attr ((bgp_attr_t *)attr);
    else {
#else
    {
#endif
        pthread_mutex_lock (&attr->mutex_lock);
        if (--attr->ref_count <= 0) {
	    deref_nexthop (attr->nexthop);
            pthread_mutex_destroy (&attr->mutex_lock);
	    Delete (attr);
    	    num_active_generic_attr--;
	    return;
	}
        pthread_mutex_unlock (&attr->mutex_lock);
    }
}


rib_t *
New_Rib (int maxbitlen)
{
    rib_t *rib = New (rib_t);

    /*rib->ll_route_nodes = LL_Create (0); */
    rib->ll_changed_route_nodes = LL_Create (0);

    /*if (LL_ATTR_MEMORY == NULL)
       init_attr_memory (); */

    if (pthread_mutex_init (&rib->mutex_lock, NULL) != 0) {
	perror ("rib pthread_mutex_init");
	exit (0);
    }
    rib->radix_tree = New_Radix (maxbitlen);
    return (rib);
}


static int
check_networks (prefix_t * prefix, LINKED_LIST * ll_networks)
{
    prefix_t *network;

    assert (ll_networks);
    LL_Iterate (ll_networks, network) {
	if (network->family == prefix->family &&
	    a_include_b (network, prefix))
	    return (1);
    }
    return (0);
}


static int
rib_redistribute_request2 (rib_t *rib, int from, int to, int on)
{
    route_head_t *route_head;
    route_node_t *route_node;

    rib_open (rib);
    if (on) {
	if (BGP4_BIT_TEST (rib->redistribute_mask[from], to)) {
	    rib_close (rib);
	    return (0);
	}
	BGP4_BIT_SET (rib->redistribute_mask[from], to);
    }
    else {
	if (!BGP4_BIT_TEST (rib->redistribute_mask[from], to)) {
	    rib_close (rib);
	    return (0);
	}
	BGP4_BIT_RESET (rib->redistribute_mask[from], to);
    }

    RIB_RADIX_WALK (rib, route_head) {
	int skip = 0;

	if (BGP4_BIT_TEST (rib->ll_networks_mask[from], to)) {
	    assert (rib->ll_networks[from]);
	    if (check_networks (route_head->prefix, rib->ll_networks[from]))
		skip++;
	}

if (!skip)
	LL_Iterate (route_head->ll_route_nodes, route_node) {
    	    if (BIT_TEST (route_node->flags, MRT_RTOPT_SUPPRESS))
	        continue;

	    if (route_node->attr->type != to)
		continue;

	    if (on)
    		MRT->proto_update_route[from] (route_head->prefix,
					       route_node->attr, NULL,
					       route_node->pref);
	    else
    		MRT->proto_update_route[from] (route_head->prefix,
					       NULL, route_node->attr,
					       0);
	} 
    } RIB_RADIX_WALK_END;
    rib_close (rib);
    return (1);
}


/* 
   router rip
   redistribute static
     => rib->redistribute_mask[RIP] |= (1 << STATIC)
*/
static int
rib_redistribute_request (int from, int to, int on)
{
    int ret = 0;

    if (from == PROTO_BGP || from == PROTO_OSPF || from == PROTO_RIP) {
	ret = rib_redistribute_request2 (RIB, from, to, on);
    }
#ifdef HAVE_IPV6
    if (from == PROTO_BGP || from == PROTO_RIPNG) {
	int ret2;
	ret2 = rib_redistribute_request2 (RIBv6, from, to, on);
	if (ret < 0 || ret2 < 0)
	    ret = -1;
	else if (ret > 0 || ret2 > 0)
	    ret = 1;
	else
	    ret = 0;
    }
#endif /* HAVE_IPV6 */
    return (ret);
}


/* 
   router rip
   network 1.2.3.4
     => rib->ll_networks_mask[RIP] |= (1 << CONNECTED)
     => rib->ll_networks[RIP] += (1.2.3.4)
*/
static int
rib_redistribute_network (int from, prefix_t *network, int on)
{
    route_head_t *route_head;
    route_node_t *route_node;
    rib_t *rib = RIB;
    u_long mask = 0;

    switch (from) {
    case PROTO_BGP:
	BGP4_BIT_SET (mask, PROTO_RIP);
	BGP4_BIT_SET (mask, PROTO_OSPF);
	BGP4_BIT_SET (mask, PROTO_STATIC);
#ifdef HAVE_IPV6
	BGP4_BIT_SET (mask, PROTO_RIPNG);
#endif /* HAVE_IPV6 */
    default:
	BGP4_BIT_SET (mask, PROTO_CONNECTED);
	break;
    }
#ifdef HAVE_IPV6
    if (network->family == AF_INET6)
	rib = RIBv6;
#endif /* HAVE_IPV6 */

        rib_open (rib);
	if (on) {
	    if (rib->ll_networks[from]) {
		prefix_t *prefix;
		LL_Iterate (rib->ll_networks[from], prefix) {
		    if (prefix_compare (prefix, network)) {
			rib_close (rib);
			return (-1);
		    }
		}
	    }
	    else {
		rib->ll_networks[from] = LL_Create (LL_DestroyFunction, 
						    Deref_Prefix, 0);
		BIT_SET (rib->ll_networks_mask[from], mask);
	    }
	    LL_Add (rib->ll_networks[from], Ref_Prefix (network));
	}
	else {
	    if (rib->ll_networks[from] == NULL) {
		rib_close (rib);
		return (-1);
	    }
	    else {
		prefix_t *prefix;
		LL_Iterate (rib->ll_networks[from], prefix) {
		    if (prefix_compare (prefix, network))
			break;
		}
		if (prefix == NULL) {
		    rib_close (rib);
		    return (-1);
		}
	        LL_Remove (rib->ll_networks[from], network);
		if (LL_GetCount (rib->ll_networks[from]) <= 0) {
		    LL_Destroy (rib->ll_networks[from]);
		    rib->ll_networks[from] = NULL;
		    rib->ll_networks_mask[from] = 0;
		}
	    }
	}

        RIB_RADIX_WALK (rib, route_head) {
	    assert (route_head);

	    if (a_include_b (network, route_head->prefix)) {

	      LL_Iterate (route_head->ll_route_nodes, route_node) {

    		if (BIT_TEST (route_node->flags, MRT_RTOPT_SUPPRESS))
		    continue;

		if (!BGP4_BIT_TEST (rib->redistribute_mask[from], 
				   route_node->attr->type) &&
		    BGP4_BIT_TEST (rib->ll_networks_mask[from], 
				    route_node->attr->type)) {
		if (on)
    		    MRT->proto_update_route[from] (route_head->prefix,
					           route_node->attr, NULL,
					           route_node->pref);
		else
    		    MRT->proto_update_route[from] (route_head->prefix,
					           NULL, route_node->attr, 
					           0);
		}
	      }
	    }
        } RIB_RADIX_WALK_END;
        rib_close (rib);
    return (1);
}


void
init_rib (trace_t * tr)
{
    assert (RIB == NULL);
    RIB = New_Rib (32);
    RIB->family = AF_INET;
    RIB->trace = trace_copy (tr);
    set_trace (RIB->trace, TRACE_PREPEND_STRING, "RIB", 0);

#ifdef HAVE_IPV6
    assert (RIBv6 == NULL);
    RIBv6 = New_Rib (128);
    RIBv6->family = AF_INET6;
    RIBv6->trace = trace_copy (tr);
    set_trace (RIBv6->trace, TRACE_PREPEND_STRING, "RIB6", 0);
#endif /* HAVE_IPV6 */
    MRT->rib_update_route = rib_update_route;
    MRT->rib_redistribute_request = rib_redistribute_request;
    MRT->rib_redistribute_network = rib_redistribute_network;
}


/*
 * get "head" of routing table entry for a given prefix. Create a head
 * and radix-tree entry if one does not aleady exist
 */
static route_head_t *
rib_get_route_head (rib_t * rib, prefix_t * prefix)
{
    route_head_t *route_head;
    radix_node_t *radix_node;

    assert (pthread_mutex_trylock (&rib->mutex_lock));
    assert (prefix->family == rib->family);

    radix_node = radix_search_exact (rib->radix_tree, prefix);
    if (radix_node == NULL) {
	route_head = New_Route_Head (prefix);
	radix_node = radix_lookup (rib->radix_tree, prefix);
	assert (radix_node->data == NULL);
        RADIX_DATA_SET (radix_node, route_head);
	route_head->radix_node = radix_node;
	rib->num_active_routes++;

	if (TR_TRACE & rib->trace->flags)
	  trace (TR_TRACE, rib->trace, "new route head for %s\n",
		 prefix_toax (prefix));
    }
    else {
	route_head = RADIX_DATA_GET (radix_node, route_head_t);
    }

    return (route_head);
}


/*
 * given a prefix AND type, find route node entry
 * return NULL if one does not exist
 */
static route_node_t *
rib_find_route_node (rib_t * rib,
		     prefix_t * prefix, generic_attr_t * attr)
{
    route_head_t *route_head;
    route_node_t *route_node;
    radix_node_t *radix_node;

    assert (pthread_mutex_trylock (&rib->mutex_lock));

    if ((radix_node = radix_search_exact (rib->radix_tree, prefix)) != NULL) {
	route_head = RADIX_DATA_GET (radix_node, route_head_t);
	assert (route_head);
	LL_Iterate (route_head->ll_route_nodes, route_node) {
	    if (route_node->attr->type == attr->type)
		return (route_node);
	}
    }

    return (NULL);
}


static void
rib_kernel_call_fn (prefix_t *prefix, 
		    generic_attr_t *new, u_long flags, int pref,
		    generic_attr_t *old, u_long oflags)
{
    if (MRT->kernel_update_route == NULL)
	return;

    if (new && (/* new->type == PROTO_KERNEL || */
             new->type == PROTO_CONNECTED))
	new = NULL;

    if (old && (/* old->type == PROTO_KERNEL || */
             old->type == PROTO_CONNECTED))
	old = NULL;

    if (BIT_TEST (flags, MRT_RTOPT_NOINSTALL))
	new = NULL;

    if (BIT_TEST (oflags, MRT_RTOPT_NOINSTALL))
	old = NULL;

    if (new != NULL || old != NULL)
	MRT->kernel_update_route (prefix, new, old, pref);
}


static void
rib_update_call_fn (rib_t *rib, prefix_t *prefix, 
		    generic_attr_t *new, u_long flags, int pref,
		    generic_attr_t *old, u_long oflags)
{
    int type, i;

    assert (prefix);
    if (BIT_TEST (flags, MRT_RTOPT_SUPPRESS))
	new = NULL;
    if (BIT_TEST (oflags, MRT_RTOPT_SUPPRESS))
	old = NULL;
    if (new == NULL && old == NULL)
	return;
    if (new && old) {
	if (new->type != old->type) {
	    /* kernel route may be replaced by others */
	    /* add new then withdraw old */
	    rib_update_call_fn (rib, prefix, new, flags, pref, NULL, 0);
	    rib_update_call_fn (rib, prefix, NULL, 0, 0, old, oflags);
	    return;
	}
	type = new->type;
	/* they can overwrite so that old is not required */
	old = NULL;
    }
    else if (new) {
	type = new->type;
    }
    else /* if (old) */ {
	assert (old);
	type = old->type;
    }

    for (i = PROTO_MIN; i <= PROTO_MAX; i++) {
	if (BGP4_BIT_TEST (rib->redistribute_mask[i], type)) {
            MRT->proto_update_route[i] (prefix, new, old, pref);
	    continue;
	}
	if (BGP4_BIT_TEST (rib->ll_networks_mask[i], type)) {
	    assert (rib->ll_networks[i]);
	    if (check_networks (prefix, rib->ll_networks[i]))
                MRT->proto_update_route[i] (prefix, new, old, pref);
	}
    }
}


static void 
rib_replace_active (rib_t * rib, route_head_t * route_head,
		    route_node_t * best)
{
    route_node_t *route_node;

    if (best == NULL) {
	/* find new active route */
	LL_Iterate (route_head->ll_route_nodes, route_node) {
	    if (best == NULL || route_node->pref < best->pref) {
		best = route_node;
	    }
	}
    }

    if (best == NULL) {	/* there is no other route nodes */

	assert (route_head->active);
	rib_kernel_call_fn (route_head->prefix, NULL, 0, 0, 
			    route_head->active->attr, 
			    route_head->active->flags);

	if (TR_TRACE & rib->trace->flags)
	  trace (TR_TRACE, rib->trace, "delete route head for %s\n",
		 prefix_toax (route_head->prefix));
	radix_remove (rib->radix_tree, route_head->radix_node);
	Delete_Route_Head (route_head);
	rib->num_active_routes--;
	return;
    }

    if (route_head->active == best)
	return;

    /* route update */

    if (BIT_TEST (rib->trace->flags, TR_TRACE)) {
	buffer_t *buffer = New_Buffer (0);
	route_toa_buffer (route_head->prefix, best->attr, best->pref, 
			  best->flags, buffer);
        trace (TR_TRACE, rib->trace, "active: %s\n", buffer_data (buffer));
	Delete_Buffer (buffer);
    }

    if (route_head->active &&
	    route_head->active->attr->nexthop == 
			best->attr->nexthop /* nexthop unchange */) {
	/* they are the same, so it's OK if replaced. */
	/* this is required in case active is being deleted. */
	route_head->active = best;
	return;
    }

    if (route_head->active != NULL) {
        rib_kernel_call_fn (route_head->prefix, 
			best->attr, best->flags, best->pref,
			route_head->active->attr, route_head->active->flags);
    }
    else {
        rib_kernel_call_fn (route_head->prefix, 
			best->attr, best->flags, best->pref, NULL, 0);
    }
    route_head->active = best;
}


static void 
rib_delete_route_node (rib_t * rib, route_node_t * route_node)
{
    route_head_t *route_head;

    assert (pthread_mutex_trylock (&rib->mutex_lock));

    route_head = route_node->route_head;
    LL_RemoveFn (route_head->ll_route_nodes, route_node, 0);

    if (BIT_TEST (rib->trace->flags, TR_TRACE)) {
	buffer_t *buffer = New_Buffer (0);
	route_toa_buffer (route_head->prefix, route_node->attr, 
			  route_node->pref, route_node->flags, buffer);
        trace (TR_TRACE, rib->trace, "delete: %s\n", buffer_data (buffer));
	Delete_Buffer (buffer);
    }

    if (route_node->protocol_mask == 0) {
	/* what's protocol_mask? */
    }

    if (route_head->active == route_node) {
	rib_replace_active (rib, route_head, NULL);
    }

    /* delete here */
    Delete_Route_Node (route_node);
}



static void 
rib_route_sweep (rib_t * rib, int type)
{
    route_head_t *route_head;
    route_node_t *route_node;
    LINKED_LIST *ll;

    assert (pthread_mutex_trylock (&rib->mutex_lock));

    ll = LL_Create (0);
    RIB_RADIX_WALK (rib, route_head) {
	LL_Iterate (route_head->ll_route_nodes, route_node) {
	    if (route_node->attr->type == type)
		LL_Add (ll, route_node);
	}
    } RIB_RADIX_WALK_END;

    LL_Iterate (ll, route_node) {
        rib_update_call_fn (rib, route_node->route_head->prefix, NULL, 0, 0, 
			    route_node->attr, route_node->flags);
	rib_delete_route_node (rib, route_node);
    }
    LL_Destroy (ll);
}


void 
rib_update_route (prefix_t *prefix, generic_attr_t *new,  
		  generic_attr_t *old, int pref, u_long flags)
{
    rib_t *rib;

    /* XXX hack! */
    if (prefix == NULL) {
	assert (new);
	delete_all_route_with_type (AF_INET, new->type);
#ifdef HAVE_IPV6
	delete_all_route_with_type (AF_INET6, new->type);
#endif /* HAVE_IPV6 */
	return;
    }

    if (prefix->family == AF_INET)
        rib = RIB;
#ifdef HAVE_IPV6
    else if (prefix->family == AF_INET6)
        rib = RIBv6;
#endif /* HAVE_IPV6 */
    else
        return;
    
    rib_open (rib);
    if (new) {
	rib_add_route (rib, prefix, new, pref, flags);
    }
    else {
	assert (old);
	rib_del_route (rib, prefix, old);
    }
    rib_close (rib);
}


void
delete_all_route_with_type (int family, int type)
{
    if (family == AF_INET) {
        rib_open (RIB);
        rib_route_sweep (RIB, type);
        rib_close (RIB);
    }
#ifdef HAVE_IPV6
    if (family == AF_INET6) {
        rib_open (RIBv6);
        rib_route_sweep (RIBv6, type);
        rib_close (RIBv6);
    }
#endif /* HAVE_IPV6 */
}


/*
 * create a route_node and add it to the appropriate route_head,
 * or create a route_head if not already exist. Returnthe new route_node.
 */

route_node_t *
rib_add_route (rib_t *rib, prefix_t *prefix, generic_attr_t *attr, int pref,
	       u_long flags)
{
    route_node_t *route_node;
    route_head_t *route_head;

    assert (pthread_mutex_trylock (&rib->mutex_lock));

    route_head = rib_get_route_head (rib, prefix);

    /* check to see if we already got this route from the same proto */

    LL_Iterate (route_head->ll_route_nodes, route_node) {
        if (route_node->attr->type == attr->type)
	    break;
	/* kernel route with the same nexthop will be overwritten */
        if (route_node->attr->type == PROTO_KERNEL &&
	        route_node->attr->nexthop == attr->nexthop) {
	    BIT_SET (flags, MRT_RTOPT_KERNEL);
	    break;
	}
	/* mark the route */
        if (attr->type == PROTO_KERNEL &&
	        route_node->attr->nexthop == attr->nexthop) {
	    BIT_SET (route_node->flags, MRT_RTOPT_KERNEL);
	    return (route_node);
	}
    }

    if (route_node) {
	route_node_t *new_route;

	/* if the existing route has MRT_RTOPT_KERNEL,
	   it will be inherited to the new route */
	if (BIT_TEST (route_node->flags, MRT_RTOPT_KERNEL))
	    BIT_SET (flags, MRT_RTOPT_KERNEL);

	if (pref <= route_node->pref) {
	    /* equal or better preference */

	    if (pref == route_node->pref && flags == route_node->flags &&
		    generic_attr_equal (attr, route_node->attr)) {

	      if (BIT_TEST (rib->trace->flags, TR_TRACE)) {
        	  buffer_t *buffer = New_Buffer (0);
		  route_toa_buffer (prefix, route_node->attr, route_node->pref,
			            route_node->flags, buffer);
		  trace (TR_TRACE, rib->trace, "ignore: %s (same)\n", 
			 buffer_data (buffer));
        	  Delete_Buffer (buffer);
	      }
	      return (route_node);
	    }

	    /* implicit withdarw of old route -- delete it */
	    /* on deleting, 
		do I have to notify the proto engine which supplied? */

	    if (BIT_TEST (rib->trace->flags, TR_TRACE)) {
        	buffer_t *buffer = New_Buffer (0);
		route_toa_buffer (prefix, route_node->attr, route_node->pref, 
				  route_node->flags, buffer);
	        trace (TR_TRACE, rib->trace, 
		       "update: %s (better pref %d)\n", 
		       buffer_data (buffer), pref);
        	Delete_Buffer (buffer);
	    }

    	    new_route = New_Route_Node (route_head, attr, pref, flags);
    	    LL_Add (route_head->ll_route_nodes, new_route);
	    LL_RemoveFn (route_head->ll_route_nodes, route_node, 0);

    	    rib_update_call_fn (rib, prefix, attr, flags, pref, 
				route_node->attr, route_node->flags);
	    rib_replace_active (rib, route_head, NULL);

    	    Delete_Route_Node (route_node);
	    return (route_node);
	}
	else {
	  /* same or bad .. ignore it */
	  if (BIT_TEST (rib->trace->flags, TR_TRACE)) {
	      buffer_t *buffer = New_Buffer (0);
	      route_toa_buffer (prefix, attr, pref, flags, buffer);
	      trace (TR_TRACE, rib->trace, "ignore: %s (now pref %d)\n",
		   buffer_data (buffer), route_node->pref);
              Delete_Buffer (buffer);
	   }
	   return (route_node);
	}
    }

    /* first time we've heard this in this proto */
    route_node = New_Route_Node (route_head, attr, pref, flags);
    LL_Add (route_head->ll_route_nodes, route_node);
    rib_update_call_fn (rib, prefix, attr, flags, pref, NULL, 0);

    if (BIT_TEST (rib->trace->flags, TR_TRACE)) {
        buffer_t *buffer = New_Buffer (0);
	route_toa_buffer (prefix, attr, pref, flags, buffer);
        trace (TR_TRACE, rib->trace, "add: %s\n", buffer_data (buffer));
        Delete_Buffer (buffer);
    }
    /* is this new route better than the active route ? */
    if (route_head->active == NULL ||
	route_node->pref < route_head->active->pref) {
	rib_replace_active (rib, route_head, route_node);
    }

    return (route_node);
}


/* XXX
 * attr is not used seriously, only attr->type is unsed.
 */
int
rib_del_route (rib_t * rib, prefix_t * prefix, generic_attr_t * attr)
{
    route_node_t *route_node;

    assert (pthread_mutex_trylock (&rib->mutex_lock));
    if ((route_node = rib_find_route_node (rib, prefix, attr)) == NULL)
	return (-1);
    assert (route_node);
    rib_update_call_fn (rib, prefix, NULL, 0, 0, attr, route_node->flags);
    rib_delete_route_node (rib, route_node);
    return (1);
}


void 
rib_open (rib_t * rib)
{
    if (pthread_mutex_trylock (&rib->mutex_lock) != 0) {
	trace (TR_DEBUG, rib->trace, "Going to block in rib_open\n");
	pthread_mutex_lock (&rib->mutex_lock);
    }
}


void 
rib_close (rib_t * rib)
{
    pthread_mutex_unlock (&rib->mutex_lock);
}


/* called from BGP to look up a nexthop for mutihop neighbor */
prefix_t *
rib_find_best_route (prefix_t *prefix)
{
    rib_t *rib = RIB;
    route_head_t *route_head;
    radix_node_t *radix_node;
    prefix_t *nexthop = NULL;

#ifdef HAVE_IPV6
    if (prefix->family == AF_INET6) {
	rib = RIBv6;
    }
#endif /* HAVE_IPV6 */
    rib_open (rib);
    radix_node = radix_search_best (rib->radix_tree, prefix);
    if (radix_node != NULL) {
	route_head = RADIX_DATA_GET (radix_node, route_head_t);
	assert (route_head && route_head->active);
	nexthop = route_head->active->attr->nexthop->prefix;
    }
    rib_close (rib);
    return (nexthop);
}
