/* 
 * $Id: nexthop.c,v 1.6 1999/04/29 19:02:40 masaki Exp $
 */

#include <mrt.h>


static u_int
nexthop_hash_fn (nexthop_t *nexthop, u_int size)
{
    if (prefix_is_unspecified (nexthop->prefix)) {
	int val;
	if (nexthop->interface)
	    val = nexthop->interface->index;
	else
	    val = 0;
	return (val);
    }

#ifdef HAVE_IPV6
    if (nexthop->prefix->family == AF_INET6) {
	u_int val, buff[4];
	memcpy (buff, prefix_tochar (nexthop->prefix), 16);
	val = buff[0] ^ buff[1] ^ buff[2] ^ buff[3];
	val ^= (val >> 16);
	if (prefix_is_linklocal (nexthop->prefix) && nexthop->interface)
	    val += nexthop->interface->index;
	val = val % size;
	return (val);
    }
    else
#endif /* HAVE_IPV6 */
    if (nexthop->prefix->family == AF_INET) {
	u_int val;
	u_char dest[4];
	memcpy (dest, prefix_tochar (nexthop->prefix), 4);
	val = dest[0] + dest[1] + dest[2] + dest[3];
	val = val % size;
	return (val);
    }
    else {
	assert (0);
    }
    /* NEVER REACHES */
    return (0);
}


static u_int
nexthop_lookup_fn (nexthop_t *a, nexthop_t *b)
{
    if (prefix_compare2 (a->prefix, b->prefix) == 0) {
        if (prefix_is_unspecified (a->prefix))
	    return (a->interface == b->interface);
#ifdef HAVE_IPV6
	else if (a->prefix->family == AF_INET6 &&
	        prefix_is_linklocal (a->prefix))
	    return (a->interface == b->interface);
#endif /* HAVE_IPV6 */
	else
	    return (1);
    }
    return (0);
}


nexthop_t *
ref_nexthop (nexthop_t *nexthop)
{
    if (nexthop) {
        pthread_mutex_lock (&nexthop->mutex_lock);
        assert (nexthop->ref_count > 0);
        nexthop->ref_count++;
        pthread_mutex_unlock (&nexthop->mutex_lock);
    }
    return (nexthop);
}


#define NEXTHOP_HASH_SIZE 1023
/* 
 * find nexthop or create a new one if it does not exit
 */
nexthop_t *
add_nexthop (prefix_t *prefix, interface_t *interface)
{
    nexthop_t nh, *nexthop;
    mrt_hash_table_t *hash = &MRT->hash_table;

    if (prefix == NULL)
	return (NULL);

#ifdef HAVE_IPV6
    if (prefix->family == AF_INET6)
	hash = &MRT->hash_table6;
#endif /* HAVE_IPV6 */

    pthread_mutex_lock (&hash->mutex_lock);

    if (hash->table == NULL) {
        hash->table = HASH_Create (NEXTHOP_HASH_SIZE, 
		    HASH_EmbeddedKey, True,
		    HASH_KeyOffset, 0,
                    HASH_LookupFunction, nexthop_lookup_fn,
                    HASH_HashFunction, nexthop_hash_fn, NULL);
    }

    nh.prefix = prefix;
    nh.interface = interface;
    nh.ref_count = 1;
    if ((nexthop = HASH_Lookup (hash->table, &nh))) {
	nexthop = ref_nexthop (nexthop);
	if (nexthop->interface != interface && interface != NULL) {
	    /* this happens only in case of global addresses */
	    /* can not reset the interface with NULL */
	    if (nexthop->interface != NULL) {
                trace ((nexthop->interface)? TR_WARN: TR_STATE, MRT->trace, 
		       "add_nexthop: %s on %s -> %s (interface change)\n",
                       prefix_toa (nexthop->prefix),
                       (nexthop->interface)?nexthop->interface->name:"?",
                       interface->name);
	    }
	    nexthop->interface = interface;
	}
        pthread_mutex_unlock (&hash->mutex_lock);
	return (nexthop);
    }

/*
    assert (prefix_is_unspecified (prefix) == 0 || interface != NULL);
    assert (prefix_is_linklocal (prefix) == 0 || interface != NULL);
*/

    nexthop = New (nexthop_t);
    nexthop->prefix = Ref_Prefix (prefix);
    nexthop->interface = interface;
    nexthop->ref_count = 1;
    pthread_mutex_init (&nexthop->mutex_lock, NULL);
    HASH_Insert (hash->table, nexthop);
    pthread_mutex_unlock (&hash->mutex_lock);
    trace (TR_STATE, MRT->trace, "add_nexthop: %s on %s\n",
           prefix_toa (nexthop->prefix), (interface)?interface->name:"?");
    return (nexthop);
}


#ifdef notdef
nexthop_t *
find_nexthop (prefix_t *prefix, interface_t *interface)
{
    nexthop_t nh, *nexthop = &nh;
    mrt_hash_table_t *hash = &MRT->hash_table;

    if (prefix == NULL)
	return (NULL);

#ifdef HAVE_IPV6
    if (prefix->family == AF_INET6)
	hash = &MRT->hash_table6;
#endif /* HAVE_IPV6 */

    if (hash->table == NULL)
	return (NULL);

    pthread_mutex_lock (&hash->mutex_lock);

    nexthop->prefix = prefix;
    nexthop->interface = interface;

    if ((nexthop = HASH_Lookup (hash->table, nexthop))) {
        pthread_mutex_unlock (&hash->mutex_lock);
	return (nexthop);
    }
    return (NULL);
}
#endif


void
deref_nexthop (nexthop_t *nexthop)
{
    if (nexthop == NULL)
	return;
    pthread_mutex_lock (&nexthop->mutex_lock);
    assert (nexthop->ref_count > 0);
    if (nexthop->ref_count <= 1) {
	mrt_hash_table_t *hash = &MRT->hash_table;
#ifdef HAVE_IPV6
	if (nexthop->prefix->family == AF_INET6)
	    hash = &MRT->hash_table6;
#endif /* HAVE_IPV6 */
        pthread_mutex_lock (&hash->mutex_lock);
        /* someone may be searching in the table 
	       and found this at the same time */
	if (nexthop->ref_count <= 1) {
	    HASH_Remove (hash->table, nexthop);
            pthread_mutex_unlock (&hash->mutex_lock);
	    pthread_mutex_destroy (&nexthop->mutex_lock);
	    Deref_Prefix (nexthop->prefix);
	    Delete (nexthop);
	    return;
	}
        pthread_mutex_unlock (&hash->mutex_lock);
    }
    nexthop->ref_count--;
    pthread_mutex_unlock (&nexthop->mutex_lock);
    return;
}


gateway_t *
find_gateway (prefix_t *prefix)
{
    nexthop_t *nexthop, *found = NULL;
    mrt_hash_table_t *hash = &MRT->hash_table;

    if (prefix == NULL)
	return (NULL);

#ifdef HAVE_IPV6
    if (prefix->family == AF_INET6)
	hash = &MRT->hash_table6;
#endif /* HAVE_IPV6 */

    if (hash->table == NULL)
	return (NULL);

    pthread_mutex_lock (&hash->mutex_lock);
    HASH_Iterate (hash->table, nexthop) {
	if (prefix_compare2 (nexthop->prefix, prefix) == 0) {
	    if (found) {
    		trace (TR_WARN, MRT->trace, 
		      "duplicated nexthop detected: %s on %s and %s\n", 
		       prefix_toa (prefix),
		      (nexthop->interface)?nexthop->interface->name:"?",
		      (found->interface)?found->interface->name:"?");
    		pthread_mutex_unlock (&hash->mutex_lock);
		return (NULL);
	    }
	    found = nexthop;
	}
    }
    pthread_mutex_unlock (&hash->mutex_lock);
    return (found);
}
