/*
 * $Id: common.c,v 1.15 1999/04/29 19:02:21 masaki Exp $
 */

#include <mrt.h>
#include <api6.h>
#include <interface.h>
#include <ctype.h>


interface_t *
new_interface (char *name, u_long flags, int mtu, int index)
{
    interface_t *interface;
    char tmpx[MAXLINE];

    assert (INTERFACE_MASTER);

    interface = find_interface_byname (name);
    if (interface == NULL) {
        INTERFACE_MASTER->number++;
        if (index <= 0)
	    index = INTERFACE_MASTER->number;
        if (index > MAX_INTERFACES) {
            trace (TR_FATAL, INTERFACE_MASTER->trace,
	       "Too many interfaces %d (should be <= %d)\n", 
	       INTERFACE_MASTER->number, MAX_INTERFACES);
	    return (NULL);
        }

        interface = New (interface_t);
        interface->ll_addr = LL_Create (0);
#if 0
        memset (interface->dlist_in, 0, sizeof (interface->dlist_in));
        memset (interface->dlist_out, 0, sizeof (interface->dlist_out));
        interface->metric_in = 1;
        interface->metric_out = 0;
        interface->default_pref = -1;
#ifdef HAVE_IPV6
        interface->default_pref6 = -1;
#endif /* HAVE_IPV6 */
#endif
        strcpy (interface->name, name);

        interface->index = index;
        INTERFACE_MASTER->index2if[index] = interface;
	pthread_mutex_lock (&INTERFACE_MASTER->mutex_lock);
        LL_Add (INTERFACE_MASTER->ll_interfaces, interface);
	pthread_mutex_unlock (&INTERFACE_MASTER->mutex_lock);
    }
    interface->flags = flags;
    interface->mtu = mtu;

    if (INTERFACE_MASTER->max_mtu < mtu)
	INTERFACE_MASTER->max_mtu = mtu;

    trace (TR_INFO, INTERFACE_MASTER->trace,
	   "Interface %s flags %s mtu %d index %d\n",
	   name, print_iflags (tmpx, sizeof (tmpx), flags), mtu, index);

    return (interface);
}


ll_addr_t *
New_Addr (int family, void *addr, int bitlen, void *broadcast)
{

    ll_addr_t *if_addr;

    if_addr = New (ll_addr_t);
    assert (addr);
    if_addr->prefix = New_Prefix (family, addr, bitlen);
    if (broadcast)
	if_addr->broadcast = New_Prefix (family, broadcast, bitlen);
    else
	if_addr->broadcast = NULL;

    return (if_addr);
}


static int 
compare_interface (interface_t *a, interface_t *b)
{
	if (BIT_TEST (a->flags, IFF_LOOPBACK) !=
	         BIT_TEST (b->flags, IFF_LOOPBACK)) {
	    return (BIT_TEST (a->flags, IFF_LOOPBACK)? 1: -1);
	}
	if (BIT_TEST (a->flags, IFF_TUNNEL) !=
	         BIT_TEST (b->flags, IFF_TUNNEL)) {
	    return (BIT_TEST (a->flags, IFF_TUNNEL)? 1: -1);
	}
	if (BIT_TEST (a->flags, IFF_UP) !=
	         BIT_TEST (b->flags, IFF_UP)) {
	    return (BIT_TEST (a->flags, IFF_UP)? -1: 1);
	}

	if (BIT_TEST (a->flags, IFF_POINTOPOINT) !=
	         BIT_TEST (b->flags, IFF_POINTOPOINT)) {
	    return (BIT_TEST (a->flags, IFF_POINTOPOINT)? 1: -1);
	}

    /* choose the largest ip number */
    if (a->primary && b->primary) {
	if (ntohl (prefix_tolong (b->primary->prefix)) !=
	    ntohl (prefix_tolong (a->primary->prefix))) {
	    return ((ntohl (prefix_tolong (a->primary->prefix)) >
		     ntohl (prefix_tolong (b->primary->prefix)))? -1: 1);
	}
    }
    return (0);
}


static void
update_default_interface (void)
{
    interface_t *interface;
#ifdef HAVE_IPV6
    interface_t *candidate = NULL;
#endif /* HAVE_IPV6 */

    pthread_mutex_lock (&INTERFACE_MASTER->mutex_lock);
    LL_SortFn (INTERFACE_MASTER->ll_interfaces, 
	       (LL_CompareProc) compare_interface);

    LL_Iterate (INTERFACE_MASTER->ll_interfaces, interface) {
	if (interface->primary)
	    break;
    }
    INTERFACE_MASTER->default_interface = interface;
    if (interface) {
        trace (TR_INFO, INTERFACE_MASTER->trace,
	           "DEFAULT Interface %s\n", interface->name);
	MRT->default_id = prefix_tolong (interface->primary->prefix);
    }
#ifdef HAVE_IPV6
    LL_Iterate (INTERFACE_MASTER->ll_interfaces, interface) {
	if (interface->primary6) {
	    if (candidate == NULL) {
		candidate = interface;
	    }
	    else if (!IN6_IS_ADDR_UC_GLOBAL 
		(prefix_toaddr6 (candidate->primary6->prefix)) &&
	      IN6_IS_ADDR_UC_GLOBAL 
		(prefix_toaddr6 (interface->primary6->prefix))) {
	        candidate = interface;
	    }
	}
    }
    INTERFACE_MASTER->default_interface6 = candidate;
    if (candidate)
        trace (TR_INFO, INTERFACE_MASTER->trace,
	           "DEFAULT V6 Interface %s\n", candidate->name);
#endif /* HAVE_IPV6*/
    pthread_mutex_unlock (&INTERFACE_MASTER->mutex_lock);
}


/*
 * I know I need to introduce a mutex lock since it may be called from 
 * kernel asynchronously
 */
interface_t *
update_addr_of_interface (int cmd,
			  interface_t * interface, int family,
		          void *addr, int bitlen, void *broadcast)
{
    ll_addr_t *if_addr;
    char ifa[256];
    char tmp6[64];
    char tmp7[64];

    assert (interface);
    assert (INTERFACE_MASTER);

#ifdef HAVE_IPV6
    assert (family == AF_INET || family == AF_INET6);
#else
    assert (family == AF_INET);
#endif /* HAVE_IPV6 */

    if (broadcast)
        sprintf (ifa, "%s %s %s/%d %s %s", interface->name,
	         (family == AF_INET)? "inet": "inet6",
	         inet_ntop (family, addr, tmp6, sizeof tmp6), bitlen,
	         BIT_TEST (interface->flags, IFF_POINTOPOINT) ?
	         "dest" : "broadcast",
	         inet_ntop (family, broadcast, tmp7, sizeof tmp7));
    else
        sprintf (ifa, "%s %s %s/%d", interface->name,
	         (family == AF_INET)? "inet": "inet6",
	         inet_ntop (family, addr, tmp6, sizeof tmp6), bitlen);

#ifdef HAVE_IPV6
    /* we don't accept these ones */
    if (family == AF_INET6) {

        /* IN6_IS_ADDR_LOOPBACK is required since IN6_IS_ADDR_V4COMPAT
	   on some implementations includes ::1 */
	if (!IN6_IS_ADDR_LOOPBACK ((struct in6_addr *) addr) && (
	       IN6_IS_ADDR_V4MAPPED ((struct in6_addr *) addr) ||
	       IN6_IS_ADDR_V4COMPAT ((struct in6_addr *) addr))) {
	    trace (TR_TRACE, INTERFACE_MASTER->trace,
	           "Interface %s skipped\n", ifa);
	    return (NULL);
	}
	if (broadcast && !IN6_IS_ADDR_LOOPBACK ((struct in6_addr *) addr) && (
	      IN6_IS_ADDR_V4MAPPED ((struct in6_addr *) broadcast) ||
	      IN6_IS_ADDR_V4COMPAT ((struct in6_addr *) broadcast))) {
	    trace (TR_TRACE, INTERFACE_MASTER->trace,
	           "Interface %s skipped\n", ifa);
	    return (NULL);
	}
    }
#endif

    LL_Iterate (interface->ll_addr, if_addr) {
	int plen = (family == AF_INET)? 4: 16;
	if (if_addr->prefix->family != family)
	    continue;
	if (memcmp (prefix_tochar (if_addr->prefix), addr, plen) == 0 &&
	        if_addr->prefix->bitlen == bitlen)
	    break;
    }

    if (cmd == 'D') {
	if (if_addr == NULL) {
	    trace (TR_WARN, INTERFACE_MASTER->trace,
	           "Interface %s not found for deletion\n", ifa);
	    return (NULL);
	}
        LL_Remove (interface->ll_addr, if_addr);
	if (if_addr->prefix)
	    Deref_Prefix (if_addr->prefix);
	if (if_addr->broadcast)
	    Deref_Prefix (if_addr->broadcast);
	Delete (if_addr);
	trace (TR_WARN, INTERFACE_MASTER->trace,
	       "Interface %s deleted\n", ifa);
	return (NULL);
    }
    assert (cmd == 'A');

    if (if_addr == NULL) {
        if_addr = New_Addr (family, addr, bitlen, broadcast);
        LL_Add (interface->ll_addr, if_addr);
    }

    if (family == AF_INET && interface->primary == NULL) {
	interface->primary = if_addr;
    }
#ifdef HAVE_IPV6
    else if (family == AF_INET6) {
	if (IN6_IS_ADDR_V4MAPPED ((struct in6_addr *) addr) &&
	    IN6_IS_ADDR_V4COMPAT ((struct in6_addr *) addr)) {
	    return (NULL);
	}

	if (interface->primary6 == NULL || (!IN6_IS_ADDR_UC_GLOBAL (
			    prefix_toaddr6 (interface->primary6->prefix)) &&
		     IN6_IS_ADDR_UC_GLOBAL ((struct in6_addr *) addr))) {
	    interface->primary6 = if_addr;
	}
	if (interface->link_local == NULL &&
	    IN6_IS_ADDR_LINKLOCAL ((struct in6_addr *) addr)) {
	    interface->link_local = if_addr;
	}
    }
#endif /* HAVE_IPV6 */

    trace (TR_INFO, INTERFACE_MASTER->trace, "Interface %s\n", ifa);
    return (interface);
}


interface_t *
add_addr_to_interface (interface_t * interface, int family,
                       void *addr, int len, void *broadcast)
{
    return (update_addr_of_interface ('A', interface, family, addr, len,
				      broadcast));
}


int
init_interfaces (trace_t * ltrace)
{
    assert (INTERFACE_MASTER == NULL);
    INTERFACE_MASTER = New (interface_master_t);
    INTERFACE_MASTER->ll_interfaces = LL_Create (0);
    INTERFACE_MASTER->trace = trace_copy (ltrace);
    set_trace (INTERFACE_MASTER->trace, TRACE_PREPEND_STRING, "IF", 0);
    INTERFACE_MASTER->number = 0;
    INTERFACE_MASTER->max_mtu = 576;
    INTERFACE_MASTER->default_interface = NULL;
    pthread_mutex_init (&INTERFACE_MASTER->mutex_lock, NULL);
    if ((INTERFACE_MASTER->sockfd = socket (AF_INET, SOCK_DGRAM, 0)) < 0) {
        trace (TR_FATAL, INTERFACE_MASTER->trace, "socket for AF_INET (%m)\n");
    }
#ifdef HAVE_IPV6
    if ((INTERFACE_MASTER->sockfd6 = socket (AF_INET6, SOCK_DGRAM, 0)) < 0) {
        trace (TR_WARN, INTERFACE_MASTER->trace, "socket for AF_INET6 (%m)\n");
    }
#endif /* HAVE_IPV6 */
    read_interfaces ();
    /* it's better to update default interface 
	whenever interfaces change, though */
    update_default_interface ();

    if (INTERFACE_MASTER->default_interface == NULL) {
        trace (TR_FATAL, INTERFACE_MASTER->trace, 
	       "Interface initialization Failed, Aborting\n");
	return (-1);
    }
    return (1);
}


/* 
 * Given a prefix, find an interface that prefix is on
 */
interface_t *
find_interface (prefix_t * prefix)
{
    return (find_interface_flags (prefix, 0));
}


/* 
 * Given a prefix, find an interface with the specified flags 
 * that prefix is on if flags == 0, then see everything
 */
interface_t *
find_interface_flags (prefix_t * prefix, u_long flags)
{
    interface_t *interface;
    ll_addr_t *ll_addr;
    u_char *addr, *dest;
    int llocal = 0;

    /* bgpsim doesn't initialize ifs */
    if (INTERFACE_MASTER == NULL)
	return (NULL);

    addr = prefix_touchar (prefix);
#ifdef HAVE_IPV6
    if (prefix->family == AF_INET6 &&
            IN6_IS_ADDR_LINKLOCAL ((struct in6_addr *) addr)) {
	llocal++;
    }
#endif /* HAVE_IPV6 */
    pthread_mutex_lock (&INTERFACE_MASTER->mutex_lock);
    LL_Iterate (INTERFACE_MASTER->ll_interfaces, interface) {

	if (flags && !BIT_TEST (interface->flags, flags))
	    continue;

	if (interface->ll_addr) {
	    LL_Iterate (interface->ll_addr, ll_addr) {
		if (ll_addr->prefix == NULL)
		    continue;
		if (ll_addr->prefix->family != prefix->family)
		    continue;
		if (!llocal && ll_addr->prefix->bitlen > prefix->bitlen)
		    continue;

		dest = prefix_touchar (ll_addr->prefix);
		if (BIT_TEST (interface->flags, IFF_POINTOPOINT) &&
		    ll_addr->broadcast) {
		    /* Linux's sit is p-to-p but doesn't have dest addr */
		    dest = prefix_touchar (ll_addr->broadcast);
		}

#if 0
		printf ("name = %s ", interface->name);
		if (ll_addr->prefix->family == AF_INET6) {
		    printf ("addr = %s ", inet_ntop (AF_INET6, addr, tmp6, sizeof tmp6));

		    printf ("dest = %s ", inet_ntop (AF_INET6, dest, tmp6, sizeof tmp6));
		}
		else {
		    printf ("addr = %s ", inet_ntop (AF_INET, addr, tmp6, sizeof tmp6));
		    printf ("dest = %s ", inet_ntop (AF_INET, dest, tmp6, sizeof tmp6));
		}
		printf ("bitlen = %d\n", ll_addr->prefix->bitlen);
#endif
		if (addr && dest && comp_with_mask (addr, dest,
				  (llocal) ? 128 : ll_addr->prefix->bitlen)) {
    		    pthread_mutex_unlock (&INTERFACE_MASTER->mutex_lock);
		    return (interface);
		}
	    }
	}
    }

    pthread_mutex_unlock (&INTERFACE_MASTER->mutex_lock);
    return (NULL);
}


/*
 * Given a prefix, find interfaces on that prefix's network
 *   matches exactly or includes
 */
LINKED_LIST *
find_network (prefix_t * prefix)
{
    interface_t *interface;
    ll_addr_t *ll_addr;
    u_char *addr, *dest;
    LINKED_LIST *ll = NULL;

    if (INTERFACE_MASTER == NULL) return (NULL);

    addr = prefix_touchar (prefix);
    
    pthread_mutex_lock (&INTERFACE_MASTER->mutex_lock);

    LL_Iterate (INTERFACE_MASTER->ll_interfaces, interface) {
#ifdef notdef
	if (BIT_TEST (interface->flags, IFF_LOOPBACK))
	    continue;
#endif

	if (interface->ll_addr) {
	    LL_Iterate (interface->ll_addr, ll_addr) {
		if (ll_addr->prefix == NULL)
		    continue;
		if (ll_addr->prefix->family != prefix->family)
		    continue;
		if (ll_addr->prefix->bitlen < prefix->bitlen)
		    continue;

		dest = prefix_touchar (ll_addr->prefix);
		if (BIT_TEST (interface->flags, IFF_POINTOPOINT)) {
		    if (ll_addr->broadcast == NULL)
			continue;
		    dest = prefix_touchar (ll_addr->broadcast);
		}

#if 0
		printf ("name = %s ", interface->name);
		if (ll_addr->prefix->family == AF_INET6) {
		    printf ("addr = %s ", inet_ntop (AF_INET6, addr, NULL, 0));
		    printf ("dest = %s ", inet_ntop (AF_INET6, dest, NULL, 0));
		}
		else {
		    printf ("addr = %s ", inet_ntop (AF_INET, addr, NULL, 0));
		    printf ("dest = %s ", inet_ntop (AF_INET, dest, NULL, 0));
		}
		printf ("bitlen = %d\n", prefix->bitlen);
#endif
		if (addr && dest &&
		    comp_with_mask (addr, dest, prefix->bitlen)) {
		    if (ll == NULL)
			ll = LL_Create (0);
		    LL_Add (ll, interface);
		    break;
		}
	    }
	}
    }

    pthread_mutex_unlock (&INTERFACE_MASTER->mutex_lock);
    return (ll);
}


/* 
 * Given a name, find an interface that has the name
 */
interface_t *
find_interface_byname (char *name)
{
    interface_t *interface;

    if (INTERFACE_MASTER == NULL) return (NULL);

    pthread_mutex_lock (&INTERFACE_MASTER->mutex_lock);
    LL_Iterate (INTERFACE_MASTER->ll_interfaces, interface) {
	if (strcasecmp (name, interface->name) == 0) {
    	    pthread_mutex_unlock (&INTERFACE_MASTER->mutex_lock);
	    return (interface);
	}
    }
    pthread_mutex_unlock (&INTERFACE_MASTER->mutex_lock);
    return (NULL);
}


/* if a digit doesn't follow the name, all interfaces starting with the name
   will be returned */
LINKED_LIST *
find_interface_byname_all (char *name)
{
    LINKED_LIST *ll = NULL;
    int all = 0;
    int len = strlen (name);
    interface_t *interface;

    if (INTERFACE_MASTER == NULL) return (ll);

    if (!isdigit (name[len - 1]))
	all++;

    pthread_mutex_lock (&INTERFACE_MASTER->mutex_lock);

    LL_Iterate (INTERFACE_MASTER->ll_interfaces, interface) {
	int match = 0;

	if (all == 0) {
	    if (strcasecmp (name, interface->name) == 0)
	        match++;
	}
	else {
	    if (strncasecmp (name, interface->name, len) == 0 &&
		   (strlen (interface->name) == len ||
		    isdigit (interface->name [len])))
	        match++;
	}
	if (match) {
	    if (ll == NULL)
		ll = LL_Create (0);
	    LL_Add (ll, interface);
	}
    }
    pthread_mutex_unlock (&INTERFACE_MASTER->mutex_lock);
    return (ll);
}


/* 
 * if it belongs to a local interface, its pointer returned
 * this is used to detect when we have received a broadcast packet from 
 * ourselves in rip, and ripng
 */
interface_t *
find_interface_local (prefix_t * prefix)
{
    interface_t *interface;
    ll_addr_t *ll_addr;
    int family = prefix->family;
    u_char *addr = prefix_touchar (prefix);

    if (INTERFACE_MASTER == NULL) return (NULL);

#ifndef HAVE_IPV6
    assert (prefix->family == AF_INET);
#else
    assert (prefix->family == AF_INET || prefix->family == AF_INET6);
#endif /* HAVE_IPV6 */

    pthread_mutex_lock (&INTERFACE_MASTER->mutex_lock);
    LL_Iterate (INTERFACE_MASTER->ll_interfaces, interface) {
	if (interface->ll_addr) {
	    LL_Iterate (interface->ll_addr, ll_addr) {
		if (ll_addr->prefix == NULL)
		    continue;
		if (ll_addr->prefix->family != family)
		    continue;

		/*
		 * The local side addresses are checked even in case of p-to-p
		 */
		addr = prefix_touchar (ll_addr->prefix);
		if (memcmp (prefix_touchar (prefix), addr, 
			(family == AF_INET) ? 4 : 16) == 0) {
    		    pthread_mutex_unlock (&INTERFACE_MASTER->mutex_lock);
		    return (interface);
		}
	    }
	}
    }
    pthread_mutex_unlock (&INTERFACE_MASTER->mutex_lock);
    return (NULL);
}


int
is_prefix_on (prefix_t *prefix, interface_t *interface)
{
    ll_addr_t *ll_addr;
    u_char *addr, *dest;
    int bitlen;

    assert (prefix);
    assert (interface);

    if (interface->ll_addr == NULL)
	return (0);

    addr = prefix_touchar (prefix);

    LL_Iterate (interface->ll_addr, ll_addr) {
	if (ll_addr->prefix == NULL)
	    continue;
	if (ll_addr->prefix->family != prefix->family)
	    continue;
/*
	if (ll_addr->prefix->bitlen > prefix->bitlen)
	    continue;
*/
	dest = prefix_touchar (ll_addr->prefix);
	bitlen = ll_addr->prefix->bitlen;

	if (BIT_TEST (interface->flags, IFF_POINTOPOINT)) {
	    /* there is no destionation addr for sit on Linux */
	    if (ll_addr->broadcast == NULL)
		continue;
	    dest = prefix_touchar (ll_addr->broadcast);
	    bitlen = 32;
#ifdef HAVE_IPV6
	    if (ll_addr->broadcast->family == AF_INET6)
	        bitlen = 128;
#endif /* HAVE_IPV6 */
	}

	if (comp_with_mask (addr, dest, bitlen))
	    return (1);
    }
    return (0);
}


/* 
 * compatible purpose
 */
interface_t *
local_interface (int family, void *cp)
{
    prefix_t ptmp, *prefix = &ptmp;

    if (INTERFACE_MASTER == NULL) return (NULL);

    if ((prefix->family = family) == AF_INET) {
	prefix->bitlen = 32;
	memcpy (&prefix->add.sin, cp, 4);
    }
#ifdef HAVE_IPV6
    else if (family == AF_INET6) {
	prefix->bitlen = 128;
	memcpy (&prefix->add.sin6, cp, 16);
    }
#endif /* HAVE_IPV6 */

    return (find_interface_local (prefix));
}


typedef struct {
    u_int bit;
    char *name;
} bits;
static bits iflags[] =
{
    {IFF_UP, "UP"},
    {IFF_BROADCAST, "BROADCAST"},
    {IFF_DEBUG, "DEBUG"},
    {IFF_LOOPBACK, "LOOPBACK"},
    {IFF_POINTOPOINT, "POINTOPOINT"},
#ifdef  IFF_NOTRAILERS
    {IFF_NOTRAILERS, "NOTRAILERS"},
#endif				/* IFF_NOTRAILERS */
    {IFF_RUNNING, "RUNNING"},
#ifdef  IFF_NOARP
    {IFF_NOARP, "NOARP"},
#endif				/* IFF_NOARP */
    {IFF_PROMISC, "PROMISC"},
    {IFF_ALLMULTI, "ALLMULTI"},
#ifdef  IFF_OACTIVE
    {IFF_OACTIVE, "OACTIVE"},
#endif				/* IFF_OACTIVE */
#ifdef  IFF_SIMPLEX
    {IFF_SIMPLEX, "SIMPLEX"},
#endif				/* IFF_SIMPLEX */
#ifdef  IFF_LINK0
    {IFF_LINK0, "LINK0"},
    {IFF_LINK1, "LINK1"},
    {IFF_LINK2, "LINK2"},
#endif				/* IFF_LINK0 */
#ifdef  IFF_MULTICAST
    {IFF_MULTICAST, "MULTICAST"},
#endif				/* IFF_MULTICAST */
#ifdef  IFF_USERDEF
    {IFF_USERDEF, "USERDEF"},
#endif				/* IFF_USERDEF */
#ifdef  IFF_TUNNEL
    {IFF_TUNNEL, "TUNNEL"},
#endif				/* IFF_TUNNEL */
    {0, NULL}
};


char *
print_iflags (char *tmpx, int len, u_long flags)
{
    int i, l, n = 0;

    assert (len >= 3);
    tmpx[n++] = '<';
    for (i = 0; iflags[i].bit; i++) {
	if (!BIT_TEST (flags, iflags[i].bit))
	    continue;
	if (n > 1)
	    tmpx[n++] = ',';
	l = strlen (iflags[i].name);
	if (n + l + 1 > len - 1)
	    break;
	strcpy (tmpx + n, iflags[i].name);
	n += l;
    }
    tmpx[n++] = '>';
    tmpx[n] = '\0';
    return (tmpx);
}


int 
mask2len (void *vmask, int bytes)
{
    int i, j, bitlen = 0;
    u_char *mask = vmask;

    if (bytes == 0 || mask == NULL)
	return (bitlen);
    for (i = 0; i < bytes; i++, bitlen += 8) {
	if (mask[i] != 0xff)
	    break;
    }
    if (i != bytes && mask[i]) {
	for (j = 7; j > 0; j--, bitlen++) {
	    if ((mask[i] & (1 << j)) == 0)
		break;
	}
    }
    /* this doesn't check if the rest of bits are all 0 or not */
    /* test in another place if contiguous or not */

#if 0
    if (bitlen == 0)
	bitlen = bytes * 8;
#endif
    return (bitlen);
}


u_char *
len2mask (int bitlen, void *vmask, int bytes)
{
    int i;
    u_char *mask = vmask;

    assert (bytes * 8 >= bitlen);
    if (bitlen == 0) {
	memset (mask, 0, bytes);
	return (mask);
    }
    if (mask == NULL || bytes == 0)
	return (NULL);
    for (i = 0; i < (bitlen / 8); i++)
	mask[i] = 0xff;
    if (bitlen < (bytes * 8))
	mask[i++] = 0xff << (8 - (bitlen & 7));
    for (; i < bytes; i++)
	mask[i] = 0;
    return (mask);
}


interface_t *
find_interface_byindex (int index)
{
    if (INTERFACE_MASTER == NULL) return (NULL);

#ifdef notdef
    LL_Iterate (INTERFACE_MASTER->ll_interfaces, interface) {
	if (interface->index == index)
	    return (interface);
    }
    return (NULL);
#else
    if (index < 0 || index >= MAX_INTERFACES)
	return (NULL);
    return (INTERFACE_MASTER->index2if[index]);
#endif
}


/* with proto and add/delete */
int 
update_kernel_route (int cmd,
		     int family, void *dest, void *nhop, int masklen, int index,
		     int proto)
{
    prefix_t *prefix, *nexthop;
    int len = 32;
    interface_t *interface = NULL;
    generic_attr_t *attr;
    u_long flags = MRT_RTOPT_KERNEL;

#ifdef HAVE_IPV6
    assert (family == AF_INET || family == AF_INET6);
#else
    assert (family == AF_INET);
#endif /* HAVE_IPV6 */

    if (family == AF_INET) {
	if (*(u_long *) nhop == INADDR_ANY) {
	    /* proto = PROTO_CONNECTED; */
	    /* proto = PROTO_KERNEL; */
	    /* return (0); */
	}
    }
#ifdef HAVE_IPV6
    else {
        struct in6_addr zero6;

	memset (&zero6, 0, sizeof (zero6));
	/* linux way of direct connected route */
	if (memcmp (nhop, &zero6, 16) == 0) {
	    /* proto = PROTO_CONNECTED; */
	    /* proto = PROTO_KERNEL; */
	    /* return (0); */
	}
	/*
	 * I'm not sure but for now these prefixs are ignored
	 */
	if (((memcmp (dest, &zero6, 16) != 0 || masklen > 0) &&
	        !IN6_IS_ADDR_LOOPBACK ((struct in6_addr *) dest) &&
	        IN6_IS_ADDR_V4COMPAT ((struct in6_addr *) dest)) ||
	    IN6_IS_ADDR_LINKLOCAL ((struct in6_addr *) dest) ||
	    IN6_IS_ADDR_MULTICAST ((struct in6_addr *) dest))
	    return (0);
	len = 128;
    }
#endif /* HAVE_IPV6 */

    if (index > 0) {
	interface = find_interface_byindex (index);
    }

    nexthop = New_Prefix (family, nhop, len);
    prefix = New_Prefix (family, dest, masklen);
    trace (TR_INFO, MRT->trace, "KERNEL READ(%c) %p %a index %d proto %s\n",
	   cmd, prefix, nexthop, index, proto2string (proto));
    assert (proto == PROTO_KERNEL || proto == PROTO_STATIC
				  || proto == PROTO_CONNECTED); /* XXX */
    attr = New_Generic_Attr (proto);
    attr->nexthop = add_nexthop (nexthop, interface);
    attr->gateway = add_gateway (nexthop, 0, interface);

    /* don't announce a route for loopback */
    if (prefix_is_loopback (prefix))
	flags |= MRT_RTOPT_SUPPRESS;

    if (MRT->rib_update_route) {
	assert (cmd == 'A' || cmd == 'D');
	if (cmd == 'A')
            MRT->rib_update_route (prefix, attr, NULL, KERNEL_PREF, 
				   flags);
	else if (cmd == 'D')
            MRT->rib_update_route (prefix, NULL, attr, KERNEL_PREF, 
				   flags);
    }
    Deref_Generic_Attr (attr);
    Deref_Prefix (prefix);
    Deref_Prefix (nexthop);
    return (1);
}


int 
add_kernel_route (int family, void *dest, void *nhop, int masklen, int index)
{
    return (update_kernel_route ('A',
			         family, dest, nhop, masklen, index, 
			         PROTO_KERNEL));
}

static mtimer_t *timeout;

static void
kernel_route_timeout (void)
{
    generic_attr_t *attr;
    attr = New_Generic_Attr (PROTO_KERNEL);
    /* special convention to delete all routes with the type */
    if (MRT->rib_update_route)
        MRT->rib_update_route (NULL, attr, NULL, 0, 0);
    Deref_Generic_Attr (attr);
}


void
kernel_read_rt_table (int seconds)
{
    if (MRT->rib_update_route) {
	sys_kernel_read_rt_table ();
        if (seconds > 0) {
            timeout = New_Timer2 ("kernel routes timeout timer",
			          seconds, TIMER_AUTO_DELETE, NULL,
			          kernel_route_timeout, 0);
            Timer_Turn_ON (timeout);
	}

    }
}


/*
 * add all of the interface routes to the appropriate ribs
 */
void
add_interfaces_to_rib ()
{
    interface_t *interface;
    ll_addr_t *ll_addr;
    prefix_t *network, *nexthop, *nexthop4;
    generic_attr_t *attr;
#ifdef HAVE_IPV6
    prefix_t *nexthop6;

    /* ipv6 dummy nexthop for direct connected route */
    nexthop6 = ascii2prefix (AF_INET6, "::/128");
#endif /* HAVE_IPV6 */
    /* ipv4 dummy nexthop for direct connected route */
    nexthop4 = ascii2prefix (AF_INET, "0.0.0.0/32");

    LL_Iterate (INTERFACE_MASTER->ll_interfaces, interface) {
	u_long flags = 0;

	if (interface->ll_addr == NULL)
	    continue;
	if (!BIT_TEST (interface->flags, IFF_UP))
	    continue;
	if (BIT_TEST (interface->flags, IFF_LOOPBACK))
	    flags |= MRT_RTOPT_SUPPRESS;
	    /* continue; */

	LL_Iterate (interface->ll_addr, ll_addr) {
	    prefix_t *pointopoint = NULL;

	    if (ll_addr->prefix == NULL)
		continue;

	    network = ll_addr->prefix;
	    if (BIT_TEST (interface->flags, IFF_POINTOPOINT))
		pointopoint = ll_addr->broadcast;

	    if (network->family == AF_INET) {
		/* OK */
		nexthop = nexthop4;
	    }
#ifdef HAVE_IPV6
	    else if (network->family == AF_INET6) {

		if (prefix_is_loopback (network))
		    goto ok;
		if (ipv6_compat_addr (prefix_toaddr6 (network))
		    && ll_addr->prefix->bitlen > 0)
		    continue;
		if (ipv6_link_local_addr (prefix_toaddr6 (network)))
		    continue;
		if (ipv6_multicast_addr (prefix_toaddr6 (network)))
		    continue;
	    ok:
		nexthop = nexthop6;
	    }
#endif /* HAVE_IPV6 */
	    else {
		/* skip others */
		continue;
	    }

	    /* create new one for masking */
	    network = New_Prefix (network->family, prefix_tochar (network),
				  network->bitlen);
	    /* masking may not be needed */
	    netmasking (network->family, prefix_tochar (network),
			network->bitlen);

	    trace (TR_INFO, MRT->trace, "Adding an interface route for %s\n",
		   interface->name);
	    attr = New_Generic_Attr (PROTO_CONNECTED);
    	    attr->nexthop = add_nexthop (nexthop, interface);
    	    attr->gateway = add_gateway (nexthop, 0, interface);
    	    if (MRT->rib_update_route)
        	MRT->rib_update_route (network, attr, NULL, CONNECTED_PREF, 
				       flags);
	    Deref_Generic_Attr (attr);
	    Deref_Prefix (network);

	    if (pointopoint) {
		/* create new one for masking */
		pointopoint = New_Prefix (pointopoint->family,
					  prefix_tochar (pointopoint),
					  pointopoint->bitlen);
		/* masking may not be needed */
		netmasking (pointopoint->family, prefix_tochar (pointopoint),
			    pointopoint->bitlen);

		trace (TR_INFO, MRT->trace, 
		       "Adding an interface route for %s\n", interface->name);
	        attr = New_Generic_Attr (PROTO_CONNECTED);
    	        attr->nexthop = add_nexthop (nexthop, interface);
    	        attr->gateway = add_gateway (nexthop, 0, interface);
    	        if (MRT->rib_update_route)
        	    MRT->rib_update_route (pointopoint, attr, NULL, 
					   CONNECTED_PREF, flags);
	        Deref_Generic_Attr (attr);
		Deref_Prefix (pointopoint);
	    }
	}
    }
    Deref_Prefix (nexthop4);
#ifdef HAVE_IPV6
    Deref_Prefix (nexthop6);
#endif /* HAVE_IPV6 */
}


void
kernel_update_route (prefix_t * dest, generic_attr_t * new, 
		     generic_attr_t * old, int pref)
/* pref is not used */
{
    int index = 0, oldindex = 0, rc;
    prefix_t *nexthop = NULL, *oldhop = NULL;

    if (dest->family == AF_INET && !MRT->kernel_install_flag4)
	return;
#ifdef HAVE_IPV6
    if (dest->family == AF_INET6 && !MRT->kernel_install_flag6)
	return;
#endif /* HAVE_IPV6 */

    if (new) {
	assert (new->gateway);
        index = (new->gateway->interface)?new->gateway->interface->index:0;
        nexthop = new->nexthop->prefix;
    }
    if (old) {
	assert (old->gateway);
        oldindex = (old->gateway->interface)?old->gateway->interface->index:0;
        oldhop = old->nexthop->prefix;
    }
    rc = sys_kernel_update_route (dest, nexthop, oldhop, index, oldindex);

    trace (TR_TRACE, MRT->trace, "KERNEL %s %s nexthop %s index %d %s\n",
	   (nexthop && oldhop == NULL) ? "ADD" : 
	   (nexthop == NULL && oldhop) ? "DEL" :
	   (nexthop && oldhop) ? "CHG" : "???",
	   prefix_toax (dest),
	   prefix_toa (nexthop? nexthop: oldhop), index,
	   (rc < 0) ? "FAILED" : "OK");
}

