/*
 * $Id: bsd.c,v 1.8 1999/04/29 21:35:58 masaki Exp $
 */

#include <mrt.h>

#include <sys/sysctl.h>
#include <net/if_dl.h>
#include <net/route.h>

union sockunion {
    struct sockaddr sa;
    struct sockaddr_in sin;
    struct sockaddr_dl sdl;
#ifdef HAVE_IPV6
    struct sockaddr_in6 sin6;
#endif	/* HAVE_IPV6 */
};


static int sys_kernel_rt_msg (u_char *buf, int bufsize);

static char *
sockunion2char (union sockunion *u)
{
    if (u->sa.sa_family == AF_INET) {
	return ((char *) &u->sin.sin_addr);
    }
#ifdef HAVE_IPV6
    else if (u->sa.sa_family == AF_INET6) {
	return ((char *) &u->sin6.sin6_addr);
    }
#endif /* HAVE_IPV6 */
    else if (u->sa.sa_family == AF_LINK) {
	return ((char *) u->sdl.sdl_data);
    }
    return (NULL);
}


int 
read_interfaces ()
{
    size_t needed;
    char *buf;

#define ROUNDUP(a) \
   ((a) > 0 ? (1 + (((a) - 1) | (sizeof(long) - 1))) : sizeof(long))

    int mib[] =
    {CTL_NET, PF_ROUTE, 0, 0, NET_RT_IFLIST, 0};

    if (sysctl (mib, 6, NULL, &needed, NULL, 0) < 0) {
	trace (TR_FATAL, INTERFACE_MASTER->trace, "sysctl: %m\n");
	return (-1);
    }
    needed = ROUNDUP (needed);
    buf = NewArray (char, needed);
    if (sysctl (mib, 6, buf, &needed, NULL, 0) < 0) {
	trace (TR_FATAL, INTERFACE_MASTER->trace, "sysctl: %m\n");
	return (-1);
    }

    sys_kernel_rt_msg (buf, needed);
    Delete (buf);
    return (1);
}


struct m_rtmsg {
    struct rt_msghdr m_rtm;
    char m_space[512];
};


static void kernel_rtmsg_rcv (int sockfd);
static int route_sockfd = -1;

int
kernel_init (void)
{
    if ((route_sockfd = socket (PF_ROUTE, SOCK_RAW, 0)) < 0) {
        trace (TR_ERROR, MRT->trace, "PF_ROUTE socket (%m)\n");
        return (-1);
    }   
    select_add_fd_event ("kernel_rtmsg_rcv", route_sockfd, SELECT_READ, TRUE, 
			  NULL, kernel_rtmsg_rcv, 1, route_sockfd);
    return (0);
}


/* 
 */
int 
sys_kernel_update_route (prefix_t * dest, prefix_t * nexthop, 
			 prefix_t * oldhop, int index, int oldindex)
{
    int rlen;
    struct m_rtmsg m_rtmsg;
    char *cp = m_rtmsg.m_space;
    union sockunion so_dst, so_gate, so_mask, so_genmask, so_ifa, so_ifp;

    int flags, rtm_addrs;
    static int seq = 0;
    struct rt_metrics rt_metrics;
    register int l;
#ifdef notdef
    interface_t *interface;
#endif
    int len = dest->bitlen;

    if (route_sockfd < 0)
	return (-1);

    bzero ((char *) &m_rtmsg, sizeof (m_rtmsg));
    bzero ((char *) &rt_metrics, sizeof (rt_metrics));
    bzero (&so_dst, sizeof (so_dst));
    bzero (&so_gate, sizeof (so_gate));
    bzero (&so_mask, sizeof (so_mask));
    bzero (&so_genmask, sizeof (so_genmask));
    bzero (&so_ifp, sizeof (so_ifp));
    bzero (&so_ifa, sizeof (so_ifa));

    if (nexthop && oldhop == NULL) {
	m_rtmsg.m_rtm.rtm_type = RTM_ADD;
    }
    else if (nexthop == NULL && oldhop) {
	nexthop = oldhop;
	index = oldindex;
	m_rtmsg.m_rtm.rtm_type = RTM_DELETE;
    }
    else if (nexthop && oldhop) {
        /* probably, ADD works like CHANGE */
	/* there is no way to specify index of route to be removed */
	/* m_rtmsg.m_rtm.rtm_type = RTM_ADD; */
	m_rtmsg.m_rtm.rtm_type = RTM_CHANGE;
    }

    if (dest->family == AF_INET) {
	so_dst.sin.sin_addr.s_addr = prefix_tolong (dest);
	so_dst.sin.sin_len = sizeof (struct sockaddr_in);
	so_dst.sin.sin_family = AF_INET;
    	netmasking (AF_INET, (char *)&so_dst.sin.sin_addr, len);

	so_gate.sin.sin_addr.s_addr = prefix_tolong (nexthop);
	so_gate.sin.sin_len = sizeof (struct sockaddr_in);
	so_gate.sin.sin_family = AF_INET;

	so_mask.sin.sin_len = sizeof (struct sockaddr_in);
        so_mask.sin.sin_family = AF_UNSPEC;
	len2mask (len, (char *) &so_mask.sin.sin_addr, 4);

	rtm_addrs = RTA_DST | RTA_GATEWAY | RTA_NETMASK;

	if (index > 0) {
	    so_ifp.sdl.sdl_index = index;
	    so_ifp.sdl.sdl_len = sizeof (struct sockaddr_dl);
	    so_ifp.sdl.sdl_family = AF_LINK;
	    rtm_addrs |= RTA_IFP;
	}


    }
#ifdef HAVE_IPV6
    else if (dest->family == AF_INET6) {
	memcpy (&so_dst.sin6.sin6_addr, prefix_tochar (dest), 16);
	so_dst.sin6.sin6_len = sizeof (struct sockaddr_in6);
	so_dst.sin6.sin6_family = AF_INET6;
    	netmasking (AF_INET6, (char *)&so_dst.sin6.sin6_addr, len);

        memcpy (&so_gate.sin6.sin6_addr, prefix_tochar (nexthop), 16);
	so_gate.sin6.sin6_len = sizeof (struct sockaddr_in6);
	so_gate.sin6.sin6_family = AF_INET6;

	so_mask.sin6.sin6_len = sizeof (struct sockaddr_in6);
        so_mask.sin6.sin6_family = AF_UNSPEC;
	len2mask (len, (char *) &so_mask.sin6.sin6_addr, 16);

	rtm_addrs = RTA_DST | RTA_GATEWAY | RTA_NETMASK;

	if (index > 0) {
	     so_ifp.sdl.sdl_index = index;
	     so_ifp.sdl.sdl_len = sizeof (struct sockaddr_dl);
	     so_ifp.sdl.sdl_family = AF_LINK;
	     rtm_addrs |= RTA_IFP;
	}

#ifdef __KAME__
	/* KAME IPV6 still requires an index here */
	if (IN6_IS_ADDR_LINKLOCAL (&so_gate.sin6.sin6_addr)) {
       	    so_gate.sin6.sin6_addr.s6_addr8[2] = index >> 8;;
       	    so_gate.sin6.sin6_addr.s6_addr8[3] = index;
	}
#endif /* __KAME__ */
#ifdef notdef
	interface = find_interface_byindex (index);

	assert (interface->primary6);
	memcpy (&so_ifa.sin6.sin6_addr,
		prefix_tochar (
			BIT_TEST (interface->flags, IFF_POINTOPOINT)?
			    interface->primary6->broadcast:
			interface->primary6->prefix), 16);
	so_ifa.sin6.sin6_len = sizeof (struct sockaddr_in6);
	so_ifa.sin6.sin6_family = AF_INET6;
#endif

    }
#endif /* HAVE_IPV6 */
    else			/* unknown family */
	return (-1);

    flags = RTF_UP;
    if (dest->family == AF_INET) {
#ifdef notdef
	/* this doesn't work when removing it */
	if (len == 32)
	    flags |= RTF_HOST;
#endif
	if (so_dst.sin.sin_addr.s_addr != INADDR_ANY)
	    flags |= RTF_GATEWAY;
#ifdef RTF_REJECT
#ifndef INADDR_LOOPBACK
#define INADDR_LOOPBACK htonl(0x7f000001)
#endif /* INADDR_LOOPBACK */
	if (so_dst.sin.sin_addr.s_addr == INADDR_LOOPBACK)
	    flags |= RTF_REJECT;
#endif /* RTF_REJECT */
    }
#ifdef HAVE_IPV6
    else if (dest->family == AF_INET6) {
#ifdef notdef
	if (len == 128)
	    flags |= RTF_HOST;
#endif
	if (!IN6_IS_ADDR_UNSPECIFIED (&so_dst.sin6.sin6_addr))
	    flags |= RTF_GATEWAY;
#ifdef RTF_REJECT
	if (IN6_IS_ADDR_LOOPBACK (&so_dst.sin6.sin6_addr))
	    flags |= RTF_REJECT;
#endif /* RTF_REJECT */
    }
#endif /* HAVE_IPV6 */

#define NEXTADDRP(w, u) \
	if (rtm_addrs & (w)) {\
	    l = ROUNDUP(u.sa.sa_len); bcopy((char *)&(u), cp, l); cp += l;\
	}

    m_rtmsg.m_rtm.rtm_flags = flags;
    m_rtmsg.m_rtm.rtm_version = RTM_VERSION;
    m_rtmsg.m_rtm.rtm_seq = ++seq;
    m_rtmsg.m_rtm.rtm_addrs = rtm_addrs;
    m_rtmsg.m_rtm.rtm_rmx = rt_metrics;
    /*m_rtmsg.m_rtm.rtm_inits = 0; */

    NEXTADDRP (RTA_DST, so_dst);
    NEXTADDRP (RTA_GATEWAY, so_gate);
    NEXTADDRP (RTA_NETMASK, so_mask);
    NEXTADDRP (RTA_GENMASK, so_genmask);
    NEXTADDRP (RTA_IFP, so_ifp);
    NEXTADDRP (RTA_IFA, so_ifa);

    m_rtmsg.m_rtm.rtm_msglen = l = cp - (char *) &m_rtmsg;

again:
    if ((rlen = write (route_sockfd, (char *) &m_rtmsg, l)) < 0) {
	/* I don't know why the kernel distinguishes 
		host and prefixlen = 32 (ipv6) or 128 (ipv6) */
	if (m_rtmsg.m_rtm.rtm_type == RTM_DELETE && (
    		(dest->family == AF_INET && len == 32 && 
		 !BIT_TEST (flags, RTF_HOST))
#ifdef HAVE_IPV6
    	     || (dest->family == AF_INET6 && len == 128 && 
		 !BIT_TEST (flags, RTF_HOST))
#endif /* HAVE_IPV6*/
	     )) {
	    trace (TR_WARN, MRT->trace, 
		   "route socket dst %p nh %a (%m), trying with RTF_HOST\n", 
	            dest, nexthop);
    	    m_rtmsg.m_rtm.rtm_flags |= RTF_HOST;
	    goto again;
	}
	if (m_rtmsg.m_rtm.rtm_type == RTM_CHANGE) {
	    int r = 0;
	    trace (TR_WARN, MRT->trace, 
		   "route socket dst %p nh %a (%m), trying delete/add\n", 
	            dest, nexthop);
	    r = sys_kernel_update_route (dest, NULL, oldhop, 0, oldindex);
	    r = sys_kernel_update_route (dest, nexthop, NULL, index, 0);
	    return (r);
	}
	if (errno == EEXIST) {
	    return (1);
	}			/* route already exists */
	trace (TR_ERROR, MRT->trace, "route socket dst %p nh %a (%m)\n", 
	       dest, nexthop);
	return (-1);
    }
    return (1);
}


#define NEXTADDRR(w, u) \
   if (rtm_addrs & (w)) {  \
           l = ROUNDUP(((struct sockaddr *)cp)->sa_len); \
           bcopy(cp, (char *)&(u), l); cp += l; \
   }


static int
sys_kernel_rt_msg (u_char *buf, int bufsize)
{

    union sockunion so_dst, so_gate, so_mask, so_genmask, so_brd, 
		    so_ifa, so_ifp;
    char *next, *lim;
    struct rt_msghdr *rtm;

    lim = buf + bufsize;
    for (next = buf; next < lim; next += rtm->rtm_msglen) {

        int family = 0, masklen = 0, l, index = 0;
        u_char *dest, *nexthop, *mask;
	char *addr = NULL, *broadcast = NULL;
	char *cp;
	u_long rtm_addrs;

	rtm = (struct rt_msghdr *) next;

	if (rtm->rtm_version != RTM_VERSION) {
	    trace (TR_ERROR, MRT->trace,
		   "rtm_version mismatch: %d should be %d\n",
		   rtm->rtm_version, RTM_VERSION);
	}

	if (rtm->rtm_type == RTM_IFINFO) {
	    struct if_msghdr *ifm = (struct if_msghdr *) rtm;
            struct sockaddr_dl *sdl;
	    char name[IFNAMSIZ];

            cp = (u_char *)(ifm + 1);
	    rtm_addrs = ifm->ifm_addrs;

            if (ifm->ifm_addrs != RTA_IFP) {
                trace (TR_ERROR, INTERFACE_MASTER->trace,
		       "bad ifm_addrs 0x%x\n", ifm->ifm_addrs);
                continue;
	    }
            sdl = (struct sockaddr_dl *) cp;
            if (sdl->sdl_family != AF_LINK) {
                trace (TR_ERROR, INTERFACE_MASTER->trace,
		       "bad sdl family %d\n", sdl->sdl_family);
                continue;
	    }
	    memcpy (name, sdl->sdl_data, sdl->sdl_nlen);
	    name[sdl->sdl_nlen] = '\0';
            if (sdl->sdl_nlen >= IFNAMSIZ) {
                trace (TR_ERROR, INTERFACE_MASTER->trace,
		       "too long name %d\n", sdl->sdl_nlen);
                continue;
            }
	    if (ifm->ifm_index <= 0)
		ifm->ifm_index = sdl->sdl_index;

	    new_interface (name, ifm->ifm_flags & 0xffff, 
				 ifm->ifm_data.ifi_mtu, ifm->ifm_index);
	}
	else if (rtm->rtm_type == RTM_NEWADDR ||
	         rtm->rtm_type == RTM_DELADDR) {
	    struct ifa_msghdr *ifam = (struct ifa_msghdr *) rtm;
	    interface_t *interface;

	    cp = (char *) ifam + sizeof (*ifam);
	    rtm_addrs = ifam->ifam_addrs;

	    NEXTADDRR (RTA_DST, so_dst);
	    NEXTADDRR (RTA_GATEWAY, so_gate);
	    NEXTADDRR (RTA_NETMASK, so_mask);
	    NEXTADDRR (RTA_IFP, so_ifp);
	    NEXTADDRR (RTA_IFA, so_ifa);
	    NEXTADDRR (RTA_BRD, so_brd);

	    if (ifam->ifam_index <= 0) {
	        if (BIT_TEST (rtm_addrs, RTA_IFP)) {
                    struct sockaddr_dl *sdl;
                    sdl = (struct sockaddr_dl *) &so_ifp;
                    if (sdl->sdl_family == AF_LINK)
			ifam->ifam_index = sdl->sdl_index;
		}
	    }

	    if ((interface = find_interface_byindex (ifam->ifam_index)) == NULL)
	        continue;

	    if (BIT_TEST (ifam->ifam_addrs, RTA_BRD)) {
		broadcast = sockunion2char (&so_brd);
	    }
	    if (BIT_TEST (ifam->ifam_addrs, RTA_IFA)) {
		family = so_ifa.sa.sa_family;
		addr = sockunion2char (&so_ifa);
#ifdef __KAME__
		/* KAME IPV6 still has an index here */
		if (family == AF_INET6 && 
			IN6_IS_ADDR_LINKLOCAL ((struct in6_addr *)addr)) {
            	    addr[2] = 0;
            	    addr[3] = 0;
		}
#endif /* __KAME__ */
		if (BIT_TEST (ifam->ifam_addrs, RTA_NETMASK)) {
		    so_mask.sa.sa_family = so_ifa.sa.sa_family;
		    masklen = mask2len (sockunion2char (&so_mask),
					(family == AF_INET) ? 4 :
#ifdef HAVE_IPV6
					(family == AF_INET6) ? 16 :
#endif /* HAVE_IPV6 */
					0);
		}
		if (addr) {
		    update_addr_of_interface (
		   	(rtm->rtm_type == RTM_NEWADDR)? 'A': 'D',
				interface, family, addr, masklen,
				(broadcast) ? broadcast : NULL);
		}
	    }
	}
	else if (rtm->rtm_type == RTM_GET || rtm->rtm_type == RTM_ADD ||
		rtm->rtm_type == RTM_DELETE || rtm->rtm_type == RTM_CHANGE) {
	    int proto = PROTO_KERNEL;
#if 0
	    printf ("rtm->rtm_flags=%x rtm->rtm_addrs=%x\n", 
		rtm->rtm_flags, rtm->rtm_addrs);
#endif
	    if (rtm->rtm_type == RTM_CHANGE) {
	        trace (TR_TRACE, MRT->trace, "RTM_CHANGE not yet supported\n");
		rtm->rtm_type = RTM_ADD; /* XXX */
	    }
	    if (rtm->rtm_errno != 0)
		continue;

	    /* REJECT and BLACKHOLE are not handled yet */
	    if (rtm->rtm_flags &
		    ~(RTF_UP|RTF_GATEWAY|RTF_HOST|RTF_STATIC
			|RTF_CLONING /* direct if */
			|RTF_XRESOLVE /* direct if for IPv6 */
			|RTF_DONE /* on injection */
#ifdef RTF_PRCLONING
			|RTF_PRCLONING /* usual route has this */
#endif /* RTF_PRCLONING */
#ifdef RTF_LOCAL /* OpenBSD does not have RTF_LOCAL */
			|RTF_LOCAL /* loopback route has this */

#endif /* RTF_LOCAL */
			|RTF_REJECT|RTF_BLACKHOLE))
		continue;
/* RTM_DELETE doesn't have RTF_UP */
/*
	    if (!BIT_TEST (rtm->rtm_flags, RTF_UP))
		continue;
*/
	    /* currently I don't want to loop back my request */
	    if (MRT->pid == rtm->rtm_pid)
	        continue;

	    if (!BIT_TEST (rtm->rtm_flags, RTF_GATEWAY))
		/* connected route is added by interface */
		proto = PROTO_CONNECTED;
	    if (BIT_TEST (rtm->rtm_flags, RTF_STATIC))
		proto = PROTO_STATIC;

	    index = rtm->rtm_index;
	    cp = (char *) rtm + sizeof (*rtm);
	    rtm_addrs = rtm->rtm_addrs;

	    NEXTADDRR (RTA_DST, so_dst);
	    NEXTADDRR (RTA_GATEWAY, so_gate);
	    NEXTADDRR (RTA_NETMASK, so_mask);
	    NEXTADDRR (RTA_GENMASK, so_genmask);
	    NEXTADDRR (RTA_IFP, so_ifp);
	    NEXTADDRR (RTA_IFA, so_ifa);

	    if (index <= 0) {
	        if (BIT_TEST (rtm_addrs, RTA_IFP)) {
                    struct sockaddr_dl *sdl;
                    sdl = (struct sockaddr_dl *) &so_ifp;
                    if (sdl->sdl_family == AF_LINK)
			index = sdl->sdl_index;
		}
	    }

	    family = so_dst.sa.sa_family;
	    dest = sockunion2char (&so_dst);
	    nexthop = sockunion2char (&so_gate);
	    so_mask.sa.sa_family = family; /* required by sockunion2char */
	    mask = sockunion2char (&so_mask);

	    if (family != AF_INET
#ifdef HAVE_IPV6
		&& family != AF_INET6
#endif /* HAVE_IPV6 */
		)
		continue;

	    if (BIT_TEST (rtm->rtm_flags, RTF_HOST)) {
		masklen = (family == AF_INET) ? 32 : 128;
	    }
	    else {
		/* I don't know the reason, 
		   but it's needed for getting rid of strange subnetmask */
		if (family == AF_INET && *(u_long *) dest == *(u_long *) mask)
		    so_mask.sa.sa_len = 0;

		if (!BIT_TEST (rtm->rtm_addrs, RTA_NETMASK) || 
			so_mask.sa.sa_len <= 2) {
		    /* not specific netmask */
		    if (family == AF_INET) {
			/* natural mask */
			if (*(u_long *) dest == INADDR_ANY)
			    masklen = 0;
			else
			    masklen = (dest[0] < 128) ? 8 :
				(dest[0] < 192) ? 16 :
				(dest[0] < 224) ? 24 : 32;
		    }
		    else {
			masklen = 0;
		    }
		}
		else {
		    char tmp[16];
		    memset (tmp, 0, sizeof (tmp));
		    memcpy (tmp, sockunion2char (&so_mask), so_mask.sa.sa_len -
			    (sockunion2char (&so_mask) - (char *) &so_mask));
		    masklen = mask2len (tmp, (family == AF_INET) ? 4 : 16);
		}
	    }

	    if (so_gate.sa.sa_family == AF_LINK) {
		/* forget the physical if info  */
		memset (nexthop, 0, (family == AF_INET)? 4: 16);
	        so_gate.sa.sa_family = family;
	    }
	    if (so_gate.sa.sa_family != family)
		continue;

#ifdef __KAME__
	    /* KAME IPV6 still has an index here */
	    if (family == AF_INET6 && 
		    IN6_IS_ADDR_LINKLOCAL ((struct in6_addr *)nexthop)) {
       	        nexthop[2] = 0;
       	        nexthop[3] = 0;
	    }
#endif /* __KAME__ */

	    update_kernel_route ((rtm->rtm_type == RTM_DELETE)? 'D': 'A',
				 family, dest, nexthop, masklen, index, proto);
	}
    }
    return (0);
}


int
sys_kernel_read_rt_table (void)
{
    size_t needed;
    int mib[] =
    {CTL_NET, PF_ROUTE, 0, 0, NET_RT_DUMP, 0};
    char *buf;

    if (sysctl (mib, 6, NULL, &needed, NULL, 0) < 0) {
	trace (TR_ERROR, MRT->trace, "sysctl: %m\n");
	return (-1);
    }
    buf = NewArray (char, needed);
    if (sysctl (mib, 6, buf, &needed, NULL, 0) < 0) {
	trace (TR_ERROR, MRT->trace, "sysctl: %m\n");
	return (-1);
    }

    sys_kernel_rt_msg (buf, needed);
    Delete (buf);
    return (1);
}


static void 
kernel_rtmsg_rcv (int sockfd)
{
    int n;
    char msgbuf[4096]; /* I don't know how much is enough */

    if ((n = read (sockfd, msgbuf, sizeof (msgbuf))) > 0) {
        sys_kernel_rt_msg (msgbuf, n);
    }
    else {
	trace (TR_ERROR, MRT->trace, "read on routing socket %d (%m)\n", 
	      sockfd);
    }
    select_enable_fd (sockfd);
}
