/*
 * $Id: filter.c,v 1.6 1999/04/30 15:42:32 masaki Exp $
 */

#include <mrt.h>
#include <bgp.h>
#include <filter.h>


#define MAX_ROUTE_MAP 100
static LINKED_LIST *ll_route_map[MAX_ROUTE_MAP];


static void
del_route_map_memory (route_map_t *route_map)
{
    assert (route_map);
    if (route_map->attr)
	 bgp_deref_attr (route_map->attr);
    Delete (route_map);
}


route_map_t *
add_route_map (int num, bgp_attr_t * attr, u_long flag)
{
    route_map_t *route_map;

    if (num <= 0 && num >= MAX_ROUTE_MAP) {
	return (NULL);
    }
    route_map = New (route_map_t);
    route_map->attr = bgp_ref_attr (attr);
    route_map->flag = flag;
    /* XXX condition */
    if (ll_route_map[num] == NULL)
	ll_route_map[num] = 
	    LL_Create (LL_DestroyFunction, 
		       del_route_map_memory, 0);
    LL_Add (ll_route_map[num], route_map);
    return (route_map);
}


route_map_t *
get_route_map (int num, bgp_attr_t * attr)
{
    route_map_t *route_map;

    if (num <= 0 && num >= MAX_ROUTE_MAP) {
	return (NULL);
    }
    if (ll_route_map[num] == NULL)
	return (NULL);

    if (attr == NULL)
	return (LL_GetHead (ll_route_map[num]));
    LL_Iterate (ll_route_map[num], route_map) {
	if (attr == route_map->attr)
	    return (route_map);
    }
    return (NULL);
}


void
del_route_map (int num)
{
    if (num <= 0 && num >= MAX_ROUTE_MAP) {
	return;
    }
    if (ll_route_map[num])
        LL_Clear (ll_route_map[num]);
}


/* destractive */
bgp_attr_t *
apply_route_map (int num, bgp_attr_t * attr)
{
    route_map_t *route_map;

    if (num >= MAX_ROUTE_MAP)
	return (NULL);

    if (ll_route_map[num] == NULL)
	return (attr);

    LL_Iterate (ll_route_map[num], route_map) {
	assert (route_map->attr);
	/* check condition */

	if (BGP4_BIT_TEST (route_map->attr->attribs, PA4_TYPE_ORIGIN)) {
	    attr->origin = route_map->attr->origin;
	    BGP4_BIT_SET (attr->attribs, PA4_TYPE_ORIGIN);
	}
	if (BGP4_BIT_TEST (route_map->attr->attribs, PA4_TYPE_ASPATH)) {
	    if (BIT_TEST (route_map->flag, ROUTE_MAP_ASPATH_PREPEND))
		attr->aspath = aspath_prepend (attr->aspath,
					       route_map->attr->aspath);
	    else {
		if (BGP4_BIT_TEST (attr->attribs, PA4_TYPE_ASPATH))
		    Delete_ASPATH (attr->aspath);
		attr->aspath = aspath_copy (route_map->attr->aspath);
	    }
	    BGP4_BIT_SET (attr->attribs, PA4_TYPE_ASPATH);
	}
	if (BGP4_BIT_TEST (route_map->attr->attribs, PA4_TYPE_NEXTHOP)) {
#ifdef HAVE_IPV6
	    if (route_map->attr->link_local) {
		if (attr->link_local)
		    deref_nexthop (attr->link_local);
		attr->link_local = ref_nexthop (route_map->attr->link_local);
	    }
	    if (route_map->attr->nexthop4) {
		if (attr->nexthop4)
		    deref_nexthop (attr->nexthop4);
		attr->nexthop4 = ref_nexthop (route_map->attr->nexthop4);
	    }
#endif /* HAVE_IPV6 */
	    if (BGP4_BIT_TEST (attr->attribs, PA4_TYPE_NEXTHOP))
		deref_nexthop (attr->nexthop);
	    attr->nexthop = ref_nexthop (route_map->attr->nexthop);
	    BGP4_BIT_SET (attr->attribs, PA4_TYPE_NEXTHOP);
	}
	if (BGP4_BIT_TEST (route_map->attr->attribs, PA4_TYPE_METRIC)) {
	    attr->multiexit = route_map->attr->multiexit;
	    BGP4_BIT_SET (attr->attribs, PA4_TYPE_METRIC);
	}
	if (BGP4_BIT_TEST (route_map->attr->attribs, PA4_TYPE_LOCALPREF)) {
	    attr->local_pref = route_map->attr->local_pref;
	    BGP4_BIT_SET (attr->attribs, PA4_TYPE_LOCALPREF);
	}
	if (BGP4_BIT_TEST (route_map->attr->attribs, PA4_TYPE_ATOMICAGG)) {
	    BGP4_BIT_SET (attr->attribs, PA4_TYPE_ATOMICAGG);
	}
	if (BGP4_BIT_TEST (route_map->attr->attribs, PA4_TYPE_AGGREGATOR)) {
	    if (BGP4_BIT_TEST (attr->attribs, PA4_TYPE_AGGREGATOR))
		Deref_Prefix (attr->aggregator.prefix);
	    attr->aggregator.as = route_map->attr->aggregator.as;
	    attr->aggregator.prefix =
		Ref_Prefix (route_map->attr->aggregator.prefix);
	    BGP4_BIT_SET (attr->attribs, PA4_TYPE_AGGREGATOR);
	}
	if (BGP4_BIT_TEST (route_map->attr->attribs, PA4_TYPE_COMMUNITY)) {
	    if (BGP4_BIT_TEST (attr->attribs, PA4_TYPE_COMMUNITY))
		Delete_community (attr->community);
	    attr->community = community_copy (route_map->attr->community);
	    BGP4_BIT_SET (attr->attribs, PA4_TYPE_COMMUNITY);
	}
	if (BGP4_BIT_TEST (route_map->attr->attribs, PA4_TYPE_DPA)) {
	    attr->dpa = route_map->attr->dpa;
	    BGP4_BIT_SET (attr->attribs, PA4_TYPE_DPA);
	}
#ifdef notdef
/* XXX pref is in route_t */
	if (route_map->attr->pref >= 0)
	    attr->pref = route_map->attr->pref;
#endif
    }
    return (attr);
}


char *
route_map_toa (int num)
{
    route_map_t *route_map;
    char line[MAXLINE], *tmpx = line;
    int len = 0;

    assert (num < MAX_ROUTE_MAP);

    if (ll_route_map[num] == NULL || LL_GetCount (ll_route_map[num]) <= 0)
	return (NULL);

    /* estimate a storage size required */
    LL_Iterate (ll_route_map[num], route_map) {
	len += route_map_toa2 (route_map, NULL);
    }

    tmpx = NewArray (char, len + 1);
    /* now printing */
    len = 0;
    LL_Iterate (ll_route_map[num], route_map) {
	len += route_map_toa2 (route_map, tmpx + len);
    }

    return (tmpx);
}


int
route_map_toa2 (route_map_t * route_map, char *cp)
{
    char line[MAXLINE];
    bgp_attr_t *attr = route_map->attr;
    int len, total = 0;

    if (BGP4_BIT_TEST (attr->attribs, PA4_TYPE_ORIGIN)) {
	len = sprintf (line, "  set origin %s\n",
		       origin2string (attr->origin));
	if (cp) strcpy (cp + total, line);
	total += len;
    }
    if (BGP4_BIT_TEST (attr->attribs, PA4_TYPE_ASPATH)) {
	len = sprintf (line, "  set as-path ");
	if (cp) strcpy (cp + total, line);
	total += len;

	if (BIT_TEST (route_map->flag, ROUTE_MAP_ASPATH_PREPEND)) {
	    len = sprintf (line, "prepend ");
	    if (cp) strcpy (cp + total, line);
	    total += len;
	}
	len = sprintf (line, "%s\n", aspath_toa (attr->aspath));
	if (cp) strcpy (cp + total, line);
	total += len;
    }

    if (BGP4_BIT_TEST (attr->attribs, PA4_TYPE_NEXTHOP)) {
#ifdef HAVE_IPV6
	if (attr->link_local) {
	    len = sprintf (line, "  set next-hop %s\n",
			   prefix_toa (attr->link_local->prefix));
	    if (cp) strcpy (cp + total, line);
	    total += len;
	}
	if (attr->nexthop4) {
	    len = sprintf (line, "  set next-hop %s\n",
			   prefix_toa (attr->nexthop4->prefix));
	    if (cp) strcpy (cp + total, line);
	    total += len;
	}
#endif /* HAVE_IPV6 */
	len = sprintf (line, "  set next-hop %s\n",
		       prefix_toa (attr->nexthop->prefix));
	if (cp) strcpy (cp + total, line);
	total += len;
    }
    if (BGP4_BIT_TEST (attr->attribs, PA4_TYPE_METRIC)) {
	len = sprintf (line, "  set metric %ld\n", attr->multiexit);
	if (cp) strcpy (cp + total, line);
	total += len;
    }
    if (BGP4_BIT_TEST (attr->attribs, PA4_TYPE_LOCALPREF)) {
	len = sprintf (line, "  set local-preference %ld\n",
		       attr->local_pref);
	if (cp) strcpy (cp + total, line);
	total += len;
    }
    if (BGP4_BIT_TEST (attr->attribs, PA4_TYPE_ATOMICAGG)) {
	len = sprintf (line, "  set atomic-aggregate\n");
	if (cp) strcpy (cp + total, line);
	total += len;
    }
    if (BGP4_BIT_TEST (attr->attribs, PA4_TYPE_AGGREGATOR)) {
	len = sprintf (line, "  set aggregator as %d %s\n",
		 attr->aggregator.as, prefix_toa (attr->aggregator.prefix));
	if (cp) strcpy (cp + total, line);
	total += len;
    }
    if (BGP4_BIT_TEST (attr->attribs, PA4_TYPE_COMMUNITY)) {
	int i;

	for (i = 0; i < attr->community->len; i++) {
	    if (i == 0) {
		len = sprintf (line, "  set community %s\n",
				community_toa2 (attr->community->value[i]));
	    }
	    else {
		len = sprintf (line, "  set community %s additive\n",
				community_toa2 (attr->community->value[i]));
	    }
	    if (cp) strcpy (cp + total, line);
	    total += len;
	}
    }
    if (BGP4_BIT_TEST (attr->attribs, PA4_TYPE_DPA)) {
	len = sprintf (line, "  set dpa as %d %ld\n",
		       attr->dpa.as, attr->dpa.value);
	if (cp) strcpy (cp + total, line);
	total += len;
    }
#ifdef notdef
/* XXX pref is in route_t */
    if (attr->pref >= 0) {
	len = sprintf (line, "  set weight %d\n", attr->pref);
	if (cp) strcpy (cp + total, line);
	total += len;
    }
#endif
    return (total);
}
