/*
 * $Id: view.c,v 1.44 1999/04/30 23:47:56 masaki Exp $
 */

#include <ctype.h>
#include <mrt.h>
#include <config_file.h>
#include <bgp.h>


static bgp_route_t *
bgp_add_route2 (view_t * view, prefix_t * prefix, bgp_attr_t * attr, 
		int weight, u_long flags);

static aggregate_t *
view_check_aggregate (view_t * view, prefix_t *prefix, int withdraw);

static generic_attr_t *
bgp2gen (bgp_attr_t *attr)
{
    generic_attr_t *gattr;

    if (attr == NULL)
	return (NULL);
    gattr = New_Generic_Attr (attr->type);
    gattr->gateway = attr->gateway;
    if (attr->nexthop2) {
        gattr->nexthop = ref_nexthop (attr->nexthop2);
    }
#ifdef HAVE_IPV6
    else if (attr->link_local) {
        gattr->nexthop = ref_nexthop (attr->link_local);
    }
#endif /* HAVE_IPV6 */
    else {
	assert (attr->nexthop);
        gattr->nexthop = ref_nexthop (attr->nexthop);
    }
    return (gattr);
}


void static
bgp_rt_update_call (prefix_t * prefix, bgp_attr_t * new, bgp_attr_t * old)
{
    generic_attr_t *gnew;
    generic_attr_t *gold;
    int pref = BGP_PREF;
    u_long flags = 0;

    assert (prefix);

    if (new == NULL && old == NULL)
	return;
    if (MRT->rib_update_route == NULL)
        return;
    /* check to see if iBGP */
    if (new && BIT_TEST (new->options, BGP_INTERNAL)) {
	BIT_SET (flags, MRT_RTOPT_SUPPRESS);
	pref = IBGP_PREF;
    }
    gnew = bgp2gen (new);
    gold = bgp2gen (old);

    MRT->rib_update_route (prefix, gnew, gold, pref, flags);

    Deref_Generic_Attr (gnew);
    Deref_Generic_Attr (gold);
}


static bgp_route_t *
New_Bgp_Route (bgp_route_head_t * bgp_route_head, bgp_attr_t * attr, 
	       int weight, u_long flags)
{
    bgp_route_t *route = New (bgp_route_t);

    route->head = bgp_route_head;
    assert (weight >= 0);
    route->weight = weight;
    route->attr = bgp_ref_attr (attr);
    route->flags = flags;
    time (&route->time); /* this is actually pretty expensive call... what to do??? */
    BGP->bgp_num_active_route_node++;

    /* only bother converting to ASCII if tracing, otherwise it can be
     * really slow under bgpsim 
     */
      trace (TR_TRACE, BGP->trace, 
	     "New Route: %p nh %a proto %r weight %d flags 0x%x\n",
	     bgp_route_head->prefix,
	     attr->nexthop->prefix, attr->type, weight, flags);
    if (attr->type == PROTO_BGP)
        LL_Add (bgp_route_head->ll_routes, route);
    else
        LL_Add (bgp_route_head->ll_imported, route);
    return (route);
}


static void 
Delete_Bgp_Route (bgp_route_t * rt_node)
{

    trace (TR_TRACE, BGP->trace, "Delete Route: %p nh %a proto %r\n",
	   rt_node->head->prefix,
	   rt_node->attr->nexthop->prefix,
	   rt_node->attr->type);
    bgp_deref_attr (rt_node->attr);
    Delete (rt_node);
    BGP->bgp_num_active_route_node--;
}


static int
bgp_compare_routes (bgp_route_t * a, bgp_route_t * b)
{

    int a1, a2;

    if (BIT_TEST (a->flags, BGP_RT_FILTERED) &&
            !BIT_TEST (b->flags, BGP_RT_FILTERED))
	return (1);
    if (!BIT_TEST (a->flags, BGP_RT_FILTERED) &&
            BIT_TEST (b->flags, BGP_RT_FILTERED))
	return (-1);

    /* these values may be u_long, so I can't subtract them to compare */

    /* larger administrative weight wins */
    if (a->weight > b->weight)
	return (-1);
    if (a->weight < b->weight)
	return (1);

    /* tie breaking */

#ifdef notdef
    /* smaller pref2 wins */
    if (a->pref2 < b->pref2)
	return (-1);
    if (a->pref2 > b->pref2)
	return (1);
#endif

    /* check local preference */
    if (BGP4_BIT_TEST (a->attr->attribs, PA4_TYPE_LOCALPREF) &&
        BGP4_BIT_TEST (b->attr->attribs, PA4_TYPE_LOCALPREF)) {

        /* higer local_pref wins */
        if (a->attr->local_pref < b->attr->local_pref)
	    return (1);
        if (a->attr->local_pref > b->attr->local_pref)
	    return (-1);
    }

    /* shorter aspath wins */
    a1 = aspath_length (a->attr->aspath);
    a2 = aspath_length (b->attr->aspath);
    if (a1 < a2)
	return (-1);
    if (a1 > a2)
	return (1);

    /* lower origin code wins */
    if (a->attr->origin < b->attr->origin)
	return (-1);
    if (a->attr->origin > b->attr->origin)
	return (1);

    /* XXX should be configurable */
    /* lower metric wins */
    if (a->attr->multiexit < b->attr->multiexit)
        return (-1);
    if (a->attr->multiexit > b->attr->multiexit)
	return (1);

    if (a->attr->gateway && b->attr->gateway) {
        /* smaller routerid wins */
        if (ntohl (a->attr->gateway->routerid) < 
		ntohl (b->attr->gateway->routerid))
	    return (-1);
        if (ntohl (a->attr->gateway->routerid) > 
		ntohl (b->attr->gateway->routerid))
	    return (1);
    }

    return (0);
}


static bgp_route_head_t *
New_Bgp_Route_Head (prefix_t * prefix)
{
    bgp_route_head_t *route_head;

    assert (prefix);
    route_head = New (bgp_route_head_t);
    route_head->prefix = Ref_Prefix (prefix);
    route_head->ll_routes = LL_Create (LL_DestroyFunction, Delete_Bgp_Route, 
				       LL_CompareFunction, bgp_compare_routes,
                                       LL_AutoSort, True, 0);
    route_head->active = NULL;
    route_head->ll_imported = LL_Create (LL_DestroyFunction, Delete_Bgp_Route, 
				         LL_CompareFunction, bgp_compare_routes,
                                         LL_AutoSort, True, 0);
    BGP->bgp_num_active_route_head++;

    trace (TR_TRACE, BGP->trace, "Add Route Head: %p\n", prefix);
    return (route_head);
}


static void 
Delete_Bgp_Route_Head (bgp_route_head_t * head)
{
    /* This has done in view_delete_bgp_route */
    trace (TR_TRACE, BGP->trace, "Delete Route Head: %p\n", head->prefix);
    LL_Destroy (head->ll_imported);
    LL_Destroy (head->ll_routes);
    Deref_Prefix (head->prefix);
    Delete (head);
    BGP->bgp_num_active_route_head--;
}


static update_bucket_t *
New_Update_Bucket ()
{
    update_bucket_t *bucket = New (update_bucket_t);

    /* NOTE: update bucket memory belongs to view */
    bucket->ll_prefix = LL_Create (LL_DestroyFunction, Deref_Prefix, 0);
    bucket->attr = NULL;
    return (bucket);
}


void 
delete_update_bucket (update_bucket_t * update_bucket)
{
    assert (update_bucket);
    assert (update_bucket->ll_prefix);
    LL_Destroy (update_bucket->ll_prefix);
    if (update_bucket->attr)
        bgp_deref_attr (update_bucket->attr);
    Delete (update_bucket);
}


static bgp_route_head_t *
view_find_route_head (view_t * view, prefix_t * prefix)
{
    radix_node_t *radix_node;

    radix_node = radix_search_exact (view->radix_tree, prefix);
    if (radix_node) {
	assert (radix_node->data);
	return (RADIX_DATA_GET (radix_node, bgp_route_head_t));
    }
    return (NULL);
}


static void
view_remove_route_head (view_t * view, bgp_route_head_t * route_head)
{
    assert (route_head->radix_node);
    assert (RADIX_DATA_GET (route_head->radix_node, bgp_route_head_t)
	    == route_head);
    BIT_SET (route_head->state_bits, VRTS_DELETE);
    radix_remove (view->radix_tree, route_head->radix_node);
    view->num_active_routes--;
}


#ifdef notdef
static bgp_route_t *
bgp_find_best_route (bgp_route_head_t * bgp_route_head)
{

    bgp_route_t *best = NULL, *route;

    /* find new active route */
    LL_Iterate (bgp_route_head->ll_routes, route) {

	/* no best */
	if (best == NULL) {
	    best = route;
	    continue;
	}

	if (bgp_compare_routes (route, best) < 0) {
	    best = route;
	    continue;
	}

    }
    return (best);
}
#endif


static int
view_add_ll_list (view_t *view, bgp_route_head_t *bgp_route_head, int withdraw)
{
  /* bgp_route_head_t *rh; */
    LINKED_LIST *ll;

    assert (view);
    assert (bgp_route_head);

    ll = (withdraw)? view->ll_with_routes: view->ll_ann_routes;

#ifdef notdef
    LL_Iterate (ll, rh) {
	if (rh == bgp_route_head) {
	    /* this is warning for a while to check out this */
	    trace (TR_WARN, view->trace,
		   "Duplicate %s prefix: %p\n",
		   (withdraw)? "withdraw": "announce",
		   bgp_route_head->prefix);
	    if (withdraw)
		return (0);
	    LL_RemoveFn (ll, rh, NULL);
	    break;
	}
    }
#else
    if (BITX_TEST (&bgp_route_head->view_mask, view->viewno) != 0) {

      trace (TR_WARN, view->trace,
	     "Duplicate %s prefix: %p\n",
	     (withdraw)? "withdraw": "announce",
	     bgp_route_head->prefix);
      if (withdraw)
	return (0);
      LL_RemoveFn (ll, bgp_route_head, NULL);
    }
    else {
        BITX_SET (&bgp_route_head->view_mask, view->viewno);
    }
#endif
    /* we could check withdraw list because implicit withdraw is OK */
    LL_Append (ll, bgp_route_head);
    return (1);
}


static int
view_add_change_list (view_t * view, bgp_route_head_t * bgp_route_head,
		      int withdraw)
{
    aggregate_t *agg;

    agg = view_check_aggregate (view, bgp_route_head->prefix, withdraw);
    if (agg) {
        if (BIT_TEST (agg->option, BGP_AGGOPT_SUMMARY_ONLY)) {
	    BIT_SET (bgp_route_head->state_bits, VRTS_SUPPRESS);
	    
	      trace (TR_TRACE, view->trace,
		     "Change: %p (but suppressed by %p)\n",
		     bgp_route_head->prefix, agg->prefix);
	    return (-1);
        }
    }

    /* add the route head to the change list */
    view_add_ll_list (view, bgp_route_head, withdraw);
    
      trace (TR_TRACE, view->trace, "Change: %p (%s)\n",
	     bgp_route_head->prefix,
	     (withdraw) ? "withdraw" : "announce");

    return (1);
}


static bgp_route_t *
view_add_bgp_route (view_t * view, bgp_route_head_t * bgp_route_head,
		    bgp_attr_t * attr, int weight, u_long flags)
{
    bgp_route_t *bgp_route, *old_best, *imported;

    assert (attr->type == PROTO_BGP);
    LL_Iterate (bgp_route_head->ll_routes, bgp_route) {
	if (bgp_route->attr->gateway == attr->gateway)
	    assert (0);
    }

    /* first time we've heard from this gateway */
    old_best = LL_GetHead (bgp_route_head->ll_routes);
    imported = LL_GetHead (bgp_route_head->ll_imported);
    bgp_route = New_Bgp_Route (bgp_route_head, attr, weight, flags);

    if (BIT_TEST (bgp_route->flags, BGP_RT_FILTERED))
	return (bgp_route);

    if (LL_GetHead (bgp_route_head->ll_routes) != bgp_route) {
	return (bgp_route);
    }

    /* the best bgp route changed */
    bgp_rt_update_call (bgp_route_head->prefix,
			bgp_route->attr, NULL);

    if (imported != NULL && bgp_route_head->active == imported) {
	if (bgp_route_head->active->weight <= bgp_route->weight) {
	    bgp_route_head->active = bgp_route;
    	    /* add the route head to the change list */
    	    view_add_change_list (view, bgp_route_head, 0);
	}
    }
    else {
	bgp_route_head->active = bgp_route;
	if (imported != NULL && 
		bgp_route_head->active->weight < imported->weight)
	    bgp_route_head->active = imported;
    	/* add the route head to the change list */
    	view_add_change_list (view, bgp_route_head, 0);
    }

    return (bgp_route);
}


/* 
 * delete a route from view. If active route, set change list
 * and modify route_head
 */
static int
view_delete_bgp_route (view_t * view, bgp_route_t * bgp_route)
{
    bgp_route_head_t *bgp_route_head;
    bgp_route_t *best, *old_best, *new_best;
    bgp_route_t *imported, *route;

    assert (bgp_route->attr->type == PROTO_BGP);
    best = NULL;
    bgp_route_head = bgp_route->head;
    old_best = LL_GetHead (bgp_route_head->ll_routes);

    LL_Iterate (bgp_route_head->ll_routes, route) {
	if (route == bgp_route)
	    break;
    }
    assert (route != NULL);

    /* remove from list in any case without destroying it */
    LL_RemoveFn (bgp_route_head->ll_routes, bgp_route, NULL);

    /* we are NOT the active route - just delete */
    if (BIT_TEST (bgp_route->flags, BGP_RT_FILTERED)) {
	/* assert (bgp_route_head->active != bgp_route); */
        Delete_Bgp_Route (bgp_route);
	return (1);
    }

    if (old_best != bgp_route) {
        Delete_Bgp_Route (bgp_route);
	return (1);
    }

    new_best = LL_GetHead (bgp_route_head->ll_routes);

    if (bgp_route_head->active != bgp_route) {
	if (new_best != NULL) {
	    bgp_rt_update_call (bgp_route_head->prefix,
				new_best->attr, NULL);
	}
	else {
	    bgp_rt_update_call (bgp_route_head->prefix,
				NULL, bgp_route->attr);
        }
        Delete_Bgp_Route (bgp_route);
        return (1);
    }

    /* It was the active route */

    /* find new active route */
    imported = LL_GetHead (bgp_route_head->ll_imported);

    /* prefix withdrawn with nothing to replace it */
    if (new_best == NULL) {

	bgp_rt_update_call (bgp_route_head->prefix,
			    NULL, bgp_route->attr);

	if (imported != NULL) {
	    Delete_Bgp_Route (bgp_route);
	    bgp_route_head->active = imported;
	    view_add_change_list (view, bgp_route_head, 0);
	    return (0);
	}

	/* no bgp routes nor imported routes */
	/* remove route_head first */
	view_remove_route_head (view, bgp_route_head);
	/* the route pointed by active will be freed later */
        /* push the route back to the list */
        LL_Add (bgp_route_head->ll_routes, bgp_route);
	assert (bgp_route_head->active == bgp_route);
	view_add_change_list (view, bgp_route_head, 1);
	return (1);
    }

    /* the new active found */
    bgp_rt_update_call (bgp_route_head->prefix,
			new_best->attr, bgp_route_head->active->attr);

    Delete_Bgp_Route (bgp_route);
    /* set to new best */
    bgp_route_head->active = new_best;

    if (imported != NULL && new_best->weight < imported->weight)
	bgp_route_head->active = imported;
    view_add_change_list (view, bgp_route_head, 0);
    return (1);
}


/*
 * this will be called from rib, that is, they are called from
 * other threads. Schedule it.
 */

static void
bgp_import (prefix_t * prefix, bgp_attr_t * bgp_new, bgp_attr_t * bgp_old)
{
    int i;
    view_t *view;

    assert (bgp_new == NULL || bgp_new->type != PROTO_BGP);
    assert (bgp_old == NULL || bgp_old->type != PROTO_BGP);

    if (bgp_new == NULL && bgp_old == NULL)
	return;

    for (i = 0; i < MAX_BGP_VIEWS; i++) {

	if ((view = BGP->views[i]) == NULL ||
	    view->family != prefix->family)
	    continue;

	view_open (view);

	if (bgp_old) {
	    bgp_del_route (view, prefix, bgp_old);
	}

	if (bgp_new) {
	    bgp_add_route (view, prefix, bgp_new);
	}

	bgp_process_changes (view);
	view_close (view);
    }

    Deref_Prefix (prefix);
    if (bgp_new) {
	bgp_deref_attr (bgp_new);
    }
    if (bgp_old) {
	bgp_deref_attr (bgp_old);
    }
}


static LINKED_LIST *
build_update_buckets (LINKED_LIST * ll_routes, LINKED_LIST *ll_update_buckets)
{
    update_bucket_t *update_bucket;
    bgp_route_t *route;
    int found;

    if (ll_update_buckets == NULL)
        ll_update_buckets = LL_Create (0);

    /* replace this with a hash, or something a bit more reasonable */
    LL_Iterate (ll_routes, route) {
	found = 0;
	LL_Iterate (ll_update_buckets, update_bucket) {
	    if (update_bucket->attr == route->attr) {
		LL_Add (update_bucket->ll_prefix,
			Ref_Prefix (route->head->prefix));
		found = 1;
		break;
	    }
	}

	if (found == 0) {
	    update_bucket = New_Update_Bucket ();
	    update_bucket->attr = bgp_ref_attr (route->attr);

	    LL_Add (ll_update_buckets, update_bucket);
	    LL_Add (update_bucket->ll_prefix,
		    Ref_Prefix (route->head->prefix));
	}
    }

    return (ll_update_buckets);
}


/*
 * get "head" of routing table entry for a given prefix. Create a head
 * and radix-tree entry if one does not aleady exist
 */
static bgp_route_head_t *
view_get_route_head (view_t * view, prefix_t * prefix)
{
    bgp_route_head_t *bgp_route_head;
    radix_node_t *radix_node;

    assert (view);
    assert (prefix);
    radix_node = radix_search_exact (view->radix_tree, prefix);

    if (radix_node) {
	assert (radix_node->data);
	bgp_route_head = RADIX_DATA_GET (radix_node, bgp_route_head_t);
    }
    else {
	bgp_route_head = New_Bgp_Route_Head (prefix);
	radix_node = radix_lookup (view->radix_tree, prefix);
	assert (radix_node->data == NULL);
	RADIX_DATA_SET (radix_node, bgp_route_head);
	bgp_route_head->radix_node = radix_node;
	view->num_active_routes++;
    }
    return (bgp_route_head);
}


#ifdef notdef
static void
bgp_send_update_schedule (bgp_peer_t * peer, int len, u_char * data)
{
    schedule_event2 ("bgp_send_update", peer->schedule, 
    		      (event_fn_t) bgp_send_update, 3, peer, len, data);
}
#endif


void
view_set_nexthop_self (bgp_peer_t *peer, bgp_attr_t *attr)
{
    if (attr->nexthop) {
	deref_nexthop (attr->nexthop);
	attr->nexthop = NULL;
        BGP4_BIT_RESET (attr->attribs, PA4_TYPE_NEXTHOP);
    }

    attr->nexthop = add_nexthop (peer->local_addr, peer->gateway->interface);
    if (attr->nexthop != NULL) {
        BGP4_BIT_SET (attr->attribs, PA4_TYPE_NEXTHOP);
#ifdef HAVE_IPV6
        if (attr->nexthop->prefix->family == AF_INET6) {
#define CISCO_BGP4MP
#ifdef CISCO_BGP4MP
	    /* BGP4+ doesn't require ipv4 nexthop for MP NLRI,
	       but cisco expects */
            static prefix_t *my_id_prefix = NULL;

            if (my_id_prefix == NULL)
#ifndef BGP_MULTI
                my_id_prefix = New_Prefix (AF_INET, &BGP->my_id, 32);
#else /* BGP_MULTI */
                my_id_prefix = New_Prefix (AF_INET, &peer->my_id, 32);
#endif /* BGP_MULTI */

            if (attr->nexthop4)
	        deref_nexthop (attr->nexthop4);
            attr->nexthop4 = add_nexthop (my_id_prefix, find_interface (my_id_prefix));
#endif /* CISCO_BGP4MP */

            if (attr->link_local)
	        deref_nexthop (attr->link_local);
            if (peer->gateway && peer->gateway->interface &&
	        peer->gateway->interface->link_local &&

/* XXX Pedro's draft says link_local is sent in case the link is shared.
        (1) it's shared if not IFF_POINTOPOINT (ok)
	(2) it's shared if IFF_POINTOPOINT and masklen != 128 (I'm not sure)
	This is required at least by INRIA IPV6 implementation since
	if the nexthop is link-local in the routing table, 
	the source of originated packets may be a link-local. */

	(!BIT_TEST (peer->gateway->interface->flags, IFF_POINTOPOINT) ||
	   (peer->gateway->interface->primary6 &&
	    peer->gateway->interface->primary6->prefix->bitlen != 128)))

	        attr->link_local = 
	            add_nexthop (peer->gateway->interface->link_local->prefix,
				 peer->gateway->interface);
            else
	        attr->link_local = NULL;
	}
#endif /* HAVE_IPV6 */
    }
}


static int
view_attach_reflector_info (bgp_peer_t *peer, bgp_attr_t *attr)
{
    u_long my_cluster_id;

    if (!BGP4_BIT_TEST (attr->attribs, PA4_TYPE_ORIGINATOR_ID)) {
	prefix_t *my_id_prefix;
#ifndef BGP_MULTI
	my_id_prefix = New_Prefix (AF_INET, &BGP->my_id, 32);
#else /* BGP_MULTI */
	my_id_prefix = New_Prefix (AF_INET, &BGP->current_bgp->this_id, 32);
#endif /* BGP_MULTI */
        attr->originator = my_id_prefix;
        BGP4_BIT_SET (attr->attribs, PA4_TYPE_ORIGINATOR_ID);
    }

#ifndef BGP_MULTI
    my_cluster_id = BGP->cluster_id;
#else /* BGP_MULTI */
    my_cluster_id = BGP->current_bgp->cluster_id;
#endif /* BGP_MULTI */
    if (!BGP4_BIT_TEST (attr->attribs, PA4_TYPE_CLUSTER_LIST)) {
	attr->cluster_list = LL_Create (0);
	BGP4_BIT_SET (attr->attribs, PA4_TYPE_CLUSTER_LIST);
    }
    LL_Add (attr->cluster_list, (DATA_PTR) my_cluster_id);
    return (0);
}


/* 
 * here is where we actually build update packets and 
 * schedule them to be sent off. NOTE: It is is the responsibility of
 * bgp peer to delete buffer memory!
 */
/* 
 * modify attr to announce neighbors, create packets, and schedule them
 */
static int 
view_announce_peer (bgp_peer_t * peer)
{
    LINKED_LIST * ll_with_prefixes = NULL;
    LINKED_LIST * ll_ann_prefixes;
    bgp_attr_t * attr;
    update_bucket_t * bucket;

    assert (peer);

    pthread_mutex_lock (&peer->update_mutex_lock);
    assert (LL_GetCount (peer->ll_update_out) > 0);
    LL_Iterate (peer->ll_update_out, bucket) {
	if (bucket->attr == NULL) {
	    if (ll_with_prefixes) {
    		bgp_create_pdu (ll_with_prefixes, NULL, NULL,
			    peer, (void_fn_t) bgp_send_update, NULL, 0, 0);
		LL_Destroy (ll_with_prefixes);
	    }
	    ll_with_prefixes = bucket->ll_prefix;
	    /* Delete (bucket); */
	    continue;
	}

	attr = bucket->attr;
	ll_ann_prefixes = bucket->ll_prefix;
	assert (ll_ann_prefixes);
	/* Delete (bucket); */
	/* it could be good if check to see if the next is withdraw or not */

	/* attach reflector_id (iBGP -- iBGP) */
	if (BIT_TEST (peer->options, BGP_INTERNAL) &&
		BIT_TEST (attr->options, BGP_INTERNAL) &&
	        BIT_TEST (peer->options, BGP_ROUTE_REFLECTOR_CLIENT)) {
	    view_attach_reflector_info (peer, attr);
	}
	else {
	    /* drop them since ther are non-transitive */
	    if (BGP4_BIT_TEST (attr->attribs, PA4_TYPE_ORIGINATOR_ID)) {
	        Deref_Prefix (attr->originator);
	        attr->originator = NULL;
	        BGP4_BIT_RESET (attr->attribs, PA4_TYPE_ORIGINATOR_ID);
	    }
	    if (BGP4_BIT_TEST (attr->attribs, PA4_TYPE_CLUSTER_LIST)) {
	        Delete_cluster_list (attr->cluster_list);
	        attr->cluster_list = NULL;
	        BGP4_BIT_RESET (attr->attribs, PA4_TYPE_CLUSTER_LIST);
	    }
	}

	/* nexthop modification if needed */
	if (BGPSIM_TRANSPARENT ||
		BIT_TEST (peer->options, BGP_TRANSPARENT_NEXTHOP)) {
	    /* OK, nothing to do */
	}
	else if (BIT_TEST (peer->options, BGP_NEXTHOP_SELF)) {
	    view_set_nexthop_self (peer, attr);
	}
	else if (attr->type == PROTO_CONNECTED) {
	   /*
	    * always set the next hop with a local address
	    */
	    view_set_nexthop_self (peer, attr);
	}
	else if (BIT_TEST (peer->options, BGP_INTERNAL)) { /* iBGP */
	    if (BGP4_BIT_TEST (attr->attribs, PA4_TYPE_NEXTHOP) &&
	       (prefix_is_unspecified (attr->nexthop->prefix) ||
	        prefix_is_loopback (attr->nexthop->prefix))) {
	        view_set_nexthop_self (peer, attr);
	    }
	}
	else if (BGP4_BIT_TEST (attr->attribs, PA4_TYPE_NEXTHOP)) {
	    if (prefix_is_unspecified (attr->nexthop->prefix) ||
	        prefix_is_loopback (attr->nexthop->prefix)) {
	        view_set_nexthop_self (peer, attr);
	    }
	    else if (!(peer->gateway && peer->gateway->interface &&
	           is_prefix_on (attr->nexthop->prefix, 
				 peer->gateway->interface))) {
	        view_set_nexthop_self (peer, attr);
	    }
	    else if (BIT_TEST (peer->options, BGP_EBGP_MULTIHOP)) {
	        view_set_nexthop_self (peer, attr);
#ifdef HAVE_IPV6
	        if (attr->link_local) {
		    deref_nexthop (attr->link_local);
		    attr->link_local = NULL;
		}
#endif /* HAVE_IPV6 */
	    }

	}
	else {
	    /* replace nexthop with local address */
	    view_set_nexthop_self (peer, attr);
	}

	/* bgpsim */
        if (!BGPSIM_TRANSPARENT &&
	    !BGP4_BIT_SET (attr->attribs, PA4_TYPE_ORIGIN)) {
	    attr->origin = 2;
	    BGP4_BIT_SET (attr->attribs, PA4_TYPE_ORIGIN);
	}

        /* iBGP */
	if (BIT_TEST (peer->options, BGP_INTERNAL)) {
	    if (!BGP4_BIT_TEST (attr->attribs, PA4_TYPE_LOCALPREF)) {
	        attr->local_pref = BGP->default_local_pref;
	        BGP4_BIT_SET (attr->attribs, PA4_TYPE_LOCALPREF);
	    }
	}
	else {
	    BGP4_BIT_RESET (attr->attribs, PA4_TYPE_LOCALPREF);
	    if (BGP4_BIT_TEST (attr->attribs, PA4_TYPE_METRIC))
	        BGP4_BIT_RESET (attr->attribs, PA4_TYPE_METRIC);
	}

	/* route map is applied at the last stage (strongest) */
	if (peer->route_map_out > 0) {
	    bgp_trace_attr2 (attr, peer->trace);
            attr = apply_route_map (peer->route_map_out, attr);
	    trace (TR_TRACE, peer->trace,
		   "route-map %d applied for output\n", peer->route_map_out);
	}

	/* automatic my as prepend is the last */
	/* eBGP */
	if (!BIT_TEST (peer->options, BGP_INTERNAL)) {

            if (!BGPSIM_TRANSPARENT &&
			!BIT_TEST (peer->options, BGP_TRANSPARENT_AS)) {
#ifndef BGP_MULTI
	        attr->aspath = aspath_prepend_as (attr->aspath, BGP->my_as);
#else /* BGP_MULTI */
	        attr->aspath = aspath_prepend_as (attr->aspath, peer->local_bgp->this_as);
#endif /* BGP_MULTI */
	        BGP4_BIT_SET (attr->attribs, PA4_TYPE_ASPATH);
	    }
	}

        bgp_create_pdu (ll_with_prefixes, ll_ann_prefixes, attr,
			        peer, (void_fn_t) bgp_send_update, NULL, 0, 0);
        bgp_deref_attr (attr);
	if (ll_with_prefixes) {
	    LL_Destroy (ll_with_prefixes);
	    ll_with_prefixes = NULL;
	}
	LL_Destroy (ll_ann_prefixes);
    }
    LL_ClearFn (peer->ll_update_out, free);
    pthread_mutex_unlock (&peer->update_mutex_lock);
    if (ll_with_prefixes) {
	bgp_create_pdu (ll_with_prefixes, NULL, NULL,
			peer, (void_fn_t) bgp_send_update, NULL, 0, 0);
	LL_Destroy (ll_with_prefixes);
    }
    return (0);
}


static void
bgp_pick_best (view_t * view, bgp_route_head_t * bgp_route_head, 
	       bgp_route_t * old_best, u_long old_flags)
{
    bgp_route_t *new_best;
    bgp_route_t *imported;

    new_best = LL_GetHead (bgp_route_head->ll_routes);
    imported = LL_GetHead (bgp_route_head->ll_imported);
    assert (new_best);

    if (BIT_TEST (new_best->flags, BGP_RT_FILTERED) &&
        BIT_TEST (old_flags, BGP_RT_FILTERED)) {
	assert (old_best != bgp_route_head->active);
	return;
    }

    else if (BIT_TEST (new_best->flags, BGP_RT_FILTERED) &&
        !BIT_TEST (old_flags, BGP_RT_FILTERED)) {
    	bgp_rt_update_call (bgp_route_head->prefix,
			    NULL, old_best->attr);
	assert (bgp_route_head->active);
	if (old_best == bgp_route_head->active) {
	    if (imported) {
	        bgp_route_head->active = imported;
	        view_add_change_list (view, bgp_route_head, 0);
	    }
	    else {
    		BIT_SET (bgp_route_head->state_bits, VRTS_HOLD);
	        view_add_change_list (view, bgp_route_head, 1);
	    }
	}
    }
    else if (!BIT_TEST (new_best->flags, BGP_RT_FILTERED) &&
        BIT_TEST (old_flags, BGP_RT_FILTERED)) {
	assert (old_best != bgp_route_head->active);
        bgp_rt_update_call (bgp_route_head->prefix,
			    new_best->attr, NULL);
	if (imported == NULL || imported->weight <= new_best->weight) {
	    assert (imported == bgp_route_head->active);
	    bgp_route_head->active = new_best;
	    view_add_change_list (view, bgp_route_head, 0);
	}
    }
    else if (!BIT_TEST (new_best->flags, BGP_RT_FILTERED) &&
        !BIT_TEST (old_flags, BGP_RT_FILTERED)) {
        bgp_rt_update_call (bgp_route_head->prefix,
			    new_best->attr, old_best->attr);

	assert (bgp_route_head->active);
	if (bgp_route_head->active == imported) {
	    if (imported->weight <= new_best->weight) {
	        bgp_route_head->active = new_best;
	        view_add_change_list (view, bgp_route_head, 0);
	    }
	}
	else {
	    bgp_route_head->active = new_best;
	    if (imported != NULL && new_best->weight < imported->weight)
	        bgp_route_head->active = imported;
	    view_add_change_list (view, bgp_route_head, 0);
	}
    }
}


static bgp_route_t *
bgp_implicit_withdraw (view_t * view, bgp_route_t * bgp_route,
		       bgp_attr_t * attr, int weight, u_long flags)
{
    bgp_route_head_t *bgp_route_head;
    bgp_route_t *old_best, *new_bgp_route;

    assert (attr->type == PROTO_BGP);
    bgp_route_head = bgp_route->head;

    if (bgp_route->flags == flags && bgp_route->weight == weight &&
	    bgp_compare_attr (bgp_route->attr, attr) > 0) {

        trace (TR_TRACE, view->trace,
	       "Implicit withdraw but same: %p (ignore)\n",
	       bgp_route_head->prefix);
	/* Do we need to update the time received? */
        return (bgp_route);
    }

    old_best = LL_GetHead (bgp_route_head->ll_routes);
    /* remove but not destroy it */
    LL_RemoveFn (bgp_route_head->ll_routes, bgp_route, NULL);
    new_bgp_route = New_Bgp_Route (bgp_route_head, attr, weight, flags);
    bgp_pick_best (view, bgp_route_head, old_best, old_best->flags);
    Delete_Bgp_Route (bgp_route);
    return (new_bgp_route);
}


/*
 * tentative for a bgp policy routing...
 */
static int
bgp_policy (prefix_t * prefix, bgp_attr_t * attr, bgp_peer_t * peer, int inout)
{
    int num;

    assert (peer);
    if (inout == 0 /* announce */|| inout == 2 /* withdraw */) {
	/* input policy processing */

	/* verify the prefix */
#ifdef HAVE_IPV6
	if (prefix->family == AF_INET) {
#endif /* HAVE_IPV6 */

	    if (prefix->bitlen > 32) {
		trace (TR_PACKET, peer->trace,
		       "  x %p (invalid prefix length)\n", prefix);
		return (0);
	    }
#ifdef HAVE_IPV6
	}
	else {
	    assert (prefix->family == AF_INET6);

	    if (prefix->bitlen > 128) {
		trace (TR_PACKET, peer->trace,
		       "  x %p (invalid prefix length)\n", prefix);
		return (0);
	    }
	}
#endif /* HAVE_IPV6 */

#ifdef notdef
/* bgpsim doesn't put a gateway */
	if (attr->gateway == NULL)
	    return (1);
#endif

#ifdef notdef
/* XXX it might be too strict */
	if (peer->local_addr->family != prefix->family) {
	    trace (TR_PACKET, peer->trace,
		   "  x %p (family %d on %d)\n", prefix,
		   prefix->family, peer->local_addr->family);
	    return (0);
	}
#endif

	/* check distribute-list for in */
	if ((num = peer->dlist_in) > 0) {
	    if (apply_access_list (num, prefix) == 0) {

		trace (TR_PACKET, peer->trace,
		       "  x %p (d-list %d)\n", prefix, num);
		return (0);
	    }
	}

	/* check filter-list for in */
	/* apply for announce only */
	if (inout == 0 && (num = peer->flist_in) > 0) {
	    if (!BGP4_BIT_TEST (attr->attribs, PA4_TYPE_ASPATH) ||
		apply_as_access_list (num, attr->aspath) == 0) {
		trace (TR_PACKET, peer->trace,
		       "  x %p (f-list %d)\n", prefix, num);
		return (0);
	    }
	}
    }
    else {
	/* output policy processing */

#ifdef notdef
/* bgpsim doesn't put a gateway */
	if (attr->gateway == NULL)
	    return (1);
#endif

	/* BGPsim skips these checks */
        if (!BGPSIM_TRANSPARENT &&
		!BIT_TEST (peer->options, BGP_TRANSPARENT_AS)) {

	    /* do not announce to peer who gave us route */
	    if (attr->gateway == peer->gateway) {

		trace (TR_PACKET, peer->trace,
		       "  x %p (split horizon)\n", prefix);
		return (0);
	    }

	    if (BGP4_BIT_TEST (attr->attribs, PA4_TYPE_ORIGINATOR_ID)) {
		if (prefix_tolong (attr->originator) == 
			peer->gateway->routerid) {
		    trace (TR_PACKET, peer->trace,
		           "  x %p (originator %a)\n",
		           prefix, attr->originator);
		    return (0);
		}
	    }

	    /* do not announce to peer who once passed it */
	    if (BGP4_BIT_TEST (attr->attribs, PA4_TYPE_ASPATH) &&
	        bgp_check_aspath_loop (attr->aspath, 
				       peer->gateway->AS)) {

		trace (TR_PACKET, peer->trace,
		       "  x %p (peer as %d in path)\n",
		       prefix, peer->gateway->AS);
		return (0);
	    }

	    /* do not announce to iBGP if larned via iBGP */
	    if (BIT_TEST (peer->options, BGP_INTERNAL) &&
		BIT_TEST (attr->options, BGP_INTERNAL)) {
		/* iBGP -- iBGP case */
		/* non-client to non-client */
		if (!BIT_TEST (attr->options, BGP_ROUTE_REFLECTOR_CLIENT) &&
		    !BIT_TEST (peer->options, BGP_ROUTE_REFLECTOR_CLIENT)) {
		    trace (TR_PACKET, peer->trace,
		           "  x %p (iBGP - iBGP)\n", prefix);
		    return (0);
		}
	    }

	    /* do not announce to iBGP if larned via IGP */
	    if (BIT_TEST (peer->options, BGP_INTERNAL) &&
	           (attr->type == PROTO_RIPNG || attr->type == PROTO_RIP ||
	            attr->type == PROTO_OSPF)) {
		trace (TR_PACKET, peer->trace,
		       "  x %p (IGP - iBGP)\n", prefix);
		return (0);
	    }

	    /* do not announce connected route to peer on the interface */
	    if (attr->type == PROTO_CONNECTED && 
		    attr->gateway->interface &&
		    peer->nexthop == NULL /* immediate neghbors */ &&
		    attr->gateway->interface == 
			peer->gateway->interface) {

		trace (TR_PACKET, peer->trace,
		       "  x %p (on the interface %s)\n",
		       prefix, peer->gateway->interface->name);
		return (0);
	    }
        }

#ifdef notdef
	if (attr->type != PROTO_BGP &&
	    check_bgp_networks (prefix)) {
	    /* OK */
	}

	else if (attr->type != PROTO_BGP &&
	    !BIT_TEST (MRT->redist[attr->type], (1 << PROTO_BGP))) {
	    trace (TR_PACKET, peer->trace,
		   "  x %p (proto %r)\n", prefix, attr->type);
	    return (0);
	}
#endif

#ifdef notdef
/* XXX it might be too strict */
	if (peer->local_addr->family != prefix->family) {
	    trace (TR_PACKET, peer->trace,
		   "  x %p (family %d on %d)\n",
		   prefix, prefix->family, peer->local_addr->family);
	    return (0);
	}
#endif
	/* check distribute-list for out */
	if ((num = peer->dlist_out) > 0) {
	    if (apply_access_list (num, prefix) == 0) {

		trace (TR_PACKET, peer->trace,
		       "  x %p (d-list %d)\n", prefix, num);
		return (0);
	    }
	}

	/* check filter-list for out */
	if ((num = peer->flist_out) > 0) {
	    if (!BGP4_BIT_TEST (attr->attribs, PA4_TYPE_ASPATH) ||
	        apply_as_access_list (num, attr->aspath) == 0) {

		trace (TR_PACKET, peer->trace,
		       "  x %p (f-list %d)\n", prefix, num);
		return (0);
	    }
	}
    }
    return (1);
}


/*
 * process route changes - run policy, and then build and schedule 
 * updates to peers. This is call after receipt of update packets
 */
/* public */
int 
bgp_process_changes (view_t * view)
{
    bgp_peer_t *peer;
    bgp_route_head_t *head;

    /* maybe check that view is open? */

    /* update each peer of view with changes */
    pthread_mutex_lock (&BGP->peers_mutex_lock);
    LL_Iterate (BGP->ll_bgp_peers, peer) {
        update_bucket_t *wbucket = NULL;
        LINKED_LIST *ll_ann = NULL;

	if (BGP->views[peer->view] != view)
	    continue;

	/* see if peer is established */
	if (peer->state != BGPSTATE_ESTABLISHED)
	    continue;

	/* build withdraw list */
	LL_Iterate (view->ll_with_routes, head) {
    	    bgp_route_t *route;

	    /* 
	     * Withdraw 
	     */

	    route = head->active;
	    assert (route);

	    /* do not give withdraw to person who gave us the route */
	    /* if not announced, then do not withdraw */
	    if (route->attr->gateway == peer->gateway ||
		!BITX_TEST (&head->peer_mask, peer->index))
		continue;

	    BITX_RESET (&head->peer_mask, peer->index); /* reset announce bit */
	    if (wbucket == NULL)
	        wbucket = New_Update_Bucket ();
	    LL_Add (wbucket->ll_prefix, Ref_Prefix (head->prefix));
	}

	/* build announce list */
	LL_Iterate (view->ll_ann_routes, head) {
    	    bgp_route_t *route;

	    /* 
	     * Add
	     */

	    route = head->active;
	    assert (route);

	    /* policy */
	    if (!bgp_policy (head->prefix, route->attr, peer, 1)) {
		/* if policy fails and route announced, withdraw the puppy */
		if (BITX_TEST (&head->peer_mask, peer->index)) {
	    	    /* reset announce bit */
		    BITX_RESET (&head->peer_mask, peer->index); 
	    	    if (wbucket == NULL)
	                wbucket = New_Update_Bucket ();
	    	    LL_Add (wbucket->ll_prefix, Ref_Prefix (head->prefix));
		}
		continue;
	    }
	    BITX_SET (&head->peer_mask, peer->index);	/* set announce bit */
	    if (ll_ann == NULL)
		ll_ann = LL_Create (0);
	    LL_Add (ll_ann, route);
	}

	pthread_mutex_lock (&peer->update_mutex_lock);

	if (wbucket) {
	    LL_Append (peer->ll_update_out, wbucket);
	    if (LL_GetCount (peer->ll_update_out) == 1) {
    	        schedule_event2 ("view_announce_peer", peer->schedule, 
    		          (event_fn_t) view_announce_peer, 1, peer);
	    }
	}

	if (ll_ann) {
	    update_bucket_t *bucket;
	    LINKED_LIST *ll_buckets;
	    /* organize by attr type for easy building of packets */
	    ll_buckets = build_update_buckets (ll_ann, NULL);

	    LL_Iterate (ll_buckets, bucket) {
	        bgp_attr_t *attr;

	        /* don't announce if not sent */
	        /* don't announce if we learned it from this gateway */
	        if (bucket->attr->gateway == peer->gateway) {
		    delete_update_bucket (bucket);
		    continue;
	        }

	        attr = bgp_copy_attr (bucket->attr);
                bgp_deref_attr (bucket->attr);
	        bucket->attr = attr;
	        LL_Append (peer->ll_update_out, bucket);
	        if (LL_GetCount (peer->ll_update_out) == 1) {
    	            schedule_event2 ("view_announce_peer", peer->schedule, 
   				     (event_fn_t) view_announce_peer, 1, peer);
	        }
	    }
	    LL_Destroy (ll_buckets);
	    LL_Destroy (ll_ann);
	}

	pthread_mutex_unlock (&peer->update_mutex_lock);
    }
    pthread_mutex_unlock (&BGP->peers_mutex_lock);

    LL_Iterate (view->ll_with_routes, head) {
        assert (BITX_TEST (&head->view_mask, view->viewno));
        BITX_RESET (&head->view_mask, view->viewno);
	/* I don't think both VRTS_HOLD & VRTS_DELETE hold the same time */
	if (BIT_TEST (head->state_bits, VRTS_HOLD)) {
	    head->active = NULL;
	    BIT_RESET (head->state_bits, VRTS_HOLD);
	}
	if (BIT_TEST (head->state_bits, VRTS_DELETE)) {
	/* now active is always connected to either of lists */
	/*  if (head->active)
		Delete_Bgp_Route (head->active); */
	    Delete_Bgp_Route_Head (head);
	}
    }
    LL_Iterate (view->ll_ann_routes, head) {
        assert (BITX_TEST (&head->view_mask, view->viewno));
        BITX_RESET (&head->view_mask, view->viewno);
    }
    LL_ClearFn (view->ll_with_routes, NULL);
    LL_ClearFn (view->ll_ann_routes, NULL);

    return (1);
}


/* public */
int
process_bgp_update (bgp_peer_t * peer, u_char * cp, int length)
{
    prefix_t *prefix;
    int i;
#ifdef BGP_MULTI
    int skip;
#endif /* BGP_MULTI */
    int weight = 0;
    bgp_attr_t *attr;
    LINKED_LIST *ll_announce, *ll_withdraw;

    assert (peer);

#ifdef notdef
if (!BGP->dump_new_format) {
    /*
     * Okay, first of all, we dump the packet and the current route table.
     * So, it may include a packet with errors.
     */
    /* view_open (BGP->views[peer->view]); */
    bgp_write_mrt_msg (peer, MSG_BGP_UPDATE, cp, length);
    /* view_close (BGP->views[peer->view]); */
}
#endif

    if (bgp_process_update_packet (cp, length, peer) < 0) {
	bgp_send_notification (peer, peer->code, peer->subcode);
	return (-1);
    }
    attr = peer->attr; peer->attr = NULL;
    ll_announce =  peer->ll_announce; peer->ll_announce = NULL;
    ll_withdraw =  peer->ll_withdraw; peer->ll_withdraw = NULL;

    if (ll_announce && attr) { /* announce */

#ifndef BGP_MULTI
        if (bgp_check_attr (peer, attr, BGP->my_as) < 0) {
#else /* BGP_MULTI */
        if (bgp_check_attr (peer, attr, peer->local_bgp->this_as) < 0) {
#endif /* BGP_MULTI */
	    /* notification was sent */
	    bgp_deref_attr (attr);
	    attr = NULL;
            if (ll_announce) {
	      /* only discard the announcements, but eventually close 
		 the connection to withdraw all heard from the peer */
		if (LL_GetCount (ll_announce) == 1) {
		    prefix = LL_GetHead (ll_announce);
	    	    trace (TR_WARN, peer->trace,
	           	   "dropping off the announce: %p\n", prefix);
		}
		else if (LL_GetCount (ll_announce) > 1) {
	    	    trace (TR_WARN, peer->trace,
	           	    "dropping off the announce:\n");
		    LL_Iterate (ll_announce, prefix) {
        	        trace (TR_WARN, peer->trace, "  %p\n", prefix);
		    }
		}
	        LL_Destroy (ll_announce);
		ll_announce = NULL;
	    }
	    /* cisco sends me a route which has my own as,
	       so quietly discards them for now */
	    if (ll_withdraw == NULL)
		return (0);
	    /* go through if there is withdraw */
        }

	/* set nexthop differntly for IBGP and EBGP -- make sure attr
	 * still exists!
	 */
	if (attr != NULL) {

	    /* copy peer's options */
	    attr->options = peer->options;

	    if (BIT_TEST (peer->options, BGP_PEER_IS_MULTIHOP)) {
	        assert (peer->nexthop);
	        attr->nexthop2 = ref_nexthop (peer->nexthop);

	        trace (TR_PACKET, peer->trace, "nexthop2 %a (multihop)\n", 
		       attr->nexthop2->prefix);
	    }
	    else if (BIT_TEST (peer->options, BGP_INTERNAL)) {
	        /* This is violation of BGP4 RFC. MRTd neglects nexthop of iBGP
	           routes that should be evaluated by looking up IGP routing 
	           table. Therefore, MRTd doesn't require distributing a route 
	           for eBGP neighbor's nexthop in IGP */

	        attr->nexthop2 = ref_nexthop (peer->gateway);
	        trace (TR_PACKET, peer->trace, "nexthop2 %a (iBGP)\n", 
		       attr->nexthop2->prefix);
	    }
	}
    }

    if (attr == NULL) {
	attr = bgp_new_attr (PROTO_BGP);
	trace (TR_PACKET, peer->trace,
	       "Creating a pseudo attribute for the withdraw\n");
    }
    attr->gateway = peer->gateway;

    /* XXX */
    if (peer->default_weight >= 0)
	weight = peer->default_weight;

    if (ll_announce || ll_withdraw) {
	/* iterate through views adding the announce/withdraws */
	for (i = 0; i < MAX_BGP_VIEWS; i++) {
	    if (BGP->views[i] == NULL)
		continue;
#ifdef BGP_MULTI
	    if (i != peer->local_bgp->view_no)
	      continue;
#endif /* BGP_MULTI */

	    view_open (BGP->views[i]);

	    if (ll_withdraw) {

		LL_Iterate (ll_withdraw, prefix) {
		    if (BGP->views[i]->family != prefix->family)
			continue;
#ifdef notdef
		    /* apply policy */
		    if (!bgp_policy (prefix, attr, peer, 2))
			continue;
#endif
		    bgp_del_route (BGP->views[i], prefix, attr);
		}
	    }

	    if (ll_announce) {
		LL_Iterate (ll_announce, prefix) {
		    u_long flags = 0;

#ifdef BGP_MULTI
#ifdef undef
		    skip = 0;
		    LL_Iterate(BGP->views[i]->ll_gateways, gateway) {
		      if (compare (gateway, attr->gateway)) {
			skip = 1;
			continue;
		      }
		    }

		    if (skip == 1)
		      continue;
#endif /* undef */
#endif /* BGP_MULTI */
		    if (BGP->views[i]->family != prefix->family)
		      continue;


		    /* apply route_map_in first */
		    if (peer->route_map_in > 0) {
			attr->original = bgp_copy_attr (attr);
		        apply_route_map (peer->route_map_in, attr);
			trace (TR_TRACE, peer->trace,
	       		       "route-map %d applied for input\n",
	       		       peer->route_map_in);
			bgp_trace_attr2 (attr, peer->trace);
		    }
		    /* XXX weight should be changed by route map */
		    /* apply policy */
		    if (!bgp_policy (prefix, attr, peer, 0)) {
			BIT_SET (flags, BGP_RT_FILTERED);
		    }
		    bgp_add_route2 (BGP->views[i], prefix, attr, weight,
				    flags);
		}
	    }

	    /* process change list -- send out updates to peers */
	    bgp_process_changes (BGP->views[i]);

	    view_close (BGP->views[i]);
	}
    }

    if (attr)
	bgp_deref_attr (attr);
    if (ll_withdraw)
	LL_Destroy (ll_withdraw);
    if (ll_announce)
	LL_Destroy (ll_announce);

    return (1);
}


/* public */
void
bgp_re_evaluate_in (bgp_peer_t * peer)
{
    int i;
    int weight = 0;

    assert (peer);

    if (peer->default_weight >= 0)
	weight = peer->default_weight;

    for (i = 0; i < MAX_BGP_VIEWS; i++) {
        bgp_route_head_t *route_head;
	view_t *view;

	if ((view = BGP->views[i]) == NULL)
	    continue;

	view_open (view);

	VIEW_RADIX_WALK (view, route_head) {
    	    bgp_route_t *bgp_route;
	    bgp_route_t *old_best;
	    int old_flags = 0;
	    int changed = 0;

	    old_best = LL_GetHead (route_head->ll_routes);
	    if (old_best != NULL)
		old_flags = old_best->flags;

            LL_Iterate (route_head->ll_routes, bgp_route) {
    		bgp_attr_t *attr = bgp_route->attr;
    		prefix_t *prefix = bgp_route->head->prefix;

		if (attr->original) {
		    bgp_attr_t *original = attr->original;
		    bgp_deref_attr (bgp_route->attr);
		    bgp_route->attr = original;
		}

	        /* apply route_map_in first */
	        if (peer->route_map_in > 0) {
		    attr->original = bgp_copy_attr (attr);
		    apply_route_map (peer->route_map_in, attr);
		    trace (TR_TRACE, peer->trace,
	       	           "route-map %d applied for input\n",
	       		    peer->route_map_in);
		    bgp_trace_attr2 (attr, peer->trace);
		    changed++;
	        }

		/* XXX should be effected by bgp_policy() */
		if (bgp_route->weight != weight) {
		    bgp_route->weight = weight;
		    changed++;
		}

	        /* apply policy */
	        if (!bgp_policy (prefix, attr, peer, 0)) {
		    if (!BIT_TEST (bgp_route->flags, BGP_RT_FILTERED)) {
		        BIT_SET (bgp_route->flags, BGP_RT_FILTERED);
			changed++;
		    }
		}
		else {
		    if (BIT_TEST (bgp_route->flags, BGP_RT_FILTERED)) {
		        BIT_RESET (bgp_route->flags, BGP_RT_FILTERED);
			changed++;
		    }
		}
	    }
	    if (changed) {
		LL_Sort (route_head->ll_routes);
		bgp_pick_best (view, route_head, old_best, old_flags);
	    }

	} VIEW_RADIX_WALK_END;

	/* process change list -- send out updates to peers */
	bgp_process_changes (view);
	view_close (view);
    }
}


static int
origin_merge (o1, o2)
{
    int origin = 0;

    if (o1 >= 2 || o2 >= 2 /* incomplete */ )
	origin = 2;
    else if (o1 == 1 || o2 == 1 /* egp */ )
	origin = 1;
    return (origin);
}


/* merge attr into attn, then return attn */
static bgp_attr_t *
view_aggregate_attr_merge (bgp_attr_t *attn, bgp_attr_t *attr, 
			   aggregate_t *agg, aspath_t *tail)
{
	    assert (attn);
	    assert (attr);
#ifdef notdef
		if (attn->gateway != attr->gateway)
		    attn->gateway = NULL;
#endif

		if (BGP4_BIT_TEST (attn->attribs, PA4_TYPE_NEXTHOP) &&
		    BGP4_BIT_TEST (attr->attribs, PA4_TYPE_NEXTHOP) &&
		    attn->nexthop == attr->nexthop) {
			/* OK */
		}
		else {
		    if (BGP4_BIT_TEST (attn->attribs, PA4_TYPE_NEXTHOP)) {
		        deref_nexthop (attn->nexthop);
		        attn->nexthop = NULL;
		        BGP4_BIT_RESET (attn->attribs, PA4_TYPE_NEXTHOP);
		    }
		}

		if (BGP4_BIT_TEST (attn->attribs, PA4_TYPE_ORIGIN) ||
		    BGP4_BIT_TEST (attr->attribs, PA4_TYPE_ORIGIN)) {
		    attn->origin = origin_merge (attn->origin, attr->origin);
		    BGP4_BIT_SET (attn->attribs, PA4_TYPE_ORIGIN);
		}

		if (BGP4_BIT_TEST (attn->attribs, PA4_TYPE_ASPATH) ||
		    BGP4_BIT_TEST (attr->attribs, PA4_TYPE_ASPATH)) {
                    attn->aspath = aspath_merge (attn->aspath, attr->aspath, tail);
		    BGP4_BIT_SET (attn->attribs, PA4_TYPE_ASPATH);
		}

#ifdef notdef
		/* is it non-transitive ??? */
		if (BGP4_BIT_TEST (attn->attribs, PA4_TYPE_METRIC) &&
		    BGP4_BIT_TEST (attr->attribs, PA4_TYPE_METRIC) &&
		    attn->multiexit == attr->multiexit) {
			/* OK */
		}
		else {
		    BGP4_BIT_RESET (attn->attribs, PA4_TYPE_METRIC);
		}
#endif

		if (BGP4_BIT_TEST (attn->attribs, PA4_TYPE_LOCALPREF) &&
		    BGP4_BIT_TEST (attr->attribs, PA4_TYPE_LOCALPREF) &&
		    attn->local_pref == attr->local_pref) {
			/* OK */
		}
		else {
		    BGP4_BIT_RESET (attn->attribs, PA4_TYPE_LOCALPREF);
		}

		if (BGP4_BIT_TEST (attn->attribs, PA4_TYPE_DPA) &&
		    BGP4_BIT_TEST (attr->attribs, PA4_TYPE_DPA) &&
		    attn->dpa.as == attr->dpa.as &&
		    attn->dpa.value == attr->dpa.value) {
			/* OK */
		}
		else {
		    BGP4_BIT_RESET (attn->attribs, PA4_TYPE_DPA);
		}

		if (BGP4_BIT_TEST (attn->attribs, PA4_TYPE_ATOMICAGG) ||
		    BGP4_BIT_TEST (attr->attribs, PA4_TYPE_ATOMICAGG))
        	    BGP4_BIT_SET (attn->attribs, PA4_TYPE_ATOMICAGG);

#ifdef HAVE_IPV6
		if (attn->link_local && attr->link_local &&
		    attn->link_local == attr->link_local) {
			/* OK */
		}
		else {
		    if (attn->link_local) {
		        deref_nexthop (attn->link_local);
		        attn->link_local = NULL;
		    }
		}
		if (attn->nexthop4 && attr->nexthop4 &&
		    attn->nexthop4 == attr->nexthop4) {
			/* OK */
		}
		else {
		    if (attn->nexthop4) {
		        deref_nexthop (attn->nexthop4);
		        attn->nexthop4 = NULL;
		    }
		}
#endif /* HAVE_IPV6 */
#ifdef notdef
		if (attn->nexthop2 && attr->nexthop2 &&
		    attn->nexthop2 == attr->nexthop2) {
			/* OK */
		}
		else {
		    if (attn->nexthop2) {
		        deref_nexthop (attn->nexthop2);
		        attn->nexthop2 = NULL;
		    }
		}
#endif
    return (attn);
}


/* finalize attn, then return attn */
static bgp_attr_t *
view_aggregate_attr_final (bgp_attr_t *attn, aggregate_t *agg, aspath_t *tail)
{
    assert (attn);

    /* local gateway */
    attn->gateway = agg->attr->gateway;

    if (attn->nexthop2) {
        /* don't inherit nexthop2 */
         deref_nexthop (attn->nexthop2);
    }
    /* always local */
    attn->nexthop2 = ref_nexthop (agg->attr->gateway);

    if (!BGP4_BIT_TEST (attn->attribs, PA4_TYPE_NEXTHOP)) {
        attn->nexthop = ref_nexthop (agg->attr->nexthop);
        BGP4_BIT_SET (attn->attribs, PA4_TYPE_NEXTHOP);
    }

    /* ignore aggregator of contributing route */
    if (BGP4_BIT_TEST (attn->attribs, PA4_TYPE_AGGREGATOR)) {
        Deref_Prefix (attn->aggregator.prefix);
        attn->aggregator.prefix = NULL;
        BGP4_BIT_RESET (attn->attribs, PA4_TYPE_AGGREGATOR);
    }

    if (BGP4_BIT_TEST (agg->attr->attribs, PA4_TYPE_AGGREGATOR)) {
        attn->aggregator.as = agg->attr->aggregator.as;
        attn->aggregator.prefix = Ref_Prefix (agg->attr->aggregator.prefix);
        BGP4_BIT_SET (attn->attribs, PA4_TYPE_AGGREGATOR);
    }

    /* ignore communities of contributing route */
    if (BGP4_BIT_TEST (attn->attribs, PA4_TYPE_COMMUNITY)) {
        Delete_community (attn->community);
        attn->community = NULL;
        BGP4_BIT_RESET (attn->attribs, PA4_TYPE_COMMUNITY);
    }

#ifdef HAVE_IPV6
    if (attn->link_local == NULL && agg->attr->link_local)
        attn->link_local = ref_nexthop (agg->attr->link_local);
    if (attn->nexthop4 == NULL && agg->attr->nexthop4)
        attn->nexthop4 = ref_nexthop (agg->attr->nexthop4);
#endif /* HAVE_IPV6 */

/* CHECK_AGG: */

    if (BIT_TEST (agg->option, BGP_AGGOPT_AS_SET)) {
	if (tail && LL_GetCount (tail)) {
	    attn->aspath = aspath_append (attn->aspath, tail);
            BGP4_BIT_SET (attn->attribs, PA4_TYPE_ASPATH);
	}
        return (attn);
    }

    /* atmoc-aggregate is already set if at least one of routes has */

    if (BGP4_BIT_TEST (attn->attribs, PA4_TYPE_ATOMICAGG))
        return (attn);

    if (BIT_TEST (agg->option, BGP_AGGOPT_SUMMARY_ONLY) &&
	    tail && LL_GetCount (tail)) {
	/* as path information lost, so set atomic-aggregate */
        BGP4_BIT_SET (attn->attribs, PA4_TYPE_ATOMICAGG);
    }

    return (attn);
}


static bgp_attr_t *
view_compute_agg (view_t *view, aggregate_t *agg, 
		  bgp_route_head_t *bgp_route_head)
{
    radix_node_t *node;
    aspath_t *tail = NULL;
    bgp_attr_t *attr = NULL;
    int n = 0;

    assert (view); assert (agg);
    assert (bgp_route_head);
    assert (bgp_route_head->radix_node);
    RADIX_WALK (bgp_route_head->radix_node, node) {

        if (n++ == 0) {
            /* the first iteration is aggregate route itself */
        }
        else {
	    bgp_route_head_t *route_head;
	    bgp_route_t *route;

	    route_head = RADIX_DATA_GET (node, bgp_route_head_t);
	if ((route = route_head->active) != NULL) {

	    if (BIT_TEST (agg->option, BGP_AGGOPT_SUMMARY_ONLY) &&
	       !BIT_TEST (bgp_route_head->state_bits, VRTS_SUPPRESS)) {
		if (!BIT_TEST (route_head->state_bits, VRTS_SUPPRESS)) {
    		    /* add the route head to the change list (withdraw) */
    		    view_add_ll_list (view, route_head, 1);
		    BIT_SET (route_head->state_bits, VRTS_SUPPRESS);
		}
	    }

	    trace (TR_TRACE, view->trace,
	           "AGG merge attr for %p from %p\n",
	           agg->prefix, route_head->prefix);

	    if (attr == NULL) {
	        attr = bgp_copy_attr (route->attr);
	        attr->type = PROTO_BGP;
	    }
	    else {
		if (tail == NULL)
		    tail = New_ASPATH ();
	        attr = view_aggregate_attr_merge (attr, route->attr, agg, 
						  tail);

	        if (BIT_TEST (route->flags, BGP_RT_AGGREGATED))
		    RADIX_WALK_BREAK;
	    }
        }
	}
    } RADIX_WALK_END;

    if (attr == NULL)
	return (NULL);

    attr = view_aggregate_attr_final (attr, agg, tail);
    if (tail) Delete_ASPATH (tail);

    trace (TR_TRACE, view->trace, "AGG merge result for %p\n", agg->prefix);
    bgp_trace_attr2 (attr, view->trace);
    return (attr);
}


/* public */
aggregate_t *
view_add_aggregate (view_t * view, prefix_t * prefix, u_long opt)
{
    aggregate_t *agg;
    radix_node_t *radix_node;
    bgp_route_head_t *bgp_route_head;
    bgp_attr_t *attr;

    assert (view);
    assert (prefix);
    assert (view->family == prefix->family);

    radix_node = radix_lookup (view->agg_tree, prefix);
    assert (radix_node);

    if ((agg = RADIX_DATA_GET (radix_node, aggregate_t)) != NULL) {
	trace (TR_TRACE, view->trace,
	       "Aggregate Entry: %p already exist\n", prefix);
	return (agg);
    }

    agg = New (aggregate_t);
    agg->prefix = Ref_Prefix (prefix);
    agg->option = opt;
    RADIX_DATA_SET (radix_node, agg);
    agg->radix_node = radix_node;
    agg->route = NULL;

    {
	static prefix_t *my_id_prefix = NULL;
	static nexthop_t *my_id_nexthop = NULL;
	static nexthop_t *nexthop_self = NULL;
	static gateway_t *gateway_self = NULL;
#ifdef HAVE_IPV6
	static nexthop_t *nexthop_self6 = NULL;
	static nexthop_t *linklocal_self6 = NULL;
	static gateway_t *gateway_self6 = NULL;
#endif /* HAVE_IPV6 */
	prefix_t *prefix;

	if (my_id_prefix == NULL) {
#ifndef BGP_MULTI
	    my_id_prefix = New_Prefix (AF_INET, &BGP->my_id, 32);
#else /* BGP_MULTI */
	    my_id_prefix = New_Prefix (AF_INET, &BGP->current_bgp->this_id, 32);
#endif /* BGP_MULTI */

	    my_id_nexthop = add_nexthop (my_id_prefix, 
					 find_interface (my_id_prefix));
	    prefix = ascii2prefix (AF_INET, "127.0.0.1/32");
	    nexthop_self = add_nexthop (prefix, find_interface (prefix));
	    /* nexthop_self = ascii2prefix (AF_INET, "0.0.0.0/32"); */ /* ank */
	    gateway_self = ref_nexthop (nexthop_self);
#ifndef BGP_MULTI
	    gateway_self->AS = BGP->my_as; /* to avoid iBGP loop */
#else /* BGP_MULTI */
	    /* I should try to get away from this BGP->current_as stuff
	       if possible.  I have a bad feeling that it won't work so
	       well as more changes are made. -- binkertn */
	    gateway_self->AS = BGP->current_bgp->this_as; /* to avoid iBGP loop */
#endif /* BGP_MULTI */
	    Deref_Prefix (prefix);
#ifdef HAVE_IPV6
	    prefix = ascii2prefix (AF_INET6, "::1/128");
	    nexthop_self6 = add_nexthop (prefix, find_interface (prefix));
	    /* linklocal_self6 = ascii2prefix (AF_INET6, "fe80::1/128"); */
	    /* nexthop_self6 = ascii2prefix (AF_INET6, "::/128"); */ /* ank */
	    /* linklocal_self6 = ascii2prefix (AF_INET6, "::/128"); */ /* ank */
	    gateway_self6 = ref_nexthop (nexthop_self6);
#ifndef BGP_MULTI
	    gateway_self6->AS = BGP->my_as; /* to avoid iBGP loop */
#else /* BGP_MULTI */
	    /* I should try to get away from this BGP->current_as stuff
	       if possible.  I have a bad feeling that it won't work so
	       well as more changes are made. -- binkertn */
	    gateway_self6->AS = BGP->current_bgp->this_as; /* to avoid iBGP loop */
#endif /* BGP_MULTI */

	    Deref_Prefix (prefix);
#endif /* HAVE_IPV6 */
	}

	agg->attr = bgp_new_attr (PROTO_BGP);
#ifdef HAVE_IPV6
	if (agg->prefix->family == AF_INET6) {
	    agg->attr->nexthop = ref_nexthop (nexthop_self6);
	    agg->attr->link_local = ref_nexthop (linklocal_self6);
	    agg->attr->nexthop4 = ref_nexthop (my_id_nexthop);
	    agg->attr->gateway = gateway_self6;
	}
	else
#endif /* HAVE_IPV6 */
	{
	    agg->attr->nexthop = ref_nexthop (nexthop_self);
	    agg->attr->gateway = gateway_self;
	}
	BGP4_BIT_SET (agg->attr->attribs, PA4_TYPE_NEXTHOP);
	agg->attr->aspath = NULL;
	BGP4_BIT_SET (agg->attr->attribs, PA4_TYPE_ASPATH);
	agg->attr->origin = 3; /* Invalid. Should not be exported in this form
                                  in any case! -- alexey */;
	BGP4_BIT_SET (agg->attr->attribs, PA4_TYPE_ORIGIN);
#ifdef notdef
	if (BIT_TEST (opt, BGP_AGGOPT_SUMMARY_ONLY))
	    BGP4_BIT_SET (agg->attr->attribs, PA4_TYPE_ATOMICAGG);
	/* BIT_TEST (opt, BGP_AGGOPT_AS_SET); */
#endif
#ifndef BGP_MULTI
	agg->attr->aggregator.as = BGP->my_as;
#else /* BGP_MULTI */
	agg->attr->aggregator.as = BGP->current_bgp->this_as;
#endif /* BGP_MULTI */
	agg->attr->aggregator.prefix = Ref_Prefix (my_id_prefix);
	BGP4_BIT_SET (agg->attr->attribs, PA4_TYPE_AGGREGATOR);
    }
    trace (TR_TRACE, view->trace, "AGG New Entry: %p\n", agg->prefix);
    bgp_trace_attr2 (agg->attr, view->trace);

    /* I need to add a route_head to locate it among the radix tree
       and to see if there are more specific routes below. */
    bgp_route_head = view_get_route_head (view, agg->prefix);
    if (BIT_TEST (agg->option, BGP_AGGOPT_SUMMARY_ONLY))
        BIT_SET (bgp_route_head->state_bits, VRTS_SUMMARY);

    attr = view_compute_agg (view, agg, bgp_route_head);
    if (attr == NULL) {
	if (bgp_route_head->active == NULL) {
	    view_remove_route_head (view, bgp_route_head);
            Delete_Bgp_Route_Head (bgp_route_head);
	}
	return (agg);
    }

    agg->route = view_add_bgp_route (view, bgp_route_head, attr, 0,
				     BGP_RT_AGGREGATED);
    bgp_deref_attr (attr);

    bgp_process_changes (view);
    return (agg);
}


/* public */
int
view_del_aggregate (view_t * view, prefix_t * prefix)
{
    aggregate_t *agg;
    radix_node_t *radix_node;

    assert (view);
    assert (prefix);
    assert (view->family == prefix->family);

    radix_node = radix_search_exact (view->agg_tree, prefix);
    if (radix_node == NULL) {
	trace (TR_TRACE, view->trace,
	       "Aggregate Entry: %p not found\n", prefix);
	return (-1);
    }

    radix_remove (view->agg_tree, radix_node);
    agg = RADIX_DATA_GET (radix_node, aggregate_t);

    if (agg->route) {
	radix_node_t *node;

	if (!BIT_TEST (agg->route->head->state_bits, VRTS_SUPPRESS) &&
    	     BIT_TEST (agg->option, BGP_AGGOPT_SUMMARY_ONLY)) {

            RADIX_WALK (agg->route->head->radix_node, node) {
                bgp_route_head_t *route_head;
                route_head = RADIX_DATA_GET (node, bgp_route_head_t);

		if (BIT_TEST (route_head->state_bits, VRTS_SUPPRESS)) {
    		    /* add the route head to the change list (announce) */
    		    view_add_ll_list (view, route_head, 0);
		    BIT_RESET (route_head->state_bits, VRTS_SUPPRESS);
		}
            } RADIX_WALK_END;
	}
	view_delete_bgp_route (view, agg->route);
        bgp_process_changes (view);
    }
    Deref_Prefix (agg->prefix);
    bgp_deref_attr (agg->attr);
    Delete (agg);
    return (1);
}


/* public */
void
view_print_aggregate (int viewno, void (*fn)())
{
    aggregate_t *agg;
    radix_node_t *node;
    view_t *view = BGP->views[viewno];

    if (view == NULL || view->agg_tree->head == NULL)
	return;

    RADIX_WALK (view->agg_tree->head, node) {
         agg = RADIX_DATA_GET (node, aggregate_t);
        (*fn) (agg->prefix, agg->option, viewno);
   } RADIX_WALK_END;
}


#ifdef notdef
static int
prefix_compare_inclusive (prefix_t * prefix, prefix_t * it)
{
    if (prefix->family != it->family)
	return (0);
    if (prefix->bitlen >= it->bitlen)	/* should not the same */
	return (0);

    return (comp_with_mask (prefix_tochar (prefix),
			    prefix_tochar (it),
			    prefix->bitlen));
}
#endif


/* pseudo code Alexey (kuznet@ms2.inr.ac.ru) provide me:

> * If a route, contributing to an aggregate changes, added, removed, then:
>
>   * Find all routes, contributing to aggregate.
>   * Get the longest common AS_SEQ. Denote it AS_SEQ_HEAD.
>   * Tails of all aspathes are gathered to one AS_SET: AS_SEQ_TAIL.
>   * If AS_SEQ_TAIL is empty, goto CHECK_AAG
>   * If "as-set" is not specified, drop AS_SEQ_TAIL, set origin
>     to unspecified and goto CHECK_AAG
>
> CHECK_AAG:
>   * If atomic-aggregate is set at least on one route then
>       set atomic-agg on aggregate and RETURN.
>     * Now no routes have atomic-aggregate attr *
>   * If AS_SEQ_TAIL was empty or as-set was specified; RETURN
>   * If some routes are suppressed (by summary-only or by suppress-map),
>     and a suppresses route has AS in its aspath not present
>     in aspath of aggregate, set atomic-aggregate on aggregate.
>
> One more note: atomic aggregate should not be referenced anywhere outside
> this piece of pseudocode, because mrt does not make deaggregation.

*/


static aggregate_t *
view_check_aggregate (view_t * view, prefix_t *prefix, int withdraw)
{
    aggregate_t *agg;
    radix_node_t *node;
    bgp_route_head_t *bgp_route_head;
    bgp_attr_t *attr;

    assert (view);
    assert (prefix);
    assert (view->family == prefix->family); /* XXX */

    node = radix_search_best2 (view->agg_tree, prefix, 0);
    if (node == NULL) {
	return (NULL);
    }
    assert (prefix_compare (prefix, node->prefix) == 0);

    agg = RADIX_DATA_GET (node, aggregate_t);
    assert (agg);

    trace (TR_TRACE, view->trace,
	   "AGG contribute: %p -> %p (%s)\n",
	   prefix, agg->prefix, (withdraw) ? "withdraw" : "announce");

    /* the first route causing the aggregate route */
    if (agg->route == NULL) {

	assert (withdraw == 0);
	trace (TR_TRACE, view->trace,
		"AGG activate %p triggered by %p\n", agg->prefix, prefix);

	bgp_route_head = view_get_route_head (view, agg->prefix);
        if (BIT_TEST (agg->option, BGP_AGGOPT_SUMMARY_ONLY))
	    BIT_SET (bgp_route_head->state_bits, VRTS_SUMMARY);
    }
    else {
	bgp_route_head = agg->route->head;
    }

    attr = view_compute_agg (view, agg, bgp_route_head);
    if (attr == NULL) {
	assert (withdraw);
	view_delete_bgp_route (view, agg->route);
	agg->route = NULL;
	/* reseting after deleting the route would be safer */
        if (BIT_TEST (agg->option, BGP_AGGOPT_SUMMARY_ONLY))
	    BIT_RESET (bgp_route_head->state_bits, VRTS_SUMMARY);
	return (NULL);
    }

    if (agg->route == NULL) {
        agg->route = view_add_bgp_route (view, bgp_route_head, attr, 0,
					 BGP_RT_AGGREGATED);
    }
    else {
        agg->route = bgp_implicit_withdraw (view, agg->route, attr, BGP_PREF,
					    BGP_RT_AGGREGATED);
    }
    bgp_deref_attr (attr);
    return (agg);
}


static bgp_route_t *
bgp_add_route2 (view_t * view, prefix_t * prefix, bgp_attr_t * attr, 
		int weight, u_long flags)
{
    bgp_route_t *bgp_route;
    bgp_route_head_t *bgp_route_head;

    assert (view);
    assert (prefix);
    assert (attr);
    if (view->family != prefix->family) {
	trace (TR_ERROR, view->trace,
	       "Not Accept: %p weight %d proto %r\n",
	       prefix, weight, attr->type);
	return (NULL);
    }

    assert (pthread_mutex_trylock (&view->mutex_lock));
    bgp_route_head = view_get_route_head (view, prefix);

    if (attr->type != PROTO_BGP) {	/* extern (imported) route */
        bgp_route_t *new_best, *old_best, *bgp_best;

	old_best = LL_GetHead (bgp_route_head->ll_imported);

	LL_Iterate (bgp_route_head->ll_imported, bgp_route) {
	    if (bgp_route->attr->type == attr->type)
		break;
	}

	if (bgp_route) {
	    LL_Remove (bgp_route_head->ll_imported, bgp_route);
	}
	bgp_route = New_Bgp_Route (bgp_route_head, attr, weight, flags);
	new_best = LL_GetHead (bgp_route_head->ll_imported);

	if (old_best == new_best) {
	    /* the didn't change */
	    return (bgp_route);
	}

	bgp_best = LL_GetHead (bgp_route_head->ll_routes);

	if (old_best && bgp_route_head->active == old_best) {
	    bgp_route_head->active = new_best;
	    if (bgp_best && bgp_route_head->active->weight <= bgp_best->weight)
		bgp_route_head->active = bgp_best;
	    view_add_change_list (view, bgp_route_head, 0);
	}
	else {
	    if (bgp_route_head->active == NULL ||
		    bgp_route_head->active->weight < new_best->weight) {
		bgp_route_head->active = new_best;
		view_add_change_list (view, bgp_route_head, 0);
	    }
	}

	return (bgp_route);
    }

    /* check to see if we already got this route from the gateway 
     *  if we have, then this is an implicit withdraw of the old route and
     *  we add the new route */
    LL_Iterate (bgp_route_head->ll_routes, bgp_route) {

	if (bgp_route->attr->gateway == attr->gateway)
	    return (bgp_implicit_withdraw (view, bgp_route, attr, 
					   weight, flags));
	/* set implicit withdraw flag */
	/* we need a special flag because a host due to policy may not
	   accept the new route, then we have to explicitly withdraw this
	   prefix
	 */
    }

    return (view_add_bgp_route (view, bgp_route_head, attr, weight, flags));
}


/* public */
bgp_route_t *
bgp_add_route (view_t * view, prefix_t * prefix, bgp_attr_t * attr)
{
    int weight = 0;
    if (attr->type != PROTO_BGP)
	weight = BGP_ORIGINATE_WEIGHT;
    return (bgp_add_route2 (view, prefix, attr, weight, 0));
}


/* public */
int
bgp_del_route (view_t * view, prefix_t * prefix, bgp_attr_t * attr)
{
    bgp_route_t *bgp_route;
    bgp_route_head_t *bgp_route_head;

    assert (view);
    assert (prefix);
    assert (attr);
    if (view->family != prefix->family) {
	trace (TR_ERROR, view->trace, "Not Accept: %p proto %r\n",
	       prefix, attr->type);
	return (0);
    }

    assert (pthread_mutex_trylock (&view->mutex_lock));

    bgp_route_head = view_find_route_head (view, prefix);
    if (bgp_route_head == NULL) {
	/* peer can not know if his announce was filtered or not */
	trace (TR_TRACE, view->trace, "Delete: %p not found\n", prefix);
	return (0);
    }

    if (attr->type != PROTO_BGP) {	/* extern (imported) route */

        bgp_route_t *imported = LL_GetHead (bgp_route_head->ll_imported);

	LL_Iterate (bgp_route_head->ll_imported, bgp_route) {
	    if (bgp_route->attr->type == attr->type)
		break;
	}

	if (bgp_route == NULL) {
	    trace (TR_TRACE, view->trace,
		   "Delete: %p proto %r not found\n", prefix, attr->type);
	    return (-1);
	}

	if (LL_GetCount (bgp_route_head->ll_routes) <= 0 &&
		LL_GetCount (bgp_route_head->ll_imported) == 1) {
	    assert (bgp_route_head->active == imported);
	    assert (bgp_route_head->active == bgp_route);
	    /* remove route_head first */
	    view_remove_route_head (view, bgp_route_head);
	    /* add the route head to the change list */
	    view_add_change_list (view, bgp_route_head, 1);
	}
	else {
	    LL_Remove (bgp_route_head->ll_imported, bgp_route);
	    if (bgp_route_head->active == imported) {
        	bgp_route_t *bgp_best = LL_GetHead (bgp_route_head->ll_routes);
		bgp_route_head->active = 
		    LL_GetHead (bgp_route_head->ll_imported);
		if (bgp_best && !BIT_TEST (bgp_best->flags, BGP_RT_FILTERED) &&
			(bgp_route_head->active == NULL || 
			bgp_route_head->active->weight < bgp_best->weight))
		    bgp_route_head->active = bgp_best;
		assert (bgp_route_head->active);
	        /* add the route head to the change list */
	        view_add_change_list (view, bgp_route_head, 0);
	    }
	}
	return (1);
    }

    LL_Iterate (bgp_route_head->ll_routes, bgp_route) {
	if (bgp_route->attr->gateway == attr->gateway) {
	    view_delete_bgp_route (view, bgp_route);
	    return (1);
	}
    }
    trace (TR_WARN, view->trace,
	   "Delete: %p not found\n", prefix);
    return (0);
}


#ifdef notdef
/* public */
int 
view_close (view_t * view)
{

    pthread_mutex_unlock (&view->mutex_lock);
    return (1);
}


/* public */
int 
view_open (view_t * view)
{

    if (pthread_mutex_trylock (&view->mutex_lock) != 0) {
	trace (TR_DEBUG, view->trace, "Going to block in view_open\n");
	pthread_mutex_lock (&view->mutex_lock);
    }
    return (1);
}
#endif


/*
 * this is used in finding nexthop for multihop peer
 */
/* public */
prefix_t *
view_find_best_route (view_t * view, prefix_t * prefix)
{
    radix_node_t *radix_node;
    bgp_route_head_t *route_head;
    bgp_attr_t *attr = NULL;

    assert (view);
    assert (prefix);
    radix_node = radix_search_best (view->radix_tree, prefix);
    if (radix_node) {
	assert (radix_node->data);
	route_head = RADIX_DATA_GET (radix_node, bgp_route_head_t);
	assert (route_head);
	if (route_head->active) {
	    attr = route_head->active->attr;
	    if (attr && BGP4_BIT_TEST (attr->attribs, PA4_TYPE_NEXTHOP))
	        return (attr->nexthop->prefix);
	}
    }
    return (NULL);
}


/* 
 * given a prefix, find route node entry
 * return NULL if one does not exist
 */
/* public */
bgp_route_t *
view_find_bgp_active (view_t * view, prefix_t * prefix)
{
    bgp_route_head_t *bgp_route_head;

    assert (view);
    assert (prefix);
    if ((bgp_route_head = view_find_route_head (view, prefix)) != NULL)
	return (bgp_route_head->active);
    return (NULL);
}


#ifdef notdef
/* view_find_bgp_route
 * given a prefix AND gateway, find route node entry
 * return NULL if one does not exist
 */
bgp_route_t *
view_find_bgp_route (view_t * view,
		     prefix_t * prefix, gateway_t * gateway)
{
    bgp_route_head_t *bgp_route_head;
    bgp_route_t *bgp_route;

    assert (view);
    assert (prefix);
    bgp_route_head = view_get_route_head (view, prefix);

    LL_Iterate (bgp_route_head->ll_routes, bgp_route) {
	if (bgp_route->attr->gateway == gateway) {
	    return (bgp_route);
	}
    }

    return (NULL);
}
#endif


#ifdef notdef
static int
bgp_keep_it (prefix_t *prefix, int type)
{
    assert (type != PROTO_BGP);
    /* network ... */
    if ((type == PROTO_CONNECTED || type == PROTO_STATIC) &&
	check_bgp_networks (prefix))
	return (1);
    /* redistribute */
    if (BIT_TEST (MRT->redist[type], (1 << PROTO_BGP)))
	return (1);
    return (0);
}
#endif


/* public */
void
bgp_update_route (prefix_t * prefix, generic_attr_t * new,
		  generic_attr_t * old, int pref)
/* bgp doesn't use pref (administrative distance). 
   instead, it uses administrative weight */
{
    bgp_attr_t *bgp_new = NULL, *bgp_old = NULL;

    assert (prefix);

#ifdef notdef
    /*
     * BGP dones't need to hold routes that will not redistibute
     */
    if (new && !bgp_keep_it (prefix, new->type)) {
	new = NULL;
    }
    if (old && !bgp_keep_it (prefix, old->type)) {
	old = NULL;
    }
#endif
    if (new == NULL && old == NULL)
	return;

    assert (new == NULL || new->type != PROTO_BGP);
    assert (old == NULL || old->type != PROTO_BGP);

    if (old) {

	trace (TR_TRACE, BGP->trace, "Imported Delete: %p nh %a proto %r\n",
	       prefix, old->nexthop->prefix, old->type);
	bgp_old = bgp_new_attr (old->type);
	bgp_old->gateway = old->gateway;
	if (old->nexthop) {
	    bgp_old->nexthop = ref_nexthop (old->nexthop);
            BGP4_BIT_SET (bgp_old->attribs, PA4_TYPE_NEXTHOP);
	}
    }
    if (new) {
	trace (TR_TRACE, BGP->trace, "Imported Add: %p nh %a proto %r\n",
	       prefix, new->nexthop->prefix, new->type);
	bgp_new = bgp_new_attr (new->type);
	bgp_new->gateway = new->gateway;
	if (new->nexthop) {
	    bgp_new->nexthop = ref_nexthop (new->nexthop);
            BGP4_BIT_SET (bgp_new->attribs, PA4_TYPE_NEXTHOP);
	}

	assert (new->type != PROTO_BGP);
	/* XXX need to check if cisco makes even static routes igp */
	if ((new->type == PROTO_CONNECTED || new->type == PROTO_STATIC) 
		&& check_bgp_networks (prefix))
            bgp_new->origin = 0;
	else if (new->type == PROTO_RIPNG || new->type == PROTO_RIP ||
	         new->type == PROTO_OSPF)
            bgp_new->origin = 0;
        else
            bgp_new->origin = 2;
        BGP4_BIT_SET (bgp_new->attribs, PA4_TYPE_ORIGIN);

        if (!BGP4_BIT_TEST (bgp_new->attribs, PA4_TYPE_ASPATH)) {
	    bgp_new->aspath = NULL;
            BGP4_BIT_SET (bgp_new->attribs, PA4_TYPE_ASPATH);
	}
    }

    assert (bgp_new || bgp_old);

    prefix = Ref_Prefix (prefix);
    schedule_event2 ("bgp_import", BGP->schedule, bgp_import, 
		     3, prefix, bgp_new, bgp_old);
}


/* public */
view_t *
New_View (u_int maxbitlen)
{
    view_t *view = New (view_t);

    view->ll_with_routes = LL_Create (0);
    view->ll_ann_routes = LL_Create (0);
    view->agg_tree = New_Radix (maxbitlen);

    /* XXX I know this is a bad way */
    view->family = AF_INET;
#ifdef HAVE_IPV6
    if (maxbitlen == 128) {
	view->family = AF_INET6;
    }
#endif	/* HAVE_IPV6 */

    /*if (LL_ATTR_MEMORY == NULL)
       init_attr_memory (); */

    pthread_mutex_init (&view->mutex_lock, NULL);
    view->radix_tree = New_Radix (maxbitlen);
    return (view);
}


/*
 * withdraw all routes sent by a peer and reset peer bit
 * on all route given to peer when peer leaves established state
 */
/* public */
void
view_delete_peer (view_t * view, bgp_peer_t * peer)
{
    bgp_route_head_t *rt_head;
    bgp_route_t *route;
    LINKED_LIST *ll_route_with;

    assert (view);
    assert (peer);
    view_open (view);

    ll_route_with = LL_Create (0);

    /* find all routes announced by this peer */
    VIEW_RADIX_WALK (view, rt_head) {
	/* we may need to check family? */
	LL_Iterate (rt_head->ll_routes, route) {
	    if (route->attr->gateway == peer->gateway)
		LL_Add (ll_route_with, route);
	    /* reset announce bit */
	    if (view->viewno == peer->view)
	        BITX_RESET (&rt_head->peer_mask, peer->index);
	}
    }
    VIEW_RADIX_WALK_END;

    LL_Iterate (ll_route_with, route) {
	view_delete_bgp_route (view, route);
    }
    LL_Destroy (ll_route_with);
    bgp_process_changes (view);
    view_close (view);
}


/*
 * when a peer is first established, run through view and build
 * BGP updates for all routes per policy
 * now this can be used when the policy changed:
     if announce bit is not set and policy allows, announce it
     if announce bit is set and policy doesn't allows, withdraw it
 */
/* public */
void
bgp_establish_peer (bgp_peer_t * peer, int force_announce)
{
    bgp_route_head_t *head;
    bgp_route_t *route;
    LINKED_LIST *ll_ann, *ll_with, *ll_buckets;
    update_bucket_t *bucket;
    view_t *view;
    long i = 0;

    assert (peer);
    ll_ann = LL_Create (0);
    ll_with = LL_Create (0);

    view = BGP->views[peer->view];
    view_open (view);

    VIEW_RADIX_WALK (view, head) {

    if ((i++) % 5000 == 1) {
      trace (NORM, peer->trace, "Scanned %d routes of view 0\n", i);
    }
    if ((route = head->active)) {

	assert (route);
	if (bgp_policy (head->prefix, route->attr, peer, 1)) {
	    if (!BIT_TEST (head->state_bits, VRTS_SUPPRESS)) {
		if (force_announce || 
			BITX_TEST (&head->peer_mask, peer->index) == 0) {
		    /* set announce bit */
	            BITX_SET (&head->peer_mask, peer->index);
		    LL_Add (ll_ann, route);
		}
	    }
	    else {
		/* must not be announced */
	        assert (BITX_TEST (&head->peer_mask, peer->index) == 0);
	    }
	}
	else {
	    if (!BIT_TEST (head->state_bits, VRTS_SUPPRESS)) {
		if (BITX_TEST (&head->peer_mask, peer->index) != 0) {
		    /* reset announce bit */
	            BITX_RESET (&head->peer_mask, peer->index);
		    LL_Add (ll_with, route);
		}
	    }
	    else {
		/* must not be announced */
	        assert (BITX_TEST (&head->peer_mask, peer->index) == 0);
	    }
	}
    }
    } VIEW_RADIX_WALK_END;

    /* In case that there are somthing changed already */
    LL_ClearFn (view->ll_with_routes, 0);
    LL_ClearFn (view->ll_ann_routes, 0);
    view_close (view);

    ll_buckets = build_update_buckets (ll_ann, NULL);

    pthread_mutex_lock (&peer->update_mutex_lock);
    if (LL_GetCount (ll_with) > 0) {
	bucket = New_Update_Bucket ();
	bucket->ll_prefix = ll_with;
	bucket->attr = NULL;
	LL_Append (peer->ll_update_out, bucket);
	if (LL_GetCount (peer->ll_update_out) == 1) {
    	    schedule_event2 ("view_announce_peer", peer->schedule, 
    		              (event_fn_t) view_announce_peer, 1, peer);
	}
    }
    LL_Iterate (ll_buckets, bucket) {
	bgp_attr_t *attr;
	attr = bgp_copy_attr (bucket->attr);
        bgp_deref_attr (bucket->attr);
	bucket->attr = attr;
	LL_Append (peer->ll_update_out, bucket);
	if (LL_GetCount (peer->ll_update_out) == 1) {
    	    schedule_event2 ("view_announce_peer", peer->schedule, 
   			     (event_fn_t) view_announce_peer, 1, peer);
	}
    }
    pthread_mutex_unlock (&peer->update_mutex_lock);

    LL_Destroy (ll_ann);
    LL_Destroy (ll_with);
    LL_Destroy (ll_buckets);
}


/* public */
int 
show_view (uii_connection_t * uii, int view)
{
    if (view < 0 || view > MAX_BGP_VIEWS || BGP->views[view] == NULL) {
	config_notice (TR_ERROR, uii, 
		       "invalid or unconfigured view %d\n", view);
	return (-1);
    }

    /* pthread_mutex_lock (&BGP->mutex_lock); */
    view_open (BGP->views[view]);
    bgp_dump_view (uii, BGP->views[view], NULL, NULL, NULL);
    view_close (BGP->views[view]);
    /* pthread_mutex_unlock (&BGP->mutex_lock); */
    return (1);
}


static int 
trace_bgp_view_op (uii_connection_t * uii, char *s, int op)
{
    int i;
    view_t *view;

    if (strcasecmp (s, "*") == 0) {
        for (i = 0; i < MAX_BGP_VIEWS; i++) {
	    if ((view = BGP->views[i]) != NULL) {
	        view_open (view);
        	set_trace (view->trace, op, TR_ALL, NULL);
		view_close (view);
	    }
	}
    }
    else if (isdigit (*s)) {
	i = atoi (s);
	if (i < 0 || i > MAX_BGP_VIEWS ||
	       (view = BGP->views[i]) == NULL) {
	    config_notice (TR_ERROR, uii, 
		       "invalid or unconfigured view %d\n", i);
	    Delete (s);
	    return (-1);
	}
	else {
	    view_open (view);
            set_trace (view->trace, op, TR_ALL, NULL);
	    view_close (view);
	}
    }
    else if (strcasecmp (s, "inet") == 0) {
        for (i = 0; i < MAX_BGP_VIEWS; i++) {
	    if ((view = BGP->views[i]) != NULL &&
		 view->family == AF_INET) {
	        view_open (view);
        	set_trace (view->trace, op, TR_ALL, NULL);
		view_close (view);
	    }
	}
    }
#ifdef HAVE_IPV6
    else if (strcasecmp (s, "inet6") == 0) {
        for (i = 0; i < MAX_BGP_VIEWS; i++) {
	    if ((view = BGP->views[i]) != NULL &&
		 view->family == AF_INET6) {
	        view_open (view);
        	set_trace (view->trace, op, TR_ALL, NULL);
		view_close (view);
	    }
	}
    }
#endif /* HAVE_IPV6 */
    else {
	config_notice (TR_ERROR, uii, "invalid or unconfigured view %s\n", s);
        Delete (s);
        return (-1);
    }
    Delete (s);
    return (1);
}

int 
trace_bgp_view (uii_connection_t * uii, char *s)
{
    return (trace_bgp_view_op (uii, s, TRACE_ADD_FLAGS));
}

int 
no_trace_bgp_view (uii_connection_t * uii, char *s)
{
    return (trace_bgp_view_op (uii, s, TRACE_DEL_FLAGS));
}


int 
trace_f_bgp (uii_connection_t * uii, int family)
{
    int op = TRACE_ADD_FLAGS;

    if (uii->negative)
	op = TRACE_DEL_FLAGS;

    if (family == 0 || family == AF_INET) {
        view_open (BGP->views[0]);
       	set_trace (BGP->views[0]->trace, op, TR_ALL, NULL);
	view_close (BGP->views[0]);
    }
#ifdef HAVE_IPV6
    if (family == 0 || family == AF_INET6) {
        view_open (BGP->views[1]);
       	set_trace (BGP->views[1]->trace, op, TR_ALL, NULL);
	view_close (BGP->views[1]);
    }
#endif /* HAVE_IPV6 */
    return (1);
}


#ifdef notdef
void
bgp_redistribute_request (int from, int to, int on)
{
    view_t *view = BGP->views[0];
    bgp_route_head_t *head;
    generic_attr_t *attr;

    assert (to == PROTO_BGP);
#ifdef HAVE_IPV6
    if (from == PROTO_RIPNG)
        view = BGP->views[1];
#endif /* HAVE_IPV6 */
    VIEW_RADIX_WALK (view, head) {
	if (head->active) {
	    assert (head->active->attr->type == PROTO_BGP);
	    if (BIT_TEST (head->active->attr->options, BGP_INTERNAL))
		continue;

	    if (BIT_TEST (head->state_bits, VRTS_SUPPRESS))
	        continue;

	    attr = bgp2gen (head->active->attr);
	    if (on)
	        MRT->proto_update_route[to] (head->prefix, attr, NULL);
	    else
	        MRT->proto_update_route[to] (head->prefix, NULL, attr);
    	    Deref_Generic_Attr (attr);
	}
    } VIEW_RADIX_WALK_END;
}
#endif

