/* 
 * $Id: bgp_util.c,v 1.56 1999/04/30 23:47:55 masaki Exp $
 */

#include <mrt.h>
#include <config_file.h>
#include <bgp.h>
#include <stdarg.h>
#include <ctype.h>

u_int aspath_hash_fn (aspath_t * aspath, u_int size);
void bgp_schedule_timer (mtimer_t * timer, bgp_peer_t * peer);
static void stop_bgp_peer (bgp_peer_t *peer, int lockf);


static void
set_nonblocking (int sockfd)
{
#ifndef HAVE_LIBPTHREAD
#ifdef FIONBIO
    int optval = 1;
    if (ioctl (sockfd, FIONBIO, &optval) < 0) {
	trace (TR_ERROR, MRT->trace, "ioctl FIONBIO failed (%s)\n",
	       strerror (errno));
    }
#else
    if (fcntl (sockfd, F_SETFL, O_NONBLOCK) < 0) {
	trace (TR_ERROR, MRT->trace, "fcntl F_SETFL O_NONBLOCK failed (%s)\n",
	       strerror (errno));
    }
#endif /* FIONBIO */
#endif /* HAVE_LIBPTHREAD */
}


prefix_t *
get_local_addr (int sockfd)
{
    union sockaddr_any {
	struct sockaddr sa;
	struct sockaddr_in sin;
#ifdef HAVE_IPV6
	struct sockaddr_in6 sin6;
#endif /* HAVE_IPV6 */
    } addr;
    prefix_t *prefix = NULL;
    int len, family;

    len = sizeof (addr);
    memset (&addr, 0, sizeof (addr));
    if (getsockname (sockfd, (struct sockaddr *) &addr, &len) >= 0) {

	if ((family = addr.sa.sa_family) == AF_INET) {
	    struct sockaddr_in *sin = (struct sockaddr_in *) &addr;
	    prefix = New_Prefix (AF_INET, &sin->sin_addr, 32);
	}
#ifdef HAVE_IPV6
	else if (family == AF_INET6) {
	    struct sockaddr_in6 *sin6 = (struct sockaddr_in6 *) &addr;
	    if (IN6_IS_ADDR_V4MAPPED (&sin6->sin6_addr))
		prefix = New_Prefix (AF_INET, ((u_char *) &sin6->sin6_addr) + 12, 32);
	    else
		prefix = New_Prefix (AF_INET6, &sin6->sin6_addr, 128);
	}
#endif /* HAVE_IPV6 */
	return (prefix);
    }
    return (NULL);
}


static void
peer_check_gateway (bgp_peer_t *peer)
{
    if (peer->local_addr)
	Deref_Prefix (peer->local_addr);
    peer->local_addr = get_local_addr (peer->sockfd);

    if ((INTERFACE_MASTER && peer->gateway->interface &&
	 !is_prefix_on (peer->gateway->prefix, peer->gateway->interface)) ||
        (INTERFACE_MASTER && peer->gateway->interface == NULL &&
	(peer->gateway->interface = find_interface_local (peer->local_addr)))) {

	prefix_t *prefix = NULL;

	/* set regardless of immediate nexthop */
	BIT_SET (peer->options, BGP_PEER_IS_MULTIHOP);

	if (MRT->rib_find_best_route)
	    prefix = MRT->rib_find_best_route (peer->gateway->prefix);
	if (prefix) {
	    if (!BIT_TEST (peer->options, BGP_INTERNAL))
	        BIT_SET (peer->options, BGP_EBGP_MULTIHOP);
	    if (peer->nexthop)
		deref_nexthop (peer->nexthop);
	    peer->nexthop = add_nexthop (prefix, peer->gateway->interface);
	    trace (TR_INFO, peer->trace, 
		   "Multihop! immediate nexthop will be %a\n", prefix);
	}
	else {
	    trace (TR_ERROR, peer->trace, 
	    	   "Multihop! immediate nexthop unknown\n");
	}
    }
}


static void
bgp_connect_ready (bgp_peer_t * peer, int async)
{
    struct sockaddr name;
    int namelen = sizeof (name);
#ifdef FIONBIO
    int optval = 0;
#endif /* FIONBIO */

    if (async) {
        if (peer->sockfd < 0) {
	    trace (TR_WARN, peer->trace,
	           "connect to %g succeeded but sockfd has been closed\n",
	           peer->gateway);
	    /* no event */
	    return;
        }

        if (peer->state != BGPSTATE_CONNECT) {
	    trace (TR_WARN, peer->trace,
	           "connect to %s succeeded but too late\n", peer->gateway);
	    /* no event */
	    return;
	}
        select_delete_fd2 (peer->sockfd);
    }

    /* see if we are really connected */
    if (getpeername (peer->sockfd, &name, &namelen) < 0) {
	trace (TR_INFO, peer->trace,
	       "connect to %g failed (%m)\n", peer->gateway);
	close (peer->sockfd);
	peer->sockfd = -1;
	if (async)
	    bgp_sm_process_event (peer, BGPEVENT_OPENFAIL);
	else
	    schedule_event2 ("bgp_sm_process_event",
			     peer->schedule, bgp_sm_process_event,
			     2, peer, BGPEVENT_OPENFAIL);
	return;
    }

    trace (TR_INFO, peer->trace, "Outgoing connection SUCCEEDED\n");

#ifdef FIONBIO
    if (ioctl (peer->sockfd, FIONBIO, &optval) < 0) {
        trace (TR_ERROR, peer->trace, "ioctl FIONBIO failed (%m)\n");
	/* OK, continue */
    }   
#else          
    if (fcntl (peer->sockfd, F_SETFL, 0) < 0) {
        trace (TR_ERROR, peer->trace, "fcntl F_SETFL 0 failed (%m)\n");
	/* OK, continue */
    }   
#endif /* FIONBIO */     
    peer_check_gateway (peer);
    if (async)
        bgp_sm_process_event (peer, BGPEVENT_OPEN);
    else
        schedule_event2 ("bgp_sm_process_event",
		         peer->schedule, bgp_sm_process_event,
		         2, peer, BGPEVENT_OPEN);
}


/*
 * Starts tcp connection to peer. Returns 1 on sucess, -1 oltherwise
 */
int 
bgp_start_transport_connection (bgp_peer_t * peer)
{
    int optval = 1;
    int ret;
    int family, len;
    struct sockaddr_in sin;
    struct sockaddr *s;
#ifdef HAVE_IPV6
    struct sockaddr_in6 sin6;
#endif /* HAVE_IPV6 */

    /* okay, start a TCP connection */

    if ((family = peer->gateway->prefix->family) == AF_INET) {
	memset (&sin, 0, sizeof (sin));
	sin.sin_family = family;
	sin.sin_port = htons (BGP->cport);
	memcpy (&sin.sin_addr, prefix_tochar (peer->gateway->prefix), 4);
	s = (struct sockaddr *) &sin;
	len = sizeof (sin);
    }

#ifdef HAVE_IPV6
    else if (family == AF_INET6) {
	memset (&sin6, 0, sizeof (sin6));
	sin6.sin6_family = family;
	sin6.sin6_port = htons (BGP->cport);
	memcpy (&sin6.sin6_addr, prefix_tochar (peer->gateway->prefix), 16);
	s = (struct sockaddr *) &sin6;
	len = sizeof (sin6);
    }
#endif /* HAVE_IPV6 */
    else {
        return (-1);
    }

    if (peer->sockfd >= 0) {
	select_delete_fd (peer->sockfd);
    }

    if ((peer->sockfd = socket (family, SOCK_STREAM, 0)) < 0) {
	trace (TR_ERROR, peer->trace, "socket open failed (%m)\n");
	return (-1);
    }

    /* transport connection succeeded */
    setsockopt (peer->sockfd, SOL_SOCKET, SO_REUSEADDR,
		(char *) &optval, sizeof (optval));

#ifdef SO_BINDTODEVICE
    if (peer->bind_if) {
	struct ifreq ifr;
	safestrncpy ((char *)&ifr.ifr_ifrn.ifrn_name, peer->bind_if->name, 
		     sizeof (ifr.ifr_ifrn.ifrn_name));
	if (setsockopt (peer->sockfd, SOL_SOCKET, SO_BINDTODEVICE,
				  &ifr, sizeof (ifr)) < 0)
	    trace (TR_ERROR, peer->trace, "SO_BINDTODEVICE %s (%m)\n",
				   peer->bind_if->name);
    }
#endif /* SO_BINDTODEVICE */

    if (peer->bind_addr) {
	    int llen;
	    struct sockaddr_in l_sin;
	    struct sockaddr *l_s;
#ifdef HAVE_IPV6
	    struct sockaddr_in6 l_sin6;
#endif /* HAVE_IPV6 */

	    if (family  == AF_INET) {
		    memset (&l_sin, 0, sizeof (sin));
		    l_sin.sin_family = family;
		    memcpy (&l_sin.sin_addr, prefix_tochar (peer->bind_addr), 
			    4);
		    l_s = (struct sockaddr *) &l_sin;
		    llen = sizeof (sin);
	    }
#ifdef HAVE_IPV6
	    else if (family == AF_INET6) {
		    memset (&l_sin6, 0, sizeof (l_sin6));
		    l_sin6.sin6_family = family;
		    memcpy (&l_sin6.sin6_addr, prefix_tochar (peer->bind_addr),
			    16);
		    l_s = (struct sockaddr *) &l_sin6;
		    llen = sizeof (sin6);
	    }
#endif /* HAVE_IPV6 */
	    else {
		assert (0);
		return (-1);
	    }
	    if (bind (peer->sockfd, l_s, len) < 0) {
		    trace (TR_ERROR, peer->trace, "socket bind (%m)\n");
		    return (-1);
	    }
    }

    /* always non-blocking. 
      if connect doesn't return, there is no way to resume it. */
    set_nonblocking (peer->sockfd);

    ret = connect (peer->sockfd, s, len);
    if (ret < 0) {
	if (errno != EINPROGRESS) {
	    close (peer->sockfd);
	    peer->sockfd = -1;
	    return (-1);
	}
	trace (TR_PACKET, peer->trace, "waiting on %d for write\n",
	       peer->sockfd);
	select_add_fd_event ("bgp_connect_ready", peer->sockfd, SELECT_WRITE, 
			     TRUE, 
			     peer->schedule, bgp_connect_ready, 2, peer, TRUE);
	return (0);
    }
    bgp_connect_ready (peer, FALSE);
    return (1);
}


int
bgp_in_recv_open (bgp_peer_t * peer)
{
    int len;
    u_char *cp;
    int type, length;

    int version;
    int as;
    u_long holdtime;
    int authcode;
    u_long id;

    /* closed by outgoing */
    if (peer->sockfd_in < 0)
	return (-1);

    len = bgp_read (peer, peer->sockfd_in, peer->read_ptr_in, BGPMAXPACKETSIZE);
    if (len < 0) {
        peer->read_ptr_in = peer->buffer_in; /* eat up the input */
    	select_delete_fd2 (peer->sockfd_in);
	close (peer->sockfd_in);
	peer->sockfd_in = -1;
        return (-1);
    }

    peer->read_ptr_in += len;

    if ((len = peer->read_ptr_in - peer->buffer_in) < BGP_HEADER_LEN) {
	/* keep reading */
    	select_enable_fd_mask (peer->sockfd_in, SELECT_READ);
	return (-1);
    }

    cp = peer->buffer_in;

    BGP_GET_HDRTYPE (type, cp);
    BGP_GET_HDRLEN (length, cp);

    if (length < BGP_HEADER_LEN || length > BGPMAXPACKETSIZE) {
        select_delete_fd2 (peer->sockfd_in);
	trace (TR_ERROR, peer->trace,
	    "rejecting the incoming connection (bad length %d)\n", length);
        goto reject;
    }

    if (len < length) {
	/* keep reading */
    	select_enable_fd_mask (peer->sockfd_in, SELECT_READ);
	return (-1);
    }

    /* this happens only once */
    select_delete_fd2 (peer->sockfd_in);

    if (type >= BGP_OPEN && type < BGP_PACKET_MAX) {
        trace (TR_PACKET, peer->trace, "recv %s (%d bytes) [incoming]\n",
               sbgp_pdus[type], length);
    }
    else {
	trace (TR_ERROR, peer->trace,
	    "rejecting the incoming connection (unknown type code %d)\n", 
	    type);
	goto reject;
    }

    if (type != BGP_OPEN) {
	trace (TR_ERROR, peer->trace,
	    "rejecting the incoming connection (%s: type is not OPEN)\n",
	       sbgp_pdus[type]);
	goto reject;
    }

    BGP_SKIP_HEADER (cp);
    BGP_GET_OPEN (version, as, holdtime, id, authcode, cp);

    if (peer->state == BGPSTATE_IDLE) {
	bgp_change_state (peer, BGPSTATE_ACTIVE, BGPEVENT_OPEN);
        Timer_Turn_OFF (peer->timer_Start);
    }

    if (peer->state == BGPSTATE_ESTABLISHED 
	    || peer->state == BGPSTATE_IDLE) {

	trace (TR_INFO, peer->trace,
	    "rejecting the incoming connection (state is %s)\n",
	       sbgp_states[peer->state]);
      reject:
#ifdef notdef
	bgp_send_notification2 (peer->sockfd_in, peer->gateway->prefix, 
				BGP->cport, BGP_CEASE, 0);
#endif
	close (peer->sockfd_in);
	peer->sockfd_in = -1;
	return (-1);
    }

    if (peer->state == BGPSTATE_OPENSENT ||
	peer->state == BGPSTATE_OPENCONFIRM) {

	char his[MAXLINE], mine[MAXLINE];

	trace (TR_INFO, peer->trace,
	       "routerid comparison (in %s <=> out %s)\n", 
		inet_ntop (AF_INET, &id, his, sizeof (his)),
#ifndef BGP_MULTI
		inet_ntop (AF_INET, &BGP->my_id, mine, sizeof (mine)));
	if (ntohl (id) <= ntohl (BGP->my_id)) {
#else /* BGP_MULTI */
		inet_ntop (AF_INET, &peer->local_bgp->this_id, mine, sizeof (mine)));
	if (ntohl (id) <= ntohl (peer->local_bgp->this_id)) {
#endif /* BGP_MULTI */
	    trace (TR_INFO, peer->trace,
	        "rejecting the incoming connection (router id)\n");
	    goto reject;
	}
    }

    /* state is CONNECT or ACTIVE */

	trace (TR_INFO, peer->trace,
	       "accepting incoming connection (state was %s)\n",
	       sbgp_states[peer->state]);

	bgp_peer_dead (peer); /* this may clear schedule queue */

	/* switch the inputs */
	peer->sockfd = peer->sockfd_in;
	peer->sockfd_in = -1;
	peer->packet = peer->buffer_in;

	/* if something remains */
	if (len > length) {
	    assert (peer->read_ptr == peer->start_ptr);
	    assert (peer->read_ptr == peer->buffer);
	    memcpy (peer->read_ptr, peer->buffer_in + length, len - length);
	    peer->read_ptr += (len - length);
	}

        peer_check_gateway (peer);

#ifdef notdef
        if (peer->state != BGPSTATE_OPENSENT) {
	    /* this was pending */
	    bgp_change_state (peer, BGPSTATE_OPENSENT, BGPEVENT_OPEN);
	}
#endif

	if (bgp_process_open (peer) > 0) {
            select_add_fd_event ("bgp_get_pdu", peer->sockfd, SELECT_READ,
                                 1 /* on */, peer->schedule, 
				 (event_fn_t) bgp_get_pdu, 1, peer);
            select_add_fd_event ("bgp_flush_queue", peer->sockfd, SELECT_WRITE,
                                 0 /* off */, peer->schedule, 
				 (event_fn_t) bgp_flush_queue, 1, peer);
	    /* no hold timer for this open since open has been received */
	    if (bgp_send_open (peer) >= 0) {
		bgp_change_state (peer, BGPSTATE_OPENSENT, BGPEVENT_OPEN);
	        Timer_Turn_ON (peer->timer_KeepAlive);
	        if (bgp_send_keepalive (peer) >= 0) {
		    if (peer->HoldTime_Interval > 0)
	                Timer_Reset_Time (peer->timer_HoldTime);
	            bgp_change_state (peer, BGPSTATE_OPENCONFIRM, 
				      BGPEVENT_RECVOPEN);
		    return (1);
	        }
	    }
	}
	/* OPEN was bad */
	/* notification already sent */
	/* outgoing connection has been closed. no way. */
	bgp_change_state (peer, BGPSTATE_IDLE, BGPEVENT_RECVOPEN);
	bgp_peer_dead (peer);
	Timer_Turn_ON (peer->timer_Start);
	return (-1);
}


static void
bgp_schedule_in_recv_open (bgp_peer_t * peer)
{
    peer->read_ptr_in = peer->buffer_in; /* reset the input */
    if (peer->state == BGPSTATE_IDLE) {
	/* suspend it until start timer fires */
        select_delete_fd2 (peer->sockfd_in);
	peer->pending_in_recv_open = 1;
	trace (TR_INFO, peer->trace, "suspend incoming\n");
    }
    else {
        schedule_event2 ("bgp_in_recv_open", peer->schedule, 
			 (void_fn_t) bgp_in_recv_open, 1, peer);
    }
}


static void
bgp_in_accept_connection (int fd)
{
    int sockfd, family, len;
    u_short port;
    bgp_peer_t *peer;
#ifndef HAVE_IPV6
    struct sockaddr_in addr, local;
#ifdef BGP_MULTI
    struct sockaddr_in remote;
#endif /* BGP_MULTI */
#else
    union sockaddr_any {
	struct sockaddr sa;
	struct sockaddr_in sin;
	struct sockaddr_in6 sin6;
    } addr, local;

#ifdef BGP_MULTI
    union sockaddr_any remote;
#endif /* BGP_MULTI */
#endif /* HAVE_IPV6 */


    prefix_t *prefix, *local_prefix;

    len = sizeof (addr);

    if ((sockfd = accept (fd, (struct sockaddr *) &addr, &len)) < 0) {
	trace (TR_ERROR, BGP->trace, "accept (%m)\n");
	select_enable_fd_mask (fd, SELECT_READ);
	return;
    }

    len = sizeof (local);
    if (getsockname (sockfd, (struct sockaddr *) &local, &len) < 0) {
        trace (TR_ERROR, BGP->trace, "getsockname (%m)\n");
        select_enable_fd_mask (fd, SELECT_READ);
        return;
    }

#ifdef BGP_MULTI
    len = sizeof (local);
    if (getpeername (sockfd, (struct sockaddr *) &remote, &len) < 0) {
        trace (TR_ERROR, BGP->trace, "getpeername on fd %d (%m)\n", sockfd);
/*        select_enable_fd_mask (fd, SELECT_READ); */
        return;
    }
#endif /* BGP_MULTI */

#ifndef HAVE_LIBPTHREAD
    set_nonblocking (fd);
#endif /* HAVE_LIBPTHREAD */
    select_enable_fd_mask (fd, SELECT_READ);

#ifndef HAVE_IPV6
    if ((family = addr.sin_family) == AF_INET) {
#else
    if ((family = addr.sa.sa_family) == AF_INET) {
#endif /* HAVE_IPV6 */
	struct sockaddr_in *sin = (struct sockaddr_in *) &addr;
	port = ntohs (sin->sin_port);
	prefix = New_Prefix (AF_INET, &sin->sin_addr, 32);
	sin = (struct sockaddr_in *) &local;
        local_prefix = New_Prefix (AF_INET, &sin->sin_addr, 32);
    }
#ifdef HAVE_IPV6
    else if (family == AF_INET6) {
	struct sockaddr_in6 *sin6 = (struct sockaddr_in6 *) &addr;
	port = ntohs (sin6->sin6_port);
	if (IN6_IS_ADDR_V4MAPPED (&sin6->sin6_addr))
	    prefix = New_Prefix (AF_INET, ((char *) &sin6->sin6_addr) + 12, 
					  32);
	else
	    prefix = New_Prefix (AF_INET6, &sin6->sin6_addr, 128);
	sin6 = (struct sockaddr_in6 *) &local;
        if (IN6_IS_ADDR_V4MAPPED (&sin6->sin6_addr))
            local_prefix = New_Prefix (AF_INET, 
				      ((char *)&sin6->sin6_addr) + 12, 32);
        else
            local_prefix = New_Prefix (AF_INET6, &sin6->sin6_addr, 128);
    }
#endif /* HAVE_IPV6 */
    else {
	trace (TR_ERROR, BGP->trace, "unknown connection family = %d\n",
	       family);
	close (sockfd);
	return;
    }

    pthread_mutex_lock (&BGP->peers_mutex_lock);
    peer = Find_BGP_Peer (prefix, local_prefix, NULL);

    if (peer) {
	if (peer->gateway) {
            trace (TR_INFO, peer->trace, 
	           "Valid incoming connection detected at %a\n", local_prefix);
	}
	else {
	    bgp_peer_t *child;
	    child = Add_BGP_Peer (NULL, prefix, NULL /* local_prefix */, NULL,
				  0, peer->trace);
            BIT_SET (child->options, BGP_CONNECT_PASSIVE);
	    child->parent = peer;
	    if (peer->children == NULL)
		peer->children = LL_Create (0);
	    LL_Add (peer->children, child);
	    trace (TR_INFO, peer->trace, 
		   "Child from %a created at %a (but remains passive)\n", 
		   prefix, local_prefix);
	    /* force to change BGP state */
	    bgp_change_state (child, BGPSTATE_ACTIVE, BGPEVENT_START);
	}
    }
    else {
	if (BGP->accept_all_peers) {
	    /* at this point, as is unknown */
	    /* modifying BGP structure should be safe */
	    peer = Add_BGP_Peer (NULL, prefix, NULL /* local_prefix */, NULL, 
				 0, BGP->trace);
	    /* safer way */
            BIT_SET (peer->options, BGP_CONNECT_PASSIVE);
	    trace (TR_INFO, peer->trace, 
		   "Anonymous peer created at %a (but remains passive)\n", 
		   local_prefix);
	    /* force to change BGP state */
	    bgp_change_state (peer, BGPSTATE_ACTIVE, BGPEVENT_START);
	}
	else {
	    trace (TR_INFO, BGP->trace, "Unconfigured peer %a at %a\n",
		   prefix, local_prefix);
#ifdef notdef
	    bgp_send_notification2 (sockfd, prefix, ntohs (port),
				    BGP_ERR_OPEN, BGP_ERROPN_AUTH);
#endif
	    Deref_Prefix (prefix);
	    Deref_Prefix (local_prefix);
	    close (sockfd);
    	    pthread_mutex_unlock (&BGP->peers_mutex_lock);
	    return;
	}
    }

    pthread_mutex_unlock (&BGP->peers_mutex_lock);

    if (peer->state == BGPSTATE_ESTABLISHED
	    /* send it anyway because it may proceed from idle later */
            /* || peer->state == BGPSTATE_IDLE */) {
	trace (TR_INFO, peer->trace, "reject the incoming connection (%s)\n",
	       sbgp_states[peer->state]);
#ifdef notdef
	bgp_send_notification2 (sockfd, prefix, ntohs (port), BGP_CEASE, 0);
#endif
        Deref_Prefix (prefix);
        Deref_Prefix (local_prefix);
	close (sockfd);
	return;
    }

    Deref_Prefix (prefix);
    Deref_Prefix (local_prefix);

    /* XXX We should run a timer to timeout the incoming connection */

    if (peer->sockfd_in >= 0) {
	trace (TR_INFO, peer->trace, 
	       "Closing previous incoming connection with fd %d\n",
	       peer->sockfd_in);
	select_delete_fd (peer->sockfd_in);
	peer->pending_in_recv_open = 0;
    }
    peer->sockfd_in = sockfd;
    peer->read_ptr_in = peer->buffer_in;
    select_add_fd_event ("bgp_schedule_in_recv_open", sockfd, 
			 SELECT_READ, TRUE, NULL, 
			 bgp_schedule_in_recv_open, 1, peer);
}


int 
aspath_lookup_fn (aspath_t * tmp1, aspath_t * tmp2)
{
    u_int tmpx;

    if ((tmp1 == NULL) || (tmp2 == NULL)) {
	return (-1);
    }

    tmpx = compare_aspaths (tmp1, tmp2);
    return (tmpx);
}

#ifdef BGP_MULTI
void
init_bgp_local (bgp_local_t *local)
{
  pthread_mutex_init (&local->mutex_lock, NULL);
  local->this_as = /* DEFAULT_LOCAL_AS */ 0 /* must be set */;
  local->this_id = /* htonl (DEFAULT_ROUTER_ID) */ 0 /* must be set */;
  local->view_no = 0;
  local->bind_interface_only = 0;
  local->num_interfaces = 0;
  local->ll_interfaces = LL_Create (0);
} 
#endif /* BGP_MULTI */

void 
init_BGP (trace_t * ltrace)
{
#ifdef BGP_MULTI
    int count;
#endif /* BGP_MULTI */

#ifdef notdef
    bgp_attr_t attr;
#endif
    char str[MAXLINE];

    BGP = New (bgp_t);
    pthread_mutex_init (&BGP->peers_mutex_lock, NULL);
#ifdef BGP_MULTI
    pthread_mutex_init (&BGP->locals_mutex_lock, NULL);
#endif /* BGP_MULTI */

    BGP->ll_bgp_peers = LL_Create (0);
    BGP->ll_networks = LL_Create (0);

    BGP->lport = BGP_PORT;
    BGP->cport = BGP_PORT;
    BGP->ll_listen_sockets = LL_Create (0);

#ifdef BGP_MULTI
    BGP->ll_bgp_locals = LL_Create (0);
#endif /* BGP_MULTI */

/*
    BGP->sockfd = -1;
    BGP->sockfd_count = 0;
*/

    BGP->accept_all_peers = 0;
#ifndef BGP_MULTI
    BGP->my_as = /* DEFAULT_MY_AS */ 0 /* must be set */;
    BGP->my_id = /* htonl (DEFAULT_MY_ID) */ 0 /* must be set */;
    BGP->cluster_id = BGP->my_id /* must be set */;
#else /* BGP_MULTI */
    BGP->current_bgp = NULL;
    for (count = 0; count <= MAX_AS_NUMBER; count++)
      BGP->router_bgp[count] = NULL;
#endif /* BGP_MULTI */
    BGP->trace = trace_copy (ltrace);
    set_trace (BGP->trace, TRACE_PREPEND_STRING, "BGP", 0);
#ifdef BGP_MULTI
    BGP->view_count = 2;
#endif /* BGP_MULTI */
    BGP->views[0] = New_View (32);
    BGP->views[0]->viewno = 0;
    BGP->views[0]->trace = trace_copy (ltrace);
    BGP->views[0]->rib = 1;
    sprintf (str, "BGP view %d", BGP->views[0]->viewno);
    set_trace (BGP->views[0]->trace, TRACE_PREPEND_STRING, str, 0);
#ifdef HAVE_IPV6
    BGP->views[1] = New_View (128);
    BGP->views[1]->viewno = 1;
    BGP->views[1]->trace = trace_copy (ltrace);
    BGP->views[1]->rib = 1;
    sprintf (str, "BGP4+ view %d", BGP->views[1]->viewno);
    set_trace (BGP->views[1]->trace, TRACE_PREPEND_STRING, str, 0);
#endif /* HAVE_IPV6 */
    BGP->default_local_pref = DEFAULT_LOCAL_PREF;
    BGP->bgp_num_active_route_head = 0;
    BGP->bgp_num_active_route_node = 0;
    BGP->Default_Start_Interval = DEFAULT_START_INTERVAL;
    BGP->Default_ConnectRetry_Interval = DEFAULT_CONNET_RETRY_INTERVAL;
    BGP->Default_KeepAlive_Interval = DEFAULT_KEEPALIVE_INTERVAL;
    BGP->Default_HoldTime_Interval = DEFAULT_HOLDTIME_INTERVAL;

    pthread_mutex_init (&BGP->mutex_lock, NULL);

#ifdef notdef
    BGP->attr_hash =
	HASH_Create (100, HASH_KeyOffset, HASH_Offset (&attr, &attr.aspath),
		     HASH_LookupFunction, aspath_lookup_fn,
		     HASH_HashFunction, aspath_hash_fn, NULL);
#endif

    BGP->schedule = New_Schedule ("BGP", BGP->trace);
    MRT->proto_update_route[PROTO_BGP] = bgp_update_route;
    if (MRT->default_id > 0) {
#ifndef BGP_MULTI
            BGP->my_id = MRT->default_id;
            BGP->cluster_id = MRT->default_id;
#else /* BGP_MULTI */
/* TODO:  I've got to figure this one out.  What do we do with the router_id
   stuff? I have to put it into some sort of initialization function.
   hmmm...later. -- binkertn */

/*          BGP->current_bgp->this_id = MRT->default_id; */
/*          BGP->current_bgp->cluster_id = MRT->default_id; */
#endif /* BGP_MULTI */
    }   
    BGP->update_call_fn = process_bgp_update;
    BGP->dump_direction = DUMP_DIR_RECV;
    mrt_thread_create2 ("BGP", BGP->schedule, NULL, NULL);
}


/* 
 * Start BGP listening on port (usually 179)
 */
/* XXX bind_if may not work on some implemenations */
listen_socket_t * 
init_BGP_listen (prefix_t *bind_addr, interface_t *bind_if)
{
    struct sockaddr *sa;
    int sockfd, len, optval = 1;
    listen_socket_t *listen_socket;

    struct sockaddr_in serv_addr;
#ifdef HAVE_IPV6
    struct sockaddr_in6 serv_addr6;
#endif /* HAVE_IPV6 */

    pthread_mutex_lock (&BGP->mutex_lock);
    LL_Iterate (BGP->ll_listen_sockets, listen_socket) {
	/* skip if only one is null */
	if ((bind_addr == NULL || listen_socket->prefix == NULL) &&
		bind_addr != listen_socket->prefix)
	    continue;
	if ((bind_addr == listen_socket->prefix /* the both is null */||
		prefix_compare2 (bind_addr, listen_socket->prefix) == 0) &&
	        bind_if == listen_socket->interface)
	    break;
    }
    pthread_mutex_unlock (&BGP->mutex_lock);
    if (listen_socket) {
        pthread_mutex_lock (&listen_socket->mutex_lock);
	listen_socket->ref_count++;
        pthread_mutex_unlock (&listen_socket->mutex_lock);
	return (listen_socket);
    }

    /* check that NOT listening to portmaster! */

    memset (&serv_addr, 0, sizeof (serv_addr));
    serv_addr.sin_port = htons (BGP->lport);
    serv_addr.sin_family = AF_INET;
    if (bind_addr && bind_addr->family == AF_INET)
        memcpy (&serv_addr.sin_addr, prefix_tochar (bind_addr), 4);

    sa = (struct sockaddr *) &serv_addr;
    len = sizeof (serv_addr);
#ifdef HAVE_IPV6
    if (bind_addr == NULL || bind_addr->family == AF_INET6) {
        memset (&serv_addr6, 0, sizeof (serv_addr6));
        serv_addr6.sin6_port = htons (BGP->lport);
        serv_addr6.sin6_family = AF_INET6;
        if (bind_addr && bind_addr->family == AF_INET6)
            memcpy (&serv_addr6.sin6_addr, prefix_tochar (bind_addr), 16);
        sa = (struct sockaddr *) &serv_addr6;
        len = sizeof (serv_addr6);
    }
#endif /* HAVE_IPV6 */

    if ((sockfd = socket (sa->sa_family, SOCK_STREAM, 0)) < 0) {
#ifdef HAVE_IPV6
	if (sa->sa_family == AF_INET)
	    goto error;
	sa = (struct sockaddr *) &serv_addr;
	len = sizeof (serv_addr);
        if ((sockfd = socket (sa->sa_family, SOCK_STREAM, 0)) < 0) {
error:
#endif /* HAVE_IPV6 */
	    trace (TR_ERROR, BGP->trace, 
		   "Could not get socket (%m)\n");
	    return (NULL);
#ifdef HAVE_IPV6
	}
#endif /* HAVE_IPV6 */
    }

    /* allow the reuse this port */
    if (setsockopt (sockfd, SOL_SOCKET, SO_REUSEADDR,
		    (char *) &optval, sizeof (optval)) < 0)
	trace (TR_ERROR, BGP->trace, "setsockopt failed (%m)\n");
#ifdef SO_REUSERPORT
    setsockopt (sockfd, SOL_SOCKET, SO_REUSERPORT, (char *)&optval, sizeof (optval));
#endif /* SO_REUSERPORT */

#ifdef SO_BINDTODEVICE
    if (bind_if) {
            struct ifreq ifr;
            safestrncpy (ifr.ifr_ifrn.ifrn_name, bind_if->name, 
			 sizeof (ifr.ifr_ifrn.ifrn_name));
            if (setsockopt (sockfd, SOL_SOCKET, SO_BINDTODEVICE,
                                  &ifr, sizeof (ifr)) < 0)
                 trace (TR_ERROR, BGP->trace, "BINDTODEVICE %s (%m)\n",
                                   bind_if->name);
	    else
                 trace (TR_INFO, BGP->trace, "binding to interface %s\n",
			bind_if->name);
    }
#endif /* SO_BINDTODEVICE */

    if (bind (sockfd, sa, len) < 0) {
	trace (TR_ERROR, BGP->trace, 
	       "Could not bind to port %d (%m), aborting\n", BGP->lport);
	close (sockfd);
	return (NULL);
    }

    listen (sockfd, 5);
    select_add_fd_event ("bgp_in_accept_connection", sockfd, SELECT_READ, 
			 TRUE, NULL, 
			 bgp_in_accept_connection, 1, sockfd);

    if (bind_addr)
        trace (TR_INFO, BGP->trace, 
	       "listening for connections on port %d at %a\n",
	       BGP->lport, bind_addr);
    else
        trace (TR_INFO, BGP->trace, "listening for connections on port %d\n",
	       BGP->lport);

    listen_socket = New (listen_socket_t);
    listen_socket->sockfd = sockfd;
    listen_socket->ref_count = 1;
    pthread_mutex_init (&listen_socket->mutex_lock, NULL);
    listen_socket->prefix = Ref_Prefix (bind_addr);
    listen_socket->interface = bind_if;
    pthread_mutex_lock (&BGP->mutex_lock);
    LL_Add (BGP->ll_listen_sockets, listen_socket);
    pthread_mutex_unlock (&BGP->mutex_lock);
    return (listen_socket);
}


bgp_peer_t *
Find_BGP_Peer (prefix_t * prefix, prefix_t * usrc, interface_t *interface)
{
    int score = -1;
    bgp_peer_t *best = NULL;
    bgp_peer_t *peer;

    /* scan through peers */
    LL_Iterate (BGP->ll_bgp_peers, peer) {
	    int quality = 0;

	if (peer->gateway == NULL && peer->neighbor_list >= 0) {
	    if (apply_access_list (peer->neighbor_list, prefix)) {
		return (peer);
	    }
	}
	else if (prefix_compare (peer->gateway->prefix, prefix))
		goto chk_2;

	if (peer->aliases) {
	    prefix_t *alias;

	    LL_Iterate (peer->aliases, alias) {
		if (prefix_compare (alias, prefix))
		    goto chk_2;
	    }
	}
	continue;

chk_2:
	if (interface && usrc) {
		if (peer->bind_if && peer->bind_if != interface)
			continue;
		if (peer->bind_addr &&
		    !prefix_compare (peer->bind_addr, usrc))
			continue;
		if (peer->bind_addr)
			quality |= 2;
		if (peer->bind_if)
			quality |= 1;
	} else if (usrc) {
		if (peer->bind_addr &&
		    !prefix_compare (peer->bind_addr, usrc)) {
			trace (TR_ERROR, peer->trace, "not matched usrc=%a\n",
			       usrc);
			continue;
		}
		if (peer->bind_addr)
			quality |= 2;
		if (!peer->bind_if)
			quality |= 1;
	} else if (interface) {
		if (peer->bind_if && peer->bind_if != interface)
			continue;
		if (!peer->bind_addr)
			quality |= 2;
		if (peer->bind_if)
			quality |= 1;
	} else {
		if (!peer->bind_addr)
			quality |= 2;
		if (!peer->bind_if)
			quality |= 1;
	}
	    if (quality > score) {
		    best = peer;
		    score = quality;
	    }
    }
    return (best);
}


bgp_peer_t *
Find_BGP_Peer_ByID (char *name)
{
    bgp_peer_t *peer = NULL;

    /* scan through peers */
    LL_Iterate (BGP->ll_bgp_peers, peer) {
	    if (peer->name && strcasecmp (peer->name, name) == 0)
		    break;
    }
    return (peer);
}


static int
bgp_obtain_index ()
{
    int i;

    /* reserve zero for the future use */
    for (i = 1 ; i < MAX_BGP_PEERS; i++) {
	if (!BITX_TEST (&BGP->bits_assigned, i)) {
	    BITX_SET (&BGP->bits_assigned, i);
	    return (i);
        }
    }
    return (-1);
}

static void
bgp_return_index (int index)
{
    assert (BITX_TEST(&BGP->bits_assigned, index));
    BITX_RESET (&BGP->bits_assigned, index);
}


void
peer_set_as (bgp_peer_t *peer, int as)
{
    if (peer->gateway && peer->gateway->AS != as) {
        peer->gateway = add_gateway (peer->gateway->prefix, as, 
				     peer->gateway->interface);
    }

#ifndef BGP_MULTI
    if (as == BGP->my_as)
#else /* BGP_MULTI */
    if (as == BGP->current_bgp->this_as)
#endif /* BGP_MULTI */
	BIT_SET (peer->options, BGP_INTERNAL);
}


/* 
 * Initialize BGP peer and add to list of peers
 */
bgp_peer_t *
Add_BGP_Peer (char *name, prefix_t * prefix, 
	      prefix_t *usrc, interface_t *interface, int as, trace_t * tr)
{
    char str[MAXLINE];
    bgp_peer_t *peer;

    assert (name != NULL || prefix != NULL);

    if (name) {
        if ((peer = Find_BGP_Peer_ByID (name)) != NULL) {
	    return (NULL);
        }
    }
    if (prefix) {
        if ((peer = Find_BGP_Peer (prefix, usrc, interface)) != NULL) {
	    return (NULL);
        }
    }
#ifdef HAVE_IPV6
    if (prefix && prefix->family == AF_INET6)
        sprintf (str, "BGP4+ %s", (name)? name: prefix_toa (prefix));
    else
#endif /* HAVE_IPV6 */
    sprintf (str, "BGP %s", (name)? name: prefix_toa (prefix));

    peer = New (bgp_peer_t);
    peer->name = (name)? strdup (name): NULL;
    peer->trace = trace_copy (tr);
    set_trace (peer->trace, TRACE_PREPEND_STRING, str, 0);
#ifdef BGP_MULTI
    peer->local_bgp = BGP->current_bgp;
#endif /* BGP_MULTI */

#ifdef HAVE_IPV6
    if (prefix && prefix->family == AF_INET6)
#ifndef BGP_MULTI
	peer->view = 1;
#else /* BGP_MULTI */
	peer->view = BGP->current_bgp->view_no;
#endif /* BGP_MULTI */
    else
#endif /* HAVE_IPV6 */

#ifndef BGP_MULTI
    { peer->view = 0; BGP->num_ipv4_peers++; }
#else /* BGP_MULTI */
    { peer->view = BGP->current_bgp->view_no; BGP->num_ipv4_peers++; }
#endif /* BGP_MULTI */
    peer->index = bgp_obtain_index ();
    BGP->num_peers++;

    peer->gateway = (prefix)? 
	add_gateway (prefix, as, find_interface (prefix)): NULL;
    peer->bind_addr = Ref_Prefix (usrc);
    peer->bind_if = interface;
    peer->options = 0;
    peer_set_as (peer, as);

    peer->aliases = NULL;
    peer->sockfd = -1;
    peer->sockfd_in = -1;
    peer->listen_socket = NULL;

    peer->neighbor_list = -1;
    peer->default_weight = -1;

    peer->dlist_in = peer->dlist_out = -1;
    peer->flist_in = peer->flist_out = -1;
    peer->route_map_in = peer->route_map_out = -1;

    peer->read_ptr = peer->buffer;
    peer->start_ptr = peer->buffer;
    peer->packet = NULL;
    peer->read_ptr_in = peer->buffer_in;

    peer->state = BGPSTATE_IDLE;

    peer->maximum_prefix = 0;
    peer->pending_in_recv_open = 0;

    peer->Start_Interval = -1;
    peer->ConnectRetry_Interval = -1;
    peer->KeepAlive_Interval = -1;
    peer->HoldTime_Interval = -1;

    if (peer->gateway) {
        /* create timers */
        peer->timer_ConnectRetry = New_Timer (bgp_schedule_timer,
					 BGP->Default_ConnectRetry_Interval,
					 "ConnectRetry", peer);
        peer->timer_KeepAlive = New_Timer (bgp_schedule_timer,
				      BGP->Default_KeepAlive_Interval,
				      "KeepALive", peer);
        peer->timer_HoldTime = New_Timer (bgp_schedule_timer,
				     BGP->Default_HoldTime_Interval,
				     "HoldTime", peer);
        peer->timer_Start = New_Timer (bgp_schedule_timer, 
				   BGP->Default_Start_Interval,
				  "StartTime", peer);
        timer_set_flags (peer->timer_ConnectRetry, TIMER_ONE_SHOT);
	timer_set_flags (peer->timer_Start, TIMER_JITTER2, -25, 0);
        /* timer_set_jitter2 (peer->timer_KeepAlive, -25, 0); */
        timer_set_flags (peer->timer_HoldTime, TIMER_ONE_SHOT);
        timer_set_flags (peer->timer_Start, TIMER_ONE_SHOT);
        timer_set_flags (peer->timer_Start, TIMER_EXPONENT);
#define BGP_TIMER_START_MAX 4
	timer_set_flags (peer->timer_Start, TIMER_EXPONENT_MAX, 
					    BGP_TIMER_START_MAX);

        peer->send_queue = LL_Create (LL_DestroyFunction, Delete_Buffer, 0);
        pthread_mutex_init (&peer->send_mutex_lock, NULL);

        peer->ll_update_out = LL_Create (LL_DestroyFunction, 
					 delete_update_bucket, 0);
        pthread_mutex_init (&peer->update_mutex_lock, NULL);

    }
    pthread_mutex_init (&peer->mutex_lock, NULL);
    peer->schedule = New_Schedule ((name)? name: gateway_toa2 (peer->gateway), 
				    peer->trace);

    LL_Add (BGP->ll_bgp_peers, peer);
    mrt_thread_create2 (str, peer->schedule, NULL, NULL);
    return (peer);
}


static void
bgp_remove_me (bgp_peer_t *peer, bgp_peer_t *child)
{
    assert (peer->children);
    LL_Remove (peer->children, child);
    if (peer->state == BGPSTATE_DESTROYED &&
	    LL_GetCount (peer->children) <= 0) {
	LL_Destroy (peer->children);
        delete_schedule (peer->schedule);
        if (peer->name) Delete (peer->name);
        Destroy_Trace (peer->trace);
        pthread_mutex_destroy (&peer->mutex_lock);
	Delete (peer);
        mrt_thread_exit ();
    }
}


bgp_peer_t *
Remove_BGP_Peer (bgp_peer_t *peer)
{
    bgp_peer_t *bp;

    LL_Iterate (BGP->ll_bgp_peers, bp) {
	if (bp == peer) {
    	    LL_RemoveFn (BGP->ll_bgp_peers, peer, NULL);
	    return (peer);
	}
    }
    return (NULL);
}


static void
bgp_destroy_peer (bgp_peer_t *peer)
{
#ifdef HAVE_IPV6
    if (peer->gateway->prefix->family != AF_INET6)
#endif /* HAVE_IPV6 */
    BGP->num_ipv4_peers++;
    BGP->num_peers--;
    bgp_return_index (peer->index);

    if (peer->gateway) {
        /* peer is not on the list now, but the routunes should not care */
	if (peer->state == BGPSTATE_ESTABLISHED) {
	    bgp_peer_down (peer);
	}
        bgp_peer_dead (peer);

        Destroy_Timer (peer->timer_ConnectRetry);
        Destroy_Timer (peer->timer_KeepAlive);
        Destroy_Timer (peer->timer_HoldTime);
        Destroy_Timer (peer->timer_Start);

        LL_Destroy (peer->send_queue);
        pthread_mutex_destroy (&peer->send_mutex_lock);
        LL_Destroy (peer->ll_update_out);
        pthread_mutex_destroy (&peer->update_mutex_lock);

	assert (peer->children == NULL);
	if (peer->parent)
    	    schedule_event2 ("bgp_remove_me",
		      	     peer->parent->schedule, bgp_remove_me, 2, 
			     peer->parent, peer);
    }
    else {
        clear_schedule (peer->schedule);
	if (peer->children && LL_GetCount (peer->children) > 0) {
	    bgp_peer_t *child;
	    LL_Iterate (peer->children, child) {
		stop_bgp_peer (child, 1);
        	pthread_mutex_lock (&BGP->peers_mutex_lock);
		Destroy_BGP_Peer (child);
        	pthread_mutex_unlock (&BGP->peers_mutex_lock);
	    }
	    /* waiting all children are destroyed */
	    peer->state = BGPSTATE_DESTROYED;
	    return;
	}
	else {
	    /* delete now */
	    if (peer->children)
		LL_Destroy (peer->children);
	}
    }

    delete_schedule (peer->schedule);
    if (peer->name) Delete (peer->name);
    Destroy_Trace (peer->trace);
    pthread_mutex_destroy (&peer->mutex_lock);
    Delete (peer);
    mrt_thread_exit ();
}


void
Destroy_BGP_Peer (bgp_peer_t *peer)
{
    if (Remove_BGP_Peer (peer) == NULL)
	return;
    schedule_event2 ("bgp_destroy_peer",
		      peer->schedule, bgp_destroy_peer, 1, peer);
}


void
bgp_peer_down (bgp_peer_t * peer)
{
        int i;
	/* withdraw all routes associated with the peer */
        for (i = 0; i < MAX_BGP_VIEWS; i++) {
	    if (BGP->views[i])
	        view_delete_peer (BGP->views[i], peer);
        }
        if (BGP->peer_down_call_fn)
	    BGP->peer_down_call_fn (peer);
}


/*
 * if peer is leaving established state, withdraw routes and free resources
 * close both sockets, reset inputs, stop timers, and clear schedule queue.
 */
void 
bgp_peer_dead (bgp_peer_t * peer)
{
    trace (TR_INFO, peer->trace, "shutdown the connection\n");

    if (peer->sockfd >= 0) {
        /* delete sockfd stuff */
	select_delete_fd (peer->sockfd);
	trace (TR_INFO, peer->trace, "outgoing connection (fd %d) closed\n",
	       peer->sockfd);
	peer->sockfd = -1;
    }
    /* keep incoming socket open */

    if (peer->gateway) {
        /* stop all timers except start */
        Timer_Turn_OFF (peer->timer_ConnectRetry);
        Timer_Turn_OFF (peer->timer_KeepAlive);
        Timer_Turn_OFF (peer->timer_HoldTime);
        /* Timer_Turn_OFF (peer->timer_Start); */
	/* delete output */
	pthread_mutex_lock (&peer->send_mutex_lock);
        LL_Clear (peer->send_queue);
	pthread_mutex_unlock (&peer->send_mutex_lock);
	pthread_mutex_lock (&peer->update_mutex_lock);
        LL_Clear (peer->ll_update_out);
	pthread_mutex_unlock (&peer->update_mutex_lock);
        if (peer->attr) {
            bgp_deref_attr (peer->attr);
	    peer->attr = NULL;
	}
        if (peer->ll_withdraw) {
            LL_Destroy (peer->ll_withdraw);
	    peer->ll_withdraw = NULL;
	}
        if (peer->ll_announce) {
            LL_Destroy (peer->ll_announce);
	    peer->ll_announce = NULL;
	}
    }
    else {
	if (peer->children && LL_GetCount (peer->children) > 0) {
	    bgp_peer_t *child;
	    LL_Iterate (peer->children, child)
                bgp_stop_peer (child);
	}
    }

    /* should flush input buffer here */
    peer->read_ptr = peer->buffer;
    peer->start_ptr = peer->buffer;
    peer->packet = NULL;
    peer->read_ptr_in = peer->buffer_in;
    clear_schedule (peer->schedule);
    if (peer->new_as > 0 && peer->gateway) {
	int old_as = peer->gateway->AS;
	peer_set_as (peer, peer->new_as);
	trace (TR_INFO, peer->trace, "AS number changed from %d to %d\n",
	       old_as, peer->new_as);
	peer->new_as = 0;
    }
}


int
bgp_kill_peer (gateway_t *gateway)
{
    bgp_peer_t *peer;

    if ((peer = find_bgp_peer (gateway)) != NULL) {
	bgp_sm_process_event (peer, BGPEVENT_STOP);
	return (1);
    }
    return (-1);
}


void
bgp_start_peer (bgp_peer_t *peer)
{
    assert (peer);
    schedule_event2 ("bgp_sm_process_event",
		      peer->schedule, bgp_sm_process_event,
		      2, peer, BGPEVENT_START);
}


void
bgp_stop_peer (bgp_peer_t *peer)
{
    assert (peer);
    schedule_event2 ("bgp_sm_process_event",
		      peer->schedule, bgp_sm_process_event,
		      2, peer, BGPEVENT_STOP);
}


void
bgp_kill_all ()
{
    bgp_peer_t *peer;

    pthread_mutex_lock (&BGP->peers_mutex_lock);
    LL_Iterate (BGP->ll_bgp_peers, peer) {
	if (peer->parent == NULL) /* children will be signaled by the parent */
            bgp_stop_peer (peer);
    }
    pthread_mutex_unlock (&BGP->peers_mutex_lock);
}


int 
set_BGP (int first, ...)
{
    va_list ap;
    enum BGP_ATTR attr;
    int id;

    assert (BGP);
    pthread_mutex_lock (&BGP->mutex_lock);
    /* Process the Arguments */
    va_start (ap, first);
    for (attr = (enum BGP_ATTR) first; attr; 
				       attr = va_arg (ap, enum BGP_ATTR)) {
	switch (attr) {
#ifndef BGP_MULTI
	case BGP_MY_AS:
	    BGP->my_as = va_arg (ap, int);
	    break;
	case BGP_MY_ID:
	    BGP->my_id = va_arg (ap, int);
	    if (!BGP->cluster_id)
		BGP->cluster_id = BGP->my_id;
	    break;
#else /* BGP_MULTI */
	case BGP_CURRENT_BGP:
	    BGP->current_bgp = va_arg (ap, bgp_local_t *);
	    break;
#endif /* BGP_MULTI */
	case BGP_PEER_DOWN_FN:
	    BGP->peer_down_call_fn = va_arg (ap, void_fn_t);
	    break;
	case BGP_PEER_ESTABLISHED_FN:
	    BGP->peer_established_call_fn = va_arg (ap, void_fn_t);
	    break;
	case BGP_RECV_UPDATE_FN:
	    BGP->update_call_fn = va_arg (ap, int_fn_t);
	    break;
	case BGP_SEND_UPDATE_FN:
	    BGP->send_update_call_fn = va_arg (ap, int_fn_t);
	    break;
/*
	case BGP_RT_UPDATE_FN:
	    BGP->rt_update_call_fn = va_arg (ap, rib_update_route_t);
	    break;
*/
	case BGP_STATE_CHANGE_FN:
	    BGP->state_change_fn = va_arg (ap, void_fn_t);
	    break;
	case BGP_ACCEPT_ALL_PEERS:
	    BGP->accept_all_peers = va_arg (ap, int);
	    break;
	case BGP_LPORT:
	    BGP->lport = va_arg (ap, int);
	    break;
	case BGP_CPORT:
	    BGP->cport = va_arg (ap, int);
	    break;
	case BGP_TRACE_STRUCT:
	    BGP->trace = va_arg (ap, trace_t *);
	    break;
	case BGP_DUMP_ROUTE_FORM:
	    id = va_arg (ap, int);
	    assert (id >= 0 && id < MAX_BGP_VIEWS);
	    if (BGP->dump_route_form[id])
		free (BGP->dump_route_form[id]);
	    BGP->dump_route_form[id] = va_arg (ap, char *);
	    if (BGP->dump_route_form[id]) {
		BGP->dump_route_form[id] = strdup (BGP->dump_route_form[id]);
	        BGP->dump_route_interval[id] = va_arg (ap, int);
	        BGP->dump_route_type[id] = va_arg (ap, int);
	        BGP->dump_route_time[id] = 0;
	    }
	    break;
	case BGP_DUMP_UPDATE_FORM:
	    if (BGP->dump_update_form)
		free (BGP->dump_update_form);
	    BGP->dump_update_form = va_arg (ap, char *);
	    if (BGP->dump_update_form) {
	        BGP->dump_update_form = strdup (BGP->dump_update_form);
	        BGP->dump_update_interval = va_arg (ap, int);
	        BGP->dump_update_time = 0;
	        BGP->dump_update_types = va_arg (ap, u_long);
	        BGP->dump_update_family = va_arg (ap, int);
	    }
	    break;
	/* 0 -- receiving, 1 -- sending, 2 - both */
	case BGP_DUMP_DIRECTION:
	    BGP->dump_direction = va_arg (ap, int);
	    break;
	default:
	    assert (0);
	    break;
	}
    }
    va_end (ap);
    pthread_mutex_unlock (&BGP->mutex_lock);

    return (1);
}



/* find_bgp_peer
 * given a prefix, find the corresponding bgp peer
 * structure.
 */
bgp_peer_t *
find_bgp_peer (gateway_t * gateway)
{
    bgp_peer_t *peer;

    pthread_mutex_lock (&BGP->peers_mutex_lock);
    LL_Iterate (BGP->ll_bgp_peers, peer) {
	if (peer->gateway == gateway) {
    	    pthread_mutex_unlock (&BGP->peers_mutex_lock);
	    return (peer);
	}
    }
    pthread_mutex_unlock (&BGP->peers_mutex_lock);
    return (NULL);
}


void
bgp_start_listening (bgp_peer_t *peer)
{
    assert (peer->listen_socket == NULL);

    peer->listen_socket = init_BGP_listen (peer->bind_addr, peer->bind_if);
    if (peer->listen_socket == NULL && (peer->bind_addr || peer->bind_if)) {
        peer->listen_socket = init_BGP_listen (NULL, NULL);
    }
}


void
start_bgp_peer (bgp_peer_t *peer)
{
    /* I can not start listening here since
       bind address may be supplied later */
    timer_set_flags (peer->timer_Start, TIMER_EXPONENT_SET, 0);
    Timer_Turn_ON (peer->timer_Start);
}


static void
stop_bgp_peer (bgp_peer_t *peer, int lockf)
{
    if (peer->sockfd_in >= 0) {
        /* delete sockfd_in stuff */
	select_delete_fd (peer->sockfd_in);
	trace (TR_INFO, peer->trace, "incoming connection (fd %d) closed\n",
	       peer->sockfd_in);
	peer->sockfd_in = -1;
        peer->pending_in_recv_open = 0;
    }
    if (peer->listen_socket) {
        pthread_mutex_lock (&peer->listen_socket->mutex_lock);
	peer->listen_socket->ref_count--;
        if (peer->listen_socket->ref_count <= 0) {
            if (lockf) pthread_mutex_lock (&BGP->mutex_lock);
	    LL_Remove (BGP->ll_listen_sockets, peer->listen_socket);
            if (lockf) pthread_mutex_unlock (&BGP->mutex_lock);
	    Deref_Prefix (peer->listen_socket->prefix);
            select_delete_fd (peer->listen_socket->sockfd);
            pthread_mutex_destroy (&peer->listen_socket->mutex_lock);
	    Delete (peer->listen_socket);
	    peer->listen_socket = NULL;
	}
	else
            pthread_mutex_unlock (&peer->listen_socket->mutex_lock);
    }

    timer_set_flags (peer->timer_Start, TIMER_EXPONENT_SET, 0);
    Timer_Turn_ON (peer->timer_Start);
}


/*
 *  start listening on socket 
 */
void
start_bgp (void)
{
    /* move to init_BGP */
    /* mrt_thread_create2 ("BGP", BGP->schedule, NULL, NULL); */
}


void 
stop_bgp (void)
{
    bgp_peer_t *peer;
    prefix_t *prefix;
    int i;

    pthread_mutex_lock (&BGP->peers_mutex_lock);
    LL_Iterate (BGP->ll_bgp_peers, peer) {
	if (peer->parent == NULL) {
	    bgp_peer_t *prev;
	    prev = LL_GetPrev (BGP->ll_bgp_peers, peer);
	    stop_bgp_peer (peer, 0);
	    Destroy_BGP_Peer (peer);
	    peer = prev;
	}
    }
    pthread_mutex_unlock (&BGP->peers_mutex_lock);

    pthread_mutex_lock (&BGP->mutex_lock);
/*
    if (BGP->sockfd >= 0) {
        select_delete_fd (BGP->sockfd);
	BGP->sockfd = -1;
	BGP->sockfd_count = 0;
    }
*/

    LL_Iterate (BGP->ll_networks, prefix) {
        if (MRT->rib_redistribute_network)
            MRT->rib_redistribute_network (PROTO_BGP, prefix, 0);
    }
    LL_Clear (BGP->ll_networks);

    for (i = 0; i <= PROTO_MAX; i++) {
        if (BGP4_BIT_TEST (BGP->redistribute_mask, i)) {
            if (MRT->rib_redistribute_request)
                MRT->rib_redistribute_request (PROTO_BGP, i, 0);
	}
    }
    BGP->redistribute_mask = 0;

    for (i = 0; i < MAX_BGP_VIEWS; i++) {
        /* delete aggregate */
    }
   /* just leave BGP main thread idling */
 /* schedule_event2 ("mrt_thread_exit", BGP->schedule, mrt_thread_exit, 0); */
    pthread_mutex_unlock (&BGP->mutex_lock);
    /* asynchronous return */
}


#ifdef BGP_MULTI
/* 
 * dump various BGP stats to a socket
 * usually called by UII (user interactive interface)
 */
int 
show_bgp_local (uii_connection_t * uii)
{
    bgp_local_t *local;
    char tmpx[MAXLINE];

    pthread_mutex_lock (&BGP->mutex_lock);
#ifdef HAVE_IPV6
    uii_add_bulk_output (uii, "Routing Protocol is \"BGP4+\"\n");
#else
    uii_add_bulk_output (uii, "Routing Protocol is \"BGP\"\n");
#endif /* HAVE_IPV6 */

    if (BGP->trace != NULL)
	uii_add_bulk_output (uii, "Trace flags 0x%x\n\n", BGP->trace->flags);
    pthread_mutex_unlock (&BGP->mutex_lock);

    pthread_mutex_lock (&BGP->locals_mutex_lock);
    LL_Iterate (BGP->ll_bgp_locals, local) {
        pthread_mutex_lock (&local->mutex_lock);
	uii_add_bulk_output (uii, "Local Router BGP AS%d\n", local->this_as);
	uii_add_bulk_output (uii, "    AS Number: %d    Router ID: %s\n",
			     local->this_as, 
		inet_ntop (AF_INET, &local->this_id, tmpx, sizeof (tmpx)));
	uii_add_bulk_output (uii, "    Local View Number: %d\n\n", local->view_no);
        pthread_mutex_unlock (&local->mutex_lock);
    }
    pthread_mutex_unlock (&BGP->locals_mutex_lock);
    return (1);
}
#endif BGP_MULTI


/* 
 * dump various BGP stats to a socket
 * usually called by UII (user interactive interface)
 */
int 
show_f_bgp (uii_connection_t * uii, int family)
{
    return (show_f_bgp_summary (uii, family, FALSE));
}


int 
show_f_bgp_summary (uii_connection_t * uii, int family, int summary)
{
    bgp_peer_t *peer;
    char tmpx[MAXLINE], date[MAXLINE], peername[MAXLINE];
    time_t now;

    pthread_mutex_lock (&BGP->mutex_lock);
    time (&now);
#ifdef HAVE_IPV6
    uii_add_bulk_output (uii, "Routing Protocol is \"BGP4+\"\n");
#else
    uii_add_bulk_output (uii, "Routing Protocol is \"BGP\"\n");
#endif /* HAVE_IPV6 */
#ifndef BGP_MULTI
    /* We should update the config output to display the ASes represented
       by the current configuration */
    uii_add_bulk_output (uii, "Local Router ID is %s\n", 
		   inet_ntop (AF_INET, &BGP->my_id, tmpx, sizeof (tmpx)));
    uii_add_bulk_output (uii, "Local AS is %d\n", BGP->my_as);
#else /* BGP_MULTI */
    /* We should update the config output to display the ASes represented
       by the current configuration.  This is somewhat out of context
       -- binkertn */
    uii_add_bulk_output (uii, "Current Router ID is %s\n", 
		   inet_ntop (AF_INET, &BGP->current_bgp->this_id, tmpx, sizeof (tmpx)));
    uii_add_bulk_output (uii, "Current AS is %d\n", BGP->current_bgp->this_as);
#endif /* BGP_MULTI */
if (!summary) {
    if (BGP->trace != NULL)
	uii_add_bulk_output (uii, "Trace flags 0x%x\n", BGP->trace->flags);
    /* uii_add_bulk_output (uii, "%d Attribute Blocks\n", 
	              HASH_GetCount (BGP->attr_hash)); */
    if (BGP->bgp_num_active_route_head > 0)
        uii_add_bulk_output (uii, "bgp active route heads: %d\n",
		       BGP->bgp_num_active_route_head);
    if (BGP->bgp_num_active_route_node > 0)
        uii_add_bulk_output (uii, "bgp active route nodes: %d\n",
		       BGP->bgp_num_active_route_node);
}
    pthread_mutex_unlock (&BGP->mutex_lock);

if (summary) {
	uii_add_bulk_output (uii, "\n");
	uii_add_bulk_output (uii, "%-25s %1s %5s %11s %7s %3s/%-3s %5s %s\n",
		"Neighbor", "V", "AS", "Update(R/S)", "Notify",
		"Up", "Dwn", "Hours", "State");
}
    pthread_mutex_lock (&BGP->peers_mutex_lock);
    LL_Iterate (BGP->ll_bgp_peers, peer) {
	char xxx[MAXLINE];
	int bgp4plus_version = 0;
	int elapsed = (peer->time > 0)?(now - peer->time):-1;

	/* skip template or group */
	if (peer->gateway == NULL)
	    continue;

	if (family != 0 && peer->gateway->prefix->family != family)
	    continue;

        pthread_mutex_lock (&peer->mutex_lock);

#ifdef HAVE_IPV6
	if (BIT_TEST (peer->options, BGP_BGP4PLUS_01))
	    bgp4plus_version = 1;
#endif /* HAVE_IPV6 */

        if (peer->name)
	    sprintf (peername, "%s (%s)", peer->name, 
		     prefix_toa (peer->gateway->prefix));
        else if (peer->description)
	    sprintf (peername, "%s (%s)", peer->description, 
		     prefix_toa (peer->gateway->prefix));
	else
	    sprintf (peername, "%s", prefix_toa (peer->gateway->prefix));

if (summary) {
	peername[25] = '\0';
	sprintf (date, "-----");
	if (elapsed >= 0) {
	    if (elapsed / 3600 > 99)
	        sprintf (date, "%5d", elapsed / 3600);
	    else
	        sprintf (date, "%d.%02d", elapsed / 3600,
                            (elapsed % 3600) * 100 / 3600);
	}

	uii_add_bulk_output (uii, 
		"%-25s %1s %5d %5d/%-5d %3d/%-3d %3d/%-3d %5s %s\n",
            peername,
	    (peer->gateway->prefix->family == AF_INET)?"4": 
	        (bgp4plus_version == 0)?"-": "+",
	    peer->gateway->AS, 
	    peer->num_updates_recv, peer->num_updates_sent,
	    peer->num_notifications_recv, peer->num_notifications_sent,
	    peer->num_connections_established, peer->num_connections_dropped,
	    date,
	    sbgp_states[peer->state]);
}
else {

        time2date (elapsed, date);
	strcpy (xxx, "BGP4");
#ifdef HAVE_IPV6
	if (peer->gateway->prefix->family == AF_INET6)
	    sprintf (xxx, "BGP4+ draft %d", bgp4plus_version);
#endif /* HAVE_IPV6 */

	uii_add_bulk_output (uii, "\n");
	uii_add_bulk_output (uii, "  peer %s AS%d on %s [%s] %s\n", peername,
		       peer->gateway->AS, 
	    (peer->gateway->interface)? peer->gateway->interface->name: "???",
		       sbgp_states[peer->state], date);

	uii_add_bulk_output (uii, 
		 "    Router ID %s (index #%d) View #%d",
	    inet_ntop (AF_INET, &peer->gateway->routerid, tmpx, sizeof (tmpx)),
		 peer->index, peer->view);
	if (peer->default_weight >= 0)
	    uii_add_bulk_output (uii, 
		 " weight %d", peer->default_weight);
	uii_add_bulk_output (uii, " %c%s\n",
		 BIT_TEST (peer->options, BGP_INTERNAL)? 'i': 'e', xxx);
	/* if (peer->state == BGPSTATE_OPENSENT ||
                peer->state == BGPSTATE_OPENCONFIRM ||
	        peer->state == BGPSTATE_ESTABLISHED) */
	if (peer->local_addr)
	    uii_add_bulk_output (uii, 
		 "    Local Address %s (socket %d)\n", 
		 prefix_toa (peer->local_addr), peer->sockfd);
#ifdef BGP_MULTI
	    uii_add_bulk_output (uii, "    Local AS: %-5d  Local ID: %s\n",
				 peer->local_bgp->this_as,
		inet_ntop (AF_INET, &peer->local_bgp->this_id, tmpx, sizeof (tmpx)));
#endif /* BGP_MULTI */
	if (peer->listen_socket && peer->listen_socket->prefix)
	    uii_add_bulk_output (uii, 
		 "    Listening at  %s (socket %d)\n",
		 peer->listen_socket->prefix?
		     prefix_toa (peer->listen_socket->prefix): "*",
		 peer->listen_socket->sockfd);
	if (peer->timer_KeepAlive->time_next_fire <= 0)
	    uii_add_bulk_output (uii, "    KeepAlive Off");
	else
	    uii_add_bulk_output (uii, "    KeepAlive %d",
			   peer->timer_KeepAlive->time_next_fire - now);
	if (peer->timer_Start->time_next_fire <= 0)
	    uii_add_bulk_output (uii, "  Starttimer Off");
	else
	    uii_add_bulk_output (uii, "  Starttimer %d",
			   peer->timer_Start->time_next_fire - now);
	if (peer->timer_HoldTime->time_next_fire <= 0)
	    uii_add_bulk_output (uii, "  Holdtime Off");
	else
	    uii_add_bulk_output (uii, "  Holdtime %d",
			   peer->timer_HoldTime->time_next_fire - now);
	if (peer->timer_ConnectRetry->time_next_fire <= 0)
	    uii_add_bulk_output (uii, "  ConnectRetry Off\n");
	else
	    uii_add_bulk_output (uii, "  ConnectRetry %d\n",
			   peer->timer_ConnectRetry->time_next_fire - now);
	uii_add_bulk_output (uii, 
		"    Packets Recv %d  Updates Recv %d  Notifications Recv %d\n",
		     peer->num_packets_recv, peer->num_updates_recv,
		     peer->num_notifications_recv);
	uii_add_bulk_output (uii, 
		"    Packets Sent %d  Updates Sent %d  Notifications Sent %d\n",
		     peer->num_packets_sent, peer->num_updates_sent,
		     peer->num_notifications_sent);
	uii_add_bulk_output (uii, 
		"    Connections Established %d  Connections dropped %d\n",
		     peer->num_connections_established, 
		     peer->num_connections_dropped);

}
        pthread_mutex_unlock (&peer->mutex_lock);
    }
    pthread_mutex_unlock (&BGP->peers_mutex_lock);
    return (1);
}


/* bgp_debug
 * set the debug flags or file for BGP
 * called by config routines
 */
int 
bgp_debug (uii_connection_t * uii, int opt_num, char *s)
{
    char *token, *arg = s;

    pthread_mutex_lock (&BGP->mutex_lock);
    /* get flag */
    if (opt_num == 0 || (token = uii_parse_line (&s)) == NULL) {
        set_trace (BGP->trace, TRACE_FLAGS, TR_ALL, NULL);
    }
    else {
        u_long flag;

        if ((flag = trace_flag (token)) == 0) {
            config_notice (TR_ERROR, uii, "unknown debug option\n");
	    Delete (arg);
	    return (-1);
	}
        set_trace (BGP->trace, TRACE_FLAGS, flag, NULL);

        /* get file name (OPTIONAL) */
        if ((token = uii_parse_line (&s)) == NULL) {
	    Delete (arg);
	    return (1);
        }
        set_trace (BGP->trace, TRACE_LOGFILE, token, NULL);
    }
    pthread_mutex_unlock (&BGP->mutex_lock);
    return (1);
}


int 
trace_bgp (uii_connection_t * uii)
{
    pthread_mutex_lock (&BGP->mutex_lock);
    set_trace (BGP->trace, TRACE_ADD_FLAGS, TR_ALL, NULL);
    pthread_mutex_unlock (&BGP->mutex_lock);
    return (1);
}


int 
no_trace_bgp (uii_connection_t * uii)
{
    pthread_mutex_lock (&BGP->mutex_lock);
    set_trace (BGP->trace, TRACE_DEL_FLAGS, TR_ALL, NULL);
    pthread_mutex_unlock (&BGP->mutex_lock);
    return (1);
}


static void
bgp_put_a_line (uii_connection_t *uii, bgp_route_t *route, time_t now)
{
    char buff[MAXLINE], *cp = buff;
    int c1, c2;

    c1 = (route == route->head->active) ? '>' : '*';
    if (BIT_TEST (route->flags, BGP_RT_FILTERED))
	c1 = ' ';
    if (BIT_TEST (route->head->state_bits, VRTS_SUPPRESS))
	c1 = 's';
    if (BIT_TEST (route->head->state_bits, VRTS_DELETE))
	c1 = 'D';

    strcpy (cp, "");
/* I need to change ipma stuff that depends on route format */
#if 0
    if (BGP4_BIT_TEST (route->attr->attribs, PA4_TYPE_METRIC)) {
	sprintf (cp, "%6ld", route->attr->multiexit);
	cp += strlen (cp);
    }
    else {
	sprintf (cp, "%6s", "");
	cp += strlen (cp);
    }
    if (BGP4_BIT_TEST (route->attr->attribs, PA4_TYPE_LOCALPREF)) {
	sprintf (cp, " %6ld", route->attr->local_pref);
	cp += strlen (cp);
    }
    else {
	sprintf (cp, " %6s", "");
	cp += strlen (cp);
    }
    sprintf (cp, " %6d", route->weight);
    cp += strlen (cp);
#endif
    if (BGP4_BIT_TEST (route->attr->attribs, PA4_TYPE_ASPATH)) {
	int l;
#if 0
	*cp++ = ' ';
#endif
	strcpy (cp, aspath_toa (route->attr->aspath));

	if ((l = strlen (cp)) > 0) {
	    /* check to see if it's null as path */
	    cp += l;
	    *cp++ = ' ';
	    *cp = '\0';
	}
    }
    if (BGP4_BIT_TEST (route->attr->attribs, PA4_TYPE_ORIGIN)) {
	*cp++ = origin2char (route->attr->origin);
	*cp = '\0';
    }

    c2 = BIT_TEST (route->attr->options, BGP_INTERNAL)? 'i': ' ';
    c2 = (BIT_TEST (route->flags, BGP_RT_AGGREGATED))? 'a': c2;
    rib_show_route_line (uii, c1, c2, route->attr->type, 
			 0, now - route->time,
			 route->head->prefix, 
			 route->attr->nexthop->prefix,
/* (route->attr->link_local)? route->attr->link_local: route->attr->nexthop, */
			 route->attr->gateway, buff);
}


/* bgp_dump_view
 * dump BGP routes to socket. usually called by UII
 */
void
bgp_dump_view (uii_connection_t * uii, view_t *view,
		as_regexp_code_t *code, bgp_peer_t *peer, condition_t *cond)
{
    bgp_route_head_t *route_head;
    bgp_route_t *route;
    time_t now;

    if (peer) {
        view = BGP->views[peer->view];
        uii_add_bulk_output (uii, "Peer %s View %d\n", 
		       prefix_toa (peer->gateway->prefix), view->viewno);
    }
    else {
        uii_add_bulk_output (uii, "View #%d %s %d routes\n",
		   view->viewno, family2string (view->family), 
		   view->num_active_routes);
    }
    uii_add_bulk_output (uii,
	    "Status code: s suppressed, * valid, > best, i - internal\n");
    uii_add_bulk_output (uii,
	    "Origin codes: i - IGP, e - EGP, ? - incomplete, a - aggregate\n");
    uii_add_bulk_output (uii, "\n");
#if 0
    rib_show_route_head (uii, "Metric LocPrf Weight Path");
#else
    rib_show_route_head (uii, "Path");
#endif

    time (&now);

    VIEW_RADIX_WALK (view, route_head) {

if (peer) {
	    /* show only active one */
	if ((route = route_head->active) &&
	    (cond == NULL || apply_condition (cond, route_head->prefix) > 0)) {

	    if (BITX_TEST (&route_head->peer_mask, peer->index)) {
		bgp_put_a_line (uii, route, now);
		if (cond) {
		    bgp_uii_attr (uii, route->attr);
		}
	    }
	}
}
else {
    if (cond == NULL || apply_condition (cond, route_head->prefix) > 0) {

#ifdef notdef
	/* can not use continue in RADIX_WALK macro */
	if (BIT_TEST (route_head->state_bits, VRTS_DELETE))
	    goto skip;
#endif

	LL_Iterate (route_head->ll_imported, route) {

	    if (code && as_regexp_exec (code, NULL) < 0)
		continue;

	    bgp_put_a_line (uii, route, now);
	}

	LL_Iterate (route_head->ll_routes, route) {

	    if (BGP4_BIT_TEST (route->attr->attribs, PA4_TYPE_ASPATH)) {
		if (code && as_regexp_exec (code, route->attr->aspath) < 0)
		    continue;
	    }
	    else {
		if (code && as_regexp_exec (code, NULL) < 0)
		    continue;
	    }
	    bgp_put_a_line (uii, route, now);
	    if (cond) {
		bgp_uii_attr (uii, route->attr);
	    }

	}
    }
}
#ifdef notdef
      skip:
#endif
    }
    VIEW_RADIX_WALK_END;
}

#ifdef BGP_MULTI
/* show_bgp_routing table
 * dump BGP routes to socket. usually called by UII
 */
int
show_bgp_rt_view (uii_connection_t * uii, view_t *view)
{
  if (!view) return(0);
  
  /* pthread_mutex_lock (&BGP->mutex_lock); */
  view_open (view);
  bgp_dump_view (uii, view, NULL, NULL, NULL);
  view_close (view);
  /* pthread_mutex_unlock (&BGP->mutex_lock); */
  uii_send_bulk_data (uii);
  return (1);
}

int
show_bgp_rt_view_no(uii_connection_t * uii, int view_no)
{
  if (view_no >= MAX_BGP_VIEWS)
    return (0);
  
  return show_bgp_rt_view(uii, BGP->views[view_no]);
}

int
show_bgp_rt_as(uii_connection_t * uii, int as)
{
  bgp_local_t *local = BGP->router_bgp[as];
  if (!local)
    return (0);

  return show_bgp_rt_view_no(uii, local->view_no);
}
#endif /* BGP_MULTI */

/* show_bgp_routing table
 * dump BGP routes to socket. usually called by UII
 */
int 
show_f_bgp_rt_regexp (uii_connection_t * uii, int family, char *expr)
{
    as_regexp_code_t *code = NULL;
    int pos = 0;
#ifdef BGP_MULTI
    int count;
#endif /* BGP_MULTI */
    view_t *view;

    if (expr && (code = as_regexp_comp (expr, &pos)) == NULL) {
    	uii_add_bulk_output (uii, "%s\n", expr);
    	uii_add_bulk_output (uii, "%*c\n", pos, '^');
	Delete (expr);
	return (-1);
    }
    Delete (expr);
#ifdef BGP_MULTI
    for (count = 0; count < MAX_BGP_VIEWS; count++) {
      view = BGP->views[count];
      if (!view) continue;
#else /* BGP_MULTI */
    view = BGP->views[0];
#endif /* BGP_MULTI */

if (family == 0 || family == AF_INET) {
    /* pthread_mutex_lock (&BGP->mutex_lock); */
    view_open (view);
    bgp_dump_view (uii, view, code, NULL, NULL);
    view_close (view);
    /* pthread_mutex_unlock (&BGP->mutex_lock); */
}

#ifndef BGP_MULTI
#ifdef HAVE_IPV6
if (family == 0 || family == AF_INET6) {
    if (family == 0)
        uii_add_bulk_output (uii, "\n");
    /* pthread_mutex_lock (&BGP->mutex_lock); */
    view = BGP->views[1];
    view_open (view);
    bgp_dump_view (uii, view, code, NULL, NULL);
    view_close (view);
    /* pthread_mutex_unlock (&BGP->mutex_lock); */
}
#endif /* HAVE_IPV6 */
#endif /* BGP_MULTI */

    if (code) Delete (code);

    uii_send_bulk_data (uii);
#ifdef BGP_MULTI
    }
#endif /* BGP_MULTI */
    return (1);
}


/* 
 * show BGP routes to match
 */
int 
show_bgp_routes (uii_connection_t * uii, prefix_t *prefix, char *options)
{
    condition_t *condition;
    int permit = 1, refine = 0, exact = 0;
    prefix_t *wildcard = NULL;
#ifdef BGP_MULTI
    int count;
#endif /* BGP_MULTI */
    view_t *view;

    if (options) {
	get_alist_options (options, &wildcard, &refine, &exact);
        Delete (options);
    }
    condition = New (condition_t);
    condition->permit = permit;
    /* expects Ref_Prefix can handle even a case of prefix == NULL */
    condition->prefix = prefix;
    condition->wildcard = wildcard;
    condition->exact = exact;
    condition->refine = refine;

#ifdef BGP_MULTI
    for(count = 0; count < MAX_BGP_VIEWS; count++) {
      view = BGP->views[count];
      if (!view) continue;
#else /* BGP_MULTI */
#ifdef HAVE_IPV6
    if (prefix->family == AF_INET6)
	view = BGP->views[1];
    else
#endif /* HAVE_IPV6 */
      view = BGP->views[0];
#endif /* BGP_MULTI */

    /* pthread_mutex_lock (&BGP->mutex_lock); */
    view_open (view);
    bgp_dump_view (uii, view, NULL, NULL, condition);
    view_close (view);
    /* pthread_mutex_unlock (&BGP->mutex_lock); */
    Deref_Prefix (prefix);
    Deref_Prefix (wildcard);
    Delete (condition);

    uii_send_bulk_data (uii);
#ifdef BGP_MULTI
    }
#endif /* BGP_MULTI */

    return (1);
}


#ifdef notdef
/* bgp_get_attr
 * attribute are stored in global hash. This 
 * 1) fetches attr if exists 
 * 2) or creates new attr
 */
bgp_attr_t *
bgp_get_attr (bgp_attr_t * attr)
{
    bgp_attr_t *tmpx;

    tmpx = HASH_Lookup (BGP->attr_hash, attr);

    if (tmpx == NULL) {
	HASH_Insert (BGP->attr_hash, bgp_ref_attr (attr));
	tmpx = attr;
    }

    return (tmpx);
}
#endif


int
show_f_bgp_neighbors_errors (uii_connection_t * uii, int family, 
			     char *peer_or_star)
{
    bgp_peer_t *peer;

    pthread_mutex_lock (&BGP->peers_mutex_lock);
    if (strcmp (peer_or_star, "*") != 0) {
	prefix_t *prefix = ascii2prefix (0, peer_or_star);
	/* XXX this happens due to a bug */
	if (prefix == NULL) {
	    if ((peer = Find_BGP_Peer_ByID (peer_or_star)) != NULL) {
		print_error_list (uii, peer->trace);
	    }
	    else {
                config_notice (TR_ERROR, uii, "Peer %s does not exist\n",
			       peer_or_star);
	        Delete (peer_or_star);
    		pthread_mutex_unlock (&BGP->peers_mutex_lock);
	        return (-1);
	    }
	}
        else if ((peer = Find_BGP_Peer (prefix, NULL, NULL)) == NULL) {
            config_notice (TR_ERROR, uii, "Peer %s does not exist\n",
		prefix_toa (prefix));
            Deref_Prefix (prefix);
	    Delete (peer_or_star);
    	    pthread_mutex_unlock (&BGP->peers_mutex_lock);
            return (-1);
        }
	else {
	    print_error_list (uii, peer->trace);
            Deref_Prefix (prefix);
	}
    }
    else {
        LL_Iterate (BGP->ll_bgp_peers, peer) {
	    if (peer->gateway == NULL)
		continue;
	    if (family != 0 && peer->gateway->prefix->family != family)
		continue;
	    print_error_list (uii, peer->trace);
	}
    }
    pthread_mutex_unlock (&BGP->peers_mutex_lock);
    Delete (peer_or_star);
    return (1);
}


int
show_f_bgp_neighbors_routes (uii_connection_t * uii, int family, 
			     char *peer_or_star)
{
    bgp_peer_t *peer;

    pthread_mutex_lock (&BGP->peers_mutex_lock);
    if (strcmp (peer_or_star, "*") != 0) {
	prefix_t *prefix = ascii2prefix (0, peer_or_star);
	/* XXX this happens due to a bug */
	if (prefix == NULL) {
	    if ((peer = Find_BGP_Peer_ByID (peer_or_star)) != NULL) {
		print_error_list (uii, peer->trace);
	    }
	    else {
                config_notice (TR_ERROR, uii, "Peer %s does not exist\n",
			       peer_or_star);
	        Delete (peer_or_star);
    		pthread_mutex_unlock (&BGP->peers_mutex_lock);
	        return (-1);
	    }
	}
        else if ((peer = Find_BGP_Peer (prefix, NULL, NULL)) == NULL) {
            config_notice (TR_ERROR, uii, "Peer %s does not exist\n",
			   prefix_toa (prefix));
            Deref_Prefix (prefix);
	    Delete (peer_or_star);
    	    pthread_mutex_unlock (&BGP->peers_mutex_lock);
            return (-1);
        }
	else {
	    bgp_dump_view (uii, NULL, NULL, peer, NULL);
            Deref_Prefix (prefix);
	}
    }
    else {
	int first = 1;
        LL_Iterate (BGP->ll_bgp_peers, peer) {
	    if (peer->gateway == NULL)
		continue;
	    if (family != 0 && peer->gateway->prefix->family != family)
		continue;
            if (first) first = 0;
	    else uii_add_bulk_output (uii, "\n");
	    /* XXX DEADLOCK MAY HAPPEN */
	    bgp_dump_view (uii, NULL, NULL, peer, NULL);
	}
    }
    pthread_mutex_unlock (&BGP->peers_mutex_lock);
    Delete (peer_or_star);
    return (1);
}


int
bgp_check_attr (bgp_peer_t * peer, bgp_attr_t * attr, int as)
{
    int peer_as;

    if (attr == NULL)
	return (0);

    assert (peer);

    if (!BGP4_BIT_TEST (attr->attribs, PA4_TYPE_NEXTHOP)) {
/* XXX */
/* This is not sufficient because there could be both IPv4 and IPv6 info */
	trace (TR_ERROR, peer->trace, "attribute: nexthop missing\n");
	bgp_send_notify_byte (peer, BGP_ERR_UPDATE, BGP_ERRUPD_MISSING, 
			      PA4_TYPE_NEXTHOP);
	return (-1);
	
    }
    else {
	/* XXX should check nexthop well */

#ifdef HAVE_IPV6
	if (attr->nexthop->prefix->family == AF_INET6 && attr->link_local &&
	        peer && peer->gateway && peer->gateway->interface &&
	       (BIT_TEST (peer->gateway->interface->flags, IFF_POINTOPOINT) &&
                peer->gateway->interface->primary6 &&
                peer->gateway->interface->primary6->prefix->bitlen == 128)) {
	    trace (TR_PACKET, peer->trace, 
		   "attribute: dropping link_local of unshared link: %a\n",
		   attr->link_local->prefix);
	    deref_nexthop (attr->link_local);
	    attr->link_local = NULL;
	}
#endif /* HAVE_IPV6 */
    }

    if (!BGP4_BIT_TEST (attr->attribs, PA4_TYPE_ASPATH)) {
	trace (TR_ERROR, peer->trace, "attribute: aspath missing\n");
	bgp_send_notify_byte (peer, BGP_ERR_UPDATE, BGP_ERRUPD_MISSING, 
			      PA4_TYPE_ASPATH);
	return (-1);
    }
    else {
        if (as != peer->gateway->AS /* eBGP */ &&
	    !BGPSIM_TRANSPARENT && /* bgpsim doesn't get thru here, though */
	    !BIT_TEST (peer->options, BGP_TRANSPARENT_AS)) {
	    if ((peer_as = bgp_get_home_AS (attr->aspath)) <= 0) {
		trace (TR_ERROR, peer->trace,
		       "attribute: no home for eBGP\n");
	        bgp_send_notification (peer, BGP_ERR_UPDATE, 
				      BGP_ERRUPD_ATTRLIST);
		return (-1);
	    }
	    else if (peer->gateway->AS != peer_as) {
		trace (TR_ERROR, peer->trace,
		       "attribute: strange home in aspath %s as %d\n",
		       aspath_toa (attr->aspath), peer->gateway->AS);
	        bgp_send_notification (peer, BGP_ERR_UPDATE, 
				      BGP_ERRUPD_ATTRLIST);
		return (-1);
	    }
	}
	if (as > 0 && bgp_check_aspath_loop (attr->aspath, as)) {
	    trace (TR_WARN, peer->trace,
		   "attribute: my own as %d in aspath %s\n",
		   as, aspath_toa (attr->aspath));
	    /* just discard */
	    return (-1);
	}
    }

    if (!BGP4_BIT_TEST (attr->attribs, PA4_TYPE_ORIGIN)) {
	trace (TR_ERROR, peer->trace, "attribute: origin missing\n");
	bgp_send_notify_byte (peer, BGP_ERR_UPDATE, BGP_ERRUPD_MISSING, 
			      PA4_TYPE_ORIGIN);
	return (-1);
    }

    if (BIT_TEST (peer->options, BGP_INTERNAL) &&
	    !BGP4_BIT_TEST (attr->attribs, PA4_TYPE_LOCALPREF)) {
	trace (TR_ERROR, peer->trace, "attribute: localpref missing\n");
	bgp_send_notify_byte (peer, BGP_ERR_UPDATE, BGP_ERRUPD_MISSING, 
			      PA4_TYPE_LOCALPREF);
	return (-1);
    }

    if (!BIT_TEST (peer->options, BGP_INTERNAL) && (
	    BGP4_BIT_TEST (attr->attribs, PA4_TYPE_ORIGINATOR_ID) ||
	    BGP4_BIT_TEST (attr->attribs, PA4_TYPE_CLUSTER_LIST))) {
	char *str;
	str = BGP4_BIT_TEST (attr->attribs, PA4_TYPE_ORIGINATOR_ID)?
		bgptype2string (PA4_TYPE_ORIGINATOR_ID):
		bgptype2string (PA4_TYPE_CLUSTER_LIST);
        trace (TR_ERROR, peer->trace,
               "strange attribute %s from eBGP\n", str);
    }

    if (/* BIT_TEST (peer->options, BGP_INTERNAL) && */
	    BGP4_BIT_TEST (attr->attribs, PA4_TYPE_ORIGINATOR_ID)) {
	u_long my_id;
#ifndef BGP_MULTI
    	my_id = BGP->my_id;
#else /* BGP_MULTI */
    	my_id = BGP->current_bgp->this_id;
#endif /* BGP_MULTI */
	if (my_id == prefix_tolong (attr->originator)) {
            char atext[64];
	    /* just ignore. don't need to send a notification */
            trace (TR_ERROR, peer->trace,
                   "originator id %a is the same as mine %s\n",
		    attr->originator,
                   inet_ntop (AF_INET, &my_id, atext, sizeof (atext)));
	    return (-1);
	}
    }

    if (/* BIT_TEST (peer->options, BGP_INTERNAL) && */
	    BGP4_BIT_TEST (attr->attribs, PA4_TYPE_CLUSTER_LIST)) {
	u_long my_cluster_id;
	DATA_PTR ptr;
#ifndef BGP_MULTI
    	my_cluster_id = BGP->cluster_id;
#else /* BGP_MULTI */
    	my_cluster_id = BGP->current_bgp->cluster_id;
#endif /* BGP_MULTI */
	assert (attr->cluster_list);
        LL_Iterate (attr->cluster_list, ptr) {
            char atext[64];
	    u_long id = (u_long) ptr;
            if (id == my_cluster_id) {
                trace (TR_ERROR, peer->trace,
                    "Duplicate my cluster id %s in %s",
                    inet_ntop (AF_INET, &id, atext, sizeof (atext)),
                     cluster_list_toa (attr->cluster_list));
	        /* just ignore. don't need to send a notification */
                return (-1);
	    }
        }
    }
    return (1);
}


int       
check_bgp_networks (prefix_t *prefix)
{
    prefix_t *network;

    assert (BGP->ll_networks);
    LL_Iterate (BGP->ll_networks, network) {
        if (a_include_b (network, prefix))
            return (1);
    }
    return (0);
}
