/*
 LiquidClient (C)QPL Daniel M. Duley 2001-2002. All rights reserved.
 */


#include "liquidclient.h"
#include <qabstractlayout.h>
#include <qlayout.h>
#include <qtoolbutton.h>
#include <qlabel.h>
#include <qdrawutil.h>
#include <qbitmap.h>
#include <kdrawutil.h>
#include <klocale.h>
#include <qdatetime.h>
#include <qimage.h>
#include <qtimer.h>
#include "workspace.h"
#include "options.h"

#include "embeddata.h"
#include "bitmapdata.h"

using namespace KWinInternal;

bool delayButtonPainting;

static FrameSet *leftFrameSet = 0;
static FrameSet *rightFrameSet = 0;
static FrameSet *midFrameSet = 0;

static QPixmap *active_titlebar_left = 0;
static QPixmap *active_titlebar_right = 0;
static QPixmap *active_titlebar_mid = 0;
static QPixmap *inactive_titlebar_left = 0;
static QPixmap *inactive_titlebar_right = 0;
static QPixmap *inactive_titlebar_mid = 0;


static QPixmap* frame_pix = 0;
static QPixmap* dis_frame_pix = 0;

static bool pixmaps_created = false;

static QImage* adjustHSVImage(QImage &img, const QColor &c, QColor bgColor,
                              bool blend)
{
    if(img.depth() != 32)
        img = img.convertDepth(32);
    QImage *dest = new QImage(img.width(), img.height(), 32);
    dest->setAlphaBuffer(true);
    unsigned int *data = (unsigned int *)img.bits();
    unsigned int *destData = (unsigned int*)dest->bits();
    int total = img.width()*img.height();
    int current;
    int delta;
    int destR, destG, destB, alpha;
    int srcR = c.red()+20;
    int srcG = c.green()+20;
    int srcB = c.blue()+20;
    float srcPercent, destPercent;
    for(current=0; current < total; ++current){
        alpha = qAlpha(data[current]);
        delta = 255-qRed(data[current]);
        destR = srcR-delta;
        destG = srcG-delta;
        destB = srcB-delta;
        if(destR < 0) destR = 0;
        if(destG < 0) destG = 0;
        if(destB < 0) destB = 0;
        if(destR > 255) destR = 255;
        if(destG > 255) destG = 255;
        if(destB > 255) destB = 255;

        if(blend && alpha != 255 && alpha !=0){
            srcPercent = ((float)alpha)/255.0;
            destPercent = 1.0-srcPercent;
            destR = (int)((srcPercent*destR) + (destPercent*bgColor.red()));
            destG = (int)((srcPercent*destG) + (destPercent*bgColor.green()));
            destB = (int)((srcPercent*destB) + (destPercent*bgColor.blue()));
            alpha = 255;
        }
        destData[current] = qRgba(destR, destG, destB, alpha);
    }
    return(dest);
}

static void create_pixmaps()
{
    if(pixmaps_created)
        return;
    pixmaps_created = true;
    QColor aColor = options->color(Options::TitleBar, true);
    QColor iColor = options->color(Options::TitleBar, false);
    QColor btnColor = options->color(Options::ButtonBg, true);
    QColor ibtnColor = options->color(Options::ButtonBg, false);

    leftFrameSet = new FrameSet("wm_left",  btnColor, ibtnColor,
                                aColor, iColor);
    rightFrameSet = new FrameSet("wm_right",  btnColor, ibtnColor,
                                 aColor, iColor);
    midFrameSet = new FrameSet("wm_mid",  btnColor, ibtnColor,
                               aColor, iColor);

    QImage img, *tmpImg;

    QColor tbColor(aColor.light(115));
    QColor tbiColor(iColor.light(115));

    img = uic_findImage_KWinLiquidDefault("titlebar_left");
    img.detach();
    tmpImg = adjustHSVImage(img, tbColor, aColor, true);
    active_titlebar_left = new QPixmap;
    active_titlebar_left->convertFromImage(*tmpImg);
    delete tmpImg;
    tmpImg = adjustHSVImage(img, tbiColor, iColor, true);
    inactive_titlebar_left = new QPixmap;
    inactive_titlebar_left->convertFromImage(*tmpImg);
    delete tmpImg;

    img = uic_findImage_KWinLiquidDefault("titlebar_mid");
    img.detach();
    tmpImg = adjustHSVImage(img, tbColor, aColor, true);
    active_titlebar_mid = new QPixmap;
    active_titlebar_mid->convertFromImage(*tmpImg);
    delete tmpImg;
    tmpImg = adjustHSVImage(img, tbiColor, iColor, true);
    inactive_titlebar_mid = new QPixmap;
    inactive_titlebar_mid->convertFromImage(*tmpImg);
    delete tmpImg;

    img = uic_findImage_KWinLiquidDefault("titlebar_right");
    img.detach();
    tmpImg = adjustHSVImage(img, tbColor, aColor, true);
    active_titlebar_right = new QPixmap;
    active_titlebar_right->convertFromImage(*tmpImg);
    delete tmpImg;
    tmpImg = adjustHSVImage(img, tbiColor, iColor, true);
    inactive_titlebar_right = new QPixmap;
    inactive_titlebar_right->convertFromImage(*tmpImg);
    delete tmpImg;


    QPainter painter;
    int i;
    frame_pix = new QPixmap(32, 24);
    frame_pix->fill(aColor.rgb());
    painter.begin(frame_pix);
    for(i=0; i < 24; i+=4){
        painter.setPen(aColor.dark(105));
        painter.drawLine(0, i, 32, i);
        painter.setPen(aColor.dark(103));
        painter.drawLine(0, i+1, 32, i+1);
    }
    painter.end();
    dis_frame_pix = new QPixmap(32, 24);
    dis_frame_pix->fill(iColor.rgb());
    painter.begin(dis_frame_pix);
    for(i=0; i < 24; i+=4){
        painter.setPen(iColor.dark(105));
        painter.drawLine(0, i, 32, i);
        painter.setPen(iColor.dark(103));
        painter.drawLine(0, i+1, 32, i+1);
    }
    painter.end();

    pixmaps_created = true;
}

static void delete_pixmaps()
{

    delete leftFrameSet;
    leftFrameSet = 0;
    delete rightFrameSet;
    rightFrameSet = 0;
    delete midFrameSet;
    midFrameSet = 0;

    delete active_titlebar_left;
    active_titlebar_left = 0;
    delete active_titlebar_right;
    active_titlebar_right = 0;
    delete active_titlebar_mid;
    active_titlebar_mid = 0;

    delete inactive_titlebar_left;
    inactive_titlebar_left = 0;
    delete inactive_titlebar_right;
    inactive_titlebar_right = 0;
    delete inactive_titlebar_mid;
    inactive_titlebar_mid = 0;

    delete frame_pix;
    frame_pix = 0;
    delete dis_frame_pix;
    dis_frame_pix = 0;

    pixmaps_created = false;
}


FrameSet::FrameSet(const QString &embed_name, const QColor &btnColor,
                   const QColor &inactiveBtnColor,
                   const QColor &activeBgColor,
                   const QColor &inactiveBgColor)
{
    QImage img;
    QImage *tmpImg;

    img = uic_findImage_KWinLiquidDefault(embed_name);
    if(img.isNull()){
        // debug - shouldn't happen
        qWarning("Couldn't find embedded image: %s!", embed_name.latin1());
        return;
    }
    img.detach();

    // TODO - make a loop
    // do active buttons
    tmpImg = adjustHSVImage(img, btnColor, activeBgColor, true);
    activeFrames[2] = new QPixmap;
    activeFrames[2]->convertFromImage(*tmpImg);
    delete tmpImg;

    tmpImg = adjustHSVImage(img, btnColor.light(105), activeBgColor, true);
    activeFrames[3] = new QPixmap;
    activeFrames[3]->convertFromImage(*tmpImg);
    delete tmpImg;

    tmpImg = adjustHSVImage(img, btnColor.light(110), activeBgColor, true);
    activeFrames[4] = new QPixmap;
    activeFrames[4]->convertFromImage(*tmpImg);
    delete tmpImg;

    tmpImg = adjustHSVImage(img, btnColor.dark(105), activeBgColor, true);
    activeFrames[1] = new QPixmap;
    activeFrames[1]->convertFromImage(*tmpImg);
    delete tmpImg;

    tmpImg = adjustHSVImage(img, btnColor.dark(110), activeBgColor, true);
    activeFrames[0] = new QPixmap;
    activeFrames[0]->convertFromImage(*tmpImg);
    delete tmpImg;


    // do inactive buttons
    tmpImg = adjustHSVImage(img, inactiveBtnColor, inactiveBgColor,
                            true);
    inactiveFrames[2] = new QPixmap;
    inactiveFrames[2]->convertFromImage(*tmpImg);
    delete tmpImg;

    tmpImg = adjustHSVImage(img, inactiveBtnColor.light(105),
                            inactiveBgColor, true);
    inactiveFrames[3] = new QPixmap;
    inactiveFrames[3]->convertFromImage(*tmpImg);
    delete tmpImg;

    tmpImg = adjustHSVImage(img, inactiveBtnColor.light(110),
                            inactiveBgColor, true);
    inactiveFrames[4] = new QPixmap;
    inactiveFrames[4]->convertFromImage(*tmpImg);
    delete tmpImg;

    tmpImg = adjustHSVImage(img, inactiveBtnColor.dark(105),
                            inactiveBgColor, true);
    inactiveFrames[1] = new QPixmap;
    inactiveFrames[1]->convertFromImage(*tmpImg);
    delete tmpImg;

    tmpImg = adjustHSVImage(img, inactiveBtnColor.dark(110),
                            inactiveBgColor, true);
    inactiveFrames[0] = new QPixmap;
    inactiveFrames[0]->convertFromImage(*tmpImg);
    delete tmpImg;

}

FrameSet::~FrameSet()
{
    int i;
    for(i=0; i < 5; ++i){
        delete activeFrames[i];
        delete inactiveFrames[i];
    }
}

FrameToolButton::FrameToolButton(FrameSet *frameSet, bool isActive,
                                 const unsigned char *bitmap,
                                 const unsigned char *sunken_bitmap,
                                 QWidget *parent,
                                 const char *name, const QString &tip,
                                 bool sticky, bool max)
    : KWinToolButton(parent, name, tip)
{
    setBackgroundMode(NoBackground);

    frames = frameSet;
    active = isActive;
    mainBmp = new QBitmap(12, 12, bitmap, true);
    mainBmp->setMask(*mainBmp);
    if(sunken_bitmap){
        pressedBmp = new QBitmap(12, 12, sunken_bitmap, true);
        pressedBmp->setMask(*pressedBmp);
    }
    else
        pressedBmp = NULL;
    currentFrame = 2;
    ascending = true;
    buffer = new QPixmap(18, 18);
    isSticky = sticky;
    isMax = max;
}

FrameToolButton::~FrameToolButton()
{
    delete mainBmp;
    delete buffer;
    if(pressedBmp)
        delete pressedBmp;
}

void FrameToolButton::reset(FrameSet *frameSet, bool act)
{
    frames = frameSet;
    active = act;
    repaint(false);
}

void FrameToolButton::setActiveButton(bool on)
{
    active = on;
    repaint(false);
}

void FrameToolButton::slotFrameTimer()
{
    if(ascending){
        ++currentFrame;
        if(currentFrame == 5){
            currentFrame = 3;
            ascending = false;
        }
    }
    else{
        --currentFrame;
        if(currentFrame == -1){
            currentFrame = 1;
            ascending = true;
        }
    }
    repaint(false);
}

void FrameToolButton::paintEvent(QPaintEvent* /*ev*/)
{
    if(delayButtonPainting)
        return;
    QPixmap *bgPix = active ? frame_pix : dis_frame_pix;
    if(!bgPix)
        return;
    if(timer.isActive())
        timer.stop();
    if(buffer->size() != size())
        buffer->resize(width(), height());
    QPainter p(buffer);
    int w = width();
    int h = height();
    bool sunken = isDown();
    bool on = false;
    bool hover = hasMouse();

    if(isSticky && ((KWinInternal::Client *)parent())->isSticky())
        on = true;
    if(isMax && ((KWinInternal::Client *)parent())->isMaximized())
        on = true;

    p.drawTiledPixmap(0, 0, width(), height(), *bgPix, x(), y());
    if(sunken)
        p.drawPixmap(0, 0, *frames->sunkenFrame(active));
    else if(hover)
        p.drawPixmap(0, 0, *frames->frame(active, currentFrame));
    else
        p.drawPixmap(0, 0, *frames->normalFrame(active));

    p.setPen(qGray(options->color(Options::ButtonBg, active).rgb()) > 130 ?
             Qt::black : Qt::white);
    if(sunken)
        p.drawPixmap(4, 4, on ? pressedBmp ? *pressedBmp : *mainBmp :
                     *mainBmp);
    else
        p.drawPixmap(3, 3, on ? pressedBmp ? *pressedBmp : *mainBmp :
                     *mainBmp);
    p.end();
    bitBlt(this, 0, 0, buffer, 0, 0, w, h, Qt::CopyROP, true);

    if(hover && !sunken)
        timer.singleShot(150, this, SLOT(slotFrameTimer()));
    else{
        currentFrame = 2;
        ascending = true;
    }
}

void MenuToolButton::paintEvent(QPaintEvent* /*ev*/)
{
    if(delayButtonPainting)
        return;

    bool active = ((KWinInternal::Client *)parent())->isActive();
    QPixmap *bgPix = active ? frame_pix : dis_frame_pix;
    if(!bgPix || !menuPix)
        return;

    bool on = isOn() || isDown();
    int dx = (width()-menuPix->width())/2;
    int dy = (height()-menuPix->height())/2;
    if(on){
        ++dx;
        ++dy;
    }

    QPainter p(this);
    p.drawTiledPixmap(0, 0, width(), height(), *bgPix, x(), y());
    p.drawPixmap(dx, dy, *menuPix);
    if(hasMouse()){
        QColor c = options->color(Options::TitleBar, active);
        int x2 = width()-1;
        int y2 = height()-1;
        p.setPen(on ? c.dark(120) : c.light(130));
        p.drawLine(0, 0, x2, 0);
        p.drawLine(0, 0, 0, y2);
        p.setPen(on ? c.light(120) : c.dark(130));
        p.drawLine(x2, 0, x2, y2);
        p.drawLine(0, y2, x2, y2);
    }
}


void LiquidClient::slotReset()
{
    *menuTmpPix = miniIcon();
    // do menu button
    bool act = isActive();

    button[0]->reset(leftFrameSet, act);
    int i = 1;
    if (isMinimizable())
	    button[i++]->reset(midFrameSet, act);
    if (isMaximizable())
	    button[i++]->reset(midFrameSet, act);
    button[i]->reset(rightFrameSet, act);

    delayButtonPainting = false;

    setFont(options->font(true));
}


LiquidClient::LiquidClient( Workspace *ws, WId w, QWidget *parent, const char *name )
: Client(ws, w, parent, name, WResizeNoErase| WRepaintNoErase)
{
    delayButtonPainting = false;
    setBackgroundMode(QWidget::NoBackground);
    setFont(options->font(isActive()));
    connect(options, SIGNAL(resetClients()), this, SLOT(slotReset()));

    QGridLayout* g = new QGridLayout(this, 0, 0, 3, 2);
    g->setRowStretch(1, 10);
    g->addWidget(windowWrapper(), 1, 1);
    g->addItem(new QSpacerItem(0, 0, QSizePolicy::Fixed, QSizePolicy::Expanding));

    g->addColSpacing(0, 1);
    g->addColSpacing(2, 1);
    g->addRowSpacing(2, 1);

    bool act = isActive();

    menuButton = new MenuToolButton(this, 0, i18n("Menu"));
    menuButton->setFixedSize(18, 18);
    menuButton->setAutoRaise(true);
    
    button[0] = new FrameToolButton(leftFrameSet, act,
                                    sticky_off_bits, sticky_on_bits,
                                    this, 0, 0, true, false);
    button[0]->setTipText(isSticky() ? i18n("Un-Sticky") :
                          i18n("Sticky"));
    connect(button[0], SIGNAL(clicked()), this, (SLOT(toggleSticky())));
    
    bn = 1;
    if (isMinimizable()) {
	    button[bn] = new FrameToolButton(midFrameSet, act,
        	                            minimize_bits, NULL,
                	                    this, 0, i18n("Minimize"));
    	    connect(button[bn], SIGNAL(clicked()), this, (SLOT(iconify())));
	    bn++;
    }

    if (isMaximizable()) {
	    button[bn] = new FrameThreeButtonButton(midFrameSet, act,
			    maximize_bits, minmax_bits,
			    this, 0, 0, false, true);
	    button[bn]->setTipText(isMaximized() ? i18n("Restore") :
        	                  i18n("Maximize"));
	    connect(button[bn], SIGNAL(clicked(int)), this, (SLOT(maxButtonClicked(int))));
	    bn++;
    }
    
    button[bn] = new FrameToolButton(rightFrameSet, act,
                                    close_bits, NULL,
                                    this, 0, i18n("Close"));
    connect(button[bn], SIGNAL(clicked()), this, (SLOT(closeWindow())));

    QHBoxLayout* hb = new QHBoxLayout(NULL, 0, 0);
    g->addLayout(hb, 0, 1);
    hb->addWidget(menuButton);
    int fh = fontMetrics().lineSpacing();

    titlebar = new QSpacerItem(10, fh, QSizePolicy::Expanding,
			       QSizePolicy::Minimum);
    hb->addItem(titlebar);

    for(int i = 0; i <= bn; i++) {
	button[i]->setFocusPolicy(NoFocus);
        button[i]->setFixedSize(18, 18);
        button[i]->setAutoRaise(true);
	hb->addWidget(button[i]);
    }
    menuTmpPix = new QPixmap(miniIcon());
    menuButton->setPixmap(menuTmpPix);

    connect(menuButton, SIGNAL(pressed()), this, SLOT(menuButtonPressed()));

    tbBuffer = new QPixmap;
}

LiquidClient::~LiquidClient()
{
    delete menuTmpPix;
    delete tbBuffer;
}

void LiquidClient::activeChange(bool act)
{
    menuButton->repaint(false);
    for (int i = 0; i <= bn; i++) 
	    button[i]->setActiveButton(act);
    Client::activeChange(act);
}

void LiquidClient::resizeEvent( QResizeEvent* e)
{
    Client::resizeEvent(e);
    QRegion rr = rect();
    QRect t = titlebar->geometry();

    if(isVisibleToTLW() && !testWFlags(WStaticContents)){
	QRect cr(2, 2, width()-4, height()-4);
	erase(QRegion(cr).subtract(t));
    }
}

void LiquidClient::captionChange( const QString& )
{
    repaint(titlebar->geometry(), false);
}


void LiquidClient::maximizeChange(bool m)
{
    // FIXME
    if (isMaximizable()) {
	   int i = 2;
	   if (!isMinimizable()) 
		   i = 1;
		   
	   button[i]->setTipText(m ? i18n("Restore") : i18n("Maximize"));
	   button[i]->repaint(false);
    }
}


void LiquidClient::stickyChange(bool s)
{
    button[0]->setTipText(s ? i18n("Un-Sticky") : i18n("Sticky"));
    button[0]->repaint(false);
}

void LiquidClient::paintEvent( QPaintEvent* )
{
    QPainter p(this);
    QRect t = titlebar->geometry();

    int x2 = width()-1;
    int y2 = height()-1;
    p.setPen(options->color(Options::TitleBar, isActive()).light(120));
    p.drawLine(0, 0, x2, 0);
    p.drawLine(0, 0, 0, y2);
    p.setPen(options->color(Options::TitleBar, isActive()).dark(130));
    p.drawLine(0, y2, x2, y2);
    p.drawLine(x2, 0, x2, y2);

    QPixmap *pix = isActive() ? frame_pix : dis_frame_pix;
    p.drawTiledPixmap(1, 1, 2, height()-2, *pix, 1, 1);
    p.drawTiledPixmap(x2-2, 1, 2, height()-2,
                       *pix, 1, 1);
    p.drawTiledPixmap(3, y2-2, width()-6, 2, *pix, 4, y2-2);

    p.setPen(options->color(Options::TitleBar, isActive()).dark(130));
    p.drawLine(3, t.height()+4, 3, y2-3);
    p.drawLine(3, t.height()+4, x2-3, t.height()+4);
    p.setPen(options->color(Options::TitleBar, isActive()).light(120));
    p.drawLine(x2-3, t.height()+4, x2-3, y2-3);
    p.drawLine(3, y2-3, x2-3, y2-3);

    t.setLeft(t.left()+4);
    t.setRight(t.right()-2);

    if(tbBuffer->width() != width())
        tbBuffer->resize(width(), 21);

    QPainter tmpPainter;
    tmpPainter.begin(tbBuffer);
    tmpPainter.setFont(options->font(isActive()));
    tmpPainter.setPen(options->color(Options::TitleBar,
                                     isActive()).dark(120));
    int textLen = tmpPainter.fontMetrics().width(caption());
    bool clippedText = false;

    tmpPainter.drawTiledPixmap(0, 0, tbBuffer->width(),
                               tbBuffer->height(), isActive() ?
                               *frame_pix : *dis_frame_pix, 1, 1);
    t.moveTopLeft(QPoint(t.x()-3, t.y()-1));
    if(t.width() > 18){
        int fillLen = textLen + 16;
        if(fillLen > t.width()){
            fillLen = t.width();
            clippedText = true;
        }
        int fx = (t.x()+(t.width()-fillLen)/2);
        tmpPainter.drawPixmap(fx, t.y(), isActive() ?
                              *active_titlebar_left :
                              *inactive_titlebar_left);
        tmpPainter.drawTiledPixmap(fx+8, t.y(), fillLen-16, 18,
                                   isActive() ? *active_titlebar_mid :
                                   *inactive_titlebar_mid);
        tmpPainter.drawPixmap(fx+fillLen-8, t.y(), isActive() ?
                              *active_titlebar_right :
                              *inactive_titlebar_right);
    }
    if(!clippedText){
        tmpPainter.drawText(t.x()+1, t.y()+1, t.width(), t.height(),
                            AlignCenter|AlignVCenter|SingleLine,
                            caption());
        tmpPainter.setPen(options->color(Options::Font, isActive()));
        tmpPainter.drawText(t.x(), t.y(), t.width(), t.height(),
                            AlignCenter|AlignVCenter|SingleLine,
                            caption());
    }
    else{
        tmpPainter.drawText(t.x()+8, t.y()+1, t.width()-16, t.height(),
                            AlignLeft|AlignVCenter|SingleLine, caption());
        tmpPainter.setPen(options->color(Options::Font, isActive()));
        tmpPainter.drawText(t.x()+7, t.y(), t.width()-16, t.height(),
                            AlignLeft|AlignVCenter|SingleLine, caption());
    }
    tmpPainter.end();
    p.drawPixmap(3, 1, *tbBuffer, 0, 0, tbBuffer->width()-6,
                tbBuffer->height());

}


void LiquidClient::mouseDoubleClickEvent( QMouseEvent * e )
{
    if(titlebar->geometry().contains(e->pos()))
	workspace()->performWindowOperation(this, options->operationTitlebarDblClick());
}


void LiquidClient::init()
{
    Client::init();
}

void LiquidClient::iconChange()
{
    *menuTmpPix = miniIcon();
    menuButton->setPixmap(menuTmpPix);
}


/*!
  Indicates that the menu button has been clicked. One press shows
  the window operation menu, a double click closes the window.
 */
void LiquidClient::menuButtonPressed()
{
    static QTime* t = 0;
    static LiquidClient* tc = 0;
    if(!t)
	t = new QTime;

    bool double_clicked = ( tc == this && t->elapsed() <= QApplication::doubleClickInterval());
    tc = this;
    t->start();

    if( !double_clicked ){
        QPoint menupoint(menuButton->rect().bottomLeft().x()-1,
                           menuButton->rect().bottomLeft().y()+2);
        workspace()->showWindowMenu(menuButton->mapToGlobal(menupoint), this);
        menuButton->setDown(false);
    }else
	closeWindow();
}


void LiquidClient::maxButtonClicked( int button )
{
    switch (button){
    case MidButton:
	maximize(MaximizeVertical);
	break;
    case RightButton:
	maximize(MaximizeHorizontal);
	break;
    default: //LeftButton:
	maximize(MaximizeFull);
	break;
    }
}


LiquidToolClient::LiquidToolClient( Workspace *ws, WId w, QWidget *parent, const char *name )
    : Client( ws, w, parent, name )
{
    connect(options, SIGNAL(resetClients()), this, SLOT(slotReset()));
    setFont(options->font(isActive(), true));

    QGridLayout* g = new QGridLayout(this, 0, 0, 2);
    g->setRowStretch(1, 10);
    g->addWidget(windowWrapper(), 1, 1);
    g->addItem(new QSpacerItem(0, 0, QSizePolicy::Fixed, QSizePolicy::Expanding));

    g->addColSpacing(0, 1);
    g->addColSpacing(2, 1);
    g->addRowSpacing(2, 1);

    closeBtn = new FrameToolButton(rightFrameSet, isActive(),
                                   close_bits, NULL,
                                   this, 0, i18n("Close"));
    connect(closeBtn, SIGNAL(clicked()), this, (SLOT( closeWindow())));
    closeBtn->setFixedSize(18, 18);
    slotReset();

    QHBoxLayout* hb = new QHBoxLayout;
    g->addLayout(hb, 0, 1);

    int fh = fontMetrics().lineSpacing()+2;

    titlebar = new QSpacerItem(10, fh, QSizePolicy::Expanding,
			       QSizePolicy::Minimum);
    hb->addItem(titlebar);
    hb->addWidget(closeBtn);
}

LiquidToolClient::~LiquidToolClient()
{
}

void LiquidToolClient::resizeEvent(QResizeEvent* e)
{
    Client::resizeEvent( e );
}

void LiquidToolClient::paintEvent(QPaintEvent*)
{
    QPainter p(this);
    QRect t = titlebar->geometry();
    QRect r = rect();
    qDrawWinPanel(&p, r, colorGroup());
    r.setTop(t.bottom()+1);
    qDrawWinPanel(&p, r, colorGroup());
    p.fillRect(QRect( QPoint(t.topLeft() ), QPoint( width() - t.left(), t.bottom())),
               options->color(Options::TitleBar, isActive()));

    p.setPen(options->color(Options::TitleBar, isActive()).light());
    t.setLeft(t.left() + 4);
    t.setRight(t.right() - 2);
    p.setPen(options->color(Options::Font, isActive()));
    p.setFont(options->font(isActive(), true));
    p.drawText(t, AlignLeft|AlignVCenter|SingleLine, caption());
}


void LiquidToolClient::mouseDoubleClickEvent(QMouseEvent * e)
{
    if(titlebar->geometry().contains(e->pos()))
        workspace()->performWindowOperation(this, options->operationTitlebarDblClick());
    workspace()->requestFocus(this);
}

void LiquidToolClient::init()
{
}

void LiquidToolClient::captionChange(const QString&)
{
    repaint(titlebar->geometry(), FALSE);
}

void LiquidToolClient::activeChange( bool on )
{
    closeBtn->setActiveButton(on);
}


void LiquidToolClient::slotReset()
{
    closeBtn->reset(rightFrameSet, isActive());
    delayButtonPainting = false;
    setFont(options->font(true));
}

extern "C"
{
  Client * allocate(Workspace * workSpace, WId winId, int /*tool*/ )
  {
      // (tool)
      //  return new LiquidToolClient(workSpace, winId);
      return new LiquidClient(workSpace, winId);
  }
  void init()
  {
      qInitImages_KWinLiquidDefault();
      create_pixmaps();
  }
  void reset()
  {
      delayButtonPainting = true;
      delete_pixmaps();
      create_pixmaps();
      // Ensure change in tooltip state gets applied
      Workspace::self()->slotResetAllClientsDelayed();
  }
  void deinit()
  {
      delete_pixmaps();
      qCleanupImages_KWinLiquidDefault();
  }
}


#include "liquidclient.moc"
