/*
 * Copyright (c) 2003-2004 Jason L. Wright (jason@thought.net)
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
 * ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

/*
 * A morse code player based on ARRL timing.
 *
 * References:
 * "A Standard for Morse Timing Using the Farnsworth Technique", QEX,
 *  April 1990 (http://www.arrl.org/files/infoserv/tech/code-std.txt)
 */

#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/audioio.h>
#include <string.h>
#include <fcntl.h>
#include <err.h>
#include <unistd.h>
#include <math.h>
#include <stdlib.h>
#include <stdio.h>
#include <sysexits.h>
#include <errno.h>

struct a_sound {
	u_int len;
	int8_t *buf;
} dit, dah, inSym, inChar, inWord;

double overallwpm = 5.0;
double charwpm = 18.0;
double attackpercent = 10.0;

struct play_list {
	struct a_sound *pl_snd;
	struct play_list *pl_nxt;
};

int play_codes(int, struct play_list *, int);
int build_dit(struct a_sound *, audio_info_t *, double);
int build_dah(struct a_sound *, audio_info_t *, double);
int build_inSym(struct a_sound *, audio_info_t *);
int build_inChar(struct a_sound *, audio_info_t *);
int build_inWord(struct a_sound *, audio_info_t *);
void play_morse(int, const char *);
void play_message(int, char *, size_t);
int do_input(int, FILE *);
int open_audio(char *, audio_info_t *);
int get_double(const char *, double *);
int build_parcel(audio_info_t *, struct a_sound *, double, double, u_int);
int attack_len(audio_info_t *);

int
play_codes(int fd, struct play_list *pl, int drain) {
	while (pl != NULL) {
		if (write(fd, pl->pl_snd->buf, pl->pl_snd->len) == -1) {
			warn("write");
			return (-1);
		}
		pl = pl->pl_nxt;
	}
	if (drain && ioctl(fd, AUDIO_DRAIN, NULL) == -1) {
		warn("drain");
		return (-1);
	}
	return (0);
}

int
build_parcel(audio_info_t *ainfo, struct a_sound *snd, double hz,
    double amp, u_int srate)
{
	u_int i, attack1, attack2;
	double m;

	snd->buf = (int8_t *)malloc(snd->len);
	if (snd->buf == NULL)
		return (-1);

	attack1 = attack_len(ainfo);
	attack2 = snd->len - attack1;

	for (i = 0; i < snd->len; i++) {
		double d;

		if (attack1 >= attack2)
			m = 1.0;
		else if (i < attack1)
			m = (float)i / (float)attack1;
		else if (i > attack2)
			m = ((float)snd->len - 1.0 - (float)i)/((float)attack1);
		else
			m = 1.0;

		d = rint(m * amp * 127.0 * sin(((double)i / (double)srate) *
		    (2 * M_PI * hz)));
		snd->buf[i] = d;
	}
	return (0);
}

int
attack_len(audio_info_t *ainfo)
{
	float wpm, u;

	if (overallwpm >= 18.0)
		wpm = overallwpm;
	else
		wpm = 18.0;

	u = 1.2 / wpm;

	u *= (float)ainfo->play.sample_rate * attackpercent;
	return ((int)u);
}

int
build_dit(struct a_sound *snd, audio_info_t *ainfo, double hz)
{
	float u;

	u = 1.2 / charwpm;

	snd->len = rintf(u * (float)ainfo->play.sample_rate);
	if (build_parcel(ainfo, snd, hz, 1.0, ainfo->play.sample_rate)) {
		warn("dit");
		return (-1);
	}
	return (0);
}

int
build_dah(struct a_sound *snd, audio_info_t *ainfo, double hz)
{
	float u;

	u = 1.2 / charwpm;

	snd->len = rintf(3.0 * u * (float)ainfo->play.sample_rate);
	if (build_parcel(ainfo, snd, hz, 1.0, ainfo->play.sample_rate)) {
		warn("dah");
		return (-1);
	}
	return (0);
}

int
build_inSym(struct a_sound *snd, audio_info_t *ainfo) {
	float u;

	u = 1.2 / charwpm;

	snd->len = rintf(u * (float)ainfo->play.sample_rate);
	if (build_parcel(ainfo, snd, 1.0, 0.0, ainfo->play.sample_rate)) {
		warn("inSym");
		return (-1);
	}
	return (0);
}

int
build_inChar(struct a_sound *snd, audio_info_t *ainfo) {
	float srate, samplen;

	srate = (float)ainfo->play.sample_rate;

	if (overallwpm >= charwpm) {
		float u;

		u = 1.2 / overallwpm;
		samplen = 3.0 * u * srate;
	} else {
		float wpm, Ta, Tc;

		wpm = charwpm;
		Ta = ((60.0 * wpm) - (37.2 * overallwpm)) / (wpm * overallwpm);
		Tc = (3.0 * Ta) / 19.0;
		samplen = Tc * srate;
	}

	snd->len = rintf(samplen);
	if (build_parcel(ainfo, snd, 1.0, 0.0, ainfo->play.sample_rate)) {
		warn("inSym");
		return (-1);
	}
	return (0);
}

int
build_inWord(struct a_sound *snd, audio_info_t *ainfo) {
	float srate, samplen;

	srate = (float)ainfo->play.sample_rate;

	if (overallwpm >= charwpm) {
		float u;

		u = 1.2 / overallwpm;
		samplen = 7.0 * u * srate;
	} else {
		float wpm, Ta, Tw;

		wpm = charwpm;
		Ta = ((60.0 * wpm) - (37.2 * overallwpm)) / (wpm * overallwpm);
		Tw = (7.0 * Ta) / 19.0;
		samplen = Tw * srate;
	}

	snd->len = rintf(samplen);
	if (build_parcel(ainfo, snd, 1.0, 0.0, ainfo->play.sample_rate)) {
		warn("inWord");
		return (-1);
	}
	return (0);
}

void
play_morse(int fd, const char *ditdahs)
{
	struct play_list pl[2];
	const char *p;

	pl[1].pl_snd = &inSym;
	pl[1].pl_nxt = NULL;
	for (p = ditdahs; p[0] != '\0'; p++) {
		if (p[0] == '.')
			pl[0].pl_snd = &dit;
		else
			pl[0].pl_snd = &dah;

		if (p[1] == '\0') {
			pl[0].pl_nxt = NULL;
			play_codes(fd, pl, 0);
		} else {
			pl[0].pl_nxt = &pl[1];
			play_codes(fd, pl, 0);
		}
	}
}

const struct morse_chars {
	const char *sym;
	const char *code;
} all_chars[] = {
#define	ALLCHAR_OFFSET_A		0
	{ "A",	".-" },
	{ "B",	"-..." },
	{ "C",	"-.-." },
	{ "D",	"-.." },
	{ "E",	"." },
	{ "F",	"..-." },
	{ "G",	"--." },
	{ "H",	"...." },
	{ "I",	".." },
	{ "J",	".---" },
	{ "K",	"-.-", },
	{ "L",	".-.." },
	{ "M",	"--" },
	{ "N",	"-." },
	{ "O",	"---" },
	{ "P",	".--." },
	{ "Q",	"--.-" },
	{ "R",	".-." },
	{ "S",	"..." },
	{ "T",	"-" },
	{ "U",	"..-" },
	{ "V",	"...-" },
	{ "W",	".--" },
	{ "X",	"-..-" },
	{ "Y",	"-.--" },
	{ "Z",	"--.." },
	{ "0",	"-----" },
#define	ALLCHAR_OFFSET_0		('Z' - 'A' + 1)
	{ "1",	".----" },
	{ "2",	"..---" },
	{ "3",	"...--" },
	{ "4",	"....-" },
	{ "5",	"....." },
	{ "6",	"-...." },
	{ "7",	"--..." },
	{ "8",	"---.." },
	{ "9",	"----." },
	{ "*",	"...-.-" },		/* SK */
#define	ALLCHAR_OFFSET_SK		('Z' - 'A' + 1 + '9' - '0' + 1 + 0)
#define	ALLCHAR_OFFSET_VA		('Z' - 'A' + 1 + '9' - '0' + 1 + 0)
	{ ".",	".-.-.-" },		/* . */
#define	ALLCHAR_OFFSET_PERIOD		('Z' - 'A' + 1 + '9' - '0' + 1 + 1)
	{ "+",	".-.-." },		/* AR, + */
#define	ALLCHAR_OFFSET_AR		('Z' - 'A' + 1 + '9' - '0' + 1 + 2)
#define	ALLCHAR_OFFSET_PLUS		('Z' - 'A' + 1 + '9' - '0' + 1 + 2)
	{ "=",	"-...-" },		/* BT, = */
#define	ALLCHAR_OFFSET_BT		('Z' - 'A' + 1 + '9' - '0' + 1 + 3)
#define	ALLCHAR_OFFSET_EQUAL		('Z' - 'A' + 1 + '9' - '0' + 1 + 3)
	{ "/",	"-..-." },		/* / */
#define	ALLCHAR_OFFSET_SLASH		('Z' - 'A' + 1 + '9' - '0' + 1 + 4)
	{ "?",	"..--.." },		/* ? */
#define	ALLCHAR_OFFSET_QUESTION		('Z' - 'A' + 1 + '9' - '0' + 1 + 5)
	{ ",",	"--..--" },		/* , */
#define	ALLCHAR_OFFSET_COMMA		('Z' - 'A' + 1 + '9' - '0' + 1 + 6)
	{ "(",	"-.--." },		/* ( */
#define	ALLCHAR_OFFSET_LPAREN		('Z' - 'A' + 1 + '9' - '0' + 1 + 7)
#define	ALLCHAR_OFFSET_KN		('Z' - 'A' + 1 + '9' - '0' + 1 + 7)
	{ ")",	"-.--.-" },		/* ) */
#define	ALLCHAR_OFFSET_RPAREN		('Z' - 'A' + 1 + '9' - '0' + 1 + 8)
	{ "-",	"-....-" },		/* - */
#define	ALLCHAR_OFFSET_HYPHEN		('Z' - 'A' + 1 + '9' - '0' + 1 + 9)
	{ "\"",	".-..-." },		/* " */
#define	ALLCHAR_OFFSET_DQUOTE		('Z' - 'A' + 1 + '9' - '0' + 1 + 10)
	{ "_",	"..--.-" },		/* _ */
#define	ALLCHAR_OFFSET_UNDERSCORE	('Z' - 'A' + 1 + '9' - '0' + 1 + 11)
	{ "'",	".----." },		/* ' */
#define	ALLCHAR_OFFSET_SQUOTE		('Z' - 'A' + 1 + '9' - '0' + 1 + 12)
	{ ":",	"---..." },		/* : */
#define	ALLCHAR_OFFSET_COLON		('Z' - 'A' + 1 + '9' - '0' + 1 + 13)
	{ ";",	"-.-.-." },		/* ; */
#define	ALLCHAR_OFFSET_SEMICOLON	('Z' - 'A' + 1 + '9' - '0' + 1 + 14)
	{ "$",	"...-..-" },		/* $ */
#define	ALLCHAR_OFFSET_DOLLAR		('Z' - 'A' + 1 + '9' - '0' + 1 + 15)
};

void
play_message(int fd, char *s, size_t len)
{
	struct play_list charSpace, wordSpace;
	char c, *p;
	size_t i;
	int first = 1;

	charSpace.pl_snd = &inChar;
	charSpace.pl_nxt = NULL;
	wordSpace.pl_snd = &inWord;
	wordSpace.pl_nxt = NULL;

	for (i = 0, p = s; i < len; i++, p++) {
		c = *p;

		if (first) {
			first = 0;
		} else {
			if (c == ' ' || c == '\r' || c == '\n')
				play_codes(fd, &wordSpace, 0);
			else
				play_codes(fd, &charSpace, 0);
		}

		if (c >= 'A' && c <= 'Z')
			play_morse(fd,
			    all_chars[c - 'A' + ALLCHAR_OFFSET_A].code);
		else if (c >= 'a' && c <= 'z')
			play_morse(fd,
			    all_chars[c - 'a' + ALLCHAR_OFFSET_A].code);
		else if (c == '*')
			play_morse(fd, all_chars[ALLCHAR_OFFSET_SK].code);
		else if (c == '.')
			play_morse(fd, all_chars[ALLCHAR_OFFSET_PERIOD].code);
		else if (c == '+')
			play_morse(fd, all_chars[ALLCHAR_OFFSET_AR].code);
		else if (c == '=')
			play_morse(fd, all_chars[ALLCHAR_OFFSET_BT].code);
		else if (c == '/')
			play_morse(fd, all_chars[ALLCHAR_OFFSET_SLASH].code);
		else if (c == '?')
			play_morse(fd, all_chars[ALLCHAR_OFFSET_QUESTION].code);
		else if (c == ',')
			play_morse(fd, all_chars[ALLCHAR_OFFSET_COMMA].code);
		else if (c == '(')
			play_morse(fd, all_chars[ALLCHAR_OFFSET_LPAREN].code);
		else if (c == ')')
			play_morse(fd, all_chars[ALLCHAR_OFFSET_RPAREN].code);
		else if (c == '-')
			play_morse(fd, all_chars[ALLCHAR_OFFSET_HYPHEN].code);
		else if (c == '"')
			play_morse(fd, all_chars[ALLCHAR_OFFSET_DQUOTE].code);
		else if (c == '_')
			play_morse(fd, all_chars[ALLCHAR_OFFSET_UNDERSCORE].code);
		else if (c == '\'')
			play_morse(fd, all_chars[ALLCHAR_OFFSET_SQUOTE].code);
		else if (c == ':')
			play_morse(fd, all_chars[ALLCHAR_OFFSET_COLON].code);
		else if (c == ';')
			play_morse(fd, all_chars[ALLCHAR_OFFSET_SEMICOLON].code);
		else if (c == '$')
			play_morse(fd, all_chars[ALLCHAR_OFFSET_DOLLAR].code);
		else if (c >= '0' && c <= '9')
			play_morse(fd,
			    all_chars[c - '0' + ALLCHAR_OFFSET_0].code);
	}
	play_codes(fd, NULL, 1);
}

int
do_input(int afd, FILE *fp)
{
	size_t len;
	char *line;

	while ((line = fgetln(fp, &len)) != NULL) {
		play_message(afd, line, len);
	}
	if (feof(fp))
		return (0);
	warn("fgetln");
	return (-1);
}

int
get_double(const char *str, double *dx)
{
	char *ep;

	errno = 0;
	*dx = strtod(str, &ep);
	if (str[0] == '\0' || *ep != '\0')
		return (-1);	/* not a number */
	if (errno == ERANGE && *dx == 0.0)
		return (-2);	/* underflow */
	if (errno == ERANGE && (*dx == HUGE_VAL || *dx == -HUGE_VAL))
		return (-3);	/* overflow */
	return (0);
}

int
open_audio(char *fname, audio_info_t *ainfo)
{
	int fd;

	fd = open(fname, O_WRONLY, 0);
	if (fd == -1) {
		warn("open %s", fname);
		return (-1);
	}

	AUDIO_INITINFO(ainfo);
	ainfo->play.encoding = AUDIO_ENCODING_SLINEAR;
	ainfo->play.precision = 8;
	ainfo->play.channels = 1;

	if (ioctl(fd, AUDIO_SETINFO, ainfo) == -1) {
		warn("audio-setinfo %s", fname);
		close(fd);
		return (-1);
	}

	if (ioctl(fd, AUDIO_GETINFO, ainfo) == -1) {
		warn("audio-getinfo %s", fname);
		close(fd);
		return (-1);
	}

	return (fd);
}

int
main(int argc, char **argv)
{
	char *devname = NULL;
	audio_info_t ainfo;
	int f, c;
	double hz = 720.0;

	while ((c = getopt(argc, argv, "a:d:f:w:c:")) != -1) {
		switch (c) {
		case 'c':
			if (get_double(optarg, &charwpm) ||
			    charwpm < 1.0 || charwpm > 1000.0) {
				fprintf(stderr, "bad charwpm: %s"
				    "(1.0 <= charwpm <= 1000.0)\n", optarg);
				return (EX_USAGE);
			}
			break;
		case 'd':
			devname = optarg;
			break;
		case 'f':
			if (get_double(optarg, &hz) ||
			    hz < 100.0 || hz > 10000.0) {
				fprintf(stderr, "bad frequency: %s"
				    "(100.0 <= freq <= 10000.0)\n", optarg);
				return (EX_USAGE);
			}
			break;
		case 'w':
			if (get_double(optarg, &overallwpm) ||
			    overallwpm < 1.0 || overallwpm > 1000.0) {
				fprintf(stderr, "bad wpm: %s"
				    "(1.0 <= wpm <= 1000.0)\n", optarg);
				return (EX_USAGE);
			}
			break;
		case 'a':
			if (get_double(optarg, &attackpercent) ||
			    attackpercent < 0.0 || attackpercent >= 50.0) {
				fprintf(stderr, "bad attack percentage: %s"
				    "(0.0 <= attack < 50.0)\n", optarg);
				return (EX_USAGE);
			}
			break;
		default:
			fprintf(stderr, "%s [-d device] [-f freq] [-w wpm]\n",
			    argv[0]);
			return (EX_USAGE);
		}
	}

	if (overallwpm > charwpm) {
		fprintf(stderr, "character rate must be greater or "
		    "equal to overall rate (%f < %f).\n",
		    charwpm, overallwpm);
		return (EX_USAGE);
	}

	attackpercent /= 100.0;

	if (devname == NULL)
		devname = "/dev/audio";
	if ((f = open_audio(devname, &ainfo)) == -1)
		return (1);

	/* Build up the various symbols (including silence) */
	if (build_inSym(&inSym, &ainfo) ||
	    build_inChar(&inChar, &ainfo) ||
	    build_inWord(&inWord, &ainfo) ||
	    build_dit(&dit, &ainfo, hz) || build_dah(&dah, &ainfo, hz))
		return (1);

	do_input(f, stdin);

	return (0);
}
