/*
 * Copyright (C), 2000-2007 by the monit project group.
 * All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include <config.h>

#ifdef HAVE_STDIO_H
#include <stdio.h>
#endif

#ifdef HAVE_ERRNO_H
#include <errno.h>
#endif

#ifdef HAVE_STRING_H
#include <string.h>
#endif

#include "protocol.h"


/* Private prototypes */
static int send(Socket_T s, char *msg);
static int expect(Socket_T s, int expect, int log);


/**
 * Check the server for greeting code 220 and send EHLO. If that failed
 * try HELO and test for return code 250 and finally send QUIT and check
 * for return code 221. If alive return TRUE else return FALSE.
 *
 *  @author Jan-Henrik Haukeland, <hauk@tildeslash.com>
 *  @author Michael Amster, <mamster@webeasy.com>
 *
 *  @version \$Id: smtp.c,v 1.32 2007/01/03 09:31:02 martinp Exp $
 *
 *  @file
 */
int check_smtp(Socket_T s) {
    
  ASSERT(s);
  
  if(!expect(s, 220, TRUE))
    return FALSE;
  
  if(!(send(s, "EHLO localhost\r\n") && expect(s, 250, FALSE))) {
    /* Try HELO also before giving up as of rfc2821 4.1.1.1 */
    if(!(send(s, "HELO localhost\r\n") && expect(s, 250, TRUE)))
      return FALSE;
  }

  if(!(send(s, "QUIT\r\n") && expect(s, 221, TRUE)))
    return FALSE;
    
  return TRUE;
  
}


/* --------------------------------------------------------------- Private */


static int send(Socket_T s, char *msg) {
  
  if(socket_write(s, msg, strlen(msg)) < 0) {
    LogError("SMTP: error sending data -- %s\n", STRERROR);
    return FALSE;
  }
  
  return TRUE;
  
}


static int expect(Socket_T s, int expect, int log) {
  
  int status;
  char buf[STRLEN];

  if(!socket_readln(s, buf, STRLEN)) {
    LogError("SMTP: error receiving data -- %s\n", STRERROR);
    return FALSE;
  }
  
  Util_chomp(buf);
  
  sscanf(buf, "%d%*s", &status);
  if(status != expect) {
    if(log) 
      LogError("SMTP error: %s\n", buf);
    return FALSE;
  }
  
  return TRUE;
  
}

