/*
 * Copyright (C), 2000-2006 by the monit project group.
 * All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

/**
 *  System dependent device methods.
 *
 *  @author Martin Pala, <martinp@tildeslash.com>
 *
 *  @version \$Id: sysdep_NETBSD.c,v 1.4 2006/04/27 20:16:03 martinp Exp $
 *
 *  @file
 */

#include <config.h>

#ifdef HAVE_STDIO_H
#include <stdio.h>
#endif

#ifdef HAVE_ERRNO_H
#include <errno.h>
#endif

#ifdef HAVE_STRING_H
#include <string.h>
#endif

#ifdef HAVE_SYS_PARAM_H
#include <sys/param.h>
#endif

#ifdef HAVE_SYS_MOUNT_H
#include <sys/mount.h>
#endif

#include "monitor.h"
#include "device_sysdep.h"

/**
 * NetBSD special block device mountpoint method. Filesystem must be mounted.
 * In the case of success, mountpoint is stored in device information
 * structure for later use.
 *
 * @param inf  Information structure where resulting data will be stored
 * @param blockdev Identifies block special device
 * @return         NULL in the case of failure otherwise mountpoint
 */
char *device_mountpoint_sysdep(Info_T inf, char *blockdev) {

  struct statfs usage;

  ASSERT(inf);
  ASSERT(blockdev);

  if(statfs(blockdev, &usage) != 0) {
    LogError("%s: Error getting mountpoint for device '%s' -- %s\n",
        prog, blockdev, STRERROR);
    return NULL;
  }

  return strncpy(inf->mntpath, usage.f_mntonname, sizeof(inf->mntpath));

}


/**
 * NetBSD filesystem usage statistics. In the case of success result is stored in
 * given information structure.
 *
 * @param inf Information structure where resulting data will be stored
 * @return        TRUE if informations were succesfully read otherwise FALSE
 */
int device_usage_sysdep(Info_T inf) {

  struct statfs usage;

  ASSERT(inf);

  if(statfs(inf->mntpath, &usage) != 0) {
    LogError("%s: Error getting usage statistics for device '%s' -- %s\n",
        prog, inf->mntpath, STRERROR);
    return FALSE;
  }

  inf->f_bsize=           usage.f_bsize;
  inf->f_blocks=          usage.f_blocks;
  inf->f_blocksfree=      usage.f_bavail;
  inf->f_blocksfreetotal= usage.f_bfree;
  inf->f_files=           usage.f_files;
  inf->f_filesfree=       usage.f_ffree;

  return TRUE;

}

