#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <assert.h>
#include <math.h>

#include "mlocale.h"
#include "mhash.h"
#include "misc.h"


#include "datatypes/count/datatype.h"
#include "datatypes/location/datatype.h"

#include "web.h"
#include "plugin_config.h"
#include "pictures.h"
#include "template.h"
#include "generate.h"

#include "md5_global.h"
#include "md5.h"

/* bit-vector */
#define BV(x) (1 << x)

const int HIGHLIGHT           = BV(0);
const int GROUPING            = BV(1);
const int VISITS              = BV(2);
const int INDEX               = BV(3);
const int BROKEN_LINK         = BV(4);
const int PERCENT             = BV(5);
const int RESOLVE_TLD         = BV(6);
const int VISITS_TRAFFIC      = BV(7);
const int SORT_BY_KEY         = BV(8);
const int VIEW_DURATION       = BV(9);
const int TIME                = BV(10);
const int SORT_BY_VCOUNT      = BV(11);
const int SORT_BY_QUOTIENT    = BV(12);
const int LOCATION_CITY       = BV(13);
const int LOCATION_PROVINCE   = BV(14);
const int LOCATION_COUNTRY    = BV(15);
const int LOCATION_PROVIDER   = BV(16);

const char* TABLE_TITLE       = "TABLE_TITLE";
const char* TABLE_COL_SPAN    = "TABLE_COL_SPAN";

const char* TABLE_ROW         = "table_row";
const char* TABLE_CELL        = "table_cell";

const char* CELL_CLASS        = "CELL_CLASS";
const char* CELL_TAGS         = "CELL_TAGS";

const char* CELL_CONTENT      = "CELL_CONTENT";
const char* CELL_ALIGN        = "CELL_ALIGN";

const char* CELL_ALIGN_RIGHT  = "right";
const char* CELL_ALIGN_CENTER = "center";
const char* CELL_ALIGN_LEFT   = "left";

const int BOTTOM_THRESHOLD    = 16;

/* just a step, never the final */

typedef enum {
	CT_TITLE,
	CT_LEFTHEADER, CT_HEADER, CT_RIGHTHEADER,
	CT_LEFTLINE, CT_LINE, CT_RIGHTLINE,
	CT_LEFTFOOTER, CT_FOOTER, CT_RIGHTFOOTER,
	CT_LEFTHLINE, CT_HLINE, CT_RIGHTHLINE,
	CT_LEFTGLINE, CT_GLINE, CT_RIGHTGLINE } cell_type;

typedef enum {
	CA_LEFT, CA_CENTER, CA_RIGHT } cell_align;



/**
  *  Render a single table cell
 */
void render_cell (mconfig *ext_conf, tmpl_main *tmpl, const char* content, cell_type type, cell_align align) {
	config_output *conf = ext_conf->plugin_conf;
	tmpl_set_current_block(tmpl, TABLE_CELL);

	switch (align) {
		case CA_LEFT:
			tmpl_set_var(tmpl, CELL_ALIGN, CELL_ALIGN_LEFT);
			break;
		case CA_CENTER:
			tmpl_set_var(tmpl, CELL_ALIGN, CELL_ALIGN_CENTER);
			break;
		case CA_RIGHT:
			tmpl_set_var(tmpl, CELL_ALIGN, CELL_ALIGN_RIGHT);
			break;
	}

	switch (type) {
		case CT_TITLE:
			tmpl_set_var(tmpl, CELL_CLASS, conf->cell_class_title);
			tmpl_set_var(tmpl, CELL_TAGS, conf->cell_tags_title);
			break;
		case CT_LEFTHEADER:
			tmpl_set_var(tmpl, CELL_CLASS, conf->cell_class_leftheader);
			tmpl_set_var(tmpl, CELL_TAGS, conf->cell_tags_leftheader);
			break;
		case CT_HEADER:
			tmpl_set_var(tmpl, CELL_CLASS, conf->cell_class_header);
			tmpl_set_var(tmpl, CELL_TAGS, conf->cell_tags_header);
			break;
		case CT_RIGHTHEADER:
			tmpl_set_var(tmpl, CELL_CLASS, conf->cell_class_rightheader);
			tmpl_set_var(tmpl, CELL_TAGS, conf->cell_tags_rightheader);
			break;
		case CT_LEFTLINE:
			tmpl_set_var(tmpl, CELL_CLASS, conf->cell_class_leftline);
			tmpl_set_var(tmpl, CELL_TAGS, conf->cell_tags_leftline);
			break;
		case CT_LINE:
			tmpl_set_var(tmpl, CELL_CLASS, conf->cell_class_line);
			tmpl_set_var(tmpl, CELL_TAGS, conf->cell_tags_line);
			break;
		case CT_RIGHTLINE:
			tmpl_set_var(tmpl, CELL_CLASS, conf->cell_class_rightline);
			tmpl_set_var(tmpl, CELL_TAGS, conf->cell_tags_rightline);
			break;
		case CT_LEFTFOOTER:
			tmpl_set_var(tmpl, CELL_CLASS, conf->cell_class_leftfooter);
			tmpl_set_var(tmpl, CELL_TAGS, conf->cell_tags_leftfooter);
			break;
		case CT_FOOTER:
			tmpl_set_var(tmpl, CELL_CLASS, conf->cell_class_footer);
			tmpl_set_var(tmpl, CELL_TAGS, conf->cell_tags_footer);
			break;
		case CT_RIGHTFOOTER:
			tmpl_set_var(tmpl, CELL_CLASS, conf->cell_class_rightfooter);
			tmpl_set_var(tmpl, CELL_TAGS, conf->cell_tags_rightfooter);
			break;
		case CT_LEFTHLINE:
			tmpl_set_var(tmpl, CELL_CLASS, conf->cell_class_lefthline);
			tmpl_set_var(tmpl, CELL_TAGS, conf->cell_tags_lefthline);
			break;
		case CT_HLINE:
			tmpl_set_var(tmpl, CELL_CLASS, conf->cell_class_hline);
			tmpl_set_var(tmpl, CELL_TAGS, conf->cell_tags_hline);
			break;
		case CT_RIGHTHLINE:
			tmpl_set_var(tmpl, CELL_CLASS, conf->cell_class_righthline);
			tmpl_set_var(tmpl, CELL_TAGS, conf->cell_tags_righthline);
			break;
		case CT_LEFTGLINE:
			tmpl_set_var(tmpl, CELL_CLASS, conf->cell_class_leftgline);
			tmpl_set_var(tmpl, CELL_TAGS, conf->cell_tags_leftgline);
			break;
		case CT_GLINE:
			tmpl_set_var(tmpl, CELL_CLASS, conf->cell_class_gline);
			tmpl_set_var(tmpl, CELL_TAGS, conf->cell_tags_gline);
			break;
		case CT_RIGHTGLINE:
			tmpl_set_var(tmpl, CELL_CLASS, conf->cell_class_rightgline);
			tmpl_set_var(tmpl, CELL_TAGS, conf->cell_tags_rightgline);
			break;
	}

	tmpl_set_var(tmpl, CELL_CONTENT, content);

	tmpl_parse_current_block(tmpl);
}

/**
  * Build a single table line from a mhash
 */
int show_mhash_web (mconfig *ext_conf, mstate *state, tmpl_main *tmpl, mhash *h, int count, int opt) {
	int i;
	config_output *conf = ext_conf->plugin_conf;
	long sum = 0;
	double sum_vcount = 0;
	cell_type line_type = CT_LINE; /* default line type */
	mdata **md;
	
#ifdef MTIMER_ENABLED
	static mtimer timer;

	MTIMER_START(timer);
#endif
	if (!h) {
		fprintf(stderr, "%s.%d: no hash for me\n", __FILE__, __LINE__);
		return 0;
	}

	sum = mhash_sumup(h);
	if (opt & VISITS && opt & PERCENT)
		sum_vcount = mhash_sumup_vcount(h);

	if (opt & SORT_BY_KEY) {
		md = mhash_sorted_to_marray(h, M_SORTBY_KEY, M_SORTDIR_ASC);
	} else if (opt & SORT_BY_VCOUNT) {
		md = mhash_sorted_to_marray(h, M_SORTBY_VCOUNT, M_SORTDIR_DESC);
	} else if (opt & SORT_BY_QUOTIENT) {
		md = mhash_sorted_to_marray(h, M_SORTBY_QUOTIENT, M_SORTDIR_DESC);
	} else {
		md = mhash_sorted_to_marray(h, M_SORTBY_COUNT, M_SORTDIR_DESC);
	}

	for (i = 0; md[i] && (i < count); i++) {
		mdata *data = md[i];

		line_type = CT_LEFTLINE; /* start with left, linetype */

		if (data) {
			char *enc_url = NULL;
			int cut_url = 0;
			unsigned int c = 0;
			char buf[255];
			const char *key;
			
			key = mdata_get_key(data, state);
			
			enc_url = html_encode(key);
			cut_url = (strlen(enc_url) > 40) /*&& !conf->dont_cut_urls*/;
			
			if (cut_url) {
				enc_url[40] = '\0';
			}

			if (opt & INDEX) {
				snprintf(buf, 255,"%d", i+1);

				render_cell(ext_conf, tmpl, buf, line_type, CA_RIGHT );
				line_type = CT_LINE; /* set linetype to default */
			}

			c = mdata_get_count(data);

			if (opt & TIME) {
				render_cell(ext_conf, tmpl, (char *)seconds_to_string(c, 1), line_type, CA_RIGHT );
				line_type = CT_LINE; /* set linetype to default */
			} else {
				snprintf(buf, 255, "%d", c);
				render_cell(ext_conf, tmpl, buf, line_type, CA_RIGHT );
				line_type = CT_LINE; /* set linetype to default */
			}

			if (opt & PERCENT && sum) {
				snprintf(buf, 255, "%.2f", c * 100.0 / sum);
				render_cell(ext_conf, tmpl, buf, line_type, CA_RIGHT );
				line_type = CT_LINE; /* set linetype to default */
			}

			if (opt & VISITS && data->type == M_DATA_TYPE_VISITED ) {
				/* if the vcount is used as 'traffic' */
				if (opt & VISITS_TRAFFIC) {
					render_cell(ext_conf, tmpl, (char *)bytes_to_string(mdata_get_vcount(data)), line_type, CA_RIGHT );
					line_type = CT_LINE; /* set linetype to default */
				} else {
					snprintf(buf, 255, "%.0f", mdata_get_vcount(data));
					render_cell(ext_conf, tmpl, buf, line_type, CA_RIGHT );
					line_type = CT_LINE; /* set linetype to default */
				}
				/* percentage display for second numeric column */
				if (opt & PERCENT && sum_vcount) {
					snprintf(buf, 255, "%.2f", mdata_get_vcount(data) * 100.0 / sum_vcount);
					render_cell(ext_conf, tmpl, buf, line_type, CA_RIGHT );
					line_type = CT_LINE; /* set linetype to default */
				}
			}

			if (opt & VIEW_DURATION && data->type == M_DATA_TYPE_VISITED ) {
				/* use count as duration and vcount as hits per page view */
				snprintf(buf, 255,  "%.0f", mdata_get_vcount(data));
				render_cell(ext_conf, tmpl, buf, line_type, CA_RIGHT );
				line_type = CT_LINE; /* set linetype to default */

				render_cell(ext_conf, tmpl, mdata_get_vcount(data) ?
					(char *)seconds_to_string(mdata_get_count(data) / mdata_get_vcount(data), 1) :
					"--", line_type, CA_RIGHT );
				line_type = CT_LINE; /* set linetype to default */
			}

			/* ----------- end of the possible left part ----- */

			if ((opt & GROUPING) && mdata_is_grouped(data)) {
				tmpl_set_current_block(tmpl, TABLE_CELL);
				if (opt & BROKEN_LINK && data->type == M_DATA_TYPE_BROKENLINK) {
					tmpl_set_var(tmpl, CELL_CLASS, conf->cell_class_gline);
					tmpl_set_var(tmpl, CELL_TAGS, conf->cell_tags_gline);
				} else {
					tmpl_set_var(tmpl, CELL_CLASS, conf->cell_class_rightgline);
					tmpl_set_var(tmpl, CELL_TAGS, conf->cell_tags_rightgline);
				}
				tmpl_set_var(tmpl, CELL_ALIGN, CELL_ALIGN_LEFT);
				tmpl_set_var(tmpl, CELL_CONTENT, enc_url);
				if (cut_url) tmpl_append_var(tmpl, CELL_CONTENT, "...");
				tmpl_parse_current_block(tmpl);
			} else {
				if (opt & HIGHLIGHT) {
					char *s;
					int s_size;
					tmpl_set_current_block(tmpl, TABLE_CELL);
					if (opt & BROKEN_LINK && data->type == M_DATA_TYPE_BROKENLINK) {
						tmpl_set_var(tmpl, CELL_CLASS, conf->cell_class_hline);
						tmpl_set_var(tmpl, CELL_TAGS, conf->cell_tags_hline);
					} else {
						tmpl_set_var(tmpl, CELL_CLASS, conf->cell_class_righthline);
						tmpl_set_var(tmpl, CELL_TAGS, conf->cell_tags_righthline);
					}
					tmpl_set_var(tmpl, CELL_ALIGN, CELL_ALIGN_LEFT);
					if (conf->assumedprotocol == NULL || strstr(key, "://")) {
						s = malloc(s_size = strlen(key) +
							strlen(enc_url) +
							(cut_url ? 3 : 0) +
							strlen("<a href=\"\"></a>") +
							1);
						snprintf(s,s_size,"<a href=\"%s\">%s</a>%s",
							key,
							enc_url, cut_url ? "..." : "");
					} else {
						s = malloc(s_size = strlen(conf->assumedprotocol) +
							strlen(conf->hostname) +
							(*key == '/' ? 0 : 1) +
							strlen(key) +
							strlen(enc_url) +
							(cut_url ? 3 : 0) +
							strlen("<a href=\"://\"></a>") +
							1);
						snprintf(s,s_size,"<a href=\"%s://%s%s%s\">%s</a>%s",
							conf->assumedprotocol,
							conf->hostname,
							*key == '/' ? "" : "/",
							key,
							enc_url,
							cut_url ? "..." : "");
					}
					tmpl_set_var(tmpl, CELL_CONTENT, s);
					tmpl_parse_current_block(tmpl);
					free(s);
				} else {
					/* added option to resolve tlds on the fly */
					if (opt & RESOLVE_TLD) {
						char *c = html_encode(misoname(key));
						if (!(opt & BROKEN_LINK && data->type == M_DATA_TYPE_BROKENLINK)) {
							line_type = CT_RIGHTLINE; }
						render_cell(ext_conf, tmpl, c, line_type, CA_LEFT );
						free(c);
					} else if (data->type == M_DATA_TYPE_LOCATION) {
						/* the first row of the location data */
						if (opt & LOCATION_CITY) {
							render_cell(ext_conf, tmpl, data->data.location->city, line_type, CA_LEFT );
						} else if (opt & LOCATION_PROVINCE) {
							render_cell(ext_conf, tmpl, data->data.location->province, line_type, CA_LEFT );
						} else if (opt & LOCATION_COUNTRY) {
							render_cell(ext_conf, tmpl, data->data.location->country, line_type, CA_LEFT );
						} else if (opt & LOCATION_PROVIDER) {
							render_cell(ext_conf, tmpl, data->data.location->provider, line_type, CA_LEFT );
						}
					} else {
						tmpl_set_current_block(tmpl, TABLE_CELL);
						if (opt & BROKEN_LINK && data->type == M_DATA_TYPE_BROKENLINK) {
							tmpl_set_var(tmpl, CELL_CLASS, conf->cell_class_line);
							tmpl_set_var(tmpl, CELL_TAGS, conf->cell_tags_line);
						} else {
							tmpl_set_var(tmpl, CELL_CLASS, conf->cell_class_rightline);
							tmpl_set_var(tmpl, CELL_TAGS, conf->cell_tags_rightline);
						}
						tmpl_set_var(tmpl, CELL_ALIGN, CELL_ALIGN_LEFT);
						tmpl_set_var(tmpl, CELL_CONTENT, enc_url);
						if (cut_url) tmpl_append_var(tmpl, CELL_CONTENT, "...");
						tmpl_parse_current_block(tmpl);
					}
				}
			}

			if (opt & BROKEN_LINK && data->type == M_DATA_TYPE_BROKENLINK) {
				struct tm *_tm;
				char timebuf[32] = "";

				if (data->data.brokenlink.referrer &&
				    (0 != strcmp(data->data.brokenlink.referrer, "-"))) {

					free(enc_url);
					enc_url = html_encode(data->data.brokenlink.referrer);
					cut_url = strlen(enc_url) > 40;

					if (cut_url) {
						enc_url[40] = '\0';
					}

					tmpl_set_current_block(tmpl, TABLE_CELL);
					tmpl_set_var(tmpl, CELL_CLASS, conf->cell_class_line);
					tmpl_set_var(tmpl, CELL_TAGS, conf->cell_tags_line);
					tmpl_set_var(tmpl, CELL_ALIGN, CELL_ALIGN_LEFT);
					tmpl_set_var(tmpl, CELL_CONTENT, "<a href=\"");
					tmpl_append_var(tmpl, CELL_CONTENT, data->data.brokenlink.referrer);
					tmpl_append_var(tmpl, CELL_CONTENT, "\">");
					tmpl_append_var(tmpl, CELL_CONTENT, enc_url);
					tmpl_append_var(tmpl, CELL_CONTENT, "</a>");
					if (cut_url) tmpl_append_var(tmpl, CELL_CONTENT, "...");
					tmpl_parse_current_block(tmpl);
				} else {
					render_cell(ext_conf, tmpl, "-", line_type, CA_LEFT );
				}

				_tm = localtime(&(data->data.brokenlink.timestamp));
				if (strftime(timebuf, sizeof(timebuf)-1, "%x", _tm) == 0) {
					fprintf(stderr, "output::modlogan.show_mhash: strftime failed\n");
				}
				render_cell(ext_conf, tmpl, timebuf, CT_RIGHTLINE, CA_LEFT );
			}
			
			if (data->type == M_DATA_TYPE_LOCATION) {
				if ((opt & LOCATION_CITY) && 
				    (opt & LOCATION_PROVINCE)) {
					render_cell(ext_conf, tmpl, data->data.location->province, line_type, CA_LEFT );
				}
			
				if ((opt & LOCATION_PROVINCE) &&
				    (opt & LOCATION_COUNTRY)) {
					render_cell(ext_conf, tmpl, data->data.location->country, line_type, CA_LEFT );
				}
			
				if ((opt & LOCATION_COUNTRY) &&
				    (opt & LOCATION_PROVIDER)) {
					render_cell(ext_conf, tmpl, data->data.location->provider, line_type, CA_LEFT );
				}
			}
				
			free(enc_url);

			parse_table_row(tmpl);
		}
	}

	free(md);
#ifdef MTIMER_ENABLED
	MTIMER_STOP(timer);
	MTIMER_CALC(timer);

	fprintf(stderr, "timer %s: %ld msec\n",
		__FUNCTION__, timer.span );
#endif
	return 0;
}

void parse_table_row(tmpl_main *tmpl) {

	/* generate the prepared row */
	tmpl_set_current_block(tmpl, TABLE_ROW);
	tmpl_parse_current_block(tmpl);

	/* clear all templates */
	tmpl_clear_block(tmpl, TABLE_CELL);
	tmpl_clear_var(tmpl, CELL_ALIGN);
	tmpl_clear_var(tmpl, CELL_CLASS);
	tmpl_clear_var(tmpl, CELL_TAGS);
}

mlist * get_next_element(mhash *h) {
	mlist *ret = NULL;
	int max = 0, i;

	for ( i = 0; i < h->size; i++) {
		mlist *l = h->data[i]->list;
		while (l) {
			if (l->data) {
				mdata *data = l->data;

				if ( mdata_get_count(data) > max) {
					max = mdata_get_count(data);
					ret = l;
				}
			}
			l = l->next;
		}
	}
	/* invert the value */
	if (ret) {
		mdata_set_count(ret->data, -mdata_get_count(ret->data));
	}

	return ret;
}

int cleanup_elements(mhash *h) {
	int i;
	for ( i = 0; i < h->size; i++) {
		mlist *l = h->data[i]->list;
		while (l) {
			if (l->data) {
				mdata *data = l->data;

				if (mdata_get_count(data) > 0) {
/*					fprintf(stderr, "%s.%d: count (%d) > 0 in cleanup\n",
						__FILE__, __LINE__,
						mdata_get_count(data));*/
				} else {
					mdata_set_count(data, -mdata_get_count(data));
				}

			}
			l = l->next;
		}
	}

	return 0;
}

int show_visit_path (mconfig *ext_conf, mstate *state, tmpl_main *tmpl,  mhash *h, int count, int opt) {
	int i = 0;
	config_output *conf = ext_conf->plugin_conf;
	mlist *l;
	long sum;
	char buf[255];

	if (!h) return 0;

	sum = mhash_sumup(h);

	while ((l = get_next_element(h)) && i < count) {
		if (l->data) {
			mdata *data = l->data;
			mlist *sl = data->data.sublist.sublist;
			int c;

			c = -data->data.sublist.count;

			i++;

			snprintf(buf, 255,  "%d", i);
			render_cell(ext_conf, tmpl, buf, CT_LEFTLINE, CA_RIGHT );
			snprintf(buf, 255,  "%d", c);
			render_cell(ext_conf, tmpl, buf, CT_LINE, CA_RIGHT );
			snprintf(buf, 255,  "%.2f", c * 100.0 / sum);
			render_cell(ext_conf, tmpl, buf, CT_LINE, CA_RIGHT );

			tmpl_set_current_block(tmpl, TABLE_CELL);
			tmpl_set_var(tmpl, CELL_CLASS, conf->cell_class_rightline);
			tmpl_set_var(tmpl, CELL_TAGS, conf->cell_tags_rightline);
			tmpl_set_var(tmpl, CELL_ALIGN, CELL_ALIGN_LEFT);
			tmpl_clear_var(tmpl, CELL_CONTENT);

			for (; sl && sl->data; sl = sl->next) {
				mdata *sld = sl->data;
				
				tmpl_append_var(tmpl, CELL_CONTENT, mdata_get_key(sld, state));
				tmpl_append_var(tmpl, CELL_CONTENT, "<br />");
			}
			tmpl_parse_current_block(tmpl);

			parse_table_row(tmpl);
		}
	}

	cleanup_elements(h);

	return 0;
}

int show_status_mhash (mconfig *ext_conf, tmpl_main *tmpl, mhash *h, int count) {
	int i;
	mdata **md;
	config_output *conf = ext_conf->plugin_conf;
	char buf[255];

	if (!h) return 0;

	md = mhash_sorted_to_marray(h, M_SORTBY_KEY, M_SORTDIR_ASC);

	for (i = 0; md[i] && (i < count); i++) {
		mdata *data = md[i];

		if (data) {
			snprintf(buf, 255,  "%d", data->data.count.count);
			render_cell(ext_conf, tmpl, buf, CT_LEFTLINE, CA_RIGHT );
			
			tmpl_set_current_block(tmpl, TABLE_CELL);
			tmpl_set_var(tmpl, CELL_CLASS, conf->cell_class_rightline);
			tmpl_set_var(tmpl, CELL_TAGS, conf->cell_tags_rightline);
			tmpl_clear_var(tmpl, CELL_ALIGN);
			tmpl_set_var(tmpl, CELL_CONTENT, data->key);
			tmpl_append_var(tmpl, CELL_CONTENT, " - ");
			tmpl_append_var(tmpl, CELL_CONTENT, mhttpcodes(strtol(data->key, NULL, 10)));

			tmpl_parse_current_block(tmpl);

			parse_table_row(tmpl);
		}
	}

	free(md);

	return 0;
}

mhash *get_entry_pages(mhash *h, mstate *state) {
	mhash *ret;
	int i;

	if (!h) return 0;

	ret = mhash_init( 32 );

	for ( i = 0; i < h->size; i++) {
		mlist *l;
		for (l = h->data[i]->list; l && l->data; l = l->next) {
			mdata *data = l->data;
			mlist *sl = data->data.sublist.sublist;
			
			if (data->type != M_DATA_TYPE_SUBLIST) {
				M_DEBUG2(M_DEBUG_LEVEL_ERRORS, M_DEBUG_SECTION_INIT, M_DEBUG_LEVEL_ERRORS,
					 "datatype not a sublist: %d - %s\n", data->type, data->key);
				return NULL;
			}
			
			if (sl && sl->data) {
				mdata *sld = sl->data;
				mdata *insd;
				
				insd = mdata_Count_create(mdata_get_key(sld, state), 1, M_DATA_STATE_PLAIN);
				mhash_insert_sorted(ret, insd);
			}
		}
	}

	return ret;
}

mhash *get_exit_pages(mhash *h, mstate *state) {
	mhash *ret;
	int i;

	if (!h) return 0;

	ret = mhash_init( 32 );

	for ( i = 0; i < h->size; i++) {
		mlist *l;
		for (l = h->data[i]->list; l && l->data; l = l->next) {
			mdata *data = l->data;
			mlist *sl = data->data.sublist.sublist;

			if (sl) {
				/* go to the last element */
				while (sl->next) sl = sl->next;

				if (sl->data) {
					mdata *sld = sl->data;
					mdata *insd;
					
					insd = mdata_Count_create(mdata_get_key(sld, state), 1, M_DATA_STATE_PLAIN);
					mhash_insert_sorted(ret, insd);
				}
			}
		}
	}

	return ret;
}

mhash *get_visit_path_length(mhash *h) {
	mhash *ret;
	int i;

	if (!h) return 0;

	ret = mhash_init( 32 );

	for ( i = 0; i < h->size; i++) {
		mlist *l;
		
		for (l = h->data[i]->list; l && l->data; l = l->next) {
			mdata *data = l->data;
			mlist *sl = data->data.sublist.sublist;
			
			if (sl) {
				/* go to the last element */
				long c = 0;
				char str[255];
				mdata *insd;
				
				for (; sl; c++, sl = sl->next);
				
				snprintf(str, 255, "%5ld", c);

				insd = mdata_Count_create(str, l->data->data.sublist.count, M_DATA_STATE_PLAIN);
				mhash_insert_sorted(ret, insd);
			}
		}
	}

	return ret;
}

mhash *get_visit_duration(mhash *h, mstate *state) {
	mhash *ret;
	int i;

	if (!h) return 0;

	ret = mhash_init( 32 );

	for ( i = 0; i < h->size; i++) {
		mlist *l;
		for (l = h->data[i]->list; l; l = l->next) {
			mlist *sl;

			if (!l->data || !l->data->data.sublist.sublist) continue;

			/* get sublist */
			sl = l->data->data.sublist.sublist;

			if (sl->data) {
				time_t t1, t2;
				char str[255];
				mdata *insd;

				/* HARD breackpoint */
				if (sl->data->type != M_DATA_TYPE_BROKENLINK) {
					fprintf(stderr, "%s.%d: last link (%s) is not BROKENLINK\n",
						__FILE__, __LINE__,
						mdata_get_key(sl->data, state)
						);
					
					return NULL;
				}

				/* get timestamp of first element */
				t1 = sl->data->data.brokenlink.timestamp;

				/* go to the last element */
				for (;sl->next && sl->next->data; sl = sl->next);

				/* HARD breackpoint */
				if (sl->data->type != M_DATA_TYPE_BROKENLINK) {
					fprintf(stderr, "%s.%d: last link (%s) is not BROKENLINK\n",
						__FILE__, __LINE__,
						mdata_get_key(sl->data, state)
						);
					
					
					return NULL;
					
				}
				/* get timestamp of the last element */
				t2 = sl->data->data.brokenlink.timestamp;

				if ((t2 - t1) >= 60) {
					snprintf(str, 255, "%5ld %s", (t2 - t1) / 60, _("min"));
				} else {
					snprintf(str, 255, " < 1 %s", _("min"));
				}

				/* HARD breackpoint */
				if (t2 - t1 < 0) {
					fprintf(stderr, "%s.%d: visit duration negative: %ld, will die now\n",
						__FILE__, __LINE__, t2 - t1);
					
					return NULL;
				}

				insd = mdata_Count_create(str, l->data->data.sublist.count, M_DATA_STATE_PLAIN);
				mhash_insert_sorted(ret, insd);
			}
		}
	}

	return ret;
}

double get_visit_full_path_length(mhash *h) {
	double d = 0;
	int i;

	if (!h) return 0;

	for ( i = 0; i < h->size; i++) {
		mlist *l;

		for (l = h->data[i]->list; l; l = l->next) {
			if (l->data) {
				mdata *data = l->data;

				d += mlist_count(data->data.sublist.sublist);
			}
		}
	}

	return d;
}

double get_visit_full_duration(mhash *h) {
	double d = 0;
	int i;
	
	if (!h) return 0;

	for ( i = 0; i < h->size; i++) {
		mlist *l = h->data[i]->list;
		for (l = h->data[i]->list; l && l->data; l = l->next) {
			mdata *data = l->data;
			mlist *sl = data->data.sublist.sublist;

			if (sl && sl->data) {
				/* get the timestamp of the first entry */
				time_t t1 = sl->data->data.brokenlink.timestamp, t2;
					
				/* go to the last entry */
				for (;sl->next;sl = sl->next);
				
				/* timestamp of the last entry */
				t2 = sl->data->data.brokenlink.timestamp;
				
				d += (t2 - t1);
			}
		}
	}

	return d;
}

mhash *get_path_length(mhash *h) {
	mhash *ret;
	mlist *l;

	if (!h) return 0;

	ret = mhash_init( 32 );

	while ((l = get_next_element(h))) {
		if (l->data) {
			mdata *data = l->data;
			mlist *sl = data->data.sublist.sublist;

			if (sl) {
				/* go to the last element */
				long c = 0;
				char str[255];
				mdata *insd;
				
				for (; sl; c++, sl = sl->next);

				snprintf(str, 255, "%5ld", c);

				insd = mdata_Count_create(str, l->data->data.sublist.count, M_DATA_STATE_PLAIN);
				mhash_insert_sorted(ret, insd);
			}
		}
	}

	cleanup_elements(h);

	return ret;
}

mhash *get_location_subset(mhash *h, int opt) {
	mhash *ret;
	int i;

	if (!h) return 0;

	ret = mhash_init( 32 );

	for ( i = 0; i < h->size; i++) {
		mlist *l;
		for (l = h->data[i]->list; l && l->data; l = l->next) {
			mdata *data = l->data;
			mdata *insd;
			char md5str[33];
			MD5_CTX context;
			unsigned char digest[16];
			char *r;
			int j;

			/* build key */
			
			md5str[0] = '\0';
			MD5Init(&context);
			
			if (opt & LOCATION_CITY) {
				if (data->data.location->city) {
					MD5Update(&context, 
						  data->data.location->city, 
						  strlen(data->data.location->city));
				} else {
					MD5Update(&context, "", 0);
				}
			}
				
			if (opt & LOCATION_PROVINCE) {
				if (data->data.location->province) {
					MD5Update(&context, 
						  data->data.location->province, 
						  strlen(data->data.location->province));
				} else {
					MD5Update(&context, "", 0);
				}
			}
			
			if (opt & LOCATION_COUNTRY) {
				if (data->data.location->country) {
					MD5Update(&context, 
						  data->data.location->country, 
						  strlen(data->data.location->country));
				} else {
					MD5Update(&context, "", 0);
				}
			}
			
			if (opt & LOCATION_PROVIDER) {
				if (data->data.location->provider) {
					MD5Update(&context, 
						  data->data.location->provider, 
						  strlen(data->data.location->provider));
				} else {
					MD5Update(&context, "", 0);
				}
			}
			
			MD5Final(digest, &context);
			for (j = 0, r = md5str; j < 16; j++, r += 2) {
				sprintf(r, "%02x", digest[j]);
			}
			*r = '\0';	
			
				
			insd = mdata_Location_create(md5str, data->data.location->count, 
						     data->data.location->city,
						     data->data.location->province,
						     data->data.location->country,
						     data->data.location->provider
						     );
			mhash_insert_sorted(ret, insd);
		}
	}

	return ret;
}

const char* M_REPORT_REQ_URL                = "web_request_url";
const char* M_REPORT_REQ_URL_TRAFFIC        = "web_request_url_traffic";
const char* M_REPORT_REQ_PROT               = "web_request_protocol";
const char* M_REPORT_REQ_METH               = "web_request_method";
const char* M_REPORT_STATUS_CODES           = "web_request_status";
const char* M_REPORT_EXTENSION              = "web_request_extensions";
const char* M_REPORT_EXTENSION_TRAFFIC      = "web_request_extensions_traffic";

const char* M_REPORT_REQ_USERS              = "web_user_users";
const char* M_REPORT_REQ_USERS_TRAFFIC      = "web_user_users_traffic";
const char* M_REPORT_REF_URL                = "web_user_referer";
const char* M_REPORT_OS                     = "web_user_os";
const char* M_REPORT_OS_VISITS              = "web_user_os_visits";
const char* M_REPORT_HOSTS                  = "web_user_host";
const char* M_REPORT_HOSTS_VISITS           = "web_user_host_visits";
const char* M_REPORT_USERAGENT              = "web_user_useragent";
const char* M_REPORT_USERAGENT_VISITS       = "web_user_useragent_visits";
const char* M_REPORT_BOOKMARKS              = "web_user_bookmarks";
const char* M_REPORT_COUNTRIES              = "web_user_countries";
const char* M_REPORT_COUNTRIES_VISITS       = "web_user_countries_visits";
const char* M_REPORT_VIEW_DURATION          = "web_user_view_duration";
const char* M_REPORT_VIEW_DURATION_VISITS   = "web_user_view_duration_visits";
const char* M_REPORT_VIEW_DURATION_AVERAGE  = "web_user_view_duration_average";
const char* M_REPORT_LOCATION_CITIES        = "web_user_location_cities";
const char* M_REPORT_LOCATION_PROVINCES     = "web_user_location_provinces";
const char* M_REPORT_LOCATION_COUNTRIES     = "web_user_location_countries";
const char* M_REPORT_LOCATION_PROVIDER      = "web_user_location_provider";


const char* M_REPORT_ENTRY_PAGES            = "web_visit_entry_pages";
const char* M_REPORT_EXIT_PAGES             = "web_visit_exit_pages";
const char* M_REPORT_VISIT_PATH             = "web_visit_path";
const char* M_REPORT_VISIT_PATH_LENGTH      = "web_visit_path_length";
const char* M_REPORT_VISIT_DURATION         = "web_visit_duration";

const char* M_REPORT_INDEXED                = "web_robot_indexed_pages";
const char* M_REPORT_ROBOTS                 = "web_robot_names";
const char* M_REPORT_SEARCH_ENGINE          = "web_robot_searchengine";
const char* M_REPORT_SEARCH_STRINGS         = "web_robot_seachstrings";

const char* M_REPORT_BROKEN_LINKS           = "web_server_broken_links";
const char* M_REPORT_INTERNAL_ERROR         = "web_server_internal_errors";
const char* M_REPORT_VHOSTS                 = "web_server_vhosts";

const char* M_REPORT_SUMMARY                = "web_generic_summary";
const char* M_REPORT_HOURLY                 = "web_generic_hourly";
const char* M_REPORT_DAILY                  = "web_generic_daily";

/**
 *
 * @return copy of the report-def to generate_web
 */

const reports_def *get_reports_web (mconfig *ext_conf) {
	config_output *conf = ext_conf->plugin_conf;
	const reports_def reports[] = {
		/* 0 */
		  /* id             title */
		{ M_REPORT_REQ_URL, _("Requested URL's"),
			/* options */
			HIGHLIGHT | GROUPING | INDEX | PERCENT | VISITS | VISITS_TRAFFIC,
				0, /* show graph */
				NULL, /* graph function */
			{ /* fields */
				{ _("Hits"), "hits" },
				{ _("Traffic"), "traffic" },
				{ "%", "traffic" },
				{ _("URL"), NULL },
				{ NULL, NULL }
			}
		},
		/* 1 */
		{ M_REPORT_REF_URL, _("Referring URL's"),
			HIGHLIGHT | GROUPING | INDEX | PERCENT,
				0,
				NULL,
			{
				{ _("Hits"), "hits" },
				{ _("Referrer"), NULL },
				{ NULL, NULL },
				{ NULL, NULL },
				{ NULL, NULL }
			}
		},
		/* 2 */
		{ M_REPORT_VIEW_DURATION, _("View Durations"),
			HIGHLIGHT | GROUPING | INDEX | PERCENT | VIEW_DURATION | TIME,
				0,
				NULL,
			{
				{ _("Duration"), "duration" },
				{ _("Hits"), "hits" },
				{ _("Average"), "duration" },
				{ _("URL"), NULL },
				{ NULL, NULL }
			}
		},
		/* 3 */
		{ M_REPORT_OS, _("Used Operating Systems"),
			GROUPING | VISITS | INDEX | PERCENT,
				0,
				NULL,
			{
				{ _("Hits"), "hits" },
				{ _("Visits"), "visits" },
				{ "%", "visits" },
				{ _("Operating System"), NULL },
				{ NULL, NULL }

			}
		},
		/* 4 */
		{ M_REPORT_HOSTS, _("Hosts"),
			GROUPING | INDEX | VISITS | PERCENT,
				0,
				NULL,
			{
				{ _("Hits"), "hits" },
				{ _("Visits"), "visits" },
				{ "%", "visits" },
				{ _("Host"), NULL },
				{ NULL, NULL }
			}
		},
		/* 5 */
		{ M_REPORT_ENTRY_PAGES, _("Entry Pages"),
			HIGHLIGHT | INDEX | PERCENT,
				0,
				NULL,
			{
				{ _("Visits"), "visits" },
				{ _("Entry Page"), NULL },
				{ NULL, NULL },
				{ NULL, NULL },
				{ NULL, NULL }
			}
		},
		/* 6 */
		{ M_REPORT_EXIT_PAGES, _("Exit Pages"),
			HIGHLIGHT | INDEX | PERCENT,
				0,
				NULL,
			{
				{ _("Visits"), "visits" },
				{ _("Exit Page"), NULL },
				{ NULL, NULL },
				{ NULL, NULL },
				{ NULL, NULL }
			}
		},
		/* 7 */
		{ M_REPORT_INDEXED, _("Indexed Pages"),
			HIGHLIGHT | INDEX | PERCENT,
				0,
				NULL,
			{
				{ _("Hits"), "hits" },
				{ _("Indexed Page"), NULL },
				{ NULL, NULL },
				{ NULL, NULL },
				{ NULL, NULL }
			}
		},
		/* 8 */
		{ M_REPORT_USERAGENT, _("Used Browsers"),
			GROUPING | VISITS | INDEX | PERCENT,
				0,
				NULL,
			{
				{ _("Hits"), "hits" },
				{ _("Visits"), "visits" },
				{ "%", "visits" },
				{ _("Browser"), NULL },
				{ NULL, NULL }
			}
		},
		/* 9 */
		{ M_REPORT_REQ_PROT, _("Used Request Protocol"),
			INDEX | PERCENT,
				0,
				NULL,
			{
				{ _("Hits"), "hits" },
				{ _("Protocol"), NULL },
				{ NULL, NULL },
				{ NULL, NULL },
				{ NULL, NULL }
			}
		},
		/* 10 */
		{ M_REPORT_REQ_METH, _("Used Request Method"),
			INDEX | PERCENT,
				0,
				NULL,
			{
				{ _("Hits"), "hits" },
				{ _("Method"), NULL },
				{ NULL, NULL },
				{ NULL, NULL },
				{ NULL, NULL }
			}
		},
		/* 11 */
		{ M_REPORT_ROBOTS, _("Robots"),
			INDEX | PERCENT,
				0,
				NULL,
			{
				{ _("Hits"), "hits" },
				{ _("Robot"), NULL },
				{ NULL, NULL },
				{ NULL, NULL },
				{ NULL, NULL }
			}
		},
		/* 12 */
		{ M_REPORT_BOOKMARKS, _("Bookmarked Pages"),
			HIGHLIGHT | INDEX | PERCENT,
				0,
				NULL,
			{
				{ _("Hits"), "hits" },
				{ _("Bookmarked Page"), NULL },
				{ NULL, NULL },
				{ NULL, NULL },
				{ NULL, NULL },
				{ NULL, NULL }
			}
		},
		/* 13 */
		{ M_REPORT_BROKEN_LINKS, _("Missing File / Broken Link"),
			GROUPING | INDEX | BROKEN_LINK | PERCENT,
				0,
				NULL,
			{
				{ _("Hits"), "hits" },
				{ _("Broken Link"), NULL },
				{ _("last referrering URL"), NULL },
				{ _("Last Hit"), NULL },
				{ NULL, NULL },
				{ NULL, NULL }
			}
		},
		/* 14 */
		{ M_REPORT_INTERNAL_ERROR, _("Internal Errors"),
			INDEX | BROKEN_LINK,
				0,
				NULL,
			{
				{ _("Hits"), "hits" },
				{ _("Broken Link"), NULL },
				{ _("last referrering URL"), NULL },
				{ _("Last Hit"), NULL },
				{ NULL, NULL },
				{ NULL, NULL }
			}
		},
		/* 15 */
		{ M_REPORT_SEARCH_STRINGS, _("SearchStrings"),
			INDEX | PERCENT | GROUPING,
				0,
				NULL,
			{
				{ _("Hits"), "hits" },
				{ _("Search String"), NULL },
				{ NULL, NULL },
				{ NULL, NULL },
				{ NULL, NULL },
				{ NULL, NULL }
			}
		},
		/* 16 */
		{ M_REPORT_SEARCH_ENGINE, _("SearchEngines"),
			HIGHLIGHT | GROUPING | INDEX | PERCENT,
				0,
				NULL,

			{
				{ _("Hits"), "hits" },
				{ _("Search Engine"), NULL },
				{ NULL, NULL },
				{ NULL, NULL },
				{ NULL, NULL },
				{ NULL, NULL }
			}
		},
		/* 17 */
		{ M_REPORT_EXTENSION, _("Extensions"),
			HIGHLIGHT | GROUPING |INDEX | PERCENT | VISITS | VISITS_TRAFFIC,
				1,
				create_pic_ext,
			{
				{ _("Hits"), "hits" },
				{ _("Traffic"), "traffic" },
				{ "%", "traffic" },
				{ _("Extensions"), NULL },
				{ NULL, NULL },
				{ NULL, NULL }
			}
		},
		/* 18 */
		{ M_REPORT_VISIT_DURATION, _("Time Per Visit"),
			INDEX | PERCENT | SORT_BY_KEY,
				1,
				create_pic_vd,
			{
				{ _("Visits"), "visits" },
				{ _("Time per Visit"), NULL },
				{ NULL, NULL },
				{ NULL, NULL },
				{ NULL, NULL }
			}
		},
		/* 19 */
		{ M_REPORT_VISIT_PATH_LENGTH, _("Visit Path Length"),
			INDEX | PERCENT | SORT_BY_KEY,
				1,
				create_pic_vpl,

			{
				{ _("Visits"), "visits" },
				{ _("Pages per Visit"), NULL },
				{ NULL, NULL },
				{ NULL, NULL },
				{ NULL, NULL },
				{ NULL, NULL }
			}
		},
		/* 20 */
		{ M_REPORT_COUNTRIES, _("Top Level Domains"),
			VISITS | INDEX | PERCENT | RESOLVE_TLD,
				1,
				create_pic_countries,
			{
				{ _("Hits"), "hits" },
				{ _("Visits"), "visits" },
				{ "%", "visits" },
				{ _("Top Level Domain"), NULL },
				{ NULL, NULL },
				{ NULL, NULL }
			}
		},
		/* 21 */
		{ M_REPORT_VHOSTS, _("Vhosts"),
			GROUPING | INDEX | VISITS | PERCENT,
				1,
				create_pic_vhost,
			{
				{ _("Hits"), "hits" },
				{ _("Visits"), "visits" },
				{ "%", "visits" },
				{ _("Vhost"), NULL },
				{ NULL, NULL },
				{ NULL, NULL }
			}
		},
		/* 22 */
		  /* id             title */
		{ M_REPORT_REQ_URL_TRAFFIC, _("Requested URL's by Traffic"),
			/* options */
			HIGHLIGHT | GROUPING | INDEX | PERCENT | VISITS | VISITS_TRAFFIC | SORT_BY_VCOUNT,
				0, /* show graph */
				NULL, /* graph function */
			{ /* fields */
				{ _("Hits"), "hits" },
				{ _("Traffic"), "traffic" },
				{ "%", "traffic" },
				{ _("URL"), NULL },
				{ NULL, NULL },
				{ NULL, NULL }
			}
		},
		/* 23 */
		{ M_REPORT_EXTENSION_TRAFFIC, _("Extensions by Traffic"),
			HIGHLIGHT | GROUPING |INDEX | PERCENT | VISITS | VISITS_TRAFFIC | SORT_BY_VCOUNT,
				1,
				create_pic_ext_traffic,
			{
				{ _("Hits"), "hits" },
				{ _("Traffic"), "traffic" },
				{ "%", "traffic" },
				{ _("Extensions"), NULL },
				{ NULL, NULL },
				{ NULL, NULL }
			}
		},
		/* 24 */
		{ M_REPORT_OS_VISITS, _("Used Operating Systems by Visits"),
			GROUPING | VISITS | INDEX | PERCENT | SORT_BY_VCOUNT,
				0,
				NULL,
			{
				{ _("Hits"), "hits" },
				{ _("Visits"), "visits" },
				{ "%", "visits" },
				{ _("Operating System"), NULL },
				{ NULL, NULL },
				{ NULL, NULL }

			}
		},
		/* 25 */
		{ M_REPORT_HOSTS_VISITS, _("Hosts by Visits"),
			GROUPING | INDEX | VISITS | PERCENT | SORT_BY_VCOUNT,
				0,
				NULL,
			{
				{ _("Hits"), "hits" },
				{ _("Visits"), "visits" },
				{ "%", "visits" },
				{ _("Host"), NULL },
				{ NULL, NULL },
				{ NULL, NULL }
			}
		},
		/* 26 */
		{ M_REPORT_USERAGENT_VISITS, _("Used Browsers by Visits"),
			GROUPING | VISITS | INDEX | PERCENT | SORT_BY_VCOUNT,
				0,
				NULL,
			{
				{ _("Hits"), "hits" },
				{ _("Visits"), "visits" },
				{ "%", "visits" },
				{ _("Browser"), NULL },
				{ NULL, NULL },
				{ NULL, NULL }
			}
		},
		/* 27 */
		{ M_REPORT_COUNTRIES_VISITS, _("Countries by Visits"),
			VISITS | INDEX | PERCENT | RESOLVE_TLD | SORT_BY_VCOUNT,
				1,
				create_pic_countries_visits,
			{
				{ _("Hits"), "hits" },
				{ _("Visits"), "visits" },
				{ "%", "visits" },
				{ _("Country"), NULL },
				{ NULL, NULL },
				{ NULL, NULL }
			}
		},
		/* 28 */
		{ M_REPORT_VIEW_DURATION_VISITS, _("View Durations by Visits"),
			HIGHLIGHT | GROUPING | INDEX | PERCENT | VIEW_DURATION | TIME | SORT_BY_VCOUNT,
				0,
				NULL,
			{
				{ _("Duration"), "duration" },
				{ _("Hits"), "hits" },
				{ _("Average"), "duration" },
				{ _("URL"), NULL },
				{ NULL, NULL },
				{ NULL, NULL }
			}
		},

		/* 29 */
		{ M_REPORT_VIEW_DURATION_AVERAGE, _("View Durations by Average"),
			HIGHLIGHT | GROUPING | INDEX | PERCENT | VIEW_DURATION | TIME | SORT_BY_QUOTIENT,
				0,
				NULL,
			{
				{ _("Duration"), "duration" },
				{ _("Hits"), "hits" },
				{ _("Average"), "duration" },
				{ _("URL"), NULL },
				{ NULL, NULL },
				{ NULL, NULL }
			}
		},
		/* 30 */
		{ M_REPORT_REQ_USERS, _("Requesting Users"),
			/* options */
			GROUPING | INDEX | PERCENT | VISITS | VISITS_TRAFFIC,
				0, /* show graph */
				NULL, /* graph function */
			{ /* fields */
				{ _("Hits"), "hits" },
				{ _("Traffic"), "traffic" },
				{ "%", "traffic" },
				{ _("Users"), NULL },
				{ NULL, NULL },
				{ NULL, NULL }
			}
		},
		/* 31 */
		{ M_REPORT_REQ_USERS_TRAFFIC, _("Requesting Users by Traffic"),
			/* options */
			GROUPING | INDEX | PERCENT | VISITS | VISITS_TRAFFIC | SORT_BY_VCOUNT,
				0, /* show graph */
				NULL, /* graph function */
			{ /* fields */
				{ _("Hits"), "hits" },
				{ _("Traffic"), "traffic" },
				{ "%", "traffic" },
				{ _("Users"), NULL },
				{ NULL, NULL },
				{ NULL, NULL }
			}
		},
		
		/* 32 */
		{ M_REPORT_LOCATION_CITIES, _("Cities"),
			/* options */
			INDEX | PERCENT | LOCATION_CITY | LOCATION_PROVINCE | LOCATION_COUNTRY,
				0, /* show graph */
				NULL, /* graph function */
			{ /* fields */
				{ _("Hits"), "hits" },
				{ _("Cities"), NULL },
				{ _("Province"), NULL },
				{ _("Country"), NULL },
				{ NULL, NULL }
			}
		},
		
		/* 33 */
		{ M_REPORT_LOCATION_PROVINCES, _("Provinces"),
			/* options */
			INDEX | PERCENT |  LOCATION_PROVINCE | LOCATION_COUNTRY,
				0, /* show graph */
				NULL, /* graph function */
			{ /* fields */
				{ _("Hits"), "hits" },
				{ _("Province"), NULL },
				{ _("Country"), NULL },
				{ NULL, NULL }
			}
		},
		
		/* 34 */
		{ M_REPORT_LOCATION_COUNTRIES, _("Countries"),
			/* options */
			INDEX | PERCENT | LOCATION_COUNTRY,
				0, /* show graph */
				NULL, /* graph function */
			{ /* fields */
				{ _("Hits"), "hits" },
				{ _("Provider"), NULL },
				{ NULL, NULL }
			}
		},
		
		/* 35 */
		{ M_REPORT_LOCATION_PROVIDER, _("Provider"),
			/* options */
			INDEX | PERCENT | LOCATION_COUNTRY | LOCATION_PROVIDER,
				0, /* show graph */
				NULL, /* graph function */
			{ /* fields */
				{ _("Hits"), "hits" },
				{ _("Country"), NULL },
				{ _("Provider"), NULL },
				{ NULL, NULL }
			}
		},

		{0, NULL, 0, 0, NULL,
			{
				{ NULL, NULL },
				{ NULL, NULL },
				{ NULL, NULL },
				{ NULL, NULL },
				{ NULL, NULL },
				{ NULL, NULL }
			}
		}
	};

	if (conf->web_reports == NULL) {
		conf->web_reports = malloc(sizeof(reports));
		memcpy(conf->web_reports, reports, sizeof(reports));
	}

	return conf->web_reports;
}

void generate_web_cleanup(mconfig *ext_conf) {
	config_output *conf = ext_conf->plugin_conf;
	
	if (conf->web_reports) free(conf->web_reports);
	conf->web_reports = NULL;
}

char * generate_web_visit_path(mconfig * ext_conf, mstate * state, const char *current, int max) {
	config_output *conf = ext_conf->plugin_conf;
	tmpl_main *tmpl;
	char *s, buf[255], *fn;
	mstate_web *staweb = NULL;

	if (state == NULL)
		return NULL;

	if (state->ext == NULL)
		return NULL;

	if (state->ext_type != M_STATE_TYPE_WEB)
		return NULL;

	staweb = state->ext;

	tmpl = tmpl_init();
	assert(tmpl);

	if ((fn = generate_template_filename(ext_conf, M_TMPL_TABLE)) == NULL) {
		fprintf(stderr, "generating filename failed for '%s'\n", current);
		tmpl_free(tmpl);
		return NULL;
	}

	if (tmpl_load_template(tmpl, fn) != 0) {
		free(fn);
		fprintf(stderr, "parsing template failed for '%s'\n", current);
		tmpl_free(tmpl);
		return NULL;
	}
	free(fn);

	render_cell(ext_conf, tmpl, "#", CT_LEFTHEADER, CA_LEFT );
	render_cell(ext_conf, tmpl, _("Visits"), CT_HEADER, CA_LEFT );
	render_cell(ext_conf, tmpl, "%", CT_HEADER, CA_LEFT );
	render_cell(ext_conf, tmpl, _("Visit Path"), CT_RIGHTHEADER, CA_LEFT );
	parse_table_row(tmpl);

	show_visit_path(ext_conf, state, tmpl, staweb->visits,
			max,
			HIGHLIGHT | GROUPING | INDEX | PERCENT);

	render_cell(ext_conf, tmpl, "#", CT_LEFTFOOTER, CA_LEFT );
	render_cell(ext_conf, tmpl, _("Visits"), CT_FOOTER, CA_LEFT );
	render_cell(ext_conf, tmpl, "%", CT_FOOTER, CA_LEFT );
	render_cell(ext_conf, tmpl, _("Visit Path"), CT_RIGHTFOOTER, CA_LEFT );
	parse_table_row(tmpl);

	snprintf(buf, 255,  "%d", 4);
	tmpl_set_var(tmpl, TABLE_TITLE, _("Visit Path"));
	tmpl_set_var(tmpl, TABLE_COL_SPAN, buf);

	if (tmpl_replace(tmpl, conf->tmp_buf)) {
		tmpl_free(tmpl);
		
		return NULL;
	}

	tmpl_free(tmpl);
	
	s = strdup(conf->tmp_buf->ptr);

	return s;
}

char * generate_web_status_codes(mconfig * ext_conf, mstate * state, const char *current, int max) {
	config_output *conf = ext_conf->plugin_conf;
	tmpl_main *tmpl;
	char *s, *ref, buf[255], *fn;
	mstate_web *staweb = NULL;

	if (state == NULL)
		return NULL;

	if (state->ext == NULL)
		return NULL;

	if (state->ext_type != M_STATE_TYPE_WEB)
		return NULL;

	staweb = state->ext;


	tmpl = tmpl_init();
	assert(tmpl);

	if ((fn = generate_template_filename(ext_conf, M_TMPL_TABLE)) == NULL) {
		fprintf(stderr, "generating filename failed for '%s'\n", current);
		tmpl_free(tmpl);
		return NULL;
	}

	if (tmpl_load_template(tmpl, fn) != 0) {
		free(fn);
		fprintf(stderr, "parsing template failed for '%s'\n", current);
		tmpl_free(tmpl);
		return NULL;
	}
	free(fn);

	ref = create_pic_status(ext_conf, state);

	if (ref && strlen(ref)) {
		tmpl_set_var(tmpl, "IMAGE", ref);
	}

	render_cell(ext_conf, tmpl, _("Hits"), CT_LEFTHEADER, CA_LEFT );
	render_cell(ext_conf, tmpl, _("Status Code"), CT_RIGHTHEADER, CA_LEFT );
	parse_table_row(tmpl);

	show_status_mhash(ext_conf, tmpl, staweb->status_hash, max);

	render_cell(ext_conf, tmpl, _("Hits"), CT_LEFTFOOTER, CA_LEFT );
	render_cell(ext_conf, tmpl, _("Status Code"), CT_RIGHTFOOTER, CA_LEFT );
	parse_table_row(tmpl);

	snprintf(buf, 255,  "%d", 2);
	tmpl_set_var(tmpl, TABLE_TITLE, _("Status Code"));
	tmpl_set_var(tmpl, TABLE_COL_SPAN, buf);
	
	if (tmpl_replace(tmpl, conf->tmp_buf)) {
		tmpl_free(tmpl);
		
		return NULL;
	}

	tmpl_free(tmpl);
	
	s = strdup(conf->tmp_buf->ptr);

	return s;

}

/* this is subject to change - replace with own template */
void generate_web_summary_line1(mconfig * ext_conf, tmpl_main *tmpl, char* title, char* value) {
	config_output *conf = ext_conf->plugin_conf;

	render_cell(ext_conf, tmpl, title, CT_LEFTHLINE, CA_LEFT );
	render_cell(ext_conf, tmpl, "&nbsp;", CT_LINE, CA_LEFT );
	tmpl_set_current_block(tmpl, TABLE_CELL);
	tmpl_set_var(tmpl, CELL_CLASS, conf->cell_class_rightline);
	tmpl_set_var(tmpl, CELL_TAGS, conf->cell_tags_rightline);
	tmpl_set_var(tmpl, CELL_CONTENT, value);
	tmpl_set_var(tmpl, CELL_ALIGN, CELL_ALIGN_RIGHT);
	tmpl_parse_current_block(tmpl);
	parse_table_row(tmpl);
}

/* this is subject to change */
void generate_web_summary_line2(mconfig * ext_conf, tmpl_main *tmpl, char* title, char* value1, char* value2) {

	render_cell(ext_conf, tmpl, title, CT_LEFTHLINE, CA_LEFT );
	render_cell(ext_conf, tmpl, value1, CT_LINE, CA_RIGHT );
	render_cell(ext_conf, tmpl, value2, CT_RIGHTLINE, CA_RIGHT );
	parse_table_row(tmpl);
}

char * generate_web_summary(mconfig * ext_conf, mstate * state, const char *current, int max) {
	config_output *conf = ext_conf->plugin_conf;
	int i;
	int last_day = 1;
	tmpl_main *tmpl;
	char *s;
	char buf[255];
	char buf2[255];
	char *fn;
	mstate_web *staweb = NULL;
	data_WebHistory sumdat;
	data_WebHistory maxdat;
	unsigned int s_200;
	unsigned int s_304;
	unsigned int min;
	unsigned int sec;
	double d = 0;

	if (state == NULL)
		return NULL;

	if (state->ext == NULL)
		return NULL;

	if (state->ext_type != M_STATE_TYPE_WEB)
		return NULL;

	staweb = state->ext;

	tmpl = tmpl_init();
	assert(tmpl);

	if ((fn = generate_template_filename(ext_conf, M_TMPL_TABLE)) == NULL) {
		fprintf(stderr, "generating filename failed for '%s'\n", current);
		tmpl_free(tmpl);
		return NULL;
	}

	if (tmpl_load_template(tmpl, fn) != 0) {
		free(fn);
		fprintf(stderr, "parsing template failed for '%s'\n", current);
		tmpl_free(tmpl);
		return NULL;
	}
	free(fn);

	sumdat.files	= maxdat.files		= 0;
	sumdat.xfersize	= maxdat.xfersize	= 0;
	sumdat.hits	= maxdat.hits		= 0;
	sumdat.hosts	= maxdat.hosts		= 0;
	sumdat.pages	= maxdat.pages		= 0;
	sumdat.visits	= maxdat.visits		= 0;

	/* count the values */
	for ( i = 0; i < 31; i++) {
		if (staweb->days[i].hits) last_day = i+1;
		sumdat.files	+= staweb->days[i].files;
		sumdat.xfersize	+= staweb->days[i].xfersize;
		sumdat.hits	+= staweb->days[i].hits;
		sumdat.hosts	+= staweb->days[i].hosts;
		sumdat.pages	+= staweb->days[i].pages;
		sumdat.visits	+= staweb->days[i].visits;

		if (maxdat.files < staweb->days[i].files)
			maxdat.files	= staweb->days[i].files;
		if (maxdat.hits < staweb->days[i].hits)
			maxdat.hits	= staweb->days[i].hits;
		if (maxdat.hosts < staweb->days[i].hosts)
			maxdat.hosts	= staweb->days[i].hosts;
		if (maxdat.pages < staweb->days[i].pages)
			maxdat.pages	= staweb->days[i].pages;
		if (maxdat.visits < staweb->days[i].visits)
			maxdat.visits	= staweb->days[i].visits;
		if (maxdat.xfersize < staweb->days[i].xfersize)
			maxdat.xfersize	= staweb->days[i].xfersize;
	}

	maxdat.hosts = sumdat.hosts = mhash_count(staweb->host_hash);


	/* Totals */
	snprintf(buf, 255,  "%ld", sumdat.hits);
	generate_web_summary_line1(ext_conf, tmpl, _("Total Hits"), buf);

	snprintf(buf, 255,  "%ld", sumdat.files);
	generate_web_summary_line1(ext_conf, tmpl, _("Total Files"), buf);

	snprintf(buf, 255,  "%ld", sumdat.pages);
	generate_web_summary_line1(ext_conf, tmpl, _("Total Pages"), buf);

	snprintf(buf, 255,  "%ld", sumdat.hosts);
	generate_web_summary_line1(ext_conf, tmpl, _("Total Hosts"), buf);

	snprintf(buf, 255,  "%ld", sumdat.visits);
	generate_web_summary_line1(ext_conf, tmpl, _("Total Visits"), buf);

	generate_web_summary_line1(ext_conf, tmpl, _("Traffic"), (char *)bytes_to_string(sumdat.xfersize));

	/* averages and maximums */
	tmpl_clear_var(tmpl, CELL_ALIGN);

	render_cell(ext_conf, tmpl, "&nbsp;", CT_LEFTHEADER, CA_LEFT );
	render_cell(ext_conf, tmpl, _("avg"), CT_HEADER, CA_LEFT );
	render_cell(ext_conf, tmpl, _("max"), CT_RIGHTHEADER, CA_LEFT );
	parse_table_row(tmpl);

	snprintf(buf, 255,  "%ld", sumdat.hits / last_day);
	snprintf(buf2, 255, "%ld", maxdat.hits);
	generate_web_summary_line2(ext_conf, tmpl, _("Hits per Day"), buf, buf2);

	snprintf(buf, 255,  "%ld", sumdat.files / last_day);
	snprintf(buf2, 255, "%ld", maxdat.files);
	generate_web_summary_line2(ext_conf, tmpl, _("Files per Day"), buf, buf2);

	snprintf(buf, 255,  "%ld", sumdat.pages / last_day);
	snprintf(buf2, 255, "%ld", maxdat.pages);
	generate_web_summary_line2(ext_conf, tmpl, _("Pages per Day"), buf, buf2);

	snprintf(buf, 255,  "%ld", sumdat.hosts / last_day);
	generate_web_summary_line2(ext_conf, tmpl, _("Hosts per Day"), buf, "---");

	snprintf(buf, 255,  "%ld", sumdat.visits / last_day);
	snprintf(buf2, 255, "%ld", maxdat.visits);
	generate_web_summary_line2(ext_conf, tmpl, _("Visits per Day"), buf, buf2);

	generate_web_summary_line2(ext_conf, tmpl, _("Traffic per Day"),
		(char *)bytes_to_string(sumdat.xfersize / last_day),
		(char *)bytes_to_string(maxdat.xfersize) );

	if (sumdat.visits) {
		double allvisitduration = get_visit_full_duration(staweb->visits);

		d = (allvisitduration / sumdat.visits);
		min = d / 60;
		sec = (int)floor(d) % 60;
	} else {
		min = 0;
		sec = 0;
	}
	snprintf(buf, 255,  "%d:%02d %s", min, sec, _("min"));
	generate_web_summary_line2(ext_conf, tmpl, _("Time per visit"), buf, "---");

	if (sumdat.visits) {
		double allvisitlength = get_visit_full_path_length(staweb->visits);
		d = (double)allvisitlength / sumdat.visits;
	} else {
		d = 0;
	}
	snprintf(buf, 255,  "%.2f", d );
	generate_web_summary_line2(ext_conf, tmpl, _("Pages per visit"), buf, "---");

	if (sumdat.pages && staweb->views) {
		double allviewduration = mhash_sumup(staweb->views);
		d = (double)allviewduration / sumdat.pages;
	} else {
		d = 0;
	}
	generate_web_summary_line2(ext_conf, tmpl, _("Pages per visit"), (char *)seconds_to_string(d, 1), "---");

	s_200 = mhash_get_value(staweb->status_hash, "200");
	s_304 = mhash_get_value(staweb->status_hash, "304");

	d = ((double)s_304/(s_200+s_304)) * 100;

	snprintf(buf, 255,  "%.2f%%", d);
	generate_web_summary_line2(ext_conf, tmpl, _("Cache Hit ratio"), buf, "---");

	tmpl_set_var(tmpl, TABLE_TITLE, _("Summary"));
	tmpl_set_var(tmpl, TABLE_COL_SPAN, "3");
	
	if (tmpl_replace(tmpl, conf->tmp_buf)) {
		tmpl_free(tmpl);
		
		return NULL;
	}

	tmpl_free(tmpl);
	
	s = strdup(conf->tmp_buf->ptr);

	return s;

}

char * generate_web_daily(mconfig * ext_conf, mstate * state, const char *current, int max) {
	config_output *conf = ext_conf->plugin_conf;
	int i;
	int last_day = 1;
	tmpl_main *tmpl;
	char *s;
	char *ref;
	char buf[255];
	char *fn;
	mstate_web *staweb = NULL;

	if (state == NULL)
		return NULL;

	if (state->ext == NULL)
		return NULL;

	if (state->ext_type != M_STATE_TYPE_WEB)
		return NULL;

	staweb = state->ext;

	for ( i = 0; i < 31; i++)
		if (staweb->days[i].hits) last_day = i+1;

	tmpl = tmpl_init();
	assert(tmpl);

	if ((fn = generate_template_filename(ext_conf, M_TMPL_TABLE)) == NULL) {
		fprintf(stderr, "generating filename failed for '%s'\n", current);
		tmpl_free(tmpl);
		return NULL;
	}

	if (tmpl_load_template(tmpl, fn) != 0) {
		free(fn);
		fprintf(stderr, "parsing template failed for '%s'\n", current);
		tmpl_free(tmpl);
		return NULL;
	}
	free(fn);

	ref = create_pic_31_day(ext_conf, state);

	if (ref && strlen(ref)) {
		tmpl_set_var(tmpl, "IMAGE", ref);
	}

	/* header */
	render_cell(ext_conf, tmpl, _("Day"), CT_LEFTHEADER, CA_LEFT );
	render_cell(ext_conf, tmpl, _("Hits"), CT_HEADER, CA_LEFT );
	render_cell(ext_conf, tmpl, _("Files"), CT_HEADER, CA_LEFT );
	render_cell(ext_conf, tmpl, _("Pages"), CT_HEADER, CA_LEFT );
	render_cell(ext_conf, tmpl, _("Visits"), CT_HEADER, CA_LEFT );
	render_cell(ext_conf, tmpl, _("KBytes"), CT_RIGHTHEADER, CA_LEFT );
	parse_table_row(tmpl);

	for ( i = 0; i < last_day; i++) {

		snprintf(buf, 255,  "%d", i+1);
		render_cell(ext_conf, tmpl, buf, CT_LEFTLINE, CA_LEFT );

		snprintf(buf, 255,  "%ld", staweb->days[i].hits);
		render_cell(ext_conf, tmpl, buf, CT_LINE, CA_RIGHT );

		snprintf(buf, 255,  "%ld", staweb->days[i].files);
		render_cell(ext_conf, tmpl, buf, CT_LINE, CA_RIGHT );

		snprintf(buf, 255,  "%ld", staweb->days[i].pages);
		render_cell(ext_conf, tmpl, buf, CT_LINE, CA_RIGHT );

		snprintf(buf, 255,  "%ld", staweb->days[i].visits);
		render_cell(ext_conf, tmpl, buf, CT_LINE, CA_RIGHT );

		render_cell(ext_conf, tmpl, (char *)bytes_to_string(staweb->days[i].xfersize), CT_RIGHTLINE, CA_RIGHT );

		parse_table_row(tmpl);
	}

	/* footer */
	render_cell(ext_conf, tmpl, _("Day"), CT_LEFTFOOTER, CA_LEFT );
	render_cell(ext_conf, tmpl, _("Hits"), CT_FOOTER, CA_LEFT );
	render_cell(ext_conf, tmpl, _("Files"), CT_FOOTER, CA_LEFT );
	render_cell(ext_conf, tmpl, _("Pages"), CT_FOOTER, CA_LEFT );
	render_cell(ext_conf, tmpl, _("Visits"), CT_FOOTER, CA_LEFT );
	render_cell(ext_conf, tmpl, _("KBytes"), CT_RIGHTFOOTER, CA_LEFT );
	parse_table_row(tmpl);

	snprintf(buf, 255,  "%d", 6);
	tmpl_set_var(tmpl, TABLE_TITLE, _("Daily Statistics"));
	tmpl_set_var(tmpl, TABLE_COL_SPAN, buf);
	
	if (tmpl_replace(tmpl, conf->tmp_buf)) {
		tmpl_free(tmpl);
		
		return NULL;
	}

	tmpl_free(tmpl);
	
	s = strdup(conf->tmp_buf->ptr);


	return s;
}

char * generate_web_hourly(mconfig * ext_conf, mstate * state, const char *current, int max) {
	config_output *conf = ext_conf->plugin_conf;
	int i;
	tmpl_main *tmpl;
	char *s, *ref, buf[255], *fn;
	mstate_web *staweb = NULL;

	if (state == NULL)
		return NULL;

	if (state->ext == NULL)
		return NULL;

	if (state->ext_type != M_STATE_TYPE_WEB)
		return NULL;

	staweb = state->ext;

	tmpl = tmpl_init();
	assert(tmpl);

	if ((fn = generate_template_filename(ext_conf, M_TMPL_TABLE)) == NULL) {
		fprintf(stderr, "generating filename failed for '%s'\n", current);
		tmpl_free(tmpl);
		return NULL;
	}

	if (tmpl_load_template(tmpl, fn) != 0) {
		free(fn);
		fprintf(stderr, "parsing template failed for '%s'\n", current);
		tmpl_free(tmpl);
		return NULL;
	}
	free(fn);

	ref = create_pic_24_hour(ext_conf, state);

	if (ref && strlen(ref)) {
		tmpl_set_var(tmpl, "IMAGE", ref);
	}

	/* header */
	render_cell(ext_conf, tmpl, _("Hour"), CT_LEFTHEADER, CA_LEFT );
	render_cell(ext_conf, tmpl, _("Hits"), CT_HEADER, CA_LEFT );
	render_cell(ext_conf, tmpl, _("Files"), CT_HEADER, CA_LEFT );
	render_cell(ext_conf, tmpl, _("Pages"), CT_HEADER, CA_LEFT );
	render_cell(ext_conf, tmpl, _("Visits"), CT_HEADER, CA_LEFT );
	render_cell(ext_conf, tmpl, _("KBytes"), CT_RIGHTHEADER, CA_LEFT );
	parse_table_row(tmpl);

	for ( i = 0; i < 24; i++) {

		snprintf(buf, 255,  "%d", i);
		render_cell(ext_conf, tmpl, buf, CT_LEFTLINE, CA_RIGHT );

		snprintf(buf, 255,  "%ld", staweb->hours[i].hits);
		render_cell(ext_conf, tmpl, buf, CT_LINE, CA_RIGHT );

		snprintf(buf, 255,  "%ld", staweb->hours[i].files);
		render_cell(ext_conf, tmpl, buf, CT_LINE, CA_RIGHT );

		snprintf(buf, 255,  "%ld", staweb->hours[i].pages);
		render_cell(ext_conf, tmpl, buf, CT_LINE, CA_RIGHT );

		snprintf(buf, 255,  "%ld", staweb->hours[i].visits);
		render_cell(ext_conf, tmpl, buf, CT_LINE, CA_RIGHT );

		render_cell(ext_conf, tmpl, (char *)bytes_to_string(staweb->hours[i].xfersize), CT_RIGHTLINE, CA_RIGHT );

		parse_table_row(tmpl);
	}

	/* footer */
	render_cell(ext_conf, tmpl, _("Hour"), CT_LEFTFOOTER, CA_LEFT );
	render_cell(ext_conf, tmpl, _("Hits"), CT_FOOTER, CA_LEFT );
	render_cell(ext_conf, tmpl, _("Files"), CT_FOOTER, CA_LEFT );
	render_cell(ext_conf, tmpl, _("Pages"), CT_FOOTER, CA_LEFT );
	render_cell(ext_conf, tmpl, _("Visits"), CT_FOOTER, CA_LEFT );
	render_cell(ext_conf, tmpl, _("KBytes"), CT_RIGHTFOOTER, CA_LEFT );
	parse_table_row(tmpl);

	snprintf(buf, 255,  "%d", 6);
	tmpl_set_var(tmpl, TABLE_TITLE, _("Hourly Statistics"));
	tmpl_set_var(tmpl, TABLE_COL_SPAN, buf);
	
	if (tmpl_replace(tmpl, conf->tmp_buf)) {
		tmpl_free(tmpl);
		
		return NULL;
	}

	tmpl_free(tmpl);
	
	s = strdup(conf->tmp_buf->ptr);

	return s;
}


char * generate_web(mconfig * ext_conf, mstate * state, const char *current, int max) {
	config_output *conf = ext_conf->plugin_conf;
	int i;
	mstate_web *staweb = NULL;
	int w = 0, j;
	char *s = NULL;
	char buf[255];
	tmpl_main *tmpl;
	const reports_def *reports;
	char *fn;
	mhash *data;
	cell_type line_type = CT_LINE; /* dummy, better a wrong value as an unknown value */
#ifdef MTIMER_ENABLED
	static mtimer timer;

	MTIMER_START(timer);
#endif
	if (state == NULL) {
		M_DEBUG0(ext_conf->debug_level, M_DEBUG_SECTION_PARSING, M_DEBUG_LEVEL_ERRORS,
			"state = NULL\n");
		return NULL;
	}

	if (state->ext == NULL) {
		M_DEBUG3(ext_conf->debug_level, M_DEBUG_SECTION_PARSING, M_DEBUG_LEVEL_ERRORS,
			"state->ext = NULL, (%d, %d, %d)\n",
			 state->year,
			 state->month,
			 state->ext_type
			 );
		return NULL;
	}

	if (state->ext_type != M_STATE_TYPE_WEB) {
		M_DEBUG0(ext_conf->debug_level, M_DEBUG_SECTION_PARSING, M_DEBUG_LEVEL_ERRORS,
			"state extension != web\n");
		return NULL;
	}

	staweb = state->ext;

	reports = get_reports_web(ext_conf);

	for (i = 0; reports[i].key; i++) {
		if (0 == strcmp(reports[i].key, current))
			break;
	}

	/* unknown report */
	if (reports[i].key == NULL) {
		M_DEBUG1(ext_conf->debug_level, M_DEBUG_SECTION_PARSING, M_DEBUG_LEVEL_ERRORS,
			 "report '%s' no found here\n", current);
		return NULL;
	}
	
	switch(i) {
	case 0: data = staweb->req_url_hash; break;
	case 1: data = staweb->ref_url_hash; break;
	case 2: data = staweb->views; break;
	case 3: data = staweb->os_hash; break;
	case 4: data = staweb->host_hash; break;
	case 5: data = get_entry_pages(staweb->visits, state); break; /* free me */
	case 6: data = get_exit_pages(staweb->visits, state); break; /* free me */
	case 7: data = staweb->indexed_pages; break;
	case 8: data = staweb->ua_hash; break;
	case 9: data = staweb->req_prot_hash; break;
	case 10: data = staweb->req_meth_hash; break;
	case 11: data = staweb->robots; break;
	case 12: data = staweb->bookmarks; break;
	case 13: data = staweb->status_missing_file; break;
	case 14: data = staweb->status_internal_error; break;
	case 15: data = staweb->searchstring; break;
	case 16: data = staweb->searchsite; break;
	case 17: data = staweb->extension; break;
	case 18: data = get_visit_duration(staweb->visits, state); break; /* free me */
	case 19: data = get_visit_path_length(staweb->visits); break; /* free me */
	case 20: data = staweb->country_hash; break;
	case 21: data = staweb->vhost_hash; break;
	case 22: data = staweb->req_url_hash; break;
	case 23: data = staweb->extension; break;
	case 24: data = staweb->os_hash; break;
	case 25: data = staweb->host_hash; break;
	case 26: data = staweb->ua_hash; break;
	case 27: data = staweb->country_hash; break;
	case 28: data = staweb->views; break;
	case 29: data = staweb->views; break;
	case 30: data = staweb->users; break;
	case 31: data = staweb->users; break;
	case 32: data = get_location_subset(staweb->location, LOCATION_CITY | LOCATION_PROVINCE | LOCATION_COUNTRY); break;
	case 33: data = get_location_subset(staweb->location, LOCATION_PROVINCE | LOCATION_COUNTRY); break; /* free me */
	case 34: data = get_location_subset(staweb->location, LOCATION_COUNTRY); break; /* free me */
	case 35: data = get_location_subset(staweb->location, LOCATION_COUNTRY | LOCATION_PROVIDER); break; /* free me */
	default:
		M_DEBUG1(ext_conf->debug_level, M_DEBUG_SECTION_PARSING, M_DEBUG_LEVEL_ERRORS,
			 "report '%s' no found here - what's up ??\n", current);
		return NULL;
	}
	
	tmpl = tmpl_init();
	assert(tmpl);

	if ((fn = generate_template_filename(ext_conf, M_TMPL_TABLE)) == NULL) {
		M_DEBUG1(ext_conf->debug_level, M_DEBUG_SECTION_PARSING, M_DEBUG_LEVEL_ERRORS,
			 "generating filename failed for '%s'\n", current);
		tmpl_free(tmpl);
		return NULL;
	}

	if (tmpl_load_template(tmpl, fn) != 0) {
		free(fn);
		M_DEBUG1(ext_conf->debug_level, M_DEBUG_SECTION_PARSING, M_DEBUG_LEVEL_ERRORS,
			 "parsing template failed for '%s'\n", current);
		tmpl_free(tmpl);
		return NULL;
	}
	free(fn);

	/* only create the report if we have data */
	if (mhash_count(data)) {
		if (reports[i].show_graph && reports[i].draw_graph) {
			char *ref = reports[i].draw_graph(ext_conf, state);

			if (ref && strlen(ref)) {
				tmpl_set_var(tmpl, "IMAGE", ref);
			}
		}

		/* set the linetype to left */
		line_type = CT_LEFTHEADER;

		/* calculate the number coloumns */
		for (w = 0; reports[i].fields[w].name != NULL; w++) {};

		if (reports[i].options & INDEX) w++;
		if (reports[i].options & PERCENT) w++;

		if (reports[i].options & INDEX) {
			render_cell(ext_conf, tmpl, "#", line_type, CA_LEFT );
			line_type = CT_HEADER;
		}

		/* Generate the HEADER of the Report
		   -----------------------------------------------------------------------
		*/
		for (j = 0; reports[i].fields[j].name != NULL; j++) {
			if( reports[i].fields[j+1].name == NULL )
			{
				line_type = CT_RIGHTHEADER;
			}

			/* FIXME! header classes are still missing */
			if (reports[i].fields[j].class) {
				render_cell(ext_conf, tmpl, reports[i].fields[j].name, line_type, CA_CENTER);
				/* tmpl_set_var(tmpl, CELL_CONTENT, reports[i].fields[j].name); */
				/* tmpl_set_var(tmpl, "TABLE_ROW_CLASS", reports[i].fields[j].class); */
			} else {
				render_cell(ext_conf, tmpl, reports[i].fields[j].name, line_type, CA_CENTER);
				/* tmpl_set_var(tmpl, CELL_CONTENT, reports[i].fields[j].name); */
				/* tmpl_set_var(tmpl, "TABLE_ROW_CLASS", "none"); */
			}

			/* after the first rendered cell, set the type to middle */
			if( line_type == CT_LEFTHEADER ) line_type = CT_HEADER;

			if (j == 0 && reports[i].options & PERCENT) {
				render_cell(ext_conf, tmpl, "%", line_type, CA_CENTER);
			}
		}
		parse_table_row(tmpl);

#if 0
		fprintf(stderr, "%s.%d: %d (%s - %d)\n", __FILE__, __LINE__, i, reports[i].title, max);
#endif

		/* Generate the MAINPART of the Report
		   -----------------------------------------------------------------------
		*/
		if (show_mhash_web(ext_conf, state, tmpl, data, max, reports[i].options)) {
			fprintf(stderr, "show mhash web failed for '%s'\n", current);
		}

		/* Generate the FOOTER of the Report
		   -----------------------------------------------------------------------
		*/
		if (max > BOTTOM_THRESHOLD) {

			line_type = CT_LEFTFOOTER;

			if (reports[i].options & INDEX) {
				render_cell(ext_conf, tmpl, "#", line_type, CA_CENTER);
				line_type = CT_FOOTER;
			}

			for (j = 0; reports[i].fields[j].name != NULL; j++) {

				if( reports[i].fields[j+1].name == NULL )
				{
					line_type = CT_RIGHTFOOTER;
				}

				/* FIXME! header classes are atill missing */
				if (reports[i].fields[j].class) {
					render_cell(ext_conf, tmpl, reports[i].fields[j].name, line_type, CA_CENTER);
					/* tmpl_set_var(tmpl, CELL_CONTENT, reports[i].fields[j].name); */
					/* tmpl_set_var(tmpl, "TABLE_ROW_CLASS", reports[i].fields[j].class); */
				} else {
					render_cell(ext_conf, tmpl, reports[i].fields[j].name, line_type, CA_CENTER);
					/* tmpl_set_var(tmpl, CELL_CONTENT, reports[i].fields[j].name); */
					/* tmpl_set_var(tmpl, "TABLE_ROW_CLASS", "none"); */
				}

				/* after the first rendered cell, set the type to middle */
				if( line_type == CT_LEFTFOOTER ) line_type = CT_FOOTER;

				if (j == 0 && reports[i].options & PERCENT) {
					render_cell(ext_conf, tmpl, "%", line_type, CA_CENTER);
				}
			}
			parse_table_row(tmpl);
		}
	} else {
		/* generate a dummy page if we have no data to process */
		w = 1;

		render_cell(ext_conf, tmpl, _("Sorry, no data to display"), CT_LEFTLINE, CA_CENTER);
		parse_table_row(tmpl);
	}

	snprintf(buf, 255,  "%d", w);
	tmpl_set_var(tmpl, TABLE_TITLE, reports[i].title);
	tmpl_set_var(tmpl, TABLE_COL_SPAN, buf);
	
	if (0 == tmpl_replace(tmpl, conf->tmp_buf)) {
		s = strdup(conf->tmp_buf->ptr);
	} else {
		s = NULL;
	}

	/* cleanup */

	/* - data */
	switch(i) {
	case 5:
	case 6:
	case 18:
	case 19: 
	case 32:
	case 33:
	case 34:
	case 35:
		mhash_free(data);
	}

	/* - the template */
	tmpl_free(tmpl);

#ifdef MTIMER_ENABLED
	MTIMER_STOP(timer);
	MTIMER_CALC(timer);

	fprintf(stderr, "timer %s: %ld msec\n",
		__FUNCTION__, timer.span );
#endif
	return s;
}

int register_reports_web (mconfig *ext_conf, tmpl_reports *r) {
	int i, j;

	const reports_def *reports = get_reports_web(ext_conf);
	const char *bla[] = {
		M_REPORT_DAILY,
			M_REPORT_HOURLY,
			M_REPORT_STATUS_CODES,
			M_REPORT_VISIT_PATH,
			M_REPORT_SUMMARY,
			NULL };

	/* set the 'registered-reports-pointer' to the first unused element */
	for (i = 0; i < M_TMPL_MAX_REPORTS && r[i].key != NULL; i++);

	/* copy the reports */
	for (j = 0; reports[j].key && i < M_TMPL_MAX_REPORTS; j++, i++) {
		r[i].key = reports[j].key;
		r[i].func = generate_web;
		r[i].title = reports[j].title;
	}

	if (i < M_TMPL_MAX_REPORTS) {
		r[i].key = bla[0];
		r[i].func = generate_web_daily;
		r[i].title = _("Daily Statistics");
	}

	if (++i < M_TMPL_MAX_REPORTS) {
		r[i].key = bla[1];
		r[i].func = generate_web_hourly;
		r[i].title = _("Hourly Statistics");
	}

	if (++i < M_TMPL_MAX_REPORTS) {
		r[i].key = bla[2];
		r[i].func = generate_web_status_codes;
		r[i].title = _("Status Codes");
	}

	if (++i < M_TMPL_MAX_REPORTS) {
		r[i].key = bla[3];
		r[i].func = generate_web_visit_path;
		r[i].title = _("Visit Path");
	}

	if (++i < M_TMPL_MAX_REPORTS) {
		r[i].key = bla[4];
		r[i].func = generate_web_summary;
		r[i].title = _("Summary");
	}

	return 0;
}

static int set_line(tmpl_main *tmpl, const char *desc,
	     long hits, long files, long pages, long visits, double traffic,
	     int days_passed
	     ) {

	char buf[255];
	
	tmpl_set_current_block(tmpl, "row");

	tmpl_set_var(tmpl, "DESC", desc);

	snprintf(buf, 255,  "%ld", hits / days_passed);
	tmpl_set_var(tmpl, "AVG_HITS", buf);
	snprintf(buf, 255,  "%ld", pages / days_passed);
	tmpl_set_var(tmpl, "AVG_PAGES", buf);
	snprintf(buf, 255,  "%ld", files / days_passed);
	tmpl_set_var(tmpl, "AVG_FILES", buf);
	snprintf(buf, 255,  "%ld", visits / days_passed);
	tmpl_set_var(tmpl, "AVG_VISITS", buf);
	tmpl_set_var(tmpl, "AVG_TRAFFIC", (char *)bytes_to_string(traffic / days_passed) );

	snprintf(buf, 255,  "%ld", hits);
	tmpl_set_var(tmpl, "TOT_HITS", buf);
	snprintf(buf, 255,  "%ld", pages);
	tmpl_set_var(tmpl, "TOT_PAGES", buf);
	snprintf(buf, 255,  "%ld", files);
	tmpl_set_var(tmpl, "TOT_FILES", buf);
	snprintf(buf, 255,  "%ld", visits);
	tmpl_set_var(tmpl, "TOT_VISITS", buf);
	tmpl_set_var(tmpl, "TOT_TRAFFIC", (char *)bytes_to_string(traffic) );

	tmpl_parse_current_block(tmpl);

	return 0;
}

int mplugins_output_generate_history_output_web(mconfig *ext_conf, mlist *history, tmpl_main *tmpl) {
	mlist *l = history;
	data_History hist, yearly;
	char  *first_report;
	config_output *conf = ext_conf->plugin_conf;
	char buf[255];

	if (/*conf->show_monthly_graph*/ 1) {
		char * ref = create_pic_X_month(ext_conf, history);

		if (ref && *ref) {
			tmpl_set_var(tmpl, "IMAGE", ref);
		}
	}
	/* header vars */

#define HIST(x) \
	hist.data.web.x = 0;

	HIST(hits);
	HIST(files);
	HIST(pages);
	HIST(visits);
	HIST(xfersize);
#undef HIST
	hist.days_passed = 0;

#define HIST(x) \
	yearly.data.web.x = 0;

	HIST(hits);
	HIST(files);
	HIST(pages);
	HIST(visits);
	HIST(xfersize);
#undef HIST
	yearly.days_passed = 0;
	yearly.year = 0;
	
	/* set the correct link */
	if (conf->menu && conf->menu->data && conf->menu->data->key) {
		first_report = conf->menu->data->key;
	} else {
		first_report = conf->reports->data->key;
	}
	
	/* go to the last element */
	for (; l->next; l = l->next);
	
	/* print the history backwards */
	for (; l; l = l->prev) {
		mdata *data = l->data;

		if (!data) break;

		if (data->data.hist->days_passed != 0) {
			char *lnk;
			if (yearly.year > data->data.hist->year) {
				snprintf(buf, 255,  "%04d",
					yearly.year
					);

				set_line(tmpl,
					 buf,
					 yearly.data.web.hits,
					 yearly.data.web.files,
					 yearly.data.web.pages,
					 yearly.data.web.visits,
					 yearly.data.web.xfersize,
					 yearly.days_passed
					 );
			}

			lnk = generate_output_link(ext_conf,
						   data->data.hist->year,
						   data->data.hist->month,
						   first_report);

			snprintf(buf, 255,  "<a href=\"%s\">%s&nbsp;%04d</a>",
				lnk,
				get_month_string(data->data.hist->month, 1),
				data->data.hist->year
				);
			free(lnk);

			set_line(tmpl,
				 buf,
				 data->data.hist->data.web.hits,
				 data->data.hist->data.web.files,
				 data->data.hist->data.web.pages,
				 data->data.hist->data.web.visits,
				 data->data.hist->data.web.xfersize,
				 data->data.hist->days_passed
				 );

			if (yearly.year > data->data.hist->year) {
				yearly.year = data->data.hist->year;
				yearly.days_passed = data->data.hist->days_passed;

#define HIST(x) \
	yearly.data.web.x = data->data.hist->data.web.x;
				HIST(hits);
				HIST(files);
				HIST(pages);
				HIST(visits);
				HIST(xfersize);
#undef HIST

			} else {
#define HIST(x) \
	yearly.data.web.x += data->data.hist->data.web.x
				yearly.year = data->data.hist->year;
				yearly.days_passed += data->data.hist->days_passed; /* CHECKME */

				HIST(hits);
				HIST(files);
				HIST(pages);
				HIST(visits);
				HIST(xfersize);
#undef HIST
			}
#define HIST(x) \
	hist.data.web.x += data->data.hist->data.web.x

			HIST(hits);
			HIST(files);
			HIST(pages);
			HIST(visits);
			HIST(xfersize);
			hist.days_passed += data->data.hist->days_passed;
#undef HIST
		} else {
			M_DEBUG1(ext_conf->debug_level, M_DEBUG_SECTION_OUTPUT, M_DEBUG_LEVEL_WARNINGS,
				 "count == 0, is this ok ?? splitby for '%s' without an entry ??\n",
				 data->key);
		}
	}
	
	if (yearly.year && yearly.days_passed) {
		snprintf(buf, 255,  "%04d",
			yearly.year
			);

		set_line(tmpl,
			 buf,
			 yearly.data.web.hits,
			 yearly.data.web.files,
			 yearly.data.web.pages,
			 yearly.data.web.visits,
			 yearly.data.web.xfersize,
			 yearly.days_passed
			 );
	}

	if (hist.days_passed) {
		set_line(tmpl,
			 _("totals"),
			 hist.data.web.hits,
			 hist.data.web.files,
			 hist.data.web.pages,
			 hist.data.web.visits,
			 hist.data.web.xfersize,
			 hist.days_passed
			);
	}
	
	return 0;
}
