/*
** Modular Logfile Analyzer
** Copyright 2000 Jan Kneschke <jan@kneschke.de>
**
** Homepage: http://www.modlogan.org
**

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version, and provided that the above
    copyright and permission notice is included with all distributed
    copies of this or derived software.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA

**
** $Id: plugin_config.c,v 1.16 2003/04/18 18:40:42 ostborn Exp $
*/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <time.h>
#include <ctype.h>
#include <errno.h>

#include "mlocale.h"
#include "mplugins.h"
#include "mrecord.h"
#include "mdatatypes.h"
#include "misc.h"

#include "plugin_config.h"

int mplugins_input_realserver_dlinit(mconfig *ext_conf) {
	config_input *conf = NULL;
	const char *errptr;
	int erroffset = 0;

	if (0 != strcmp(ext_conf->version, VERSION)) {
		M_DEBUG2(ext_conf->debug_level, M_DEBUG_SECTION_INIT, M_DEBUG_LEVEL_ERRORS,
			 "version string doesn't match: (mla) %s != (plugin) %s\n", ext_conf->version, VERSION);
		return -1;
	}

	conf = malloc(sizeof(config_input));
	memset(conf, 0, sizeof(config_input));

	conf->match_os = mlist_init();
	conf->match_ua = mlist_init();

	conf->inputfilename = NULL;

	conf->buf = buffer_init();

	conf->match_clf = pcre_compile(
		"^(.*?)" /* client */
		"[ ](\\-)[ ](\\-)"
		"[ ]\\[(.*?)\\]" /* utime */
		"[ ]+\"(.*?)\"" /* request */
		"[ ]([0-9]{1,3})" /* status */
		"[ ]([-0-9]+)" /* sent_bytes */
		"[ ]\\[(.*?)\\]" /* client_info */
		"[ ]\\[(.*?)\\]" /* client_id */
		"(?:" /* Logging Levels 2-5 (StatsMask values) */
		"    (?:[ ]\\[\\])?"
		"    (?:[ ]?\\[(?=Stat1:|UNKNOWN)([^]]+)\\])? " /* stat1 */
		"    (?:[ ]?\\[(?=Stat2:|UNKNOWN)([^]]+)\\])?" /* stat2 */
		"    (?:[ ]?\\[(?=Stat3:|UNKNOWN)([^]]+)\\])?" /* stat3 */
		")?" /* end stats mask */
		"(?:" /* Logging levels 1-5 */
		"    [ ](.*?)" /* file_size */
		"    [ ](.*?)" /* file_time */
		"    [ ](.*?)" /* sent_time */
		"    [ ](.*?)" /* resends */
		"    [ ](.*?)" /* failed_requests */
		""   /* end level 1-2 */
		"    (?:" /* Level (3,4) or 5 */
		"        (?:" /* Logging levels 3-4 */
		"            [ ]\\[(.*?)\\] " /* stream_components */
		"            [ ]\\[(.*?)\\]" /* start_time */
		"            [ ](.*?)" /* server_address */
		"            (?:" /* Logging level 4 */
		"                [ ](.*?)" /* average_bitrate */
		"                [ ](.*?)" /* packets_sent */
		"            )?"
		"        )" /* end 3,4 */
		"        |[ ]([-0-9_]*)" /* presentation_id TODO match presentation ID */
		"    )?"
		")?" /* end 1-5 */
		""
		, PCRE_EXTENDED, &errptr, &erroffset, NULL);

	if (conf->match_clf == NULL) {
		fprintf(stderr, "%s.%d: rexexp compilation error at %s\n", __FILE__, __LINE__, errptr);
		return -1;
	}

	conf->match_clf_extra = pcre_study(conf->match_clf, 0, &errptr);
	if (errptr != NULL) {
		fprintf(stderr, "%s.%d: rexexp studying error at %s\n", __FILE__, __LINE__, errptr);
		return -1;
	}

	if ((conf->match_timestamp = pcre_compile(
		"^([0-9]{2})/([a-zA-Z]{3})/([0-9]{4}):([0-9]{2}):([0-9]{2}):([0-9]{2})",
		0, &errptr, &erroffset, NULL)) == NULL) {

		fprintf(stderr, "%s.%d: rexexp compilation error at %s\n", __FILE__, __LINE__, errptr);
		return -1;
	}

	if ((conf->match_url = pcre_compile(
		"^([A-Za-z]+) ([^? ]+(?:\\?([^ ]*)|))(?: (.*?)|)$",
		0, &errptr, &erroffset, NULL)) == NULL) {

		fprintf(stderr, "%s.%d: rexexp compilation error at %s\n", __FILE__, __LINE__, errptr);
		return -1;
	}

	conf->match_timestamp_extra = pcre_study(conf->match_timestamp, 0, &errptr);
	if (errptr != NULL) {
		fprintf(stderr, "%s.%d: rexexp studying error at %s\n", __FILE__, __LINE__, errptr);
		return -1;
	}
	conf->match_url_extra = pcre_study(conf->match_url, 0, &errptr);
	if (errptr != NULL) {
		fprintf(stderr, "%s.%d: rexexp studying error at %s\n", __FILE__, __LINE__, errptr);
		return -1;
	}

	ext_conf->plugin_conf = conf;

	return 0;
}

int mplugins_input_realserver_dlclose(mconfig *ext_conf) {
	config_input *conf = ext_conf->plugin_conf;

	mclose(&(conf->inputfile));

	mlist_free(conf->match_os);
	mlist_free(conf->match_ua);

	pcre_free(conf->match_clf);

	buffer_free(conf->buf);

	free(ext_conf->plugin_conf);
	ext_conf->plugin_conf = NULL;

	return 0;
}

int mplugins_input_realserver_parse_config(mconfig *ext_conf, const char *filename, const char *section) {
	config_input *conf = ext_conf->plugin_conf;

	const mconfig_values config_values[] = {
		{"inputfile",	M_CONFIG_TYPE_STRING,	M_CONFIG_VALUE_OVERWRITE, &(conf->inputfilename)},

		{NULL, M_CONFIG_TYPE_INT, 0, NULL}
	};

	return mconfig_parse_section(ext_conf, filename, section, config_values);
}


int mplugins_input_realserver_set_defaults(mconfig *ext_conf) {
	config_input *conf = ext_conf->plugin_conf;

	if (conf->inputfilename && strcmp(conf->inputfilename, "-") != 0) {
		if (mopen(&(conf->inputfile), conf->inputfilename)) {
			M_DEBUG2(ext_conf->debug_level, M_DEBUG_SECTION_INIT, M_DEBUG_LEVEL_ERRORS,
				 "%s: %s\n", conf->inputfilename, strerror(errno));
			return -1;
		}
		M_DEBUG1(ext_conf->debug_level, M_DEBUG_SECTION_INIT, M_DEBUG_LEVEL_VERBOSE,
			 "(realserver) using %s as inputfile\n", conf->inputfilename);
	} else {
		/* stdin */
		if (mopen(&(conf->inputfile), NULL)) {
			M_DEBUG2(ext_conf->debug_level, M_DEBUG_SECTION_INIT, M_DEBUG_LEVEL_ERRORS,
				 "%s: %s\n", conf->inputfilename, strerror(errno));
			return -1;
		}
		
		M_DEBUG0(ext_conf->debug_level, M_DEBUG_SECTION_INIT, M_DEBUG_LEVEL_VERBOSE,
			 "(realserver) using (stdin) as inputfile\n");
	}

	return 0;
}

int mplugins_init(mplugin *func) {
	func->dlinit = mplugins_input_realserver_dlinit;
	func->dlclose = mplugins_input_realserver_dlclose;
	func->parse_config = mplugins_input_realserver_parse_config;
	func->set_defaults = mplugins_input_realserver_set_defaults;
	func->get_next_record = mplugins_input_realserver_get_next_record;
	func->insert_record = NULL;
	func->gen_report = NULL;
	func->gen_history = NULL;

	return 0;
}
