/*
** Modular Logfile Analyzer
** Copyright 2000 Jan Kneschke <jan@kneschke.de>
**
** Homepage: http://www.modlogan.org
**

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version, and provided that the above
    copyright and permission notice is included with all distributed
    copies of this or derived software.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA

**
** $Id: generate.c,v 1.60 2003/08/01 01:25:31 ostborn Exp $
*/

#include <libintl.h>
#include <locale.h>
#include <stdlib.h>
#include <stdio.h>
#include <time.h>
#include <math.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <errno.h>
#include <assert.h>

/* ./src/ */

#include "mconfig.h"
#include "mstate.h"
#include "mlocale.h"
#include "mhash.h"
#include "mlist.h"
#include "mdatatypes.h"
#include "mplugins.h"
#include "misc.h"

#include "datatypes/brokenlink/datatype.h"

/* ./src/output/template */

#include "plugin_config.h"
#include "template.h"
#include "generate.h"
#include "web.h"
#include "mail.h"
#include "pictures.h"
#include "mtree.h"

typedef struct {
	char *key;
	char *subst;
} tmpl_vars;

char * generate_output_link (mconfig *ext_conf, int year, int month, const char *current) {
	config_output *conf = ext_conf->plugin_conf;
	tmpl_main *tmpl;
	char date[7], *fn;

	/* huge overkill, but dawm nice code */

	sprintf(date, "%04d%02d",
		year,
		month);

	if (conf->tmpl_output_link == NULL) {
		tmpl = tmpl_init();
		tmpl_load_string(tmpl,  conf->filename_pattern);
		
		conf->tmpl_output_link = tmpl;
	} else {
		tmpl = conf->tmpl_output_link;
	}
	tmpl_set_var(tmpl, "NAME", current);
	tmpl_set_var(tmpl, "DATE", date);

	if (tmpl_replace(tmpl, conf->tmp_buf)) {
		tmpl_free(tmpl);
		
		conf->tmpl_output_link = NULL;
		
		return NULL;
	}

	fn = strdup(conf->tmp_buf->ptr);

	return fn;
}

int generate_output_filename (mconfig *ext_conf, mstate *state, const char *current, buffer *output_filename) {
	char *lnk = generate_output_link(ext_conf, state->year, state->month, current);
	config_output *conf = ext_conf->plugin_conf;
	
	if (!lnk) return -1;

	buffer_strcpy(output_filename, conf->outputdir);
	buffer_strcat_len(output_filename, "/", 1);
	buffer_strcat(output_filename, lnk);
	
	free(lnk);

	return 0;
}

char * generate_template_filename(mconfig *ext_conf, int type) {
	char *fn, *t = NULL;
	config_output *conf = ext_conf->plugin_conf;

	switch(type) {
	case M_TMPL_TABLE:
		t = conf->tmpl_table;
		break;
	case M_TMPL_OUTER:
		t = conf->tmpl_outer;
		break;
	case M_TMPL_MENU:
		t = conf->tmpl_menu;
		break;
	case M_TMPL_INDEX:
		t = conf->tmpl_index;
		break;
	default:
		M_DEBUG1(ext_conf->debug_level, M_DEBUG_SECTION_PARSING, M_DEBUG_LEVEL_ERRORS,
			 "unknown type '%d'\n",
			 type);
		return NULL;
	}

	if (!t || !conf->template_path || !conf->template_name) {
		M_DEBUG4(ext_conf->debug_level, M_DEBUG_SECTION_PARSING, M_DEBUG_LEVEL_ERRORS,
			 "something is NULL: type = %d, t = %p, tmpl-path: %p, tmpl-name: %p\n",
			 type, t, conf->template_path, conf->template_name);
		return NULL;
	}

	fn = malloc(strlen(conf->template_path) +
		    strlen(conf->template_name) +
		    strlen(t) +
		    2 + 1);

	sprintf(fn, "%s/%s/%s",
		conf->template_path,
		conf->template_name,
		t);

	return fn;
}

static int gen_menu_tree(mconfig *ext_conf, mstate *state, tmpl_main *tmpl, const mtree *menu, const char *current, int level) {
	char *parent, *fn;
	int i;
	config_output *conf = ext_conf->plugin_conf;

	if (!menu) return -1;
	if (!menu->data) return -1;
	
	parent = menu->data->key;

	for (i = 0; i < level; i++) {
		tmpl_set_current_block(tmpl, "menurowspacer");
		tmpl_parse_current_block(tmpl);
	}

	tmpl_set_current_block(tmpl, "menuentry");
	tmpl_set_var(tmpl, "MENU_CLASS",
		     (0 == strcmp(parent, current))
		     ? "active"
		     : "menu");

	if (level != 0) {
		fn = generate_output_link(ext_conf, state->year, state->month, parent);
		tmpl_set_var(tmpl, "MENU_URL", fn);
		free(fn);
	} else {
		tmpl_set_var(tmpl, "MENU_URL", conf->index_filename);
	}

	tmpl_set_var(tmpl, "MENU_NAME",
		     menu->data->data.string.string ?
		     menu->data->data.string.string :
		     parent);
	tmpl_parse_current_block(tmpl);
	tmpl_clear_block(tmpl, "menurowspacer");

	if (mtree_is_child(menu, current)) {
		int i;

		for (i = 0; i < menu->num_childs; i++) {
			tmpl_clear_block(tmpl, "menusubstart");
			tmpl_clear_block(tmpl, "menusubend");
			if (i == 0) {
				tmpl_set_current_block(tmpl, "menusubstart");
				tmpl_parse_current_block(tmpl);
			}
			if (i == menu->num_childs - 1) {
				tmpl_set_current_block(tmpl, "menusubend");
				tmpl_parse_current_block(tmpl);
			}
			gen_menu_tree(ext_conf, state, tmpl, menu->childs[i], current, level + 1);
		}
	}
	return 0;
}

static int gen_menu_block(mconfig *ext_conf, mstate *state, tmpl_main *tmpl, const mtree *menu, const char *current, int level) {
	char *parent, *fn;

	if (!menu) return -1;
	if (!menu->data) return -1;
	
	parent = menu->data->key;

	tmpl_clear_block(tmpl,"menutitle");
	tmpl_clear_block(tmpl,"menuentry");

	if (menu->num_childs > 0 ) {
		tmpl_set_current_block(tmpl, "menutitle");
		tmpl_set_var(tmpl, "MENU_TITLE",
			     menu->data->data.string.string ?
			     menu->data->data.string.string : menu->data->key );
		tmpl_parse_current_block(tmpl);
	}

	if (mtree_is_child(menu, current)) {
		int i;

		for (i = 0; i < menu->num_childs; i++) {
			tmpl_set_current_block(tmpl, "menuentry");
			fn = generate_output_link(ext_conf, state->year, state->month, menu->childs[i]->data->key);
			tmpl_set_var(tmpl, "MENU_URL", fn);
			free(fn);

			tmpl_set_var(tmpl, "MENU_NAME",
					menu->childs[i]->data->data.string.string ?
					menu->childs[i]->data->data.string.string : menu->childs[i]->data->key);
			tmpl_parse_current_block(tmpl);
		}

		tmpl_set_current_block(tmpl, "menublock");
		tmpl_parse_current_block(tmpl);

		for (i = 0; i < menu->num_childs; i++) {
			gen_menu_block(ext_conf, state, tmpl, menu->childs[i], current, level + 1);
		}
	}
	return 0;
}

static char * generate_menu (mconfig *ext_conf, mstate *state, const char *current) {
	char *s, *fn;
	config_output *conf = ext_conf->plugin_conf;
	tmpl_main * tmpl;

	tmpl = tmpl_init();
	assert(tmpl);

	if ((fn = generate_template_filename(ext_conf, M_TMPL_MENU)) == NULL) {
		tmpl_free(tmpl);
		M_DEBUG0(ext_conf->debug_level, M_DEBUG_SECTION_PARSING, M_DEBUG_LEVEL_ERRORS,
			 "generating template filename failed for the menu\n");
		return NULL;
	}

	if (tmpl_load_template(tmpl, fn) != 0) {
		M_DEBUG1(ext_conf->debug_level, M_DEBUG_SECTION_PARSING, M_DEBUG_LEVEL_ERRORS,
			 "loading template failed: %s\n", fn);

		free(fn);
		tmpl_free(tmpl);
		return NULL;
	}
	free(fn);

/*	mtree_print(conf->menu);*/

	if( conf->flat_menu ) {
		gen_menu_block(ext_conf, state, tmpl, conf->menu, current, 0);
	} else {
		gen_menu_tree(ext_conf, state, tmpl, conf->menu, current, 0);
	}

	if (tmpl_replace(tmpl, conf->tmp_buf)) {
		tmpl_free(tmpl);
		
		return NULL;
	}
	
	
	tmpl_free(tmpl);
	
	s = strdup(conf->tmp_buf->ptr);

	return s;
}

static char * generate_report (mconfig *ext_conf, mstate *state, tmpl_reports *reports,  const char *current) {
	char *s = NULL;
	int i;

	/* search for a title */
	for (i = 0; reports[i].key; i++) {
		if (0 == strcmp(reports[i].key, current)) {
			/* found */
			break;
		}
	}

	if (reports[i].key == NULL) {
		M_DEBUG0(ext_conf->debug_level, M_DEBUG_SECTION_PARSING, M_DEBUG_LEVEL_ERRORS,
			 "the key of the report is NULL\n");
		return NULL;
	}

	if (reports[i].func != NULL) {
		if ((s = reports[i].func(ext_conf, state, current, 30 /* max-value */)) == NULL) {
			M_DEBUG1(ext_conf->debug_level, M_DEBUG_SECTION_PARSING, M_DEBUG_LEVEL_ERRORS,
				 "no chance for %s\n", current);
		}
	}

	return s;
}

/**
 * unfold the tree to an array
 *
 * @return last used index in the array
 */

static int mtree_to_marray (mtree *t, mdata **d, int ndx) {
	int i;

	if (!t) return ndx;

	d[ndx++] = t->data;

	for (i = 0; i < t->num_childs; i++) {
		ndx = mtree_to_marray(t->childs[i], d, ndx);
	}

	return ndx;
}

static int generate_monthly_output(mconfig *ext_conf, mstate *state) {
#define NUM_TIMER 14
	int ret;
	char *fn;
	config_output *conf = ext_conf->plugin_conf;
	tmpl_main * tmpl;
	mlist *l;
	int i, cnt, j;
	mdata **d;
	mtree *menunode; /* used for TITLE replace */

	char buf[255];
	struct tm *_tm;
	time_t t;

	mtimer timer[NUM_TIMER];

	tmpl_vars vars[] = {
		/* generated by this plugin */
		{ "LASTRECORD", NULL }, /* hard coded to 0 */
		{ "GENERATEDAT", NULL }, /* hard coded to 1 */

		/* hard values */
		{ "MLA_URL", "http://www.modlogan.org" },
		{ "MLA_PACKAGE", PACKAGE },
		{ "MLA_VERSION", VERSION },

		/* translated strings */
		{ "TXT_STATISTICS", _("Statistics for")},
		{ "TXT_LASTRECORD", _("Last record")},
		{ "TXT_GENERATEDAT", _("Generated at")},
		{ "TXT_OUTPUTBY", _("Output generated by")},
		/* { "TITLE", _("Statistics") }, */
		{ "TITLE", NULL },

		/* link handlers */
		{ "LNK_HOME", NULL },
		{ "LNK_UP", NULL }, /* 11 - hard coded !! */
		{ "LNK_PREV", NULL },
		{ "LNK_NEXT", NULL },
		{ "LNK_CURRENT", NULL },

		/* internal placeholders */
		{ "MENU", NULL },
		{ "REPORT", NULL },

		{ NULL,  NULL }
	};

	if (!conf) {
		M_DEBUG0(ext_conf->debug_level, M_DEBUG_SECTION_PARSING, M_DEBUG_LEVEL_ERRORS,
			 "conf is NULL\n");
		return -1;
	}

	vars[11].subst = conf->index_filename;

	for (i = 0; i < NUM_TIMER; i++) {
		MTIMER_RESET(timer[i]);
	}

	MTIMER_START(timer[0]);

	/* init template handle */
	tmpl = tmpl_init();
	assert(tmpl);

	tmpl->debug_level = ext_conf->debug_level;

	if ((fn = generate_template_filename(ext_conf, M_TMPL_OUTER)) == NULL) {
		M_DEBUG0(ext_conf->debug_level, M_DEBUG_SECTION_PARSING, M_DEBUG_LEVEL_ERRORS,
			 "can't generate filename of the template\n");
		tmpl_free(tmpl);

		return -1;
	}

	MTIMER_START(timer[1]);
	/* load and parse template */
	if ((ret = tmpl_load_template(tmpl, fn)) != 0) {
		M_DEBUG1(ext_conf->debug_level, M_DEBUG_SECTION_PARSING, M_DEBUG_LEVEL_ERRORS,
			 "parsing template %s failed\n",
			 fn);
		free(fn);
		tmpl_free(tmpl);

		return ret;
	}
	free(fn);
	MTIMER_STOP(timer[1]);
	MTIMER_CALC(timer[1]);

	MTIMER_START(timer[2]);
	/* set external variables from config-file */
	for (l = conf->variables; l && l->data; l = l->next) {
		char *k, *v;
			
		k = l->data->key;
		
		if (k) {
			if ((v = strchr(k, ',')) != NULL) {
				v++;
				
				/* check if the variable is used internally */
				for (i = 0; vars[i].key; i++) {
					if (0 == strncmp(k, vars[i].key, v-k-1))
						break;
				}
				
				/* not found */
				if (vars[i].key == NULL) {
					char *key = malloc(v-k);
					strncpy(key, k, v-k-1);
					key[v-k-1] = '\0';
					tmpl_set_var(tmpl, key, v);
					free(key);
				}
			} else {
				M_DEBUG1(ext_conf->debug_level, M_DEBUG_SECTION_PARSING, M_DEBUG_LEVEL_ERRORS,
					 "no ',' found in %s\n", k);
			}
		}
	}
	MTIMER_STOP(timer[2]);
	MTIMER_CALC(timer[2]);

	MTIMER_START(timer[3]);

	/* create current timestamp */
	t = time(NULL);
	_tm = localtime(&t);
	strftime(buf, sizeof(buf), "%X %x", _tm);
	tmpl_set_var(tmpl, "GENERATEDAT", buf);

	/* create timestamp of the last record */
	if ((t = state->timestamp)) {
		_tm = localtime(&t);
		strftime(buf, sizeof(buf), "%X %x", _tm);
		tmpl_set_var(tmpl, "LASTRECORD", buf);
	} else {
		tmpl_set_var(tmpl, "LASTRECORD", _("(not set)"));
	}

	tmpl_set_var(tmpl, "LNK_HOME", conf->index_filename);

	if (conf->menu && mtree_num_elements(conf->menu)) {
		cnt = mtree_num_elements(conf->menu) + 1;
		d = malloc(sizeof(mdata *) * cnt);

		mtree_to_marray(conf->menu, d, 0);
	} else {
		cnt = 0;
		d = NULL;
	}

	MTIMER_STOP(timer[3]);
	MTIMER_CALC(timer[3]);

	MTIMER_START(timer[4]);

	/* generate reports */
	for (l = conf->reports; l && l->data; l = l->next) {
		MTIMER_START(timer[7]);
		
		for (i = 0; conf->avail_reports[i].key; i++) {
			if (0 == strcmp(conf->avail_reports[i].key, l->data->key)) {
				break;
			}
		}

		if (conf->avail_reports[i].key == NULL) {
			/* report is unknown */
			M_DEBUG1(ext_conf->debug_level, M_DEBUG_SECTION_PARSING, M_DEBUG_LEVEL_WARNINGS,
					 "report-type '%s' is unknown - skipped\n",
					 l->data->key);
			continue;
		}

		/* set template variables */
		for (i = 0; vars[i].key; i++) {
			if (i != 0 && i != 1 && /* ignore GENERATEDAT and LASTRECORD as they are already set */
			    vars[i].subst != NULL) {
				if (0 != tmpl_set_var(tmpl, vars[i].key, vars[i].subst)) {
#if 0
					M_DEBUG1(ext_conf->debug_level, M_DEBUG_SECTION_PARSING, M_DEBUG_LEVEL_WARNINGS,
						 "looks like {%s} isn't in the template\n",
						 vars[i].key);
#endif
				} else {
					/* everything ok */
				}
			} else if (vars[i].subst == NULL) {
				/* perhaps this variable is handled by a function */
				char *s = NULL;

				if (0 == strcmp("TITLE", vars[i].key)) {
					/* set the current title, from menu */
					/* first search the report in the current menu tree */
					
					if ( (menunode = mtree_search(conf->menu, l->data->key)) ) {
						tmpl_set_var(tmpl, vars[i].key, menunode->data->data.string.string);
					} else {
						/* no menu entry found for this report !? ... whatever */
						tmpl_set_var(tmpl, vars[i].key, "Whatever...");
					}
				} else if (0 == strcmp("MENU", vars[i].key)) {
					MTIMER_START(timer[9]);
					
					if ((s = generate_menu(ext_conf, state, l->data->key)) == NULL) {
						M_DEBUG1(ext_conf->debug_level, M_DEBUG_SECTION_PARSING, M_DEBUG_LEVEL_ERRORS,
							 "generating %s failed\n",
							 vars[i].key);

						return -1;
					} else {
						tmpl_set_var(tmpl, vars[i].key, s);
						free(s);
					}
					MTIMER_STOP(timer[9]);
					MTIMER_CALC(timer[9]);
				} else if (0 == strcmp("REPORT", vars[i].key)) {
					MTIMER_START(timer[10]);
					
					if ((s = generate_report(ext_conf, state, conf->avail_reports, l->data->key)) == NULL) {
						M_DEBUG1(ext_conf->debug_level, M_DEBUG_SECTION_PARSING, M_DEBUG_LEVEL_ERRORS,
							 "generating %s failed\n",
							 vars[i].key);

						return -1;
					} else {
						tmpl_set_var(tmpl, vars[i].key, s);

						free(s);
					}
					MTIMER_STOP(timer[10]);
					MTIMER_CALC(timer[10]);
				} else if (0 == strcmp("LNK_CURRENT", vars[i].key)) {
					char *fn;
					MTIMER_START(timer[11]);
					
					fn = generate_output_link(ext_conf, state->year, state->month, l->data->key);

					tmpl_set_var(tmpl, vars[i].key, fn);
					free(fn);
					MTIMER_STOP(timer[11]);
					MTIMER_CALC(timer[11]);
				} else if (0 == strcmp("LNK_PREV", vars[i].key)) {
					/* locate entry */
					MTIMER_START(timer[12]);
					
					for (j = 0; j < cnt; j++) {
						if (0 == strcmp(l->data->key, d[j]->key))
							break;
					}

					/* found */
					if (j != cnt && j > 0) {
						fn= generate_output_link(ext_conf, state->year, state->month, d[j-1]->key);
						tmpl_set_var(tmpl, vars[i].key, fn);
						free(fn);
					} else {
						tmpl_set_var(tmpl, vars[i].key, conf->index_filename);
					}
					MTIMER_STOP(timer[12]);
					MTIMER_CALC(timer[12]);
				} else if (0 == strcmp("LNK_NEXT", vars[i].key)) {
					/* locate entry */
					MTIMER_START(timer[13]);
					
					for (j = 0; j < cnt; j++) {
						if (0 == strcmp(l->data->key, d[j]->key))
							break;
					}

					/* found */
					if (j != cnt && j < cnt - 1) {
						fn= generate_output_link(ext_conf, state->year, state->month, d[j+1]->key);
						tmpl_set_var(tmpl, vars[i].key, fn);
						free(fn);
					} else {
						tmpl_set_var(tmpl, vars[i].key, conf->index_filename);
					}
					MTIMER_STOP(timer[13]);
					MTIMER_CALC(timer[13]);
				}
			}
		}

		MTIMER_STOP(timer[7]);
		MTIMER_CALC(timer[7]);

		MTIMER_START(timer[8]);

		/* replace tags by the actual output */
		if (0 == tmpl_replace(tmpl, conf->tmp_output)) {
			/* index-file */
			int fd;
			
			if (-1 == generate_output_filename(ext_conf, state, l->data->key, conf->tmpl_output_filename)) {
				M_DEBUG1(ext_conf->debug_level, M_DEBUG_SECTION_PARSING, M_DEBUG_LEVEL_ERRORS,
					 "no filename for output: %s\n",
					 l->data->key);

				return -1;
			}

			if ((fd = open(conf->tmpl_output_filename->ptr, O_WRONLY | O_CREAT | O_TRUNC, 0644)) < 0) {
				M_DEBUG2(ext_conf->debug_level, M_DEBUG_SECTION_PARSING, M_DEBUG_LEVEL_ERRORS,
					 "opening '%s' failed: %s\n",
					 conf->tmpl_output_filename->ptr, strerror(errno));

				return -1;
			}
			
			write(fd, conf->tmp_output->ptr, conf->tmp_output->used - 1);
			
			close(fd);
		}
		MTIMER_STOP(timer[8]);
		MTIMER_CALC(timer[8]);
	}

	MTIMER_STOP(timer[4]);
	MTIMER_CALC(timer[4]);

	/* generate pages which are not related to a menu */

	MTIMER_START(timer[5]);

	for (i = 0; i < cnt; i++) {
		/* check if we have already generated the page */
		for (l = conf->reports; l && l->data; l = l->next) {
			if (0 == strcmp(d[i]->key, l->data->key))
				break;
		}

		if (l) {
			for (j = 0; conf->avail_reports[j].key; j++) {
				if (0 == strcmp(conf->avail_reports[j].key, d[i]->key)) {
					/* found */
					break;
				}
			}

			if (conf->avail_reports[j].key != NULL) {
				/* report is known and is generated */
				continue;
			}
		}

		/* set template variables */
		for (j = 0; vars[j].key; j++) {
			if (j != 0 && j != 1 && /* ignore GENERATEDAT and LASTRECORD as they are already set */
			    vars[j].subst != NULL) {
				if (0 != tmpl_set_var(tmpl, vars[j].key, vars[j].subst)) {
#if 0
					M_DEBUG1(ext_conf->debug_level, M_DEBUG_SECTION_PARSING, M_DEBUG_LEVEL_WARNINGS,
						 "looks like {%s} isn't in the template\n",
						 vars[j].key);
#endif
				} else {
					/* everything ok */
				}
			} else if (vars[j].subst == NULL) {
				/* perhaps this variable is handled by a function */
				char *s = NULL;

				if (0 == strcmp("MENU", vars[j].key)) {
					if ((s = generate_menu(ext_conf, state, d[i]->key)) == NULL) {
						M_DEBUG1(ext_conf->debug_level, M_DEBUG_SECTION_PARSING, M_DEBUG_LEVEL_ERRORS,
							 "generating %s failed\n",
							 vars[j].key);

						return -1;
					} else {
						tmpl_set_var(tmpl, vars[j].key, s);
						free(s);
					}
				} else if (0 == strcmp("REPORT", vars[j].key)) {
					/* no report */
					tmpl_clear_var(tmpl, vars[j].key);
				} else if (0 == strcmp("LNK_CURRENT", vars[j].key)) {
					char *fn;
					
					fn = generate_output_link(ext_conf, state->year, state->month, d[i]->key);

					tmpl_set_var(tmpl, vars[j].key, fn);
					free(fn);
				} else if (0 == strcmp("LNK_PREV", vars[j].key)) {

					if (i > 0) {
						fn = generate_output_link(ext_conf, state->year, state->month, d[i-1]->key);
						tmpl_set_var(tmpl, vars[j].key, fn);
						free(fn);
					} else {
						tmpl_set_var(tmpl, vars[j].key, conf->index_filename);
					}
				} else if (0 == strcmp("LNK_NEXT", vars[j].key)) {
					if (i < cnt - 1) {
						fn = generate_output_link(ext_conf, state->year, state->month, d[i+1]->key);
						tmpl_set_var(tmpl, vars[j].key, fn);
						free(fn);
					} else {
						tmpl_set_var(tmpl, vars[j].key, conf->index_filename);
					}
				}

			}
		}

		/* replace tags by the actual output */
		if (0 == tmpl_replace(tmpl, conf->tmp_output)) {
			/* index-file */
			int fd;
			
			if (-1 == generate_output_filename(ext_conf, state, d[i]->key, conf->tmpl_output_filename))  {
				M_DEBUG1(ext_conf->debug_level, M_DEBUG_SECTION_PARSING, M_DEBUG_LEVEL_ERRORS,
					 "no filename for output: %s\n",
					 d[i]->key);

				return -1;
			}
			
			if ((fd = open(conf->tmpl_output_filename->ptr, O_WRONLY | O_CREAT | O_TRUNC, 0644)) < 0) {
				config_output *conf = ext_conf->plugin_conf;
				char *lnk;
				
				lnk = generate_output_link(ext_conf, state->year, state->month, d[i]->key);
				M_DEBUG2(ext_conf->debug_level, M_DEBUG_SECTION_GENERATING, M_DEBUG_LEVEL_ERRORS,
					"Can't open File \"%s\" for write. error=%s\n", conf->tmpl_output_filename->ptr, strerror(errno));
				M_DEBUG3(ext_conf->debug_level, M_DEBUG_SECTION_GENERATING, M_DEBUG_LEVEL_ERRORS,
					"outputdir=\"%s\", lnk=\"%s\", d[i]->key=\"%s\"\n", conf->outputdir, lnk, d[i]->key);
				free(lnk);
			} else {
				
				write(fd, conf->tmp_output->ptr, conf->tmp_output->used - 1);
				
				close(fd);
			}
		}
	}

	free(d);
	d = NULL;

	MTIMER_STOP(timer[5]);
	MTIMER_CALC(timer[5]);

	MTIMER_START(timer[6]);

	/* copy files */
	for (l = conf->files; l && l->data; l = l->next) {
		int fd1, fd2, n;
		char buf[256];
		char *fn1, *fn2, *p, *f, *dir;
		
		/* create the filename which has to be copied */
		fn1 = malloc(strlen(conf->template_path) + strlen(conf->template_name) + strlen(l->data->key) + 1 + 2);
		sprintf(fn1, "%s/%s/%s",
			conf->template_path,
			conf->template_name,
			l->data->key);


		fn2 = malloc(strlen(conf->outputdir) + strlen(l->data->key) + 1 + 2);
		dir = malloc(strlen(conf->outputdir) + strlen(l->data->key) + 1 + 2);
		sprintf(fn2, "%s/%s",
			conf->outputdir,
			l->data->key);

		if (strlen(l->data->key) > sizeof(buf) - 10) {
			M_DEBUG1(ext_conf->debug_level, M_DEBUG_SECTION_PARSING, M_DEBUG_LEVEL_ERRORS,
				 "filename '%s' is too long\n",
				 l->data->key
				 );

			return -1;
		}

		/* build all neccesary sub-dirs */

		strcpy(buf, l->data->key);
		strcpy(dir, conf->outputdir);

		for (f = buf; (p = strchr(f, '/')) != NULL; f = p + 1) {
			struct stat st;

			/* extract the directory parts */

			*p = '\0';
			strcat(dir, "/");
			strcat(dir, f);

			if (stat(dir, &st) == 0) {
				if (S_ISDIR(st.st_mode)) {
					/* directory already exists */
					continue;
				} else {
					M_DEBUG1(ext_conf->debug_level, M_DEBUG_SECTION_PARSING, M_DEBUG_LEVEL_ERRORS,
						 "'%s' exists, but is not a diretory\n",
						 dir);
					free(dir);

					return -1;
				}
			} else {
				if (mkdir(dir,
					  S_IROTH | S_IXOTH |
					  S_IRGRP | S_IXGRP |
					  S_IRUSR | S_IXUSR | S_IWUSR) != 0) /* 0755 */
				{
					M_DEBUG2(ext_conf->debug_level, M_DEBUG_SECTION_PARSING, M_DEBUG_LEVEL_ERRORS,
						 "makeing directory %s failed: %s\n",
						 dir,
						 strerror(errno));
					free(dir);

					return -1;
				}
			}
		}
		free(dir);


		/* open both files */
		if ((fd1 = open(fn1, O_RDONLY)) == -1) {
			M_DEBUG2(ext_conf->debug_level, M_DEBUG_SECTION_PARSING, M_DEBUG_LEVEL_ERRORS,
				 "opening source %s failed: %s\n",
				 fn1,
				 strerror(errno));

			free(fn1);
			free(fn2);


			return -1;
		}
		if ((fd2 = open(fn2, O_CREAT | O_TRUNC | O_WRONLY, S_IROTH | S_IRGRP | S_IWUSR | S_IRUSR)) == -1) {
			M_DEBUG2(ext_conf->debug_level, M_DEBUG_SECTION_PARSING, M_DEBUG_LEVEL_ERRORS,
				 "opening dest file '%s' failed: %s\n",
				 fn2,
				 strerror(errno));
			free(fn1);
			free(fn2);

			return -1;
		}

		/* copy the files */
		while( (n = read(fd1, buf, sizeof(buf)) )) {
			write(fd2, buf, n);
		}

		/* clean-up */
		close(fd1);
		close(fd2);

		free(fn1);
		free(fn2);
	}

	tmpl_free(tmpl);

	MTIMER_STOP(timer[6]);
	MTIMER_CALC(timer[6]);

	MTIMER_STOP(timer[0]);
	MTIMER_CALC(timer[0]);

/*	for (i = 0; i < NUM_TIMER; i++) {
		fprintf(stderr, "timer %d: %ld msec\n",
			i, timer[i].span);
	}
 */

	return 0;
}

static int generate_history_output(mconfig *ext_conf, mlist *history) {
	mlist *l = history;
	char filename[255];
	config_output *conf = ext_conf->plugin_conf;
	tmpl_main * tmpl;
	char *fn;
	int ret;
	char *s;
	int i;

	char buf[255];
	struct tm *_tm;
	time_t t;

	tmpl_vars vars[] = {
		/* generated by this plugin */
		{ "LASTRECORD", NULL },
		{ "GENERATEDAT", NULL },

		/* hard values */
		{ "MLA_URL", "http://www.modlogan.org" },
		{ "MLA_PACKAGE", PACKAGE },
		{ "MLA_VERSION", VERSION },

		/* translated strings */
		{ "TXT_STATISTICS", _("Statistics for")},
		{ "TXT_LASTRECORD", _("Last record")},
		{ "TXT_GENERATEDAT", _("Generated at")},
		{ "TXT_HISTORY", _("History")},
		{ "TXT_HITS", _("Hits")},
		{ "TXT_PAGES", _("Pages")},
		{ "TXT_FILES", _("Files")},
		{ "TXT_VISITS", _("Visits")},
		{ "TXT_TRAFFIC", _("Traffic")},
		{ "TXT_MONTH", _("Month")},
		{ "TXT_AVERAGE_DAY", _("Average/Day")},
		{ "TXT_TOTALS", _("Totals")},
		{ "TXT_OUTPUTBY", _("Output generated by")},

		{ "TITLE", _("Statistics") },

		{ NULL,  NULL }
	};

	if (history == NULL || history->data == NULL) return -1;
	
	/* suffix for the generated statistics page files */
	sprintf(filename, "%s/%s",
		conf->outputdir,
		conf->index_filename);

	tmpl = tmpl_init();
	assert(tmpl);

	tmpl->debug_level = ext_conf->debug_level;

	if ((fn = generate_template_filename(ext_conf, M_TMPL_INDEX)) == NULL) {
		M_DEBUG0(ext_conf->debug_level, M_DEBUG_SECTION_PARSING, M_DEBUG_LEVEL_ERRORS,
			 "can't generate filename of the template\n");
		tmpl_free(tmpl);

		return -1;
	}

	/* load and parse template */
	if ((ret = tmpl_load_template(tmpl, fn)) != 0) {
		M_DEBUG1(ext_conf->debug_level, M_DEBUG_SECTION_PARSING, M_DEBUG_LEVEL_ERRORS,
			 "parsing template %s failed\n",
			 fn);
		tmpl_free(tmpl);
		free(fn);

		return ret;
	}
	free(fn);
	
	switch(history->data->data.hist->type) {
	case M_HIST_TYPE_WEB:
		if (0 != mplugins_output_generate_history_output_web(ext_conf, history, tmpl)) {
			M_WP();
			return -1;
		}

		break;
	case M_HIST_TYPE_MAIL:
		mplugins_output_generate_history_output_mail(ext_conf, history, tmpl);

		break;
	default:
		M_DEBUG1(ext_conf->debug_level, M_DEBUG_SECTION_PARSING, M_DEBUG_LEVEL_ERRORS,
			 "unknown history type: %d\n", history->data->data.hist->type);
		break;
	}

	/* set external variables from config-file */
	for (l = conf->variables; l && l->data; l = l->next) {
		char *k, *v;
		
		if (!l->data->key) continue;

		k = l->data->key;

		/* split string */
		if ((v = strchr(k, ',')) != NULL) {
			v++;

			/* check if the variable is used internally */
			for (i = 0; vars[i].key; i++) {
				if (0 == strncmp(k, vars[i].key, v-k-1))
					break;
			}

			/* not found */
			if (vars[i].key == NULL) {
				char *key = malloc(v-k);
				strncpy(key, k, v-k-1);
				key[v-k-1] = '\0';
				tmpl_set_var(tmpl, key, v);
				free(key);
			}
		} else {
			M_DEBUG1(ext_conf->debug_level, M_DEBUG_SECTION_PARSING, M_DEBUG_LEVEL_ERRORS,
				 "no ',' found in %s\n", k);
		}
	}
	
	/* create current timestamp */
	t = time(NULL);
	_tm = localtime(&t);
	strftime(buf, sizeof(buf), "%X %x", _tm);
	tmpl_set_var(tmpl, "GENERATEDAT", buf);

	/* set template variables */
	for (i = 0; vars[i].key; i++) {
		if (i != 0 && i != 1 && /* ignore GENERATEDAT and LASTRECORD as they are already set */
		    vars[i].subst != NULL &&
		    0 != tmpl_set_var(tmpl, vars[i].key, vars[i].subst)) {
			M_DEBUG1(ext_conf->debug_level, M_DEBUG_SECTION_PARSING, M_DEBUG_LEVEL_WARNINGS,
				 "substituing key %s failed\n",
				 vars[i].key);
		}
	}

	/* replace tags by the actual output */
	if (0 == tmpl_replace(tmpl, conf->tmp_output)) {
		/* index-file */
		int fd;
		
		if ((fd = open(filename, O_WRONLY | O_CREAT | O_TRUNC, 0644)) < 0) {
			M_DEBUG2(ext_conf->debug_level, M_DEBUG_SECTION_GENERATING, M_DEBUG_LEVEL_ERRORS,
				 "Can't open File \"%s\" for write. errno=%d\n", filename, errno);
		} else {
			write(fd, conf->tmp_output->ptr, conf->tmp_output->used - 1);
			
			close(fd);
		}
		tmpl_free(tmpl);
	} else {
		tmpl_free(tmpl);
		return -1;
	}

	return 0;
}


int mplugins_output_generate_monthly_output(mconfig * ext_conf, mstate * state, const char *subpath) {
	int ret;
	if (!mplugins_output_template_patch_config(ext_conf)) {
		M_DEBUG0(ext_conf->debug_level, M_DEBUG_SECTION_PARSING, M_DEBUG_LEVEL_ERRORS,
			 "could not patch config\n");
		return -1;
	}

	if (subpath) {
		char *s;
		config_output *conf = ext_conf->plugin_conf;
		/* create the subdir if neccesary */

		s = malloc(strlen(subpath) + strlen(conf->outputdir) + strlen("//") + 1);
		sprintf(s, "%s/%s/",
			conf->outputdir,
			subpath);
		if (-1 == mkdir(s, 0755)) {
			if (errno != EEXIST) {
				M_DEBUG1(ext_conf->debug_level, M_DEBUG_SECTION_PARSING, M_DEBUG_LEVEL_ERRORS,
					 "mkdir failed: %s\n", strerror(errno));
				return -1;
			}
		}

		/* modify output directory */
		free(conf->outputdir);
		conf->outputdir = s;

		fprintf(stderr, "generating output in %s\n", conf->outputdir);
	}

	if ((ret = generate_monthly_output(ext_conf, state))) {
		M_DEBUG0(ext_conf->debug_level, M_DEBUG_SECTION_PARSING, M_DEBUG_LEVEL_ERRORS,
			 "generate_monthly_output failed\n");
	}

	if (!mplugins_output_template_unpatch_config(ext_conf)) {
		M_DEBUG0(ext_conf->debug_level, M_DEBUG_SECTION_PARSING, M_DEBUG_LEVEL_ERRORS,
			 "could not un-patch config\n");
		return -1;
	}
	return 0;
}

int mplugins_output_generate_history_output(mconfig *ext_conf, mlist *history, const char *subpath) {
	int ret;
	if (!mplugins_output_template_patch_config(ext_conf)) {
		M_DEBUG0(ext_conf->debug_level, M_DEBUG_SECTION_PARSING, M_DEBUG_LEVEL_ERRORS,
			 "could not patch config\n");
		return -1;
	}

	if (subpath) {
		char *s;
		config_output *conf = ext_conf->plugin_conf;
		/* create the subdir if neccesary */

		s = malloc(strlen(subpath) + strlen(conf->outputdir) + strlen("//") + 1);
		sprintf(s, "%s/%s/",
			conf->outputdir,
			subpath);

		if (-1 == mkdir(s, 0755)) {
			if (errno != EEXIST) {
				M_DEBUG1(ext_conf->debug_level, M_DEBUG_SECTION_PARSING, M_DEBUG_LEVEL_ERRORS,
					 "mkdir failed: %s\n", strerror(errno));
				return -1;
			}
		}

		/* modify output directory */
		free(conf->outputdir);
		conf->outputdir = s;

		fprintf(stderr, "generating history in %s\n", conf->outputdir);
	}

	ret = generate_history_output(ext_conf, history);

	if (!mplugins_output_template_unpatch_config(ext_conf)) {
		M_DEBUG0(ext_conf->debug_level, M_DEBUG_SECTION_PARSING, M_DEBUG_LEVEL_ERRORS,
			 "could not un-patch config\n");
		return -1;
	}
	return 0;
}

