/*
** Modular Logfile Analyzer
** Copyright 2000 Jan Kneschke <jan@kneschke.de>
**
** Homepage: http://www.modlogan.org
**

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version, and provided that the above
    copyright and permission notice is included with all distributed
    copies of this or derived software.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA

**
** $Id: pictures_day.c,v 1.20 2003/07/31 23:55:47 ostborn Exp $
*/
#include <stdio.h>
#include <stdlib.h>
#include <time.h>

#include "config.h"

#ifdef HAVE_LIBGD
#ifdef HAVE_GD_GD_H
#include <gd/gd.h>
#include <gd/gdfonts.h>
#else
#include <gd.h>
#include <gdfonts.h>
#endif
#endif

#include "mconfig.h"
#include "mlist.h"
#include "mdatatypes.h"
#include "mlocale.h"
#include "misc.h"

#include "pictures.h"
#include "generate.h"

/* calculating the box
**
** +------------------------------+
** |+----------------------------+|
** ||                            ||
** ||#                           ||
** ||##  #    (1)                ||
** ||## ##                       ||
** ||#####                       ||
** ||#####                       ||
** |+----------------------------+|
** |+----------------------------+|
** ||                            ||
** ||#                           ||
** ||##  #    (2)                ||
** ||## ##                       ||
** ||#####                       ||
** ||#####                       ||
** |+----------------------------+|
** |+----------------------------+|
** ||                            ||
** ||#                           ||
** ||##  #    (3)                ||
** ||## ##                       ||
** ||#####                       ||
** ||#####                       ||
** |+----------------------------+|
** +------------------------------+
**
*/

#define IM_BOX1_HEIGHT	150
#define IM_BOX2_HEIGHT	100
#define IM_BOX3_HEIGHT	IM_BOX2_HEIGHT
#define IM_CELLSPACING	  4

#define IM_BOX1_BAR_WIDTH	  8
#define IM_BOX1_BAR_SHIFT	  2
#define IM_BOX1_BAR_SPACE	  8

#define IM_BOX2_BAR_WIDTH	  8
#define IM_BOX2_BAR_SHIFT	  2
#define IM_BOX2_BAR_SPACE	  8

#define IM_BOX3_BAR_WIDTH	  8
#define IM_BOX3_BAR_SHIFT	  2
#define IM_BOX3_BAR_SPACE	  8

/* gdSmallFont */
#define IM_FONT_HEIGHT	 13
#define IM_FONT_WIDTH	  6

#define IM_BOX1_X1	IM_CELLSPACING * 2 + IM_FONT_HEIGHT
#define IM_BOX1_Y1	IM_CELLSPACING * 2 + IM_FONT_HEIGHT
#define IM_BOX1_Y2	IM_BOX1_Y1 + IM_BOX1_HEIGHT - IM_CELLSPACING

#define IM_BOX2_X1	IM_BOX1_X1
#define IM_BOX2_Y1	IM_BOX1_Y2 + 3 * IM_CELLSPACING
#define IM_BOX2_Y2	IM_BOX2_Y1 + IM_BOX2_HEIGHT - IM_CELLSPACING

#define IM_BOX3_X1	IM_BOX1_X1
#define IM_BOX3_Y1	IM_BOX2_Y2 + 3 * IM_CELLSPACING
#define IM_BOX3_Y2	IM_BOX3_Y1 + IM_BOX3_HEIGHT - IM_CELLSPACING

#define IM_HEIGHT	IM_BOX3_Y2 + IM_CELLSPACING * 2 + IM_FONT_HEIGHT

#define IM_LAB1_Y1	IM_HEIGHT - IM_FONT_HEIGHT - IM_CELLSPACING

#define IM_FILENAME	"daily_usage_"
#define IM_FILEEXT	".png"

char *mplugin_modlogan_create_pic_31_day(mconfig *ext_conf, mstate *state) {
#ifdef HAVE_LIBGD
	data_WebHistory max;

	/* The number of bars in the graph depends on the month */
	int leapyear = (state->year%4 == 0 && state->year%100 != 0) || state->year%400 == 0;
	int month_days[] = {31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31};
	int IM_BARS = month_days[state->month > 0 && state->month <= 12 ? state->month - 1 : 0]
		      + ((leapyear && state->month == 2)?1:0);
	int IM_BOX1_X2 = IM_BOX1_X1 + IM_BARS * IM_BOX1_BAR_WIDTH + (IM_BARS - 1) * IM_BOX1_BAR_SPACE + IM_BOX1_BAR_SHIFT;
	int IM_BOX2_X2 = IM_BOX1_X2;
	int IM_BOX3_X2 = IM_BOX1_X2;
	int IM_WIDTH = IM_BOX1_X2 + IM_CELLSPACING * 2 + IM_FONT_HEIGHT;

	int i = IM_BARS;
	gdImage *im;
	FILE *f;
	char filename[255];
	rgb_tripple rgb;
	int x1, y1, x2, y2;
	char int2str[20]; /* 20 is a guess only !! */
	config_output *conf = ext_conf->plugin_conf;
	mstate_web *staweb = state->ext;
	char *str;

	struct tm tm;

	static char href[255] = "";

	int col_border, col_backgnd,
		col_pages, col_files,
		col_visits, col_border_sh,
		col_xfer, col_hits;

	max.hits	= 0;
	max.files	= 0;
	max.pages	= 0;
	max.visits	= 0;
	max.hosts	= 0;
	max.xfersize	= 0;

#define	SET_MAX(x) if (staweb->days[i].x > max.x) max.x = staweb->days[i].x

	for ( i = 0; i < IM_BARS; i++) {
		SET_MAX(hits);
		SET_MAX(files);
		SET_MAX(pages);
		SET_MAX(visits);
		SET_MAX(hosts);
		SET_MAX(xfersize);
	}
#undef SET_MAX
	im = gdImageCreate( IM_WIDTH + 1,IM_HEIGHT + 1 );

	col_border	= gdImageColorAllocate(im, 0x00, 0x00, 0x00);

	html3torgb3(conf->col_shadow, &rgb);
	col_border_sh	= gdImageColorAllocate(im, rgb.r, rgb.g, rgb.b);

	html3torgb3(conf->col_backgnd, &rgb);
	col_backgnd	= gdImageColorAllocate(im, rgb.r, rgb.g, rgb.b);
	html3torgb3(conf->col_hits, &rgb);
	col_hits	= gdImageColorAllocate(im, rgb.r, rgb.g, rgb.b);
	html3torgb3(conf->col_files, &rgb);
	col_files	= gdImageColorAllocate(im, rgb.r, rgb.g, rgb.b);
	html3torgb3(conf->col_pages, &rgb);
	col_pages	= gdImageColorAllocate(im, rgb.r, rgb.g, rgb.b);
	html3torgb3(conf->col_visits, &rgb);
	col_visits	= gdImageColorAllocate(im, rgb.r, rgb.g, rgb.b);
	html3torgb3(conf->col_xfer, &rgb);
	col_xfer	= gdImageColorAllocate(im, rgb.r, rgb.g, rgb.b);

/* background */
	gdImageFilledRectangle(im, 0, 0, IM_WIDTH-1, IM_HEIGHT-1, col_backgnd);
/* surrounding border */
	gdImageRectangle( im, 1, 1, IM_WIDTH-1, IM_HEIGHT-1, col_border );
	gdImageRectangle( im, 0, 0, IM_WIDTH, IM_HEIGHT, col_border_sh );

/* left border - max hits */
	sprintf(int2str, "%li", max.hits);
	x1 = IM_CELLSPACING;
	y1 = IM_BOX1_Y1 + strlen(int2str) * IM_FONT_WIDTH;
	gdImageStringUp(im, gdFontSmall, x1, y1, int2str, col_border);
/* left border - max visits */
	sprintf(int2str, "%li", max.visits);
	x1 = IM_CELLSPACING;
	y1 = IM_BOX2_Y1 + strlen(int2str) * IM_FONT_WIDTH;
	gdImageStringUp(im, gdFontSmall, x1, y1, int2str, col_border);
/* left border - max kbytes */
	sprintf(int2str, "%.0f", max.xfersize / 1024);
	x1 = IM_CELLSPACING;
	y1 = IM_BOX3_Y1 + strlen(int2str) * IM_FONT_WIDTH;
	gdImageStringUp(im, gdFontSmall, x1, y1, int2str, col_border);

/* right border - 'pages/files/hits' */
	x1 = IM_WIDTH - IM_CELLSPACING - IM_FONT_HEIGHT;
	y1 = IM_BOX1_Y1 + strlen(_("Hits")) * IM_FONT_WIDTH;
	gdImageStringUp(im, gdFontSmall, x1+1, y1+1, _("Hits"), col_border_sh);
	gdImageStringUp(im, gdFontSmall, x1, y1, _("Hits"), col_hits);

	y1 += IM_FONT_WIDTH;
	gdImageStringUp(im, gdFontSmall, x1+1, y1+1, "/", col_border_sh);
	gdImageStringUp(im, gdFontSmall, x1, y1, "/", col_border);

	y1 += strlen(_("Files")) * IM_FONT_WIDTH;
	gdImageStringUp(im, gdFontSmall, x1+1, y1+1, _("Files"), col_border_sh);
	gdImageStringUp(im, gdFontSmall, x1, y1, _("Files"), col_files);

	y1 += IM_FONT_WIDTH;
	gdImageStringUp(im, gdFontSmall, x1+1, y1+1, "/", col_border_sh);
	gdImageStringUp(im, gdFontSmall, x1, y1, "/", col_border);

	y1 += strlen(_("Pages")) * IM_FONT_WIDTH;
	gdImageStringUp(im, gdFontSmall, x1+1, y1+1, _("Pages"), col_border_sh);
	gdImageStringUp(im, gdFontSmall, x1, y1, _("Pages"), col_pages);
/* right border - 'visits' */
	x1 = IM_WIDTH - IM_CELLSPACING - IM_FONT_HEIGHT;
	y1 = IM_BOX2_Y1 + strlen(_("Visits")) * IM_FONT_WIDTH;
	gdImageStringUp(im, gdFontSmall, x1+1, y1+1, _("Visits"), col_border_sh);
	gdImageStringUp(im, gdFontSmall, x1, y1, _("Visits"), col_visits);
/* right border - 'kbytes' */
	x1 = IM_WIDTH - IM_CELLSPACING - IM_FONT_HEIGHT;
	y1 = IM_BOX3_Y1 + strlen(_("KBytes")) * IM_FONT_WIDTH;
	gdImageStringUp(im, gdFontSmall, x1+1, y1+1, _("KBytes"), col_border_sh);
	gdImageStringUp(im, gdFontSmall, x1, y1, _("KBytes"), col_xfer);

/* header */
	x1 = IM_BOX1_X1;
	y1 = IM_CELLSPACING;
	str = malloc( 1 + strlen(_("Daily usage for %1$s %2$04d")) - 10 + 4 +
                      strlen( get_month_string(state->month,0)));
	sprintf( str, _("Daily usage for %1$s %2$04d"), get_month_string(state->month,0),
		 state->year );
	gdImageString(im, gdFontSmall, x1, y1, str, col_border);
	free(str);
	
/* border for BOX1 */
	gdImageRectangle(im,
		IM_BOX1_X1 - IM_CELLSPACING,
		IM_BOX1_Y1 - IM_CELLSPACING,
		IM_BOX1_X2 + IM_CELLSPACING,
		IM_BOX1_Y2 + IM_CELLSPACING,
		col_border);
	gdImageRectangle(im,
		IM_BOX1_X1 - IM_CELLSPACING + 1,
		IM_BOX1_Y1 - IM_CELLSPACING + 1,
		IM_BOX1_X2 + IM_CELLSPACING + 1,
		IM_BOX1_Y2 + IM_CELLSPACING + 1,
		col_border_sh);
/* border for BOX2 */
	gdImageRectangle(im,
		IM_BOX2_X1 - IM_CELLSPACING,
		IM_BOX2_Y1 - IM_CELLSPACING,
		IM_BOX2_X2 + IM_CELLSPACING,
		IM_BOX2_Y2 + IM_CELLSPACING,
		col_border);
	gdImageRectangle(im,
		IM_BOX2_X1 - IM_CELLSPACING + 1,
		IM_BOX2_Y1 - IM_CELLSPACING + 1,
		IM_BOX2_X2 + IM_CELLSPACING + 1,
		IM_BOX2_Y2 + IM_CELLSPACING + 1,
		col_border_sh);
/* border for BOX3 */
	gdImageRectangle(im,
		IM_BOX3_X1 - IM_CELLSPACING,
		IM_BOX3_Y1 - IM_CELLSPACING,
		IM_BOX3_X2 + IM_CELLSPACING,
		IM_BOX3_Y2 + IM_CELLSPACING,
		col_border);
	gdImageRectangle(im,
		IM_BOX3_X1 - IM_CELLSPACING + 1,
		IM_BOX3_Y1 - IM_CELLSPACING + 1,
		IM_BOX3_X2 + IM_CELLSPACING + 1,
		IM_BOX3_Y2 + IM_CELLSPACING + 1,
		col_border_sh);

/* calc the weekday and check if date is valid */
	memset(&tm, 0, sizeof(struct tm));

	tm.tm_sec = 0;
	tm.tm_min = 0;
	tm.tm_hour = 0;
	tm.tm_year = state->year-1900;
	tm.tm_mon = state->month-1;
	tm.tm_mday = i + 1;

	for ( i = 0; i < IM_BARS; i++) {
		/* Box 1 */
		tm.tm_mday = i + 1;

		if (mktime (&tm) == -1) break;

		if (max.hits) {
			int y2b = IM_BOX1_Y2;
			int y1b = IM_BOX1_Y1;

			x1 = IM_BOX1_X1 + i * (IM_BOX1_BAR_SPACE + IM_BOX1_BAR_WIDTH);
			x2 = x1 + IM_BOX1_BAR_WIDTH;

			y2 = y2b;
			y1 = y2 - ((double)staweb->days[i].hits / max.hits) * (y2b - y1b - 1);

			if (y1 != y2) {
				gdImageFilledRectangle(im, x1, y1, x2, y2, col_hits);
				gdImageRectangle(im, x1, y1, x2, y2, col_border);
			}

			x1 += IM_BOX1_BAR_SHIFT;
			x2 += IM_BOX1_BAR_SHIFT;

			y1 = y2 - ((double)staweb->days[i].files / max.hits) * (y2b - y1b - 1);

			if (y1 != y2) {
				gdImageFilledRectangle(im, x1, y1, x2, y2, col_files);
				gdImageRectangle(im, x1, y1, x2, y2, col_border);
			}

			x1 += IM_BOX1_BAR_SHIFT;
			x2 += IM_BOX1_BAR_SHIFT;

			y1 = y2 - ((double)staweb->days[i].pages / max.hits) * (y2b - y1b - 1);
			if (y1 != y2) {
				gdImageFilledRectangle(im, x1, y1, x2, y2, col_pages);
				gdImageRectangle(im, x1, y1, x2, y2, col_border);
			}
		}
		/* Box 2 */

		if (max.visits) {
			int y2b = IM_BOX2_Y2;
			int y1b = IM_BOX2_Y1;

			x1 = IM_BOX2_X1 + i * (IM_BOX2_BAR_SPACE + IM_BOX2_BAR_WIDTH);
			x2 = x1 + IM_BOX2_BAR_WIDTH;

			y2 = y2b;
			y1 = y2 - ((double)staweb->days[i].visits / max.visits) * (y2b - y1b - 1);

			if (y1 != y2) {
				gdImageFilledRectangle(im, x1, y1, x2, y2, col_visits);
				gdImageRectangle(im, x1, y1, x2, y2, col_border);
			}
		}

		/* Box 3 */

		if (max.xfersize) {
			double percent = (double)staweb->days[i].xfersize / max.xfersize;
			int y2b = IM_BOX3_Y2;
			int y1b = IM_BOX3_Y1;

			x1 = IM_BOX3_X1 + i * (IM_BOX3_BAR_SPACE + IM_BOX3_BAR_WIDTH);
			x2 = x1 + IM_BOX3_BAR_WIDTH;

			y2 = y2b;
			y1 = y2 - (percent * (y2b - y1b - 1));

			if (y1 != y2) {
				gdImageFilledRectangle(im, x1, y1, x2, y2, col_xfer);
				gdImageRectangle(im, x1, y1, x2, y2, col_border);
			}
		}

		x1 = IM_BOX1_X1 + i * (IM_BOX1_BAR_SPACE + IM_BOX1_BAR_WIDTH);
		y1 = IM_LAB1_Y1;

		sprintf(int2str, "%2i", i+1);

		gdImageString(im, gdFontSmall, x1, y1, int2str, (tm.tm_wday == 0 || tm.tm_wday == 6) ? col_hits : col_border);
	}
	sprintf(filename, "%s/%s%04d%02d%s",
		conf->outputdir,
		IM_FILENAME,
		state->year, state->month,
		IM_FILEEXT);

	if ((f = fopen (filename, "wb"))) {
		gdImagePng(im, f);

		fclose(f);
	}

	gdImageDestroy(im);

	sprintf(href, "<center><img src=\"%s%04d%02d%s\" alt=\"%s\" width=\"%d\" height=\"%d\"></center>\n",
		IM_FILENAME,
		state->year, state->month,
		IM_FILEEXT,
		_("Daily usage"), IM_WIDTH+1, IM_HEIGHT+1);
	return href;
#else
	return NULL;
#endif
}

