/*
** Modular Logfile Analyzer
** Copyright 2000 Jan Kneschke <jan@kneschke.de>
** Copyright 2002 Thomas Schweikle <tschweikle@fiducia.de>
**
** Homepage: http://www.modlogan.org
**

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version, and provided that the above
    copyright and permission notice is included with all distributed
    copies of this or derived software.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA

**
** $Id: parse.c,v 1.4 2003/04/18 18:40:42 ostborn Exp $
*/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <time.h>
#include <ctype.h>
#include <errno.h>

#include "mlocale.h"
#include "mplugins.h"
#include "mrecord.h"
#include "mdatatypes.h"
#include "misc.h"

#include "plugin_config.h"

int parse_record_pcre(mconfig *ext_conf, mlogrec *record, buffer *b) {
#define N 20 + 1
	const char **list;
	int ovector[3 * N], n;
        struct tm tm;

	config_input *conf = ext_conf->plugin_conf;
	mlogrec_web *recweb = NULL;
	mlogrec_web_ftp *recftp = NULL;

	if (record->ext_type != M_RECORD_TYPE_WEB) {
		if (record->ext_type != M_RECORD_TYPE_UNSET) {
			mrecord_free_ext(record);
		}
		
		record->ext_type = M_RECORD_TYPE_WEB;
		record->ext = mrecord_init_web();
	}

	recweb = record->ext;
	
	if (recweb == NULL) return M_RECORD_HARD_ERROR;

	recweb->ext = mrecord_init_web_ftp();
	recweb->ext_type = M_RECORD_TYPE_WEB_FTP;

	recftp = recweb->ext;

	if (recftp == NULL) return M_RECORD_HARD_ERROR;

	if ((n = pcre_exec(conf->match_rsync, conf->match_rsync_extra, b->ptr, b->used - 1, 0, 0, ovector, 3 * N)) < 0) {
		if (n == PCRE_ERROR_NOMATCH) {
			fprintf(stderr, "%s.%d: string doesn't match: %s\n", __FILE__, __LINE__, b->ptr);
			return M_RECORD_CORRUPT;
		} else {
			fprintf(stderr, "%s.%d: execution error while matching: %d\n", __FILE__, __LINE__, n);
			return M_RECORD_HARD_ERROR;
		}
	}

	if (n) {
		pcre_get_substring_list(b->ptr, ovector, n, &list);

		/* the meaning of the different fields (rsyncd.log) */

		/*
		 * 1-6 - timestamp
		 *  7  - process-id (unused)
		 *  8  - method
		 *  9  - hostname (unused)
		 * 10  - host-ip
		 * 11  - module
		 * 12  - user
		 * 13  - file
		 * 14  - transfered bytes
		 *
		 * url = module/file
		 */

		tm.tm_year = strtol(list[1], NULL, 10) - 1900;
		tm.tm_mon  = strtol(list[2], NULL, 10) - 1;
		tm.tm_mday = strtol(list[3], NULL, 10);
		tm.tm_hour = strtol(list[4], NULL, 10);
		tm.tm_min  = strtol(list[5], NULL, 10);
		tm.tm_sec  = strtol(list[6], NULL, 10);

		/* Since no timezone we'll ignore it */
		if ((record->timestamp = mkutctime (&tm)) == -1) {
			int i;

			printf ("Err: Time: ");
			for (i=1;i<=14;i++)
				printf ("%s ", list[i]);
			printf ("\n");
		}

		switch(*list[8]) {
		case 's':
			recftp->trans_command = M_RECORD_FTP_COMMAND_GET;
			break;
		case 'r':
			recftp->trans_command = M_RECORD_FTP_COMMAND_PUT;
			break;
		default:
			break;
		}

		buffer_strcpy(recweb->req_host_ip, (char *)list[10]);

		buffer_strcpy(recweb->req_user, (char *)list[12]);

		buffer_strcpy(recweb->req_url, (char *)list[11]);
		buffer_strcat(recweb->req_url, "/");
		buffer_strcat(recweb->req_url, (char *)list[13]);

		recweb->xfersize = strtol(list[14], NULL, 10);

		free(list);
	}

	return M_RECORD_NO_ERROR;
}


int mplugins_input_rsync_get_next_record(mconfig *ext_conf, mlogrec *record) {
	int ret = 0;
	config_input *conf = ext_conf->plugin_conf;

	if (record == NULL) return M_RECORD_HARD_ERROR;

	/* fill the line buffer */
	if (NULL == mgets(&(conf->inputfile), conf->buf)) return M_RECORD_EOF;
	
	ret = parse_record_pcre   (ext_conf, record, conf->buf);
	
	if (ret == M_RECORD_CORRUPT) {
		M_DEBUG1(ext_conf->debug_level, M_DEBUG_SECTION_PARSING, M_DEBUG_LEVEL_WARNINGS,
			 "affected Record: %s\n",
			 conf->buf->ptr
			 );
	}
	return ret;
}
