/* ====================================================================
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 2003 MaxMind LLC  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        MaxMind (http://www.maxmind.com/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The names "MaxMind" and "GeoIP" must
 *    not be used to endorse or promote products derived from this
 *    software without prior written permission. For written
 *    permission, please contact support@maxmind.com.
 *
 * 5. Products derived from this software may not be called "GeoIP",
 *    nor may "MaxMind" appear in their name, without prior written
 *    permission of the MaxMind.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 */

/*
 * Module definition information - the part between the -START and -END
 * lines below is used by Configure. This could be stored in a separate
 * instead.
 *
 * MODULE-DEFINITION-START
 * Name: geoip_module
 * ConfigStart
     GEOIP_LIB="-L/usr/local/lib -lGeoIP"
     if [ "X$GEOIP_LIB" != "X" ]; then
         LIBS="$LIBS $GEOIP_LIB"
         echo " + using $GEOIP_LIB for GeoIP support"
     fi
 * ConfigEnd
 * MODULE-DEFINITION-END
 */

/* geoip module
 *
 * Version 1.1.0
 *
 * This module sets an environment variable to the remote country
 * based on the requestor's IP address.  It uses the GeoIP library
 * to lookup the country by IP address.
 *
 * Copyright 2003 MaxMind LLC
 * June 26th, 2002

To use the module you have to compile it into the frontend part of
your server, I usually copy the module to apache-1.3/src/modules/extra/
and use APACI like:

  ./configure --prefix=/usr/local/apache \
     --activate-module=src/modules/extra/mod_geoip.c \
     [... more apaci options ...]    

You should also be able to compile and use this module as a
dynamically loaded module (DSO).
 
 */

#include "httpd.h"
#include "http_config.h"
#include "http_core.h"
#include "http_log.h"
#include "GeoIP.h"
#include "GeoIPCity.h"

typedef struct {
  GeoIP **gips;
  int numGeoIPFiles;
  char **GeoIPFilenames;
  int GeoIPEnable;
  char GeoIPOutput;
  int GeoIPFlags;
  int *GeoIPFlags2;
  int xForwardedFor;
} geoip_server_cfg;

module MODULE_VAR_EXPORT geoip_module;

module MODULE_VAR_EXPORT proxy_add_uri_module;

static const int GEOIP_NONE    = 0;
static const int GEOIP_DEFAULT = 1;
static const int GEOIP_NOTES   = 2;
static const int GEOIP_ENV     = 4;
static const int GEOIP_ALL     = 6;
static const int GEOIP_INIT    = 7;

static const int GEOIP_UNKNOWN = -1;

char dmacodestr[100];
char areacodestr[100];
char latstr[100];
char lonstr[100];

static void *geoip_server_config(pool *p, server_rec *s) {
  geoip_server_cfg *cfg = ap_pcalloc(p, sizeof(geoip_server_cfg));
  if (!cfg)
    return NULL;

  cfg->gips = NULL;
  cfg->numGeoIPFiles = 0;
  cfg->GeoIPFilenames = NULL;
  cfg->GeoIPEnable = 0;
  cfg->GeoIPOutput = GEOIP_INIT;
  cfg->GeoIPFlags = GEOIP_STANDARD;
  cfg->GeoIPFlags2 = NULL;
  cfg->xForwardedFor = NULL;
  return (void *)cfg;
}

static const char *geoip_enable(cmd_parms *cmd, void *dummy, int flag) {
  server_rec *s = cmd->server;
  geoip_server_cfg *cfg = (geoip_server_cfg *)ap_get_module_config(s->module_config, &geoip_module);
  cfg->GeoIPEnable = flag;
  return NULL;
}

static const char *geoip_xforward(cmd_parms *cmd, void *dummy, int flag) {
  server_rec *s = cmd->server;
  geoip_server_cfg *cfg = (geoip_server_cfg *)ap_get_module_config(s->module_config, &geoip_module);
  cfg->xForwardedFor = flag;
  return NULL;
}

/* TODO we will have to change this to ITERATE2 to support multiple flags per database file */
static const char *geoip_set_filename(cmd_parms *cmd, void *dummy, char *GeoIPFilename, char *arg2) {
  server_rec *s = cmd->server;
  geoip_server_cfg *cfg = (geoip_server_cfg *)ap_get_module_config(s->module_config, &geoip_module);
  int i = cfg->numGeoIPFiles;
  cfg->numGeoIPFiles++;
  cfg->GeoIPFilenames = realloc(cfg->GeoIPFilenames, cfg->numGeoIPFiles * sizeof(char *));
  cfg->GeoIPFilenames[i] = GeoIPFilename;
  cfg->GeoIPFlags2 = realloc(cfg->GeoIPFlags2, cfg->numGeoIPFiles * sizeof(int));
  if (arg2 == NULL) {
    cfg->GeoIPFlags2[i] = GEOIP_UNKNOWN;
  } else if (!strcmp(arg2, "Standard")) {
    cfg->GeoIPFlags2[i] = GEOIP_STANDARD;
  } else if (!strcmp(arg2, "MemoryCache")) {
    cfg->GeoIPFlags2[i] = GEOIP_MEMORY_CACHE;
  } else if (!strcmp(arg2, "CheckCache")) {
    cfg->GeoIPFlags2[i] = GEOIP_CHECK_CACHE;
  }
  return NULL;
}

static const char *geoip_set_output(cmd_parms *cmd, void *dummy, const char *arg) {
  server_rec *s = cmd->server;
  geoip_server_cfg *cfg = (geoip_server_cfg *)ap_get_module_config(s->module_config, &geoip_module);
  if (cfg->GeoIPOutput & GEOIP_DEFAULT) {
    /* was set to default, clear so can be reset with user specified values */
    cfg->GeoIPOutput = GEOIP_NONE;
  }
  if (!strcmp(arg, "Notes")) {
    cfg->GeoIPOutput |= GEOIP_NOTES;
  } else if (!strcmp(arg, "Env")) {
    cfg->GeoIPOutput |= GEOIP_ENV;
  } else if (!strcmp(arg, "All")) {
    cfg->GeoIPOutput |= GEOIP_ALL;
  } else {
    ap_log_error(APLOG_MARK, APLOG_ERR, s, "[mod_geoip]: Invalid Value for GeoIPOutput: %s", arg);
  }
  return NULL;
}

/* TODO we will have to change this to ITERATE2 to support multiple flags per database file */
static const char *geoip_set_flags(cmd_parms *cmd, void *dummy, const char *arg) {
  server_rec *s = cmd->server;
  geoip_server_cfg *cfg = (geoip_server_cfg *)ap_get_module_config(s->module_config, &geoip_module);
  if (!strcmp(arg, "MemoryCache")) {
    cfg->GeoIPFlags &= GEOIP_MEMORY_CACHE;
  } else if (!strcmp(arg, "CheckCache")) {
    cfg->GeoIPFlags &= GEOIP_CHECK_CACHE;
  }
  return NULL;
}

static command_rec geoip_cmds[] = {
  { "GeoIPDBFile", geoip_set_filename, NULL,
    OR_ALL, TAKE12, "GeoIP Data Files" },
  { "GeoIPEnable", geoip_enable, NULL,
    OR_ALL, FLAG, "Enable mod_geoip" },
  { "GeoIPOutput", geoip_set_output, NULL,
    OR_ALL, ITERATE, "Specify output method(s)" },
  { "GeoIPFlags", geoip_set_flags, NULL,
    OR_ALL, ITERATE, "GeoIP flags" },
  { "GeoIPXForwardedFor", geoip_xforward, NULL,
    OR_ALL, FLAG, "Get IP from X-Forwarded-For" },
  { NULL }
};

static int geoip_post_read_request (request_rec *r) {
  geoip_server_cfg *cfg = (geoip_server_cfg *)ap_get_module_config(r->server->module_config, &geoip_module);
  char *ipaddr;
  char *orgorisp;
  short int country_id;
  GeoIPRecord * gir;
  GeoIPRegion * giregion;
  const char *country_code, *country_name;
  const char *netspeedstring;
  unsigned char databaseType;
  int i;
  int netspeed = 0;

  /* For spliting X-Forwarded-For */
  char *ipaddr_ptr;
  int ipaddr_allocated;
  char *comma_ptr;

  if(!cfg->GeoIPEnable)
    return DECLINED;

  if(!cfg->xForwardedFor)
    ipaddr = r->connection->remote_ip;
  else {
    ipaddr_allocated = 0;
    ipaddr = ap_table_get(r->headers_in, "X-Forwarded-For");
    /* Check to ensure that the X-Forwarded-For header is
     * not a comma separated list of addresses, which would
     * cause mod_geoip to return no country code. If the
     * header is a comma separated list, return the first
     * IP address in the list, which is (hopefully!) the
     * real client IP.
     */
    ipaddr_ptr = calloc(16, sizeof(char));
    if (ipaddr_ptr != NULL) {
      ipaddr_allocated = 1;
      /* Copy the first 15 characters of ipaddr into
       * ipaddr_ptr, so that the X-Forwarded-For value
       * can be modified, if needed.
       */
      strncpy(ipaddr_ptr, ipaddr, 15);
      ipaddr_ptr[15] = (char) NULL;
      comma_ptr = ipaddr_ptr;
      for (i = 0; i < 16; ++i) {
        if (comma_ptr[i] == ',') {
          /* A comma has been found, so replace it with NULL,
           * so that when ipaddr is used, only the first
           * address will be found.
           */
          comma_ptr[i] = (char) NULL;
          /* Set the ipaddr pointer to be the modified
           * X-Forwarded-For value.
           */
          ipaddr = ipaddr_ptr;
          break;
        }
      }
    }
  }
  if(!ipaddr)
    return DECLINED;

  if(!cfg->gips) {
    if(cfg->GeoIPFilenames != NULL) {
      cfg->gips = malloc(sizeof(GeoIP *) * cfg->numGeoIPFiles);
      for (i = 0; i < cfg->numGeoIPFiles; i++) {
	/* Use flags associated with filename, or if not available, use global flags */
	cfg->gips[i] = GeoIP_open(cfg->GeoIPFilenames[i],
				  (cfg->GeoIPFlags2[i] == GEOIP_UNKNOWN) ? cfg->GeoIPFlags : cfg->GeoIPFlags2[i]);
	if(!cfg->gips[i]) {
	  ap_log_error(APLOG_MARK, APLOG_ERR, r->server, "[mod_geoip]: Error while opening data file %s", cfg->GeoIPFilenames[i]);
	  return DECLINED;
	}
      }
    } else {
      cfg->gips = malloc(sizeof(GeoIP *));
      cfg->gips[0] = GeoIP_new(cfg->GeoIPFlags);
      if(!cfg->gips[0]) {
	ap_log_error(APLOG_MARK, APLOG_ERR, r->server, "[mod_geoip]: Error while opening data file");
	return DECLINED;
      }
      cfg->numGeoIPFiles = 1;
    }
  }

  for (i = 0; i < cfg->numGeoIPFiles; i++) {
    databaseType = GeoIP_database_edition(cfg->gips[i]);
    switch (databaseType) {
    case GEOIP_NETSPEED_EDITION:
      netspeed = GeoIP_id_by_name (cfg->gips[i], ipaddr);	
      if (netspeed == GEOIP_UNKNOWN_SPEED) {
        netspeedstring = "unknown";
      }
      else if (netspeed == GEOIP_DIALUP_SPEED) {
        netspeedstring = "dialup";
      }
      else if (netspeed == GEOIP_CABLEDSL_SPEED) {
        netspeedstring = "cabledsl";
      }
      else if (netspeed == GEOIP_CORPORATE_SPEED) {
        netspeedstring = "corporate";
      }
      if (cfg->GeoIPOutput & GEOIP_NOTES){
        ap_table_set(r->notes,"GEOIP_NETSPEED",netspeedstring);
      }
      if (cfg->GeoIPOutput & GEOIP_ENV){
        ap_table_set(r->subprocess_env,"GEOIP_NETSPEED",netspeedstring);
      }
    break;
    case GEOIP_COUNTRY_EDITION:
      country_id = GeoIP_country_id_by_addr(cfg->gips[i], ipaddr);

      if (country_id > 0) {
	country_code = GeoIP_country_code[country_id];
	country_name = GeoIP_country_name[country_id];
	if (cfg->GeoIPOutput & GEOIP_NOTES) {
	  ap_table_set(r->notes, "GEOIP_COUNTRY_CODE", country_code);
	  ap_table_set(r->notes, "GEOIP_COUNTRY_NAME", country_name);
	}
	if (cfg->GeoIPOutput & GEOIP_ENV) {
	  ap_table_set(r->subprocess_env, "GEOIP_COUNTRY_CODE", country_code);
	  ap_table_set(r->subprocess_env, "GEOIP_COUNTRY_NAME", country_name);
	}
      }
      break;
    case GEOIP_REGION_EDITION_REV0:
    case GEOIP_REGION_EDITION_REV1:
      giregion = GeoIP_region_by_name (cfg->gips[i], ipaddr);
      if (giregion != NULL) {
	if (cfg->GeoIPOutput & GEOIP_NOTES) {
	  ap_table_set(r->notes, "GEOIP_COUNTRY_CODE", giregion->country_code);
	  if (giregion->region[0]) {
	    ap_table_set(r->notes, "GEOIP_REGION", giregion->region);
	  }
	}
	if (cfg->GeoIPOutput & GEOIP_ENV) {
	  ap_table_set(r->subprocess_env, "GEOIP_COUNTRY_CODE", giregion->country_code);
	  if (giregion->region[0]) {
	    ap_table_set(r->subprocess_env, "GEOIP_REGION", giregion->region);
	  }
	}
	GeoIPRegion_delete(giregion);
      }
      break;
    case GEOIP_CITY_EDITION_REV0:
    case GEOIP_CITY_EDITION_REV1:
      gir = GeoIP_record_by_addr(cfg->gips[i], ipaddr);
      if (gir != NULL) {
        sprintf(dmacodestr,"%d",gir->dma_code);
        sprintf(areacodestr,"%d",gir->area_code);
        sprintf(latstr,"%f",gir->latitude);
	sprintf(lonstr,"%f",gir->longitude);
        if (cfg->GeoIPOutput & GEOIP_NOTES) {
	  ap_table_set(r->notes, "GEOIP_COUNTRY_CODE", gir->country_code);
	  ap_table_set(r->notes, "GEOIP_COUNTRY_NAME", gir->country_name);
	  if (gir->region != NULL)
	    ap_table_set(r->notes, "GEOIP_REGION", gir->region);
	  if (gir->city != NULL)
	    ap_table_set(r->notes, "GEOIP_CITY", gir->city);
          ap_table_set(r->notes,"GEOIP_DMA_CODE",dmacodestr);
          ap_table_set(r->notes,"GEOIP_AREA_CODE",areacodestr);
          ap_table_set(r->notes,"GEOIP_LATITUDE",latstr);
          ap_table_set(r->notes,"GEOIP_LONGITUDE",lonstr);
          ap_table_set(r->notes,"GEOIP_POSTAL_CODE",gir->postal_code);
	}
	if (cfg->GeoIPOutput & GEOIP_ENV) {
	  ap_table_set(r->subprocess_env, "GEOIP_COUNTRY_CODE", gir->country_code);
	  ap_table_set(r->subprocess_env, "GEOIP_COUNTRY_NAME", gir->country_name);
	  if (gir->region != NULL)
	    ap_table_set(r->subprocess_env, "GEOIP_REGION", gir->region);
	  if (gir->city != NULL)
	    ap_table_set(r->subprocess_env, "GEOIP_CITY", gir->city);
          ap_table_set(r->subprocess_env,"GEOIP_DMA_CODE",dmacodestr);
          ap_table_set(r->subprocess_env,"GEOIP_AREA_CODE",areacodestr);
          ap_table_set(r->subprocess_env,"GEOIP_LATITUDE",latstr);
          ap_table_set(r->subprocess_env,"GEOIP_LONGITUDE",lonstr);
          ap_table_set(r->subprocess_env,"GEOIP_POSTAL_CODE",gir->postal_code);
	}
      }
      break;
    case GEOIP_ORG_EDITION:
      orgorisp = GeoIP_name_by_addr(cfg->gips[i], ipaddr);
      if (orgorisp != NULL) {
	if (cfg->GeoIPOutput & GEOIP_NOTES) {
	  ap_table_set(r->notes, "GEOIP_ORGANIZATION", orgorisp);
	}
	if (cfg->GeoIPOutput & GEOIP_ENV) {
	  ap_table_set(r->subprocess_env, "GEOIP_ORGANIZATION", orgorisp);
	}
      }
      break;
    case GEOIP_ISP_EDITION:
      orgorisp = GeoIP_name_by_addr(cfg->gips[i], ipaddr);
      if (orgorisp != NULL) {
	if (cfg->GeoIPOutput & GEOIP_NOTES) {
	  ap_table_set(r->notes, "GEOIP_ISP", orgorisp);
	}
	if (cfg->GeoIPOutput & GEOIP_ENV) {
	  ap_table_set(r->subprocess_env, "GEOIP_ISP", orgorisp);
	}
      }
      break;
    }
  }

  /* Cleanup spliting of X-Forwarded-For */
  if (ipaddr_allocated == 1) {
    free(ipaddr_ptr);
  }

  return OK;
}

static void geoip_child_exit(server_rec *r, pool *p) {
  int i;
  geoip_server_cfg *cfg = (geoip_server_cfg *)ap_get_module_config(r->module_config, &geoip_module);
  if(cfg->gips != NULL) {
    for (i = 0; i < cfg->numGeoIPFiles; i++)
      if(cfg->gips[i] != NULL)
	GeoIP_delete(cfg->gips[i]);
    free(cfg->gips);
  }
  if(cfg->GeoIPFilenames != NULL) {
    free(cfg->GeoIPFilenames);
  }
  if(cfg->GeoIPFlags2 != NULL) {
    free(cfg->GeoIPFlags2);
  }
}

module MODULE_VAR_EXPORT geoip_module = {
    STANDARD_MODULE_STUFF,
    NULL,                       /* initializer */
    NULL,                       /* dir config creater */
    NULL,                       /* dir merger --- default is to override */
    geoip_server_config,        /* server config */
    NULL,                       /* merge server configs */
    geoip_cmds,                 /* command table */
    NULL,                       /* handlers */
    NULL,                       /* filename translation */
    NULL,                       /* check_user_id */
    NULL,                       /* check auth */
    NULL,                       /* check access */
    NULL,                       /* type_checker */
    NULL,                       /* fixups */
    NULL,                       /* logger */
    NULL,                       /* header parser */
    NULL,                       /* child_init */
    geoip_child_exit,           /* child_exit */
    geoip_post_read_request,    /* post read-request */
};
