/* ====================================================================
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 2002 MaxMind.com.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        MaxMind (http://www.maxmind.com/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The names "MaxMind" and "GeoIP" must
 *    not be used to endorse or promote products derived from this
 *    software without prior written permission. For written
 *    permission, please contact support@maxmind.com.
 *
 * 5. Products derived from this software may not be called "GeoIP",
 *    nor may "MaxMind" appear in their name, without prior written
 *    permission of the MaxMind.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 */

/*
 * Module definition information - the part between the -START and -END
 * lines below is used by Configure. This could be stored in a separate
 * instead.
 *
 * MODULE-DEFINITION-START
 * Name: geoip_module
 * ConfigStart
     GEOIP_LIB="-L/usr/local/lib -lGeoIP"
     if [ "X$GEOIP_LIB" != "X" ]; then
         LIBS="$LIBS $GEOIP_LIB"
         echo " + using $GEOIP_LIB for GeoIP support"
     fi
 * ConfigEnd
 * MODULE-DEFINITION-END
 */

/* geoip module
 *
 * Version 1.1.0
 *
 * This module sets an environment variable to the remote country
 * based on the requestor's IP address.  It uses the GeoIP library
 * to lookup the country by IP address.
 *
 * Copyright 2002, MaxMind.com
 * June 26th, 2002

To use the module you have to compile it into the frontend part of
your server, I usually copy the module to apache-1.3/src/modules/extra/
and use APACI like:

  ./configure --prefix=/usr/local/apache \
     --activate-module=src/modules/extra/mod_geoip.c \
     [... more apaci options ...]    

You should also be able to compile and use this module as a
dynamically loaded module (DSO).
 
 */

#include "httpd.h"
#include "http_config.h"
#include "http_core.h"
#include "http_log.h"
#include "GeoIP.h"
#include "GeoIPCity.h"

typedef struct {
  GeoIP *gip;
  char *GeoIPFilename;
  int GeoIPEnable;
  char GeoIPOutput;
  int GeoIPFlags;
} geoip_server_cfg;

module MODULE_VAR_EXPORT geoip_module;

module MODULE_VAR_EXPORT proxy_add_uri_module;

static const int GEOIP_NONE    = 0;
static const int GEOIP_DEFAULT = 1;
static const int GEOIP_NOTES   = 2;
static const int GEOIP_ENV     = 4;
static const int GEOIP_ALL     = 6;
static const int GEOIP_INIT    = 7;

static void *geoip_server_config(pool *p, server_rec *s) {
  geoip_server_cfg *cfg = ap_pcalloc(p, sizeof(geoip_server_cfg));
  if (!cfg)
    return NULL;

  cfg->gip = NULL;
  cfg->GeoIPFilename = NULL;
  cfg->GeoIPEnable = 0;
  cfg->GeoIPOutput = GEOIP_INIT;
  cfg->GeoIPFlags = GEOIP_STANDARD;
  return (void *)cfg;
}

static const char *geoip_enable(cmd_parms *cmd, void *dummy, int flag) {
  server_rec *s = cmd->server;
  geoip_server_cfg *cfg = (geoip_server_cfg *)ap_get_module_config(s->module_config, &geoip_module);
  cfg->GeoIPEnable = flag;
  return NULL;
}

static const char *geoip_set_filename(cmd_parms *cmd, void *dummy, char *GeoIPFilename) {
  server_rec *s = cmd->server;
  geoip_server_cfg *cfg = (geoip_server_cfg *)ap_get_module_config(s->module_config, &geoip_module);
  cfg->GeoIPFilename = GeoIPFilename;
  return NULL;
}

static const char *geoip_set_output(cmd_parms *cmd, void *dummy, const char *arg) {
  server_rec *s = cmd->server;
  geoip_server_cfg *cfg = (geoip_server_cfg *)ap_get_module_config(s->module_config, &geoip_module);
  if (cfg->GeoIPOutput & GEOIP_DEFAULT) {
    /* was set to default, clear so can be reset with user specified values */
    cfg->GeoIPOutput = GEOIP_NONE;
  }
  if (strcmp(arg, "Notes")) {
    cfg->GeoIPOutput &= GEOIP_NOTES;
  } else if (strcmp(arg, "Env")) {
    cfg->GeoIPOutput &= GEOIP_ENV;
  } else if (strcmp(arg, "All")) {
    cfg->GeoIPOutput &= GEOIP_ALL;
  } else {
    ap_log_error(APLOG_MARK, APLOG_ERR, s, "[mod_geoip]: Invalid Value for GeoIPOutput: %s", arg);
  }
}

static const char *geoip_set_flags(cmd_parms *cmd, void *dummy, const char *arg) {
  server_rec *s = cmd->server;
  geoip_server_cfg *cfg = (geoip_server_cfg *)ap_get_module_config(s->module_config, &geoip_module);
  if (strcmp(arg, "MemoryCache")) {
    cfg->GeoIPFlags &= GEOIP_MEMORY_CACHE;
  }
}

static command_rec geoip_cmds[] = {
  { "GeoIPDBFile", geoip_set_filename, NULL,
    OR_ALL, TAKE1, "GeoIP Data File" },
  { "GeoIPEnable", geoip_enable, NULL,
    OR_ALL, FLAG, "Enable mod_geoip" },
  { "GeoIPOutput", geoip_set_output, NULL,
    OR_ALL, ITERATE, "Specify output method(s)" },
  { "GeoIPFlags", geoip_set_flags, NULL,
    OR_ALL, ITERATE, "GeoIP flags" },
  { NULL }
};

static int geoip_post_read_request (request_rec *r) {
  geoip_server_cfg *cfg = (geoip_server_cfg *)ap_get_module_config(r->server->module_config, &geoip_module);
  char *ipaddr;
  short int country_id;
  GeoIPRecord * gir;
  GeoIPRegion * giregion;
  const char *country_code, *country_name;
  unsigned char databaseType;

  if(!cfg->GeoIPEnable)
    return DECLINED;

  ipaddr = r->connection->remote_ip;
  if(!ipaddr)
    return DECLINED;

  if(!cfg->gip) {
    if(cfg->GeoIPFilename != NULL) {
      cfg->gip = GeoIP_open(cfg->GeoIPFilename, cfg->GeoIPFlags);
    } else {
      cfg->gip = GeoIP_new(cfg->GeoIPFlags);
    }
    if(!cfg->gip) {
      ap_log_error(APLOG_MARK, APLOG_ERR, r->server, "[mod_geoip]: Error while opening data file");
      return DECLINED;
    }
  }

  databaseType = GeoIP_database_edition(cfg->gip);
  switch (databaseType) {
  case GEOIP_COUNTRY_EDITION:
    country_id = GeoIP_country_id_by_addr(cfg->gip, ipaddr);

    if (country_id > 0) {
      country_code = GeoIP_country_code[country_id];
      country_name = GeoIP_country_name[country_id];
      if (cfg->GeoIPOutput & GEOIP_NOTES) {
	ap_table_set(r->notes, "GEOIP_COUNTRY_CODE", country_code);
	ap_table_set(r->notes, "GEOIP_COUNTRY_NAME", country_name);
      }
      if (cfg->GeoIPOutput & GEOIP_ENV) {
	ap_table_set(r->subprocess_env, "GEOIP_COUNTRY_CODE", country_code);
	ap_table_set(r->subprocess_env, "GEOIP_COUNTRY_NAME", country_name);
      }
    }
    break;
  case GEOIP_REGION_EDITION:
    giregion = GeoIP_region_by_name (cfg->gip, ipaddr);
    if (giregion != NULL) {
      if (cfg->GeoIPOutput & GEOIP_NOTES) {
        ap_table_set(r->notes, "GEOIP_COUNTRY_CODE", giregion->country_code);
        if (NULL != giregion->region) {
          ap_table_set(r->notes, "GEOIP_REGION", giregion->region);
        }
      }
      if (cfg->GeoIPOutput & GEOIP_ENV) {
        ap_table_set(r->subprocess_env, "GEOIP_COUNTRY_CODE", giregion->country_code);
        if (NULL != giregion->region) {
          ap_table_set(r->subprocess_env, "GEOIP_REGION", giregion->region);
        }
      }
      GeoIPRegion_delete(giregion);
    }
    break;
  case GEOIP_CITY_EDITION_REV0:
  case GEOIP_CITY_EDITION_REV1:
    gir = GeoIP_record_by_addr(cfg->gip, ipaddr);
    if (gir != NULL) {
      if (cfg->GeoIPOutput & GEOIP_NOTES) {
        ap_table_set(r->notes, "GEOIP_COUNTRY_CODE", gir->country_code);
        ap_table_set(r->notes, "GEOIP_COUNTRY_NAME", gir->country_name);
        if (gir->region != NULL)
          ap_table_set(r->notes, "GEOIP_REGION", gir->region);
        if (gir->city != NULL)
          ap_table_set(r->notes, "GEOIP_CITY", gir->city);
      }
      if (cfg->GeoIPOutput & GEOIP_ENV) {
        ap_table_set(r->subprocess_env, "GEOIP_COUNTRY_CODE", gir->country_code);
        ap_table_set(r->subprocess_env, "GEOIP_COUNTRY_NAME", gir->country_name);
        if (gir->region != NULL)
          ap_table_set(r->subprocess_env, "GEOIP_REGION", gir->region);
        if (gir->city != NULL)
          ap_table_set(r->subprocess_env, "GEOIP_CITY", gir->city);
      }
    }
    break;
  }
  return OK;
}

static void geoip_child_exit(server_rec *r, pool *p) {
  geoip_server_cfg *cfg = (geoip_server_cfg *)ap_get_module_config(r->module_config, &geoip_module);
  if(cfg->gip != NULL)
    GeoIP_delete(cfg->gip);
}

module MODULE_VAR_EXPORT geoip_module = {
    STANDARD_MODULE_STUFF,
    NULL,                       /* initializer */
    NULL,                       /* dir config creater */
    NULL,                       /* dir merger --- default is to override */
    geoip_server_config,        /* server config */
    NULL,                       /* merge server configs */
    geoip_cmds,                 /* command table */
    NULL,                       /* handlers */
    NULL,                       /* filename translation */
    NULL,                       /* check_user_id */
    NULL,                       /* check auth */
    NULL,                       /* check access */
    NULL,                       /* type_checker */
    NULL,                       /* fixups */
    NULL,                       /* logger */
    NULL,                       /* header parser */
    NULL,                       /* child_init */
    geoip_child_exit,           /* child_exit */
    geoip_post_read_request,    /* post read-request */
};
