/*
   mkvmerge -- utility for splicing together matroska files
   from component media subtypes

   Distributed under the GPL
   see the file COPYING for details
   or visit http://www.gnu.org/copyleft/gpl.html

   $Id: p_aac.cpp 3838 2008-08-21 15:04:24Z mosu $

   AAC output module

   Written by Moritz Bunkus <moritz@bunkus.org>.
*/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>

#include "aac_common.h"
#include "hacks.h"
#include "matroska.h"
#include "p_aac.h"

using namespace libmatroska;

aac_packetizer_c::aac_packetizer_c(generic_reader_c *_reader,
                                   int _id,
                                   int _profile,
                                   int _samples_per_sec,
                                   int _channels,
                                   track_info_c &_ti,
                                   bool _emphasis_present,
                                   bool _headerless)
  throw (error_c):
  generic_packetizer_c(_reader, _ti),
  bytes_output(0), packetno(0), last_timecode(-1), num_packets_same_tc(0),
  bytes_skipped(0),
  samples_per_sec(_samples_per_sec), channels(_channels), id(_id),
  profile(_profile), headerless(_headerless),
  emphasis_present(_emphasis_present) {

  set_track_type(track_audio);
  set_track_default_duration((int64_t)(1024 * 1000000000.0 / samples_per_sec));
}

aac_packetizer_c::~aac_packetizer_c() {
}

unsigned char *
aac_packetizer_c::get_aac_packet(unsigned long *header,
                                 aac_header_t *aacheader) {
  int pos, i, up_shift, down_shift, size;
  unsigned char *buf, *src, *packet_buffer;

  packet_buffer = byte_buffer.get_buffer();
  size = byte_buffer.get_size();
  pos = find_aac_header(packet_buffer, size, aacheader, emphasis_present);
  if (pos < 0) {
    if (10 < size) {
      bytes_skipped += size - 10;
      byte_buffer.remove(size - 10);
    }
    return NULL;
  }
  if ((pos + aacheader->bytes) > size)
    return NULL;

  bytes_skipped += pos;
  if (verbose && (0 < bytes_skipped))
    mxwarn(FMT_TID "skipping " LLD " bytes (no valid AAC header "
           "found). This might make audio/video go out of sync, but this "
           "stream is damaged.\n", ti.fname.c_str(), (int64_t)0,
           bytes_skipped);
  bytes_skipped = 0;

  if ((aacheader->header_bit_size % 8) == 0)
    buf = (unsigned char *)safememdup(packet_buffer + pos +
                                      aacheader->header_byte_size,
                                      aacheader->data_byte_size);
  else {
    // Header is not byte aligned, i.e. MPEG-4 ADTS
    // This code is from mpeg4ip/server/mp4creator/aac.cpp
    buf = (unsigned char *)safemalloc(aacheader->data_byte_size);

    up_shift = aacheader->header_bit_size % 8;
    down_shift = 8 - up_shift;
    src = packet_buffer + pos + aacheader->header_bit_size / 8;

    buf[0] = src[0] << up_shift;
    for (i = 1; i < aacheader->data_byte_size; i++) {
      buf[i - 1] |= (src[i] >> down_shift);
      buf[i] = (src[i] << up_shift);
    }
  }

  byte_buffer.remove(pos + aacheader->bytes);

  return buf;
}

void
aac_packetizer_c::set_headers() {
  if (!hack_engaged(ENGAGE_OLD_AAC_CODECID))
    set_codec_id(MKV_A_AAC);
  else if (id == AAC_ID_MPEG4) {
    if (profile == AAC_PROFILE_MAIN)
      set_codec_id(MKV_A_AAC_4MAIN);
    else if (profile == AAC_PROFILE_LC)
      set_codec_id(MKV_A_AAC_4LC);
    else if (profile == AAC_PROFILE_SSR)
      set_codec_id(MKV_A_AAC_4SSR);
    else if (profile == AAC_PROFILE_LTP)
      set_codec_id(MKV_A_AAC_4LTP);
    else if (profile == AAC_PROFILE_SBR)
      set_codec_id(MKV_A_AAC_4SBR);
    else
      die("aac_packetizer: Unknown AAC MPEG-4 object type %d.", profile);
  } else {
    if (profile == AAC_PROFILE_MAIN)
      set_codec_id(MKV_A_AAC_2MAIN);
    else if (profile == AAC_PROFILE_LC)
      set_codec_id(MKV_A_AAC_2LC);
    else if (profile == AAC_PROFILE_SSR)
      set_codec_id(MKV_A_AAC_2SSR);
    else if (profile == AAC_PROFILE_SBR)
      set_codec_id(MKV_A_AAC_2SBR);
    else
      die("aac_packetizer: Unknown AAC MPEG-2 profile %d.", profile);
  }
  set_audio_sampling_freq((float)samples_per_sec);
  set_audio_channels(channels);

  if ((ti.private_data != NULL) && (ti.private_size > 0))
    set_codec_private(ti.private_data, ti.private_size);
  else if (!hack_engaged(ENGAGE_OLD_AAC_CODECID)) {
    unsigned char buffer[5];
    int length;

    length = create_aac_data(buffer, profile == AAC_PROFILE_SBR ?
                             AAC_PROFILE_LC : profile,
                             channels, samples_per_sec,
                             profile == AAC_PROFILE_SBR ? samples_per_sec * 2 :
                             samples_per_sec,
                             profile == AAC_PROFILE_SBR);
    set_codec_private(buffer, length);
  }

  generic_packetizer_c::set_headers();
}

int
aac_packetizer_c::process_headerless(packet_cptr packet) {
  int64_t new_timecode;

  if (packet->timecode != -1) {
    new_timecode = packet->timecode;
    if (last_timecode == packet->timecode) {
      num_packets_same_tc++;
      new_timecode += (int64_t)(num_packets_same_tc * 1024 * 1000000000.0 / samples_per_sec);
    } else {
      last_timecode = packet->timecode;
      num_packets_same_tc = 0;
    }
  } else
    new_timecode = (int64_t)(1024 * 1000000000.0 * packetno / samples_per_sec);

  packetno++;
  packet->duration = (int64_t)(1024 * 1000000000.0 / samples_per_sec);
  packet->timecode = new_timecode;

  add_packet(packet);

  return FILE_STATUS_MOREDATA;
}

int
aac_packetizer_c::process(packet_cptr packet) {
  if (headerless)
    return process_headerless(packet);

  unsigned char *aac_packet;
  unsigned long header;
  aac_header_t aacheader;

  byte_buffer.add(packet->data->get(), packet->data->get_size());
  while ((aac_packet = get_aac_packet(&header, &aacheader)) != NULL) {
    int64_t my_timecode = -1 == packet->timecode ? (int64_t)(1024 * 1000000000.0 * packetno / samples_per_sec) : packet->timecode;
    add_packet(new packet_t(new memory_c(aac_packet, aacheader.data_byte_size, true), my_timecode, (int64_t)(1024 * 1000000000.0 / samples_per_sec)));
    packetno++;
  }

  return FILE_STATUS_MOREDATA;
}

connection_result_e
aac_packetizer_c::can_connect_to(generic_packetizer_c *src,
                                 string &error_message) {
  aac_packetizer_c *asrc;

  asrc = dynamic_cast<aac_packetizer_c *>(src);
  if (asrc == NULL)
    return CAN_CONNECT_NO_FORMAT;

  connect_check_a_samplerate(samples_per_sec, asrc->samples_per_sec);
  connect_check_a_channels(channels, asrc->channels);
  if (profile != asrc->profile) {
    error_message = mxsprintf("The AAC profiles are different: %d and %d",
                              (int)profile, (int)asrc->profile);
    return CAN_CONNECT_NO_PARAMETERS;
  }
  return CAN_CONNECT_YES;
}
