/* $Id: milter-regex.c,v 1.11 2003/10/06 08:49:35 dhartmei Exp $ */

/*
 * Copyright (c) 2003 Daniel Hartmeier
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 *    - Redistributions of source code must retain the above copyright
 *      notice, this list of conditions and the following disclaimer.
 *    - Redistributions in binary form must reproduce the above
 *      copyright notice, this list of conditions and the following
 *      disclaimer in the documentation and/or other materials provided
 *      with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT HOLDERS OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
 * ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 */

static const char rcsid[] = "$Id: milter-regex.c,v 1.11 2003/10/06 08:49:35 dhartmei Exp $";

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <ctype.h>
#include <errno.h>
#include <netdb.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <syslog.h>
#include <time.h>
#include <unistd.h>
#include <libmilter/mfapi.h>

#include "rules.h"

static const char	*rule_file_name = "/etc/milter-regex.conf";
static struct ruleset	 ruleset;
static int		 debug = 0;

static sfsistat	 cb_connect(SMFICTX *ctx, char *name, _SOCK_ADDR *sa);
static sfsistat	 cb_helo(SMFICTX *ctx, char *arg);
static sfsistat	 cb_envfrom(SMFICTX *ctx, char **args);
static sfsistat	 cb_envrcpt(SMFICTX *ctx, char **args);
static sfsistat	 cb_header(SMFICTX *ctx, char *name, char *value);
static sfsistat	 cb_eoh(SMFICTX *ctx);
static sfsistat	 cb_body(SMFICTX *ctx, u_char *chunk, size_t size);
static sfsistat	 cb_close(SMFICTX *ctx);
static void	 usage(void);
static void	 reload_ruleset(void);
void		 log(int, const char *, ...);

#define OCONN	"unix:/var/spool/milter-regex"
#define	RCODE	"554"
#define XCODE	"5.7.1"

#ifdef LINUX
#define	ST_MTIME st_mtime
extern size_t	 strlcpy(char *, const char *, size_t);
extern int	 vasprintf(char **, const char *, va_list);
#else
#define	ST_MTIME st_mtimespec
#endif

static void
reload_ruleset(void)
{
	static time_t last_check = 0;
	static struct stat sbo;
	time_t t = time(NULL);

	if (!last_check)
		memset(&sbo, 0, sizeof(sbo));
	if (t - last_check >= 10) {
		struct stat sb;

		last_check = t;
		memset(&sb, 0, sizeof(sb));
		if (stat(rule_file_name, &sb)) {
			log(LOG_ERR, "reload_ruleset: stat: %s: %s",
			    rule_file_name, strerror(errno));
			return;
		}
		if (memcmp(&sb.ST_MTIME, &sbo.ST_MTIME, sizeof(sb.ST_MTIME))) {
			memcpy(&sbo.ST_MTIME, &sb.ST_MTIME, sizeof(sb.ST_MTIME));
			log(LOG_INFO, "loading configuration file");
			free_ruleset(&ruleset);
			memset(&ruleset, 0, sizeof(ruleset));
			if (parse_file(rule_file_name, &ruleset))
				log(LOG_ERR,
				    "reload_ruleset: failed to load "
				    "configuration file");
			else
				log(LOG_INFO,
				    "configuration file loaded successfully");
		}
	}
}

struct context {
	char		 buf[2048];	/* longer body lines are wrapped */
	unsigned	 pos;		/* write position within buf */
};

static void
setreply(SMFICTX *ctx, const struct rule *rule, const char *fmt, ...)
{
	va_list ap;
	char msg[8192];

	switch (rule->action) {
	case SMFIS_REJECT:
		snprintf(msg, sizeof(msg), "REJECT %s ", rule->message);
		break;
	case SMFIS_TEMPFAIL:
		snprintf(msg, sizeof(msg), "TEMPFAIL %s ", rule->message);
		break;
	case SMFIS_DISCARD:
		strlcpy(msg, "DISCARD ", sizeof(msg));
		break;
	default:
		return;
	}
	va_start(ap, fmt);
	vsnprintf(msg + strlen(msg), sizeof(msg) - strlen(msg), fmt, ap);
	va_end(ap);
	log(LOG_INFO, "%s", msg);
	if (rule->action == SMFIS_REJECT ||
	    rule->action == SMFIS_TEMPFAIL)
		if (smfi_setreply(ctx, RCODE, XCODE, (char *)rule->message) !=
		    MI_SUCCESS)
			log(LOG_ERR, "smfi_setreply");
}

static sfsistat
cb_connect(SMFICTX *ctx, char *name, _SOCK_ADDR *sa)
{
	char host[64];
	const struct rule *rule;
	struct context *context;

	strlcpy(host, "unknown", sizeof(host));
	switch (sa->sa_family) {
	case AF_INET: {
		struct sockaddr_in *sin = (struct sockaddr_in *)sa;

		if (inet_ntop(AF_INET, &sin->sin_addr.s_addr, host,
		    sizeof(host)) == NULL)
			log(LOG_ERR, "cb_connect: inet_ntop: %s",
			    strerror(errno));
		break;
	}
	case AF_INET6: {
		struct sockaddr_in6 *sin6 = (struct sockaddr_in6 *)sa;

		if (inet_ntop(AF_INET6, &sin6->sin6_addr, host,
		    sizeof(host)) == NULL)
			log(LOG_ERR, "cb_connect: inet_ntop: %s",
			    strerror(errno));
		break;
	}
	}
	log(LOG_DEBUG, "cb_connect('%s', '%s')", name, host);
	reload_ruleset();
	rule = match_rules(ruleset.connect, name, host);
	if (rule != NULL) {
		setreply(ctx, rule, "connect '%s' '%s'", name, host);
		return (rule->action);
	} else {
		if (ruleset.helo == NULL &&
		    ruleset.envfrom == NULL &&
		    ruleset.envrcpt == NULL &&
		    ruleset.header == NULL &&
		    ruleset.body == NULL) {
			log(LOG_DEBUG, "ACCEPT");
			return (SMFIS_ACCEPT);
		}
	}
	context = (struct context *)malloc(sizeof(*context));
	if (context == NULL) {
		log(LOG_ERR, "cb_connect: malloc: %s", strerror(errno));
		return (SMFIS_ACCEPT);
	}
	if (smfi_setpriv(ctx, context) != MI_SUCCESS) {
		free(context);
		log(LOG_ERR, "cb_connect: smfi_setpriv");
		return (SMFIS_ACCEPT);
	}
	return (SMFIS_CONTINUE);
}

static sfsistat
cb_helo(SMFICTX *ctx, char *arg)
{
	const struct rule *rule;

	log(LOG_DEBUG, "cb_helo('%s')", arg);
	rule = match_rules(ruleset.helo, arg, NULL);
	if (rule != NULL) {
		setreply(ctx, rule, "helo '%s'", arg);
		return (rule->action);
	} else {
		if (ruleset.envfrom == NULL &&
		    ruleset.envrcpt == NULL &&
		    ruleset.header == NULL &&
		    ruleset.body == NULL) {
			log(LOG_DEBUG, "ACCEPT");
			return (SMFIS_ACCEPT);
		}
	}
	return (SMFIS_CONTINUE);
}

static sfsistat
cb_envfrom(SMFICTX *ctx, char **args)
{
	const struct rule *rule;

	if (args[0] == NULL) {
		log(LOG_ERR, "cb_envfrom: args[0] == NULL");
		return (SMFIS_ACCEPT);
	}
	log(LOG_DEBUG, "cb_envfrom('%s')", args[0]);
	rule = match_rules(ruleset.envfrom, args[0], NULL);
	if (rule != NULL) {
		setreply(ctx, rule, "envfrom '%s'", args[0]);
		return (rule->action);
	} else {
		if (ruleset.envrcpt == NULL &&
		    ruleset.header == NULL &&
		    ruleset.body == NULL) {
			log(LOG_DEBUG, "ACCEPT");
			return (SMFIS_ACCEPT);
		}
	}
	return (SMFIS_CONTINUE);
}

static sfsistat
cb_envrcpt(SMFICTX *ctx, char **args)
{
	const struct rule *rule;

	if (args[0] == NULL) {
		log(LOG_ERR, "cb_envrcpt: args[0] == NULL");
		return (SMFIS_ACCEPT);
	}
	log(LOG_DEBUG, "cb_envrcpt('%s')", args[0]);
	rule = match_rules(ruleset.envrcpt, args[0], NULL);
	if (rule != NULL) {
		setreply(ctx, rule, "envrcpt '%s'", args[0]);
		return (rule->action);
	} else {
		if (ruleset.header == NULL &&
		    ruleset.body == NULL) {
			log(LOG_DEBUG, "ACCEPT");
			return (SMFIS_ACCEPT);
		}
	}
	return (SMFIS_CONTINUE);
}

static sfsistat
cb_header(SMFICTX *ctx, char *name, char *value)
{
	const struct rule *rule;

	log(LOG_DEBUG, "cb_header('%s', '%s')", name, value);
	rule = match_rules(ruleset.header, name, value);
	if (rule != NULL) {
		setreply(ctx, rule, "header '%s' '%s'", name, value);
		return (rule->action);
	}
	return (SMFIS_CONTINUE);
}

static sfsistat
cb_eoh(SMFICTX *ctx)
{
	struct context *context;

	log(LOG_DEBUG, "cb_eoh()");
	context = (struct context *)smfi_getpriv(ctx);
	if (context == NULL) {
		log(LOG_ERR, "cb_eoh: smfi_getpriv");
		return (SMFIS_ACCEPT);
	}
	memset(context->buf, 0, sizeof(context->buf));
	context->pos = 0;
	if (ruleset.body == NULL) {
		log(LOG_DEBUG, "ACCEPT");
		return (SMFIS_ACCEPT);
	}
	return (SMFIS_CONTINUE);
}

static sfsistat
cb_body(SMFICTX *ctx, u_char *chunk, size_t size)
{
	struct context *context;

	log(LOG_DEBUG, "cb_body()");
	context = (struct context *)smfi_getpriv(ctx);
	if (context == NULL) {
		log(LOG_ERR, "cb_body: smfi_getpriv");
		return (SMFIS_ACCEPT);
	}
	for (; size > 0; size--, chunk++) {
		context->buf[context->pos] = *chunk;
		if (context->buf[context->pos] == '\n' ||
		    context->pos == sizeof(context->buf) - 1) {
			const struct rule *rule;

			if (context->pos > 0 &&
			    context->buf[context->pos - 1] == '\r')
				context->buf[context->pos - 1] = 0;
			else
				context->buf[context->pos] = 0;
			context->pos = 0;
			rule = match_rules(ruleset.body, context->buf, NULL);
			if (rule != NULL) {
				setreply(ctx, rule, "body '%s'", context->buf);
				return (rule->action);
			}
		} else
			context->pos++;
	}
	return (SMFIS_CONTINUE);
}

static sfsistat
cb_close(SMFICTX *ctx)
{
	struct context *context;

	log(LOG_DEBUG, "cb_close()");
	context = (struct context *)smfi_getpriv(ctx);
	if (context != NULL) {
		smfi_setpriv(ctx, NULL);
		free(context);
	}
	return (SMFIS_CONTINUE);
}

struct smfiDesc smfilter = {
	"milter-regex",	/* filter name */
	SMFI_VERSION,	/* version code -- do not change */
	SMFIF_ADDHDRS|SMFIF_CHGHDRS|SMFIF_ADDRCPT|SMFIF_DELRCPT, /* flags */
	cb_connect,	/* connection info filter */
	cb_helo,	/* SMTP HELO command filter */
	cb_envfrom,	/* envelope sender filter */
	cb_envrcpt,	/* envelope recipient filter */
	cb_header,	/* header filter */
	cb_eoh,		/* end of header */
	cb_body,	/* body block */
	NULL,		/* end of message */
	NULL,		/* message aborted */
	cb_close	/* connection cleanup */
};

void
log(int priority, const char *fmt, ...)
{
	va_list ap;
	char *msg = NULL;

	va_start(ap, fmt);
	if (vasprintf(&msg, fmt, ap) != -1) {
		if (debug)
			printf("%s\n", msg);
		syslog(priority, "%s", msg);
		free(msg);
	}
	va_end(ap);
}

static void
usage(void)
{
	extern char *__progname;

	fprintf(stderr, "usage: %s [-d] [-c config] [-p pipe]\n", __progname);
	exit(1);
}

void
die(const char *reason)
{
	log(LOG_ERR, "die: %s", reason);
	smfi_stop();
	sleep(60);
	/* not reached, smfi_stop() kills thread */
	abort();
}

int
main(int argc, char **argv)
{
	int ch;
	char *oconn = OCONN;
	sfsistat r = MI_FAILURE;

	tzset();
	openlog("milter-regex", LOG_PID | LOG_NDELAY, LOG_DAEMON);

	while ((ch = getopt(argc, argv, "dc:p:")) != -1) {
		switch (ch) {
		case 'd':
			debug = 1;
			break;
		case 'c':
			rule_file_name = optarg;
			break;
		case 'p':
			oconn = optarg;
			break;
		default:
			usage();
		}
	}
	if (argc != optind) {
		fprintf(stderr, "unknown command line argument: %s ...",
		    argv[optind]);
		usage();
	}

	memset(&ruleset, 0, sizeof(ruleset));
	if (filter_init())
		goto done;

	log(LOG_INFO, "started");

	if (smfi_setconn(oconn) != MI_SUCCESS) {
		log(LOG_ERR, "smfi_setconn: %s", oconn);
		goto done;
	}
	if (!strncmp(oconn, "unix:", 5))
		unlink(oconn + 5);
	else if (!strncmp(oconn, "local:", 6))
		unlink(oconn + 6);

	if (smfi_register(smfilter) != MI_SUCCESS) {
		log(LOG_ERR, "smfi_register");
		goto done;
	}

	if (!debug && daemon(0, 0)) {
		log(LOG_ERR, "daemon: %s", strerror(errno));
		goto done;
	}

	log(LOG_DEBUG, "calling smfi_main()");
	r = smfi_main();

done:
	if (r == MI_SUCCESS)
		log(LOG_INFO, "terminating gracefully");
	else
		log(LOG_ERR, "terminating due to error");
	return (r);
}
