/* $Id: mreadline.c,v 1.11 2001/06/17 21:56:27 kuhlmann Exp $ */

/*****************************************************
 * mreadline - small line editing and history code
 * Copyright (C) 1998 Sergey Shkonda (serg@bcs.zp.ua)
 *
 * This software is provided AS IS to be used in
 * whatever way you see fit and is placed in the
 * public domain.
 *
 * Author : Sergey Shkonda Nov 27, 1998
 * Changes:
 * * Lalo Martins (lalo@webcom.com) Feb 26, 1999
 *   added tab completion (added get_tab() and changed
 *   R_process_input()
 * * Lalo Martins (lalo@webcom.com) Feb 26, 1999
 *   added more editing commands: delete (as VEOF),
 *   ^A, ^E, ^K, ^U and ^Y, all in R_process_input()
 *****************************************************/

#include "micq.h"

#ifdef USE_MREADLINE

#include "mreadline.h"
#include "util_ui.h"
#include "util.h"
#include "ui.h"
#include "tabs.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <signal.h>

#if HAVE_UNISTD_H
#include <unistd.h>
#endif
#if HAVE_TERMIOS_H
#include <termios.h>
#endif

#define HISTORY_LINES 10
#define HISTORY_LINE_LEN 1024

static struct termios t_attr;
static void tty_prepare (void);
static void tty_restore (void);

static void R_process_input_backspace (void);
static void R_process_input_delete (void);

static RETSIGTYPE micq_ttystop_handler (int);
static RETSIGTYPE micq_cont_handler (int);

static char *history[HISTORY_LINES + 1];
static int history_cur = 0;
static char s[HISTORY_LINE_LEN];
static char y[HISTORY_LINE_LEN];
static int cpos = 0;
static int clen = 0;
static int istat = 0;

void R_init (void)
{
    int k;
    static int inited = 0;

    if (inited)
        return;
    for (k = 0; k < HISTORY_LINES + 1; k++)
    {
        history[k] = (char *) malloc (HISTORY_LINE_LEN);
        history[k][0] = 0;
    }
    s[0] = 0;
    inited = 1;
    signal (SIGTSTP, &micq_ttystop_handler);
    signal (SIGCONT, &micq_cont_handler);
    tty_prepare ();
    atexit (tty_restore);
}

static RETSIGTYPE micq_ttystop_handler (int a)
{
    tty_restore ();
    signal (SIGTSTP, SIG_DFL);
    raise (SIGTSTP);
}

static RETSIGTYPE micq_cont_handler (int a)
{
    tty_prepare ();
    R_redraw ();
    signal (SIGTSTP, &micq_ttystop_handler);
    signal (SIGCONT, &micq_cont_handler);
}

void R_pause (void)
{
    tty_restore ();
}

void R_resume (void)
{
    tty_prepare ();
}

void R_process_input_backspace (void)
{
    if (cpos)
    {
        clen--;
        cpos--;
        strcpy (s + cpos, s + cpos + 1);
#ifndef ANSI_COLOR
        {
            int i;
            print ("\b%s", s + cpos);
            for (i = clen - cpos; i; i--)
                print ("\b");
        }
#else
        printf ("\b\033[K%s", s + cpos);
        if (cpos < clen)
            printf ("\033[%dD", clen - cpos);
#endif
    }
}

void R_process_input_delete (void)
{
    if (cpos < clen)
    {
        clen--;
        strcpy (s + cpos, s + cpos + 1);
#ifndef ANSI_COLOR
        {
            int i;
            print (s + cpos);
            for (i = clen - cpos; i; i--)
                print ("\b");
        }
#else
        printf ("\033[K%s", s + cpos);
        if (cpos < clen)
            printf ("\033[%dD", clen - cpos);
#endif
    }
}

void R_process_input_tab (void)
{
    UDWORD uin;

    if (strpbrk (s, UIN_DELIMS) && strpbrk (s, UIN_DELIMS) - s < strlen (s) - 1)
    {
        M_print ("\a");
        return;
    }
    if (strncmp (s, message_cmd, strlen (s) < strlen (message_cmd) ? strlen (s) : strlen (message_cmd)))
    {
        M_print ("\a");
        return;
    }

    if ((uin = TabGetNext ()))
        sprintf (s, "%s %s/", message_cmd, UIN2Name (uin));
    else
        sprintf (s, "%s ", message_cmd);

    clen = cpos = strlen (s);
    R_undraw ();
    R_redraw ();
}

int R_process_input (void)
{
    char ch;
    int k;
    char s1[HISTORY_LINE_LEN];

    if (!read (STDIN_FILENO, &ch, 1))
        return 0;
    if (!istat)
    {
        if (ch == t_attr.c_cc[VERASE] && t_attr.c_cc[VERASE] != _POSIX_VDISABLE)
        {
            if (del_is_bs)
                R_process_input_backspace ();
            else
                R_process_input_delete ();
            return 0;
        }
        if (ch == t_attr.c_cc[VEOF] && t_attr.c_cc[VERASE] != _POSIX_VDISABLE)
        {
            if (clen)
            {
                R_process_input_delete ();
                return 0;
            }
            strcpy (s, "q");
            printf ("\n");
            return 1;
        }
        if (ch == t_attr.c_cc[VKILL] && t_attr.c_cc[VERASE] != _POSIX_VDISABLE)
        {
            strcpy (y, s);
            s[0] = '\0';
            cpos = clen = 0;
            R_undraw ();
            R_redraw ();
            return 0;
        }
#ifdef    VREPRINT
        if (ch == t_attr.c_cc[VREPRINT] && t_attr.c_cc[VERASE] != _POSIX_VDISABLE)
        {
            R_undraw ();
            R_redraw ();
            return 0;
        }
#endif /* VREPRINT */
        if (ch == '\t' && t_attr.c_cc[VERASE] != _POSIX_VDISABLE)
        {
            R_process_input_tab ();
            return 0;
        }
        if (ch >= 0 && ch < ' ')
        {
            switch (ch)
            {
                case 1:        /* ^A */
                    printf ("\033[%dD", cpos);
                    cpos = 0;
                    break;
                case 5:        /* ^E */
                    printf ("\033[%dC", clen - cpos);
                    cpos = clen;
                    break;
                case 8:        /* ^H = \b */
                    R_process_input_backspace ();
                    return 0;
                case 11:       /* ^K, as requested by Bernhard Sadlowski */
                    clen = cpos - 1;
                    printf ("\033[K");
                    break;
                case '\n':
                case '\r':
                    s[clen + 1] = 0;    /* just to be sure */
                    M_print ("\n");
                    history_cur = 0;
                    TabReset ();
                    strcpy (history[0], s);
                    if (!s[0])
                        return 1;
                    if (strcmp (s, history[1]))
                        for (k = HISTORY_LINES; k; k--)
                            strcpy (history[k], history[k - 1]);
                    return 1;
                case 12:       /* ^L */
                    system ("clear");
                    R_redraw ();
                    break;
                case 25:       /* ^Y */
                    strcpy (s, y);
                    clen = cpos = strlen (s);
                    R_undraw ();
                    R_redraw ();
#ifdef ANSI_COLOR
                case 27:       /* ESC */
                    istat = 1;
                    break;
#endif
            }
        }
        else if (clen + 1 < HISTORY_LINE_LEN)
        {
            printf ("%c", ch);
#ifdef ANSI_COLOR
            printf ("%s", s + cpos);
            strcpy (s1, s + cpos);
            strcpy (s + cpos + 1, s1);
            if (cpos < clen)
                printf ("\033[%dD", clen - cpos);
#endif
            s[cpos++] = ch;
            clen++;
            s[clen] = 0;
        }
        return 0;
    }
#ifdef ANSI_COLOR
    switch (istat)
    {
        case 1:                /* ESC */
            if (ch == '[' || ch == 'O')
                istat = 2;
            else
                istat = 0;
            break;
        case 2:                /* CSI */
            istat = 0;
            switch (ch)
            {
                case 'A':      /* Up key */
                case 'B':      /* Down key */
                    if ((ch == 'A' && history_cur >= HISTORY_LINES)
                        || (ch == 'B' && history_cur == 0))
                        break;
                    k = history_cur;
                    strcpy (history[history_cur], s);
                    if (ch == 'A')
                        history_cur++;
                    else
                        history_cur--;
                    if (history[history_cur][0] || history_cur == 0)
                    {
                        strcpy (s, history[history_cur]);
                        cpos = clen = strlen (s);
                        R_undraw ();
                        R_redraw ();
                    }
                    else
                    {
                        history_cur = k;
                    }
                    break;
                case 'C':      /* Right key */
                    if (cpos == clen)
                        break;
                    cpos++;
                    printf ("\033[C");
                    break;
                case 'D':      /* Left key */
                    if (!cpos)
                        break;
                    cpos--;
                    printf ("\033[D");
                    break;
                case '3':      /* ESC [ 3 ~ = Delete */
                    istat = 3;
                    break;
                default:
                    printf ("\007");
            }
            break;
        case 3:                /* Del Key */
            istat = 0;
            switch (ch)
            {
                case '~':      /* Del Key */
                    R_process_input_delete ();
                    break;
                default:
                    printf ("\007");
            }
    }
#endif
    return 0;
}

void R_redraw (void)
{
    R_prompt ();
    printf ("%s", s);
#ifdef ANSI_COLOR
    if (cpos != clen)
        printf ("\033[%dD", clen - cpos);
#endif
}

void R_getline (char *buf, int len)
{
    strncpy (buf, s, len);
    cpos = 0;
    clen = 0;
    s[0] = 0;
}

static const char *curprompt = NULL;
void R_setprompt (const char *prompt)
{
    curprompt = prompt;
}

void R_prompt (void)
{
    if (curprompt)
        M_print (curprompt);
}

void R_doprompt (const char *prompt)
{
    R_setprompt (prompt);
/*    M_print( "\n\a" );*/
    R_prompt ();
}

void R_undraw ()
{
    M_print ("\r");             /* for tab stop reasons */
    printf ("\033[K");
}

static struct termios saved_attr;
static int attrs_saved = 0;

static void tty_restore (void)
{
    if (!attrs_saved)
        return;
#if HAVE_TCGETATTR
    if (tcsetattr (STDIN_FILENO, TCSAFLUSH, &saved_attr) != 0)
        perror ("can't restore tty modes");
    else
#endif
        attrs_saved = 0;
}

static void tty_prepare (void)
{
    istat = 0;
#if HAVE_TCGETATTR
    if (tcgetattr (STDIN_FILENO, &t_attr) != 0)
        return;
    if (!attrs_saved)
    {
        saved_attr = t_attr;
        attrs_saved = 1;
    }

    t_attr.c_lflag &= ~(ECHO | ICANON);
    t_attr.c_cc[VMIN] = 1;
    if (tcsetattr (STDIN_FILENO, TCSAFLUSH, &t_attr) != 0)
        perror ("can't change tty modes");
#endif
}


#endif /* USE_MREADLINE */
