/*
 *  Mico --- an Open Source CORBA implementation
 *  Copyright Frank Pilhofer
 *
 *  This file is part of MicoCCM, written by Frank Pilhofer.
 *  The MICO CORBA Component Project was sponsored by Alcatel.
 */

#include <CORBA.h>
#include <mico/ir3.h>
#include <mico/ir_creator.h>
#include <mico/util.h>
#include <mico/os-misc.h>
#include "db.h"
#include "parser.h"
#include "params.h"
#include "idlparser.h"

/*
 * IDL 3 to IDL 2 Transformer
 */

class CCMTransformer {
public:
  CCMTransformer (DB & db, CORBA::Repository_ptr targetrepo);

  bool transform (CORBA::Container_ptr src);

private:
  CORBA::Contained_ptr   transform_Contained (CORBA::Contained_ptr src);
  CORBA::Container_ptr   transform_Container (CORBA::Container_ptr src);
  CORBA::IDLType_ptr     transform_IDLType   (CORBA::IDLType_ptr src);

  CORBA::ModuleDef_ptr   transform_Module    (CORBA::ModuleDef_ptr src);
  CORBA::ConstantDef_ptr transform_Constant  (CORBA::ConstantDef_ptr src);
  CORBA::StructDef_ptr   transform_Struct    (CORBA::StructDef_ptr src);
  CORBA::UnionDef_ptr    transform_Union     (CORBA::UnionDef_ptr src);
  CORBA::EnumDef_ptr     transform_Enum      (CORBA::EnumDef_ptr src);
  CORBA::AliasDef_ptr    transform_Alias     (CORBA::AliasDef_ptr src);
  CORBA::ExceptionDef_ptr transform_Exception (CORBA::ExceptionDef_ptr src);
  CORBA::AttributeDef_ptr transform_Attribute (CORBA::AttributeDef_ptr src);
  CORBA::OperationDef_ptr transform_Operation (CORBA::OperationDef_ptr src);
  CORBA::InterfaceDef_ptr transform_Interface (CORBA::InterfaceDef_ptr src);
  CORBA::ValueMemberDef_ptr transform_ValueMember (CORBA::ValueMemberDef_ptr src);
  CORBA::ValueDef_ptr     transform_Value     (CORBA::ValueDef_ptr src);
  CORBA::ValueBoxDef_ptr  transform_ValueBox  (CORBA::ValueBoxDef_ptr src);

  CORBA::OperationDef_ptr
  transform_Provides (CORBA::ComponentIR::ProvidesDef_ptr src);

  CORBA::OperationDef_ptr
  transform_Uses (CORBA::ComponentIR::UsesDef_ptr src);

  CORBA::OperationDef_ptr
  transform_Emits (CORBA::ComponentIR::EmitsDef_ptr src);

  CORBA::OperationDef_ptr
  transform_Publishes (CORBA::ComponentIR::PublishesDef_ptr src);

  CORBA::OperationDef_ptr
  transform_Consumes (CORBA::ComponentIR::ConsumesDef_ptr src);

  CORBA::InterfaceDef_ptr
  transform_Component (CORBA::ComponentIR::ComponentDef_ptr src);

  CORBA::OperationDef_ptr
  transform_Factory (CORBA::ComponentIR::FactoryDef_ptr src);

  CORBA::OperationDef_ptr
  transform_Finder (CORBA::ComponentIR::FinderDef_ptr src);

  CORBA::InterfaceDef_ptr
  transform_Home (CORBA::ComponentIR::HomeDef_ptr src);

  DB & _db;
  CORBA::Repository_var _repo;
};

CCMTransformer::CCMTransformer (DB & db, CORBA::Repository_ptr targetrepo)
  : _db (db)
{
  _repo = CORBA::Repository::_duplicate (targetrepo);
}

/*
 * Traverse the Interface Repository in Pre-Order
 */

bool
CCMTransformer::transform (CORBA::Container_ptr src)
{
  CORBA::Contained_var cs = CORBA::Contained::_narrow (src);

  if (!CORBA::is_nil (cs)) {
    CORBA::Contained_var tc = transform_Contained (cs);
  }

  CORBA::ContainedSeq_var contents = src->contents (CORBA::dk_all, 1);
  CORBA::Contained_var dummy1;
  CORBA::Container_var dummy2;

  for (CORBA::ULong i=0; i<contents->length(); i++) {
    dummy1 = transform_Contained (contents[i]);
    dummy2 = CORBA::Container::_narrow (contents[i]);
    if (!CORBA::is_nil (dummy2)) {
      if (!transform (dummy2)) {
	return false;
      }
    }
  }

  return true;
}

/*
 * Meta-Transformers
 */

CORBA::Contained_ptr
CCMTransformer::transform_Contained (CORBA::Contained_ptr src)
{
  CORBA::Contained_ptr res;

  switch (src->def_kind()) {
  case CORBA::dk_Module: {
    CORBA::ModuleDef_var el = CORBA::ModuleDef::_narrow (src);
    res = transform_Module (el);
    break;
  }

  case CORBA::dk_Constant: {
    CORBA::ConstantDef_var el = CORBA::ConstantDef::_narrow (src);
    res = transform_Constant (el);
    break;
  }

  case CORBA::dk_Struct: {
    CORBA::StructDef_var el = CORBA::StructDef::_narrow (src);
    res = transform_Struct (el);
    break;
  }

  case CORBA::dk_Union: {
    CORBA::UnionDef_var el = CORBA::UnionDef::_narrow (src);
    res = transform_Union (el);
    break;
  }

  case CORBA::dk_Enum: {
    CORBA::EnumDef_var el = CORBA::EnumDef::_narrow (src);
    res = transform_Enum (el);
    break;
  }

  case CORBA::dk_Alias: {
    CORBA::AliasDef_var el = CORBA::AliasDef::_narrow (src);
    res = transform_Alias (el);
    break;
  }

  case CORBA::dk_Exception: {
    CORBA::ExceptionDef_var el = CORBA::ExceptionDef::_narrow (src);
    res = transform_Exception (el);
    break;
  }

  case CORBA::dk_Attribute: {
    CORBA::AttributeDef_var el = CORBA::AttributeDef::_narrow (src);
    res = transform_Attribute (el);
    break;
  }

  case CORBA::dk_Operation: {
    CORBA::OperationDef_var el = CORBA::OperationDef::_narrow (src);
    res = transform_Operation (el);
    break;
  }

  case CORBA::dk_Interface: 
  case CORBA::dk_AbstractInterface: 
  case CORBA::dk_LocalInterface: {
    CORBA::InterfaceDef_var el = CORBA::InterfaceDef::_narrow (src);
    res = transform_Interface (el);
    break;
  }

  case CORBA::dk_ValueMember: {
    CORBA::ValueMemberDef_var el = CORBA::ValueMemberDef::_narrow (src);
    res = transform_ValueMember (el);
    break;
  }

  case CORBA::dk_Value: {
    CORBA::ValueDef_var el = CORBA::ValueDef::_narrow (src);
    res = transform_Value (el);
    break;
  }

  case CORBA::dk_ValueBox: {
    CORBA::ValueBoxDef_var el = CORBA::ValueBoxDef::_narrow (src);
    res = transform_ValueBox (el);
    break;
  }

  case CORBA::dk_Provides: {
    CORBA::ComponentIR::ProvidesDef_var el =
      CORBA::ComponentIR::ProvidesDef::_narrow (src);
    res = transform_Provides (el);
    break;
  }

  case CORBA::dk_Uses: {
    CORBA::ComponentIR::UsesDef_var el =
      CORBA::ComponentIR::UsesDef::_narrow (src);
    res = transform_Uses (el);
    break;
  }

  case CORBA::dk_Emits: {
    CORBA::ComponentIR::EmitsDef_var el =
      CORBA::ComponentIR::EmitsDef::_narrow (src);
    res = transform_Emits (el);
    break;
  }

  case CORBA::dk_Publishes: {
    CORBA::ComponentIR::PublishesDef_var el =
      CORBA::ComponentIR::PublishesDef::_narrow (src);
    res = transform_Publishes (el);
    break;
  }

  case CORBA::dk_Consumes: {
    CORBA::ComponentIR::ConsumesDef_var el =
      CORBA::ComponentIR::ConsumesDef::_narrow (src);
    res = transform_Consumes (el);
    break;
  }

  case CORBA::dk_Component: {
    CORBA::ComponentIR::ComponentDef_var el =
      CORBA::ComponentIR::ComponentDef::_narrow (src);
    res = transform_Component (el);
    break;
  }

  case CORBA::dk_Factory: {
    CORBA::ComponentIR::FactoryDef_var el =
      CORBA::ComponentIR::FactoryDef::_narrow (src);
    res = transform_Factory (el);
    break;
  }

  case CORBA::dk_Finder: {
    CORBA::ComponentIR::FinderDef_var el =
      CORBA::ComponentIR::FinderDef::_narrow (src);
    res = transform_Finder (el);
    break;
  }

  case CORBA::dk_Home: {
    CORBA::ComponentIR::HomeDef_var el =
      CORBA::ComponentIR::HomeDef::_narrow (src);
    res = transform_Home (el);
    break;
  }

  default:
    assert (0);
    break;
  }

  assert (!CORBA::is_nil (res));
  return res;
}

CORBA::Container_ptr
CCMTransformer::transform_Container (CORBA::Container_ptr src)
{
  CORBA::Contained_var cs = CORBA::Contained::_narrow (src);
  CORBA::Container_ptr target;

  if (!CORBA::is_nil (cs)) {
    CORBA::Contained_var ct = transform_Contained (cs);
    target = CORBA::Container::_narrow (ct);
  }
  else {
    target = CORBA::Container::_duplicate (_repo);
  }

  assert (!CORBA::is_nil (target));
  return target;
}

CORBA::IDLType_ptr
CCMTransformer::transform_IDLType (CORBA::IDLType_ptr src)
{
  CORBA::Contained_var cs = CORBA::Contained::_narrow (src);
  CORBA::IDLType_ptr target;

  if (!CORBA::is_nil (cs)) {
    CORBA::Contained_var ct = transform_Contained (cs);
    target = CORBA::IDLType::_narrow (ct);
  }
  else {
    /*
     * Handle anonymous types
     */

    switch (src->def_kind()) {
    case CORBA::dk_Primitive: {
      CORBA::PrimitiveDef_var el = CORBA::PrimitiveDef::_narrow (src);
      target = _repo->get_primitive (el->kind ());
      break;
    }

    case CORBA::dk_String: {
      CORBA::StringDef_var el = CORBA::StringDef::_narrow (src);
      target = _repo->create_string (el->bound());
      break;
    }

    case CORBA::dk_Wstring: {
      CORBA::WstringDef_var el = CORBA::WstringDef::_narrow (src);
      target = _repo->create_wstring (el->bound());
      break;
    }

    case CORBA::dk_Fixed: {
      CORBA::FixedDef_var el = CORBA::FixedDef::_narrow (src);
      target = _repo->create_fixed (el->digits(), el->scale());
      break;
    }

    case CORBA::dk_Sequence: {
      CORBA::SequenceDef_var el = CORBA::SequenceDef::_narrow (src);
      CORBA::IDLType_var st = el->element_type_def ();
      CORBA::IDLType_var tt = transform_IDLType (st);
      target = _repo->create_sequence (el->bound(), tt);
      break;
    }

    case CORBA::dk_Array: {
      CORBA::ArrayDef_var el = CORBA::ArrayDef::_narrow (src);
      CORBA::IDLType_var st = el->element_type_def ();
      CORBA::IDLType_var tt = transform_IDLType (st);
      target = _repo->create_array (el->length(), tt);
      break;
    }

    default:
      assert (0);
    }
  }

  assert (!CORBA::is_nil (target));
  return target;
}

/*
 * Transformers for Interface Repository nodes
 */

CORBA::ModuleDef_ptr
CCMTransformer::transform_Module (CORBA::ModuleDef_ptr src)
{
  CORBA::String_var id = src->id ();
  CORBA::String_var name = src->name ();
  CORBA::String_var version = src->version ();

  CORBA::Contained_var existing = _repo->lookup_id (id);

  if (!CORBA::is_nil (existing)) {
    return CORBA::ModuleDef::_narrow (existing);
  }

  CORBA::Container_var defin = src->defined_in ();
  CORBA::Container_var dest = transform_Container (defin);

  return dest->create_module (id, name, version);
}

CORBA::ConstantDef_ptr
CCMTransformer::transform_Constant (CORBA::ConstantDef_ptr src)
{
  CORBA::String_var id = src->id ();
  CORBA::String_var name = src->name ();
  CORBA::String_var version = src->version ();

  CORBA::Contained_var existing = _repo->lookup_id (id);

  if (!CORBA::is_nil (existing)) {
    return CORBA::ConstantDef::_narrow (existing);
  }

  CORBA::Container_var defin = src->defined_in ();
  CORBA::Container_var dest = transform_Container (defin);

  CORBA::IDLType_var st = src->type_def ();
  CORBA::IDLType_var tt = transform_IDLType (st);

  CORBA::Any_var value = src->value ();

  return dest->create_constant (id, name, version, tt, value);
}

CORBA::StructDef_ptr
CCMTransformer::transform_Struct (CORBA::StructDef_ptr src)
{
  CORBA::String_var id = src->id ();
  CORBA::String_var name = src->name ();
  CORBA::String_var version = src->version ();
  CORBA::StructDef_ptr res;

  CORBA::Contained_var existing = _repo->lookup_id (id);

  if (!CORBA::is_nil (existing)) {
    return CORBA::StructDef::_narrow (existing);
  }

  CORBA::Container_var defin = src->defined_in ();
  CORBA::Container_var dest = transform_Container (defin);

  CORBA::StructMemberSeq_var sm = src->members ();
  CORBA::StructMemberSeq tm;

  res = dest->create_struct (id, name, version, tm);

  tm.length (sm->length ());

  for (CORBA::ULong i=0; i<sm->length(); i++) {
    tm[i].name = sm[i].name;
    tm[i].type = CORBA::_tc_void;
    tm[i].type_def = transform_IDLType (sm[i].type_def);
  }

  res->members (tm);
  return res;
}

CORBA::UnionDef_ptr
CCMTransformer::transform_Union (CORBA::UnionDef_ptr src)
{
  CORBA::String_var id = src->id ();
  CORBA::String_var name = src->name ();
  CORBA::String_var version = src->version ();

  CORBA::Contained_var existing = _repo->lookup_id (id);
  CORBA::UnionDef_ptr res;

  if (!CORBA::is_nil (existing)) {
    return CORBA::UnionDef::_narrow (existing);
  }

  CORBA::Container_var defin = src->defined_in ();
  CORBA::Container_var dest = transform_Container (defin);

  CORBA::IDLType_var sd = src->discriminator_type_def ();
  CORBA::IDLType_var td = transform_IDLType (sd);

  CORBA::UnionMemberSeq_var sm = src->members ();
  CORBA::UnionMemberSeq tm;

  res = dest->create_union (id, name, version, td, tm);

  tm.length (sm->length ());

  for (CORBA::ULong i=0; i<sm->length(); i++) {
    tm[i].name  = sm[i].name;
    tm[i].label = sm[i].label;
    tm[i].type  = CORBA::_tc_void;
    tm[i].type_def = transform_IDLType (sm[i].type_def);
  }

  res->members (tm);
  return res;
}

CORBA::EnumDef_ptr
CCMTransformer::transform_Enum (CORBA::EnumDef_ptr src)
{
  CORBA::String_var id = src->id ();
  CORBA::String_var name = src->name ();
  CORBA::String_var version = src->version ();

  CORBA::Contained_var existing = _repo->lookup_id (id);

  if (!CORBA::is_nil (existing)) {
    return CORBA::EnumDef::_narrow (existing);
  }

  CORBA::Container_var defin = src->defined_in ();
  CORBA::Container_var dest = transform_Container (defin);

  CORBA::EnumMemberSeq_var em = src->members ();
  
  return dest->create_enum (id, name, version, em);
}

CORBA::AliasDef_ptr
CCMTransformer::transform_Alias (CORBA::AliasDef_ptr src)
{
  CORBA::String_var id = src->id ();
  CORBA::String_var name = src->name ();
  CORBA::String_var version = src->version ();

  CORBA::Contained_var existing = _repo->lookup_id (id);

  if (!CORBA::is_nil (existing)) {
    return CORBA::AliasDef::_narrow (existing);
  }

  CORBA::Container_var defin = src->defined_in ();
  CORBA::Container_var dest = transform_Container (defin);

  CORBA::IDLType_var st = src->original_type_def ();
  CORBA::IDLType_var tt = transform_IDLType (st);

  return dest->create_alias (id, name, version, tt);
}

CORBA::ExceptionDef_ptr
CCMTransformer::transform_Exception (CORBA::ExceptionDef_ptr src)
{
  CORBA::String_var id = src->id ();
  CORBA::String_var name = src->name ();
  CORBA::String_var version = src->version ();

  CORBA::Contained_var existing = _repo->lookup_id (id);

  if (!CORBA::is_nil (existing)) {
    return CORBA::ExceptionDef::_narrow (existing);
  }

  CORBA::Container_var defin = src->defined_in ();
  CORBA::Container_var dest = transform_Container (defin);

  CORBA::StructMemberSeq_var sm = src->members ();
  CORBA::StructMemberSeq tm;
  tm.length (sm->length ());

  for (CORBA::ULong i=0; i<sm->length(); i++) {
    tm[i].name = sm[i].name;
    tm[i].type = CORBA::_tc_void;
    tm[i].type_def = transform_IDLType (sm[i].type_def);
  }

  return dest->create_exception (id, name, version, tm);
}

CORBA::AttributeDef_ptr
CCMTransformer::transform_Attribute (CORBA::AttributeDef_ptr src)
{
  CORBA::String_var id = src->id ();
  CORBA::String_var name = src->name ();
  CORBA::String_var version = src->version ();

  CORBA::Contained_var existing = _repo->lookup_id (id);

  if (!CORBA::is_nil (existing)) {
    return CORBA::AttributeDef::_narrow (existing);
  }

  CORBA::Container_var defin = src->defined_in ();
  CORBA::Container_var dest = transform_Container (defin);
  CORBA::InterfaceDef_var destiface = CORBA::InterfaceDef::_narrow (dest);
  CORBA::ValueDef_var destval = CORBA::ValueDef::_narrow (dest);

  CORBA::IDLType_var st = src->type_def ();
  CORBA::IDLType_var tt = transform_IDLType (st);

  CORBA::AttributeDef_ptr res;

  if (!CORBA::is_nil (destiface)) {
    res = destiface->create_attribute (id, name, version, tt, src->mode());
  }
  else {
    res = destval->create_attribute (id, name, version, tt, src->mode());
  }

  return res;
}

CORBA::OperationDef_ptr
CCMTransformer::transform_Operation (CORBA::OperationDef_ptr src)
{
  CORBA::String_var id = src->id ();
  CORBA::String_var name = src->name ();
  CORBA::String_var version = src->version ();

  CORBA::Contained_var existing = _repo->lookup_id (id);

  if (!CORBA::is_nil (existing)) {
    return CORBA::OperationDef::_narrow (existing);
  }

  CORBA::Container_var defin = src->defined_in ();
  CORBA::Container_var dest = transform_Container (defin);
  CORBA::InterfaceDef_var destiface = CORBA::InterfaceDef::_narrow (dest);
  CORBA::ValueDef_var destval = CORBA::ValueDef::_narrow (dest);

  CORBA::IDLType_var st = src->result_def ();
  CORBA::IDLType_var tt = transform_IDLType (st);

  CORBA::ParDescriptionSeq_var sp = src->params ();
  CORBA::ParDescriptionSeq tp;
  tp.length (sp->length());

  for (CORBA::ULong i1=0; i1<sp->length(); i1++) {
    tp[i1].name = sp[i1].name;
    tp[i1].type = CORBA::_tc_void;
    tp[i1].mode = sp[i1].mode;
    tp[i1].type_def = transform_IDLType (sp[i1].type_def);
  }

  CORBA::OperationMode sm = src->mode ();
  CORBA::ContextIdSeq_var sc = src->contexts ();

  CORBA::ExceptionDefSeq_var se = src->exceptions ();
  CORBA::ExceptionDefSeq te;
  te.length (se->length());

  for (CORBA::ULong i2=0; i2<se->length(); i2++) {
    te[i2] = transform_Exception (se[i2]);
  }

  CORBA::OperationDef_ptr res;

  if (!CORBA::is_nil (destiface)) {
    res = destiface->create_operation (id, name, version, tt, sm, tp, te, sc);
  }
  else {
    res = destval->create_operation (id, name, version, tt, sm, tp, te, sc);
  }

  return res;
}

CORBA::InterfaceDef_ptr
CCMTransformer::transform_Interface (CORBA::InterfaceDef_ptr src)
{
  CORBA::String_var id = src->id ();
  CORBA::String_var name = src->name ();
  CORBA::String_var version = src->version ();

  CORBA::Contained_var existing = _repo->lookup_id (id);

  if (!CORBA::is_nil (existing)) {
    return CORBA::InterfaceDef::_narrow (existing);
  }

  CORBA::Container_var defin = src->defined_in ();
  CORBA::Container_var dest = transform_Container (defin);

  CORBA::InterfaceDefSeq_var sb = src->base_interfaces ();
  CORBA::InterfaceDefSeq tb;
  tb.length (sb->length());

  for (CORBA::ULong i=0; i<sb->length(); i++) {
    CORBA::IDLType_var ttemp = transform_IDLType (sb[i]);
    tb[i] = CORBA::InterfaceDef::_narrow (ttemp);
  }

  CORBA::InterfaceDef_ptr res;

  switch (src->def_kind()) {
  case CORBA::dk_Interface:
    res = dest->create_interface (id, name, version, tb);
    break;

  case CORBA::dk_AbstractInterface: {
    CORBA::AbstractInterfaceDefSeq ab;
    res = dest->create_abstract_interface (id, name, version, ab);
    res->base_interfaces (tb);
    break;
  }

  case CORBA::dk_LocalInterface:
    res = dest->create_local_interface (id, name, version, tb);
    break;

  default:
    assert (0);
    break;
  }

  assert (!CORBA::is_nil (res));
  return res;
}

CORBA::ValueMemberDef_ptr
CCMTransformer::transform_ValueMember (CORBA::ValueMemberDef_ptr src)
{
  CORBA::String_var id = src->id ();
  CORBA::String_var name = src->name ();
  CORBA::String_var version = src->version ();

  CORBA::Contained_var existing = _repo->lookup_id (id);

  if (!CORBA::is_nil (existing)) {
    return CORBA::ValueMemberDef::_narrow (existing);
  }

  CORBA::Container_var defin = src->defined_in ();
  CORBA::Container_var dest = transform_Container (defin);
  CORBA::ValueDef_var destval = CORBA::ValueDef::_narrow (dest);

  CORBA::IDLType_var st = src->type_def ();
  CORBA::IDLType_var tt = transform_IDLType (st);

  return destval->create_value_member (id, name, version, tt, src->access());
}
  
CORBA::ValueDef_ptr
CCMTransformer::transform_Value (CORBA::ValueDef_ptr src)
{
  CORBA::String_var id = src->id ();
  CORBA::String_var name = src->name ();
  CORBA::String_var version = src->version ();

  CORBA::Contained_var existing = _repo->lookup_id (id);

  if (!CORBA::is_nil (existing)) {
    return CORBA::ValueDef::_narrow (existing);
  }

  CORBA::Container_var defin = src->defined_in ();
  CORBA::Container_var dest = transform_Container (defin);

  CORBA::InterfaceDefSeq_var ss = src->supported_interfaces ();
  CORBA::InterfaceDefSeq ts;
  ts.length (ss->length());

  for (CORBA::ULong i1=0; i1<ss->length(); i1++) {
    CORBA::IDLType_var ttemp = transform_IDLType (ss[i1]);
    ts[i1] = CORBA::InterfaceDef::_narrow (ttemp);
  }

  CORBA::InitializerSeq_var si = src->initializers ();
  CORBA::InitializerSeq ti;
  ti.length (si->length());

  for (CORBA::ULong i2=0; i2<si->length(); i2++) {
    ti[i2].name = si[i2].name;
    ti[i2].members.length (si[i2].members.length());

    for (CORBA::ULong j=0; j<si[i2].members.length(); j++) {
      ti[i2].members[j].name = si[i2].members[j].name;
      ti[i2].members[j].type = CORBA::_tc_void;
      ti[i2].members[j].type_def = transform_IDLType (si[i2].members[j].type_def);
    }
  }

  CORBA::ValueDef_var sb = src->base_value ();
  CORBA::ValueDef_var tb = CORBA::ValueDef::_nil ();

  if (!CORBA::is_nil (sb)) {
    CORBA::IDLType_var ttemp = transform_IDLType (sb);
    tb = CORBA::ValueDef::_narrow (ttemp);
  }

  CORBA::ValueDefSeq_var sab = src->abstract_base_values ();
  CORBA::ValueDefSeq tab;
  tab.length (sab->length ());

  for (CORBA::ULong i3=0; i3<sab->length(); i3++) {
    CORBA::IDLType_var ttemp = transform_IDLType (sab[i3]);
    tab[i3] = CORBA::ValueDef::_narrow (ttemp);
  }

  return dest->create_value (id, name, version,
			     src->is_custom (),
			     src->is_abstract (),
			     tb,
			     src->is_truncatable (),
			     tab,
			     ts,
			     ti);
}

CORBA::ValueBoxDef_ptr
CCMTransformer::transform_ValueBox (CORBA::ValueBoxDef_ptr src)
{
  CORBA::String_var id = src->id ();
  CORBA::String_var name = src->name ();
  CORBA::String_var version = src->version ();

  CORBA::Contained_var existing = _repo->lookup_id (id);

  if (!CORBA::is_nil (existing)) {
    return CORBA::ValueBoxDef::_narrow (existing);
  }

  CORBA::Container_var defin = src->defined_in ();
  CORBA::Container_var dest = transform_Container (defin);

  CORBA::IDLType_var st = src->original_type_def ();
  CORBA::IDLType_var tt = transform_IDLType (st);

  return dest->create_value_box (id, name, version, tt);
}

/*
 * Here it finally gets more interesting
 */

CORBA::OperationDef_ptr
CCMTransformer::transform_Provides (CORBA::ComponentIR::ProvidesDef_ptr src)
{
  CORBA::String_var id = src->id ();
  CORBA::String_var name = src->name ();
  CORBA::String_var version = src->version ();

  /*
   * Original:
   *                  provides <interface_type> <name>;
   *
   * Equivalent IDL:
   *                  <interface_type> provide_<name> ();
   */

  /*
   * Compute new name and id for the resulting operation
   */

  string target_name = "provide_";
  target_name += name.in ();

  string target_id = id.in ();
  target_id.replace (target_id.length() - 4 - strlen (name.in()),
		     strlen (name.in()),
		     target_name);

  CORBA::Contained_var existing = _repo->lookup_id (target_id.c_str());

  if (!CORBA::is_nil (existing)) {
    return CORBA::OperationDef::_narrow (existing);
  }

  CORBA::Container_var defin = src->defined_in ();
  CORBA::Container_var dest = transform_Container (defin);
  CORBA::InterfaceDef_var destiface = CORBA::InterfaceDef::_narrow (dest);

  CORBA::IDLType_var st = src->interface_type ();
  CORBA::IDLType_var tt = transform_IDLType (st);

  CORBA::ParDescriptionSeq tp;
  CORBA::ContextIdSeq tc;
  CORBA::ExceptionDefSeq te;

  string origid = id.in();
  string fname = _db.get_fname_for_repoid (origid);
  _db.register_repoid (target_id, fname);

  return destiface->create_operation (target_id.c_str(), target_name.c_str(),
				      version, tt, CORBA::OP_NORMAL,
				      tp, te, tc);
}

CORBA::OperationDef_ptr
CCMTransformer::transform_Uses (CORBA::ComponentIR::UsesDef_ptr src)
{
  CORBA::String_var id = src->id ();
  CORBA::String_var name = src->name ();
  CORBA::String_var version = src->version ();

  CORBA::OperationDef_var tempop;

  CORBA::Container_var defin = src->defined_in ();
  CORBA::Container_var dest = transform_Container (defin);
  CORBA::InterfaceDef_var destiface = CORBA::InterfaceDef::_narrow (dest);

  /*
   * Original:
   *                  uses <interface_type> <name>;
   *
   * Equivalent IDL:
   *                  void connect_<name> (in <interface_type> conxn)
   *                    raises (Components::AlreadyConnected,
   *                            Components::InvalidConnection);
   *
   *                  <interface_type> disconnect_<name> ()
   *                    raises (Components::NoConnection);
   *
   *                  <interface_type> get_connection_<name> ()
   */

  if (!src->is_multiple ()) {
    /*
     * connect_<name> operation
     */

    {
      string target_name = "connect_";
      target_name += name.in ();

      string target_id = id.in ();
      target_id.replace (target_id.length() - 4 - strlen (name.in()),
			 strlen (name.in()),
			 target_name);
      
      CORBA::Contained_var existing = _repo->lookup_id (target_id.c_str());
      
      if (!CORBA::is_nil (existing)) {
	return CORBA::OperationDef::_narrow (existing);
      }
      
      CORBA::IDLType_var tt = _repo->get_primitive (CORBA::pk_void);

      CORBA::IDLType_var spt = src->interface_type ();
      
      CORBA::ParDescriptionSeq tp;
      tp.length (1);
      tp[0].name = CORBA::string_dup ("conxn");
      tp[0].type = CORBA::_tc_void;
      tp[0].type_def = transform_IDLType (spt);
      tp[0].mode = CORBA::PARAM_IN;

      CORBA::ExceptionDefSeq te;
      CORBA::Contained_var tempco;
      te.length (2);

      tempco = _repo->lookup ("Components::AlreadyConnected");
      te[0] = CORBA::ExceptionDef::_narrow (tempco);

      tempco = _repo->lookup ("Components::InvalidConnection");
      te[1] = CORBA::ExceptionDef::_narrow (tempco);

      CORBA::ContextIdSeq tc;

      tempop = destiface->create_operation (target_id.c_str(),
					    target_name.c_str(),
					    version,
					    tt, CORBA::OP_NORMAL,
					    tp, te, tc);

      string origid = id.in();
      string fname = _db.get_fname_for_repoid (origid);
      _db.register_repoid (target_id, fname);
    }

    /*
     * disconnect_<name> operation
     */

    {
      string target_name = "disconnect_";
      target_name += name.in ();

      string target_id = id.in ();
      target_id.replace (target_id.length() - 4 - strlen (name.in()),
			 strlen (name.in()),
			 target_name);
      
      CORBA::IDLType_var st = src->interface_type ();
      CORBA::IDLType_var tt = transform_IDLType (st);
      
      CORBA::ExceptionDefSeq te;
      CORBA::Contained_var tempco;
      te.length (1);

      tempco = _repo->lookup ("Components::NoConnection");
      te[0] = CORBA::ExceptionDef::_narrow (tempco);

      CORBA::ParDescriptionSeq tp;
      CORBA::ContextIdSeq tc;

      tempop = destiface->create_operation (target_id.c_str(),
					    target_name.c_str(),
					    version,
					    tt, CORBA::OP_NORMAL,
					    tp, te, tc);

      string origid = id.in();
      string fname = _db.get_fname_for_repoid (origid);
      _db.register_repoid (target_id, fname);
    }

    /*
     * get_connection_<name> operation
     */

    {
      string target_name = "get_connection_";
      target_name += name.in ();

      string target_id = id.in ();
      target_id.replace (target_id.length() - 4 - strlen (name.in()),
			 strlen (name.in()),
			 target_name);
      
      CORBA::IDLType_var st = src->interface_type ();
      CORBA::IDLType_var tt = transform_IDLType (st);
      
      CORBA::ParDescriptionSeq tp;
      CORBA::ContextIdSeq tc;
      CORBA::ExceptionDefSeq te;

      tempop = destiface->create_operation (target_id.c_str(),
					    target_name.c_str(),
					    version,
					    tt, CORBA::OP_NORMAL,
					    tp, te, tc);

      string origid = id.in();
      string fname = _db.get_fname_for_repoid (origid);
      _db.register_repoid (target_id, fname);
    }
  }

  /*
   * Original:
   *                  uses multiple <interface_type> <name>;
   *
   * Equivalent IDL:
   *                  struct <name>Connection {
   *                    <interface_type> objref;
   *                    Components::Cookie ck;
   *                  };
   *
   *                  typedef sequence<<name>Connection> <name>Connections;
   *
   *                  Components::Cookie
   *                    connect_<name> (in <interface_type> connection)
   *                      raises (Components::ExceededConnectionLimit,
   *                              Components::InvalidConnection);
   *
   *                  <interface_type>
   *                    disconnect_<name> (in Components::Cookie ck)
   *                      raises (Components::InvalidConnection);
   *
   *                  <name>Connections get_connections_<name> ();
   */

  else {
    CORBA::IDLType_var conn;
    CORBA::IDLType_var conns;

    /*
     * struct <name>Connection
     */

    {
      string target_name = name.in();
      target_name += "Connection";

      string target_id = id.in ();
      target_id.replace (target_id.length() - 4 - strlen (name.in()),
			 strlen (name.in()),
			 target_name);
      
      CORBA::Contained_var existing = _repo->lookup_id (target_id.c_str());
      
      if (!CORBA::is_nil (existing)) {
	target_name = "connect_";
	target_name += name.in();
	existing = destiface->lookup (target_name.c_str());
	return CORBA::OperationDef::_narrow (existing);
      }
      
      CORBA::IDLType_var st = src->interface_type ();

      CORBA::StructMemberSeq tms;
      CORBA::Contained_var tempco;
      tms.length (2);

      tms[0].name = CORBA::string_dup ("objref");
      tms[0].type = CORBA::_tc_void;
      tms[0].type_def = transform_IDLType (st);

      tempco = _repo->lookup ("Components::Cookie");

      tms[1].name = CORBA::string_dup ("ck");
      tms[1].type = CORBA::_tc_void;
      tms[1].type_def = CORBA::IDLType::_narrow (tempco);

      conn = dest->create_struct (target_id.c_str(),
				  target_name.c_str(),
				  version,
				  tms);

      string origid = id.in();
      string fname = _db.get_fname_for_repoid (origid);
      _db.register_repoid (target_id, fname);
    }

    /*
     * typedef sequence <<name>Connection> <name>Connections
     */

    {
      string target_name = name.in();
      target_name += "Connections";

      string target_id = id.in ();
      target_id.replace (target_id.length() - 4 - strlen (name.in()),
			 strlen (name.in()),
			 target_name);
      
      CORBA::SequenceDef_var dummy = _repo->create_sequence (0, conn);
      conns = dest->create_alias (target_id.c_str(),
				  target_name.c_str(),
				  version,
				  dummy);

      string origid = id.in();
      string fname = _db.get_fname_for_repoid (origid);
      _db.register_repoid (target_id, fname);
    }

    /*
     * connect_<name> operation
     */

    {
      string target_name = "connect_";
      target_name += name.in ();

      string target_id = id.in ();
      target_id.replace (target_id.length() - 4 - strlen (name.in()),
			 strlen (name.in()),
			 target_name);
      
      CORBA::Contained_var tempco;
      tempco = _repo->lookup ("Components::Cookie");
      CORBA::IDLType_var tt = CORBA::IDLType::_narrow (tempco);

      CORBA::IDLType_var spt = src->interface_type ();
      
      CORBA::ParDescriptionSeq tp;
      tp.length (1);
      tp[0].name = CORBA::string_dup ("connection");
      tp[0].type = CORBA::_tc_void;
      tp[0].type_def = transform_IDLType (spt);
      tp[0].mode = CORBA::PARAM_IN;

      CORBA::ExceptionDefSeq te;
      te.length (2);

      tempco = _repo->lookup ("Components::ExceededConnectionLimit");
      te[0] = CORBA::ExceptionDef::_narrow (tempco);

      tempco = _repo->lookup ("Components::InvalidConnection");
      te[1] = CORBA::ExceptionDef::_narrow (tempco);

      CORBA::ContextIdSeq tc;

      tempop = destiface->create_operation (target_id.c_str(),
					    target_name.c_str(),
					    version,
					    tt, CORBA::OP_NORMAL,
					    tp, te, tc);

      string origid = id.in();
      string fname = _db.get_fname_for_repoid (origid);
      _db.register_repoid (target_id, fname);
    }

    /*
     * disconnect_<name> operation
     */

    {
      string target_name = "disconnect_";
      target_name += name.in ();

      string target_id = id.in ();
      target_id.replace (target_id.length() - 4 - strlen (name.in()),
			 strlen (name.in()),
			 target_name);
      
      CORBA::IDLType_var st = src->interface_type ();
      CORBA::IDLType_var tt = transform_IDLType (st);
      
      CORBA::Contained_var tempco;
      tempco = _repo->lookup ("Components::Cookie");

      CORBA::ParDescriptionSeq tp;
      tp.length (1);
      tp[0].name = CORBA::string_dup ("ck");
      tp[0].type = CORBA::_tc_void;
      tp[0].type_def = CORBA::IDLType::_narrow (tempco);
      tp[0].mode = CORBA::PARAM_IN;

      CORBA::ExceptionDefSeq te;
      te.length (1);

      tempco = _repo->lookup ("Components::InvalidConnection");
      te[0] = CORBA::ExceptionDef::_narrow (tempco);

      CORBA::ContextIdSeq tc;

      tempop = destiface->create_operation (target_id.c_str(),
					    target_name.c_str(),
					    version,
					    tt, CORBA::OP_NORMAL,
					    tp, te, tc);

      string origid = id.in();
      string fname = _db.get_fname_for_repoid (origid);
      _db.register_repoid (target_id, fname);
    }

    /*
     * get_connections_<name> operation
     */

    {
      string target_name = "get_connections_";
      target_name += name.in ();

      string target_id = id.in ();
      target_id.replace (target_id.length() - 4 - strlen (name.in()),
			 strlen (name.in()),
			 target_name);
      
      CORBA::ParDescriptionSeq tp;
      CORBA::ContextIdSeq tc;
      CORBA::ExceptionDefSeq te;

      tempop = destiface->create_operation (target_id.c_str(),
					    target_name.c_str(),
					    version,
					    conns, CORBA::OP_NORMAL,
					    tp, te, tc);

      string origid = id.in();
      string fname = _db.get_fname_for_repoid (origid);
      _db.register_repoid (target_id, fname);
    }
  }

  return tempop._retn();
}

/*
 * The equivalent IDL for events is currently buggy (ptc/99-10-04):
 * emitters and publishers are not compatible with the consumers. So
 * they are ignored for the moment.
 */

CORBA::OperationDef_ptr
CCMTransformer::transform_Emits (CORBA::ComponentIR::EmitsDef_ptr src)
{
  assert (0);
  return CORBA::OperationDef::_nil ();
}

CORBA::OperationDef_ptr
CCMTransformer::transform_Publishes (CORBA::ComponentIR::PublishesDef_ptr src)
{
  assert (0);
  return CORBA::OperationDef::_nil ();
}

CORBA::OperationDef_ptr
CCMTransformer::transform_Consumes (CORBA::ComponentIR::ConsumesDef_ptr src)
{
  assert (0);
  return CORBA::OperationDef::_nil ();
}

CORBA::InterfaceDef_ptr
CCMTransformer::transform_Component (CORBA::ComponentIR::ComponentDef_ptr src)
{
  CORBA::String_var id = src->id ();
  CORBA::String_var name = src->name ();
  CORBA::String_var version = src->version ();

  CORBA::Contained_var existing = _repo->lookup_id (id);

  if (!CORBA::is_nil (existing)) {
    return CORBA::InterfaceDef::_narrow (existing);
  }

  CORBA::Container_var defin = src->defined_in ();
  CORBA::Container_var dest = transform_Container (defin);

  /*
   * Original:
   *                  component <name> { ... };
   * Equivalent IDL:
   *                  interface <name> : Components::CCMObject { ... };
   *
   * Original:
   *                  component <name> supports <if1>, <if2> { ... };
   * Equivalent IDL:
   *                  interface <name> : Components::CCMObject, <if1>, <if2>
   *                  { ... };
   *
   * Original:
   *                  component <name> : <base> { ... };
   * Equivalent IDL:
   *                  interface <name> : <base> { ... };
   *
   * Original:
   *                  component <name> : <base> supports <if1>, <if2>
   *                  { ... };
   * Equivalent IDL:
   *                  interface <name> : <base>, <if1>, <if2>
   *                  { ... };
   *
   * In other words, the component is represented by an interface that
   * inherits all supported interfaces, plus the equivalent interface
   * for the base component, if it exists, or CCMObject otherwise.
   */

  CORBA::ComponentIR::ComponentDef_var base = src->base_component ();
  CORBA::InterfaceDefSeq_var supported = src->supported_interfaces ();
  CORBA::InterfaceDefSeq tb;

  tb.length (supported->length() + 1);

  if (CORBA::is_nil (base)) {
    CORBA::Contained_var tempco = _repo->lookup ("Components::CCMObject");
    tb[0] = CORBA::InterfaceDef::_narrow (tempco);
  }
  else {
    CORBA::IDLType_var ttemp = transform_IDLType (base);
    tb[0] = CORBA::InterfaceDef::_narrow (ttemp);
  }

  for (CORBA::ULong i=0; i<supported->length(); i++) {
    CORBA::IDLType_var ttemp = transform_IDLType (supported[i]);
    tb[i+1] = CORBA::InterfaceDef::_narrow (ttemp);
  }

  return dest->create_interface (id, name, version, tb);
}

CORBA::OperationDef_ptr
CCMTransformer::transform_Factory (CORBA::ComponentIR::FactoryDef_ptr src)
{
  CORBA::String_var id = src->id ();
  CORBA::String_var name = src->name ();
  CORBA::String_var version = src->version ();

  /*
   * Original:
   *                  factory <name> (...);
   *
   * Equivalent IDL:
   *                  <component> <name> (...);
   *
   * The factory operation must be added to the home's explicit interface.
   */
  
  string target_id = id.in ();
  target_id.insert (target_id.length() - 4 - strlen (name.in()), "Explicit/");

  CORBA::Contained_var existing = _repo->lookup_id (target_id.c_str());

  if (!CORBA::is_nil (existing)) {
    return CORBA::OperationDef::_narrow (existing);
  }

  CORBA::Container_var defin = src->defined_in ();
  CORBA::ComponentIR::HomeDef_var home =
    CORBA::ComponentIR::HomeDef::_narrow (defin);
  CORBA::ComponentIR::ComponentDef_var managed = home->managed_component ();
  CORBA::Container_var homeif = transform_Container (defin);

  /*
   * Look up explicit interface
   */

  CORBA::String_var homeifid = home->id ();
  string homeexid = homeifid.in();
  homeexid.insert (homeexid.length() - 4, "Explicit");

  CORBA::Contained_var dest = _repo->lookup_id (homeexid.c_str());
  CORBA::InterfaceDef_var destiface = CORBA::InterfaceDef::_narrow (dest);

  CORBA::ParDescriptionSeq_var sp = src->params ();
  CORBA::ParDescriptionSeq tp;
  tp.length (sp->length());

  for (CORBA::ULong i1=0; i1<sp->length(); i1++) {
    tp[i1].name = sp[i1].name;
    tp[i1].type = CORBA::_tc_void;
    tp[i1].mode = sp[i1].mode;
    tp[i1].type_def = transform_IDLType (sp[i1].type_def);
  }

  CORBA::OperationMode sm = src->mode ();
  CORBA::ContextIdSeq_var sc = src->contexts ();

  CORBA::ExceptionDefSeq_var se = src->exceptions ();
  CORBA::ExceptionDefSeq te;
  te.length (se->length());

  for (CORBA::ULong i2=0; i2<se->length(); i2++) {
    te[i2] = transform_Exception (se[i2]);
  }

  CORBA::IDLType_var tt = transform_IDLType (managed);

  string origid = id.in();
  string fname = _db.get_fname_for_repoid (origid);
  _db.register_repoid (target_id, fname);

  return destiface->create_operation (target_id.c_str(), name, version,
				      tt, CORBA::OP_NORMAL,
				      tp, te, sc);
}

CORBA::OperationDef_ptr
CCMTransformer::transform_Finder (CORBA::ComponentIR::FinderDef_ptr src)
{
  CORBA::String_var id = src->id ();
  CORBA::String_var name = src->name ();
  CORBA::String_var version = src->version ();

  /*
   * Original:
   *                  finder <name> (...);
   *
   * Equivalent IDL:
   *                  <component> <name> (...);
   *
   * The finder operation must be added to the home's explicit interface.
   */
  
  string target_id = id.in ();
  target_id.insert (target_id.length() - 4 - strlen (name.in()), "Explicit/");

  CORBA::Contained_var existing = _repo->lookup_id (target_id.c_str());

  if (!CORBA::is_nil (existing)) {
    return CORBA::OperationDef::_narrow (existing);
  }

  CORBA::Container_var defin = src->defined_in ();
  CORBA::ComponentIR::HomeDef_var home =
    CORBA::ComponentIR::HomeDef::_narrow (defin);
  CORBA::ComponentIR::ComponentDef_var managed = home->managed_component ();
  CORBA::Container_var homeif = transform_Container (defin);

  /*
   * Look up explicit interface
   */

  CORBA::String_var homeifid = home->id ();
  string homeexid = homeifid.in();
  homeexid.insert (homeexid.length() - 4, "Explicit");

  CORBA::Contained_var dest = _repo->lookup_id (homeexid.c_str());
  CORBA::InterfaceDef_var destiface = CORBA::InterfaceDef::_narrow (dest);

  CORBA::ParDescriptionSeq_var sp = src->params ();
  CORBA::ParDescriptionSeq tp;
  tp.length (sp->length());

  for (CORBA::ULong i1=0; i1<sp->length(); i1++) {
    tp[i1].name = sp[i1].name;
    tp[i1].type = CORBA::_tc_void;
    tp[i1].mode = sp[i1].mode;
    tp[i1].type_def = transform_IDLType (sp[i1].type_def);
  }

  CORBA::OperationMode sm = src->mode ();
  CORBA::ContextIdSeq_var sc = src->contexts ();

  CORBA::ExceptionDefSeq_var se = src->exceptions ();
  CORBA::ExceptionDefSeq te;
  te.length (se->length());

  for (CORBA::ULong i2=0; i2<se->length(); i2++) {
    te[i2] = transform_Exception (se[i2]);
  }

  CORBA::IDLType_var tt = transform_IDLType (managed);

  string origid = id.in();
  string fname = _db.get_fname_for_repoid (origid);
  _db.register_repoid (target_id, fname);

  return destiface->create_operation (target_id.c_str(), name, version,
				      tt, CORBA::OP_NORMAL,
				      tp, te, sc);
}

CORBA::InterfaceDef_ptr
CCMTransformer::transform_Home (CORBA::ComponentIR::HomeDef_ptr src)
{
  CORBA::String_var id = src->id ();
  CORBA::String_var name = src->name ();
  CORBA::String_var version = src->version ();

  CORBA::Contained_var existing = _repo->lookup_id (id);

  if (!CORBA::is_nil (existing)) {
    return CORBA::InterfaceDef::_narrow (existing);
  }

  CORBA::Container_var defin = src->defined_in ();
  CORBA::Container_var dest = transform_Container (defin);

  CORBA::ComponentIR::HomeDef_var base = src->base_home ();
  CORBA::ComponentIR::ComponentDef_var managed = src->managed_component ();
  CORBA::ValueDef_var key = src->primary_key ();

  /*
   * (1) Home definitions with no primary key
   *
   * Original:
   *                  home <name> manages <component> { ... };
   * Equivalent IDL:
   *                  interface <name>Explicit : Components::CCMHome
   *                  { ... };
   *
   *                  interface <name>Implicit : Components::KeylessCCMHome {
   *                  {
   *                    <component> create ();
   *                  };
   *
   *                  interface <name> : <name>Explicit, <name>Implicit {};
   *
   *
   * (2) Home definitions with primary key
   *
   * Original:
   *                  home <name> manages <component> primaryKey <key>
   *                  { ... };
   * Equivalent IDL:
   *                  interface <name>Explicit : Components::CCMHome
   *                  { ... };
   *
   *                  interface <name>Implicit {
   *                    <component> create (in <key> key);
   *                    <component> find_by_primary_key (in <key> key);
   *                    <component> remove (in <key> key);
   *                    <key> get_primary_key (in <component> comp);
   *                  };
   *
   *                  interface <name> : <name>Explicit, <name>Implicit {};
   */

  /*
   * Explicit Interface
   */

  CORBA::InterfaceDef_var explif, implif, res;

  {
    string target_name = name.in();
    target_name += "Explicit";

    string target_id = id.in ();
    target_id.replace (target_id.length() - 4 - strlen (name.in()),
		       strlen (name.in()),
		       target_name);

    CORBA::Contained_var existing = _repo->lookup_id (target_id.c_str());

    if (!CORBA::is_nil (existing)) {
      return CORBA::InterfaceDef::_narrow (existing);
    }

    CORBA::InterfaceDefSeq tb;
    tb.length (1);

    if (CORBA::is_nil (base)) {
      CORBA::Contained_var tempco = _repo->lookup ("Components::CCMHome");
      tb[0] = CORBA::InterfaceDef::_narrow (tempco);
    }
    else {
      CORBA::IDLType_var ttemp = transform_IDLType (base);
      tb[0] = CORBA::InterfaceDef::_narrow (ttemp);
    }

    explif = dest->create_interface (target_id.c_str(),
				     target_name.c_str(),
				     version, tb);

    string origid = id.in();
    string fname = _db.get_fname_for_repoid (origid);
    _db.register_repoid (target_id, fname);
  }

  /*
   * Implicit Interface
   */

  {
    string target_name = name.in();
    target_name += "Implicit";

    string target_id = id.in ();
    target_id.replace (target_id.length() - 4 - strlen (name.in()),
		       strlen (name.in()),
		       target_name);

    CORBA::InterfaceDefSeq tb;

    if (CORBA::is_nil (key)) {
      CORBA::Contained_var tempco = _repo->lookup ("Components::KeylessCCMHome");
      tb.length (1);
      tb[0] = CORBA::InterfaceDef::_narrow (tempco);
    }

    implif = dest->create_interface (target_id.c_str(),
				     target_name.c_str(),
				     version, tb);

    string origid = id.in();
    string fname = _db.get_fname_for_repoid (origid);
    _db.register_repoid (target_id, fname);

    /*
     * Operations on implicit interface differ for keyless and keyful homes
     */

    if (CORBA::is_nil (key)) {
      /*
       * <component> create ();
       */

      string target_id = id.in ();
      target_id.insert (target_id.length() - 4, "Implicit/create");

      CORBA::ParDescriptionSeq tp;
      CORBA::ContextIdSeq tc;
      CORBA::ExceptionDefSeq te;

      CORBA::IDLType_var result = transform_IDLType (managed);

      _db.register_repoid (target_id, fname);

      CORBA::OperationDef_var tempop =
	implif->create_operation (target_id.c_str(), "create", version,
				  result, CORBA::OP_NORMAL, tp, te, tc);
    }
    else {
      CORBA::Contained_var tempco;
      CORBA::OperationDef_var tempop;

      /*
       * <component> create (in <key> key)
       */

      {
	string target_id = id.in ();
	target_id.insert (target_id.length() - 4, "Implicit/create");

	CORBA::ParDescriptionSeq tp;
	tp.length (1);
	tp[0].name = CORBA::string_dup ("key");
	tp[0].type = CORBA::_tc_void;
	tp[0].type_def = transform_IDLType (key);
	tp[0].mode = CORBA::PARAM_IN;
	
	CORBA::ExceptionDefSeq te;
	te.length (2);

	tempco = _repo->lookup ("Components::DuplicateKeyValue");
	te[0] = CORBA::ExceptionDef::_narrow (tempco);

	tempco = _repo->lookup ("Components::InvalidKey");
	te[0] = CORBA::ExceptionDef::_narrow (tempco);

	CORBA::IDLType_var result = transform_IDLType (managed);

	CORBA::ContextIdSeq tc;

	_db.register_repoid (target_id, fname);

	tempop = implif->create_operation (target_id.c_str(),
					   "create", version,
					   result, CORBA::OP_NORMAL,
					   tp, te, tc);
      }

      /*
       * <component> find_by_primary_key (in <key> key)
       */

      {
	string target_id = id.in ();
	target_id.insert (target_id.length() - 4,
			  "Implicit/find_by_primary_key");

	CORBA::ParDescriptionSeq tp;
	tp.length (1);
	tp[0].name = CORBA::string_dup ("key");
	tp[0].type = CORBA::_tc_void;
	tp[0].type_def = transform_IDLType (key);
	tp[0].mode = CORBA::PARAM_IN;
	
	CORBA::ExceptionDefSeq te;
	te.length (2);

	tempco = _repo->lookup ("Components::UnknownKeyValue");
	te[0] = CORBA::ExceptionDef::_narrow (tempco);

	tempco = _repo->lookup ("Components::InvalidKey");
	te[0] = CORBA::ExceptionDef::_narrow (tempco);

	CORBA::IDLType_var result = transform_IDLType (managed);

	CORBA::ContextIdSeq tc;

	_db.register_repoid (target_id, fname);

	tempop = implif->create_operation (target_id.c_str(),
					   "find_by_primary_key", version,
					   result, CORBA::OP_NORMAL,
					   tp, te, tc);
      }

      /*
       * void remove (in <key> key)
       */

      {
	string target_id = id.in ();
	target_id.insert (target_id.length() - 4,
			  "Implicit/remove");

	CORBA::ParDescriptionSeq tp;
	tp.length (1);
	tp[0].name = CORBA::string_dup ("key");
	tp[0].type = CORBA::_tc_void;
	tp[0].type_def = transform_IDLType (key);
	tp[0].mode = CORBA::PARAM_IN;
	
	CORBA::ExceptionDefSeq te;
	te.length (2);

	tempco = _repo->lookup ("Components::UnknownKeyValue");
	te[0] = CORBA::ExceptionDef::_narrow (tempco);

	tempco = _repo->lookup ("Components::InvalidKey");
	te[0] = CORBA::ExceptionDef::_narrow (tempco);

	CORBA::IDLType_var result = _repo->get_primitive (CORBA::pk_void);

	CORBA::ContextIdSeq tc;

	_db.register_repoid (target_id, fname);

	tempop = implif->create_operation (target_id.c_str(),
					   "remove", version,
					   result, CORBA::OP_NORMAL,
					   tp, te, tc);
      }

      /*
       * <key> get_primary_key (in <component> comp)
       */

      {
	string target_id = id.in ();
	target_id.insert (target_id.length() - 4,
			  "Implicit/get_primary_key");

	CORBA::ParDescriptionSeq tp;
	tp.length (1);
	tp[0].name = CORBA::string_dup ("comp");
	tp[0].type = CORBA::_tc_void;
	tp[0].type_def = transform_IDLType (managed);
	tp[0].mode = CORBA::PARAM_IN;
	
	CORBA::ExceptionDefSeq te;
	CORBA::ContextIdSeq tc;

	CORBA::IDLType_var result = transform_IDLType (key);

	_db.register_repoid (target_id, fname);

	tempop = implif->create_operation (target_id.c_str(),
					   "get_primary_key", version,
					   result, CORBA::OP_NORMAL,
					   tp, te, tc);
      }
    }
  }

  /*
   * Home equivalent interface
   */

  {
    CORBA::InterfaceDefSeq tb;
    tb.length (2);

    tb[0] = CORBA::InterfaceDef::_duplicate (explif);
    tb[1] = CORBA::InterfaceDef::_duplicate (implif);

    res = dest->create_interface (id, name, version, tb);
  }

  return res._retn();
}

/*
 * ----------------------------------------------------------------------
 * Main CCM Transformer
 * ----------------------------------------------------------------------
 */

CORBA::Container_ptr
CCMTransform (DB & db, IDLParam & params,
	      CORBA::Repository_ptr repo,
	      CORBA::Container_ptr cont)
{
  /*
   * Preload CCM IDL
   */

  /*
   * Step 1: Invoke the Preprocessor
   */

  string cmd = "mico-cpp -D __MICO_IDL__ ";
  cmd += params.cpp_options;
  cmd += " -I";
  cmd += ABSEXECDIR;
  cmd += "/include ";
  cmd += ABSEXECDIR;
  cmd += "/include/mico/CCM.idl";

  FILE * inp_file = OSMisc::popen (cmd.c_str(), "r");

  if (inp_file == NULL) {
    cerr << "error: cannot open " << ABSEXECDIR << "include/mico/CCM.idl" << endl;
    exit (1);
  }

  int dummy_char = fgetc (inp_file);

  if (dummy_char == EOF) {
    cerr << "error: cannot execute mico-cpp" << endl;
    exit( 1 );
  }

  ungetc (dummy_char, inp_file);
  
  /*
   * Step 2: Invoke the Parser (bison)
   */
    
  Parser parser (inp_file, "components.idl");
  parser.parse();
  
  OSMisc::pclose (inp_file);

  /*
   * Step 3: Traverse the Parse Tree and load Interface Repository
   */
  
  IDLParser idlParser (db);
  idlParser.collect (repo, parser.getRootNode());
  db.set_repoids (repo);

  /*
   * Traverse the CORBA 3 Interface Repository and transform to CORBA 2
   */
  
  CCMTransformer ct (db, repo);

  if (!ct.transform (cont)) {
    cerr << "error: transformation failed" << endl;
    exit (1);
  }

  /*
   * Find the input container in the new repository and return it
   */

  CORBA::Container_var res;

  CORBA::Contained_var ci = CORBA::Contained::_narrow (cont);

  if (!CORBA::is_nil (ci)) {
    CORBA::String_var id = ci->id ();
    ci  = repo->lookup_id (id);
    res = CORBA::Container::_narrow (ci);
    assert (!CORBA::is_nil (res));
  }
  else {
    res = CORBA::Container::_duplicate (repo);
  }

  return res;
}

