/*
 *  Mico --- a free CORBA implementation
 *  Copyright Frank Pilhofer
 *
 *  This file is part of MicoCCM, written by Frank Pilhofer.
 *  The MICO CORBA Component Project was sponsored by Alcatel.
 */

#include <CORBA.h>
#include <mico/CCM.h>
#include <mico/CosNaming.h>
#include <mico/CCMContainer.h>

/*
 * ----------------------------------------------------------------------
 * MICO Containers
 * ----------------------------------------------------------------------
 */

MICO::CCM::ContainerBase::~ContainerBase ()
{
}

/*
 * ----------------------------------------------------------------------
 * Implementation for the Service Container
 * ----------------------------------------------------------------------
 */

/*
 * Implementation of Components::SessionContext for the Service Container
 */

MICO::CCM::ServiceContext::ServiceContext (ServiceContainer * c)
{
  _container = c;
}

Components::CCMHome_ptr
MICO::CCM::ServiceContext::get_CCM_home ()
{
  return _container->get_reference_for_home ();
}

CORBA::Object_ptr
MICO::CCM::ServiceContext::get_CCM_object ()
{
  return _container->get_reference_for_component ();
}

/*
 * The Service Container itself
 */

MICO::CCM::ServiceContainer::ServiceContainer (CORBA::ORB_ptr orb)
{
  CORBA::Object_var obj;

  _orb = CORBA::ORB::_duplicate (orb);
  obj = _orb->resolve_initial_references ("RootPOA");
  _my_poa = PortableServer::POA::_narrow (obj);

  try {
    obj = _orb->resolve_initial_references ("NameService");
    _naming = CosNaming::NamingContextExt::_narrow (obj);
  }
  catch (...) {
    _naming = CosNaming::NamingContextExt::_nil ();
  }

  if (CORBA::is_nil (_naming)) {
    cerr << "Oops, I'd like to have a Naming Service." << endl;
    assert (0);
  }

  _context = new ServiceContext (this);
  _have_info = 0;
}

MICO::CCM::ServiceContainer::~ServiceContainer ()
{
}

void
MICO::CCM::ServiceContainer::load (const ComponentInfo & info)
{
  assert (!_have_info);
  _info = info;
  _have_info = 1;

  /*
   * Activate home and component singletons
   */

  _home_ref = _info.home_skel.in()->_this();
  _comp_ref = _info.comp_skel.in()->_this();

  /*
   * Register home with Naming Service
   */

  CosNaming::Name_var name = _naming->to_name (_info.home_short_name.c_str());
  _naming->rebind (name.in(), _home_ref.in());
}

void
MICO::CCM::ServiceContainer::activate ()
{
  /*
   * Activate our POA
   */

  PortableServer::POAManager_var mgr = _my_poa->the_POAManager ();
  mgr->activate ();
}

void
MICO::CCM::ServiceContainer::deactivate ()
{
  assert (0);
}

Components::CCMHome_ptr
MICO::CCM::ServiceContainer::get_reference_for_home ()
{
  return Components::CCMHome::_narrow (_home_ref.in());
}

Components::CCMObject_ptr
MICO::CCM::ServiceContainer::get_reference_for_component ()
{
  return Components::CCMObject::_narrow (_comp_ref.in());
}

Components::SessionContext_ptr
MICO::CCM::ServiceContainer::get_session_context ()
{
  return Components::SessionContext::_duplicate (_context);
}

/*
 * ----------------------------------------------------------------------
 * Implementation for the Session Container
 * ----------------------------------------------------------------------
 */

/*
 * Implementation of Components::SessionContext for the Session Container
 */

MICO::CCM::SessionContext::SessionContext (CORBA::ORB_ptr o,
					   SessionContainer * c)
{
  _container = c;
  _orb = CORBA::ORB::_duplicate (o);
  CORBA::Object_var obj = _orb->resolve_initial_references ("POACurrent");
  _current = PortableServer::Current::_narrow (obj);
}

Components::CCMHome_ptr
MICO::CCM::SessionContext::get_CCM_home ()
{
  return _container->get_reference_for_home ();
}

CORBA::Object_ptr
MICO::CCM::SessionContext::get_CCM_object ()
{
  try {
    PortableServer::POA_var poa = _current->get_POA ();
    PortableServer::ObjectId_var oid = _current->get_object_id ();
    PortableServer::ServantBase_var comp = poa->id_to_servant (oid.in());
    return _container->get_reference_for_component (comp);
  }
  catch (PortableServer::Current::NoContext &) {
    throw Components::IllegalState ();
  }
  return CORBA::Object::_nil ();
}

/*
 * The Session Container itself
 */

MICO::CCM::SessionContainer::SessionContainer (CORBA::ORB_ptr orb)
{
  CORBA::Object_var obj;
  _orb = CORBA::ORB::_duplicate (orb);
  obj = _orb->resolve_initial_references ("RootPOA");
  _home_poa = PortableServer::POA::_narrow (obj);
  _component_poa = PortableServer::POA::_narrow (obj);

  try {
    obj = _orb->resolve_initial_references ("NameService");
    _naming = CosNaming::NamingContextExt::_narrow (obj);
  } catch (...) {
    _naming = CosNaming::NamingContextExt::_nil ();
  }

  if (CORBA::is_nil (_naming)) {
    cerr << "Oops, I'd like to have a Naming Service." << endl;
    assert (0);
  }

  _context = new SessionContext (_orb, this);
  _have_info = 0;
}

MICO::CCM::SessionContainer::~SessionContainer ()
{
}

void
MICO::CCM::SessionContainer::load (const ComponentInfo & info)
{
  assert (!_have_info);
  _info = info;
  _have_info = 1;

  /*
   * Activate home singleton
   */

  PortableServer::ObjectId_var oid =
    _home_poa->activate_object (_info.home_skel.in());
  _home_ref = _home_poa->id_to_reference (oid.in());

  /*
   * Register home with Naming Service
   */

  CosNaming::Name_var name = _naming->to_name (_info.home_short_name.c_str());
  _naming->rebind (name.in(), _home_ref.in());
}

void
MICO::CCM::SessionContainer::activate ()
{
  /*
   * Set instances to active
   */

  for (InstanceMap::iterator i=active_components.begin();
       i != active_components.end(); i++) {
    Components::SessionComponent_var sc =
      Components::SessionComponent::_narrow ((*i).second);
    sc->ccm_activate ();
  }

  /*
   * Activate our POA
   */

  PortableServer::POAManager_var mgr;
  mgr = _home_poa->the_POAManager ();
  mgr->activate ();
  mgr = _component_poa->the_POAManager ();
  mgr->activate ();
}

void
MICO::CCM::SessionContainer::deactivate ()
{
  /*
   * Set instances to inactive
   */

  for (InstanceMap::iterator i=active_components.begin();
       i != active_components.end(); i++) {
    Components::SessionComponent_var sc =
      Components::SessionComponent::_narrow ((*i).second);
    sc->ccm_passivate ();
  }
}

Components::SessionContext_ptr
MICO::CCM::SessionContainer::get_session_context ()
{
  return Components::SessionContext::_duplicate (_context);
}

Components::CCMHome_ptr
MICO::CCM::SessionContainer::get_reference_for_home ()
{
  return Components::CCMHome::_narrow (_home_ref.in());
}

Components::CCMObject_ptr
MICO::CCM::SessionContainer::activate_component (CORBA::LocalObject_ptr inst,
						 PortableServer::Servant skel)
{
  PortableServer::ObjectId_var oid = _component_poa->activate_object (skel);
  CORBA::Object_var ref = _component_poa->id_to_reference (oid.in());
  active_components[oid.in()] = inst;
  return Components::CCMObject::_narrow (ref);
}

Components::CCMObject_ptr
MICO::CCM::SessionContainer::get_reference_for_component (PortableServer::Servant s)
{
  CORBA::Object_var o = _component_poa->servant_to_reference (s);
  return Components::CCMObject::_narrow (o);
}

PortableServer::Servant
MICO::CCM::SessionContainer::get_skeleton_for_reference (CORBA::Object_ptr o)
{
  return _component_poa->reference_to_servant (o);
}

Components::CCMObject_ptr
MICO::CCM::SessionContainer::get_reference_for_instance (CORBA::LocalObject *o)
{
  InstanceMap::iterator it = active_components.begin ();
  while (it != active_components.end()) {
    if ((*it).second == o) {
      CORBA::Object_var ref = _component_poa->id_to_reference ((*it).first);
      return Components::CCMObject::_narrow (ref);
    }
    it++;
  }
  return Components::CCMObject::_nil ();
}

CORBA::LocalObject_ptr
MICO::CCM::SessionContainer::get_instance_for_component (PortableServer::Servant skel)
{
  PortableServer::ObjectId_var oid = _component_poa->servant_to_id (skel);
  return CORBA::LocalObject::_duplicate (active_components[oid.in()]);
}

void
MICO::CCM::SessionContainer::deactivate_component (CORBA::Object_ptr o)
{
  PortableServer::ServantBase_var skel = get_skeleton_for_reference (o);
  deactivate_component (skel);
}

void
MICO::CCM::SessionContainer::deactivate_component (PortableServer::Servant skel)
{
  PortableServer::ObjectId_var oid = _component_poa->servant_to_id (skel);
  _component_poa->deactivate_object (oid);
  active_components.erase (oid.in());
}

CORBA::Object_ptr
MICO::CCM::SessionContainer::get_reference_for_facet (CORBA::LocalObject_ptr o)
{
  InstanceMap::iterator it = active_facets.begin ();
  while (it != active_facets.end()) {
    if ((*it).second == o) {
      return _component_poa->id_to_reference ((*it).first);
    }
    it++;
  }
  return CORBA::Object::_nil ();
}

CORBA::Object_ptr
MICO::CCM::SessionContainer::activate_facet (CORBA::LocalObject_ptr inst,
					     PortableServer::Servant skel)
{
  PortableServer::ObjectId_var oid = _component_poa->activate_object (skel);
  active_facets[oid.in()] = inst;
  return _component_poa->id_to_reference (oid.in());
}
