/*            Copyright (C) 1999, 2000, 2001, 2002, 2003 Stijn van Dongen
 *
 * This file is part of MCL.  You can redistribute and/or modify MCL under the
 * terms of the GNU General Public License; either version 2 of the License or
 * (at your option) any later version.  You should have received a copy of the
 * GPL along with MCL, in the file COPYING.
*/

#include <limits.h>
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <stdarg.h>
#include <string.h>

#include "ilist.h"
#include "perm.h"

#include "util/compile.h"
#include "util/array.h"
#include "util/alloc.h"
#include "util/types.h"
#include "util/equate.h"
#include "util/io.h"


void voidIlAlert
(  const char*       msg
)  ;


int ilWriteFile
(  const mcxIL*   src_il
,  FILE*          f_out
)
   {  int   k  =  mcxIOwriteInteger(f_out, src_il->n)
   ;  k       +=  fwrite(src_il->L, sizeof(int), src_il->n, f_out)
   ;  if (k != src_il->n + 1)
      {  fprintf
         (  stderr
         ,  "[ilWriteFile fatal] wrote %d, should be %d\n"
         ,  k
         ,  src_il->n + 1
         )
      ;  exit(1)
   ;  }
      return 0
;  }


mcxIL*   ilVA
(  mcxIL*   il
,  int      k
,  ...
)
   {  va_list     ap
   ;  mcxBuf      ibuf
   ;  int         i

   ;  if (!il)
      il    =  ilInit(NULL)

   ;  mcxBufInit(&ibuf, &(il->L), sizeof(int), 8)

   ;  {  va_start(ap, k)

      ;  for (i=0;i<k;i++)
         {  int  *xp =  mcxBufExtend(&ibuf, 1, EXIT_ON_FAIL)
         ;  *xp      =  va_arg(ap, int)
      ;  }
         va_end(ap)
   ;  }

      il->n =  mcxBufFinalize(&ibuf)
   ;  return(il)
;  }


mcxIL*   ilComplete
(  mcxIL*   il
,  int      n
)
   {  int   i

   ;  il = ilInstantiate(il, n, NULL, -1)

   ;  for (i=0;i<n;i++)
      *(il->L+i)  =  i

   ;  return(il)
;  }


mcxIL*    ilReadFile
(  mcxIL*         dst_il
,  FILE*          f_in
)
   {  int n    =  mcxIOreadInteger(f_in)
   ;  if (!n)
      {  fprintf(stderr, "[ilReadFile warning] empty list\n")
   ;  }
   ;  dst_il = ilInstantiate(dst_il, n, NULL, -1)
   ;  fread(dst_il->L, sizeof(int), n, f_in)
   ;  return dst_il
;  }


mcxIL*   ilInit
(  mcxIL*   il
)
   {  if (!il)
      il = (mcxIL *) mcxAlloc(sizeof(mcxIL), EXIT_ON_FAIL)

   ;  il->L =  NULL
   ;  il->n    =  0

   ;  return il
;  }


mcxIL*   ilInstantiate
(  mcxIL*   dst_il
,  int      n
,  int*     ints
,  int      c
)
   {  if (!dst_il) dst_il = ilInit(NULL)

   ;  dst_il->L   =  (int*) mcxRealloc
                        (  dst_il->L
                        ,  n * sizeof(int)
                        ,  RETURN_ON_FAIL
                        )

   ;  if (n && !dst_il->L)
         mcxMemDenied(stderr, "ilInstantiate", "int", n)
      ,  exit(1)

   ;  if (n && ints)
      {  memcpy(dst_il->L, ints, n * sizeof(int))
   ;  }
      else
      {  int i    =  0
                                    /* c is used only if ints == NULL */
      ;  while (i < n) *(dst_il->L+i++) = c
   ;  }

   ;  dst_il->n     =  n
   ;  return dst_il
;  }


mcxIL*   ilStore
(  mcxIL*   dst
,  int*     ints
,  int      n
)
   {  if (!dst)
      {  dst  =  ilNew(n, ints, -1)
      ;  return dst
   ;  }
      else
      {  int i    =  0
      ;  dst->L   =  (int*) mcxRealloc
                     (  dst->L
                     ,  n * sizeof(int)
                     ,  RETURN_ON_FAIL
                     )

      ;  if (n && !dst->L)
            mcxMemDenied(stderr, "ilStore", "int", n)
         ,  exit(1)

      ;  while (i<n)
         *(dst->L+i) = *(ints+i), i++

      ;  dst->n      =  n
   ;  }
      return dst
;  }


mcxIL*   ilCon
(  mcxIL*   dst
,  int*     ls
,  int      n
)
   {  if (!dst)
      return ilNew(n, ls, -1)

   ;  dst->L      =  (int*) mcxRealloc
                        (  dst->L
                        ,  (dst->n + n) * sizeof(int)
                        ,  RETURN_ON_FAIL
                        )

   ;  if ((dst->n + n) && !dst->L)
         mcxMemDenied(stderr, "ilCon", "int", (dst->n + n))
      ,  exit(1)

   ;  if (0)
      {  int   i
      ;  for (i=0;i<n;i++)
         {  *(dst->L+dst->n-n+i) = *(ls+i)
      ;  }
   ;  }
      else
      {  memcpy((dst->L)+dst->n, ls, n * sizeof(int))
   ;  }

   ;  dst->n    = dst->n + n

   ;  return dst
;  }


void     ilResize
(  mcxIL*   il
,  int      n
)
   {  if (!il) voidIlAlert("ilResize")
   ;  {  
         int   i  =  il->n

      ;  if (n > il->n)
         {  
            il->L =  (int*) mcxRealloc
                        (  il->L
                        ,  n * sizeof(int)
                        ,  RETURN_ON_FAIL
                        )

         ;  if (n && !il->L)
               mcxMemDenied(stderr, "ilResize", "int", n)
            ,  exit(1)
      ;  }

      ;  while (i<n)
         {  *(il->L+i) = 0
      ;  }

      ;  il->n    =  n
   ;  }
;  }


mcxIL*   ilInvert
(  mcxIL*   src
)  {  int      i
   ;  mcxIL*   inv
   ;  if (!src) voidIlAlert("ilInvert")

   ;  inv      =  ilNew(src->n, NULL, -1)
   ;  for (i=0;i<src->n;i++)
      {  int   next        =  *(src->L+i)
      ;  if (next < 0 || next >= src->n)
         {  fprintf
            (  stderr
            ,  "[ilInvert] index %d out of range (0, %d>\n"
            ,  next
            ,  src->n
            )
         ;  exit(1)
      ;  }
      ;  *(inv->L+next) =  i
   ;  }
   ;  return inv
;  }


int   ilIsMonotone
(  mcxIL*   src
,  int      gradient
,  int      min_diff
)
   {  int   i
   ;  if (!src) voidIlAlert("ilIsMonotone")

   ;  for(i=1;i<src->n;i++)
      {  if ( (*(src->L+i) - *(src->L+i-1)) * gradient < min_diff)
            return 0
   ;  }
   ;  return 1
;  }


int   ilIsOneOne
(  mcxIL*   src
)
   {  int   d
   ;  mcxIL*   inv   =  ilInvert(src)
   ;  mcxIL*   invv  =  ilInvert(inv)
   ;  d = intnCmp(src->L, invv->L, src->n)
   ;  ilFree(&inv)
   ;  ilFree(&invv)
   ;  return d ? 0 : 1
;  }


void   ilAccumulate
(  mcxIL*   il
)  {  int   i     =  0
   ;  int   prev  =  0
   ;  if (!il) voidIlAlert("ilIsMonotone")

   ;  for (i=0;i<il->n;i++)
      {  *(il->L+i) +=  prev
      ;  prev           =  *(il->L+i)
   ;  }
;  }
      

int ilSum
(  mcxIL*   il
)
   {  int   sum   =  0
   ;  int   i
   ;  if (!il) voidIlAlert("ilSum")

   ;  i          =  il->n
   ;  while(--i >= 0)
      {  sum += *(il->L+i)
   ;  }
   ;  return sum
;  }


void ilPrint
(  mcxIL*   il
,  const char msg[]
)
   {  int      i
   ;  if (!il) voidIlAlert("ilPrint")

   ;  for (i=0;i<il->n;i++)
      {  printf(" %-8d", *(il->L+i))
      ;  if (((i+1) % 8) == 0)
            printf("\n")
   ;  }
   ;  if (i%8 != 0) printf("\n")
   ;  fprintf  (  stdout, "[ilPrint end%s%s: size is %d]\n"
               ,  msg[0] ? ":" : ""
               ,  msg
               ,  il->n
               )
   ;  fprintf(stdout, " size is %d\n\n", il->n)
;  }

                                    /* translate an integer sequence    */
                                    /* should be generalized via Unary  */
void     ilTranslate
(  mcxIL*   il
,  int      dist
)
   {  int   i
   ;  if (!il) voidIlAlert("ilTranslate")

   ;  i          =  il->n
   ;  if (!dist) return
   ;  while (--i >= 0)
      {  *(il->L+i) += dist
   ;  }
;  }


void ilFree
(  mcxIL**  ilp
)
   {  if (*ilp)
      {  if ((*ilp)->L)  mcxFree((*ilp)->L)
      ;  mcxFree(*ilp)
   ;  }
   ;  *ilp  =  (void*) 0
;  }

                              /* shuffle an interval of integers        */
mcxIL*     ilRandPermutation
(  int      lb
,  int      rb
)
   {  int      w        =     rb - lb
   ;  int      *ip, i
   ;  mcxIL*   il

   ;  if (w < 0)
      {  fprintf  (  stderr
                  ,  "[ilRandPermutation error] bounds [%d, %d] reversed\n"
                  ,  lb
                  ,  rb
                  )
      ;  exit(1)
   ;  }

   ;  if (w == 0)
      {  fprintf  (  stderr
                  ,  "[ilRandPermutation warning] bounds [%d, %d] equal\n"
                  ,  lb
                  ,  rb
                  )
   ;  }

   ;  il    =  ilNew(w, (void*) 0, -1)
   ;  ip    =  il->L

   ;  for (i=0;i<w;i++) *(ip+i) = i       /* initialize translated interval */

   ;  for (i=w-1;i>=0;i--)                /* shuffle interval               */
      {  int l       =  (int) (rand() % (i+1))
      ;  int draw    =  *(ip+l)
      ;  *(ip+l)     =  *(ip+i)
      ;  *(ip+i)     =  draw
   ;  }

   ;  ilTranslate(il, lb)        /* translate interval to correct offset   */
   ;  return il
;  }


/*
 *   currently not in use.
 *   note the solution in genMatrix:
 *   draw for each number separately.
*/


mcxIL* ilLottery
(  int         lb
,  int         rb
,  float       p
,  int         times
)
   {  int      i
   ;  int      w        =  rb - lb
   ;  int      hits
   ;  long     prev, bar, r

   ;  mcxIL*   il       =  ilNew(0, NULL, 0)
   ;  mcxBuf   ibuf

   ;  if (w <= 0 || rb < 1)
      {  fprintf
         (  stderr
         ,  "[ilDraw warning] interval [%d, %d> ill defined\n"
         ,  lb
         ,  rb
         )
      ;  exit(1)
   ;  }

   ;  if (p < 0.0) p = 0.0
   ;  if (p > 1.0) p = 1.0

   ;  mcxBufInit(&ibuf, &(il->L), sizeof(int), times)

   ;  hits     =  0
   ;  prev     =  rand()
   ;  bar      =  p * LONG_MAX
   
   ;  for (i=0;i<times;i++)
      {  
         if ((r = rand() % INT_MAX) < bar)
         {  
            int*  jptr  =  (int*) mcxBufExtend(&ibuf, 1, EXIT_ON_FAIL)
         ;  *jptr       =  lb + ((r + prev) % w)
      ;  }
         prev = r % w
   ;  }

      il->n    =  mcxBufFinalize(&ibuf)
   ;  return il
;  }


                           /* create random partitions at grid--level   */
mcxIL*  ilGridRandPartitionSizes
(  int      w
,  int      gridsize
)
   {  int      n_blocks, i
   ;  mcxIL*   il_all   =     (void*) 0
   ;  mcxIL*   il_one   =     (void*) 0


   ;  if (gridsize > w)
         return ilRandPartitionSizes(w)

   ;  n_blocks          =     w / gridsize

   ;  il_all            =     ilNew(0, (void*) 0, -1)

   ;  for (i=0;i<n_blocks;i++)
      {
      ;  il_one         =     ilRandPartitionSizes(gridsize)
      ;  ilCon(il_all, il_one->L, il_one->n)
      ;  ilFree(&il_one)
   ;  }

   ;  if (n_blocks * gridsize < w)
      {  il_one         =     ilRandPartitionSizes(w - n_blocks * gridsize)
      ;  ilCon(il_all, il_one->L, il_one->n)
      ;  ilFree(&il_one)
   ;  }

   ;  ilSort(il_all)
   ;  return il_all
;  }


                                    /* create random partition */
mcxIL* ilRandPartitionSizes
(  int      w
)
   {  Pmt*     pmt
   ;  mcxIL*   il

   ;  if (w <= 0)
      {  fprintf(stderr, "[ilRandPartition warning] width argument %d nonpositive\n", w)
      ;  return   ilNew(0, (void*) 0, -1)
   ;  }

   ;  pmt      =  pmtRand(w)
   ;  il       =  pmtGetCycleSizes(pmt)
   ;  pmtFree(&pmt)
   ;  ilSort(il)

   ;  return il
;  }


float ilDeviation
(  mcxIL*   il
)
   {  float    dev   =  0.0
   ;  float    av    =  ilAverage(il)
   ;  int   i
   ;  if (il->n == 0)
         return 0.0

   ;  for (i=0;i<il->n;i++)
      {  dev += ((float ) (*(il->L+i)-av)) * (*(il->L+i)-av)
   ;  }
   ;  return sqrt(dev/il->n)
;  }


float ilAverage
(  mcxIL*   il
)
   {  float    d  =  (float ) ilSum(il)
   ;  return il->n ? d /  il->n : 0.0
;  }


float ilCenter
(  mcxIL*   il
)
   {  int   sum   =  ilSum(il)
   ;  return   sum ? (float ) ilSqum (il) / (float) sum : 0.0 
;  }


int ilSqum
(  mcxIL*   il
)
   {  int   sum   =  0
   ;  int   i     =  il->n
   ;  while(--i >= 0)
      {  sum += *(il->L+i) * *(il->L+i)
   ;  }
   ;  return sum
;  }


int ilSelectRltBar
(  mcxIL*   il
,  int      i1
,  int      i2
,  int      (*rlt1)(const void*, const void*)
,  int      (*rlt2)(const void*, const void*)
,  int      onlyCount
)
   {  int   i
   ;  int   j     =  0

   ;  for(i=0;i<il->n;i++)
      {  if
         (  (!rlt1 || rlt1(il->L+i, &i1))
         && (!rlt2 || rlt2(il->L+i, &i2))
         )
         {  
            if (!onlyCount && j<i)
            *(il->L+j)   =  *(il->L+i)

         ;  j++
      ;  }
   ;  }
   ;  if (!onlyCount && (i-j))
      ilResize(il, j)

   ;  return j
;  }


void voidIlAlert
(  const char*       msg
)
   {  fprintf(stderr, "[%s] void mcxIL argument\n", msg)
   ;  exit(1)
;  }


void  ilSort
(  mcxIL*   il
)
   {  if (!il)
      {  fprintf(stderr, "[ilSort] warning: uninitialized list\n")
      ;  return
   ;  }
      if (il->L)
         qsort(il->L, il->n, sizeof(int), intCmp)
;  }


void  ilRevSort
(  mcxIL*   il
)
   {  if (il->L)
      qsort(il->L, il->n, sizeof(int), intRevCmp)
;  }


int ilSelectLtBar
(  mcxIL*   il
,  int      i
)  {  return ilSelectRltBar(il, i,  0, intLt, NULL, 0)
;  }


int   ilIsDescending
(  mcxIL*   src
)  {  return ilIsMonotone(src, -1, 1)
;  }


int   ilIsNonAscending
(  mcxIL*   src
)  {  return ilIsMonotone(src, -1, 0)
;  }


int   ilIsNonDescending
(  mcxIL*   src
)  {  return ilIsMonotone(src, 1, 0)
;  }


int   ilIsAscending
(  mcxIL*   src
)  {  return ilIsMonotone(src, 1, 1)
;  }


int ilSelectGqBar
(  mcxIL*   il
,  int      i
)  {  return ilSelectRltBar(il, i,  0, intGq, NULL, 0)
;  }


mcxIL* ilNew
(  int   n
,  int*  ints
,  int   c
)  {  mcxIL* il =  ilInstantiate(NULL, n, ints, c)
   ;  return il
;  }


int ilCountLtBar
(  mcxIL*   il
,  int      i
)  {  return ilSelectRltBar(il, i,  0, intLt, NULL, 1)
;  }


