/*            Copyright (C) 2002, 2003 Stijn van Dongen
 *
 * This file is part of MCL.  You can redistribute and/or modify MCL under the
 * terms of the GNU General Public License; either version 2 of the License or
 * (at your option) any later version.  You should have received a copy of the
 * GPL along with MCL, in the file COPYING.
*/

#include <string.h>

#include "report.h"

#include "impala/matrix.h"
#include "impala/vector.h"
#include "impala/ivp.h"
#include "impala/io.h"

#include "mcl/interpret.h"
#include "mcl/clm.h"

#include "util/types.h"
#include "util/err.h"
#include "util/ting.h"
#include "util/opt.h"
#include "util/array.h"


enum
{  OPT_ENSTRICT
,  OPT_OUTPUTFILE
,  OPT_OUTPUTINFIX
,  OPT_OUTPUTSUFFIX
,  OPT_INPUTFILE
,  OPT_TIGHT
,  OPT_HELP
,  OPT_W_SELFVAL
,  OPT_W_MAXVAL
,  OPT_W_CENTER
,  OPT_SORT
,  OPT_VERSION
}  ;
   

/*    1: has argument                        */
/*    2: may occur in file argument place    */
/*    4: experimental option                 */

mcxOptAnchor options[] =
{  {  "--enstrict",        0, OPT_ENSTRICT }
,  {  "--version",         0, OPT_VERSION }
,  {  "-o",                1, OPT_OUTPUTFILE }
,  {  "-infix",            1, OPT_OUTPUTINFIX }
,  {  "-imx",              1, OPT_INPUTFILE }
,  {  "-suffix",           1, OPT_OUTPUTINFIX }
,  {  "-tight",            1, OPT_TIGHT }
,  {  "-t",                1, OPT_TIGHT }
,  {  "-h",                0, OPT_HELP }
,  {  "-sort",             1, OPT_SORT }
,  {  "-center",           5, OPT_W_CENTER }
,  {  "-self",             5, OPT_W_SELFVAL }
,  {  "-max",              5, OPT_W_MAXVAL }
,  {  NULL,                0, 0 }
}  ;

const char* usagelines[] =
{  "Usage: clmimac [options]"
,  ""
,  "Options (marked * are obligatory):"
,  "  -imx <fname>   *        input matrix file, presumably dumped mcl iterand"
,  "  -o <fname>              output cluster file"
,  "  -infix <infix>          for use with multiple -t options"
,  "  -suffix <suffix>        for use with multiple -t options"
,  "  -tight <num>            integer in range 0..100, multiple use allowed"
,  "  -t <num>                same as -tight, multiple use allowed"
,  "  -sort <str>             sort mode for clusters, str one of <revsize|size|lex>"
,  "  --enstrict              clean up clustering if not a partition"
,  "  -center <frac>          EXPERIMENTAL/EXPERT option"
,  "  -self <frac>            EXPERIMENTAL/EXPERT option"
,  "  -max <frac>             EXPERIMENTAL/EXPERT option"
,  NULL
}  ;

int main
(  int                  argc
,  const char*          argv[]
)  
   {  mcxstatus parseStatus   =  STATUS_OK

   ;  mcxbool keepOverlap     =  TRUE
   ;  mclIpretParam* ipp      =  mclIpretParamNew()
   ;  mcxTing* throlds        =  mcxTingEmpty(NULL, 20) 
   ;  mcxTing* fnstem         =  NULL
   ;  mcxTing* fninfix        =  NULL
   ;  const char* sortmode    =  "revsize"

   ;  int n_specs             =  0
   ;  int flags = 0

   ;  mcxIO* xfin
   ;  mclMatrix *mx, *cl
   ;  int o, e, m
   ;  double f
   ;  char* p, *q, *r
   ;  const char* me = "clmimac"
   ;  const char* arg_fnin = NULL
   ;  int n_args

   ;  mcxHash* hopts
   ;  mcxOptList* lopts

   ;  if (argc <= 1)
      {  mcxUsage(stdout, me, usagelines)
      ;  exit(0)
   ;  }

      hopts    =  mcxOptHash(options, NULL)
   ;  lopts    =  mcxOptExhaust
                  (hopts, (char**) argv, argc, 1, &n_args, &parseStatus)

  /*  fixme: -imx argument rather than fixed suffix argument ?
  */
   ;  if (parseStatus != STATUS_OK)
      {  mcxErr(me, "initialization failed")
      ;  exit(1)
   ;  }

      while (lopts && lopts->tag)
      {  mcxKV* kv = mcxHashSearch(lopts->tag, hopts, MCX_DATUM_FIND)
      ;  mcxOptAnchor* anch = kv ? (mcxOptAnchor*) kv->val : NULL

      ;  if (!kv)                   /* huh? impossible? fixme< */
         {  lopts = lopts->next
         ;  continue
      ;  }

         if (anch->flags & 4)
         {  mcxWarn
            (  me
            ,  "EXPERIMENTAL using option <%s>, may produce unexpected results"
            ,  anch->tag
            )
      ;  }

         switch(anch->id)
         {  case OPT_ENSTRICT
         :  keepOverlap = FALSE
         ;  break
         ;

            case OPT_VERSION
         :  report_version(me)
         ;  exit(0)
         ;

            case OPT_INPUTFILE
         :  arg_fnin = lopts->val
         ;  break
         ;

            case OPT_OUTPUTINFIX
         :  fninfix = mcxTingNew(lopts->val)
         ;  break
         ;

            case OPT_OUTPUTFILE
         :  fnstem = mcxTingNew(lopts->val)
         ;  break
         ;

            case OPT_W_SELFVAL
         :  f =  atof(lopts->val)
         /* mq: check value */
         ;  ipp->w_selfval = f/100.0
         ;  break
         ;

            case OPT_W_MAXVAL
         :  f =  atof(lopts->val)
         /* mq: check value */
         ;  ipp->w_maxval = f/100.0
         ;  break
         ;

            case OPT_W_CENTER
         :  f =  atof(lopts->val)
         /* mq: check value */
         ;  ipp->w_center = f/100.0
         ;  break
         ;

            case OPT_HELP
         :  mcxUsage(stdout, me, usagelines)
         ;  exit(0)
         ;

            case OPT_TIGHT
         :  mcxTingAppend(throlds, lopts->val)
         ;  mcxTingAppend(throlds, ":")
         ;  break
         ;

            case OPT_SORT
         :  sortmode = lopts->val
         ;  break
         ;

         }

         lopts = lopts->next
   ;  }

      if (!arg_fnin)
      {  mcxErr(me, "-imx option is required")
      ;  mcxExit(0)
   ;  }

      xfin     =  mcxIOnew(arg_fnin, "r")
   ;  mx       =  mclxRead(xfin, EXIT_ON_FAIL)
   ;  report_graph_or_exit(me, mx, xfin->fn)

   ;  ipp->w_selfval = 0.05
   ;  ipp->w_maxval  = 0.95
   ;  ipp->w_center  = 0.0

   ;  if (!throlds->len)
      mcxTingWrite(throlds, "100:")

   ;  p = throlds->str

   ;  while ((p = strchr(p+1, ':')))
      n_specs++

   ;  p = throlds->str

   ;  if (!fnstem)
      fnstem =    n_specs > 1 || fninfix
               ?  mcxTingNew(argv[argc-1])
               :  mcxTingNew("out.imac")

   ;  while ((q = strchr(p, ':')))
      {  mcxIO* xfout

      ;  if (q == p)
         {  p++
         ;  continue
      ;  }

         *q = '\0'
      ;  f  =  atof(p)
      ;  r  =  p
      ;  p  =  q+1

      ;  if (f < 0.5)
         f = 0.5
      ;  else if (f > 99.5)
         f = 99.5

      ;  xfout     =  mcxIOnew(fnstem->str, "w")

      ;  ipp->w_maxval  = f/100.0
      ;  ipp->w_selfval = (100.0-f)/100.0

      ;  if (fninfix)
         mcxTingAppend(xfout->fn, fninfix->str)

      ;  if (n_specs > 1)
         mcxTingAppend(xfout->fn, r)

      ;  cl = mclInterpret(mx, ipp)  

      ;  if (keepOverlap)
         flags |= ENSTRICT_KEEP_OVERLAP

      ;  mclcEnstrict(cl, &o, &m, &e, flags)  

      ;  if (o)
         mcxTell
         (  me
         ,  "%s <%d> instances of overlap (t parameter %.0f)"
         ,  keepOverlap ? "found" : "removed"
         ,  (int) o
         ,  (double) f
         )

      ;  if (m)
         mcxTell
         (  me
         ,  "collected <%d> missing nodes (t parameter %.0f)"
         ,  (int) m
         ,  (double) f
         )

      ;  if (e)
         mcxTell
         (  me
         ,  "removed <%d> empty clusters (t parameter %.0f)"
         ,  (int) e
         ,  (double) f
         )

      ;  if (!strcmp(sortmode, "size"))
         mclxColumnsRealign(cl, mclvSizeCmp)
      ;  else if (!strcmp(sortmode, "revsize"))
         mclxColumnsRealign(cl, mclvSizeRevCmp)
      ;  else if (!strcmp(sortmode, "lex"))
         mclxColumnsRealign(cl, mclvLexCmp)
      ;  else if (!strcmp(sortmode, "none"))
        /* ok */
      ;  else
         mcxErr(me, "ignoring unknown sort mode <%s>", sortmode)

      ;  mclxWriteAscii(cl, xfout, -1, EXIT_ON_FAIL)
      ;  mcxIOfree(&xfout)
   ;  }

      return 0
;  }


