/*            Copyright (C) 2001, 2002, 2003 Stijn van Dongen
 *
 * This file is part of MCL.  You can redistribute and/or modify MCL under the
 * terms of the GNU General Public License; either version 2 of the License or
 * (at your option) any later version.  You should have received a copy of the
 * GPL along with MCL, in the file COPYING.
*/

#include <string.h>
#include <stdio.h>
#include <errno.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/types.h>

#include "report.h"

#include "impala/matrix.h"
#include "impala/vector.h"
#include "impala/io.h"
#include "impala/tab.h"
#include "impala/compose.h"
#include "impala/iface.h"
#include "impala/scan.h"

#include "mcl/interpret.h"
#include "mcl/clm.h"

#include "util/io.h"
#include "util/types.h"
#include "util/err.h"
#include "util/opt.h"
#include "util/minmax.h"


/*
 * html output:
 *    file#c3  cluster 3 (top) anchor
 *    file#e6  element 6 anchor
 *    file#i3  cluster 3 inner nodes anchor
 *    file#o3  cluster 3 outer nodes anchor
 *    file#b3  cluster 3 bottom anchor
 *
 *    div.c cluster/cluster listing.
 *    div.e innernodes/outernodes listing.
 *    div.d alien cluster listing below innernode/outernode listing.
*/

/*
 * TODO:
 *
 *    Separate and modularize formatting code.  a good design has not yet come
 *    to mind though, and as long as there is no pressing need it probably
 *    wont.
 *    The present code is awful if you look one way at it.
 *    It's ok if you look the other way at it.
 *    Treat/tread with care.
 *
 *    make intermediate output format; e.g. M4 macros that
 *    are then reprocessed.
 *
 *    Print useful information in html/head/title: cluster id etc.
 *
 *    index file:
 *    possibly some sample nodes from each cluster. (ideally these would
 *    be as disparate as possible).
 *    Append index.html to fmt base, so foo-index.html or foo/index.html
 *       futurefeature: implement disparacy maximizing algorithm.
 *       futurefeature: user-time reselection of random nodes.
 *
 *    test: html{0,1} x split{0,1} x tab{0,1}
 *
 *    / Are single element clusters done right ?
 *
 *    It might be nice to show node/node edge weights, but that will
 *    require an extra layer of design.
 *    Vision: current focus on cluster/node context.
 *       cluster/graph/node context
 *       cluster1/cluster2/node context require interface design first. 
 *
 *    fix plain text formatting (newline errors).
 *
 *    Nuther threshold for outer listing.
 *
 *    HTML encoding of tab labels.
 *
 *    This thing might eventually evolve into some visualization tool.
 *    It's *code* is basically a playground for coding ideas.
 *    It's output specification should be fairly stable.
 *    The code is totally unreadable. Not to me, I think, but it
 *    does not look remotely maintainable. Some of the reasons are hard.
*/

/* Checklist for cluster/graph properties.
 *    defective clustering (overlap, missing nodes, empty clusters)
 *    sparse cluster (column) domain
 *    sparse node domain (cluster rows, graph rows/cols)
 *    graph node domain subsumes cluster node domain.
 *    cluster node domain subsumes graph node domain.
 *    graph/cluster node domains induce tri-sphere.
*/


const char* me = "clmformat";

const char* usagelines[] =
{  "Usage: clmformat <options>"
,  ""
,  "Options (marked * are obligatory):"
,  "-icl <fname>  *  read clustering matrix from file"
,  "-imx <fname>  *  read corresponding graph matrix from file"
,  "-tab <fname>     read tab file"
,  ""
,  "-fmt <fname>     output file name or output base name"
,  "-dir <dname>     use directory as base entry point, split output"
,  "-infix <str>     insert <str> after base name or base directory"
,  ""
,  "-lump-size <n>   cluster size threshold for split behaviour"
,  "-lump-count <n>  batch size threshold for split behaviour"
,  ""
,  "-nsm <fname>     write node stickiness matrix to file"
,  "-ccm <fname>     write cluster cohesion matrix to file"
,  ""
,  "--adapt          allow non-matching domains and defective clusterings"
,  "-pi <f>          apply inflation to input graph"
,  ""
,  "-dump <fname>    dump line-based output"
,  "-dump-node-sep <str> the separator for node indices or node labels"
,  "                 (default tab) - more dump options are on the TODO list"
,  NULL
}  ;

/*  "--subgraph       take subgraph (when domains don't match)" */

/*
,  "Node stickiness matrix: columns range over all nodes in the graph."
,  "  Given a node c, the entry in row zero shows the mass fraction of edges"
,  "  that have c as tail and for which the head is in the same cluster as c,"
,  "  relative to the mass of all edges that have c as tail."
,  "  The entry in row one lists the number of edges originating from c."
,  "Cluster cohesion matrix: columns range over all clusters."
,  "  Given a cluster C, the entry in row zero shows the average of the node"
,  "  stickiness (defined above) for all nodes c in C."
,  "  The entry in row one lists the number of nodes in c."
,  "Residue projection matrix:"
,  "  Contains projection value of missing nodes onto each of the clusterings."
*/


static const char* html_header =
"<html>\n"
"<head>\n"
"<style type=\"text/css\">\n"
"body { font-family: courier, mono; background: white }\n"
"a:link { text-decoration: none; color: #1111dd; }\n"
"a:visited { text-decoration: none; color: #11dd11; }\n"
"a:active { text-decoration: none; color: #dd1111; }\n"
"div.c { margin-left:6%; whitespace:pre; }\n"
"div.e { margin-left:6%; whitespace:pre; }\n"
"div.d { margin-left:12%; whitespace:pre; }\n"
"</style>\n"
"</head>\n"
"<body>\n" ;

static const char* txt_rule =
"------------------------------------------------------------------------------"
;
static const char* html_footer =
"</body>\n"
"</html>\n" ;

const char* fmt_label_open = "" ;
const char* fmt_label_close = "" ;
const char* fmt_br = "" ;
const char* fmt_2en = "" ;

const char* dump_node_sep = "\t";


void freenames
(  char** fnames
)
   {  char** baseptr = fnames
   ;  if (!fnames)
      return
   ;  while (*fnames)
      {  mcxFree(*fnames)
      ;  fnames++
   ;  }
      mcxFree(baseptr)
;  }


/* todo:
 * options for printing cluster index, cluster confidence,
 * nrof elements ...
 * format for nodes: %i:%t <%i t="%t"> etc
*/
void dodump
(  mcxIO*   xf_dump
,  mclx*    cl
,  mclTab*  tab      /* NULL allowed */
,  mclx*    clvals   /* NULL allowed */
)
   {  int i, j
   ;  for (i=0;i<N_COLS(cl);i++)
      {  mclv* clus = cl->cols+i
      ;  if (clvals)
         {  mclv* clvv = clvals->cols+i
         ;  fprintf
            (  xf_dump->fp
            ,  "%ld%s%ld%s%.2f%s%.2f%s%.2f"
            ,  (long) clus->vid
            ,  dump_node_sep
            ,  (long) clus->n_ivps
            ,  dump_node_sep
            ,  (double) clvv->ivps[0].val
            ,  dump_node_sep
            ,  (double) clvv->ivps[1].val
            ,  dump_node_sep
            ,  (double) clvv->ivps[2].val
            )
      ;  }
         for (j=0;j<clus->n_ivps;j++)
         {  long elid = clus->ivps[j].idx
         ;  if (j || clvals)
            fprintf
            (  xf_dump->fp
            ,  "%s"
            ,  dump_node_sep
            )

         ;  if (tab)
            {  long elos = mclvGetIvpOffset(tab->domain, elid, -1)
            ;  char* lbl = elos >= 0 ? tab->labels[elos] : "_"
            ;  fprintf(xf_dump->fp, "%s", lbl)
            ;  if (elos<0)
               mcxErr
               (  "clmformat dump"
               ,  "Node <%ld> not found in tab file"
               ,  elid
               )
         ;  }
            else
            fprintf(xf_dump->fp, "%ld", (long) elid)
      ;  }
         if (clus->n_ivps)
         fprintf(xf_dump->fp, "\n")
   ;  }
   }


char** mknames
(  mclMatrix* cl
,  const char* base
,  const char* suffix
,  int cutoff
,  int batchsize
,  int do_dir
)
   {  int i
   ;  char** fnames = mcxAlloc((N_COLS(cl)+1) * sizeof(char*), EXIT_ON_FAIL)
   ;  mcxTing* txt = mcxTingEmpty(NULL, 80)
   ;  mcxTing* bch = mcxTingEmpty(NULL, 80)
   ;  mcxTing* res = NULL
   ;  mcxTing* emp = mcxTingNew("")
   ;  int carry = 0

   ;  if (batchsize)
      cutoff = 0

   ;  if (!mcldIsCanonical(cl->dom_cols))
      mcxDie(1, me, "Currently I need sequentially numbered clusters")

   ;  if (do_dir && cutoff)
         res = mcxTingNew(base)
      ,  mcxTingPrintAfter
         (  res
         ,  "%scut%s"
         ,  res->len ? "." : ""
         ,  suffix ? suffix : ""
         )

   ;  for (i=0;i<N_COLS(cl);i++)
      {  int clid = cl->cols[i].vid
      ;  int cut = cutoff && cl->cols[i].n_ivps <= cutoff

      ;  if (!do_dir)
         {  fnames[i] = mcxTingStr(emp)   /* necessary; it *is* printed */
         ;  continue
      ;  }

         if (cut)
         {  fnames[i] = mcxTingStr(res)
         ;  continue
      ;  }

         if (batchsize)
         {  if (!carry)
            {  int j
            ;  for (j=i;j<N_COLS(cl) && carry<batchsize;j++)
               carry+= cl->cols[j].n_ivps
            ;  carry = 0
            ;  mcxTingWrite(bch, base)
            ;  mcxTingPrintAfter
               (  bch
               ,  "%s%d"
               ,  bch->len ? "." : ""
               ,  clid
               )
            ;  if (j-i > 1)
               mcxTingPrintAfter
               (  bch
               ,  "-%d"
               ,  j-1
               )
            ;  if (suffix)
               mcxTingAppend(bch, suffix)
         ;  }
            carry += cl->cols[i].n_ivps

         ;  fnames[i] = mcxTingStr(bch)
         ;  if (carry > batchsize)
            carry = 0
      ;  }
         else
         {  mcxTingWrite(txt, base)
         ;  mcxTingPrintAfter
            (  txt
            ,  "%s%d"
            ,  txt->len ? "." : ""
            ,  clid
            )
         ;  if (suffix)
            mcxTingAppend(txt, suffix)
         ;  fnames[i] = mcxTingStr(txt)
      ;  }
   ;  }

      fnames[N_COLS(cl)] = NULL

   ;  mcxTingFree(&txt)
   ;  mcxTingFree(&bch)
   ;  mcxTingFree(&res)
   ;  mcxTingFree(&emp)
   ;  return fnames
;  }


void settoself
(  mclMatrix* cl
,  mclMatrix* el_on_cl
)
   {  int i, j
   ;  for (i=0;i<N_COLS(cl);i++)
      {  mclv* clvec = cl->cols+i
      ;  long  clid  = clvec->vid
      ;  mclv* elclvec = NULL
      ;  mclp* valivp = NULL
      ;  for (j=0;j<clvec->n_ivps;j++)
         {  long elid = clvec->ivps[j].idx
         ;  elclvec = mclxGetVector(el_on_cl, elid, EXIT_ON_FAIL, elclvec)
         ;  valivp = mclvGetIvp(elclvec, clid, NULL)
         ;  if (!valivp && clvec->n_ivps > 1)
            mcxErr(me, "match error: el %ld cl %ld", elid, clid)
         ;  clvec->ivps[j].val = valivp ? MAX(0.01, valivp->val) : 0.01
      ;  }
      }
   }


void prune_it
(  mclMatrix* el_to_cl  /* must be conforming */
,  mclMatrix* el_on_cl
,  double pct
,  int max
)
   {  int i
   ;  for (i=0;i<N_COLS(el_on_cl);i++)
      {  mclv* elclvec  =  el_on_cl->cols+i
      ;  long clid      =  el_to_cl->cols[i].ivps[0].idx
      ;  mclp* elivp    =  mclvGetIvp(elclvec, clid, NULL)
      ;  double elfrac  =  elivp ? elivp->val : 0.01
      ;  double sum     =  elivp ? elivp->val : 0.05
      ;  int n_others   =  0
      ;  int k          =  -1

      ;  mclvSort(elclvec, mclpValRevCmp)

      ;  while (++k < elclvec->n_ivps && sum < pct && n_others < max)
         {  long y = elclvec->ivps[k].idx
         ;  if (y == clid)
            continue
         ;  sum += elclvec->ivps[k].val
         ;  n_others++
      ;  }

         mclvResize(elclvec, k)
      ;  mclvSort(elclvec, mclpIdxCmp)
      ;  if (!mclvGetIvp(elclvec, clid, NULL))
         mclvInsertIdx(elclvec, clid, elfrac)
   ;  }
   }


/* 0: self value
 * 1: coverage
 * 2: maxcoverage
*/

mclMatrix* mkclvals
(  mclMatrix* mx
,  mclMatrix* cl
,  mclMatrix* cl_on_cl
)
   {  int i
   ;  mclxScore xscore
   ;  mclMatrix* clvals
      =  mclxCartesian
         (  mclvCopy(NULL, cl->dom_cols), mclvCanonical(NULL, 3, 1.0), 1.0 )

   ;  for(i=0;i<N_COLS(cl);i++)
      {  mclv* clvec = cl->cols+i
      ;  mclp* tivp = mclvGetIvp(cl_on_cl->cols+i, clvec->vid, NULL)

      ;  clvals->cols[i].ivps[0].val = tivp ? MAX(tivp->val, 0.001) : 0.001
      ;  mclxSubScan
         (mx, clvec, clvec, &xscore)
      ;  clvals->cols[i].ivps[1].val = MAX(xscore.cov, 0.001)
      ;  clvals->cols[i].ivps[2].val = MAX(xscore.covmax, 0.001)
   ;  }
      return clvals
;  }


long getclusid
(  mclMatrix* el_to_cl
,  long elid
)
   {  long offset = mclxGetVectorOffset(el_to_cl, elid, EXIT_ON_FAIL, -1)
   ;  return el_to_cl->cols[offset].ivps[0].idx
;  }


void mkanindex
(  mcxIO* xf_ind1    /*  print to first; make ref to second */
,  mcxTing* fn_ind2
,  char* title_ind2
,  mclMatrix* mx
,  mclMatrix* cl
,  mclMatrix* el_to_cl
,  mclMatrix* clvals
,  mclTab* tab
,  char** fnames
,  int do_html
,  mclVector* cllist
,  mclVector* ellist
)
   {  double coh, cov, maxcov
   ;  mcxTing* txtIND = mcxTingEmpty(NULL, 80)
   ;  int i
   ;  if (do_html)
      fprintf(xf_ind1->fp, "%s", html_header)

   ;  cov = mclxCoverage(mx, cl, &maxcov)

   ;  coh = mclvSum(cllist)
   ;  if (cllist->n_ivps)
      coh /=  cllist->n_ivps

   ;  if (do_html)
      fprintf
      (  xf_ind1->fp
      ,  "<a name=\"top\"></a>"
      )

   ;  fprintf
      (  xf_ind1->fp
      ,  "Clustering has %ld clusters%s"
         "General measures: cohesion %.2f, cov %.2f, maxcov %.2f\n"
      ,  (long) N_COLS(cl)
      ,  fmt_br
      ,  coh
      ,  cov
      ,  maxcov
      )

   /* hierverder: add average, center, max, min, count */
   /* perhaps even a little 'staafdiagram' to show size distribution */

   ;  if (do_html)
      fprintf
      (  xf_ind1->fp
      ,  "<br><a href=\"#cl\">Clusters</a>, "
         "<a href=\"#el\">Nodes</a>, "
         "<a href=\"%s\">%s</a>"
         "\n%s"
      ,  fn_ind2->str
      ,  title_ind2
      ,  "<p><a name=\"cl\"></a>"
      )
   ;  else
      fprintf(xf_ind1->fp, "\n")

   ;  for (i=0;i<cllist->n_ivps;i++)
      {  long clusid = cllist->ivps[i].idx
      ;  long clusos = mclvGetIvpOffset(clvals->dom_cols, clusid, -1)
            /* redundant; we require canonical col domain somewhere */
            /* fixme; check clusos! */
      ;  mcxTingEmpty(txtIND, 80)

      ;  if (do_html)
         mcxTingPrintAfter
         (  txtIND
         ,  "<a name=\"c%ld\"></a><a href=\"%s#c%ld\">"
         ,  clusid
         ,  fnames[clusos]
         ,  clusid
         )
         /* todo: make this a table? (think js sorting as well) */
      ;  mcxTingPrintAfter
         (  txtIND
         ,  "Cluster %ld%s, sz %ld, cohesion %.2f, coverage %.2f-%.2f%s\n"
         ,  clusid
         ,  do_html ? "</a>" : ""
         ,  (long)   cl->cols[clusos].n_ivps
         ,  (double) clvals->cols[clusos].ivps[0].val
         ,  (double) clvals->cols[clusos].ivps[1].val
         ,  (double) clvals->cols[clusos].ivps[2].val
         ,  do_html ? "<br>" : ""
         )  
      ;  fprintf
         (  xf_ind1->fp
         ,  "%s"
         ,  txtIND->str
         )
   ;  }

      if (do_html)
      fprintf
      (  xf_ind1->fp
      ,  "<p><a name=\"el\"></a>"
      )
   ;  else
      fprintf(xf_ind1->fp, "\n")

   ;  for (i=0;i<ellist->n_ivps;i++)
      {  long clusid = getclusid(el_to_cl, ellist->ivps[i].idx)
      ;  long elid = ellist->ivps[i].idx
      ;  long elos = mclxGetVectorOffset(el_to_cl, elid, EXIT_ON_FAIL, -1)
      ;  mcxTingEmpty(txtIND, 80)

      ;  if (tab)
         mcxTingPrintAfter
         (  txtIND
         ,  "%s%s%s%s"
         ,  fmt_label_open
         ,  tab->labels[elos] 
         ,  fmt_label_close
         ,  fmt_2en
         )

      ;  if (do_html)
         mcxTingPrintAfter
         (  txtIND
         ,  "<a href=\"%s#e%ld\">"
         ,  fnames[clusid] 
         ,  elid
         )

      ;  mcxTingPrintAfter
         (  txtIND
         ,  "Node %ld%s in "
         ,  elid
         ,  do_html ? "</a>" : ""
         )

      ;  if (do_html)
         mcxTingPrintAfter
         (  txtIND
         ,  "<a href=\"%s#c%ld\">"
         ,  fnames[clusid]
         ,  clusid
         )

      ;  mcxTingPrintAfter
         (  txtIND
         ,  "Cluster %ld%s, %.2f%s\n"
         ,  clusid
         ,  do_html ? "</a>" : ""
         ,  (double) ellist->ivps[i].val
         ,  do_html ? "<br>" : ""
         )
      ;  fprintf(xf_ind1->fp, "%s", txtIND->str)
   ;  }

      if (do_html)
      fprintf(xf_ind1->fp, "%s", html_footer)
   ;  mcxTingFree(&txtIND)
;  }


void mkindex
(  mcxIO* xf_ind1
,  mcxIO* xf_ind2
,  mclMatrix* mx
,  mclMatrix* cl
,  mclMatrix* el_to_cl
,  mclMatrix* clvals
,  mclTab* tab
,  char** fnames
,  int do_html
)
   {  mclv* cllist = mclvCopy(NULL, clvals->dom_cols)  /* coverage */
   ;  mclv* ellist = mclvCopy(NULL, cl->dom_rows)
   ;  int i

   ;  for (i=0;i<cllist->n_ivps;i++)
      {  cllist->ivps[i].val = clvals->cols[i].ivps[0].val
      ;  mcldMerge(cl->cols+i, ellist, ellist)
   ;  }

      mcxIOopen(xf_ind1, EXIT_ON_FAIL)
   ;  mcxIOopen(xf_ind2, EXIT_ON_FAIL)

   ;  mcxTell(me, "writing index file [%s]", xf_ind1->fn->str)
   ;  mkanindex
      (  xf_ind1
      ,  xf_ind2->fn
      ,  "Index II"
      ,  mx, cl
      ,  el_to_cl
      ,  clvals
      ,  tab
      ,  fnames
      ,  do_html
      ,  cllist
      ,  ellist
      )

   ;  mclvSort(cllist, mclpValRevCmp)
   ;  mclvSort(ellist, mclpValRevCmp)

   ;  mcxTell(me, "writing index file [%s]", xf_ind2->fn->str)
   ;  mkanindex
      (  xf_ind2
      ,  xf_ind1->fn
      ,  "Index I"
      ,  mx, cl
      ,  el_to_cl
      ,  clvals
      ,  tab
      ,  fnames
      ,  do_html
      ,  cllist
      ,  ellist
      )

   ;  mclvFree(&cllist)
   ;  mclvFree(&ellist)
   ;  mcxIOclose(xf_ind1)
   ;  mcxIOclose(xf_ind2)
;  }


int main
(  int                  argc
,  const char*          argv[]
)
   {  mcxIO *xf_cl = NULL, *xf_mx = NULL, *xf_cut = NULL
      , *xf_nsm = NULL, *xf_ccm = NULL, *xf_tab = NULL, *xf_fmt = NULL
      , *xf_ind1 = NULL
      , *xf_ind2 = NULL
      , *xf_dump = NULL

   ;  mcxTing*    fn_fmt      =  mcxTingNew("out.html")   /* file name */
   ;  mcxTing*    dn_fmt      =  mcxTingNew("out.fmd")   /* dir name */
   ;  mclMatrix   *cl         =  NULL
   ;  mclMatrix   *mx         =  NULL
   ;  mclx* el_to_cl = NULL, *el_on_cl = NULL, *cl_on_cl = NULL
            , *cl_on_el = NULL, *clvals = NULL
   ;  mclv* clclvec = NULL

   ;  mclVector   *meet       =  NULL
   ;  mclTab* tab = NULL

   ;  int         a           =  1
   ;  int         status      =  0
   ;  int         do_html     =  1
   ;  int         do_bf       =  1
   ;  int         duh         =  0
   ;  const char* arg = NULL
   ;  int i, j
   ;  int o, m, e, n_err = 0
   ;  int adapt = 0, subgraph = 0
   ;  int cutoff = 0
   ;  int batchsize = 500
   ;  mcxbool do_dir = FALSE
   ;  const char* infix = NULL
   ;  const char* suffix = NULL
   ;  char** fnames = NULL
   ;  int n_cut = 0
   ;  float inflation = 0.0

   ;  mcxTing *txtCL0 = mcxTingEmpty(NULL, 80)
   ;  mcxTing *txtEL1 = mcxTingEmpty(NULL, 80)
   ;  mcxTing *txtEL2 = mcxTingEmpty(NULL, 80)
   ;  mcxTing *txtCL1 = mcxTingEmpty(NULL, 80)
   ;  mcxTing *txtCL2 = mcxTingEmpty(NULL, 80)

   ;  mclVerbosityIoImpala    =  1

   ;  if (argc == 1)
      goto help

   ;  while(a < argc)
      {  if (!strcmp(argv[a], "-icl"))
         {  if (a++ + 1 < argc)
            xf_cl =  mcxIOnew(argv[a], "r")
         ;  else goto arg_missing
      ;  }
         else if (!strcmp(argv[a], "--adapt"))
         {  adapt = 1
      ;  }
         else if (!strcmp(argv[a], "--subgraph"))
         {  subgraph = 1
      ;  }
         else if (!strcmp(argv[a], "--version"))
         {  report_version(me)
         ;  exit(0)
      ;  }
         else if (!strcmp(argv[a], "-h"))
         {  help
         :  mcxUsage(stdout, me, usagelines)
         ;  mcxExit(status)
      ;  }
         else if (!strcmp(argv[a], "-infix"))
         {  if (a++ + 1 >= argc)
            goto arg_missing
         ;  infix = argv[a]
      ;  }
         else if (!strcmp(argv[a], "-dump-node-sep"))
         {  if (a++ + 1 >= argc)
            goto arg_missing
         ;  dump_node_sep = argv[a]
      ;  }
         else if (!strcmp(argv[a], "-dump"))
         {  if (a++ + 1 >= argc)
            goto arg_missing
         ;  xf_dump = mcxIOnew(argv[a], "w")
      ;  }
         else if (!strcmp(argv[a], "-pi"))
         {  if (a++ + 1 >= argc)
            goto arg_missing
         ;  inflation = atof(argv[a])
      ;  }
         else if (!strcmp(argv[a], "-lump-count"))
         {  if (a++ + 1 >= argc)
            goto arg_missing
         ;  batchsize = strtol(argv[a], NULL, 10)
         ;  cutoff = 0
      ;  }
         else if (!strcmp(argv[a], "-lump-size"))
         {  if (a++ + 1 >= argc)
            goto arg_missing
         ;  cutoff = strtol(argv[a], NULL, 10)
         ;  batchsize = 0
      ;  }
         else if
         (  ((!strcmp(argv[a], "-do")) && (duh = 1))
         || (  (duh = 0)
            || (!strcmp(argv[a], "-no") || !strcmp(argv[a], "-dont"))
            )
         )
         {  if (a++ + 1 >= argc)
            goto arg_missing
         ;  arg = argv[a]
         ;  if (!strcmp(arg, "bf"))
            do_bf = duh
         ;  if (!strcmp(arg, "html"))
            do_html = duh
         ;  else if (!strcmp(arg, "txt"))
            do_html = 1 - duh
         ;  else
            mcxDie(1, me, "unknown mode <%s>", arg)
      ;  }
         else if (!strcmp(argv[a], "-tab"))
         {  if (a++ + 1 >= argc)
            goto arg_missing
         ;  xf_tab = mcxIOnew(argv[a], "r")
      ;  }
         else if (!strcmp(argv[a], "-dir"))
         {  if (a++ + 1 >= argc)
            goto arg_missing
         ;  mcxTingWrite(dn_fmt, argv[a])
         ;  do_dir = TRUE
      ;  }
         else if (!strcmp(argv[a], "-fmt"))
         {  if (a++ + 1 >= argc)
            goto arg_missing
         ;  mcxTingWrite(fn_fmt, argv[a])
      ;  }
         else if (!strcmp(argv[a], "-ccm"))
         {  if (a++ + 1 >= argc)
            goto arg_missing
         ;  xf_ccm = mcxIOnew(argv[a], "w")
      ;  }
         else if (!strcmp(argv[a], "-nsm"))
         {  if (a++ + 1 >= argc)
            goto arg_missing
         ;  xf_nsm = mcxIOnew(argv[a], "w")
      ;  }
         else if (!strcmp(argv[a], "-imx"))
         {  if (a++ + 1 < argc)
            xf_mx = mcxIOnew(argv[a], "r")
         ;  else goto arg_missing
      ;  }
         else if (!strcmp(argv[a], "-h"))
         {  goto help
      ;  }
         else if (0)
         {  arg_missing:
         ;  mcxErr
            (  me
            ,  "flag <%s> needs argument; see help (-h)"
            ,  argv[argc-1]
            )
         ;  mcxExit(1)
      ;  }
         else
         {  mcxErr
            (  me
            ,  "unrecognized flag <%s>; see help (-h)"
            ,  argv[a]
            )
         ;  mcxExit(1)
      ;  }
         a++
   ;  }

      if (!xf_cl || (!xf_dump && !xf_mx))
      {  status = 1
      ;  goto help
   ;  }

      if (!do_html && !do_dir && !strcmp(fn_fmt->str, "out.html"))
      mcxTingWrite(fn_fmt, "out.txt")

      /* fixme: demand graph; make utility function for that,
       * e.g. report_nag
       *
       * also, make it a read option, so it can be detected early.
      */

   ;  if (xf_ccm)
      mcxIOopen(xf_ccm, EXIT_ON_FAIL)
   ;  if (xf_nsm)
      mcxIOopen(xf_nsm, EXIT_ON_FAIL)

   ;  cl =  mclxRead(xf_cl, 0)
   ;  mcxIOfree(&xf_cl)
   ;

      if (xf_mx)
      {  mx =  mclxRead(xf_mx, 0)
      ;  mcxIOfree(&xf_mx)

      ;  if (inflation)
         mclxInflate(mx, inflation)

      ;  if (!mcldEquate(mx->dom_cols, cl->dom_rows, MCL_DOM_EQUAL))
         {  mclVector* meet   =  mcldMeet(mx->dom_cols, cl->dom_rows, NULL)
         ;  mcxErr
            (  me
            ,  "Domain mismatch for matrix ('left') and clustering ('right')"
            )
         ;  report_domain(me, N_COLS(mx), N_ROWS(cl), meet->n_ivps)

         ;  if (adapt)
            {  mclMatrix* tmx, *tcl

            ;  tcl   =  mclxSub(cl, cl->dom_cols, meet)
            ;  mclxFree(&cl)
            ;  cl    =  tcl

            ;  tmx   =     subgraph
                        ?  mclxSub(mx, meet, meet)
                        :  mclxSub(mx, meet, mx->dom_cols)
            ;  mclxFree(&mx)
            ;  mx = tmx
         ;  }
            else
            {  report_exit(me, SHCL_ERROR_DOMAIN)
         ;  }
         }

         if (mclcEnstrict(cl, &o, &m, &e, ENSTRICT_TRULY))
         {  report_partition(me, cl, xf_cl->fn, o, m, e)
         ;  if (adapt)
            report_fixit(me, n_err++)
         ;  else
            report_exit(me, SHCL_ERROR_PARTITION)
      ;  }

         el_to_cl =  mclxTranspose(cl)
      ;  el_on_cl =  mclxCompose(el_to_cl, mx, 0)
      ;  cl_on_cl =  mclxCompose(el_on_cl, cl, 0)
      ;  mclxMakeStochastic(el_on_cl)
      ;  mclxMakeStochastic(cl_on_cl)

      ;  if (1)
         settoself(cl, el_on_cl)
      ;  prune_it(el_to_cl, el_on_cl, 0.95, 10)
      ;  cl_on_el =  mclxTranspose(el_on_cl)

      ;  for (i=0;i<N_COLS(el_to_cl);i++)
         {  long clusid
         ;  int n_nb
         ;  double meetMass, vecMass
         ;  mclVector* clus = NULL

         ;  if ((el_to_cl->cols+i)->n_ivps == 0)
            mcxErr
            (  me
            ,  "element <%ld> not in clustering"
            ,  (long) el_to_cl->cols[i].vid
            )
         ;  else
            {  clusid =  ((el_to_cl->cols+i)->ivps+0)->idx
            ;  clus   =  mclxGetVector(cl, clusid, RETURN_ON_FAIL, clus)
            ;  if (!clus)
               {  mcxErr("clmformat panic", "cluster not found")
               ;  exit(1)
            ;  }
               meet        =  mcldMeet
                              (  mx->cols+i
                              ,  clus
                              ,  meet
                              )
            ;  meetMass    =  mclvSum(meet)
            ;  vecMass     =  mclvSum(mx->cols+i)
            ;  n_nb        =  (mx->cols+i)->n_ivps
         ;  }
         }

         mclvFree(&meet)

      ;  if (xf_nsm)
         {  mclxWriteAscii(el_on_cl, xf_nsm, 6, RETURN_ON_FAIL);
         ;  mcxIOfree(&xf_nsm)
      ;  }
         if (xf_ccm)
         {  mclxWriteAscii(cl_on_cl, xf_ccm, 6, EXIT_ON_FAIL);
         ;  mcxIOfree(&xf_ccm)
      ;  }
      }

      if (xf_tab)
      {  mcxIOopen(xf_tab, EXIT_ON_FAIL)
      ;  tab =  mclTabRead(xf_tab, cl->dom_rows, EXIT_ON_FAIL)
      ;  mcxIOfree(&xf_tab)
        /*
         *  NOTE/fixme/todo: cl might have been changed by adaptation. too
         *  bad then.  we depend on cl as we jointly do a) read the tab file
         *  conditionally on cl row domain above b) mclvGetIvp on the cl row
         *  domain below.  we could go modal, do these things on the matrix
         *  domain if present (mclxGetVector etc), do these things on the cl
         *  row domain otherwise.
        */
   ;  }

     /*  note: branch below is not mem-clean
      *  todo? enable output of some confidence scores in dump?
     */
      if (xf_dump)
      {  mcxIOopen(xf_dump, EXIT_ON_FAIL)
      ;  clvals = mx ? mkclvals(mx, cl, cl_on_cl) : NULL
      ;  dodump(xf_dump, cl, tab, clvals)
      ;  return 0
   ;  }

      if (do_html)
      {  if (do_bf)
            fmt_label_open = "<b>"
         ,  fmt_label_close = "</b>"

      ;  fmt_br = "<br>\n"
      ;  fmt_2en = "&nbsp;&nbsp;"
   ;  }
      else
      {  fmt_br = "\n"
      ;  fmt_2en = "  "
   ;  }

      if (do_dir)
      {  if (chdir(dn_fmt->str))
         {  if (errno == ENOENT)
            {  if (mkdir(dn_fmt->str, 0777))
                  perror("mkdir error")
               ,  mcxDie(1, me, "Cannot create dir <%s>", dn_fmt->str)
            ;  else if (chdir(dn_fmt->str))
               mcxDie(1, me, "Cannot change dir to <%s>", dn_fmt->str)
         ;  }
            else
               perror("mkdir error")
            ,  mcxDie(1, me, "Cannot change dir to <%s>", dn_fmt->str)
      ;  }
         mcxTingEmpty(fn_fmt, 0)
      ;  if (do_html)
         suffix = ".html"
      ;  else
         suffix = ".txt"
      ;  xf_fmt = mcxIOnew("zilch", "w")

      ;  xf_ind1 = mcxIOnew(do_html ? "index.html" : "index.txt", "w")
      ;  xf_ind2 = mcxIOnew(do_html ? "index2.html" : "index2.txt", "w")
   ;  }
      else
      {  xf_fmt = mcxIOnew(fn_fmt->str, "w")
      ;  mcxIOopen(xf_fmt, EXIT_ON_FAIL)  
      ;  if (do_html)
         fprintf(xf_fmt->fp, "%s", html_header)
   ;  }

      fnames = mknames(cl, infix, suffix, cutoff, batchsize, do_dir)

   ;  clvals = mkclvals(mx, cl, cl_on_cl)

   ;  if (do_dir)
      mkindex(xf_ind1, xf_ind2, mx, cl, el_to_cl, clvals, tab, fnames, do_html)
   
   ;  for (i=0;i<N_COLS(cl);i++)
      {  mclv* clvec = cl->cols+i
      ;  mclv* clsortvec = mclvCopy(NULL, clvec)
      ;  long clid = clvec->vid
      ;  long clnext = cl->cols[(i+1) % N_COLS(cl)].vid
      ;  long clprev = cl->cols[(i-1+N_COLS(cl)) % N_COLS(cl)].vid
      ;  mclp* domivp = NULL
      ;  mclv* elclvec = NULL
      ;  mclv* elmxvec = NULL
      ;  double elfrac = 0.0
      ;  double clfrac = 0.0
      ;  int endfile = 0

/* ----------------------------------------------------------------------- */
      
      ;  mcxTingEmpty(txtCL0, 80)

      ;  if (do_dir)
         {  int cut = cutoff && cl->cols[i].n_ivps <= cutoff
         ;  int cutexists = 0
         ;  int newfile = 0

         ;  if (cut)
            {  cutexists = n_cut > 0
            ;  newfile = n_cut == 0
            ;  if (!xf_cut)
               {  mcxTingWrite(xf_fmt->fn, fnames[i])
              /* fnames[i] contains name of cut file; peculiar branch this is.
               * cut file name depends on html/txt branch;
               * perhaps it would be neater to explicitly derive that
               * name over here.
              */
               ;  mcxIOopen(xf_fmt, EXIT_ON_FAIL)  
               ;  xf_cut = xf_fmt
            ;  }
               else
               {  if (do_html)
                  fprintf(xf_cut->fp, "<p><hr noshade size=1><p>")
            ;  }

               endfile = 0
         ;  }
            else if (batchsize)
            {  if (i==0 || strcmp(fnames[i-1], fnames[i]))
               {  mcxTingWrite(xf_fmt->fn, fnames[i])
               ;  mcxIOopen(xf_fmt, EXIT_ON_FAIL)  
               ;  newfile = 1
            ;  }
            /* __ we need the shortcut: otherwise i+1 might overflow */
               if (i+1==N_COLS(cl) || strcmp(fnames[i], fnames[i+1]))
               endfile = 1
         ;  }
            else
            {  mcxTingWrite(xf_fmt->fn, fnames[i])
            ;  mcxIOopen(xf_fmt, EXIT_ON_FAIL)  
            ;  newfile = 1
            ;  endfile = 1
         ;  }

            mcxTell
            (me, "Writing cluster %ld to file [%s]", clid, xf_fmt->fn->str)
         ;  if (cut)
            n_cut++
         ;  if (newfile && do_html)
            fprintf(xf_fmt->fp, "%s", html_header)
      ;  }
         else if (i+1==N_COLS(cl))
         endfile = 1

      ;  if (do_html)
         mcxTingPrintAfter
         (  txtCL0
         ,  "<a name=\"c%ld\"></a>"
         ,  clid
         )
      ;  mcxTingPrintAfter
         (  txtCL0
         ,  "Cluster %ld, nodes %ld"
         ,  (long) clvec->vid
         ,  (long) clvec->n_ivps
         )
      ;  if (mx)
         {  mcxTingPrintAfter
            (  txtCL0
            ,  ", cohesion %.2f"
            ,  clvals->cols[i].ivps[0].val
            )
         ;  mcxTingPrintAfter
            (  txtCL0
            ,  ", coverage %.2f-%.2f"
            ,  clvals->cols[i].ivps[1].val
            ,  clvals->cols[i].ivps[2].val
            )
      ;  }
         if (do_html)
         {  mcxTingPrintAfter
            (  txtCL0
            ,  "<br><b><a href=\"%s#c%ld\">&lt;</a>"
               "&nbsp;"
               "<a href=\"%s#c%ld\">&gt;</a></b>"
            ,  fnames[clprev]
            ,  clprev
            ,  fnames[clnext]
            ,  clnext
            )
         ;  mcxTingPrintAfter
            (  txtCL0
            ,  "&nbsp;<tt><a href=\"#i%ld\">Inner nodes,</a>"
               "&nbsp;<a href=\"#o%ld\">Outer nodes,</a>"
               "&nbsp;<a href=\"#b%ld\">Bottom%s</a></tt>"
            ,  clid ,  clid ,  clid
            ,  do_dir ? "," : "."
            )

         ;  if (do_dir)
            mcxTingPrintAfter
            (  txtCL0
            ,  "&nbsp;<tt><a href=\"%s#c%ld\">Index I,</a>"
               "&nbsp;<a href=\"%s#c%ld\">Index II</a></tt>"
               "<br><tt>---</tt>"
            ,  xf_ind1->fn->str
            ,  clid
            ,  xf_ind2->fn->str
            ,  clid
            )
      ;  }
         else
         mcxTingPrintAfter
         (  txtCL0
         ,  "\n---"
         )

      ;  fprintf(xf_fmt->fp, "%s\n", txtCL0->str)

/* ----------------------------------------------------------------------- */
 /* yes, it's a sign of poverty to use such delimiters */
 /* the part could be stuffed into a sub */

      ;  if (cl_on_cl)
         {  mclv* tmpclvec
         ;  mclp* clivp
         ;  double sumcl = clfrac
         ;  double clval = 0.0
         ;  int n_cls = 0
         ;  long x = 0
         ;  clclvec = mclxGetVector(cl_on_cl, clid, EXIT_ON_FAIL, clclvec)
         ;  clivp = mclvGetIvp(clclvec, clid, NULL)
         ;  if (!clivp && clvec->n_ivps > 1)
            mcxErr
            (  me
            ,  "Cluster %ld does not project onto itself"
            ,  (long) clid
            )
         ;  clval = clivp ? clivp->val : 0.0
         ;  tmpclvec = mclvCopy(NULL, clclvec)
         ;  mcxTingEmpty(txtCL1, 80)

         ;  mclvSort(tmpclvec, mclpValRevCmp)

         ;  if (do_html)
            mcxTingPrintAfter
            (  txtCL1
            ,  "<div class=\"c\">self: %.2f</div>\n"
            ,  clval
            )
         ;  else
            mcxTingPrintAfter
            (  txtCL1
            ,  "%5sself: %.2f\n"
            ,  ""
            ,  clval
            )

         ;  while (sumcl < 0.95 && n_cls < 10 && ++x < tmpclvec->n_ivps)
            {  long otherid = tmpclvec->ivps[x].idx

            ;  if (otherid == clid)
               continue

            ;  if (do_html)
               mcxTingPrintAfter
               (  txtCL1
               ,  "<div class=\"c\">"
                  "<a href=\"%s#c%ld\">cl %ld</a>: %.2f</div>\n"
               ,  fnames[otherid]
               ,  otherid
               ,  otherid
               ,  (double) tmpclvec->ivps[x].val
               )
            ;  else
               mcxTingPrintAfter
               (  txtCL1
               ,  "%5scl %ld: %.2f\n"
               ,  ""
               ,  otherid
               ,  (double) tmpclvec->ivps[x].val
               )
            ;  sumcl += tmpclvec->ivps[x].val
            ;  n_cls++
         ;  }

            mclvFree(&tmpclvec)

         ;  if (do_html)
            mcxTingPrintAfter
            (  txtCL1
            ,  "<tt>---</tt>"
               "<br><a name=\"i%ld\"></a>Inner nodes\n"
               "<br><tt>&nbsp;&nbsp;&nbsp;&nbsp;<a href=\"#c%ld\">Top,</a>"
               "&nbsp;<a href=\"#o%ld\">Outer nodes</a></tt>"
               "<br><tt>---</tt>"
            ,  clid, clid, clid
            )
         ;  else
            mcxTingPrintAfter
            (  txtCL1
            ,  "---\nInner nodes\n---\n"
            )
     ;   }

         fprintf(xf_fmt->fp, "%s", txtCL1->str)
      ;
/* ----------------------------------------------------------------------- */

         mclvSort(clsortvec, mclpValRevCmp)

      ;  for (j=0;j<clsortvec->n_ivps;j++)
         {  long elid = clsortvec->ivps[j].idx
         ;  int elos
         ;  double selfval = clsortvec->ivps[j].val
         ;  domivp = mclvGetIvp(cl->dom_rows, elid, NULL)
               /* clsortvec must be ascending in elid if #3 != NULL*/

         ;  if (!domivp)
            mcxDie(1, me, "match error 2: el %ld cl %ld", elid, clid)

         ;  elos = domivp - cl->dom_rows->ivps

         ;  mcxTingEmpty(txtEL1, 80)

         ;  if (mx)
            elmxvec = mclxGetVector(mx, elid, EXIT_ON_FAIL, NULL)
               /* clsortvec must be ascending in elid if #4 != NULL*/

         ;  if (do_html)
            mcxTingPrintAfter
            (  txtEL1
            ,  "<div class=\"e\"><a name=\"e%ld\"></a>"
            ,  (long) clsortvec->ivps[j].idx
            )
         ;  else
            mcxTingPrintAfter(txtEL1, "%5s", "")

         ;  if (tab)
            mcxTingPrintAfter
            (  txtEL1
            ,  "%s%s%s"
            ,  fmt_label_open
            ,  tab->labels[domivp - cl->dom_rows->ivps]
            ,  fmt_label_close
            )

         ;  mcxTingPrintAfter
            (  txtEL1
            ,  "%sel %ld"
            ,  tab ? "   " : ""
            ,  (long) clsortvec->ivps[j].idx
            )

         ;  if (mx)
            mcxTingPrintAfter
            (  txtEL1
            ,  ", nb %ld"
            ,  (long) elmxvec->n_ivps
            )

         ;  if (do_html)
            mcxTingPrintAfter(txtEL1, "</div>")

         ;  fprintf(xf_fmt->fp, "%s\n", txtEL1->str)

/* ----------------------------------------------------------------------- */

         ;  if(mx)
            {  mclvScore vscore
            ;  double cov, covmax
            ;  mclp *elivp

            ;  elclvec = mclxGetVector(el_on_cl, elid, EXIT_ON_FAIL, NULL)
               /* clsortvec must be ascending in elid if #3 != NULL*/
            ;  elivp = mclvGetIvp(elclvec, clid, NULL)
            ;  elfrac = elivp ? elivp->val : 0.0

            ;  mcxTingEmpty(txtCL2, 80)

            ;  if (do_html)
               mcxTingPrintAfter
               (  txtCL2
               ,  "<div class=\"d\">self: %.2f"
               ,  selfval
               )
            ;  else
               mcxTingPrintAfter
               (  txtCL2
               ,  "%10sself: %.2f"
               ,  ""
               ,  elfrac
               )
            ;  if (1)   /* coverage branch */
               {  mclvSubScan
                  (  mx->cols+elos
                  ,  clvec
                  ,  &vscore
                  )
               ;  mclvScoreCoverage(&vscore, &cov, &covmax)
               ;  mcxTingPrintAfter
                  (  txtCL2
                  ,  ", cov %.2f-%.2f"
                  ,  cov
                  ,  covmax
                  )
            ;  }

               mcxTingPrintAfter
               (  txtCL2
               ,  "%s\n"
               ,  do_html ? "</div>" : ""
               )

            ;  if (1)   /* outer nodes branch */
               {  mclv* tmpelvec = mclvCopy(NULL, elclvec)
               ;  long k
               ;  mclvSort(tmpelvec, mclpValRevCmp)

               ;  for (k=0;k<tmpelvec->n_ivps;k++)
                  {  long y = tmpelvec->ivps[k].idx
                  ;  if (y == clid)
                     continue

                  ;  if (do_html)
                     mcxTingPrintAfter
                     (  txtCL2
                     ,  "<div class=\"d\">"
                        "<a href=\"%s#c%ld\">cl %ld</a>: %.2f</div>\n"
                     ,  fnames[y]
                     ,  y
                     ,  y
                     ,  (double) tmpelvec->ivps[k].val
                     )
                  ;  else
                     mcxTingPrintAfter
                     (  txtCL2
                     ,  "%10scl %ld: %.2f\n"
                     ,  ""
                     ,  y
                     ,  (double) tmpelvec->ivps[k].val
                     )
               ;  }
                  mclvFree(&tmpelvec)
            ;  }

               fprintf(xf_fmt->fp, "%s", txtCL2->str)
         ;  }
         }

/* ----------------------------------------------------------------------- */

      ;  if (mx)
         {  mclv* aliens = mcldMinus(cl_on_el->cols+i, clvec, NULL)
         ;  long j
         ;  mcxTingEmpty(txtEL2, 80)

         ;  mclvSort(aliens, mclpValRevCmp)

         ;  if (do_html)
            mcxTingPrintAfter
            (  txtEL2
            ,  "<tt>---</tt>"
               "<br><a name=\"o%ld\"></a>Outer nodes\n"
               "<br><tt>&nbsp;&nbsp;&nbsp;&nbsp;<a href=\"#c%ld\">Top,</a>"
               "&nbsp;<a href=\"#c%ld\">Inner nodes</a></tt>"
               "<br><tt>---</tt>"
            ,  clid, clid, clid
            )
         ;  else
            mcxTingPrintAfter
            (  txtEL2
            ,  "---\nOuter nodes\n---\n"
            )

         ;  for (j=0;j<aliens->n_ivps;j++)
            {  long alelid = aliens->ivps[j].idx
            ;  double alclval  = aliens->ivps[j].val
            ;  long alclid = getclusid(el_to_cl, alelid)
            ;  domivp = mclvGetIvp(cl->dom_rows, alelid, NULL)
                  /* note that alelid need not be ascending */

            ;  if (!do_html)
               mcxTingPrintAfter(txtEL2, "%5s", "")
            ;  else
               mcxTingPrintAfter(txtEL2, "<div class=\"e\">")

            ;  if (tab)
               mcxTingPrintAfter
               (  txtEL2
               ,  "%s%s%s"
               ,  fmt_label_open
               ,  tab->labels[domivp - cl->dom_rows->ivps]
               ,  fmt_label_close
               )

            ;  if (do_html)
               mcxTingPrintAfter
               (  txtEL2
               ,  "%s<a href=\"%s#e%ld\">"
               ,  tab && !do_html ? "   " : " "
               ,  fnames[alclid]
               ,  alelid
               )

            ;  mcxTingPrintAfter
               (  txtEL2
               ,  "%sel %ld%s in "
               ,  tab ? "  " : ""
               ,  alelid
               ,  do_html ? "</a>" : ""
               )

            ;  if (do_html)
               mcxTingPrintAfter
               (  txtEL2
               ,  "<a href=\"%s#c%ld\">"
               ,  fnames[alclid]
               ,  alclid
               )

            ;  mcxTingPrintAfter(txtEL2, "cl %ld", alclid)

            ;  if (do_html)
               mcxTingPrintAfter(txtEL2, "</a>: %.2f\n</div>", alclval)
            ;  else
               mcxTingPrintAfter(txtEL2, ": %.2f\n", alclval)
         ;  }
            mclvFree(&aliens)

         ;  if (do_html)
            mcxTingPrintAfter
            (  txtEL2
            ,  "<a name=\"b%ld\"></a><br>---<br>\n"
               "&nbsp;<tt><a href=\"#c%ld\">Top,</a>"
               "&nbsp;<a href=\"#i%ld\">Inner nodes,</a>"
               "&nbsp;<a href=\"#o%ld\">Outer nodes</a></tt>"
            ,  clid
            ,  clid
            ,  clid
            ,  clid
            )
         ;  fprintf(xf_fmt->fp, "%s", txtEL2->str)
      ;  }

         if (endfile)
         {  if (do_html)
            fprintf(xf_fmt->fp, "%s\n", html_footer)
         ;  mcxIOclose(xf_fmt)
      ;  }
         else if (do_html && xf_fmt != xf_cut)
         fprintf(xf_fmt->fp, "<p><hr noshade size=1><p>")
      ;  else
         fprintf(xf_fmt->fp, "\n%s\n\n", txt_rule)

      ;  mclvFree(&clsortvec)
   ;  }

      if (xf_cut)
      {  if (do_html)
         fprintf(xf_cut->fp, "%s\n", html_footer)
      ;  mcxIOclose(xf_cut)
   ;  }

      freenames(fnames)

   ;  if (xf_cut != xf_fmt)
      mcxIOfree(&xf_cut)

   ;  mcxIOfree(&xf_fmt)

   ;  mcxIOfree(&xf_ind1)
   ;  mcxIOfree(&xf_ind2)

   ;  mclTabFree(&tab)
   ;  mclxFree(&cl)
   ;  mclxFree(&mx)
   ;  mclxFree(&el_to_cl)
   ;  mclxFree(&el_on_cl)
   ;  mclxFree(&cl_on_cl)
   ;  mclxFree(&cl_on_el)
   ;  mclxFree(&clvals)

   ;  mcxTingFree(&fn_fmt)
   ;  mcxTingFree(&dn_fmt)
   ;  mcxTingFree(&txtCL0)
   ;  mcxTingFree(&txtEL1)
   ;  mcxTingFree(&txtEL2)
   ;  mcxTingFree(&txtCL1)
   ;  mcxTingFree(&txtCL2)
   ;  return 0
;  }

