/*           Copyright (C) 1999, 2000, 2001, 2002, 2003 Stijn van Dongen
 *
 * This file is part of MCL.  You can redistribute and/or modify MCL under the
 * terms of the GNU General Public License; either version 2 of the License or
 * (at your option) any later version.  You should have received a copy of the
 * GPL along with MCL, in the file COPYING.
*/

#include <time.h>

#include "proc.h"
#include "dpsd.h"
#include "expand.h"
#include "inflate.h"
#include "procinit.h"

#include "impala/io.h"
#include "impala/matrix.h"

#include "util/ting.h"
#include "util/err.h"
#include "util/io.h"
#include "util/types.h"
#include "util/alloc.h"
#include "util/minmax.h"


#define ITERATION_INITIAL  1
#define ITERATION_MAIN     2


static int mclVerbosityStart    =  1;        /* mq static */




void  mclDumpMatrix
(  mclMatrix*     mx
,  mclProcParam*  mpp
,  const char*    affix
,  const char*    pfix
,  int            n
,  mcxbool        printValue
)  ;


void  mclDumpVector
(  mclProcParam*  mpp
,  const char*    affix
,  const char*    pfix
,  int            n
,  mcxbool        printValue
)  ;


int doIteration
(  mclMatrix**          mxin
,  mclMatrix**          mxout
,  mclProcParam*        mpp
,  int                  type
)  ;


mclProcParam* mclProcParamNew
(  void
)  
   {  mclProcParam* mpp    =  (mclProcParam*)
                              mcxAlloc(sizeof(mclProcParam), EXIT_ON_FAIL)
   ;  int i

   ;  mpp->mxp             =  mclExpandParamNew()
   ;  mpp->ipretParam      =  mclIpretParamNew()

   ;  mpp->n_ithreads      =  0

   ;  for (i=0;i<5;i++)
      mpp->marks[i]        =  100
   ;  mpp->massLog         =  mcxTingEmpty(NULL, 200)

   ;  mpp->devel           =  0

   ;  mpp->dumpIterands    =  FALSE
   ;  mpp->dumpClusters    =  FALSE
   ;  mpp->dumpChr         =  FALSE
   ;  mpp->dumpModulo      =  1
   ;  mpp->dumpOffset      =  0
   ;  mpp->dumpBound       =  5
   ;  mpp->dumpStem        =  "mcl"
   ;  mpp->dumpMode        =  'a'

   ;  mpp->chaosLimit      =  0.0001
   ;  mpp->lap             =  0.0
   ;  mpp->n_ite           =  0

   ;  mpp->vec_attr        =  NULL

   ;  mpp->mainInflation   =  2
   ;  mpp->mainLoopLength  =  10000
   ;  mpp->initInflation   =  2
   ;  mpp->initLoopLength  =  0

   ;  mpp->inflateFirst    =  FALSE
   ;  mpp->expandOnly      =  FALSE

   ;  mpp->printDigits     =  3
   ;  mpp->printMatrix     =  0

   ;  mpp->dimension       =  0
   ;  return mpp
;  }


void mclProcParamFree
(  mclProcParam** ppp
)
   {  mclProcParam* mpp = *ppp
   ;  mclExpandParamFree(&(mpp->mxp))
   ;  mclIpretParamFree(&(mpp->ipretParam))
   ;  mcxTingFree(&(mpp->massLog))
   ;  mcxFree(mpp)
   ;  *ppp = NULL
;  }


mclMatrix*  mclProcess
(  mclMatrix* mxEven
,  mclProcParam* mpp
)
   {  mclMatrix*        mxOdd       =  NULL
   ;  mclMatrix*        mxCluster   =  NULL
   ;  int               n_cols      =  N_COLS(mxEven)
   ;  int               digits      =  mpp->printDigits
   ;  mclExpandParam    *mxp        =  mpp->mxp
   ;  int               i           =  0
   ;  FILE*             fv          =  stderr
   ;  clock_t           t1          =  clock()

                                       /* mq check memleak for param and stats
                                        * structs and members
                                       */
   ;  if (!mxp->stats)
      mclExpandParamDim(mxp, mxEven)

   ;  mclExpandInitLog(mpp->massLog, mxp)

   ;  if (mpp->printMatrix)
      mclFlowPrettyPrint
      (  mxEven
      ,  stdout
      ,  digits
      ,  "1 After centering (if) and normalization"
      )

   ;  if (mpp->dumpIterands)
      mclDumpMatrix(mxEven, mpp, "ite", "", 0, TRUE)

   ;  if (mxp->v_vectorProgress)
      {  
         if (!mxp->v_pruning)
         fprintf(fv, " ite ")

      ;  for (i=0;i<n_cols/mxp->vectorProgression;i++)
         fputc('-', fv)

      ;  fprintf(fv, mxp->v_pruning ? "\n" : "  chaos  time\n")
   ;  }
      else if (mxp->v_matrixProgress)
      fprintf(fv, " ite  chaos  time\n")

   ;  if (mpp->inflateFirst)
      {  if (mpp->n_ithreads)
         mclxInflateBoss(mxEven, mpp->mainInflation, mpp)
      ;  else
         mclxInflate(mxEven, mpp->mainInflation)
   ;  }

                              /*  mq this is somewhat close to a hack      */
                              /*  doIteration inspects expandOnly as well  */
                              /*  and mclAlgorithm does so too             */
      if (mpp->expandOnly)
      {  doIteration(&mxEven, &mxOdd, mpp, ITERATION_MAIN) 
      ;  mclxFree(&mxEven)
      ;  mxEven = mxOdd
      ;  return mxEven
   ;  }

      for (i=0;i<mpp->initLoopLength;i++)
      {  doIteration 
         (  &mxEven
         ,  &mxOdd
         ,  mpp
         ,  ITERATION_INITIAL
         )
      ;  mclxFree(&mxEven)
      ;  mpp->n_ite++
      ;  mxEven  =  mxOdd
   ;  }

      if (  mpp->initLoopLength
         && (  mxp->v_pruning
            || mxp->v_vectorProgress
            )
         )
      fprintf
      (  fv
      ,  "====== Changing from initial to main inflation now ======\n"
      )

   ;  for (i=0;i<mpp->mainLoopLength;i++)
      {  int convergence
         =  doIteration
            (  &mxEven
            ,  &mxOdd
            ,  mpp
            ,  ITERATION_MAIN
            )

      ;  mclxFree(&mxEven)
      ;  mpp->n_ite++
      ;  mxEven  =  mxOdd

      ;  if (convergence)
         {  if (mxp->v_pruning)
            fprintf(fv, "\n")
         ;  break
      ;  }
      }

      mpp->lap = ((double) (clock() - t1)) / CLOCKS_PER_SEC

   ;  mxCluster = mclInterpret(mxEven, mpp->ipretParam)
   ;  mclxFree(&mxEven)

   ;  return mxCluster
;  }


void mclxCenter
(  mclMatrix*        mx
,  double            w_center
,  double            w_selfval
)
   {  int      col

   ;  for (col=0;col<N_COLS(mx);col++)
      {  mclVector*  vec      =  mx->cols+col
      ;  mclIvp*     match    =  NULL
      ;  int         offset   =  -1
      ;  double      selfval

      ;  if
         (  vec->ivps
         && (  mclvIdxVal(vec, vec->vid, &offset)
            ,  offset >= 0
            )
         )  
         {  match = (vec->ivps+offset)
         ;  selfval     =  match->val
         ;  match->val  =  0.0
      ;  }
         else                    /* create extra room in vector */
         {  mclvResize (vec, (vec->n_ivps)+1)
         ;  match       =  vec->ivps+(vec->n_ivps-1)
         ;  match->val  =  0.0
         ;  match->idx  =  vec->vid
         ;  mclvSort (vec, mclpIdxCmp)
                                 /* ^^^ this could be done by shifting */

         ;  mclvIdxVal(vec, vec->vid, &offset)

         ;  if (offset < 0)
               mcxErr("mclxCenter", "error: insertion failed ?!?")
            ,  mcxExit(1)

         ;  match    =  (vec->ivps+offset)
         ;  selfval  =  0.0
      ;  }

         {  double sum =  mclvSum(vec)

         ;  if (sum == 0.0)
            {  match->val =  1
         ;  }
            else
            {  match->val =   w_center == 0.0
                              ?  0.0
                              :  (  w_center * mclvPowSum(vec, 2)
                                    +  w_selfval * selfval
                                 ) / sum
         ;  }
         }

         mclvNormalize(vec)
   ;  }
   }


int doIteration
(  mclMatrix**          mxin
,  mclMatrix**          mxout
,  mclProcParam*        mpp
,  int                  type
)
   {  int               digits         =  mpp->printDigits
   ;  FILE*             fv             =  stderr
   ;  mclExpandParam*   mxp            =  mpp->mxp  
   ;  mclExpandStats*   stats          =  mxp->stats
   ;  int               bInitial       =  (type == ITERATION_INITIAL)
   ;  const char        *when          =  bInitial ? "initial" : "main"
   ;  int               n_ite          =  mpp->n_ite
   ;  char              msg[80]
   ;  double            inflation      =  bInitial
                                          ?  mpp->initInflation
                                          :  mpp->mainInflation

   ;  if (mxp->v_vectorProgress && !mxp->v_pruning)
      fprintf(fv, "%3d  ", (int) n_ite+1)

   ;  if
      (  (mxp->modeExpand == MCL_EXPAND_DENSE)
      && (n_ite > 0)
      && mclxNrofEntries(*mxin) < pow(N_COLS(*mxin), 1.5)
      )
      mxp->modeExpand = MCL_EXPAND_SPARSE
   /* apparently I don't use DENSE mode anymore
    * and probably the modeExpand button neither.
   */

   ;  *mxout =  mclExpand(*mxin, mxp)

   ;  if (n_ite >= 0 && n_ite < 5)
      mpp->marks[n_ite] = (int)(100.001*mxp->stats->mass_final_low[mxp->nj])

   ;  if (n_ite >= 0 && n_ite < mxp->nl)
      mclExpandAppendLog(mpp->massLog, mxp->stats, n_ite)

   ;  if (mpp->dumpChr)
      {  mclMatrix* chr = mxp->stats->mx_chr
      ;  int k

      ;  for (k=0;k<N_COLS(chr);k++)
         ((chr->cols+k)->ivps+0)->val
         =  mclvIdxVal((*mxout)->cols+k, k, NULL)
   ;  }

      if (mpp->printMatrix)
      {  sprintf
         (  msg, "%d%s%s%s"
         ,  (int) 2*n_ite+1, " After expansion (", when, ")"
         )
      ;  if (mxp->v_vectorProgress)
         fprintf(stdout, "\n")
      ;  mclFlowPrettyPrint(*mxout, stdout, digits, msg)
   ;  }

      if (mxp->v_vectorProgress)
      {  if (mxp->v_pruning)
         fprintf
         (  fv
         ,  "\nchaos <%.2f> time <%.2f>\n"
         ,  (double) mxp->stats->chaosMax
         ,  (double) mxp->stats->lap
         )
      ;  else
         fprintf
         (fv, " %6.2f %5.2f\n", (double) stats->chaosMax, (double) stats->lap)
   ;  }
      else if (!mxp->v_pruning && mxp->v_matrixProgress)
      fprintf
      (  fv
      ,  "%3d  %6.2f %5.2f\n"
      ,  (int) n_ite+1
      ,  (double) stats->chaosMax
      ,  (double) stats->lap
      )

   ;  if (mpp->expandOnly)
      return 1

   ;  if (mxp->v_pruning || mpp->dumpClusters)
      {  if (mxp->v_pruning)
         {  if (mclVerbosityStart && mxp->v_explain)
            {  mclExpandStatsHeader(fv, stats, mxp)
            ;  mclVerbosityStart = 0
         ;  }
            mclExpandStatsPrint(stats, fv)
      ;  }

         if (mpp->dumpClusters)
         {  mclMatrix*  clus  =  mclInterpret(*mxout, mpp->ipretParam)
         ;  mclDumpMatrix(clus, mpp, "cls", "", n_ite+1, FALSE)
         ;  mclxFree(&clus)
      ;  }
      }

      if (mpp->n_ithreads)
      mclxInflateBoss(*mxout, inflation, mpp)
   ;  else
      mclxInflate(*mxout, inflation)

   ;  if (mpp->printMatrix)
      {  sprintf
         (  msg,  "%d%s%s%s"
         ,  (int) 2*n_ite+2, " After inflation (", when, ")"
         )
      ;  if (mxp->v_vectorProgress)
         fprintf(stdout, "\n")
      ;  mclFlowPrettyPrint(*mxout, stdout, digits, msg)
   ;  }

      if (mpp->dumpIterands)
      mclDumpMatrix(*mxout, mpp, "ite", "", n_ite+1, TRUE)

   ;  if (mpp->dumpChr)
      {  mclMatrix* chr = mxp->stats->mx_chr
      ;  int k

      ;  for (k=0;k<N_COLS(chr);k++)
         ((chr->cols+k)->ivps+1)->val
         =  mclvIdxVal((*mxout)->cols+k, k, NULL)

      ;  mclDumpMatrix
         (  chr
         ,  mpp
         ,  "chr"
         ,  ""
         ,  n_ite+1
         ,  TRUE
         )
   ;  }

      if (stats->chaosMax < mpp->chaosLimit)
      return 1
   ;  else
      return 0
;  }


void  mclDumpMatrix
(  mclMatrix*     mx
,  mclProcParam*  mpp  
,  const char*    affix
,  const char*    postfix
,  int            n
,  mcxbool        printValue
)
   {  mcxIO*   xfDump
   ;  char snum[18]
   ;  mcxTing*  fname

   ;  if (  (  mpp->dumpOffset
            && (n<mpp->dumpOffset)
            )
         || (  mpp->dumpBound
            && (n >= mpp->dumpBound)
            )
         || (  ((n-mpp->dumpOffset) % mpp->dumpModulo) != 0)
         )
         return

   ;  fname = mcxTingNew(mpp->dumpStem)
   ;  mcxTingAppend(fname, affix)

   ;  sprintf(snum, "%d", (int) n)
   ;  mcxTingAppend(fname, snum)
   ;  mcxTingAppend(fname, postfix)

   ;  xfDump   =  mcxIOnew(fname->str, "w")

   ;  if (mcxIOopen(xfDump, RETURN_ON_FAIL) != STATUS_OK)
      {  mcxErr
         ("mclDumpMatrix", "cannot open stream <%s>, ignoring", xfDump->fn->str)
      ;  return
   ;  }
      else
      {  if (mpp->dumpMode == 'a')
         mclxWriteAscii(mx, xfDump, printValue ? 8 : -1, RETURN_ON_FAIL)
      ;  else
         mclxWriteBinary(mx, xfDump, RETURN_ON_FAIL)
   ;  }

      mcxIOfree(&xfDump)
   ;  mcxTingFree(&fname)
;  }


void  mclDumpVector
(  mclProcParam*  mpp
,  const char*    affix
,  const char*    postfix
,  int            n
,  mcxbool        printValue
)
   {  mcxIO*   xf
   ;  char snum[18]
   ;  mcxTing*  fname

   ;  if (  (  mpp->dumpOffset
            && (n<mpp->dumpOffset)
            )
         || (  mpp->dumpBound
            && (n >= mpp->dumpBound)
            )
         )
         return

   ;  fname = mcxTingNew(mpp->dumpStem)
   ;  mcxTingAppend(fname, affix)

   ;  sprintf(snum, "%d", (int) n)
   ;  mcxTingAppend(fname, snum)
   ;  mcxTingAppend(fname, postfix)

   ;  xf =  mcxIOnew(fname->str, "w")
   ;  if (mcxIOopen(xf, RETURN_ON_FAIL) == STATUS_FAIL)
      {  mcxTingFree(&fname)
      ;  mcxIOfree(&xf)
      ;  return
   ;  }

   ;  if (mpp->dumpMode == 'a')
      mclvWriteAscii(mpp->vec_attr, xf->fp, printValue ? 8 : -1)
   ;  else
      mclvWrite(mpp->vec_attr, xf, RETURN_ON_FAIL)

   ;  mcxIOfree(&xf)
   ;  mcxTingFree(&fname)
;  }


void mclProcPrintInfo
(  FILE* fp
,  mclProcParam* mpp  
)
   {  mclShowSettings(fp, mpp, FALSE)
;  }


