/*           Copyright (C) 1999, 2000, 2001, 2002, 2003 Stijn van Dongen
 *
 * This file is part of MCL.  You can redistribute and/or modify MCL under the
 * terms of the GNU General Public License; either version 2 of the License or
 * (at your option) any later version.  You should have received a copy of the
 * GPL along with MCL, in the file COPYING.
*/


#ifndef impala_matrix_h
#define impala_matrix_h

#include <stdio.h>

#include "ivp.h"
#include "vector.h"
#include "pval.h"

#include "util/types.h"



typedef struct
{  mclVector*     cols
;  mclVector*     dom_cols
;  mclVector*     dom_rows
;
}  mclMatrix      ;

#define mclx mclMatrix


/* INTEGRITY
 *    Remember the constraint on a vector: The idx members of the ivp array
 *    need to be nonnegative and ascending.  The val members are usually
 *    positive, but this is not mandated.
 *
 *    cols is an array of vectors. The count of vectors is equal to
 *    dom_cols->n_ivps. The successive vids (the vid member) of the vectors in
 *    cols are the indices (the idx members of the ivps) in dom_cols. These
 *    vids must be ascending and nonnegative.
 *    Obviously there is duplication of data here; the vid members of the
 *    vectors in cols are exactly the indices of the dom_cols member.  This
 *    duplication is quite useful however, as many matrix operations require
 *    domain access, checking, and manipulation that can easily be formulated
 *    in terms of base vector methods.
 *    Only when doing non-standard stuff with matrices one must take care
 *    to maintain data integrity; i.e. when adding a column to a matrix.
 *    An example of this is found in mcl/clm.c
 *
 *    The vectors accessible via cols have entries (i.e. ivp idx members) that
 *    must be present as indices (of the ivps in) in dom_rows.
 *
 *    The setting cols == NULL, dom_cols->n_ivps == 0, dom_cols->ivps == NULL
 *    is obviously legal.
 *
 *    In normal mode of operation, all matrix operations are simply carried out
 *    by routines in this interface, and they will ensure the integrity of the
 *    matrix structure. If you need to do something out of the ordinary, beware
 *    of the constraints just given.
 *
 * NOTE
 *    All unary and binary operations remove resulting entries that are zero.
 *    E.g. mclxAdd, mclxHadamard (both binary).  Unary examples are currently
 *    perhaps not present.
 *
 *    This means it is dangerous to use those operations on matrices
 *    representing value tables that contain zero-valued entries -
 *    one might want to combine two different such matrices on different
 *    domains simply by using mclxAdd, but the zero-valued entries will
 *    be removed.
 *
 *    There is currently no easy way around this. See also ../README .
*/

 /* only allowed as rvalue */
#define N_COLS(mx) ((long) ((mx)->dom_cols->n_ivps * 1))
#define N_ROWS(mx) ((long) ((mx)->dom_rows->n_ivps * 1))

#define mclxRowCanonical(mx) mcldIsCanonical((mx)->dom_rows)
#define mclxColCanonical(mx) mcldIsCanonical((mx)->dom_cols)

/* args become members of object */

mclMatrix* mclxAllocZero
(  mclVector*  dom_cols
,  mclVector*  dom_rows
)  ;


/* args become members of object
*/

mclMatrix* mclxCartesian
(  mclVector*  dom_cols
,  mclVector*  dom_rows
,  double  val
)  ;


/*    All arguments remain owned by caller.
 *
 *    The domains of the new matrix will always be clones of the selection
 *    domains as supplied by caller, even if those are not subdomains of the
 *    corresponding matrix domains.
*/

mclMatrix*  mclxSub
(  const mclMatrix*     mx
,  const mclVector*     colSelect
,  const mclVector*     rowSelect
)  ;


double  mclxSelectValues
(  mclMatrix*  mx
,  double      *lft        /* NULL for turning of lft comparison  */
,  double      *rgt        /* NULL for turning of rgt comparison  */
,  mcxbits     equate      /*  0,1,or 2 of { MCLX_GQ,  MCLX_LQ }  */
)  ;


mclMatrix* mclxIdentity
(  mclVector* vec
)  ;


mclMatrix* mclxCopy
(  const mclMatrix*        mx
)  ;


void mclxFree
(  mclMatrix**    mx
)  ;


mclMatrix* mclxDiag
(  mclVector* vec
)  ;


mclMatrix* mclxConstDiag
(  mclVector* vec
,  double  c
)  ;


void mclxMakeSparse
(  mclMatrix*     mtx
,  int                     maxDensity
)  ;


void mclxMakeStochastic
(  mclMatrix*     mx
)  ;


mclMatrix* mclxTranspose
(  const mclMatrix*        src
)  ;


void mclxMakeCharacteristic
(  mclMatrix*     mtx
)  ;

void mclxHdp
(  mclMatrix*     mx
,  double                   power
)  ; 


long mclxSubNrofEntries
(  const mclMatrix*        m
,  const mclVector*        colSelect
,  const mclVector*        rowSelect
)  ;


long mclxNrofEntries
(  const mclMatrix*        m
)  ;


double mclxSubMass
(  const mclMatrix*        m
,  const mclVector*        colSelect
,  const mclVector*        rowSelect
)  ;


double mclxMass
(  const mclMatrix*  m
)  ;


/* operation's second arg should be double */

void mclxUnary
(  mclMatrix*  m1
,  double  (*operation)(pval, void*)
,  void*  arg
)  ;


mclVector* mclxColNums
(  const mclMatrix*  m
,  double           (*f)(const mclVector * vec)
,  mcxenum           mode
)  ;


mclMatrix* mclxMax
(  const mclMatrix*  m1
,  const mclMatrix*  m2
)  ;


mclMatrix* mclxAdd
(  const mclMatrix*  m1
,  const mclMatrix*  m2
)  ;


mclMatrix* mclxHadamard
(  const mclMatrix*  m1
,  const mclMatrix*  m2
)  ;


void mclxInflate
(  mclMatrix*  mx
,  double                power
)  ;


/* result has col domain: the union of col domains
 * and row domain: the union of row domains
*/

mclMatrix* mclxBinary
(  const mclMatrix*  m1
,  const mclMatrix*  m2
,  double (*operation)(pval, pval)
)  ;


double mclxMaxValue
(  const mclMatrix*  m
)  ;


void mclxScale
(  const mclMatrix*  m
,  double            f
)  ;


/* mode one of
 *    MCL_VECTOR_COMPLETE or
 *    MCL_VECTOR_SPARSE
*/

mclVector* mclxColSums
(  const mclMatrix*  m
,  mcxenum           mode
)  ;


mclVector* mclxColSizes
(  const mclMatrix*  m
,  mcxenum           mode
)  ;


int mclxGetVectorOffset
(  const mclMatrix*  mx
,  long              vid
,  mcxOnFail         ON_FAIL
,  long              offset
)  ;


mclVector* mclxGetVector
(  const mclMatrix*  mx
,  long              vid
,  mcxOnFail         ON_FAIL
,  const mclVector*  offset
)  ;


mclVector* mclxGetNextVector
(  const mclMatrix*  mx
,  long              vid
,  mcxOnFail         ON_FAIL
,  const mclVector*  offset
)  ;


/* vids-column association is disrupted! */

void  mclxColumnsRealign
(  mclMatrix*        m
,  int              (*cmp)(const void* vec1, const void* vec2)
)  ;


/* These can be used to map or permute domains.
 *
 * Permutation or mapping of a matrix can also be achieved using matrix
 * multiplication.  These two methods do in-place modification.
*/

mcxstatus mclxMapRows
(  mclMatrix   *mx
,  mclMatrix  *map
)  ;


mcxstatus mclxMapCols
(  mclMatrix  *mx
,  mclMatrix  *map
)  ;


mclx* mclxMakeMap
(  mclVector*  dom_cols
,  mclVector*  new_dom_cols
)  ;


#endif

