/* -*- C -*- 
   mboxgrep - scan mailbox for messages matching a regular expression
   Copyright (C) 2000, 2001  Daniel Spiljar
   
   Mboxgrep is free software; you can redistribute it and/or modify it 
   under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   Mboxgrep is distributed in the hope that it will be useful, but 
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with mboxgrep; if not, write to the Free Software Foundation, 
   Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
   
   $Id: scan.c,v 1.6 2001/12/02 17:27:02 dspiljar Exp $ */

#include <config.h>

#include <unistd.h>
#include <stdio.h>
#include <regex.h>
#include <stdlib.h>
#include <sys/types.h>
#include <dirent.h>
#include <time.h>
#include <errno.h>
#include <string.h>
#ifdef HAVE_LIBZ
#include <zlib.h>
#define BUFLEN 16384
#endif /* HAVE_LIBZ */
#ifdef HAVE_LIBPCRE
#include <pcre.h>
#endif /* HAVE_LIBPCRE */

#include "scan.h"
#include "mbox.h"
#include "mh.h"
#include "maildir.h"
#include "wrap.h"
#include "md5.h"

void scan_mailbox(char path[], format_t format)
{
  static FILE *boxf, *out;
  extern FILE *tmpp;
#ifdef HAVE_LIBZ
  static gzFile *zboxf;
  extern gzFile *ztmpp;
#endif /* HAVE_LIBZ */
#ifdef HAVE_LIBPCRE
  extern pcre *pcre_pattern;
  extern pcre_extra *hints;
  extern int perl;
  int of[BUFSIZ];
#endif /* HAVE_LIBPCRE */
  static DIR *boxd, *foo;
  static maildir_t *maildird;
  static message_t *msg;
  extern regex_t posix_pattern;
  extern char *pipecmd, *outboxname;
  extern action_t action;
  extern int invert, count, headers, body, merr;
  extern int dedup;
  int delete = 0;
  char date_str[80];
  int isdup = 0;
  time_t tt;
  struct tm *ct;
  extern checksum_t *cs;


  if (format == MAILDIR && action == WRITE)
    {
      foo = opendir(outboxname); /* do NOT change this to m_opendir! */
      if (foo == NULL && errno == ENOENT)
	maildir_create(outboxname);
      else closedir(foo);

      if (-1 == maildir_check(outboxname))
	{
	  if (merr)
	    fprintf(stderr, "%s: %s: not a maildir folder\n", APPNAME, 
		    outboxname);
	  exit(2);
	}
    }

  count = 0;
  if (action == DELETE)
    delete = 1;

  if (format == MBOX)
    {
      boxf = (FILE *) mbox_open(path, "r");
      if (boxf == NULL) return;
    }
#ifdef HAVE_LIBZ
  else if (format == ZMBOX)
    {
      zboxf = (gzFile *) gzmbox_open(path, "r");
      if (zboxf == NULL) return;
    }
#endif /* HAVE_LIBZ */
  else if ((format == MH) || (format == NNMH) || (format == NNML))
    {
      boxd = mh_open(path);
      if (boxd == NULL) return;
    }
  else if (format == MAILDIR)
    {
      maildird = maildir_open(path);
      if (maildird == NULL) return;
    }

  for (;;)
    {
      int res1 = 1, res2 = 1;

      if (format == MBOX)
	msg = (message_t *)mbox_read_message (boxf);
#ifdef HAVE_LIBZ
      if (format == ZMBOX)
	msg = (message_t *)gzmbox_read_message (zboxf);
#endif /* HAVE_LIBZ */
      else if ((format == MH) || (format == NNMH) || (format == NNML))
	msg = (message_t *)mh_read_message(boxd);
      else if (format == MAILDIR)
	msg = (message_t *)maildir_read_message(maildird);

      if (msg == NULL) break;

      if (msg->from == NULL) msg->from = (char *) strdup("nobody");

#ifdef HAVE_LIBPCRE
      if (perl)
	{
	  if (headers)
	    res1 = pcre_exec (pcre_pattern, hints, msg->headers,
			      (int) strlen(msg->headers), 0, 0, of, BUFSIZ);
	  if (body)
	    res2 = pcre_exec (pcre_pattern, hints, msg->body,
			      (int) strlen(msg->body), 0, 0, of, BUFSIZ);

	  res1 = res1 ^ 1;
	  res2 = res2 ^ 1;
	}
      else
#endif /* HAVE_LIBPCRE */
	{
	  if (headers)
	    res1 = regexec (&posix_pattern, msg->headers, 0, NULL, 0);
	  if (body)
	    res2 = regexec (&posix_pattern, msg->body, 0, NULL, 0);
	}

      if (dedup)
	isdup = md5_check_message (msg->body, cs);

      if (((res1 == 0) | (res2 == 0)) ^ ((invert ^ delete)) &&
	  ((dedup && !isdup) || !dedup))
	{
	  if (action == DISPLAY)
	    {
	      if (format != MBOX && format != ZMBOX
		  && 0 != strncmp("From ", msg->headers, 5))
		{
		  tt = time(NULL);
		  ct = localtime(&tt);
		  strftime(date_str, 80, "%a %b %d %H:%M:%S %Y", ct);
		  fprintf(stdout, "From %s  %s\n", msg->from, date_str);
		}
	      fprintf(stdout, "%s\n%s", msg->headers, msg->body);
	    }
	  else if (action == WRITE)
	    {
	      if (format == MAILDIR)
		maildir_write_message(msg, outboxname);
	      else if (format == MH || format == NNMH || format == NNML)
		mh_write_message(msg, outboxname);
	      else if (format == MBOX)
		{
		  out = mbox_open(outboxname, "w");
		  fprintf(out, "%s\n%s", msg->headers, msg->body);
		  mbox_close(out);
		}
	    }
	  else if (action == PIPE)
	    {
	      out = popen (pipecmd, "w");
	      if (out == NULL)
		{
		  if (merr)
		    {
		      fprintf(stderr, "%s: %s: ", APPNAME, pipecmd);
		      perror(NULL);
		    }
		  exit(2);
		} /* if */
	      fprintf(out, "%s\n%s", msg->headers, msg->body);
	      pclose (out);
	    }
	  else if (action == COUNT)
	    ++count;
	  else if (action == DELETE && format == MBOX)
	    fprintf(tmpp, "%s\n%s", msg->headers, msg->body);
#ifdef HAVE_LIBZ
	  else if (action == DELETE && format == ZMBOX)
	    {
	      int quux, len, baz;

	      quux = 0;
	      baz = strlen(msg->headers);
	      for (;;)
		{
		  len = gzwrite(ztmpp, (msg->headers+quux), 
				(((quux + BUFLEN) < baz) ? BUFLEN : 
				 (baz - quux)));
		  quux += len;
		  if (quux == baz)
		    break;
		}
	      gzwrite(ztmpp, "\n", 1);
	      quux = 0;
	      baz = strlen(msg->body);
	      for (;;)
		{
		  len = gzwrite(ztmpp, (msg->body+quux), 
				(((quux + BUFLEN) < baz) ? BUFLEN : 
				 (baz - quux)));
		  quux += len;
		  if (quux == baz)
		    break;
		}
	    }
#endif /* HAVE_LIBZ */
	} /* if */
      else if (((((res1 == 0) | (res2 == 0)) ^ invert) && delete) &&
	       ((format == MH) || (format == NNMH) 
		|| (format == NNML) || (format == MAILDIR)))
	m_unlink(msg->filename);
      free(msg->body);
      free(msg->headers);
      free(msg);
    } /* for */
  if (format == MBOX)
    mbox_close(boxf);
#ifdef HAVE_LIBZ
  if (format == ZMBOX)
    gzmbox_close(zboxf);
#endif /* HAVE_LIBZ */
  else if ((format == MH) || (format == NNMH) || (format == NNML))
    mh_close(boxd);
} /* scan_mailbox */

int md5_check_message (char *body, checksum_t *chksum)
{
  struct md5_ctx a;
  unsigned char b[16];
  int i,x;

  md5_init_ctx (&a);
  if (body == NULL)
    md5_process_bytes ("", 0, &a);
  else
    md5_process_bytes (body, strlen(body), &a);
  md5_finish_ctx(&a, b);

  for (i = 0; i < chksum->n; i++)
    {
      if (0 == strncmp (chksum->md5[i], b, 16)) 
	return 1; 
    }

  chksum->md5 = 
	(char **) xrealloc (chksum->md5, (1 + chksum->n) * sizeof (char *));
  chksum->md5[chksum->n] = strdup (b);

  (chksum->n)++;

  return 0;
}
