/* -*- C -*- 
  mboxgrep - scan mailbox for messages matching a regular expression
  Copyright (C) 2000, 2001  Daniel Spiljar

  Mboxgrep is free software; you can redistribute it and/or modify it 
  under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  Mboxgrep is distributed in the hope that it will be useful, but 
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with mboxgrep; if not, write to the Free Software Foundation, 
  Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

  $Id: main.c,v 1.9 2001/12/02 17:27:02 dspiljar Exp $ */

#include <config.h>

#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <regex.h>
#include <stdlib.h>
#include <unistd.h>
#ifdef HAVE_LIBPCRE
#include <pcre.h>
#endif /* HAVE_LIBPCRE */
#ifdef HAVE_LIBZ
#include <zlib.h>
#endif /* HAVE_LIBZ */

#include "getopt.h"
#include "mboxgrep.h"
#include "misc.h"
#include "info.h"
#include "mbox.h"
#include "mh.h"
#include "scan.h"

regex_t posix_pattern;
#ifdef HAVE_LIBPCRE
pcre *pcre_pattern;
pcre_extra *hints;
int perl;
#endif /* HAVE_LIBPCRE */
format_t format = MBOX; /* default folder format */
lockmethod_t lock = FCNTL; /* default file locking method */
char *boxname, *outboxname, *pipecmd, *tmpfilename, hostname[BUFSIZ];
action_t action = DISPLAY; /* default action on matched messages */
int pid, maildir_count = 0;
int invert = 0, count = 0;
int body = 0, headers = 0;
int dedup = 0;
int merr = 1; /* report errors by default */
FILE *tmpp;
#ifdef HAVE_LIBZ
gzFile *ztmpp;
#endif /* HAVE_LIBZ */
checksum_t *cs;

int
main (int argc, char **argv)
{
  int option_index = 0;
  int c, extended = 1, ignorecase = 0;
#ifdef HAVE_LIBPCRE
  int errptr;
  const char *error;
#endif /* HAVE_LIBPCRE */
  int haveregex = 0, havemailbox = 0;
  static char *regex_s;
  int singlefile = 0;

  int errcode = 0;
  char errbuf[BUFSIZ];

  static struct option long_options[] = {
    {"count", 0, 0, 'c'},
    {"delete", 0, 0, 'd'},
/*  {"date", 1, 0, 'D'}, */
    {"extended-regexp", 0, 0, 'E'},
    {"basic-regexp", 0, 0, 'G'},
    {"perl-regexp", 0, 0, 'P'},
    {"help", 0, 0, 'h'},
    {"ignore-case", 0, 0, 'i'},
    {"mailbox-format", 1, 0, 'm'},
    {"no", 1, 0, 'n' },
    {"pipe", 1, 0, 'p'},
    {"regexp", 1, 0, 'e'},
    {"invert-match", 0, 0, 'v'},
    {"version", 0, 0, 'V'},
    {"headers", 0, 0, 'H'},
    {"body", 0, 0, 'B'},
    {"no-messages", 0, 0, 's'},
    {"output", 1, 0, 'o'},
    {"no-file-lock", 0, 0, 'x'},  /* deprecated, use `--file-lock=none' */
    {"no-duplicates", 0, 0, 200},
    {"file-lock", 1, 0, 'l'},
    {0, 0, 0, 0}
  };

  while (1)
    {
      c = getopt_long (argc, argv, "BcdEe:GHhil:m:n:o:Pp:sVvx", long_options, 
		       &option_index);

      if (c == -1)
	break;

      switch (c)
	{
	case '?':
	  usage();
	case 'c':
	  action = COUNT;
	  break;
	case 'd':
	  action = DELETE;
	  break;
	case 'e':
	  regex_s = strdup (optarg);
	  haveregex = 1;
	  break;
	case 'o':
	  outboxname = strdup(optarg);
	  action = WRITE;
	  break;
	case 'E':
	  extended = 1;
	  break;
	case 'G':
	  extended = 0;
	  break;
	case 'P':
#ifdef HAVE_LIBPCRE
	  extended = 0;
	  perl = 1;
#else
	  fprintf(stderr, 
		  "%s: Support for Perl regular expressions not "
		  "compiled in\n");
	  exit(2);
#endif /* HAVE_LIBPCRE */
	  break;
	case 'h':
	  help ();
	  break;
	case 'i':
	  ignorecase = 1;
	  break;
	case 'm':
	  format = folder_format (optarg);
	  break;
	case 'l':
	  lock = lock_method (optarg);
	  break;
	case 'p':
	  action = PIPE;
	  pipecmd = strdup (optarg);
	  break;
	case 'V':
	  version ();
	  break;
	case 'v':
	  invert = 1;
	  break;
	case 'H':
	  headers = 1;
	  break;
	case 'B':
	  body = 1;
	  break;
	case 's':
	  merr = 0;
	  break;
	case 'x':
	  lock = 0;
	  break;
        case 200:
          dedup = 1;
          break;
        case 'n':
	  {
	    switch (optarg[0])
	      {
	        case 'd':
		  dedup = 1;
		  break;
	        default:
		  fprintf(stderr, "%s: invalid option -- n%c\n", 
			  APPNAME, optarg[0]);
		  exit(2);
	      }
	  }
	} /* switch */
    } /* while */

  if ((body == 0) && (headers == 0))
    {
      body = 1;
      headers = 1;
    }

  if (format == MAILDIR && action == WRITE)
    {
      gethostname(hostname, BUFSIZ);
      pid = (int) getpid();
    }

  cs = (checksum_t *) malloc (sizeof(checksum_t));
  cs->md5 = (char **) xcalloc (1, sizeof (char **));
  cs->n = 0;

  if (optind < argc && ! haveregex)
    {
      regex_s = strdup(argv[optind]);
      haveregex = 1;
      ++optind;
    } /* if */

  if (haveregex) 
    {
#ifdef HAVE_LIBPCRE
      if (perl)
	{
	  pcre_pattern = pcre_compile (regex_s, 
				       (ignorecase ? PCRE_CASELESS : 0),
				       &error, &errptr, NULL);
	  if (pcre_pattern == NULL)
	    {
	      if (merr)
		fprintf (stderr, "%s: %s: %s\n", APPNAME, regex_s, error);
	      exit(2);
	    }
	}
      else
#endif /* HAVE_LIBPCRE */
	{
	  int flag1 = 0, flag2 = 0;
	  
	  if (ignorecase)
	    flag1 = REG_ICASE;
	  if (extended)
	    flag2 = REG_EXTENDED;
	  
	  errcode = regcomp (&posix_pattern, regex_s, 
			     (flag1 | flag2 | REG_NEWLINE ));
	  if (0 != errcode)
	    {
	      if (merr)
		{
		  regerror(errcode, &posix_pattern, errbuf, BUFSIZ);
		  fprintf (stderr, "%s: %s: %s\n", APPNAME, regex_s, errbuf);
		}
	      exit (2);
	    } /* if */
	} /* if */
    } /* if */
  else
    usage();

  if (optind == (argc - 1))
    singlefile = 1;

  while (optind < argc)
    {
      if (action == DELETE)
	{
#ifdef HAVE_LIBZ
	  if (format == ZMBOX)
	    ztmpp = gztmpfile_open();
#endif /* HAVE_LIBZ */
	  if (format == MBOX)
	    tmpp = tmpfile_open();
	}

      boxname = strdup(argv[optind]);

      scan_mailbox(argv[optind], format);
      havemailbox = 1;
      if (action == COUNT)
	{
	  if (singlefile)
	    fprintf(stdout, "%i\n", count);
	  else
	    fprintf(stdout, "%s:%i\n", argv[optind], count);
	}
      if (action == DELETE)
	{
#ifdef HAVE_LIBZ
	  if (format == ZMBOX)
	    gzclose(ztmpp);
#endif /* HAVE_LIBZ */
	  if (format == MBOX)
	    fclose(tmpp);
	  rename(tmpfilename, argv[optind]);
	}
      ++optind;
    } /* while */

  if (! havemailbox)
    usage();

  return 0;
} /* main */
