/* $Id: section_index.c,v 1.12 2003/04/22 06:30:12 jtalkington Exp $ */

#include "common.h"
#include "section_index.h"

#ifndef DISABLE_SECTION_INDEXES

/* section_index.c 
 * This is where we output all of the programs in a particular 
 * section.
 */

/* the tree of all the programs in the section */
static struct index_alpha_node *index_tree = NULL;

/* the list of each major directory in the MANPATH */
static struct pathlist_struct *manpath_list = NULL;

/* the main output function */
void
output_section_index(char *section) {
	DIR *dir_str = NULL;
	struct pathlist_struct **cur = NULL;
	char sec_first_char = 0;
	char *dirname = NULL;
	int i = 0;
	char *script_name = get_query_value(SCRIPT_NAME);

	if((Config.disabled & D_INDEXES) || section == NULL ||
			Config.manpath == NULL) {
		output_index_denied();
		return;
	}

	sec_first_char = *section;

	generate_manpath_list();

	for(cur = &manpath_list; (*cur) != NULL; cur = &(*cur)->next) {
		for(i = 0; i < MANDIR_PREFIX_COUNT; i++) {
			dirname = strdup((*cur)->name);
			dirname = str_app_s(dirname, "/");
			dirname = str_app_s(dirname, mandir_prefixes[i]);
			dirname = str_app_s(dirname, section);

			if((dir_str = opendir(dirname)) == NULL) {
				free(dirname);
				dirname = strdup((*cur)->name);
				dirname = str_app_s(dirname, "/");
				dirname = str_app_s(dirname, mandir_prefixes[i]);
				dirname = str_app_c(dirname, sec_first_char);
				dir_str = opendir(dirname);
			}
		
			free(dirname);

			if(dir_str != NULL) {
				add_dir_to_index(dir_str, section);
				closedir(dir_str);
			} else {
				/* do nothing */
			}
		}
	}

	output_header("Section Index", section);

	printf("<span class=\"mono\">\n");
	if(script_name != NULL) {
		output_index_top(&index_tree);
	}
	printf("</span>\n");
	printf("\n<hr />\n");
	output_index_alpha_node(&index_tree, script_name, section);
	output_footer();

#ifdef M2W_CLEANUP
	cleanup_index_alpha_node(&index_tree);
	cleanup_manpath_list(manpath_list);
#endif /* M2W_CLEANUP */

}

/* generate_manpath_list()
 * goes through the manpath and adds each directory to the manpath list
 */
void
generate_manpath_list(void) {
	char *cur = Config.manpath;
	struct pathlist_struct **mp = &manpath_list;
	char *end = NULL;
	char *name = NULL;

	while((end = strchr(cur, ':')) != NULL) {
		name = strndup(cur, end - cur);

		if((*mp) == NULL) {
			(*mp) = calloc(1, sizeof(struct pathlist_struct));

		} else {

			while((*mp)->next != NULL) {
				mp = &(*mp)->next;
			}

			(*mp)->next = calloc(1, sizeof(struct pathlist_struct));
			mp = &(*mp)->next;
		}

		(*mp)->name = name;
		cur = end + 1;
	}

	if((*mp) == NULL) {
		(*mp) = calloc(1, sizeof(struct pathlist_struct));
	} else {
		while((*mp)->next != NULL) {
			mp = &(*mp)->next;
		}

		(*mp)->next = calloc(1, sizeof(struct pathlist_struct));
		mp = &(*mp)->next;
	}

	(*mp)->name = strdup(cur);
}

/* add_dir_to_index()
 * get the contents of a particular directory, and add the programs to the
 * index_tree
 */
void
add_dir_to_index(DIR *dir, char *section) {
	struct dirent *dir_entry = NULL;
	char *filename = NULL;
	char *program = NULL;
	char *prog_end = NULL;
	char *secstr = NULL;

	if(dir == NULL || section == NULL) {
		return;
	}

	secstr = strdup(".");
	secstr = str_app_s(secstr, section);

	while((dir_entry = readdir(dir)) != NULL) {
		filename = strdup(dir_entry->d_name);

		/* skip dotfiles */
		if(*filename == '.') {
			free(filename);
			filename = NULL;
		} else if((prog_end = strstr(filename, secstr)) != NULL) {
				program = strndup(filename, prog_end - filename);
				add_to_index(&index_tree, *program, program);
				free(program);
				
				program = NULL;
				prog_end = NULL;
				
				free(filename);
				filename = NULL;
		} else {
			free(filename);
			filename = NULL;
		}
	}

	free(secstr);
	secstr = NULL;
}

/* add_to_index()
 * recursively searches for the right place to stick an entry in the index_tree
 * and sticks it to it.
 */

void
add_to_index(struct index_alpha_node **node, char index, char *name) {

	if((*node) == NULL) {
		(*node) = calloc(1, sizeof(struct index_alpha_node));

		(*node)->name = index;
		add_index_prog_node(&(*node)->data, name);
	} else {
		if((*node)->name == index) {
			add_index_prog_node(&(*node)->data, name);
		} else if((*node)->name < index) {
			add_to_index(&(*node)->right, index, name);
		} else {
			add_to_index(&(*node)->left, index, name);
		}
	}
}

/* add_index_prog_node()
 * inserts a program name into a section node, if it doesn't already exist
 * does nothing if it does exist
 */

void
add_index_prog_node(struct index_prog_node **node, char *name) {
	int result = 0;

	if((*node) == NULL) {
		(*node) = calloc(1, sizeof(struct index_prog_node));

		(*node)->left = NULL;
		(*node)->right = NULL;
		(*node)->name = strdup(name);
	} else {
		result = strcmp((*node)->name, name);

		if(result == 0) {
			/* do nothing with this, it already exists */
			return;
		} else if(result < 0) {
			add_index_prog_node(&(*node)->right, name);
		} else {
			add_index_prog_node(&(*node)->left, name);
		}
	}
}

/* output_index_alpha_node
 * recursively prints out the program nodes
 */
void
output_index_alpha_node(struct index_alpha_node **node, char *script_name, char *section) {

	if((*node) == NULL) {
		return;
	}

	if((*node)->left != NULL) {
		output_index_alpha_node(&(*node)->left, script_name, section);
	}

	printf("<br /><h2><a name=\"%p\">%c</a></h2>\n", node, (*node)->name);

	output_index_prog_node(&(*node)->data, script_name, section);

	if((*node)->right != NULL) {
		output_index_alpha_node(&(*node)->right, script_name, section);
	}
}

/* output_index_prog_node
 * prints out the data elements of a section node
 */
void
output_index_prog_node(struct index_prog_node **node, char *script_name, char *section) {
	char *tmp = NULL;

	if((*node) == NULL) {
		return;
	}

	if((*node)->left != NULL) {
		output_index_prog_node(&(*node)->left, script_name, section);
	}

	/* create a link and print it out */
	if((*node)->name != NULL) {
		if(script_name != NULL) {
			tmp = gen_index_url(script_name, (*node)->name, section);
		} else {
			tmp = strdup((*node)->name);
		}
		printf("%s<br />\n", tmp);
		fflush(stdout);
		free(tmp);
		tmp = NULL;
	}

	if((*node)->right != NULL) {
		output_index_prog_node(&(*node)->right, script_name, section);
	}
}

/* gen_index_url
 * generates a url to print
 */
char *
gen_index_url(char *script_name, char *program, char *section) {
        char *output = NULL;
        
        if(script_name == NULL || program == NULL || section == NULL) {
                return(NULL);
        }
        
        output = calloc(1, strlen(INDEX_HREF) + (2 * (strlen(program) + 1)) + (strlen(section) + 1) + strlen(script_name) + 1);
        sprintf(output, INDEX_HREF, script_name, program, section, program);
        return(output);
}

/* output_index_top()
 * outputs the top of the out the index page with refs to the lower portion
 * of the page that holds that letter
 */
void
output_index_top(struct index_alpha_node **node) {
	char *tmp = NULL;
	
	if((*node) == NULL) {
		return;
	}

	if((*node)->left != NULL) {
		output_index_top(&(*node)->left);
	}

	tmp = gen_top_url(node);
	printf("%s", tmp);
	free(tmp);
	tmp = NULL;

	if((*node)->right != NULL) {
		output_index_top(&(*node)->right);
	}

}

/* gen_top_url
 * creates a url for jumping down in the page
 */
char *
gen_top_url(struct index_alpha_node **node) {
        char *output = NULL;
        
        output = calloc(1, strlen(INDEX_TOP_HREF) + 2 * sizeof(node));
        sprintf(output, INDEX_TOP_HREF, node, (*node)->name);
        return(output);
}

#ifdef M2W_CLEANUP
/* cleanup_index_alpha_node
 * frees the stuff in the index_tree
 */
void
cleanup_index_alpha_node(struct index_alpha_node **node) {
	struct index_alpha_node *right = NULL;

	if((*node) == NULL) {
		return;
	} else {
		right = (*node)->right;
	}

	if((*node)->left != NULL) {
		cleanup_index_alpha_node(&(*node)->left);
	}

	cleanup_index_prog_node(&(*node)->data);

	free((*node));

	if(right != NULL) {
		cleanup_index_alpha_node(&right);
	}
}

/* cleanup_index_prog_node
 * free()s the data
 */
void
cleanup_index_prog_node(struct index_prog_node **node) {
	struct index_prog_node *right = NULL;

	if((*node) == NULL) {
		return;
	} else {
		right = (*node)->right;
	}

	if((*node)->left != NULL) {
		cleanup_index_prog_node(&(*node)->left);
	}

	free((*node)->name);
	free((*node));

	if(right != NULL) {
		cleanup_index_prog_node(&right);
	}
}

/* cleanup_manpath_list
 * free()s the manpath_list
 */
void
cleanup_manpath_list(struct pathlist_struct *mp) {
	struct pathlist_struct *next = NULL;

	if(mp == NULL) {
		return;
	}

	next = mp->next;

	free(mp->name);
	free(mp);

	if(next != NULL) {
		cleanup_manpath_list(next);
	}
}

#endif /* M2W_CLEANUP */

#else /* DISABLE_SECTION_INDEXES */

void
output_section_index(char *section) {
	output_index_denied();
}

#endif /* DISABLE_SECTION_INDEXES */

void
output_index_denied() {
	output_header("Section Index", NULL);
	printf("<h2>Section indexes are disabled.</h2>\n");
	output_footer();
}
