/* $Id: query_string.c,v 1.14 2003/04/24 00:33:34 jtalkington Exp $ */

#include "common.h"
#include "query_string.h"

/* struct query_struct query_tree
 * this is the root for all of the options that are passed,
 * such as which section, if output should be raw, and
 * what man page we are looking for
 */
static struct query_struct *query_tree = NULL;

/* init_query_tree()
 * determines if we are in CGI mode and parses the QUERY_STRING, else
 * lets parse_options parse the command line and populate query_tree.
 */
void
init_query_tree(int argc, char *argv[]) {
	char *query_string = NULL;
	char *script_name = NULL;
	char *user_agent = NULL;

#ifdef ENABLE_LYNX
	char *server_software = getenv("SERVER_SOFTWARE");

	if(server_software != NULL) {
		if(strncmp(server_software, "Lynx", 4) == 0) {
			init_lynx(argv[0]);
		}
	}
#endif /* ENABLE_LYNX */

	query_string = getenv("QUERY_STRING");
	script_name = getenv("SCRIPT_NAME");
	user_agent = getenv("HTTP_USER_AGENT");

	if(user_agent != NULL) {
		add_query_element(&query_tree, USER_AGENT, user_agent);
	}

#ifdef ENABLE_LYNX
	if(script_name == NULL) {
		script_name = get_query_value(SCRIPT_NAME);
	}
#endif /* ENABLE_LYNX */


	/* 
	 * figure out if we are in command or CGI mode
	 * this should probably be more complex than looking for a
	 * SCRIPT_NAME environment variable, but it's not
	 */

	
	if(script_name != NULL) {
		/* CGI mode */
		char *name_start = NULL;
		char *name_end = NULL;
		char *value_start = NULL;
		char *value_end = NULL;
		char *hunk_end = NULL;
		char *name = NULL;
		char *value = NULL;
		char *script_name_dup = strdup(script_name);
		
		Config.mode = CGI_MODE;

		add_query_element(&query_tree, SCRIPT_NAME, script_name_dup);
		free(script_name_dup);

		if(query_string == NULL) {
			return;
		}

		for(name_start = query_string; *name_start != '\0'; /* none */) {
			if((hunk_end = strchr(name_start, '&'))!= NULL) {
				if((name_end = strchr(name_start, '=')) != NULL) {
					value_start = name_end + 1;
					if(value_start != hunk_end) {
						value_end = hunk_end;
						name = strndup(name_start, name_end - name_start);
						value = strndup(value_start, value_end - value_start);
						
						name = decode_escape(name);
						value = decode_escape(value);

						if(name != NULL) {
							add_query_element(&query_tree, name, value);
						
							free(name);
							name = NULL;
						}
						
						if(value != NULL) {
							free(value);
							value = NULL;
						}
						
						name_end = NULL;
						value_start = NULL;
						value_end = NULL;

						name_start = hunk_end + 1;

					} else {
						/* this name has no value but has an = */
						name = strndup(name_start, name_end - name_start);

						name = decode_escape(name);
						add_query_element(&query_tree, name, NULL);
						free(name);

						name = NULL;
						value_start = NULL;
						name_end = NULL;
						name_start = hunk_end + 1;
					}
				} else {
					/* this name has no = or value */
					name = strndup(name_start, name_end - name_start);
					name = decode_escape(name);
					
					add_query_element(&query_tree, name, NULL);
					
					free(name);
					
					name = NULL;
					name_start = hunk_end + 1;
				}

			} else if((name_end = strchr(name_start, '=')) != NULL) {
				/* this is the last hunk */
				value_start = name_end + 1;
				name = strndup(name_start, name_end - name_start);
				name = decode_escape(name);
				
				if(*value_start != '\0') {
					value = strdup(value_start);
					value = decode_escape(value);
				}
				
				add_query_element(&query_tree, name, value);
				free(name);
				
				if(value != NULL) {
					free(value);
				}
				
				/* we are done with the parameters */
				break;
			} else {
				/* this hunk is last, and has no = in it */
				name = strdup(name_start);
				name = decode_escape(name);
				add_query_element(&query_tree, name, NULL);
				free(name);
				/* we are done with the parameters */
				break;
			}
		}
	} else {
		/* command mode */

		Config.mode = COMMAND_MODE;
		parse_options(argc, argv);

	}
}

/* add_query_option()
 * a wrapper for parse_options to add to the query_tree
 */
void
add_query_option(char *name, char *value) {
	add_query_element(&query_tree, name, value);
}

/* add_query_element()
 * adds an element (such as section, or manpage name) to query_tree
 */
void
add_query_element(struct query_struct **node, char *name, char *value) {
	int result = 0;
	
	if(name == NULL) {
		return;
	}

	if((*node) == NULL) {
		(*node) = calloc(1, sizeof(struct query_struct));
		(*node)->name = strdup(name);
		if(value != NULL) {
			(*node)->value = strdup(value);
		}

	} else {
		result = strcmp(name, (*node)->name);

		if(result == 0) {
			if((*node)->value != NULL) {
				free((*node)->value);
			}

			/* replace the value */
			(*node)->value = strdup(value);

		} else if(result < 0) {
			add_query_element(&(*node)->left, name, value);
		} else {
			add_query_element(&(*node)->right, name, value);
		}
	}
}

/* get_query_value()
 * wrapper to get the value of an entry in the query_tree.
 */
char *
get_query_value(char *name) {
	struct query_struct *result = find_query_element(&query_tree, name);

	if(result != NULL) {
		return(result->value);
	} else {
		return(NULL);
	}
}

/* find_query_element()
 * internals for get_query_value, finds a node in the query_tree that matches
 * the name given, and returns the entire struct.
 */

struct query_struct *
find_query_element(struct query_struct **node, char *name) {
	int result = 0;

	if((*node) == NULL || name == NULL) {
		return(NULL);
	}

	result = strcmp(name, (*node)->name);

	if(result == 0) {
		return((*node));
	} else if(result < 0) {
		return(find_query_element(&(*node)->left, name));
	} else {
		return(find_query_element(&(*node)->right, name));
	}
}

/* decode_escape()
 * decodes HTTP escapes in the QUERY_STRING
 */
char *decode_escape(char *line) {
	char *output = NULL;
	char *cur = NULL;
	char *tmp = NULL;
	unsigned int converted_char = 0;

	if(line == NULL) {
		return(NULL);
	}

	for(cur = line; *cur != '\0'; cur++) {
		if(*cur == '%') {
			tmp = strndup(cur + 1, 2);
			converted_char = strtol(tmp, (char **)NULL, 16);

			free(tmp);
			tmp = NULL;

			if(needs_escaping(converted_char)) {
				output = str_app_c(output, '\\');
			}

			output = str_app_c(output, converted_char);

			converted_char = 0;

			cur = cur + 2;
		} else if(*cur == '+') {
			/* skip blanks, there shouldn't be any in program names
			 * or section names...
			 */
		} else {
			if(needs_escaping(*cur)) {
				output = str_app_c(output, '\\');
			}
			
			output = str_app_c(output, *cur);
		}
	}

	free(line);
	return(output);
}

/* needs_escaping()
 * returns 1 if a character is unsafe for use in shell commands
 */

int
needs_escaping(char c) {
	if(c == ';' ||
	   c == '<' ||
	   c == '>' ||
	   c == '*' ||
	   c == '|' ||
	   c == '`' ||
	   c == '&' ||
	   c == '$' ||
	   c == '!' ||
	   c == '#' ||
	   c == '(' ||
	   c == ')' ||
	   c == '[' ||
	   c == ']' ||
	   c == '{' ||
	   c == '}' ||
	   c == ':' ||
	   c == '?' ||
	   c == '\\' ||
	   c == '^' ||
	   c == '\'' ||
	   c == '\"') {
		return(1);
	} else {
		return(0);
	}
}

#ifdef M2W_CLEANUP
/* cleanup_query_tree()
 * wrapper to free the query_tree
 */
void
cleanup_query_tree() {
	cleanup_query_element(&query_tree);
}

/* cleanup_query_element()
 * cleans up all of the nodes in the query_tree
 */
void
cleanup_query_element(struct query_struct **node) {
	struct query_struct *right = NULL;

	if((*node) == NULL) {
		return;
	} else {
		right = (*node)->right;
	}

	if((*node)->left != NULL) {
		cleanup_query_element(&(*node)->left);
		(*node)->left = NULL;
	}

	if((*node)->name != NULL) {
		free((*node)->name);
		(*node)->name = NULL;
	}

	if((*node)->value != NULL) {
		free((*node)->value);
		(*node)->value = NULL;
	}

	free((*node));

	if(right != NULL) {
		cleanup_query_element(&right);
	}
}
	

#endif /* M2W_CLEANUP */


