/* The contents of this file are subject to the Mozilla Public License
 * Version 1.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * License for the specific language governing rights and limitations
 * under the License.
 *
 * The Original Code is Mobile Application Link.
 *
 * The Initial Developer of the Original Code is AvantGo, Inc.
 * Portions created by AvantGo, Inc. are Copyright (C) 1997-1999
 * AvantGo, Inc. All Rights Reserved.
 *
 * Contributor(s):
 */

/* Owner:  miket */

#include "MALSetupDLL.h"
#include <Notify.h>

typedef enum
{
    codeINSTALL_INIT_CONTINUE  = 0,
    codeINSTALL_INIT_CANCEL
}
codeINSTALL_INIT;

typedef enum
{
    codeINSTALL_EXIT_DONE = 0,
    codeINSTALL_EXIT_UNINSTALL
}
codeINSTALL_EXIT;

typedef enum
{
    codeUNINSTALL_INIT_CONTINUE = 0,
    codeUNINSTALL_INIT_CANCEL
}
codeUNINSTALL_INIT;

typedef enum
{
    codeUNINSTALL_EXIT_DONE = 0
}
codeUNINSTALL_EXIT;

BOOL APIENTRY DllMain(HANDLE hModule,
                      DWORD ul_reason_for_call,
                      LPVOID lpReserved)
{
    return TRUE;
}

const LPTSTR deleteExt = TEXT(".delete");
const LPTSTR windowsPath = TEXT("\\windows\\");
const LPTSTR wildcardCharacter = TEXT("*");
const LPTSTR regMALKey = TEXT("Software\\Mobile Application Link");
const LPTSTR regObjectKey = TEXT("Windows CE Services\\Synchronization\\Objects\\Mobile Application Link");

static void checkForRunningClient(void)
{
    HWND hwndMobileLink = FindWindow(TEXT("Mobile Link Main Window Class"),
        TEXT("Mobile Link"));
    if (IsWindow(hwndMobileLink))
        PostMessage(hwndMobileLink, WM_CLOSE, 0, 0);
}

/*  Search Windows directory for files matching *.delete, and delete them.
*/
static BOOL deleteAllDeadFiles(void)
{
    BOOL deleteFailed = FALSE;
    WIN32_FIND_DATA findData;
    TCHAR buf[MAX_PATH];
    HANDLE findHandle;

    /* Build template for matching files. */
    wcscpy(buf, windowsPath);
    wcscat(buf, wildcardCharacter);
    wcscat(buf, deleteExt);

    /* Start search for files. */
    findHandle = FindFirstFile(buf, &findData);

    /* Loop while we've found stuff. */
    while (INVALID_HANDLE_VALUE != findHandle) {

        /* Build full pathname of file to delete. */
        TCHAR deletebuf[MAX_PATH];
        wcscpy(deletebuf, windowsPath);
        wcscat(deletebuf, findData.cFileName);

        /* Delete.  If we got an error, remember that so that
        we can schedule the cleanup app to try again. */
        if (0 == DeleteFile(deletebuf))
            deleteFailed = TRUE;

        /* Get next file.  Signal done if done. */
        if (0 == FindNextFile(findHandle, &findData)) {
            
            if (ERROR_NO_MORE_FILES == GetLastError())
                findHandle = INVALID_HANDLE_VALUE;

        }

    }

    return deleteFailed;
}

/*  Given a filename and extension, deletes that file or, at a minimum,
    renames it to *.delete so that a new file of the same name can be
    installed later.
*/
static void deleteFile(LPTSTR name, LPTSTR extension)
{
    TCHAR buf[MAX_PATH];

    /* Build name of file. Assumes we're in the right directory or
    that the caller has passed in the right full path. */
    wcscpy(buf, name);
    wcscat(buf, TEXT("."));
    wcscat(buf, extension);

    /* If we couldn't delete it for any reason, rename it to *.delete. */
    if (0 == DeleteFile(buf)) {

        TCHAR buf2[MAX_PATH];

        wcscpy(buf2, name);
        wcscat(buf2, deleteExt);
        MoveFile(buf, buf2);

    }

}

static void doUninstallInit(void)
{

    deleteFile(TEXT("\\windows\\malclmgr"), TEXT("dll"));
    deleteFile(TEXT("\\windows\\malcommon"), TEXT("dll"));
    deleteFile(TEXT("\\windows\\malconfig"), TEXT("dat"));
    RegDeleteKey(HKEY_LOCAL_MACHINE, regMALKey); 
    RegDeleteKey(HKEY_LOCAL_MACHINE, regObjectKey);

    deleteAllDeadFiles();

}

static void doUninstallExit(void)
{
    /* Just in case cleanup app is still registered to be run,
    unregister it so we're not called again.  Cost:  If user did something
    perverse (installed MAL and deleted right away again without ever
    taking the device out of the cradle, which isn't what a real user
    would do), then the malcleanup.exe app never gets a chance to run, gets
    deleted, and we're left with undeleteable things lying around.  But
    that's the breaks. At least MAL isn't all that big. */
    CeRunAppAtEvent(cleanupAppName, NOTIFICATION_EVENT_NONE);
}

static void doInstallInit(void)
{
    doUninstallInit();
    doUninstallExit();
}

static void doInstallExit(void)
{
    if (deleteAllDeadFiles()) {

        /* Installer is done. Schedule app that will delete busy files when
        they're not busy. */
        HANDLE f = CreateFile(cleanupAppName,
            0,
            0,
            NULL,
            OPEN_EXISTING,
            FILE_ATTRIBUTE_NORMAL,
            NULL);

        if (INVALID_HANDLE_VALUE != f) {

            CloseHandle(f);
            CeRunAppAtEvent(cleanupAppName,
                NOTIFICATION_EVENT_RS232_DETECTED);

        }

    } else {

        DeleteFile(cleanupAppName);

    }
}

MALSETUPDLL_API int Install_Init(HWND hwndParent,
                                 BOOL fFirstCall,
                                 BOOL fPreviouslyInstalled,
                                 LPCTSTR pszInstallDir)
{
    checkForRunningClient();
    doInstallInit();
    return codeINSTALL_INIT_CONTINUE;
}

MALSETUPDLL_API int Install_Exit(HWND hwndParent,
                                 LPCTSTR pszInstallDir,
                                 WORD cFailedDirs,
                                 WORD cFailedFiles,
                                 WORD cFailedRegKeys,
                                 WORD cFailedRegVals,
                                 WORD cFailedShortcuts)
{
    doInstallExit();
    return codeINSTALL_EXIT_DONE;
}

MALSETUPDLL_API int Uninstall_Init(HWND hwndParent,
                                   LPCTSTR pszInstallDir)
{
    doUninstallInit();
    return codeUNINSTALL_INIT_CONTINUE;
}

MALSETUPDLL_API int Uninstall_Exit(HWND hwndParent)
{
    doUninstallExit();
    return codeUNINSTALL_EXIT_DONE;
}

MALSETUPDLL_API void Install_DoCleanup(void)
{
    deleteAllDeadFiles();
}