/* The contents of this file are subject to the Mozilla Public License
 * Version 1.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * License for the specific language governing rights and limitations
 * under the License.
 *
 * The Original Code is Mobile Application Link.
 *
 * The Initial Developer of the Original Code is AvantGo, Inc.
 * Portions created by AvantGo, Inc. are Copyright (C) 1997-1999
 * AvantGo, Inc. All Rights Reserved.
 *
 * Contributor(s):
 */

/* Owner:  miket */

#include <stdio.h>
#include <crtdbg.h>
#include <windowsx.h>

#include <AGUtil.h>
#include <AGClientProcessor.h>
#include <AGCommandProcessor.h>
#include <AGProtocol.h>
#include <AGBufferReader.h>
#include <AGSyncCommon.h>
#include <AGSyncCECommon.h>
#include <AGSyncCEDesktopCommands.h>
#include <AGSyncCEResourceWin32.h>
#include <AGSyncCEStatusDialog.h>
#include <AGSyncCEStreamProtocol.h>
#include <commctrl.h>
#include <rapi.h>
#include <AGDesktopInfoWinCE.h>
#include <AGTimeoutMessageBox.h>
#include <AGMobileLinkSettings.h>

/* WARNING (miket):  The resources for this message box are all screwed up.
    I couldn't get multiple scripts to be included in a single project, so
    the resource.h defines are duplicated in several places, and the
    resources themselves are duplicated.  I recommend grepping for things
    like IDC_LINK to find everything. */
#include <AGLinkMessageBox.h>

#include <stdio.h>
#include <AGShlapi.h>

#ifdef _DEBUG
#define debug_print(x) OutputDebugString(x)
#else
#define debug_print(x)
#endif

typedef sword (*netInitFunc)(AGNetCtx *ctx);
typedef sword (*netCloseFunc)(AGNetCtx *ctx);
typedef void  (*netToggleFunc)(AGNetCtx *ctx, AGBool useAltNet);
typedef int32 (*netCtxSizeFunc)(void);

typedef struct SyncInfo {
    
    AGCommandProcessor *cp;
    AGRecord * record;
    AGReader * rapiReader;
    AGWriter * rapiWriter;
    AGSyncCEStatusDialog * statusDialog;
    AGBool quit;

    /* Secure Network Library Stuff */
    AGBool          hasseclib;
    netInitFunc     secnetinit;
    netCloseFunc    secnetclose;
    netToggleFunc   sectoggle;
    netCtxSizeFunc  secctxsize;

} SyncInfo;

static int32 performCommand(void *out, int32 *errCode,
                            AGReader *commandToProcess);
static int32 initAndOpenDatabase(void *out, AGDBConfig *db, 
                                 int32 *errCode);
static int32 getNextModifiedRecord(void * out, AGRecord ** record, 
                                   int32 *errCode);
static int32 getNextRecord(void * out, AGRecord ** record,
                           int32 *errCode);
static int32 nextExpansionCommand(void *out, int32 *newCommand,
                                  int32 *commandLength, void **commandBytes);

/* ----------------------------------------------------------------------------
    void syncInfoFree(SyncInfo * pInfo)

    Free the given SyncInfo structure and everything that it knows about.

*/
void syncInfoFree(SyncInfo * pInfo)
{
    if (NULL != pInfo) {

            if (NULL != pInfo->rapiReader)
                AGReaderFree(pInfo->rapiReader);

            if (NULL != pInfo->rapiWriter)
                AGWriterFree(pInfo->rapiWriter);

        free(pInfo);
    }
}

/* ----------------------------------------------------------------------------
    SyncInfo * syncInfoNew()

    Allocate SyncInfo structure and whatever else is guaranteed to be needed,
    as well.

*/
SyncInfo * syncInfoNew()
{
    SyncInfo * pInfo;

    /* Allocate the SyncInfo structure. */
    pInfo = (SyncInfo *)malloc(sizeof(SyncInfo));
    if (NULL != pInfo) {

        bzero(pInfo, sizeof(SyncInfo));

        return pInfo;
    }

    syncInfoFree(pInfo);
    return NULL;
}

/* ----------------------------------------------------------------------------
    static AGBool writeUserConfig(AGUserConfig * user)
*/
static AGBool writeUserConfig(AGUserConfig * user)
{
    char userConfigFilename[MAX_PATH];
    char syncUserConfigFilename[MAX_PATH];

    AGCEGetCurrentDeviceInfo(NULL, userConfigFilename, syncUserConfigFilename);

    /* Write out config files to disk. */
    AGWriteUserConfigToDisk(userConfigFilename, user);
    AGWriteUserConfigToDisk(syncUserConfigFilename, user);

    return FALSE;
}

/* ----------------------------------------------------------------------------
    static AGLocationConfig * readLocationConfig(void)

    Part of initialization:  Read locationConfig file from disk.

*/
static AGLocationConfig * readLocationConfig(void)
{
    TCHAR * stringPointer;
    AGLocationConfig * result = NULL;

    stringPointer = (TCHAR*)
        AGSyncCommonGetStringConstant(agLocationConfigPath, FALSE);
    result = AGReadLocationConfigFromDisk(stringPointer);
    free(stringPointer);

    return result;
}

/* ----------------------------------------------------------------------------
    static AGPlatformCalls * setupPlatformCalls(void * out)
*/
static AGPlatformCalls * setupPlatformCalls(void * out)
{
    AGPlatformCalls * result;
    result = (AGPlatformCalls *)malloc(sizeof(AGPlatformCalls));
    bzero(result, sizeof(AGPlatformCalls));
    if (NULL != result) {
        result->out = out;
        result->nextModifiedRecordFunc = getNextModifiedRecord;
        result->nextRecordFunc = getNextRecord;
        result->openDatabaseFunc = initAndOpenDatabase;
        result->nextExpansionCommandFunc = nextExpansionCommand;

        result->performCommandOut = out;
        result->performCommandFunc = performCommand;
    }
    return result;
}

/* ----------------------------------------------------------------------------
    static AGSyncCEStatusDialog * startStatusDialog(HINSTANCE h,
                                                    AGBool * quit,
                                                    AGArray * graphic,
                                                    AGBool bitmapFormat)
*/
static AGSyncCEStatusDialog * startStatusDialog(HINSTANCE h,
                                                AGBool * quit,
                                                AGArray * graphic,
                                                AGBool bitmapFormat)
{
    char strbuf[256];
    AGSyncCEStatusDialog * result;

    result = AGSyncCEStatusDialogNew(h,
        quit,
        graphic,
        250,
        bitmapFormat);
    *quit = FALSE;

    if (NULL != result) {
        LoadString(h, IDS_DEVICE_CONNECTING, strbuf, 256);
        AGSyncCEStatusDialogSetTask(result, strbuf);
    }
    return result;
}

/* ----------------------------------------------------------------------------
    static void setStatusDialogToConnecting(AGSyncCEStatusDialog * dialog,
    HINSTANCE h)
*/
static void setStatusDialogToConnecting(AGSyncCEStatusDialog * dialog,
                                        HINSTANCE h)
{
    char strbuf[256];
    LoadString(h, IDS_CONNECTING, strbuf, 256);
    AGSyncCEStatusDialogSetTask(dialog, strbuf);
}

/* ----------------------------------------------------------------------------
    static IRAPIStream * startRAPIConnection(DWORD * dwOut, BYTE ** pOut)
*/
static IRAPIStream * startRAPIConnection(DWORD * dwOut, BYTE ** pOut)
{
    HRESULT hr;
    WCHAR * wideStringPointer[2];
    IRAPIStream * rapiStream = NULL;

    wideStringPointer[0] =
        (WCHAR*)AGSyncCommonGetStringConstant(agRAPIDLLName, TRUE);
    wideStringPointer[1] =
        (WCHAR*)AGSyncCommonGetStringConstant(agRAPIFunctionName, TRUE);
    hr = CeRapiInvoke(wideStringPointer[0], wideStringPointer[1], 0, NULL,
        dwOut, pOut, &rapiStream, 0);
    free(wideStringPointer[0]);
    free(wideStringPointer[1]);

    return rapiStream;
}

/* ----------------------------------------------------------------------------
    static AGDeviceInfo * allocateAndReadDeviceInfo(AGReader * r, AGWriter * w)
    
    Allocate a deviceInfo record; find out whether device has a deviceInfo
    record for us and, if it does, read it into what we just created.
    Otherwise just zero it out.
*/
static AGDeviceInfo * allocateAndReadDeviceInfo(AGReader * r, AGWriter * w)
{
    int32 error;
    AGDeviceInfo * result = NULL;
    AGWriteSTREAM_GET_DEVICEINFO(r, w, &error, &result);
    return result;
}

/* ----------------------------------------------------------------------------
    static AGUserConfig * syncUserConfigs(AGReader * r, AGWriter * w)

    Synchronizes UserConfig.
    
    Receives device's record, syncs with ours, and then sends 
    the properly synced record.

*/
static AGUserConfig * syncUserConfigs(AGReader * r, AGWriter * w)
{
    int32 error;
    AGUserConfig * agreedUserConfig = NULL;
    AGUserConfig * deviceUserConfig = NULL;
    AGUserConfig * desktopUserConfig = NULL;
    AGUserConfig * newAgreed = NULL;    
    char userConfigFilename[MAX_PATH];
    char syncUserConfigFilename[MAX_PATH];

    AGCEGetCurrentDeviceInfo(NULL, userConfigFilename, syncUserConfigFilename);

    /* Get desktop's userConfig. */
    desktopUserConfig = AGReadUserConfigFromDisk(userConfigFilename);

    /* Get last known agreed userConfig. */
    agreedUserConfig = AGReadUserConfigFromDisk(syncUserConfigFilename);

    /* Get device's record. */
    AGWriteSTREAM_GET_DEVICE_USERCONFIG(r, w, &error, &deviceUserConfig);

    /* Be sure an agreed userConfig exists. */
    if (NULL == agreedUserConfig)
        agreedUserConfig = AGUserConfigNew();

    if (NULL != agreedUserConfig) {

        /* Sync the three records. */
        newAgreed = AGUserConfigSynchronize(
            agreedUserConfig,
            deviceUserConfig,
            desktopUserConfig);

        /* Write synchronized serverConfig down to the device. */
        AGWriteSTREAM_PUT_DEVICE_USERCONFIG(r, w, &error, newAgreed);

        /* Clean up agreed userConfig. */
        AGUserConfigFree(agreedUserConfig);

    }

    /* Clean up desktop userConfig. */
    if (NULL != desktopUserConfig)
        AGUserConfigFree(desktopUserConfig);

    /* Clean up device userConfig. */
    if (NULL != deviceUserConfig)
        AGUserConfigFree(deviceUserConfig);

    return newAgreed;
}

/* ----------------------------------------------------------------------------
    static void doWindowsMessagePump()
    
*/
static void doWindowsMessagePump()
{
    MSG msg;
    while (PeekMessage(&msg, 0, 0, 0, PM_REMOVE | PM_NOYIELD)) {
        TranslateMessage(&msg);
        DispatchMessage(&msg);
    }
}

/* ----------------------------------------------------------------------------
    static int32 doStartServer(char * servername,
                               int32 port,
                               AGUserConfig * user,
                               AGUserConfig * agreedUser,
                               AGServerConfig ** server,
                               AGServerConfig ** agreed,
                               AGReader * r,
                               AGWriter * w,
                               int32 *errCode)

    Prepares ourselves and device for a set of MAL commands pertaining to a
    particular server.

*/
static int32 doStartServer(char * servername,
                           int32 port,
                           AGUserConfig * user,
                           AGServerConfig ** server,
                           AGReader * r,
                           AGWriter * w,
                           int32 *errCode)
{
    /* Given a name and port for a server, retrieve our serverConfig for
    that server.  If we don't know about it, something is wrong. */
    *server = AGUserConfigGetServerByNameAndPort(user, servername,
        port, AGUSERCONFIG_FIND);

    if (NULL != *server) {

        return AGWriteSTREAM_START_SERVER(r, w, errCode,
            (*server)->serverName, (*server)->serverPort);

    } else {
        return -1;
    }
}

/* ----------------------------------------------------------------------------
    static int32 doEndServer(AGReader * r,
                             AGWriter * w,
                             int32 *errCode)

    Cleanup after sequence of MAL commands.

*/
static int32 doEndServer(AGReader * r,
                         AGWriter * w,
                         int32 *errCode)
{
    return AGWriteSTREAM_END_SERVER(r, w, errCode);
}

static int32 performCommand(void *out, int32 *errCode,
                            AGReader *r)
{
    int32 result;
    SyncInfo * pInfo = (SyncInfo *)out;
    AGPerformCommandFunc defaultPerformCommand = AGCommandProcessorGetPerformFunc(pInfo->cp);
    AGBufferReader rc;

    AGBufferReaderInit(&rc, AGBufferReaderPeek((AGBufferReader *)r));

    result = (*defaultPerformCommand)(pInfo->cp, errCode, (AGReader *)&rc);

    result = AGWriteSTREAM_PERFORM_COMMAND(pInfo->rapiReader,
        pInfo->rapiWriter, errCode, r);

    AGBufferReaderFinalize(&rc);

    return result;
}

static int32 performTask(void *out, int32 *returnErrorCode,
                         char *currentTask, AGBool bufferable)
{
    SyncInfo * pInfo = (SyncInfo *)out;

    AGSyncCEStatusDialogSetTask(pInfo->statusDialog, currentTask);

    return AGCLIENT_CONTINUE;
}

static int32 performItem(void *out, int32 *returnErrorCode,
                         int32 currentItemNumber,
                         int32 totalItemCount,
                         char *currentItem)
{
    SyncInfo * pInfo = (SyncInfo *)out;

    AGSyncCEStatusDialogSetItem(pInfo->statusDialog, currentItem);
    AGSyncCEStatusDialogSetBar(pInfo->statusDialog, currentItemNumber,
        totalItemCount);

    return AGCLIENT_CONTINUE;
}

static int32 performGoodbye(void *out,
                            int32 *returnErrorCode,
                            AGSyncStatus syncStatus,
                            int32 errorCode,
                            char *errorMessage)
{
    SyncInfo * pInfo = (SyncInfo *)out;

    if (NULL != errorMessage) {
        if (0 != errorCode) {

            int result;
            char url[MAX_PATH];

            url[0] = '\0';

            result = MessageBoxWithLink(pInfo->statusDialog->hDlg,
                errorMessage,
                "Mobile Link",
                TRUE,
                url,
                kTimeoutValue);

            if (IDC_LINK == result) {

                if (NULL != url
                    && PathIsURL(url)
                    && !strncmp("http://", url, 7)) {

                    ShellExecute(pInfo->statusDialog->hDlg,
                        "open",
                        url,
                        NULL,
                        NULL,
                        1);

                }

            }
        }
    }
    return AGCLIENT_CONTINUE;
}

static int32 initAndOpenDatabase(void *out, AGDBConfig *db, 
                                 int32 *errCode)
{
    SyncInfo * pInfo = (SyncInfo *)out;

    pInfo->record = NULL;

    return AGWriteSTREAM_OPEN_DATABASE(pInfo->rapiReader, pInfo->rapiWriter,
        errCode, db);
}

static int32 nextExpansionCommand(void *out, int32 *newCommand,
                                  int32 *commandLength, void **commandBytes)
{
    SyncInfo * pInfo = (SyncInfo *)out;

    pInfo->record = NULL;

    return AGWriteSTREAM_GET_NEXT_EXPANSION_COMMAND(pInfo->rapiReader, pInfo->rapiWriter,
        newCommand, commandLength, commandBytes);
}

static int32 getRecordBase(SyncInfo * pInfo, AGBool modonly,
                           AGRecord ** record, int32 * errCode)
{
    int32 result;

    if (NULL != pInfo->record) {
        AGRecordFree(pInfo->record);
        pInfo->record = NULL;
    }

    /* Send notification to device that this is a getNextRecord or 
    getNextModifiedRecord sequence. */
    if (modonly)
        result = AGWriteSTREAM_GET_NEXT_MODIFIED_RECORD(pInfo->rapiReader, 
            pInfo->rapiWriter, errCode, &pInfo->record);
    else
        result = AGWriteSTREAM_GET_NEXT_RECORD(pInfo->rapiReader,
            pInfo->rapiWriter, errCode, &pInfo->record);

    *record = pInfo->record;

    return result;

}

static int32 getNextModifiedRecord(void * out, AGRecord ** record, 
                                   int32 *errCode)
{
    return getRecordBase((SyncInfo *)out, TRUE, record, errCode);
}

static int32 getNextRecord(void * out, AGRecord ** record,
                           int32 *errCode)
{
    return getRecordBase((SyncInfo *)out, FALSE, record, errCode);
}

typedef struct tagThreadStruct { 
    AGBool * quit; 
    AGClientProcessor *processor; 
} threadStruct; 

static DWORD WINAPI quickClientProcessorLoop(void *out) 
{ 
    int32 cpResult; 
    AGBool cancelled = FALSE; 
    threadStruct * pInfo = (threadStruct *)out; 
    AGClientProcessor *processor = pInfo->processor; 
    cpResult = AGCLIENT_CONTINUE; 
    while (AGCLIENT_CONTINUE == cpResult) { 
        cpResult = AGClientProcessorProcess(processor); 
        doWindowsMessagePump(); 
        if (AGCLIENT_CONTINUE == cpResult && *pInfo->quit) { 
            cancelled = TRUE; 
            processor->finished = TRUE; 
            cpResult = AGCLIENT_IDLE; 
        } 
    } 
   
    /* If we terminated via an error (i.e., we didn't end because of 
    an END command), signal main thread that we're done. */ 
    processor->finished = TRUE; 

    return cpResult; 
} 

static DWORD startClientProcessorThread(AGClientProcessor * processor, 
                                        AGBool * quit, 
                                        threadStruct * threadData) 
{ 
    DWORD threadID = 0; 

    processor->threadWriter = AGBufferWriterNew(1024); 
    if(processor->threadWriter) { 
        processor->readIndex = processor->writeIndex = 0; 
        processor->mutex = CreateMutex(NULL, FALSE, NULL); 
        if(processor->mutex) { 
            threadData->processor = processor; 
            threadData->quit = quit; 
            processor->threadHandle = CreateThread(NULL, 
                0, 
                quickClientProcessorLoop, 
                (LPVOID)threadData, 
                0, 
                &threadID); 
        } 
    } 
    return threadID; 
} 

static AGBool doClientProcessorLoop(SyncInfo * pInfo,
                                    AGUserConfig * userConfig,
                                    AGLocationConfig * locationConfig,
                                    AGPlatformCalls * platform,
                                    AGReader * r,
                                    AGWriter * w,
                                    AGNetCtx *ctx)
{
    int32 dummyError;
    int32 cpResult;
    int32 syncCount;
    int32 i, n;
    AGClientProcessor * clientProcessor = NULL;
    AGBool cancelled = FALSE;
    HINSTANCE hInst;
    char serverNameProto[MAX_PATH];
    char statusBarProto[MAX_PATH];
    threadStruct threadData; 
    DWORD threadID; 

    hInst = (HINSTANCE)GetWindowLong(pInfo->statusDialog->hDlg, GWL_HINSTANCE);
    LoadString(hInst,
        IDS_CONNECTING_TO_SERVER,
        serverNameProto,
        MAX_PATH);

    LoadString(hInst,
        IDS_STATUS_BAR_SYNCING,
        statusBarProto,
        MAX_PATH);

    n = AGUserConfigCount(userConfig);

    for (i = 0; i < n; ++i) {

        char serverNameBuf[MAX_PATH];

        AGDeviceInfo * deviceInfo = NULL;

        AGServerConfig * sc =
            AGUserConfigGetServerByIndex(userConfig, i);

        if (cancelled)
            continue;

        if (NULL == sc)
            continue;

        if (sc->disabled)
            continue;

        if (NULL == sc->serverName || sc->serverPort <= 0)
            continue;

        sprintf(serverNameBuf,
            serverNameProto,
            sc->serverName,
            sc->serverPort);

        AGSyncCEStatusDialogSetTask(pInfo->statusDialog,
            serverNameBuf);

        sprintf(serverNameBuf,
            statusBarProto,
            sc->serverName);

        AGSyncCEStatusDialogSetTitle(pInfo->statusDialog,
            serverNameBuf);

        pInfo->cp = AGCommandProcessorNew(sc);
        pInfo->cp->commands.out = pInfo;
        pInfo->cp->commands.performTaskFunc = performTask;
        pInfo->cp->commands.performItemFunc = performItem;
        pInfo->cp->commands.performGoodbyeFunc = performGoodbye;

        doStartServer(sc->serverName, sc->serverPort, userConfig,
            &pInfo->cp->serverConfig, r, w, &dummyError);

        syncCount = 0;
        do {
            deviceInfo = allocateAndReadDeviceInfo(r, w);

            AGCommandProcessorStart(pInfo->cp);
            clientProcessor = AGClientProcessorNew(pInfo->cp->serverConfig,
                                                   deviceInfo, 
                                                   locationConfig,
                                                   platform, 
                                                   TRUE,
                                                   ctx);
            AGClientProcessorSetBufferServerCommands(clientProcessor, FALSE);

            if (NULL == clientProcessor)
                continue;

            AGClientProcessorSync(clientProcessor);

            /* Spin up thread that will queue commands being read 
            from the network. */ 
            threadID = startClientProcessorThread(clientProcessor, 
                &pInfo->quit, 
                &threadData); 

            /* If thread creation was successful, start handling commands 
            in the main thread.  We need to do it this weird way because 
            the PalmOS can't talk to the device in anything but the original 
            thread in which the HotSync is taking place. Sure, this is CE,
            which doesn't have that restriction, but we're looking for some
            sort of code reusability here. */ 
            if (0 != threadID) { 

                /* Call the dispatcher. */ 
                AGClientProcessorBeginCommandDispatcher( 
                    clientProcessor, 
                    doWindowsMessagePump,
                    &pInfo->quit); 

                /* Once we've returned from dispatching commands, we wait 
                to make sure the second thread finished, and then get its 
                result. */ 
                WaitForSingleObject(clientProcessor->threadHandle, 
                    INFINITE); 
                cpResult = AGCLIENT_ERR; 
                GetExitCodeThread(clientProcessor->threadHandle, 
                    (LPDWORD)&cpResult); 
            } 
            else { 
                /* For some reason, we weren't able to create the second 
                thread.  So just go about syncing single-threadedly. */ 
                cpResult = AGCLIENT_CONTINUE; 
                while (AGCLIENT_CONTINUE == cpResult) { 
                    cpResult =
                        AGClientProcessorProcess(clientProcessor);
                    doWindowsMessagePump(); 
                    if (AGCLIENT_CONTINUE == cpResult && pInfo->quit) { 
                        cancelled = TRUE; 
                        cpResult = AGCLIENT_IDLE; 
                    } 
                }
            }
            AGClientProcessorFree(clientProcessor);

            if (clientProcessor->mutex) { 
                CloseHandle(clientProcessor->mutex); 
                clientProcessor->mutex = NULL; 
            } 
            if(clientProcessor->threadWriter) { 
                AGBufferWriterFree(clientProcessor->threadWriter); 
                clientProcessor->threadWriter = NULL; 
            }

            if (NULL != deviceInfo)
                AGDeviceInfoFree(deviceInfo);
        } while (!cancelled &&
                 AGCommandProcessorShouldSyncAgain(pInfo->cp) &&
                 syncCount++ < 2);

        doEndServer(r, w, &dummyError);

        AGCommandProcessorFree(pInfo->cp);
        pInfo->cp = NULL;
    }

    if (0 == n) {

        char message[MAX_PATH];
        char caption[MAX_PATH];

        LoadString(hInst,
            IDS_NO_SERVERS_MESSAGE,
            message,
            MAX_PATH);

        LoadString(hInst,
            IDS_NO_SERVERS_CAPTION,
            caption,
            MAX_PATH);

        MessageBoxTimeout(pInfo->statusDialog->hDlg,
            message,
            caption,
            MB_ICONEXCLAMATION | MB_OK,
            kTimeoutValue);

    }

    return TRUE; /* success */
}
/*---------------------------------------------------------------------------*/
static void checkForSecNet(SyncInfo * pInfo) 
{
    
    /* PENDING Must look in registry for dll */
    HINSTANCE h = LoadLibrary("secnet.dll");
    pInfo->hasseclib = 0;

    if (h) {
        pInfo->secnetinit = 
            (netInitFunc)GetProcAddress(h, "NetInit");
        pInfo->secnetclose = 
            (netCloseFunc)GetProcAddress(h, "NetClose");
        pInfo->sectoggle = 
            (netToggleFunc)GetProcAddress(h, "NetToggle");
        pInfo->secctxsize = 
            (netCtxSizeFunc)GetProcAddress(h, "NetGetCtxSize");

        if (pInfo->sectoggle && pInfo->secnetclose && 
            pInfo->secnetinit && pInfo->secctxsize)
            pInfo->hasseclib = 1;
    }
}

/* ----------------------------------------------------------------------------
*/
static AGDeviceEntry * makeDeviceEntry(char * username,
                                       AGDeviceType deviceType,
                                       char * basePath)
{
    char * temp;
    TCHAR userDir[MAX_PATH];

    strcpy(userDir, basePath);

    /* Build the path of the config file. */
    temp = (char*)AGSyncCommonGetStringConstant(agMALSubdirectoryName,
        FALSE);
    PathAppend(userDir, temp);
    free(temp);

    /* Make sure the path exists. */
    if (AG_NO_DEVICE_TYPE != deviceType)
        CreateDirectory(userDir, NULL);

    return AGMobileLinkDeviceEntryNew(strdup(username),
        deviceType,
        strdup(userDir));
}

/* ----------------------------------------------------------------------------
*/
static void setCurrentUser(void)
{
    AGDeviceEntry * dle;
    char deviceName[MAX_PATH];
    char tempDir[MAX_PATH];

    tempDir[0] = deviceName[0] = '\0';

    AGCEGetCurrentDeviceInfo(deviceName, tempDir, NULL);

    PathRemoveFileSpec(tempDir);

    if (0 == strlen(tempDir) || 0 == strlen(deviceName))
        return;
    
    dle = makeDeviceEntry(strdup(deviceName),
        AG_CE_DEVICE_TYPE,
        strdup(tempDir));

    AGMobileLinkSetCurrentDevice(dle);

    AGMobileLinkDeviceEntryFree(dle);

}

/* ----------------------------------------------------------------------------
*/
BOOL AGSyncToServer(HINSTANCE h) 
{
    _CrtSetDbgFlag( _CRTDBG_ALLOC_MEM_DF | _CRTDBG_CHECK_ALWAYS_DF |
                    _CRTDBG_CHECK_CRT_DF |
                    _CRTDBG_DELAY_FREE_MEM_DF |
                    _CRTDBG_LEAK_CHECK_DF);

    BOOL syncResult = FALSE;
    SyncInfo * pInfo = NULL;
    IRAPIStream * rapiStream = NULL;
    AGUserConfig * userConfig = NULL;
    AGUserConfig * agreedUserConfig = NULL;
    AGLocationConfig * locationConfig = NULL;
    AGPlatformCalls * platform = NULL;
    DWORD dwOut = 0;
    BYTE * pOut = NULL;
    int32 dummyError;
    AGNetCtx *ctx;
    AGArray * graphics = NULL;
    AGBool bitmapFormat;

    __try {

        setCurrentUser();

        /* NULL is ok here; just means we don't have any special
        proxy requirements. */
        locationConfig = readLocationConfig();

        rapiStream = startRAPIConnection(&dwOut, &pOut);
        if (NULL == rapiStream)
            __leave;

        pInfo = syncInfoNew();
        if (NULL == pInfo)
            __leave;

        /* Start Mobile Application Link net services. */
        checkForSecNet(pInfo);
        if (pInfo->hasseclib) {
            ctx = (AGNetCtx *)malloc((*pInfo->secctxsize)());
            (*pInfo->secnetinit)(ctx);
        } else {
            ctx = (AGNetCtx *)malloc(sizeof(AGNetCtx));
            AGNetInit(ctx);
        }

        graphics = AGSyncCommonLoadGraphics(h,
            IDI_GENERIC_PROGRESS_ICON0,
            &bitmapFormat);

        pInfo->rapiReader = AGSyncCEStreamCreateRAPIReader(rapiStream);
        pInfo->rapiWriter = AGSyncCEStreamCreateRAPIWriter(rapiStream);

        platform = setupPlatformCalls(pInfo);
        if (NULL == platform)
            __leave;

        pInfo->statusDialog = startStatusDialog(h,
            &pInfo->quit,
            graphics,
            bitmapFormat);
        if (NULL == pInfo->statusDialog)
            __leave;

        setStatusDialogToConnecting(pInfo->statusDialog, h);

        userConfig = syncUserConfigs(pInfo->rapiReader, pInfo->rapiWriter);
        writeUserConfig(userConfig);

        syncResult = doClientProcessorLoop(pInfo, userConfig, locationConfig,
                                           platform, pInfo->rapiReader, 
                                           pInfo->rapiWriter, ctx);

        userConfig = syncUserConfigs(pInfo->rapiReader, pInfo->rapiWriter);
        writeUserConfig(userConfig);

        if (!syncResult)
            __leave;

        AGWriteSTREAM_END(pInfo->rapiReader, pInfo->rapiWriter, &dummyError);
    }

    __finally {

        if (pInfo) {

            if (pInfo->hasseclib) {
                (*pInfo->secnetclose)(ctx);
            } else { 
                AGNetClose(ctx);
            }

            if (NULL != pInfo->statusDialog)
                AGSyncCEStatusDialogFree(pInfo->statusDialog);

            syncInfoFree(pInfo);

        } else {
            AGNetClose(ctx);
        }

        if (NULL != locationConfig)
            AGLocationConfigFree(locationConfig);

        if (NULL != userConfig)
            AGUserConfigFree(userConfig);

        if (NULL != platform)
            free(platform);

        if (NULL != graphics)
            AGSyncCommonFreeGraphics(graphics, bitmapFormat);

        if (NULL != pOut)
            CeRapiFreeBuffer(pOut);

        if (NULL != rapiStream)
            rapiStream->Release();

        free(ctx);

    }

    return syncResult;
}
