/* The contents of this file are subject to the Mozilla Public License
 * Version 1.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * License for the specific language governing rights and limitations
 * under the License.
 *
 * The Original Code is Mobile Application Link.
 *
 * The Initial Developer of the Original Code is AvantGo, Inc.
 * Portions created by AvantGo, Inc. are Copyright (C) 1997-1999
 * AvantGo, Inc. All Rights Reserved.
 *
 * Contributor(s):
 */

// Owner:  miket

#include <AGSyncCommon.h>
#include <AGSyncCECommon.h>
#include <AGSyncCEDeviceCommon.h>
#include <AGProtocol.h>
#include <AGClientProcessor.h>
#include <AGSyncCEDeviceDLLManager.h>

const WCHAR * REGKEY_DLL_LOC = L"Software\\Mobile Application Link\\Servers";
const WCHAR * REGKEY_DLL_SUB = L"Handler";
const WCHAR * REGKEY_DLL_DEF = L"Default";

/* ----------------------------------------------------------------------------
    AGSyncCEDeviceDLLManager * AGSyncCEDeviceDLLManagerNew(void)

    Creates a DLLManager, which handles the loading and freeing of application-
    specific DLLs.

*/
AGSyncCEDeviceDLLManager * AGSyncCEDeviceDLLManagerNew(void)
{
    AGSyncCEDeviceDLLManager * dll;

    dll = (AGSyncCEDeviceDLLManager *)malloc(sizeof(AGSyncCEDeviceDLLManager));
    if (NULL == dll)
        return NULL;

    dll->currentServerInfo = NULL;

    return dll;
}

/* ----------------------------------------------------------------------------
    void AGSyncCEDeviceDLLManagerFree(AGSyncCEDeviceDLLManager * dll)

    Release DLLManager.
*/
void AGSyncCEDeviceDLLManagerFree(AGSyncCEDeviceDLLManager * dll)
{
    AGSyncCEDeviceDLLManagerEndServer(dll);

    free(dll);
}

/* ----------------------------------------------------------------------------
    int32 AGSyncCEDeviceDLLManagerPerformCommand(
        AGSyncCEDeviceDLLManager * dll, AGCommand command, uint32 length,
        void * bytes, int32 *errCode)

    Pass the current MAL command to the correct client DLL, and return
    the given result.
*/
int32 AGSyncCEDeviceDLLManagerPerformCommand(AGSyncCEDeviceDLLManager * dll,
                                             int32 *errCode, AGReader *command)
{
    AGSyncCEDeviceDLLManagerDatabase & c =
        *(dll->currentServerInfo->currentDatabaseInfo);
    return c.PerformCommandFunc(c.out, errCode, command);
}

/* ----------------------------------------------------------------------------
    int32 AGSyncCEDeviceDLLManagerOpenDatabase(AGSyncCEDeviceDLLManager * dll,
                                               AGDBConfig * db,
                                               int32 *errCode)

    Pass the OpenDatabase command to the correct client DLL, and return
    the given result.
*/
int32 AGSyncCEDeviceDLLManagerOpenDatabase(AGSyncCEDeviceDLLManager * dll,
                                           AGDBConfig * db,
                                           int32 *errCode)
{
    AGSyncCEDeviceDLLManagerDatabase & c =
        *(dll->currentServerInfo->currentDatabaseInfo);
    return c.OpenDatabaseFunc(c.out, db, errCode);
}

/* ----------------------------------------------------------------------------
    int32 getRecordBase(AGSyncCEDeviceDLLManager * dll, AGBool modonly,
                        AGRecord ** record, int32 *errCode)

    Pass the current request for record to the correct client DLL and return
    the result.
*/
int32 getRecordBase(AGSyncCEDeviceDLLManager * dll, AGBool modonly,
                    AGRecord ** record, int32 *errCode)
{
    AGSyncCEDeviceDLLManagerDatabase & c =
        *(dll->currentServerInfo->currentDatabaseInfo);
    if (modonly)
        return c.GetNextModifiedRecordFunc(c.out, record, errCode);
    else
        return c.GetNextRecordFunc(c.out, record, errCode);
}

/* ----------------------------------------------------------------------------
    int32 AGSyncCEDeviceDLLManagerGetNextModifiedRecord(
                                                 AGSyncCEDeviceDLLManager * dll,
                                                 AGRecord ** record,
                                                 int32 *errCode)

    Pass the current request for record to the correct client DLL and return
    the result.
*/
int32 AGSyncCEDeviceDLLManagerGetNextModifiedRecord(
                                             AGSyncCEDeviceDLLManager * dll,
                                             AGRecord ** record,
                                             int32 *errCode)
{
    return getRecordBase(dll, TRUE, record, errCode);
}

/* ----------------------------------------------------------------------------
    int32 AGSyncCEDeviceDLLManagerGetNextRecord(AGSyncCEDeviceDLLManager * dll,
                                                AGRecord ** record,
                                                int32 *errCode)

    Pass the current request for record to the correct client DLL and return
    the result.
*/
int32 AGSyncCEDeviceDLLManagerGetNextRecord(AGSyncCEDeviceDLLManager * dll,
                                            AGRecord ** record,
                                            int32 *errCode)
{
    return getRecordBase(dll, FALSE, record, errCode);
}

/* ----------------------------------------------------------------------------
    static HKEY openClientDLLRegKey(AGServerConfig * serverConfig)

    Given a new serverConfig, open the root registry key pertaining to that
    server and any databases it handles.
*/
static HKEY openClientDLLRegKey(AGServerConfig * serverConfig)
{

    HKEY hKey = NULL;
    HRESULT hr;
    const TCHAR * szBackslash = L"\\";
    TCHAR * uniqueServerName;
    TCHAR * fullstr;

    /* Make server name key of format aaaa:xxxx */
    uniqueServerName = AGSyncCommonMakeUniqueServerString(
        serverConfig->serverName, serverConfig->serverPort);
    if (NULL == uniqueServerName)
        return NULL;

    fullstr = (TCHAR*)malloc(sizeof(TCHAR) *
        (lstrlen(REGKEY_DLL_LOC)
        + lstrlen(szBackslash)
        + lstrlen(uniqueServerName)
        + 1));
    if (NULL == fullstr)
        return NULL;

    lstrcpy(fullstr, REGKEY_DLL_LOC);
    lstrcat(fullstr, szBackslash);
    lstrcat(fullstr, uniqueServerName);
    free(uniqueServerName);

    /* Open the key containing all DLL path location keys. */    
    hr = RegOpenKeyEx(HKEY_LOCAL_MACHINE, fullstr, 0, NULL, &hKey);
    free(fullstr);
    if (ERROR_SUCCESS != hr) {
        fullstr = (WCHAR*)malloc(sizeof(WCHAR) *
            (lstrlen(REGKEY_DLL_LOC) + lstrlen(szBackslash) + 
            lstrlen(REGKEY_DLL_DEF) + 1));
        if (NULL != fullstr) {
            lstrcpy(fullstr, REGKEY_DLL_LOC);
            lstrcat(fullstr, szBackslash);
            lstrcat(fullstr, REGKEY_DLL_DEF);
            hr = RegOpenKeyEx(HKEY_LOCAL_MACHINE, fullstr, 0, NULL, &hKey);
            free(fullstr);
        }
    }

    return hKey;

}

/* ----------------------------------------------------------------------------
    static HMODULE loadDatabaseDLL(AGSyncCEDeviceDLLManagerServer * server,
    AGDBConfig * dbconfig)

    Given a dbconfig naming a database, load the client DLL that handles that
    database.
*/
static HMODULE loadDatabaseDLL(AGSyncCEDeviceDLLManagerServer * server,
                               AGDBConfig * dbconfig)
{

    HKEY hSubKey;
    HRESULT hr;
    WCHAR * wideDBName = NULL;
    HMODULE modresult = NULL;
    DWORD bufsize = 0;
    DWORD valueType = 0;

    if (NULL == server->registryKey)
        return NULL;

    /* Open subkey containing specific DLL path. */
    if (NULL != dbconfig) {
        int len = strlen(dbconfig->dbname);
        if (len > 0) {
            wideDBName = (WCHAR*)malloc(sizeof(WCHAR) * (len + 1));
            mbstowcs(wideDBName, dbconfig->dbname, -1);
        }
    }

    if (NULL != wideDBName) {
        hr = RegOpenKeyEx(server->registryKey,
            wideDBName, 0, NULL, &hSubKey);
        free(wideDBName);
    }
    else
        hSubKey = server->registryKey;

    hr = RegQueryValueEx(hSubKey, REGKEY_DLL_SUB, 0, &valueType,
        NULL, &bufsize);

    if (ERROR_SUCCESS == hr) {

        /* In bytes, not Unicode characters! */
        WCHAR * dllpath = (WCHAR*)malloc(bufsize + 1);

        if (NULL != dllpath) {

            RegQueryValueEx(hSubKey, REGKEY_DLL_SUB, 0, &valueType,
                (BYTE*)dllpath, &bufsize);

            if (ERROR_SUCCESS == hr)
                modresult = LoadLibrary(dllpath);

            free(dllpath);

        }

        RegCloseKey(hSubKey);

    }
    return modresult;

}

/* ----------------------------------------------------------------------------
    static void freeDatabaseDLL(AGSyncCEDeviceDLLManagerServer * server)

    Cleans up and releases the DLL that handles the current database.
*/
static void freeDatabaseDLL(AGSyncCEDeviceDLLManagerServer * server)
{
    AGSyncCEDeviceDLLManagerDatabase * db;

    db = server->currentDatabaseInfo;
    if (NULL != db->hAGDatabaseLib) {
        db->TermFunc(db->out);
        FreeLibrary(db->hAGDatabaseLib);
        db->hAGDatabaseLib = NULL;
    }

    free(server->currentDatabaseInfo);
    server->currentDatabaseInfo = NULL;

}

/* ----------------------------------------------------------------------------
    static int32 startDatabase(AGSyncCEDeviceDLLManagerServer * server,
                           AGDBConfig * dbconfig)

    Prepares for commands that will be sent to a particular database. Figures
    out which DLL handles the database, loads it, initializes it.
*/
static int32 startDatabase(AGSyncCEDeviceDLLManagerServer * server,
                           AGDBConfig * dbconfig)
{

    if (NULL != server->currentDatabaseInfo)
        return -1;

    server->currentDatabaseInfo = (AGSyncCEDeviceDLLManagerDatabase *)
        malloc(sizeof(AGSyncCEDeviceDLLManagerDatabase));

    if (NULL == server->currentDatabaseInfo)
        return -1;

    server->currentDatabaseInfo->hAGDatabaseLib =
        loadDatabaseDLL(server, dbconfig);

    if (NULL != server->currentDatabaseInfo->hAGDatabaseLib) {

        server->currentDatabaseInfo->InitFunc =
            (INITFUNC) GetProcAddress(
            server->currentDatabaseInfo->hAGDatabaseLib,
            L"Initialize");

        server->currentDatabaseInfo->TermFunc =
            (TERMFUNC) GetProcAddress(
            server->currentDatabaseInfo->hAGDatabaseLib,
            L"Terminate");

        server->currentDatabaseInfo->PerformCommandFunc =
            (AGPerformCommandFunc) GetProcAddress(
            server->currentDatabaseInfo->hAGDatabaseLib,
            L"PerformCommand");

        server->currentDatabaseInfo->OpenDatabaseFunc =
            (AGOpenDatabaseFunc) GetProcAddress(
            server->currentDatabaseInfo->hAGDatabaseLib,
            L"OpenDatabase");

        server->currentDatabaseInfo->GetNextRecordFunc =
            (AGGetNextRecordFunc) GetProcAddress(
            server->currentDatabaseInfo->hAGDatabaseLib,
            L"GetNextRecord");

        server->currentDatabaseInfo->GetNextModifiedRecordFunc =
            (AGGetNextModifiedRecordFunc) GetProcAddress(
            server->currentDatabaseInfo->hAGDatabaseLib,
            L"GetNextModifiedRecord");

        server->currentDatabaseInfo->GetNextExpansionCommandFunc =
            (AGGetNextExpansionCommandFunc) GetProcAddress(
            server->currentDatabaseInfo->hAGDatabaseLib,
            L"GetNextExpansionCommand");
        server->currentDatabaseInfo->out =
            server->currentDatabaseInfo->InitFunc(server->serverConfig);
    }
    else {
        MessageBox(NULL, TEXT("Couldn't find appropriate client DLL."),
            TEXT("Mobile Link"), MB_OK | MB_ICONEXCLAMATION);
        return 1;
    }
    if (NULL == server->currentDatabaseInfo->InitFunc ||
        NULL == server->currentDatabaseInfo->TermFunc ||
        NULL == server->currentDatabaseInfo->PerformCommandFunc ||
        NULL == server->currentDatabaseInfo->OpenDatabaseFunc ||
        NULL == server->currentDatabaseInfo->GetNextRecordFunc ||
        NULL == server->currentDatabaseInfo->GetNextModifiedRecordFunc) {
        MessageBox(NULL, TEXT("Client DLL was missing something."),
            TEXT("Mobile Link"), MB_OK | MB_ICONEXCLAMATION);
        return 1;
    }

    return 0;

}

/* ----------------------------------------------------------------------------
    static int32 endDatabase(AGSyncCEDeviceDLLManagerServer * server)

    Cleans up when commands to current database are done.

*/
static int32 endDatabase(AGSyncCEDeviceDLLManagerServer * server)
{
    if (NULL == server->currentDatabaseInfo)
        return -1;

    freeDatabaseDLL(server);

    return 0;
}

/* ----------------------------------------------------------------------------
    int32 AGSyncCEDeviceDLLManagerStartDatabase(AGSyncCEDeviceDLLManager * dll,
    AGDBConfig * dbconfig)

    Exposed wrapper for startDatabase.

*/
int32 AGSyncCEDeviceDLLManagerStartDatabase(AGSyncCEDeviceDLLManager * dll,
                                            AGDBConfig * dbconfig)
{
    return startDatabase(dll->currentServerInfo, dbconfig);
}

/* ----------------------------------------------------------------------------
    int32 AGSyncCEDeviceDLLManagerEndDatabase(AGSyncCEDeviceDLLManager * dll)

    Exposed wrapper for endDatabase.

*/
int32 AGSyncCEDeviceDLLManagerEndDatabase(AGSyncCEDeviceDLLManager * dll)
{
    return endDatabase(dll->currentServerInfo);
}

/* ----------------------------------------------------------------------------
    int32 AGSyncCEDeviceDLLManagerStartServer(AGSyncCEDeviceDLLManager * dll,
    AGServerConfig * serverConfig)

    Prepares to receive commands from a particular server.

*/
int32 AGSyncCEDeviceDLLManagerStartServer(AGSyncCEDeviceDLLManager * dll,
                                          AGServerConfig * serverConfig)
{
    /* Don't allow caller to start a new server without first finishing
    one. */
    if (NULL != dll->currentServerInfo)
        return -1;

    dll->currentServerInfo = (AGSyncCEDeviceDLLManagerServer *)
        malloc(sizeof(AGSyncCEDeviceDLLManagerServer));
    if (NULL == dll->currentServerInfo)
        return -1;
    bzero(dll->currentServerInfo, sizeof(AGSyncCEDeviceDLLManagerServer));

    dll->currentServerInfo->serverConfig = serverConfig;
    dll->currentServerInfo->registryKey = openClientDLLRegKey(serverConfig);
    dll->currentServerInfo->currentDatabaseInfo = NULL;

    /* HACK */
    startDatabase(dll->currentServerInfo, NULL);

    return 0;
}

/* ----------------------------------------------------------------------------
    int32 AGSyncCEDeviceDLLManagerEndServer(AGSyncCEDeviceDLLManager * dll)

    Finishes with current server.
*/
int32 AGSyncCEDeviceDLLManagerEndServer(AGSyncCEDeviceDLLManager * dll)
{
    if (NULL == dll->currentServerInfo)
        return 0;

    /* HACK */
    endDatabase(dll->currentServerInfo);

    /* Don't let server be closed when a database is still open. */
    if (NULL != dll->currentServerInfo->currentDatabaseInfo)
        return -1;

    RegCloseKey(dll->currentServerInfo->registryKey);
    dll->currentServerInfo->registryKey = NULL;

    free(dll->currentServerInfo);
    dll->currentServerInfo = NULL;

    return 0;
}