/* The contents of this file are subject to the Mozilla Public License
 * Version 1.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * License for the specific language governing rights and limitations
 * under the License.
 *
 * The Original Code is Mobile Application Link.
 *
 * The Initial Developer of the Original Code is AvantGo, Inc.
 * Portions created by AvantGo, Inc. are Copyright (C) 1997-1999
 * AvantGo, Inc. All Rights Reserved.
 *
 * Contributor(s):
 */

/*  Owner:  miket */

#include <AGUtil.h>
#include <AGProtocol.h>
#include <AGClientProcessor.h>
#include <AGSyncCommon.h>
#include <AGSyncCECommon.h>
#include <AGSyncCEDevice.h>
#include <AGSyncCEDeviceCommon.h>
#include <AGSyncCEDeviceDLLManager.h>
#include <AGSyncCEDeviceDataHandler.h>
#include <AGSyncCEDeviceExportFunctions.h>
#include <AGSyncCEStreamProtocol.h>

#define _ASSERT(x)

WCHAR * PREFS_NAME = L"\\windows\\malconfig.dat";

/* ----------------------------------------------------------------------------
    static AGDeviceInfo * createDeviceInfo(void)

    Create a deviceInfo record and fill it in.

*/
static AGDeviceInfo * createDeviceInfo(void)
{
    AGDeviceInfo * devInfo = NULL;

    devInfo = AGDeviceInfoNew();

    if (NULL != devInfo) {

        STORE_INFORMATION info;
        OSVERSIONINFO osver;
        HDC hdc = GetDC(NULL);
        char osverstringmajor[3];
        char osverstringminor[2];
        char osverstring[6];
        char * minorptr;

        /* Ask device how much memory it has. */
        GetStoreInformation(&info);

        /* Find out version of operating system. */
        osver.dwOSVersionInfoSize = sizeof(OSVERSIONINFO);
        GetVersionEx(&osver);
        _itoa(osver.dwMajorVersion, osverstringmajor, 10);
        if (osver.dwMinorVersion < 10) {
            strcpy(osverstringminor, "0");
            minorptr = &osverstringminor[1];
        }
        else
            minorptr = &osverstringminor[0];
        _itoa(osver.dwMinorVersion, minorptr, 10);
        strcpy(osverstring, osverstringmajor);
        strcat(osverstring, ".");
        strcat(osverstring, osverstringminor);

        devInfo->availableBytes   = info.dwFreeSize;
        devInfo->serialNumber     = strdup("");
        devInfo->osName           = strdup("WINCE_OS");
        devInfo->osVersion        = strdup(osverstring);
        devInfo->screenWidth      = GetDeviceCaps(hdc, HORZRES);
        devInfo->screenHeight     = GetDeviceCaps(hdc, VERTRES);
        devInfo->colorDepth       = GetDeviceCaps(hdc, BITSPIXEL);

        ReleaseDC(NULL, hdc);
    }
    return devInfo;
}

static void doCmdAGCE_STREAM_PERFORM_COMMAND(SyncInfo * pInfo)
{
    int32 error;
    int32 result;

    AGReadSTREAM_PERFORM_COMMAND(pInfo->reader);
    result = AGSyncCEDeviceDLLManagerPerformCommand(pInfo->dll, &error, pInfo->reader);
    AGWriteResponseSTREAM_PERFORM_COMMAND(pInfo->writer, result, error);
}

static void doCmdAGCE_STREAM_OPEN_DATABASE(SyncInfo * pInfo)
{
    int32 error, result;
    AGReadSTREAM_OPEN_DATABASE(pInfo->reader, &pInfo->currentDb);
    pInfo->record = AGRecordNew(0, AG_RECORD_UNMODIFIED, 0, 0, 0, 0);
    result = AGSyncCEDeviceDLLManagerOpenDatabase(pInfo->dll,
        pInfo->currentDb, &error);
    AGWriteResponseSTREAM_OPEN_DATABASE(pInfo->writer, result, error);    
}

static void doCmdAGCE_STREAM_START_SERVER(SyncInfo * pInfo)
{
    
    char * servername = NULL;
    int16 port;
    
    AGReadSTREAM_START_SERVER(pInfo->reader, &servername, &port);
    
    /* Look up this server in our records. If we can't find it,
    something is wrong. */
    
    pInfo->currentServerConfig = AGUserConfigGetServerByNameAndPort(
        pInfo->userConfig, servername, port, AGUSERCONFIG_FIND);
    
    if (NULL != pInfo->currentServerConfig) {
        /* Now that we have a serverconfig that works with this server,
        do whatever setup is necessary to start handling MAL requests for this
        server. */
        AGSyncCEDeviceDLLManagerStartServer(pInfo->dll,
            pInfo->currentServerConfig);
        AGWriteResponseSTREAM_START_SERVER(pInfo->writer, AGCLIENT_IDLE, 0);
    }
    else
        AGWriteResponseSTREAM_START_SERVER(pInfo->writer, AGCLIENT_ERR, 0);
    
    if (NULL != servername)
        free(servername);
}

static void doCmdAGCE_STREAM_END_SERVER(SyncInfo * pInfo)
{
    AGReadSTREAM_END_SERVER(pInfo->reader);
    AGSyncCEDeviceDLLManagerEndServer(pInfo->dll);
    AGWriteResponseSTREAM_END_SERVER(pInfo->writer, AGCLIENT_IDLE, 0);
    pInfo->currentServerConfig = NULL;
}

static int32 doGetNextRecord(SyncInfo * pInfo, AGBool modonly, int32 * error)
{
    uint32 result;

    /* Pass command onto the DLL handler. */
    if (modonly)
        result = AGSyncCEDeviceDLLManagerGetNextModifiedRecord(pInfo->dll,
        &pInfo->record, error);
    else
        result = AGSyncCEDeviceDLLManagerGetNextRecord(pInfo->dll,
        &pInfo->record, error);

    if (AGCLIENT_CONTINUE != result) {
        /* If we're done, clean up. */
        AGDBConfigFree(pInfo->currentDb);
        pInfo->currentDb = NULL;
        AGRecordFree(pInfo->record);
        pInfo->record = NULL;
    }
    return result;
}

static void doCmdAGCE_STREAM_GET_NEXT_RECORD(SyncInfo * pInfo)
{
    int32 error;
    int32 result;
    AGReadSTREAM_GET_NEXT_RECORD(pInfo->reader);
    result = doGetNextRecord(pInfo, FALSE, &error);
    AGWriteResponseSTREAM_GET_NEXT_RECORD(pInfo->writer, result, error,
        pInfo->record);
}

static void doCmdAGCE_STREAM_GET_NEXT_MODIFIED_RECORD(SyncInfo * pInfo)
{
    int32 error;
    int32 result;
    AGReadSTREAM_GET_NEXT_MODIFIED_RECORD(pInfo->reader);
    result = doGetNextRecord(pInfo, TRUE, &error);
    AGWriteResponseSTREAM_GET_NEXT_MODIFIED_RECORD(pInfo->writer,
        result, error, pInfo->record);
}

static void doCmdAGCE_STREAM_END(SyncInfo * pInfo)
{
    AGReadSTREAM_END(pInfo->reader);
    pInfo->streamDone = TRUE;
    AGWriteResponseSTREAM_END(pInfo->writer, AGCLIENT_IDLE, 0);
}

static void doCmdAGCE_STREAM_GET_DEVICE_USERCONFIG(SyncInfo * pInfo)
{
    AGReadSTREAM_GET_DEVICE_USERCONFIG(pInfo->reader);

    if (NULL != pInfo->userConfig) {
        AGWriteResponseSTREAM_GET_DEVICE_USERCONFIG(pInfo->writer,
            AGCLIENT_IDLE, 0, pInfo->userConfig);
    } else {
        AGWriteResponseSTREAM_GET_DEVICE_USERCONFIG(pInfo->writer,
            AGCLIENT_ERR, AGCLIENT_OPEN_ERR, NULL);
    }
}

static void doCmdAGCE_STREAM_PUT_DEVICE_USERCONFIG(SyncInfo * pInfo)
{
    if (NULL != pInfo->userConfig)
        AGUserConfigFree(pInfo->userConfig);
    AGReadSTREAM_PUT_DEVICE_USERCONFIG(pInfo->reader, &pInfo->userConfig);
    AGWriteResponseSTREAM_PUT_DEVICE_USERCONFIG(pInfo->writer,
        AGCLIENT_IDLE, 0);
    AGWriteUserConfigToDisk(PREFS_NAME, pInfo->userConfig);
}

static void doCmdAGCE_STREAM_GET_DEVICEINFO(SyncInfo * pInfo)
{
    int32 error = 0;
    AGDeviceInfo * devInfo;
    AGReadSTREAM_GET_DEVICEINFO(pInfo->reader);
    devInfo = createDeviceInfo();
    if (NULL != devInfo) {
        AGWriteResponseSTREAM_GET_DEVICEINFO(pInfo->writer,
            AGCLIENT_IDLE, error, devInfo);
        AGDeviceInfoFree(devInfo);
    }
    else
        AGWriteResponseSTREAM_GET_DEVICEINFO(pInfo->writer,
            AGCLIENT_ERR, AGCLIENT_OPEN_ERR, NULL);
}

/* ----------------------------------------------------------------------------
    static BOOL doWindowsMessagePump(BOOL eatQuit)

    returns TRUE only if WM_QUIT message was posted.

    if eatQuit is FALSE, we post another nearly identical quit message after
    seeing one.
    
*/
static BOOL doWindowsMessagePump(BOOL eatQuit)
{
    MSG msg;
    while (PeekMessage(&msg, 0, 0, 0, PM_REMOVE | PM_NOYIELD)) {
        TranslateMessage(&msg);
        DispatchMessage(&msg);
        if (msg.message == WM_QUIT) {
            if (!eatQuit)
                PostQuitMessage((int)msg.wParam);
            return TRUE;
        }
    }
    return FALSE;
}

#define HANDLE_COMMAND(message, buf)    \
    case (message): doCmd##message((SyncInfo*)buf); break
void doStreamCommand(SyncInfo * pInfo, AGCEStreamCommand command)
{

    switch(command) {
        HANDLE_COMMAND(AGCE_STREAM_END, pInfo);
        HANDLE_COMMAND(AGCE_STREAM_PERFORM_COMMAND, pInfo);
        HANDLE_COMMAND(AGCE_STREAM_OPEN_DATABASE, pInfo);
        HANDLE_COMMAND(AGCE_STREAM_GET_NEXT_RECORD, pInfo);
        HANDLE_COMMAND(AGCE_STREAM_GET_NEXT_MODIFIED_RECORD, pInfo);
        HANDLE_COMMAND(AGCE_STREAM_START_SERVER, pInfo);
        HANDLE_COMMAND(AGCE_STREAM_END_SERVER, pInfo);
        HANDLE_COMMAND(AGCE_STREAM_GET_DEVICE_USERCONFIG, pInfo);
        HANDLE_COMMAND(AGCE_STREAM_PUT_DEVICE_USERCONFIG, pInfo);
        HANDLE_COMMAND(AGCE_STREAM_GET_DEVICEINFO, pInfo);
    }
}

/* ----------------------------------------------------------------------------
    ExportFunc HRESULT _RAPI_HandleStream(DWORD cbInput,
          BYTE *pInput,
          DWORD *pcbOutput,
          BYTE **ppOutput,
          IRAPIStream *pIRAPIStream)

    RAPI server function.  Communicates with desktop-side DLL.

*/
HRESULT STDAPICALLTYPE _RAPI_HandleStream(DWORD cbInput,
                                          BYTE *pInput,
                                          DWORD *pcbOutput,
                                          BYTE **ppOutput,
                                          IRAPIStream *pIRAPIStream)
{
    SyncInfo * pInfo = NULL;

    *pcbOutput = 0;
    pInfo = (SyncInfo *)malloc(sizeof(SyncInfo));
    bzero(pInfo, sizeof(SyncInfo));
    pInfo->reader = AGSyncCEStreamCreateRAPIReader(pIRAPIStream);
    pInfo->writer = AGSyncCEStreamCreateRAPIWriter(pIRAPIStream);

    /* Allow client DLLs to initialize themselves. */
    pInfo->dll = AGSyncCEDeviceDLLManagerNew();
    if (NULL == pInfo->dll)
        return (HRESULT)-1;

    /* Read in our local copy of UserConfig record. */
    pInfo->userConfig = AGReadUserConfigFromDisk(PREFS_NAME);

    /* Mark that we don't currently have a server in mind. */
    pInfo->currentServerConfig = NULL;

    /* Read the downstream data and parse it. Then pass off the type of
    command to the right callback routine and send a result (and possibly
    real data) back upstream. */
    pInfo->streamDone = FALSE;
    while (!pInfo->streamDone) {
        AGCEStreamCommand command =
            (AGCEStreamCommand)AGReadCompactInt(pInfo->reader);
        doStreamCommand(pInfo, command);
        doWindowsMessagePump(TRUE);
    }

    if (NULL != pInfo) {

        if (NULL != pInfo->userConfig) {
            AGWriteUserConfigToDisk(PREFS_NAME, pInfo->userConfig);
            AGUserConfigFree(pInfo->userConfig);
        }

        if (NULL != pInfo->dll)
            AGSyncCEDeviceDLLManagerFree(pInfo->dll);

        if (NULL != pIRAPIStream) {

            if (NULL != pInfo->reader)
                AGReaderFree(pInfo->reader);

            if (NULL != pInfo->writer)
                AGWriterFree(pInfo->writer);

            pIRAPIStream->Release();

        }

        free(pInfo);
    }

    return 0;
}